/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>

#include "system.h"
#include "conv_s19.h"
#include "umutil.h"
#include "glue.h"

#include "chip_ti_am1808.h"

#define INCLUDE
#include "arch_arm_cpu.c"
#include "arch_arm_cp15.c"
#include "arch_ti_aintc.c"
#include "arch_ti_i2c.c"
#include "arch_ti_mmcsd.c"
#include "arch_ti_pll.c"
#include "arch_ti_psc.c"
#include "arch_ti_spi.c"
#include "arch_ti_syscfg.c"
#include "arch_ti_timer.c"
#include "arch_ti_uart.c"
#undef INCLUDE

#define CHIP_(x) chip_ti_am1808_ ## x

struct cpssp {
	struct process process;

	struct sig_std_logic *port_spi0_clk;
	struct sig_std_logic *port_spi0_simo;
	struct sig_std_logic *port_spi0_scsN[6];
	struct sig_std_logic *port_spi1_clk;
	struct sig_std_logic *port_spi1_simo;
	struct sig_std_logic *port_spi1_scsN[8];

	unsigned int state_gnd;
	unsigned int state_vcc;

	uint8_t ram[128*1024];
	uint8_t rom[64*1024];
	uint8_t ddr[256*1024*1024];

#define STATE
#define NAME		cpu
#define NAME_(x)	cpu_ ## x
#define CONFIG_DEBUG	0
#define CONFIG_VERSION	5
#define CONFIG_ARM	1
#define CONFIG_THUMB	1
#define CONFIG_E	1
#define CONFIG_M	0
#include "arch_arm_cpu.c"
#undef CONFIG_M
#undef CONFIG_E
#undef CONFIG_ARM
#undef CONFIG_THUMB
#undef CONFIG_VERSION
#undef CONFIG_DEBUG
#undef NAME_
#undef NAME

#define NAME		cp15
#define NAME_(x)	cp15_ ## x
#include "arch_arm_cp15.c"
#undef NAME_
#undef NAME

#define NAME		aintc
#define NAME_(x)	aintc_ ## x
#include "arch_ti_aintc.c"
#undef NAME_
#undef NAME

#define NAME		i2c0
#define NAME_(x)	i2c0_ ## x
#include "arch_ti_i2c.c"
#undef NAME_
#undef NAME

#define NAME		i2c1
#define NAME_(x)	i2c1_ ## x
#include "arch_ti_i2c.c"
#undef NAME_
#undef NAME

#define NAME		mmcsd0
#define NAME_(x)	mmcsd0_ ## x
#include "arch_ti_mmcsd.c"
#undef NAME_
#undef NAME

#define NAME		mmcsd1
#define NAME_(x)	mmcsd1_ ## x
#include "arch_ti_mmcsd.c"
#undef NAME_
#undef NAME

#define NAME		pll0
#define NAME_(x)	pll0_ ## x
#include "arch_ti_pll.c"
#undef NAME_
#undef NAME

#define NAME		pll1
#define NAME_(x)	pll1_ ## x
#include "arch_ti_pll.c"
#undef NAME_
#undef NAME

#define NAME		spi0
#define NAME_(x)	spi0_ ## x
#include "arch_ti_spi.c"
#undef NAME_
#undef NAME

#define NAME		spi1
#define NAME_(x)	spi1_ ## x
#include "arch_ti_spi.c"
#undef NAME_
#undef NAME

#define NAME		psc0
#define NAME_(x)	psc0_ ## x
#include "arch_ti_psc.c"
#undef NAME_
#undef NAME

#define NAME		psc1
#define NAME_(x)	psc1_ ## x
#include "arch_ti_psc.c"
#undef NAME_
#undef NAME

#define NAME		syscfg0
#define NAME_(x)	syscfg0_ ## x
#include "arch_ti_syscfg.c"
#undef NAME_
#undef NAME

#define NAME		syscfg1
#define NAME_(x)	syscfg1_ ## x
#include "arch_ti_syscfg.c"
#undef NAME_
#undef NAME

#define NAME		timer0
#define NAME_(x)	timer0_ ## x
#include "arch_ti_timer.c"
#undef NAME_
#undef NAME

#define NAME		timer1
#define NAME_(x)	timer1_ ## x
#include "arch_ti_timer.c"
#undef NAME_
#undef NAME

#define NAME		timer2
#define NAME_(x)	timer2_ ## x
#include "arch_ti_timer.c"
#undef NAME_
#undef NAME

#define NAME		timer3
#define NAME_(x)	timer3_ ## x
#include "arch_ti_timer.c"
#undef NAME_
#undef NAME

#define NAME		uart0
#define NAME_(x)	uart0_ ## x
#include "arch_ti_uart.c"
#undef NAME_
#undef NAME

#define NAME		uart1
#define NAME_(x)	uart1_ ## x
#include "arch_ti_uart.c"
#undef NAME_
#undef NAME

#define NAME		uart2
#define NAME_(x)	uart2_ ## x
#include "arch_ti_uart.c"
#undef NAME_
#undef NAME
#undef STATE
};

#define COUNT_INST(x)	;

/*
 * Forward declarations for cp15.
 */
/*forward*/ static void
cp15_mcr(struct cpssp *cpssp, uint32_t insn, uint32_t val);
/*forward*/ static void
cp15_mrc(struct cpssp *cpssp, uint32_t insn, uint32_t *valp);

/*
 * Forward declarations for aintc.
 */
/*forward*/ static void
aintc_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
aintc_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for i2c0.
 */
/*forward*/ static void
i2c0_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
i2c0_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for i2c1.
 */
/*forward*/ static void
i2c1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
i2c1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for mmcsd0.
 */
/*forward*/ static void
mmcsd0_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
mmcsd0_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for mmcsd1.
 */
/*forward*/ static void
mmcsd1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
mmcsd1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for pll0.
 */
/*forward*/ static void
pll0_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
pll0_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for pll1.
 */
/*forward*/ static void
pll1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
pll1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for psc0.
 */
/*forward*/ static void
psc0_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
psc0_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for psc1.
 */
/*forward*/ static void
psc1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
psc1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for spi0.
 */
/*forward*/ static void
spi0_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
spi0_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for spi1.
 */
/*forward*/ static void
spi1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
spi1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for syscfg0.
 */
/*forward*/ static void
syscfg0_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
syscfg0_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for syscfg1.
 */
/*forward*/ static void
syscfg1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
syscfg1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for timer0.
 */
/*forward*/ static void
timer0_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
timer0_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for timer1.
 */
/*forward*/ static void
timer1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
timer1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for timer2.
 */
/*forward*/ static void
timer2_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
timer2_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for timer3.
 */
/*forward*/ static void
timer3_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
timer3_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for uart0.
 */
/*forward*/ static void
uart0_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
uart0_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for uart1.
 */
/*forward*/ static void
uart1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
uart1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations for uart2.
 */
/*forward*/ static void
uart2_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
uart2_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Callbacks for cpu.
 */
static int
cpu_mcr(struct cpssp *cpssp, uint32_t insn, uint32_t val)
{
	switch ((insn >> 8) & 0xf) {
	case 0x0 ... 0xe:
		return -1;
	case 0xf:
		cp15_mcr(cpssp, insn, val);
		return 0;
	default:
		assert(0); /* Cannot happen. */
	}
}

static int
cpu_mrc(struct cpssp *cpssp, uint32_t insn, uint32_t *valp)
{
	switch ((insn >> 8) & 0xf) {
	case 0x0 ... 0xe:
		*valp = 0;
		return -1;
	case 0xf:
		cp15_mrc(cpssp, insn, valp);
		return 0;
	default:
		assert(0); /* Cannot happen. */
	}
}

static uint8_t
cpu_exc_ack(struct cpssp *cpssp)
{
	assert(0); /* FIXME */
	return 0;
}

static void
cpu_exc_eoi(struct cpssp *cpssp, uint8_t vec)
{
	assert(0); /* FIXME */
}

static void
cpu_SVCall_pending_set(struct cpssp *cpssp, unsigned int val)
{
	assert(0); /* FIXME */
}

static void
cpu_mw(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
#if 0
	fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
			__FUNCTION__, addr, bs, val);
#endif

	if (addr == 0x00000000) {
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n", __FUNCTION__,
				addr);

	} else if (0x01c10000 <= addr && addr < 0x01c10000 + 0x1000) {
		psc0_st(cpssp, addr, bs, val);

	} else if (0x01c11000 <= addr && addr < 0x01c11000 + 0x1000) {
		pll0_st(cpssp, addr, bs, val);

	} else if (0x01c14000 <= addr && addr < 0x01c14000 + 0x1000) {
		syscfg0_st(cpssp, addr, bs, val);

	} else if (0x01c20000 <= addr && addr < 0x01c20000 + 0x1000) {
		timer0_st(cpssp, addr, bs, val);

	} else if (0x01c21000 <= addr && addr < 0x01c21000 + 0x1000) {
		timer1_st(cpssp, addr, bs, val);

	} else if (0x01c22000 <= addr && addr < 0x01c22000 + 0x1000) {
		i2c0_st(cpssp, addr, bs, val);

	} else if (0x01c40000 <= addr && addr < 0x01c40000 + 0x1000) {
		mmcsd0_st(cpssp, addr, bs, val);

	} else if (0x01c41000 <= addr && addr < 0x01c41000 + 0x1000) {
		spi0_st(cpssp, addr, bs, val);

	} else if (0x01c42000 <= addr && addr < 0x01c42000 + 0x1000) {
		uart0_st(cpssp, addr, bs, val);

	} else if (0x01d0c000 <= addr && addr < 0x01d0c000 + 0x1000) {
		uart1_st(cpssp, addr, bs, val);

	} else if (0x01d0d000 <= addr && addr < 0x01d0d000 + 0x1000) {
		uart2_st(cpssp, addr, bs, val);

	} else if (0x01e1a000 <= addr && addr < 0x01e1a000 + 0x1000) {
		pll1_st(cpssp, addr, bs, val);

	} else if (0x01e1b000 <= addr && addr < 0x01e1b000 + 0x1000) {
		mmcsd1_st(cpssp, addr, bs, val);

	} else if (0x01e27000 <= addr && addr < 0x01e27000 + 0x1000) {
		psc1_st(cpssp, addr, bs, val);

	} else if (0x01e28000 <= addr && addr < 0x01e28000 + 0x1000) {
		i2c1_st(cpssp, addr, bs, val);

	} else if (0x01e2c000 <= addr && addr < 0x01e2c000 + 0x1000) {
		syscfg1_st(cpssp, addr, bs, val);

	} else if (0x01f0c000 <= addr && addr < 0x01f0c000 + 0x1000) {
		timer2_st(cpssp, addr, bs, val);

	} else if (0x01f0d000 <= addr && addr < 0x01f0d000 + 0x1000) {
		timer3_st(cpssp, addr, bs, val);

	} else if (0x01f0e000 <= addr && addr < 0x01f0e000 + 0x1000) {
		spi1_st(cpssp, addr, bs, val);

	} else if (0x80000000 <= addr && addr < 0x80000000 + sizeof(cpssp->ram)) {
		uint32_t off;

		off = addr - 0x80000000;
		if ((bs >> 0) & 1) {
			cpssp->ram[off + 0] = (val >> 0) & 0xff;
		}
		if ((bs >> 1) & 1) {
			cpssp->ram[off + 1] = (val >> 8) & 0xff;
		}
		if ((bs >> 2) & 1) {
			cpssp->ram[off + 2] = (val >> 16) & 0xff;
		}
		if ((bs >> 3) & 1) {
			cpssp->ram[off + 3] = (val >> 24) & 0xff;
		}

	} else if (0xfffd0000 <= addr && addr < 0xfffd0000 + sizeof(cpssp->rom)) {
		assert(0); /* Read-only */

	} else if (0xc0000000 <= addr && addr < 0xc0000000 + sizeof(cpssp->ddr)) {
		uint32_t off;

		off = addr - 0xc0000000;
		if ((bs >> 0) & 1) {
			cpssp->ddr[off + 0] = (val >> 0) & 0xff;
		}
		if ((bs >> 1) & 1) {
			cpssp->ddr[off + 1] = (val >> 8) & 0xff;
		}
		if ((bs >> 2) & 1) {
			cpssp->ddr[off + 2] = (val >> 16) & 0xff;
		}
		if ((bs >> 3) & 1) {
			cpssp->ddr[off + 3] = (val >> 24) & 0xff;
		}

	} else if (0xfffee000 <= addr && addr < 0xfffee000 + 0x2000) {
		aintc_st(cpssp, addr, bs, val);

	} else {
		fprintf(stderr, "ERROR: %s: addr=0x%08x (pc=0x%08x)\n",
				__FUNCTION__, addr, cpssp->cpu.pc);
		assert(0);
	}
}

static void
cpu_mr(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	if (addr == 0x00000000) {
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n", __FUNCTION__,
				addr);
		*valp = 0;

	} else if (0x01c10000 <= addr && addr < 0x01c10000 + 0x1000) {
		psc0_ld(cpssp, addr, bs, valp);

	} else if (0x01c11000 <= addr && addr < 0x01c11000 + 0x1000) {
		pll0_ld(cpssp, addr, bs, valp);

	} else if (0x01c14000 <= addr && addr < 0x01c14000 + 0x1000) {
		syscfg0_ld(cpssp, addr, bs, valp);

	} else if (0x01c20000 <= addr && addr < 0x01c20000 + 0x1000) {
		timer0_ld(cpssp, addr, bs, valp);

	} else if (0x01c21000 <= addr && addr < 0x01c21000 + 0x1000) {
		timer1_ld(cpssp, addr, bs, valp);

	} else if (0x01c22000 <= addr && addr < 0x01c22000 + 0x1000) {
		i2c0_ld(cpssp, addr, bs, valp);

	} else if (0x01c40000 <= addr && addr < 0x01c40000 + 0x1000) {
		mmcsd0_ld(cpssp, addr, bs, valp);

	} else if (0x01c41000 <= addr && addr < 0x01c41000 + 0x1000) {
		spi0_ld(cpssp, addr, bs, valp);

	} else if (0x01c42000 <= addr && addr < 0x01c42000 + 0x1000) {
		uart0_ld(cpssp, addr, bs, valp);

	} else if (0x01d0c000 <= addr && addr < 0x01d0c000 + 0x1000) {
		uart1_ld(cpssp, addr, bs, valp);

	} else if (0x01d0d000 <= addr && addr < 0x01d0d000 + 0x1000) {
		uart2_ld(cpssp, addr, bs, valp);

	} else if (0x01e1a000 <= addr && addr < 0x01e1a000 + 0x1000) {
		pll1_ld(cpssp, addr, bs, valp);

	} else if (0x01e1b000 <= addr && addr < 0x01e1b000 + 0x1000) {
		mmcsd1_ld(cpssp, addr, bs, valp);

	} else if (0x01e27000 <= addr && addr < 0x01e27000 + 0x1000) {
		psc1_ld(cpssp, addr, bs, valp);

	} else if (0x01e28000 <= addr && addr < 0x01e28000 + 0x1000) {
		i2c1_ld(cpssp, addr, bs, valp);

	} else if (0x01e2c000 <= addr && addr < 0x01e2c000 + 0x1000) {
		syscfg1_ld(cpssp, addr, bs, valp);

	} else if (0x01f0c000 <= addr && addr < 0x01f0c000 + 0x1000) {
		timer2_ld(cpssp, addr, bs, valp);

	} else if (0x01f0d000 <= addr && addr < 0x01f0d000 + 0x1000) {
		timer3_ld(cpssp, addr, bs, valp);

	} else if (0x01f0e000 <= addr && addr < 0x01f0e000 + 0x1000) {
		spi1_ld(cpssp, addr, bs, valp);

	} else if (0x80000000 <= addr && addr < 0x80000000 + sizeof(cpssp->ram)) {
		uint32_t off;

		off = addr - 0x80000000;

		*valp = *(uint32_t *) &cpssp->ram[off];

	} else if (0xfffd0000 <= addr && addr < 0xfffd0000 + sizeof(cpssp->rom)) {
		uint32_t off;

		off = addr - 0xfffd0000;

		*valp = *(uint32_t *) &cpssp->rom[off];

	} else if (0xc0000000 <= addr && addr < 0xc0000000 + sizeof(cpssp->ddr)) {
		uint32_t off;

		off = addr - 0xc0000000;

		*valp = *(uint32_t *) &cpssp->ddr[off];

	} else if (0xfffee000 <= addr && addr < 0xfffee000 + 0x2000) {
		aintc_ld(cpssp, addr, bs, valp);

	} else {
		fprintf(stderr, "ERROR: %s: addr=0x%08x (pc=0x%08x)\n",
				__FUNCTION__, addr, cpssp->cpu.pc);
		assert(0);
	}

#if 0
	fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
			__FUNCTION__, addr, bs, *valp);
#endif
#if 1
	if ((0xfffd0000 <= addr && addr < 0xfffd0000 + 64*1024)
	 || (0xc1080000 <= addr && addr < 0xc1080000 + 5*64*1024)) {
		if (loglevel) {
			fprintf(stderr, "DIS %08x %08x\n", addr, *valp);
		}
	}
#endif
}

static void
cpu_mx(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	cpu_mr(cpssp, addr, bs, valp);
}

static void
cpu_gpio_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
}

static void
cpu_gpio_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	*valp = 0; /* FIXME */
}

static void
cpu_stop(struct cpssp *cpssp, int deepflag)
{
}

/*
 * Callbacks for spi0.
 */
static void
spi0_clk_set(struct cpssp *cpssp, unsigned int val)
{
	sig_std_logic_set(cpssp->port_spi0_clk, cpssp, val);
}

static void
spi0_simo_set(struct cpssp *cpssp, unsigned int val)
{
	sig_std_logic_set(cpssp->port_spi0_simo, cpssp, val);
}

static void
spi0_csN_set(struct cpssp *cpssp, int n, unsigned int val)
{
	if (n < 6) {
		sig_std_logic_set(cpssp->port_spi0_scsN[n], cpssp, val);
	}
}

/*
 * Callbacks for spi1.
 */
static void
spi1_clk_set(struct cpssp *cpssp, unsigned int val)
{
	sig_std_logic_set(cpssp->port_spi1_clk, cpssp, val);
}

static void
spi1_simo_set(struct cpssp *cpssp, unsigned int val)
{
	sig_std_logic_set(cpssp->port_spi1_simo, cpssp, val);
}

static void
spi1_csN_set(struct cpssp *cpssp, int n, unsigned int val)
{
	sig_std_logic_set(cpssp->port_spi1_scsN[n], cpssp, val);
}

#define BEHAVIOR
#define NAME		cpu
#define NAME_(x)	cpu_ ## x
#define CONFIG_DEBUG	0
#define CONFIG_VERSION	5
#define CONFIG_ARM	1
#define CONFIG_THUMB	1
#define CONFIG_E	1
#define CONFIG_M	0
#include "arch_arm_cpu.c"
#undef CONFIG_M
#undef CONFIG_E
#undef CONFIG_THUMB
#undef CONFIG_ARM
#undef CONFIG_VERSION
#undef CONFIG_DEBUG
#undef NAME_
#undef NAME

#define NAME		cp15
#define NAME_(x)	cp15_ ## x
#include "arch_arm_cp15.c"
#undef NAME_
#undef NAME

#define NAME		aintc
#define NAME_(x)	aintc_ ## x
#include "arch_ti_aintc.c"
#undef NAME_
#undef NAME

#define NAME		i2c0
#define NAME_(x)	i2c0_ ## x
#include "arch_ti_i2c.c"
#undef NAME_
#undef NAME

#define NAME		i2c1
#define NAME_(x)	i2c1_ ## x
#include "arch_ti_i2c.c"
#undef NAME_
#undef NAME

#define NAME		mmcsd0
#define NAME_(x)	mmcsd0_ ## x
#include "arch_ti_mmcsd.c"
#undef NAME_
#undef NAME

#define NAME		mmcsd1
#define NAME_(x)	mmcsd1_ ## x
#include "arch_ti_mmcsd.c"
#undef NAME_
#undef NAME

#define NAME		pll0
#define NAME_(x)	pll0_ ## x
#include "arch_ti_pll.c"
#undef NAME_
#undef NAME

#define NAME		pll1
#define NAME_(x)	pll1_ ## x
#include "arch_ti_pll.c"
#undef NAME_
#undef NAME

#define NAME		psc0
#define NAME_(x)	psc0_ ## x
#include "arch_ti_psc.c"
#undef NAME_
#undef NAME

#define NAME		psc1
#define NAME_(x)	psc1_ ## x
#include "arch_ti_psc.c"
#undef NAME_
#undef NAME

#define NAME		spi0
#define NAME_(x)	spi0_ ## x
#include "arch_ti_spi.c"
#undef NAME_
#undef NAME

#define NAME		spi1
#define NAME_(x)	spi1_ ## x
#include "arch_ti_spi.c"
#undef NAME_
#undef NAME

#define NAME		syscfg0
#define NAME_(x)	syscfg0_ ## x
#include "arch_ti_syscfg.c"
#undef NAME_
#undef NAME

#define NAME		syscfg1
#define NAME_(x)	syscfg1_ ## x
#include "arch_ti_syscfg.c"
#undef NAME_
#undef NAME

#define NAME		timer0
#define NAME_(x)	timer0_ ## x
#include "arch_ti_timer.c"
#undef NAME_
#undef NAME

#define NAME		timer1
#define NAME_(x)	timer1_ ## x
#include "arch_ti_timer.c"
#undef NAME_
#undef NAME

#define NAME		timer2
#define NAME_(x)	timer2_ ## x
#include "arch_ti_timer.c"
#undef NAME_
#undef NAME

#define NAME		timer3
#define NAME_(x)	timer3_ ## x
#include "arch_ti_timer.c"
#undef NAME_
#undef NAME

#define NAME		uart0
#define NAME_(x)	uart0_ ## x
#include "arch_ti_uart.c"
#undef NAME_
#undef NAME

#define NAME		uart1
#define NAME_(x)	uart1_ ## x
#include "arch_ti_uart.c"
#undef NAME_
#undef NAME

#define NAME		uart2
#define NAME_(x)	uart2_ ## x
#include "arch_ti_uart.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

static void
rom_create(struct cpssp *cpssp, const char *name, const char *img)
{
	void *media;
	int ret;

	system_name_push(name);

	media = storage_create(system_path(),
			sizeof(cpssp->rom), buildpath(ROMDIR, img),
			conv_s19_open, conv_s19_close, conv_s19_read);
	assert(media);

	ret = storage_read(media, cpssp->rom, 5*64*1024, 0);
	assert(ret <= sizeof(cpssp->rom));

	ret = storage_destroy(media);
	assert(0 <= ret);

	system_name_pop();
}

static void
rom_destroy(struct cpssp *cpssp)
{
}

static void __attribute__((__noreturn__))
CHIP_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:	;
	while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
		cpu_clk(cpssp);
		timer0_clk(cpssp);
		timer1_clk(cpssp);
		timer2_clk(cpssp);
		timer3_clk(cpssp);
		cpssp->process.inst_cnt++;
	}

	sched_to_scheduler();
	goto again;
}

static void
CHIP_(gnd_set)(void *_cpssp, unsigned int val)
{
}

static void
CHIP_(vcc_set)(void *_cpssp, unsigned int val)
{
}

void *
CHIP_(create)(
	const char *name,
	const char *img,
	struct sig_manage *manage,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_vcc,
	struct sig_std_logic *port_spi0_clk,
	struct sig_std_logic *port_spi0_ena,
	struct sig_std_logic *port_spi0_simo,
	struct sig_std_logic *port_spi0_somi,
	struct sig_std_logic *port_XhashXspi0_scs0,
	struct sig_std_logic *port_XhashXspi0_scs1,
	struct sig_std_logic *port_XhashXspi0_scs2,
	struct sig_std_logic *port_XhashXspi0_scs3,
	struct sig_std_logic *port_XhashXspi0_scs4,
	struct sig_std_logic *port_XhashXspi0_scs5,
	struct sig_std_logic *port_spi1_clk,
	struct sig_std_logic *port_spi1_ena,
	struct sig_std_logic *port_spi1_simo,
	struct sig_std_logic *port_spi1_somi,
	struct sig_std_logic *port_XhashXspi1_scs0,
	struct sig_std_logic *port_XhashXspi1_scs1,
	struct sig_std_logic *port_XhashXspi1_scs2,
	struct sig_std_logic *port_XhashXspi1_scs3,
	struct sig_std_logic *port_XhashXspi1_scs4,
	struct sig_std_logic *port_XhashXspi1_scs5,
	struct sig_std_logic *port_XhashXspi1_scs6,
	struct sig_std_logic *port_XhashXspi1_scs7
)
{
	static const struct sig_std_logic_funcs gnd_funcs = {
		.std_logic_set = CHIP_(gnd_set),
	};
	static const struct sig_std_logic_funcs vcc_funcs = {
		.std_logic_set = CHIP_(vcc_set),
	};
	static const struct sig_std_logic_funcs spi0_somi_funcs = {
		.std_logic_set = spi0_somi_set,
	};
	static const struct sig_std_logic_funcs spi1_somi_funcs = {
		.std_logic_set = spi1_somi_set,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpu_create(name, cpssp);
	cp15_create(cpssp);
	aintc_create(cpssp);
	i2c0_create(cpssp);
	i2c1_create(cpssp);
	mmcsd0_create(cpssp);
	mmcsd1_create(cpssp);
	pll0_create(cpssp);
	pll1_create(cpssp);
	psc0_create(cpssp);
	psc1_create(cpssp);
	spi0_create(cpssp);
	spi1_create(cpssp);
	syscfg0_create(cpssp);
	syscfg1_create(cpssp);
	timer0_create(cpssp);
	timer1_create(cpssp);
	timer2_create(cpssp);
	timer3_create(cpssp);
	uart0_create(cpssp);
	uart1_create(cpssp);
	uart2_create(cpssp);
	rom_create(cpssp, "rom", img);

	cpu_reset(cpssp); /* FIXME */
	aintc_reset(cpssp); /* FIXME */
	mmcsd0_reset(cpssp); /* FIXME */
	mmcsd1_reset(cpssp); /* FIXME */
	timer0_reset(cpssp); /* FIXME */
	timer1_reset(cpssp); /* FIXME */
	timer2_reset(cpssp); /* FIXME */
	timer3_reset(cpssp); /* FIXME */

	cpssp->process.inst_hz = 8*1000*1000;
	sched_process_init(&cpssp->process, CHIP_(step), cpssp);

	/* Call */
	/* Out */
	cpssp->port_spi0_clk = port_spi0_clk;
	sig_std_logic_connect_out(port_spi0_clk, cpssp, SIG_STD_LOGIC_Z);
	cpssp->port_spi0_simo = port_spi0_simo;
	sig_std_logic_connect_out(port_spi0_simo, cpssp, SIG_STD_LOGIC_Z);
	cpssp->port_spi0_scsN[0] = port_XhashXspi0_scs0;
	sig_std_logic_connect_out(port_XhashXspi0_scs0, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi0_scsN[1] = port_XhashXspi0_scs1;
	sig_std_logic_connect_out(port_XhashXspi0_scs1, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi0_scsN[2] = port_XhashXspi0_scs2;
	sig_std_logic_connect_out(port_XhashXspi0_scs2, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi0_scsN[3] = port_XhashXspi0_scs3;
	sig_std_logic_connect_out(port_XhashXspi0_scs3, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi0_scsN[4] = port_XhashXspi0_scs4;
	sig_std_logic_connect_out(port_XhashXspi0_scs4, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi0_scsN[5] = port_XhashXspi0_scs5;
	sig_std_logic_connect_out(port_XhashXspi0_scs5, cpssp, SIG_STD_LOGIC_1);

	cpssp->port_spi1_clk = port_spi1_clk;
	sig_std_logic_connect_out(port_spi1_clk, cpssp, SIG_STD_LOGIC_Z);
	cpssp->port_spi1_simo = port_spi1_simo;
	sig_std_logic_connect_out(port_spi1_simo, cpssp, SIG_STD_LOGIC_Z);
	cpssp->port_spi1_scsN[0] = port_XhashXspi1_scs0;
	sig_std_logic_connect_out(port_XhashXspi1_scs0, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi1_scsN[1] = port_XhashXspi1_scs1;
	sig_std_logic_connect_out(port_XhashXspi1_scs1, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi1_scsN[2] = port_XhashXspi1_scs2;
	sig_std_logic_connect_out(port_XhashXspi1_scs2, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi1_scsN[3] = port_XhashXspi1_scs3;
	sig_std_logic_connect_out(port_XhashXspi1_scs3, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi1_scsN[4] = port_XhashXspi1_scs4;
	sig_std_logic_connect_out(port_XhashXspi1_scs4, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi1_scsN[5] = port_XhashXspi1_scs5;
	sig_std_logic_connect_out(port_XhashXspi1_scs5, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi1_scsN[6] = port_XhashXspi1_scs6;
	sig_std_logic_connect_out(port_XhashXspi1_scs6, cpssp, SIG_STD_LOGIC_1);
	cpssp->port_spi1_scsN[7] = port_XhashXspi1_scs7;
	sig_std_logic_connect_out(port_XhashXspi1_scs7, cpssp, SIG_STD_LOGIC_1);

	/* In */
	cpssp->state_gnd = 0;
	sig_std_logic_connect_in(port_gnd, cpssp, &gnd_funcs);

	cpssp->state_vcc = 0;
	sig_std_logic_connect_in(port_vcc, cpssp, &vcc_funcs);

	sig_std_logic_connect_in(port_spi0_somi, cpssp, &spi0_somi_funcs);

	sig_std_logic_connect_in(port_spi1_somi, cpssp, &spi1_somi_funcs);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	rom_destroy(cpssp);
	uart2_destroy(cpssp);
	uart1_destroy(cpssp);
	uart0_destroy(cpssp);
	timer3_destroy(cpssp);
	timer2_destroy(cpssp);
	timer1_destroy(cpssp);
	timer0_destroy(cpssp);
	syscfg1_destroy(cpssp);
	syscfg0_destroy(cpssp);
	spi1_destroy(cpssp);
	spi0_destroy(cpssp);
	psc1_destroy(cpssp);
	psc0_destroy(cpssp);
	pll1_destroy(cpssp);
	pll0_destroy(cpssp);
	mmcsd1_destroy(cpssp);
	mmcsd0_destroy(cpssp);
	i2c1_destroy(cpssp);
	i2c0_destroy(cpssp);
	aintc_destroy(cpssp);
	cp15_destroy(cpssp);
	cpu_destroy(cpssp);

	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
