/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue.h"

#include "chip_st_stm32f411xe.h"

#define INCLUDE
#include "chip_st_stm32f411xe_matrix.c"
#include "arch_arm_cpu.c"
#include "arch_arm_cp10.c"
#include "arch_arm_cp11.c"
#include "arch_arm_nvic.c"
#include "arch_arm_systick.c"
#include "arch_gen_flash.c"
#include "arch_gen_sram.c"
#include "arch_st_exti.c"
#include "arch_st_flashint.c"
#include "arch_st_gpio.c"
#include "arch_st_i2c.c"
#include "arch_st_pwr.c"
#include "arch_st_rcc.c"
#include "arch_st_syscfg.c"
#include "arch_st_tim.c"
#include "arch_st_usb.c"
#undef INCLUDE

#define CHIP_(x) chip_st_stm32f411xe_ ## x

struct cpssp {
	struct process process;

	struct sig_std_logic *port_pa[16];
	struct sig_std_logic *port_pb[16];
	struct sig_std_logic *port_pc[16];
	struct sig_std_logic *port_pd[16];
	struct sig_std_logic *port_pe[16];
	struct sig_std_logic *port_ph[16];

	unsigned int state_gnd;
	unsigned int state_vcc;

#define STATE

#define NAME		matrix
#define NAME_(x)	matrix_ ## x
#include "chip_st_stm32f411xe_matrix.c"
#undef NAME_
#undef NAME

#define NAME		cpu
#define NAME_(x)	cpu_ ## x
#define CONFIG_DEBUG	0
#define CONFIG_VERSION	7
#define CONFIG_ARM	0
#define CONFIG_THUMB	1
#define CONFIG_E	1
#define CONFIG_M	1
#include "arch_arm_cpu.c"
#undef CONFIG_M
#undef CONFIG_E
#undef CONFIG_ARM
#undef CONFIG_THUMB
#undef CONFIG_VERSION
#undef CONFIG_DEBUG
#undef NAME_
#undef NAME

#define NAME		cp10
#define NAME_(x)	cp10_ ## x
#include "arch_arm_cp10.c"
#undef NAME_
#undef NAME

#define NAME		cp11
#define NAME_(x)	cp11_ ## x
#include "arch_arm_cp11.c"
#undef NAME_
#undef NAME

#define NAME		nvic
#define NAME_(x)	nvic_ ## x
#define CONFIG_INTLINESNUM	96
#include "arch_arm_nvic.c"
#undef CONFIG_INTLINESNUM
#undef NAME_
#undef NAME

#define NAME		systick
#define NAME_(x)	systick_ ## x
#include "arch_arm_systick.c"
#undef NAME_
#undef NAME

#define NAME		flash
#define NAME_(x)	flash_ ## x
#define FLASH_SIZE	(512*1024)
#include "arch_gen_flash.c"
#undef FLASH_SIZE
#undef NAME_
#undef NAME

#define NAME		sram1
#define NAME_(x)	sram1_ ## x
#define SRAM_SIZE	(128*1024)
#include "arch_gen_sram.c"
#undef SRAM_SIZE
#undef NAME_
#undef NAME

#define NAME		exti
#define NAME_(x)	exti_ ## x
#include "arch_st_exti.c"
#undef NAME_
#undef NAME

#define NAME		flashint
#define NAME_(x)	flashint_ ## x
#include "arch_st_flashint.c"
#undef NAME_
#undef NAME

#define NAME		gpioa
#define NAME_(x)	gpioa_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpiob
#define NAME_(x)	gpiob_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpioc
#define NAME_(x)	gpioc_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpiod
#define NAME_(x)	gpiod_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpioe
#define NAME_(x)	gpioe_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpioh
#define NAME_(x)	gpioh_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		i2c1
#define NAME_(x)	i2c1_ ## x
#include "arch_st_i2c.c"
#undef NAME_
#undef NAME

#define NAME		i2c2
#define NAME_(x)	i2c2_ ## x
#include "arch_st_i2c.c"
#undef NAME_
#undef NAME

#define NAME		i2c3
#define NAME_(x)	i2c3_ ## x
#include "arch_st_i2c.c"
#undef NAME_
#undef NAME

#define NAME		pwr
#define NAME_(x)	pwr_ ## x
#include "arch_st_pwr.c"
#undef NAME_
#undef NAME

#define NAME		rcc
#define NAME_(x)	rcc_ ## x
#include "arch_st_rcc.c"
#undef NAME_
#undef NAME

#define NAME		syscfg
#define NAME_(x)	syscfg_ ## x
#include "arch_st_syscfg.c"
#undef NAME_
#undef NAME

#define NAME		tim1
#define NAME_(x)	tim1_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim2
#define NAME_(x)	tim2_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim3
#define NAME_(x)	tim3_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim4
#define NAME_(x)	tim4_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim5
#define NAME_(x)	tim5_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim9
#define NAME_(x)	tim9_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim10
#define NAME_(x)	tim10_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim11
#define NAME_(x)	tim11_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		usb
#define NAME_(x)	usb_ ## x
#include "arch_st_usb.c"
#undef NAME_
#undef NAME

#undef STATE
};

#define COUNT_INST(x)	;

/*
 * Forward declarations cpu.
 */
/*forward*/ static void
cpu_irq_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
cpu_SCB_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
cpu_SCB_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
cpu_reset(struct cpssp *cpssp);

/*
 * Forward declarations cp10.
 */
/*forward*/ static void
cp10_mcr(struct cpssp *cpssp, uint32_t insn, uint32_t val);
/*forward*/ static void
cp10_mrc(struct cpssp *cpssp, uint32_t insn, uint32_t *valp);
/*forward*/ static void
cp10_reset(struct cpssp *cpssp);
/*forward*/ static void
cp10_create(struct cpssp *cpssp);
/*forward*/ static void
cp10_destroy(struct cpssp *cpssp);

/*
 * Forward declarations cp11.
 */
/*forward*/ static void
cp11_mcr(struct cpssp *cpssp, uint32_t insn, uint32_t val);
/*forward*/ static void
cp11_mrc(struct cpssp *cpssp, uint32_t insn, uint32_t *valp);
/*forward*/ static void
cp11_reset(struct cpssp *cpssp);
/*forward*/ static void
cp11_create(struct cpssp *cpssp);
/*forward*/ static void
cp11_destroy(struct cpssp *cpssp);

/*
 * Forward declarations nvic.
 */
/*forward*/ static uint8_t
nvic_exc_ack(struct cpssp *cpssp);
/*forward*/ static void
nvic_exc_eoi(struct cpssp *cpssp, uint8_t vec);
/*forward*/ static void
nvic_primask_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static unsigned int
nvic_primask_get(struct cpssp *cpssp);
/*forward*/ static void
nvic_pri_set(struct cpssp *cpssp, unsigned int n, unsigned int val);
/*forward*/ static unsigned int
nvic_pri_get(struct cpssp *cpssp, unsigned int n);
/*forward*/ static void
nvic_NMI_pending_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
nvic_PendSV_pending_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
nvic_SVCall_pending_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
nvic_systick_pending_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
nvic_irq_N_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
nvic_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
nvic_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
nvic_reset(struct cpssp *cpssp);
/*forward*/ static void
nvic_create(struct cpssp *cpssp);
/*forward*/ static void
nvic_destroy(struct cpssp *cpssp);

/*
 * Forward declarations systick.
 */
/*forward*/ static void
systick_st(struct cpssp *cpssp, uint32_t addr, uint32_t val);
/*forward*/ static void
systick_ld(struct cpssp *cpssp, uint32_t addr, uint32_t *valp);
/*forward*/ static void
systick_reset(struct cpssp *cpssp);
/*forward*/ static void
systick_create(struct cpssp *cpssp);
/*forward*/ static void
systick_destroy(struct cpssp *cpssp);

/*
 * Forward declarations for matrix.
 */
/*forward*/ static void
matrix_portaNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
matrix_portbNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
matrix_portcNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
matrix_portdNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
matrix_porteNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
matrix_porthNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);

/*
 * Forward declarations for flash.
 */
/*forward*/ static void
flash_read(struct cpssp *cpssp, uint32_t addr, uint32_t *valp);

/*
 * Forward declarations for sram1.
 */
/*forward*/ static void
sram1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
sram1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);

/*
 * Forward declarations exti.
 */
/*forward*/ static void
exti_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
exti_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
exti_inN_set(struct cpssp *cpssp, int pin, unsigned int val);
/*forward*/ static void
exti_reset(struct cpssp *cpssp);
/*forward*/ static void
exti_create(struct cpssp *cpssp);
/*forward*/ static void
exti_destroy(struct cpssp *cpssp);

/*
 * Forward declarations flashint.
 */
/*forward*/ static void
flashint_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
flashint_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
flashint_reset(struct cpssp *cpssp);
/*forward*/ static void
flashint_create(struct cpssp *cpssp);
/*forward*/ static void
flashint_destroy(struct cpssp *cpssp);

/*
 * Forward declarations gpioa.
 */
/*forward*/ static void
gpioa_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
gpioa_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
gpioa_leftNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
gpioa_rightN_in_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
gpioa_reset(struct cpssp *cpssp);
/*forward*/ static void
gpioa_create(struct cpssp *cpssp);
/*forward*/ static void
gpioa_destroy(struct cpssp *cpssp);

/*
 * Forward declarations gpiob.
 */
/*forward*/ static void
gpiob_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
gpiob_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
gpiob_leftNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
gpiob_rightN_in_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
gpiob_reset(struct cpssp *cpssp);
/*forward*/ static void
gpiob_create(struct cpssp *cpssp);
/*forward*/ static void
gpiob_destroy(struct cpssp *cpssp);

/*
 * Forward declarations gpioc.
 */
/*forward*/ static void
gpioc_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
gpioc_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
gpioc_leftNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
gpioc_rightN_in_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
gpioc_reset(struct cpssp *cpssp);
/*forward*/ static void
gpioc_create(struct cpssp *cpssp);
/*forward*/ static void
gpioc_destroy(struct cpssp *cpssp);

/*
 * Forward declarations gpiod.
 */
/*forward*/ static void
gpiod_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
gpiod_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
gpiod_leftNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
gpiod_rightN_in_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
gpiod_reset(struct cpssp *cpssp);
/*forward*/ static void
gpiod_create(struct cpssp *cpssp);
/*forward*/ static void
gpiod_destroy(struct cpssp *cpssp);

/*
 * Forward declarations gpioe.
 */
/*forward*/ static void
gpioe_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
gpioe_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
gpioe_leftNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
gpioe_rightN_in_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
gpioe_reset(struct cpssp *cpssp);
/*forward*/ static void
gpioe_create(struct cpssp *cpssp);
/*forward*/ static void
gpioe_destroy(struct cpssp *cpssp);

/*
 * Forward declarations gpioh.
 */
/*forward*/ static void
gpioh_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
gpioh_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
gpioh_leftNM_in_set(struct cpssp *cpssp, int n, int alt, unsigned int val);
/*forward*/ static void
gpioh_rightN_in_set(struct cpssp *cpssp, int n, unsigned int val);
/*forward*/ static void
gpioh_reset(struct cpssp *cpssp);
/*forward*/ static void
gpioh_create(struct cpssp *cpssp);
/*forward*/ static void
gpioh_destroy(struct cpssp *cpssp);

/*
 * Forward declarations i2c1.
 */
/*forward*/ static void
i2c1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
i2c1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
i2c1_scl_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
i2c1_sda_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
i2c1_smba_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
i2c1_clk(struct cpssp *cpssp);
/*forward*/ static void
i2c1_reset(struct cpssp *cpssp);
/*forward*/ static void
i2c1_create(struct cpssp *cpssp);
/*forward*/ static void
i2c1_destroy(struct cpssp *cpssp);

/*
 * Forward declarations i2c2.
 */
/*forward*/ static void
i2c2_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
i2c2_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
i2c2_scl_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
i2c2_sda_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
i2c2_smba_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
i2c2_clk(struct cpssp *cpssp);
/*forward*/ static void
i2c2_reset(struct cpssp *cpssp);
/*forward*/ static void
i2c2_create(struct cpssp *cpssp);
/*forward*/ static void
i2c2_destroy(struct cpssp *cpssp);

/*
 * Forward declarations i2c3.
 */
/*forward*/ static void
i2c3_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
i2c3_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
i2c3_scl_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
i2c3_sda_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
i2c3_smba_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
i2c3_clk(struct cpssp *cpssp);
/*forward*/ static void
i2c3_reset(struct cpssp *cpssp);
/*forward*/ static void
i2c3_create(struct cpssp *cpssp);
/*forward*/ static void
i2c3_destroy(struct cpssp *cpssp);

/*
 * Forward declarations pwr.
 */
/*forward*/ static void
pwr_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
pwr_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
pwr_reset(struct cpssp *cpssp);
/*forward*/ static void
pwr_create(struct cpssp *cpssp);
/*forward*/ static void
pwr_destroy(struct cpssp *cpssp);

/*
 * Forward declarations rcc.
 */
/*forward*/ static void
rcc_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
rcc_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
rcc_reset(struct cpssp *cpssp);
/*forward*/ static void
rcc_create(struct cpssp *cpssp);
/*forward*/ static void
rcc_destroy(struct cpssp *cpssp);

/*
 * Forward declarations syscfg.
 */
/*forward*/ static void
syscfg_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
syscfg_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
syscfg_inNM_set(struct cpssp *cpssp, int n, int m, unsigned int val);
/*forward*/ static void
syscfg_reset(struct cpssp *cpssp);
/*forward*/ static void
syscfg_create(struct cpssp *cpssp);
/*forward*/ static void
syscfg_destroy(struct cpssp *cpssp);

/*
 * Forward declarations tim1.
 */
/*forward*/ static void
tim1_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
tim1_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
tim1_reset(struct cpssp *cpssp);
/*forward*/ static void
tim1_create(struct cpssp *cpssp);
/*forward*/ static void
tim1_destroy(struct cpssp *cpssp);

/*
 * Forward declarations tim2.
 */
/*forward*/ static void
tim2_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
tim2_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
tim2_reset(struct cpssp *cpssp);
/*forward*/ static void
tim2_create(struct cpssp *cpssp);
/*forward*/ static void
tim2_destroy(struct cpssp *cpssp);

/*
 * Forward declarations tim3.
 */
/*forward*/ static void
tim3_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
tim3_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
tim3_reset(struct cpssp *cpssp);
/*forward*/ static void
tim3_create(struct cpssp *cpssp);
/*forward*/ static void
tim3_destroy(struct cpssp *cpssp);

/*
 * Forward declarations tim4.
 */
/*forward*/ static void
tim4_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
tim4_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
tim4_reset(struct cpssp *cpssp);
/*forward*/ static void
tim4_create(struct cpssp *cpssp);
/*forward*/ static void
tim4_destroy(struct cpssp *cpssp);

/*
 * Forward declarations tim5.
 */
/*forward*/ static void
tim5_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
tim5_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
tim5_reset(struct cpssp *cpssp);
/*forward*/ static void
tim5_create(struct cpssp *cpssp);
/*forward*/ static void
tim5_destroy(struct cpssp *cpssp);

/*
 * Forward declarations tim9.
 */
/*forward*/ static void
tim9_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
tim9_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
tim9_reset(struct cpssp *cpssp);
/*forward*/ static void
tim9_create(struct cpssp *cpssp);
/*forward*/ static void
tim9_destroy(struct cpssp *cpssp);

/*
 * Forward declarations tim10.
 */
/*forward*/ static void
tim10_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
tim10_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
tim10_reset(struct cpssp *cpssp);
/*forward*/ static void
tim10_create(struct cpssp *cpssp);
/*forward*/ static void
tim10_destroy(struct cpssp *cpssp);

/*
 * Forward declarations tim11.
 */
/*forward*/ static void
tim11_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
tim11_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
tim11_reset(struct cpssp *cpssp);
/*forward*/ static void
tim11_create(struct cpssp *cpssp);
/*forward*/ static void
tim11_destroy(struct cpssp *cpssp);

/*
 * Forward declarations usb.
 */
/*forward*/ static void
usb_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
usb_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
usb_reset(struct cpssp *cpssp);
/*forward*/ static void
usb_create(struct cpssp *cpssp);
/*forward*/ static void
usb_destroy(struct cpssp *cpssp);

/*
 * Callbacks for cpu.
 */
static int
cpu_mcr(struct cpssp *cpssp, uint32_t insn, uint32_t val)
{
	switch ((insn >> 8) & 0xf) {
	case 0x0 ... 0x9:
	case 0xc ... 0xf:
		return -1;
	case 0xa:
		cp10_mcr(cpssp, insn, val);
		return 0;
	case 0xb:
		cp11_mcr(cpssp, insn, val);
		return 0;
	default:
		assert(0); /* Cannot happen. */
	}
}

static int
cpu_mrc(struct cpssp *cpssp, uint32_t insn, uint32_t *valp)
{
	switch ((insn >> 8) & 0xf) {
	case 0x0 ... 0x9:
	case 0xc ... 0xf:
		*valp = 0;
		return -1;
	case 0xa:
		cp10_mrc(cpssp, insn, valp);
		return 0;
	case 0xb:
		cp11_mrc(cpssp, insn, valp);
		return 0;
	default:
		assert(0); /* Cannot happen. */
	}
}

static uint8_t
cpu_exc_ack(struct cpssp *cpssp)
{
	return nvic_exc_ack(cpssp);
}

static void
cpu_exc_eoi(struct cpssp *cpssp, uint8_t vec)
{
	nvic_exc_eoi(cpssp, vec);
}

static void
cpu_primask_set(struct cpssp *cpssp, unsigned int val)
{
	nvic_primask_set(cpssp, val);
}

static unsigned int
cpu_primask_get(struct cpssp *cpssp)
{
	return nvic_primask_get(cpssp);
}

static void
cpu_pri_set(struct cpssp *cpssp, int n, unsigned int val)
{
	nvic_pri_set(cpssp, n, val);
}

static unsigned int
cpu_pri_get(struct cpssp *cpssp, int n)
{
	return nvic_pri_get(cpssp, n);
}

static void
cpu_NMI_pending_set(struct cpssp *cpssp, unsigned int val)
{
	nvic_NMI_pending_set(cpssp, val);
}

static void
cpu_PendSV_pending_set(struct cpssp *cpssp, unsigned int val)
{
	nvic_PendSV_pending_set(cpssp, val);
}

static void
cpu_SVCall_pending_set(struct cpssp *cpssp, unsigned int val)
{
	nvic_SVCall_pending_set(cpssp, val);
}

static void
cpu_SysTick_pending_set(struct cpssp *cpssp, unsigned int val)
{
	nvic_systick_pending_set(cpssp, val);
}

static void
CHIP_(ppb_st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
#if 0
	fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
			__FUNCTION__, addr, bs, val);
#endif

	addr &= 0x100000 - 1;

	switch (addr) {
	case 0x0e000 ... 0x0e00f:
		/* Reserved (0x000-0x00f) */
		/* FIXME */
		/* Auxiliary Control Register */
		/* B3-274 */
		assert(0); /* FIXME */
	case 0x0e010 ... 0x0e0ff:
		/* System Timer Register */
		/* B3.3.2 */
		assert(bs == 0b1111); /* UNPREDICTABLE */

		systick_st(cpssp, addr, val);
		break;
	case 0x0e100 ... 0x0ecff:
		/* Nested Vectored Interrupt Controller Register */
		/* 3.4.2 */
		nvic_st(cpssp, addr, bs, val);
		break;
	case 0x0ed00 ... 0x0ed8f:
		/* System Control Block */
		/* B3.2.2 */
		cpu_SCB_st(cpssp, addr, bs, val);
		break;
#if defined(CONFIG_DEBUG) && CONFIG_DEBUG
	case 0x0edf0 ... 0x0eeff:
		/* Debug */
		assert(0);
#endif
	default:
		/* Reserved */
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
	}
}

static void
CHIP_(ppb_ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x100000 - 1;

	switch (addr) {
	case 0x0e000 ... 0x0e00f:
		/* Reserved */
		/* FIXME */
		/* Auxiliary Control Register */
		/* B3-274 */
		assert(0); /* FIXME */
	case 0x0e010 ... 0x0e0ff:
		/* System Timer Register */
		/* B3.3.2 */
		assert(bs == 0b1111); /* UNPREDICTABLE */

		systick_ld(cpssp, addr, valp);
		break;
	case 0x0e100 ... 0x0ecff:
		/* Nested Vectored Interrupt Controller Register */
		/* 3.4.2 */
		nvic_ld(cpssp, addr, bs, valp);
		break;
	case 0x0ed00 ... 0x0ed8f:
		/* System Control and ID Registers */
		/* B3.2.2 */
		cpu_SCB_ld(cpssp, addr, bs, valp);
		break;
#if defined(CONFIG_DEBUG) && CONFIG_DEBUG
	case 0x0edf0 ... 0x0eeff:
		/* Debug */
		assert(0); /* FIXME */
#endif
	default:
		/* Reserved */
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
	}

#if 1
	fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
			__FUNCTION__, addr, bs, *valp);
#endif
}

static void
cpu_mw(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
#if 0
	fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
			__FUNCTION__, addr, bs, val);
#endif

	/*
	 * 2.3 Memory map, 37/836
	 */
	if (0x00000000 <= addr && addr < 0x00080000) {
		/* Flash/SRAM1/System memory (aliased) */
		goto flash; /* FIXME */

	} else if (0x08000000 <= addr && addr < 0x08080000) {
		/* Flash */
	flash:	;
		assert(0); /* Read-only */

	} else if (0x20000000 <= addr && addr < 0x20020000) {
		sram1_st(cpssp, addr, bs, val);

	} else if (0x40000000 <= addr && addr < 0x40000400) {
		tim2_st(cpssp, addr, bs, val);

	} else if (0x40000400 <= addr && addr < 0x40000800) {
		tim3_st(cpssp, addr, bs, val);

	} else if (0x40000800 <= addr && addr < 0x40000c00) {
		tim4_st(cpssp, addr, bs, val);

	} else if (0x40000c00 <= addr && addr < 0x40001000) {
		tim5_st(cpssp, addr, bs, val);

	} else if (0x40005400 <= addr && addr < 0x40005800) {
		i2c1_st(cpssp, addr, bs, val);

	} else if (0x40005800 <= addr && addr < 0x40005c00) {
		i2c2_st(cpssp, addr, bs, val);

	} else if (0x40005c00 <= addr && addr < 0x40006000) {
		i2c3_st(cpssp, addr, bs, val);

	} else if (0x40007000 <= addr && addr < 0x40007400) {
		pwr_st(cpssp, addr, bs, val);

	} else if (0x40010000 <= addr && addr < 0x40010400) {
		tim1_st(cpssp, addr, bs, val);

	} else if (0x40013800 <= addr && addr < 0x40013c00) {
		syscfg_st(cpssp, addr, bs, val);

	} else if (0x40013c00 <= addr && addr < 0x40014000) {
		exti_st(cpssp, addr, bs, val);

	} else if (0x40014000 <= addr && addr < 0x40014400) {
		tim9_st(cpssp, addr, bs, val);

	} else if (0x40014400 <= addr && addr < 0x40014800) {
		tim10_st(cpssp, addr, bs, val);

	} else if (0x40014800 <= addr && addr < 0x40014c00) {
		tim11_st(cpssp, addr, bs, val);

	} else if (0x40020000 <= addr && addr < 0x40020400) {
		gpioa_st(cpssp, addr, bs, val);

	} else if (0x40020400 <= addr && addr < 0x40020800) {
		gpiob_st(cpssp, addr, bs, val);

	} else if (0x40020800 <= addr && addr < 0x40020c00) {
		gpioc_st(cpssp, addr, bs, val);

	} else if (0x40020c00 <= addr && addr < 0x40021000) {
		gpiod_st(cpssp, addr, bs, val);

	} else if (0x40021000 <= addr && addr < 0x40021400) {
		gpioe_st(cpssp, addr, bs, val);

	} else if (0x40021c00 <= addr && addr < 0x40022000) {
		gpioh_st(cpssp, addr, bs, val);

	} else if (0x40023800 <= addr && addr < 0x40023c00) {
		rcc_st(cpssp, addr, bs, val);

	} else if (0x40023c00 <= addr && addr < 0x40024000) {
		flashint_st(cpssp, addr, bs, val);

	} else if (0x50000000 <= addr && addr < 0x50040000) {
		usb_st(cpssp, addr, bs, val);

	} else if (0xe0000000 <= addr && addr < 0xe0100000) {
		CHIP_(ppb_st)(cpssp, addr, bs, val);

	} else {
		fprintf(stderr, "ERROR: %s: addr=0x%08x (pc=0x%08x)\n",
				__FUNCTION__, addr, cpssp->cpu.pc);
		assert(0);
	}
}

static void
cpu_mr(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	/*
	 * 2.3 Memory map, 37/836
	 */
	if (0x00000000 <= addr && addr < 0x00080000) {
		/* Flash/SRAM1/System memory (aliased) */
		goto flash; /* FIXME */

	} else if (0x08000000 <= addr && addr < 0x08080000) {
	flash:	;
		flash_read(cpssp, addr, valp);

	} else if (0x20000000 <= addr && addr < 0x20020000) {
		sram1_ld(cpssp, addr, bs, valp);

	} else if (0x40000000 <= addr && addr < 0x40000400) {
		tim2_ld(cpssp, addr, bs, valp);

	} else if (0x40000400 <= addr && addr < 0x40000800) {
		tim3_ld(cpssp, addr, bs, valp);

	} else if (0x40000800 <= addr && addr < 0x40000c00) {
		tim4_ld(cpssp, addr, bs, valp);

	} else if (0x40000c00 <= addr && addr < 0x40001000) {
		tim5_ld(cpssp, addr, bs, valp);

	} else if (0x40005400 <= addr && addr < 0x40005800) {
		i2c1_ld(cpssp, addr, bs, valp);

	} else if (0x40005800 <= addr && addr < 0x40005c00) {
		i2c2_ld(cpssp, addr, bs, valp);

	} else if (0x40005c00 <= addr && addr < 0x40006000) {
		i2c3_ld(cpssp, addr, bs, valp);

	} else if (0x40007000 <= addr && addr < 0x40007400) {
		pwr_ld(cpssp, addr, bs, valp);

	} else if (0x40010000 <= addr && addr < 0x40010400) {
		tim1_ld(cpssp, addr, bs, valp);

	} else if (0x40013800 <= addr && addr < 0x40013c00) {
		syscfg_ld(cpssp, addr, bs, valp);

	} else if (0x40013c00 <= addr && addr < 0x40014000) {
		exti_ld(cpssp, addr, bs, valp);

	} else if (0x40014000 <= addr && addr < 0x40014400) {
		tim9_ld(cpssp, addr, bs, valp);

	} else if (0x40014400 <= addr && addr < 0x40014800) {
		tim10_ld(cpssp, addr, bs, valp);

	} else if (0x40014800 <= addr && addr < 0x40014c00) {
		tim11_ld(cpssp, addr, bs, valp);

	} else if (0x40020000 <= addr && addr < 0x40020400) {
		gpioa_ld(cpssp, addr, bs, valp);

	} else if (0x40020400 <= addr && addr < 0x40020800) {
		gpiob_ld(cpssp, addr, bs, valp);

	} else if (0x40020800 <= addr && addr < 0x40020c00) {
		gpioc_ld(cpssp, addr, bs, valp);

	} else if (0x40020c00 <= addr && addr < 0x40021000) {
		gpiod_ld(cpssp, addr, bs, valp);

	} else if (0x40021000 <= addr && addr < 0x40021400) {
		gpioe_ld(cpssp, addr, bs, valp);

	} else if (0x40021c00 <= addr && addr < 0x40022000) {
		gpioh_ld(cpssp, addr, bs, valp);

	} else if (0x40023800 <= addr && addr < 0x40023c00) {
		rcc_ld(cpssp, addr, bs, valp);

	} else if (0x40023c00 <= addr && addr < 0x40024000) {
		flashint_ld(cpssp, addr, bs, valp);

	} else if (0x50000000 <= addr && addr < 0x50040000) {
		usb_ld(cpssp, addr, bs, valp);

	} else if (0xe0000000 <= addr && addr < 0xe0100000) {
		CHIP_(ppb_ld)(cpssp, addr, bs, valp);

	} else if (0xf0000000 <= addr) {
		/* Might happen when loading exception return vector. */
		*valp = 0;

	} else {
		fprintf(stderr, "ERROR: %s: addr=0x%08x (pc=0x%08x)\n",
				__FUNCTION__, addr, cpssp->cpu.pc);
		assert(0);
	}

#if 0
	fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
			__FUNCTION__, addr, bs, *valp);
#endif
}

static void
cpu_mx(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	cpu_mr(cpssp, addr, bs, valp);
}

static void
cpu_gpio_st(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
}

static void
cpu_gpio_ld(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	*valp = 0; /* FIXME */
}

static void
cpu_stop(struct cpssp *cpssp, int deepflag)
{
}

/*
 * Callbacks for nvic.
 */
static void
nvic_irq_set(struct cpssp *cpssp, unsigned int val)
{
	cpu_irq_set(cpssp, val);
}

/*
 * Callbacks for systick.
 */
static void
systick_irq_set(struct cpssp *cpssp)
{
	nvic_systick_pending_set(cpssp, 1);
}

/*
 * Callback for matrix.
 */
static void
matrix_portaNM_out_set(
	struct cpssp *cpssp,
	int alt,
	int pin,
	unsigned int val
)
{
	gpioa_leftNM_in_set(cpssp, pin, alt, val);
}

static void
matrix_portbNM_out_set(
	struct cpssp *cpssp,
	int alt,
	int pin,
	unsigned int val
)
{
	gpiob_leftNM_in_set(cpssp, pin, alt, val);
}

static void
matrix_portcNM_out_set(
	struct cpssp *cpssp,
	int alt,
	int pin,
	unsigned int val
)
{
	gpioc_leftNM_in_set(cpssp, pin, alt, val);
}

static void
matrix_portdNM_out_set(
	struct cpssp *cpssp,
	int alt,
	int pin,
	unsigned int val
)
{
	gpiod_leftNM_in_set(cpssp, pin, alt, val);
}

static void
matrix_porteNM_out_set(
	struct cpssp *cpssp,
	int alt,
	int pin,
	unsigned int val
)
{
	gpioe_leftNM_in_set(cpssp, pin, alt, val);
}

static void
matrix_porthNM_out_set(
	struct cpssp *cpssp,
	int alt,
	int pin,
	unsigned int val
)
{
	gpioh_leftNM_in_set(cpssp, pin, alt, val);
}

/*
 * Callbacks for exti.
 */
static void
exti_irqN_set(struct cpssp *cpssp, int n, int val)
{
	fprintf(stderr, "%s: n=%d val=%d\n", __FUNCTION__, n, val);

	switch (n) {
	case 0:
		nvic_irq_N_set(cpssp, 6, val);
		break;
	case 1:
		nvic_irq_N_set(cpssp, 7, val);
		break;
	case 2:
		nvic_irq_N_set(cpssp, 8, val);
		break;
	case 3:
		nvic_irq_N_set(cpssp, 9, val);
		break;
	case 4:
		nvic_irq_N_set(cpssp, 10, val);
		break;
	case 5 ... 9:
		/* nvic_irq_N_set(cpssp, 23, val); FIXME */
		break;
	case 10 ... 15:
		/* nvic_irq_N_set(cpssp, 40, val); FIXME */
		break;
	case 16:
		nvic_irq_N_set(cpssp, 1, val);
		break;
	case 17:
		nvic_irq_N_set(cpssp, 41, val);
		break;
	case 18:
		nvic_irq_N_set(cpssp, 42, val);
		break;
	case 19 ... 20:
		/* Not used. */
		break;
	case 21:
		nvic_irq_N_set(cpssp, 2, val);
		break;
	case 22:
		nvic_irq_N_set(cpssp, 3, val);
		break;
	default:
		/* Not used. */
		assert(0);
	}
}

/*
 * Callbacks for gpioa.
 */
static void
gpioa_rightN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(cpssp->port_pa[n]);

	sig_std_logic_set(cpssp->port_pa[n], cpssp, val);
}

static void
gpioa_aN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	/* FIXME */
}

static void
gpioa_leftNM_out_set(struct cpssp *cpssp, int n, int alt, unsigned int val)
{
	matrix_portaNM_in_set(cpssp, n, alt, val);
}

/*
 * Callbacks for gpiob.
 */
static void
gpiob_rightN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	if (cpssp->port_pb[n]) {
		sig_std_logic_set(cpssp->port_pb[n], cpssp, val);
	} else {
		assert(n == 11);
	}
}

static void
gpiob_aN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	/* FIXME */
}

static void
gpiob_leftNM_out_set(struct cpssp *cpssp, int n, int alt, unsigned int val)
{
	matrix_portbNM_in_set(cpssp, n, alt, val);
}

/*
 * Callbacks for gpioc.
 */
static void
gpioc_rightN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(cpssp->port_pc[n]);

	sig_std_logic_set(cpssp->port_pc[n], cpssp, val);
}

static void
gpioc_aN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	/* FIXME */
}

static void
gpioc_leftNM_out_set(struct cpssp *cpssp, int n, int alt, unsigned int val)
{
	matrix_portcNM_in_set(cpssp, n, alt, val);
}

/*
 * Callbacks for gpiod.
 */
static void
gpiod_rightN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(cpssp->port_pd[n]);

	sig_std_logic_set(cpssp->port_pd[n], cpssp, val);
}

static void
gpiod_aN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	/* FIXME */
}

static void
gpiod_leftNM_out_set(struct cpssp *cpssp, int n, int alt, unsigned int val)
{
	matrix_portdNM_in_set(cpssp, n, alt, val);
}

/*
 * Callbacks for gpioe.
 */
static void
gpioe_rightN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(cpssp->port_pe[n]);

	sig_std_logic_set(cpssp->port_pe[n], cpssp, val);
}

static void
gpioe_aN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	/* FIXME */
}

static void
gpioe_leftNM_out_set(struct cpssp *cpssp, int n, int alt, unsigned int val)
{
	matrix_porteNM_in_set(cpssp, n, alt, val);
}

/*
 * Callbacks for gpioh.
 */
static void
gpioh_rightN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	if (cpssp->port_ph[n]) {
		sig_std_logic_set(cpssp->port_ph[n], cpssp, val);
	}
}

static void
gpioh_aN_out(struct cpssp *cpssp, int n, unsigned int val)
{
	/* FIXME */
}

static void
gpioh_leftNM_out_set(struct cpssp *cpssp, int n, int alt, unsigned int val)
{
	matrix_porthNM_in_set(cpssp, n, alt, val);
}

/*
 * Callbacks for syscfg.
 */
static void
syscfg_outN_set(struct cpssp *cpssp, int pin, unsigned int val)
{
	exti_inN_set(cpssp, pin, val);
}

/*
 * Callbacks for tim1.
 */
static void
tim1_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* None... */
}

/*
 * Callbacks for tim2.
 */
static void
tim2_irq_set(struct cpssp *cpssp, unsigned int val)
{
	nvic_irq_N_set(cpssp, 28, val);
}

/*
 * Callbacks for tim3.
 */
static void
tim3_irq_set(struct cpssp *cpssp, unsigned int val)
{
	nvic_irq_N_set(cpssp, 29, val);
}

/*
 * Callbacks for tim4.
 */
static void
tim4_irq_set(struct cpssp *cpssp, unsigned int val)
{
	nvic_irq_N_set(cpssp, 30, val);
}

/*
 * Callbacks for tim5.
 */
static void
tim5_irq_set(struct cpssp *cpssp, unsigned int val)
{
	nvic_irq_N_set(cpssp, 50, val);
}

/*
 * Callbacks for tim9.
 */
static void
tim9_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* None... */
}

/*
 * Callbacks for tim10.
 */
static void
tim10_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* None... */
}

/*
 * Callbacks for tim11.
 */
static void
tim11_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* None... */
}

#define BEHAVIOR

#define NAME		matrix
#define NAME_(x)	matrix_ ## x
#include "chip_st_stm32f411xe_matrix.c"
#undef NAME_
#undef NAME

#define NAME		cpu
#define NAME_(x)	cpu_ ## x
#define CONFIG_DEBUG	0
#define CONFIG_VERSION	7
#define CONFIG_ARM	0
#define CONFIG_THUMB	1
#define CONFIG_E	1
#define CONFIG_M	1
#include "arch_arm_cpu.c"
#undef CONFIG_M
#undef CONFIG_E
#undef CONFIG_THUMB
#undef CONFIG_ARM
#undef CONFIG_VERSION
#undef CONFIG_DEBUG
#undef NAME_
#undef NAME

#define NAME		cp10
#define NAME_(x)	cp10_ ## x
#include "arch_arm_cp10.c"
#undef NAME_
#undef NAME

#define NAME		cp11
#define NAME_(x)	cp11_ ## x
#include "arch_arm_cp11.c"
#undef NAME_
#undef NAME

#define NAME		nvic
#define NAME_(x)	nvic_ ## x
#define CONFIG_INTLINESNUM	96
#include "arch_arm_nvic.c"
#undef CONFIG_INTLINESNUM
#undef NAME_
#undef NAME

#define NAME		systick
#define NAME_(x)	systick_ ## x
#include "arch_arm_systick.c"
#undef NAME_
#undef NAME

#define NAME		flash
#define NAME_(x)	flash_ ## x
#define FLASH_SIZE	(512*1024)
#include "arch_gen_flash.c"
#undef FLASH_SIZE
#undef NAME_
#undef NAME

#define NAME		sram1
#define NAME_(x)	sram1_ ## x
#define SRAM_SIZE	(128*1024)
#include "arch_gen_sram.c"
#undef SRAM_SIZE
#undef NAME_
#undef NAME

#define NAME		exti
#define NAME_(x)	exti_ ## x
#include "arch_st_exti.c"
#undef NAME_
#undef NAME

#define NAME		flashint
#define NAME_(x)	flashint_ ## x
#include "arch_st_flashint.c"
#undef NAME_
#undef NAME

#define NAME		gpioa
#define NAME_(x)	gpioa_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpiob
#define NAME_(x)	gpiob_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpioc
#define NAME_(x)	gpioc_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpiod
#define NAME_(x)	gpiod_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpioe
#define NAME_(x)	gpioe_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		gpioh
#define NAME_(x)	gpioh_ ## x
#include "arch_st_gpio.c"
#undef NAME_
#undef NAME

#define NAME		i2c1
#define NAME_(x)	i2c1_ ## x
#include "arch_st_i2c.c"
#undef NAME_
#undef NAME

#define NAME		i2c2
#define NAME_(x)	i2c2_ ## x
#include "arch_st_i2c.c"
#undef NAME_
#undef NAME

#define NAME		i2c3
#define NAME_(x)	i2c3_ ## x
#include "arch_st_i2c.c"
#undef NAME_
#undef NAME

#define NAME		pwr
#define NAME_(x)	pwr_ ## x
#include "arch_st_pwr.c"
#undef NAME_
#undef NAME

#define NAME		rcc
#define NAME_(x)	rcc_ ## x
#include "arch_st_rcc.c"
#undef NAME_
#undef NAME

#define NAME		syscfg
#define NAME_(x)	syscfg_ ## x
#include "arch_st_syscfg.c"
#undef NAME_
#undef NAME

#define NAME		tim1
#define NAME_(x)	tim1_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim2
#define NAME_(x)	tim2_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim3
#define NAME_(x)	tim3_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim4
#define NAME_(x)	tim4_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim5
#define NAME_(x)	tim5_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim9
#define NAME_(x)	tim9_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim10
#define NAME_(x)	tim10_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		tim11
#define NAME_(x)	tim11_ ## x
#include "arch_st_tim.c"
#undef NAME_
#undef NAME

#define NAME		usb
#define NAME_(x)	usb_ ## x
#include "arch_st_usb.c"
#undef NAME_
#undef NAME

#undef BEHAVIOR

static void __attribute__((__noreturn__))
CHIP_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:	;
	while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
		cpu_clk(cpssp); /* FIXME */
		systick_clk(cpssp); /* FIXME */
		i2c1_clk(cpssp); /* FIXME */
		i2c2_clk(cpssp); /* FIXME */
		i2c3_clk(cpssp); /* FIXME */
		tim1_tick(cpssp); /* FIXME */
		tim2_tick(cpssp); /* FIXME */
		tim3_tick(cpssp); /* FIXME */
		tim4_tick(cpssp); /* FIXME */
		tim5_tick(cpssp); /* FIXME */
		tim9_tick(cpssp); /* FIXME */
		tim10_tick(cpssp); /* FIXME */
		tim11_tick(cpssp); /* FIXME */
		cpssp->process.inst_cnt++;
	}

	sched_to_scheduler();
	goto again;
}

static void
CHIP_(gnd_set)(void *_cpssp, unsigned int val)
{
}

static void
CHIP_(vcc_set)(void *_cpssp, unsigned int val)
{
}

static void
CHIP_(gpioa_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	gpioa_rightN_in_set(cpssp, n, val);
	syscfg_inNM_set(cpssp, 'a' - 'a', n, val);
}

static void
CHIP_(gpiob_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	gpiob_rightN_in_set(cpssp, n, val);
	syscfg_inNM_set(cpssp, 'b' - 'a', n, val);
}

static void
CHIP_(gpioc_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	gpioc_rightN_in_set(cpssp, n, val);
	syscfg_inNM_set(cpssp, 'c' - 'a', n, val);
}

static void
CHIP_(gpiod_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	gpiod_rightN_in_set(cpssp, n, val);
	syscfg_inNM_set(cpssp, 'd' - 'a', n, val);
}

static void
CHIP_(gpioe_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	gpioe_rightN_in_set(cpssp, n, val);
	syscfg_inNM_set(cpssp, 'e' - 'a', n, val);
}

static void
CHIP_(gpioh_set)(void *_cpssp, int n, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	gpioh_rightN_in_set(cpssp, n, val);
	syscfg_inNM_set(cpssp, 'h' - 'a', n, val);
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_std_logic *port_vss0,
	struct sig_std_logic *port_vss1,
	struct sig_std_logic *port_vss2,
	struct sig_std_logic *port_vss3,
	struct sig_std_logic *port_vss4,
	struct sig_std_logic *port_vdd0,
	struct sig_std_logic *port_vdd1,
	struct sig_std_logic *port_vdd2,
	struct sig_std_logic *port_vdd3,
	struct sig_std_logic *port_vdd4,
	struct sig_std_logic *port_vdd5,
	struct sig_std_logic *port_vrefXminusX,
	struct sig_std_logic *port_vrefXplusX,
	struct sig_std_logic *port_vdda,
	struct sig_std_logic *port_pa0,
	struct sig_std_logic *port_pa1,
	struct sig_std_logic *port_pa2,
	struct sig_std_logic *port_pa3,
	struct sig_std_logic *port_pa4,
	struct sig_std_logic *port_pa5,
	struct sig_std_logic *port_pa6,
	struct sig_std_logic *port_pa7,
	struct sig_std_logic *port_pa8,
	struct sig_std_logic *port_pa9,
	struct sig_std_logic *port_pa10,
	struct sig_std_logic *port_pa11,
	struct sig_std_logic *port_pa12,
	struct sig_std_logic *port_pa13,
	struct sig_std_logic *port_pa14,
	struct sig_std_logic *port_pa15,
	struct sig_std_logic *port_pb0,
	struct sig_std_logic *port_pb1,
	struct sig_std_logic *port_pb2,
	struct sig_std_logic *port_pb3,
	struct sig_std_logic *port_pb4,
	struct sig_std_logic *port_pb5,
	struct sig_std_logic *port_pb6,
	struct sig_std_logic *port_pb7,
	struct sig_std_logic *port_pb8,
	struct sig_std_logic *port_pb9,
	struct sig_std_logic *port_pb10,
	struct sig_std_logic *port_pb12,
	struct sig_std_logic *port_pb13,
	struct sig_std_logic *port_pb14,
	struct sig_std_logic *port_pb15,
	struct sig_std_logic *port_pc0,
	struct sig_std_logic *port_pc1,
	struct sig_std_logic *port_pc2,
	struct sig_std_logic *port_pc3,
	struct sig_std_logic *port_pc4,
	struct sig_std_logic *port_pc5,
	struct sig_std_logic *port_pc6,
	struct sig_std_logic *port_pc7,
	struct sig_std_logic *port_pc8,
	struct sig_std_logic *port_pc9,
	struct sig_std_logic *port_pc10,
	struct sig_std_logic *port_pc11,
	struct sig_std_logic *port_pc12,
	struct sig_std_logic *port_pc13,
	struct sig_std_logic *port_pc14,
	struct sig_std_logic *port_pc15,
	struct sig_std_logic *port_pd0,
	struct sig_std_logic *port_pd1,
	struct sig_std_logic *port_pd2,
	struct sig_std_logic *port_pd3,
	struct sig_std_logic *port_pd4,
	struct sig_std_logic *port_pd5,
	struct sig_std_logic *port_pd6,
	struct sig_std_logic *port_pd7,
	struct sig_std_logic *port_pd8,
	struct sig_std_logic *port_pd9,
	struct sig_std_logic *port_pd10,
	struct sig_std_logic *port_pd11,
	struct sig_std_logic *port_pd12,
	struct sig_std_logic *port_pd13,
	struct sig_std_logic *port_pd14,
	struct sig_std_logic *port_pd15,
	struct sig_std_logic *port_pe0,
	struct sig_std_logic *port_pe1,
	struct sig_std_logic *port_pe2,
	struct sig_std_logic *port_pe3,
	struct sig_std_logic *port_pe4,
	struct sig_std_logic *port_pe5,
	struct sig_std_logic *port_pe6,
	struct sig_std_logic *port_pe7,
	struct sig_std_logic *port_pe8,
	struct sig_std_logic *port_pe9,
	struct sig_std_logic *port_pe10,
	struct sig_std_logic *port_pe11,
	struct sig_std_logic *port_pe12,
	struct sig_std_logic *port_pe13,
	struct sig_std_logic *port_pe14,
	struct sig_std_logic *port_pe15,
	struct sig_std_logic *port_ph0,
	struct sig_std_logic *port_ph1,
	struct sig_std_logic *port_vbat,
	struct sig_std_logic *port_nrst,
	struct sig_std_logic *port_wcap1,
	struct sig_std_logic *port_vcap2,
	struct sig_std_logic *port_boot0
)
{
	static const struct sig_std_logic_funcs gnd_funcs = {
		.std_logic_set = CHIP_(gnd_set),
	};
	static const struct sig_std_logic_funcs vdd_funcs = {
		.std_logic_set = CHIP_(vcc_set),
	};
	static const struct sig_std_logic_funcs gpioa_funcs = {
		.set_extN = CHIP_(gpioa_set),
	};
	static const struct sig_std_logic_funcs gpiob_funcs = {
		.set_extN = CHIP_(gpiob_set),
	};
	static const struct sig_std_logic_funcs gpioc_funcs = {
		.set_extN = CHIP_(gpioc_set),
	};
	static const struct sig_std_logic_funcs gpiod_funcs = {
		.set_extN = CHIP_(gpiod_set),
	};
	static const struct sig_std_logic_funcs gpioe_funcs = {
		.set_extN = CHIP_(gpioe_set),
	};
	static const struct sig_std_logic_funcs gpioh_funcs = {
		.set_extN = CHIP_(gpioh_set),
	};
	struct cpssp *cpssp;
	unsigned int n;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* Create sub-architectures. */
	matrix_create(cpssp);
	cpu_create(name, cpssp);
	cp10_create(cpssp);
	cp11_create(cpssp);
	nvic_create(cpssp);
	systick_create(cpssp);
	flash_create(cpssp, name, "nvram");
	sram1_create(cpssp);
	exti_create(cpssp);
	flashint_create(cpssp);
	gpioa_create(cpssp);
	gpiob_create(cpssp);
	gpioc_create(cpssp);
	gpiod_create(cpssp);
	gpioe_create(cpssp);
	gpioh_create(cpssp);
	i2c1_create(cpssp);
	i2c2_create(cpssp);
	i2c3_create(cpssp);
	pwr_create(cpssp);
	rcc_create(cpssp);
	syscfg_create(cpssp);
	tim1_create(cpssp);
	tim2_create(cpssp);
	tim3_create(cpssp);
	tim4_create(cpssp);
	tim5_create(cpssp);
	tim9_create(cpssp);
	tim10_create(cpssp);
	tim11_create(cpssp);
	usb_create(cpssp);

	cpu_reset(cpssp); /* FIXME */
	cp10_reset(cpssp); /* FIXME */
	cp11_reset(cpssp); /* FIXME */
	nvic_reset(cpssp); /* FIXME */
	systick_reset(cpssp); /* FIXME */
	exti_reset(cpssp); /* FIXME */
	flashint_reset(cpssp); /* FIXME */
	gpioa_reset(cpssp); /* FIXME */
	gpiob_reset(cpssp); /* FIXME */
	gpioc_reset(cpssp); /* FIXME */
	gpiod_reset(cpssp); /* FIXME */
	gpioe_reset(cpssp); /* FIXME */
	gpioh_reset(cpssp); /* FIXME */
	i2c1_reset(cpssp); /* FIXME */
	i2c2_reset(cpssp); /* FIXME */
	i2c3_reset(cpssp); /* FIXME */
	pwr_reset(cpssp); /* FIXME */
	rcc_reset(cpssp); /* FIXME */
	syscfg_reset(cpssp); /* FIXME */
	tim1_reset(cpssp); /* FIXME */
	tim2_reset(cpssp); /* FIXME */
	tim3_reset(cpssp); /* FIXME */
	tim4_reset(cpssp); /* FIXME */
	tim5_reset(cpssp); /* FIXME */
	tim9_reset(cpssp); /* FIXME */
	tim10_reset(cpssp); /* FIXME */
	tim11_reset(cpssp); /* FIXME */
	usb_reset(cpssp); /* FIXME */

	cpssp->process.inst_hz = 8*1000*1000;
	sched_process_init(&cpssp->process, CHIP_(step), cpssp);

	/* Call */
	/* Out */

	/* In */
	cpssp->state_gnd = 0;
	sig_std_logic_connect_in(port_vss0, cpssp, &gnd_funcs);

	cpssp->state_vcc = 0;
	sig_std_logic_connect_in(port_vdd0, cpssp, &vdd_funcs);

#define conn(a, n) \
do { \
	cpssp->port_p ## a[n] = port_p ## a ## n; \
	sig_std_logic_connect_out(port_p ## a ## n, cpssp, SIG_STD_LOGIC_Z); \
	sig_std_logic_connect_inN(port_p ## a ## n, cpssp, n, &gpio ## a ## _funcs); \
} while (0)
	
	/* Port A */
	conn(a,  0); conn(a,  1); conn(a,  2); conn(a,  3);
	conn(a,  4); conn(a,  5); conn(a,  6); conn(a,  7);
	conn(a,  8); conn(a,  9); conn(a, 10); conn(a, 11);
	conn(a, 12); conn(a, 13); conn(a, 14); conn(a, 15);

	/* Port B */
	conn(b,  0); conn(b,  1); conn(b,  2); conn(b,  3);
	conn(b,  4); conn(b,  5); conn(b,  6); conn(b,  7);
	conn(b,  8); conn(b,  9); conn(b, 10); cpssp->port_pb[11] = NULL;
	conn(b, 12); conn(b, 13); conn(b, 14); conn(b, 15);

	/* Port C */
	conn(c,  0); conn(c,  1); conn(c,  2); conn(c,  3);
	conn(c,  4); conn(c,  5); conn(c,  6); conn(c,  7);
	conn(c,  8); conn(c,  9); conn(c, 10); conn(c, 11);
	conn(c, 12); conn(c, 13); conn(c, 14); conn(c, 15);

	/* Port D */
	conn(d,  0); conn(d,  1); conn(d,  2); conn(d,  3);
	conn(d,  4); conn(d,  5); conn(d,  6); conn(d,  7);
	conn(d,  8); conn(d,  9); conn(d, 10); conn(d, 11);
	conn(d, 12); conn(d, 13); conn(d, 14); conn(d, 15);

	/* Port E */
	conn(e,  0); conn(e,  1); conn(e,  2); conn(e,  3);
	conn(e,  4); conn(e,  5); conn(e,  6); conn(e,  7);
	conn(e,  8); conn(e,  9); conn(e, 10); conn(e, 11);
	conn(e, 12); conn(e, 13); conn(e, 14); conn(e, 15);

	/* Port H */
	conn(h,  0); conn(h,  1);
	for (n = 2; n < 16; n++) {
		cpssp->port_ph[n] = NULL;
	}

#undef conn

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	matrix_destroy(cpssp);
	cpu_destroy(cpssp);
	cp10_destroy(cpssp);
	cp11_destroy(cpssp);
	nvic_destroy(cpssp);
	systick_destroy(cpssp);
	flash_destroy(cpssp);
	sram1_destroy(cpssp);
	exti_destroy(cpssp);
	flashint_destroy(cpssp);
	gpioa_destroy(cpssp);
	gpiob_destroy(cpssp);
	gpioc_destroy(cpssp);
	gpiod_destroy(cpssp);
	gpioe_destroy(cpssp);
	gpioh_destroy(cpssp);
	i2c1_destroy(cpssp);
	i2c2_destroy(cpssp);
	i2c3_destroy(cpssp);
	pwr_destroy(cpssp);
	rcc_destroy(cpssp);
	syscfg_destroy(cpssp);
	tim1_destroy(cpssp);
	tim2_destroy(cpssp);
	tim3_destroy(cpssp);
	tim4_destroy(cpssp);
	tim5_destroy(cpssp);
	tim9_destroy(cpssp);
	tim10_destroy(cpssp);
	tim11_destroy(cpssp);
	usb_destroy(cpssp);

	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}
