/*
 * $Id: chip_sst_29EE020.c,v 1.52 2013-04-26 17:08:52 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define ROM_CACHE_SIZE	2

#include "config.h"

#define SZ_BIOS_ROM	(256 * 1024) /* ROM size */

#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>

#include "fixme.h"
#include "glue-main.h"
#include "glue-shm.h"
#include "glue-suspend.h"
#include "umutil.h"

#include "chip_sst_29EE020.h"

#define CHIP_(x) chip_sst_29EE020_ ## x

struct cpssp {
	struct sig_cs *port_cs;

	uint8_t *haddr;

	unsigned int state_power;
	int state_write;
	int state_write_once;
	int state_id;

	enum {
		BIOS_SEQ,
		BIOS_SEQ_AA,
		BIOS_SEQ_AA_55,
		BIOS_SEQ_AA_55_80,
		BIOS_SEQ_AA_55_80_AA,
		BIOS_SEQ_AA_55_80_AA_55,
	} state_seq;
};

static void *
CHIP_(access)(struct cpssp *cpssp, unsigned long pa)
{
	assert(/* 0 <= pa && */ pa < SZ_BIOS_ROM);

	pa &= 0xfffff000;

	return cpssp->haddr + pa;
}

static int
CHIP_(readb)(void *_css, uint32_t addr, uint8_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _css;
	uint8_t *va;

	addr &= SZ_BIOS_ROM - 1;

	assert(/* 0 <= pa && */ addr < SZ_BIOS_ROM);

	if (cpssp->state_id) {
		/*
		 * Read Manufacturer/Device ID
		 */
		switch (addr & 0x7fff) {
		case 0x0000:
			/* Manufacturer ID */
			*valp = 0xbf; /* SST */
			break;
		case 0x0001:
			/* Device ID */
			*valp = 0x10; /* SST29EE020 */
			break;
		case 0x0002:
		case 0x0003:
			/* Some bioses need this, dont know why... */
			/* 29EE020-doku says nothing about this...*/
			*valp = 0x00;
			break;
		default:
			assert(0); /* FIXME */
		}

	} else {
		/*
		 * Read Contents
		 */
		va = CHIP_(access)(cpssp, addr & 0xfffff000);
		*valp = *(va + (addr & 0x00000fff));
	}
	return 0;
}

static void
CHIP_(unmap)(struct cpssp *cpssp)
{
	sig_cs_unmap(cpssp->port_cs, cpssp, 0, SZ_BIOS_ROM);
}

static int
CHIP_(writeb)(void *_css, uint32_t addr, uint8_t val)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	switch (cpssp->state_seq) {
	case BIOS_SEQ:
		if ((addr & 0x7fff) != 0x5555) goto unknown;
		if (val != 0xaa) goto unknown;

		cpssp->state_seq = BIOS_SEQ_AA;
		break;

	case BIOS_SEQ_AA:
		if ((addr & 0x7fff) != 0x2aaa) goto unknown;
		if (val != 0x55) goto unknown;

		cpssp->state_seq = BIOS_SEQ_AA_55;
		break;

	case BIOS_SEQ_AA_55:
		if ((addr & 0x7fff) != 0x5555) goto unknown;
		switch (val) {
		case 0x80:
			cpssp->state_seq = BIOS_SEQ_AA_55_80;
			break;
		case 0x90:
			/* Software ID Entry */
			cpssp->state_id = 1;
			cpssp->state_seq = BIOS_SEQ;
			CHIP_(unmap)(cpssp);
			break;
		case 0xa0:
			/* Software Data Protect Enable & Page Write */
			cpssp->state_write = 0;
			cpssp->state_write_once = 1;
			cpssp->state_seq = BIOS_SEQ;
			break;
		case 0xf0:
			/* Software ID Exit */
			cpssp->state_id = 0;
			cpssp->state_seq = BIOS_SEQ;
			CHIP_(unmap)(cpssp);
			break;
		default:
			goto unknown;
		}
		break;

	case BIOS_SEQ_AA_55_80:
		if ((addr & 0x7fff) != 0x5555) goto unknown;
		if (val != 0xaa) goto unknown;

		cpssp->state_seq = BIOS_SEQ_AA_55_80_AA;
		break;

	case BIOS_SEQ_AA_55_80_AA:
		if ((addr & 0x7fff) != 0x2aaa) goto unknown;
		if (val != 0x55) goto unknown;

		cpssp->state_seq = BIOS_SEQ_AA_55_80_AA_55;
		break;

	case BIOS_SEQ_AA_55_80_AA_55:
		if ((addr & 0x7fff) != 0x5555) goto unknown;
		switch (val) {
		case 0x10:
			/* Software Chip-Erase */
			fixme();
			cpssp->state_seq = BIOS_SEQ;
			break;
		case 0x20:
			/* Software Data Protect Disable */
			cpssp->state_write = 1;
			cpssp->state_seq = BIOS_SEQ;
			break;
		case 0x60:
			/* Alternate Software ID Entry */
			cpssp->state_id = 1;
			cpssp->state_seq = BIOS_SEQ;
			CHIP_(unmap)(cpssp);
			break;
		default:
			assert(0); /* FIXME */
		}
		break;

	default:
		assert(0);

	unknown:;
		/* Unknown sequence. */
		cpssp->state_seq = BIOS_SEQ;
		break;
	}
	return 0;
}

static int
CHIP_(map_r)(
	void *_cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = _cpssp;

	*cfp = NULL;
	*csp = NULL;
	if (cpssp->state_id) {
		*haddr_p = NULL;
	} else {
		addr &= SZ_BIOS_ROM - 1;
		addr &= ~0xfff;
		*haddr_p = cpssp->haddr + addr;
	}

	return 0;
}

static int
CHIP_(map_w)(
	void *_cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp,
	char **haddr_p
)
{
	/* struct cpssp *cpssp = _cpssp; */

	*cfp = NULL;
	*csp = NULL;
	*haddr_p = NULL;

	return 0;
}

static void
CHIP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->state_power = val;
	/* FIXME */
}

void *
CHIP_(create)(
	const char *name,
	const char *img,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_power,
	struct sig_cs *port_cs,
	struct sig_isa_bus *port_bus
)
{
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = CHIP_(power_set),
	};
	static const struct sig_cs_funcs cs_funcs = {
		.readb = CHIP_(readb),
		.writeb = CHIP_(writeb),
		.map_r = CHIP_(map_r),
		.map_w = CHIP_(map_w),
	};
	struct cpssp *cpssp;
	const char *path;
	int fd;
	int ret;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);
	cpssp->haddr = shm_palloc(SZ_BIOS_ROM, 0);
	assert(cpssp->haddr);
	assert(! ((unsigned long) cpssp->haddr & 0xfff));

	cpssp->state_id = 0;
	cpssp->state_write = 0;
	cpssp->state_write_once = 0;
	cpssp->state_seq = BIOS_SEQ;

	path = buildpath(ROMDIR, img);
	fd = open(path, O_RDONLY);
	assert(0 <= fd);

	ret = read(fd, cpssp->haddr, SZ_BIOS_ROM);
	assert(ret == SZ_BIOS_ROM);

	ret = close(fd);
	assert(0 <= ret);

#if 0
	ret = mprotect(cpssp->haddr, SZ_BIOS_ROM, PROT_READ | PROT_EXEC);
	assert(0 <= ret);
#endif

	/* Call */
	cpssp->port_cs = port_cs;
	sig_cs_connect(port_cs, cpssp, &cs_funcs);

	/* Out */

	/* In */
	cpssp->state_power = 0;
	sig_std_logic_connect_in(port_power, cpssp, &power_funcs);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_pfree(cpssp->haddr);
	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
