/*
 * Copyright (C) 2003-2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_CONTROL_FLOW	0

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue.h"

#include "chip_intel_82093AA.h"

struct cpssp {
	struct sig_cs *port_cs;
	struct sig_isa_bus *port_isa_bus;
	struct sig_icc_bus *port_icc_bus;

	int state_power;
	int state_intin[24];

	unsigned int selectreg : 8;
	unsigned int id : 4;
	unsigned int arbitration : 4;
	struct ioapic_ioredtbl {
		unsigned int destination : 8;
		unsigned int mask : 1;
		unsigned int trigger_mode : 1;
		unsigned int remote_irr : 1;
		unsigned int polarity : 1;
		unsigned int delivery_status : 1;
		unsigned int destination_mode : 1;
		unsigned int delivery_mode : 3;
		unsigned int vector : 8;
	} ioredtbl[24];
};

static int
chip_intel_82093AA_readl(void *_ioa, uint32_t reg, uint32_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _ioa;
	struct ioapic_ioredtbl *ioredtbl;
	uint32_t val;

	switch (reg) {
	case 0x0:	/* I/O register select */
		val = cpssp->selectreg;
		break;

	case 0x1:	/* I/O window register */
		switch (cpssp->selectreg) {
		case 0x00: /* ID */
			val = 0;
			val |= cpssp->id << 24;
			break;
			
		case 0x01: /* Version */
			val = 0;
			val |= 0x17 << 16; /* # Redirection Entries - 1 */
			val |= 0x11 << 0; /* Version number */
			break;

		case 0x02: /* Arbitration ID */
			val = 0;
			val |= cpssp->arbitration << 24;
			break;

		case 0x03 ... 0x0f: /* Reserved */
			goto reserved;

		case 0x10 ... 0x3f: /* Redirection Table */
			ioredtbl = &cpssp->ioredtbl[(cpssp->selectreg - 0x10) >> 1];
			if (cpssp->selectreg & 1) {
				val = 0;
				val |= ioredtbl->destination << 24;

			} else {
				val = 0;
				val |= ioredtbl->mask << 16;
				val |= ioredtbl->trigger_mode << 15;
				val |= ioredtbl->remote_irr << 14;
				val |= ioredtbl->polarity << 13;
				val |= ioredtbl->delivery_status << 12;
				val |= ioredtbl->destination_mode << 11;
				val |= ioredtbl->delivery_mode << 8;
				val |= ioredtbl->vector << 0;
			}
			break;

		default:
		reserved:
			faum_log(FAUM_LOG_WARNING, "IOAPIC", "",
				"Reading from reserved register 0x%02x.\n",
				cpssp->selectreg);
			val = 0;
			break;
		}
		break;

	default:
		assert(0); /* FIXME VOSSI */
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "IOAPIC: Reading 0x%08lx from register %u/0x%02xx\n",
				(unsigned long) val, (unsigned int) reg,
				(unsigned int) cpssp->selectreg);
	}

	*valp = val;

	return 0;
}

static int
chip_intel_82093AA_writel(void *_ioa, uint32_t reg, uint32_t val)
{
	struct cpssp *cpssp = (struct cpssp *) _ioa;
	struct ioapic_ioredtbl *ioredtbl;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "IOAPIC: Writing 0x%08lx to register %u/0x%02x\n",
				(unsigned long) val, (unsigned int) reg,
				(unsigned int) cpssp->selectreg);
	}

	switch (reg) {
	case 0x0:	/* I/O register select */
		cpssp->selectreg	= val;
		break;

	case 0x1:	/* I/O window register */
		switch (cpssp->selectreg) {
		case 0x00: /* ID */
			cpssp->id = (val >> 24) & 0xf;
			break;

		case 0x01: /* Version */
			/* Read-only */
			goto read_only;

		case 0x02:
		read_only:
			faum_log(FAUM_LOG_WARNING, "IOAPIC", "",
				"Writing to read-only register 0x%02x\n",
				cpssp->selectreg);
			break;

		case 0x03 ... 0x0f: /* Reserved */
			goto reserved;

		case 0x10 ... 0x3f: /* Redirection Table */
			ioredtbl = &cpssp->ioredtbl[(cpssp->selectreg - 0x10) >> 1];
			if (cpssp->selectreg & 1) {
				ioredtbl->destination = (val >> 24) & 0xff;

			} else {
				ioredtbl->mask = (val >> 16) & 1;
				ioredtbl->trigger_mode = (val >> 15) & 1;
				ioredtbl->remote_irr = (val >> 14) & 1;
				ioredtbl->polarity = (val >> 13) & 1;
				/* ioredtbl->delivery_status: read-only */
				ioredtbl->destination_mode = (val >> 11) & 1;
				ioredtbl->delivery_mode = (val >> 8) & 0x7;
				ioredtbl->vector = (val >> 0) & 0xff;
			}
			break;

		default:
		reserved:
			faum_log(FAUM_LOG_WARNING, "IOAPIC", "",
				"Writing to reserved register 0x%02x.\n",
				cpssp->selectreg);
			break;
		}
		break;

	default:
		assert(0); /* FIXME VOSSI */
	}

	return 0;
}

static void
chip_intel_82093AA_irq_set(void *c, unsigned int irq, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) c;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: irq %d set to %d\n",
				__FUNCTION__, irq, val);
	}

	assert(/* 0 <= irq && */ irq < 24);
	assert(/* 0 <= val && */ val < 2);

	cpssp->state_intin[irq] = val;

	if (! cpssp->state_power) {
		return;
	}

	val ^= cpssp->ioredtbl[irq].polarity;
	val &= ~cpssp->ioredtbl[irq].mask;

	if (! val) {
		return;
	}

	if (cpssp->ioredtbl[irq].trigger_mode) {
		cpssp->ioredtbl[irq].remote_irr = 1;
	}
	sig_icc_bus_msg0(cpssp->port_icc_bus, cpssp,
			cpssp->ioredtbl[irq].destination_mode,
			cpssp->ioredtbl[irq].delivery_mode,
			1, /* Level */
			cpssp->ioredtbl[irq].trigger_mode,
			cpssp->ioredtbl[irq].vector,
			cpssp->ioredtbl[irq].destination);
}

static void
chip_intel_82093AA_irq00_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x00, val);
}

static void
chip_intel_82093AA_irq01_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x01, val);
}

static void
chip_intel_82093AA_irq02_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x02, val);
}

static void
chip_intel_82093AA_irq03_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x03, val);
}

static void
chip_intel_82093AA_irq04_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x04, val);
}

static void
chip_intel_82093AA_irq05_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x05, val);
}

static void
chip_intel_82093AA_irq06_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x06, val);
}

static void
chip_intel_82093AA_irq07_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x07, val);
}

static void
chip_intel_82093AA_irq08_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x08, val);
}

static void
chip_intel_82093AA_irq09_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x09, val);
}

static void
chip_intel_82093AA_irq0A_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x0A, val);
}

static void
chip_intel_82093AA_irq0B_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x0B, val);
}

static void
chip_intel_82093AA_irq0C_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x0C, val);
}

static void
chip_intel_82093AA_irq0D_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x0D, val);
}

static void
chip_intel_82093AA_irq0E_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x0E, val);
}

static void
chip_intel_82093AA_irq0F_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x0F, val);
}

static void
chip_intel_82093AA_irq10_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x10, val);
}

static void
chip_intel_82093AA_irq11_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x11, val);
}

static void
chip_intel_82093AA_irq12_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x12, val);
}

static void
chip_intel_82093AA_irq13_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x13, val);
}

static void
chip_intel_82093AA_irq14_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x14, val);
}

static void
chip_intel_82093AA_irq15_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x15, val);
}

static void
chip_intel_82093AA_irq16_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x16, val);
}

static void
chip_intel_82093AA_irq17_set(void *c, unsigned int val)
{
	chip_intel_82093AA_irq_set(c, 0x17, val);
}

static void
chip_intel_82093AA_eoi_receive(void *c, uint8_t vector)
{
	struct cpssp *cpssp = (struct cpssp *) c;
	unsigned int irq;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: got eoi on vector 0x%02x.\n",
				__FUNCTION__, vector);
	}

	for (irq = 0; ; irq++) {
		if (irq == 24) {
			/* Vector not found. */
			return;
		}
		if (cpssp->ioredtbl[irq].vector == vector) {
			/* Vector found. */
			break;
		}
	}

	/* Re-set remote IRR bit. */
	cpssp->ioredtbl[irq].remote_irr = 0;

	if (! cpssp->state_intin[irq]) {
		/* Interrupt not pending any more. */
		return;
	}
	if (! cpssp->ioredtbl[irq].trigger_mode) {
		/* Not in level triggered mode any more. */
		return;
	}

	/* Re-assert interrupt. */
	cpssp->ioredtbl[irq].remote_irr = 1;

	sig_icc_bus_msg0(cpssp->port_icc_bus, cpssp,
			cpssp->ioredtbl[irq].destination_mode,
			cpssp->ioredtbl[irq].delivery_mode,
			1, /* Level */
			cpssp->ioredtbl[irq].trigger_mode,
			cpssp->ioredtbl[irq].vector,
			cpssp->ioredtbl[irq].destination);
}

static void
chip_intel_82093AA_msg0_receive(
	void *c,
	unsigned int destination_mode,
	unsigned int delivery_mode,
	unsigned int level,
	unsigned int trigger_mode,
	uint8_t vector,
	uint8_t destination
)
{
	// fixme();
}

static void
chip_intel_82093AA_status0_receive(void *c, unsigned int status)
{
	// fixme();
}

static void
chip_intel_82093AA_msg1_receive(void *c, uint8_t processor_priority)
{
	// fixme();
}

static void
chip_intel_82093AA_status1_receive(void *c, unsigned int status)
{
	// fixme();
}

static void
chip_intel_82093AA_power_set(void *_css, unsigned int val)
{
	struct cpssp *css = (struct cpssp *) _css;

	css->state_power = val;
}

static void
chip_intel_82093AA_n_reset_set(void *_css, unsigned int n_val)
{
	struct cpssp *css = (struct cpssp *) _css;
	unsigned int irq;

	css->id = 8; /* FIXME */
	css->arbitration = 0x0;
	for (irq = 0; irq < 24; irq++) {
		css->ioredtbl[irq].destination = 0x00;
		css->ioredtbl[irq].mask = 1;
		css->ioredtbl[irq].trigger_mode = 0;
		css->ioredtbl[irq].remote_irr = 0;
		css->ioredtbl[irq].polarity = 0;
		css->ioredtbl[irq].delivery_status = 0;
		css->ioredtbl[irq].destination_mode = 0;
		css->ioredtbl[irq].delivery_mode = 0x0;
		css->ioredtbl[irq].vector = 0x00;
	}
}

void *
chip_intel_82093AA_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_power,
	struct sig_std_logic *port_reset_hash_,
	struct sig_cs *port_cs,
	struct sig_icc_bus *port_icc_bus,
	struct sig_isa_bus *port_isa_bus,
	struct sig_std_logic *port_intin00,
	struct sig_std_logic *port_intin01,
	struct sig_std_logic *port_intin02,
	struct sig_std_logic *port_intin03,
	struct sig_std_logic *port_intin04,
	struct sig_std_logic *port_intin05,
	struct sig_std_logic *port_intin06,
	struct sig_std_logic *port_intin07,
	struct sig_std_logic *port_intin08,
	struct sig_std_logic *port_intin09,
	struct sig_std_logic *port_intin10,
	struct sig_std_logic *port_intin11,
	struct sig_std_logic *port_intin12,
	struct sig_std_logic *port_intin13,
	struct sig_std_logic *port_intin14,
	struct sig_std_logic *port_intin15,
	struct sig_std_logic *port_intin16,
	struct sig_std_logic *port_intin17,
	struct sig_std_logic *port_intin18,
	struct sig_std_logic *port_intin19,
	struct sig_std_logic *port_intin20,
	struct sig_std_logic *port_intin21,
	struct sig_std_logic *port_intin22,
	struct sig_std_logic *port_intin23
)
{
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = chip_intel_82093AA_power_set,
	};
	static const struct sig_std_logic_funcs reset_hash__funcs = {
		.boolean_or_set = chip_intel_82093AA_n_reset_set,
	};
	static const struct sig_cs_funcs cs_funcs = {
		.readl = chip_intel_82093AA_readl,
		.writel = chip_intel_82093AA_writel,
	};
	static const struct sig_icc_bus_funcs icc_bus_funcs = {
		.eoi = chip_intel_82093AA_eoi_receive,
		.msg0 = chip_intel_82093AA_msg0_receive,
		.status0 = chip_intel_82093AA_status0_receive,
		.msg1 = chip_intel_82093AA_msg1_receive,
		.status1 = chip_intel_82093AA_status1_receive,
	};
	static const struct sig_std_logic_funcs intin00_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq00_set,
	};
	static const struct sig_std_logic_funcs intin01_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq01_set,
	};
	static const struct sig_std_logic_funcs intin02_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq02_set,
	};
	static const struct sig_std_logic_funcs intin03_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq03_set,
	};
	static const struct sig_std_logic_funcs intin04_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq04_set,
	};
	static const struct sig_std_logic_funcs intin05_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq05_set,
	};
	static const struct sig_std_logic_funcs intin06_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq06_set,
	};
	static const struct sig_std_logic_funcs intin07_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq07_set,
	};
	static const struct sig_std_logic_funcs intin08_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq08_set,
	};
	static const struct sig_std_logic_funcs intin09_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq09_set,
	};
	static const struct sig_std_logic_funcs intin10_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq0A_set,
	};
	static const struct sig_std_logic_funcs intin11_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq0B_set,
	};
	static const struct sig_std_logic_funcs intin12_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq0C_set,
	};
	static const struct sig_std_logic_funcs intin13_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq0D_set,
	};
	static const struct sig_std_logic_funcs intin14_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq0E_set,
	};
	static const struct sig_std_logic_funcs intin15_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq0F_set,
	};
	static const struct sig_std_logic_funcs intin16_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq10_set,
	};
	static const struct sig_std_logic_funcs intin17_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq11_set,
	};
	static const struct sig_std_logic_funcs intin18_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq12_set,
	};
	static const struct sig_std_logic_funcs intin19_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq13_set,
	};
	static const struct sig_std_logic_funcs intin20_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq14_set,
	};
	static const struct sig_std_logic_funcs intin21_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq15_set,
	};
	static const struct sig_std_logic_funcs intin22_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq16_set,
	};
	static const struct sig_std_logic_funcs intin23_funcs = {
		.boolean_or_set = chip_intel_82093AA_irq17_set,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/* Out */
	/* Call */
	cpssp->port_cs = port_cs;
	sig_cs_connect(port_cs, cpssp, &cs_funcs);

	cpssp->port_icc_bus = port_icc_bus;
	sig_icc_bus_connect(port_icc_bus, cpssp, &icc_bus_funcs);

	/* In */
	sig_std_logic_connect_in(port_power, cpssp, &power_funcs);

	sig_std_logic_connect_in(port_reset_hash_, cpssp, &reset_hash__funcs);

	cpssp->state_intin[0] = 0;
	sig_std_logic_connect_in(port_intin00, cpssp, &intin00_funcs);

	cpssp->state_intin[1] = 0;
	sig_std_logic_connect_in(port_intin01, cpssp, &intin01_funcs);

	cpssp->state_intin[2] = 0;
	sig_std_logic_connect_in(port_intin02, cpssp, &intin02_funcs);

	cpssp->state_intin[3] = 0;
	sig_std_logic_connect_in(port_intin03, cpssp, &intin03_funcs);

	cpssp->state_intin[4] = 0;
	sig_std_logic_connect_in(port_intin04, cpssp, &intin04_funcs);

	cpssp->state_intin[5] = 0;
	sig_std_logic_connect_in(port_intin05, cpssp, &intin05_funcs);

	cpssp->state_intin[6] = 0;
	sig_std_logic_connect_in(port_intin06, cpssp, &intin06_funcs);

	cpssp->state_intin[7] = 0;
	sig_std_logic_connect_in(port_intin07, cpssp, &intin07_funcs);

	cpssp->state_intin[8] = 0;
	sig_std_logic_connect_in(port_intin08, cpssp, &intin08_funcs);

	cpssp->state_intin[9] = 0;
	sig_std_logic_connect_in(port_intin09, cpssp, &intin09_funcs);

	cpssp->state_intin[10] = 0;
	sig_std_logic_connect_in(port_intin10, cpssp, &intin10_funcs);

	cpssp->state_intin[11] = 0;
	sig_std_logic_connect_in(port_intin11, cpssp, &intin11_funcs);

	cpssp->state_intin[12] = 0;
	sig_std_logic_connect_in(port_intin12, cpssp, &intin12_funcs);

	cpssp->state_intin[13] = 0;
	sig_std_logic_connect_in(port_intin13, cpssp, &intin13_funcs);

	cpssp->state_intin[14] = 0;
	sig_std_logic_connect_in(port_intin14, cpssp, &intin14_funcs);

	cpssp->state_intin[15] = 0;
	sig_std_logic_connect_in(port_intin15, cpssp, &intin15_funcs);

	cpssp->state_intin[16] = 0;
	sig_std_logic_connect_in(port_intin16, cpssp, &intin16_funcs);

	cpssp->state_intin[17] = 0;
	sig_std_logic_connect_in(port_intin17, cpssp, &intin17_funcs);

	cpssp->state_intin[18] = 0;
	sig_std_logic_connect_in(port_intin18, cpssp, &intin18_funcs);

	cpssp->state_intin[19] = 0;
	sig_std_logic_connect_in(port_intin19, cpssp, &intin19_funcs);

	cpssp->state_intin[20] = 0;
	sig_std_logic_connect_in(port_intin20, cpssp, &intin20_funcs);

	cpssp->state_intin[21] = 0;
	sig_std_logic_connect_in(port_intin21, cpssp, &intin21_funcs);

	cpssp->state_intin[22] = 0;
	sig_std_logic_connect_in(port_intin22, cpssp, &intin22_funcs);

	cpssp->state_intin[23] = 0;
	sig_std_logic_connect_in(port_intin23, cpssp, &intin23_funcs);

	return cpssp;
}

void
chip_intel_82093AA_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
chip_intel_82093AA_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
chip_intel_82093AA_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
