/*
 * Copyright (C) 2007-2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <inttypes.h>
#include <setjmp.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>

#include "glue.h"

#include "chip_intel_80386_int.h"

#undef NAME
#undef NAME_

#define CHIP_(x)	chip_intel_80386_int_ ## x

struct cpssp {
	/* Config */

	/* Ports */
	struct sig_host_bus *host_bus;

	/* Signals */

	/* State */
	int state_event;
	int state_interrupt_pending;
	int state_nmi_pending;
	int state_power;
	int state_n_reset;
	int state_n_init;

#define STATE
#define NAME		cache2
#define NAME_(x)	cache2_ ## x
#include "arch_gen_cpu_x86_cache2_int.c"
#undef NAME_
#undef NAME
#define NAME		mmu
#define NAME_(x)	mmu_ ## x
#include "arch_gen_cpu_x86_mmu_int.c"
#undef NAME_
#undef NAME
#define NAME		cache1i
#define NAME_(x)	cache1i_ ## x
#define CONFIG_CODE	1
#include "arch_gen_cpu_x86_cache1_int.c"
#undef CONFIG_CODE
#undef NAME_
#undef NAME
#define NAME		cache1d
#define NAME_(x)	cache1d_ ## x
#define CONFIG_DATA	1
#include "arch_gen_cpu_x86_cache1_int.c"
#undef CONFIG_DATA
#undef NAME_
#undef NAME
#define NAME		align
#define NAME_(x)	align_ ## x
#include "arch_gen_cpu_x86_align_int.c"
#undef NAME_
#undef NAME
#define NAME		core
#define NAME_(x)	core_ ## x
#include "arch_gen_cpu_x86_core_int.c"
#undef NAME_
#undef NAME
#undef STATE

	/* Processes */
	struct process process;
};

static void
CHIP_(ior)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint32_t *valp)
{
	*valp = -1;
	if (unlikely(sig_host_bus_ior(cpssp->host_bus, cpssp, port, bs, valp) != 0)) {
		sig_host_bus_addr_type(cpssp->host_bus, cpssp,
				0, port, SIG_HOST_BUS_IOR);
		/* delay... */
		sig_host_bus_read_data(cpssp->host_bus, cpssp,
				bs, valp);
	}
}

static void
CHIP_(iow)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_iow(cpssp->host_bus, cpssp, port, bs, val) != 0)) {
		sig_host_bus_addr_type(cpssp->host_bus, cpssp,
				0, port, SIG_HOST_BUS_IOW);
		/* delay... */
		sig_host_bus_write_data(cpssp->host_bus, cpssp,
				bs, val);
	}
}

static void
CHIP_(mx)(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t *valp)
{
	if (unlikely(sig_host_bus_mx(cpssp->host_bus, cpssp,
			0, addr, bs, valp) != 0)) {
		sig_host_bus_addr_type(cpssp->host_bus, cpssp,
				0, addr, SIG_HOST_BUS_MR);
		/* delay... */
		sig_host_bus_read_data(cpssp->host_bus, cpssp, /* FIXME */
				bs, valp);
	}
}

static void
CHIP_(mr)(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t *valp)
{
	if (unlikely(sig_host_bus_mr(cpssp->host_bus, cpssp,
			0, addr, bs, valp) != 0)) {
		sig_host_bus_addr_type(cpssp->host_bus, cpssp,
				0, addr, SIG_HOST_BUS_MR);
		/* delay... */
		sig_host_bus_read_data(cpssp->host_bus, cpssp,
				bs, valp);
	}
}

static void
CHIP_(mw)(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_mw(cpssp->host_bus, cpssp,
			0, addr, bs, val) != 0)) {
		sig_host_bus_addr_type(cpssp->host_bus, cpssp,
				0, addr, SIG_HOST_BUS_MW);
		/* delay... */
		sig_host_bus_write_data(cpssp->host_bus, cpssp,
				bs, val);
	}
}

/*
 * Forward declarations cache2.
 */
/*forward*/ static void
cache2_mx(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
cache2_mr(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
cache2_mw(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
cache2_create(struct cpssp *cpssp);
/*forward*/ static void
cache2_destroy(struct cpssp *cpssp);

/*
 * Forward declarations mmu.
 */
/*forward*/ static void
mmu_tlb_flush(struct cpssp *cpssp);
/*forward*/ static void
mmu_load_cr3(struct cpssp *cpssp, uint32_t val);
/*forward*/ static void
mmu_vmx(struct cpssp *cpssp, uint32_t addr, unsigned int bs,
		bool user, uint32_t *valp, uint8_t *errp);
/*forward*/ static void
mmu_vmr(struct cpssp *cpssp, uint32_t addr, unsigned int bs,
		bool user, uint32_t *valp, uint8_t *errp, bool w_test);
/*forward*/ static void
mmu_vmw(struct cpssp *cpssp, uint32_t addr, unsigned int bs, 
		bool user, uint32_t val, uint8_t *errp);

/*
 * Forward declarations cache1i.
 */
/*forward*/ static void
cache1i_mx(struct cpssp *cpssp, uint32_t addr, unsigned int bs,
		bool user, uint32_t *valp, uint8_t *errp);
/*forward*/ static void
cache1i_create(struct cpssp *cpssp);
/*forward*/ static void
cache1i_destroy(struct cpssp *cpssp);

/*
 * Forward declarations cache1d.
 */
/*forward*/ static void
cache1d_mr(struct cpssp *cpssp, uint32_t addr, unsigned int bs,
		bool user, uint32_t *valp, uint8_t *errp, bool w_test);
/*forward*/ static void
cache1d_mw(struct cpssp *cpssp, uint32_t addr, unsigned int bs,
		bool user, uint32_t val, uint8_t *errp);
/*forward*/ static void
cache1d_create(struct cpssp *cpssp);
/*forward*/ static void
cache1d_destroy(struct cpssp *cpssp);

/*
 * Forward declarations align.
 */
/*forward*/ static void
align_uiorb(struct cpssp *cpssp, uint32_t addr, uint8_t *valp);
/*forward*/ static void
align_uiorw(struct cpssp *cpssp, uint32_t addr, uint16_t *valp);
/*forward*/ static void
align_uiord(struct cpssp *cpssp, uint32_t addr, uint32_t *valp);
/*forward*/ static void
align_uiowb(struct cpssp *cpssp, uint32_t addr, uint8_t value);
/*forward*/ static void
align_uioww(struct cpssp *cpssp, uint32_t addr, uint16_t value);
/*forward*/ static void
align_uiowd(struct cpssp *cpssp, uint32_t addr, uint32_t value);
/*forward*/ static void
align_umxb(struct cpssp *cpssp, uint32_t addr, bool user,
		uint8_t *valp, uint8_t *merrp);
/*forward*/ static void
align_umxw(struct cpssp *cpssp, uint32_t addr, bool user,
		uint16_t *valp, uint8_t *merrp);
/*forward*/ static void
align_umxd(struct cpssp *cpssp, uint32_t addr, bool user,
		uint32_t *valp, uint8_t *merrp);
/*forward*/ static void
align_umrb(struct cpssp *cpssp, uint32_t addr, bool user,
		uint8_t *valp, uint8_t *merrp, bool w_test);
/*forward*/ static void
align_umrw(struct cpssp *cpssp, uint32_t addr, bool user,
		uint16_t *valp, uint8_t *merrp, bool w_test);
/*forward*/ static void
align_umrd(struct cpssp *cpssp, uint32_t addr, bool user,
		uint32_t *valp, uint8_t *merrp, bool w_test);
/*forward*/ static void
align_umwb(struct cpssp *cpssp, uint32_t addr, bool user,
		uint8_t val, uint8_t *merrp);
/*forward*/ static void
align_umww(struct cpssp *cpssp, uint32_t addr, bool user,
		uint16_t val, uint8_t *merrp);
/*forward*/ static void
align_umwd(struct cpssp *cpssp, uint32_t addr, bool user,
		uint32_t val, uint8_t *merrp);

/*
 * Callbacks for cache2.
 */
static void
cache2_bus_mx(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t *valp)
{
	CHIP_(mx)(cpssp, addr, bs, valp);
}

static void
cache2_bus_mr(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t *valp)
{
	CHIP_(mr)(cpssp, addr, bs, valp);
}

static void
cache2_bus_mw(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t val)
{
	CHIP_(mw)(cpssp, addr, bs, val);
}

/*
 * Callbacks for MMU.
 */
static void
mmu_mx(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t *valp)
{
	cache2_mx(cpssp, addr, bs, valp);
}

static void
mmu_mr(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t *valp)
{
	cache2_mr(cpssp, addr, bs, valp);
}

static void
mmu_mw(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t val)
{
	cache2_mw(cpssp, addr, bs, val);
}

/*
 * Callbacks for cache1i.
 */
static void
cache1i_mmu_mx(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	bool user,
	uint32_t *valp,
	uint8_t *errp
)
{
	mmu_vmx(cpssp, addr, bs, user, valp, errp);
}

/*
 * Callbacks for cache1d.
 */
static void
cache1d_mmu_mr(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	bool user,
	uint32_t *valp,
	uint8_t *errp,
	bool w_test
)
{
	mmu_vmr(cpssp, addr, bs, user, valp, errp, w_test);
}

static void
cache1d_mmu_mw(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	bool user,
	uint32_t val,
	uint8_t *errp
)
{
	mmu_vmw(cpssp, addr, bs, user, val, errp);
}

/*
 * Callbacks for align.
 */
static void
align_ior(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint32_t *valp)
{
	CHIP_(ior)(cpssp, port, bs, valp);
}

static void
align_iow(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint32_t val)
{
	CHIP_(iow)(cpssp, port, bs, val);
}

static void
align_vmx(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	bool user,
	uint32_t *valp,
	uint8_t *errp
)
{
	cache1i_mx(cpssp, addr, bs, user, valp, errp);
}

static void
align_vmr(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	bool user,
	uint32_t *valp,
	uint8_t *errp,
	bool w_test
)
{
	cache1d_mr(cpssp, addr, bs, user, valp, errp, w_test);
}

static void
align_vmw(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	bool user,
	uint32_t val,
	uint8_t *errp
)
{
	cache1d_mw(cpssp, addr, bs, user, val, errp);
}

/*
 * Callbacks for core.
 */
static uint8_t
core_ack(struct cpssp *cpssp)
{
	uint8_t vec;

	sig_host_bus_inta_addr(cpssp->host_bus, cpssp);
	sig_host_bus_inta_data(cpssp->host_bus, cpssp, &vec);

	return vec;
}

static void
core_tlb_flush(struct cpssp *cpssp)
{
	mmu_tlb_flush(cpssp);
}

static void
core_load_cr3(struct cpssp *cpssp, uint32_t val)
{
	mmu_load_cr3(cpssp, val);
}

static void
core_uiorb(struct cpssp *cpssp, uint32_t addr, uint8_t *valp)
{
	align_uiorb(cpssp, addr, valp);
}

static void
core_uiorw(struct cpssp *cpssp, uint32_t addr, uint16_t *valp)
{
	align_uiorw(cpssp, addr, valp);
}

static void
core_uiord(struct cpssp *cpssp, uint32_t addr, uint32_t *valp)
{
	align_uiord(cpssp, addr, valp);
}

static void
core_uiowb(struct cpssp *cpssp, uint32_t addr, uint8_t value)
{
	align_uiowb(cpssp, addr, value);
}

static void
core_uioww(struct cpssp *cpssp, uint32_t addr, uint16_t value)
{
	align_uioww(cpssp, addr, value);
}

static void
core_uiowd(struct cpssp *cpssp, uint32_t addr, uint32_t value)
{
	align_uiowd(cpssp, addr, value);
}

static void
core_umxb(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint8_t *valp,
	uint8_t *merrp
)
{
	align_umxb(cpssp, addr, user, valp, merrp);
}

static void
core_umxw(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint16_t *valp,
	uint8_t *merrp
)
{
	align_umxw(cpssp, addr, user, valp, merrp);
}

static void
core_umxd(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint32_t *valp,
	uint8_t *merrp
)
{
	align_umxd(cpssp, addr, user, valp, merrp);
}

static void
core_umrb(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint8_t *valp,
	uint8_t *merrp,
	bool w_test
)
{
	align_umrb(cpssp, addr, user, valp, merrp, w_test);
}

static void
core_umrw(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint16_t *valp,
	uint8_t *merrp,
	bool w_test
)
{
	align_umrw(cpssp, addr, user, valp, merrp, w_test);
}

static void
core_umrd(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint32_t *valp,
	uint8_t *merrp,
	bool w_test
)
{
	align_umrd(cpssp, addr, user, valp, merrp, w_test);
}

static void
core_umwb(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint8_t val,
	uint8_t *merrp
)
{
	align_umwb(cpssp, addr, user, val, merrp);
}

static void
core_umww(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint16_t val,
	uint8_t *merrp
)
{
	align_umww(cpssp, addr, user, val, merrp);
}

static void
core_umwd(
	struct cpssp *cpssp,
	uint32_t addr,
	bool user,
	uint32_t val,
	uint8_t *merrp
)
{
	align_umwd(cpssp, addr, user, val, merrp);
}

#define BEHAVIOR
#define NAME		cache2
#define NAME_(x)	cache2_ ## x
#include "arch_gen_cpu_x86_cache2_int.c"
#undef NAME_
#undef NAME
#define NAME		mmu
#define NAME_(x)	mmu_ ## x
#include "arch_gen_cpu_x86_mmu_int.c"
#undef NAME_
#undef NAME
#define NAME		cache1i
#define NAME_(x)	cache1i_ ## x
#define CONFIG_CODE	1
#include "arch_gen_cpu_x86_cache1_int.c"
#undef CONFIG_CODE
#undef NAME_
#undef NAME
#define NAME		cache1d
#define NAME_(x)	cache1d_ ## x
#define CONFIG_DATA	1
#include "arch_gen_cpu_x86_cache1_int.c"
#undef CONFIG_DATA
#undef NAME_
#undef NAME
#define NAME		align
#define NAME_(x)	align_ ## x
#include "arch_gen_cpu_x86_align_int.c"
#undef NAME_
#undef NAME
#define NAME		core
#define NAME_(x)	core_ ## x
#include "arch_gen_cpu_x86_core_int.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

void __attribute__((__noreturn__))
CHIP_(step)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:  ;
	sched_to_scheduler();

	while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
		//TODO
		if (! cpssp->state_power) {
			cpssp->process.inst_cnt = cpssp->process.inst_limit;

		} else {
			core_step(cpssp);
			cpssp->process.inst_cnt += 1;
		}
	}
	goto again;
}

static void
CHIP_(irq_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->state_event |= val;
	cpssp->state_interrupt_pending = val;
}

static void
CHIP_(nmi_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->state_event |= val;
	cpssp->state_nmi_pending = val;
}

static void
CHIP_(smi_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

//	fprintf(stderr, "SMI TODO\n");
	return;
	cpssp->state_event |= val;
	cpssp->state_nmi_pending = val;
}

static void
CHIP_(a20_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

//	fprintf(stderr, "a20 TODO\n");
	return;
	cpssp->state_event |= val;
	cpssp->state_nmi_pending = val;
}

static void
CHIP_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->state_event |= !val;
	cpssp->state_power = val;
}

static void
CHIP_(n_reset_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (n_val) {
		/* Reset gone. */
		cpssp->state_n_reset |= 1;
	} else {
		/* New reset. */
		cpssp->state_event |= 1;
		cpssp->state_n_reset = 0;
	}
}

static void
CHIP_(n_ferr_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

//	fprintf(stderr, "FERR TODO\n");
	return;
	cpssp->state_event |= val;
	cpssp->state_nmi_pending = val;
}

static void
CHIP_(n_ignne_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

//	fprintf(stderr, "IGNNE TODO\n");
	return;
	cpssp->state_event |= val;
	cpssp->state_nmi_pending = val;
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_power,
	struct sig_std_logic *port_n_reset,
	struct sig_std_logic *port_irq,
	struct sig_std_logic *port_nmi,
	struct sig_std_logic *port_smi,
	struct sig_std_logic *port_n_ferr,
	struct sig_std_logic *port_n_ignne,
	struct sig_std_logic *port_a20,
	struct sig_host_bus *port_bus
)
{
	static const struct sig_std_logic_funcs irq_func = {
		.boolean_or_set = CHIP_(irq_set),
	};
	static const struct sig_std_logic_funcs nmi_func = {
		.boolean_or_set = CHIP_(nmi_set),
	};
	static const struct sig_std_logic_funcs smi_func = {
		.boolean_or_set = CHIP_(smi_set),
	};
	static const struct sig_std_logic_funcs a20_func = {
		.boolean_or_set = CHIP_(a20_set),
	};
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = CHIP_(power_set),
	};
	static const struct sig_std_logic_funcs n_reset_funcs = {
		.boolean_or_set = CHIP_(n_reset_set),
	};
	static const struct sig_std_logic_funcs n_ferr_func = {
		.boolean_or_set = CHIP_(n_ferr_set),
	};
	static const struct sig_std_logic_funcs n_ignne_func = {
		.boolean_or_set = CHIP_(n_ignne_set),
	};

	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	memset(cpssp, 0, sizeof(*cpssp));

	cache2_create(cpssp);
	cache1i_create(cpssp);
	cache1d_create(cpssp);

	/* Running a 20MHz CPU - FIXME */
	cpssp->process.inst_hz = 20*1000*1000;
	//cpssp->process.inst_hz = 233*1000*1000;

	cpssp->host_bus = port_bus;

	sig_std_logic_connect_in(port_irq, cpssp, &irq_func);
	sig_std_logic_connect_in(port_nmi, cpssp, &nmi_func);
	sig_std_logic_connect_in(port_smi, cpssp, &smi_func);
	sig_std_logic_connect_in(port_a20, cpssp, &a20_func);
	sig_std_logic_connect_in(port_power, cpssp, &power_funcs);
	sig_std_logic_connect_in(port_n_reset, cpssp, &n_reset_funcs);
	sig_std_logic_connect_in(port_n_ferr, cpssp, &n_ferr_func);
	sig_std_logic_connect_in(port_n_ignne, cpssp, &n_ignne_func);

	sched_process_init(&cpssp->process, CHIP_(step), cpssp);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	cache1d_destroy(cpssp);
	cache1i_destroy(cpssp);
	cache2_destroy(cpssp);

	shm_free(cpssp);
}


void
CHIP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
