/*
 * $Id: chip_gen_idectrl.c,v 1.2 2013-05-10 08:49:10 vrsieh Exp $ 
 *
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "glue.h"

#include "chip_gen_idectrl.h"

#define COMP "chip_gen_idectrl"

struct cpssp {
	/*
	 * Config
	 */
	unsigned int irq;
	unsigned int port_0_7;
	unsigned int port_8;

	/*
	 * Signals
	 */
	struct sig_std_logic *sig_p5V;
	struct sig_std_logic *sig_n_reset;
	struct sig_isa_bus *sig_isa_bus;
	struct sig_std_logic *sig_isa_bus_irq;
	struct sig_ide_bus *sig_ide_bus;

	/*
	 * State
	 */
};

/*
 * IDE Bus Interface
 */
static void
isa_ide_controller_irq_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	sig_std_logic_or_set(cpssp->sig_isa_bus_irq, cpssp, val);
}

static __attribute__((__noreturn__)) void
isa_ide_controller_dmarq_set(void *_cpssp, unsigned int val)
{
	assert(0);	/* No DMA capable device. */
}

/*
 * ISA Bus Interface
 */
static int
isa_ide_controller_inb(void *_cpssp, unsigned char *valp, uint32_t port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	uint16_t val16;

	if (cpssp->port_0_7 < port && port < cpssp->port_0_7 + 8) {
		sig_ide_bus_inw(cpssp->sig_ide_bus,
				port - cpssp->port_0_7, &val16);
		*valp = val16;
		return 0;
	} else if (cpssp->port_8 == port) {
		sig_ide_bus_inw(cpssp->sig_ide_bus,
				port - cpssp->port_8 + 8, &val16);
		*valp = val16;
		return 0;
	}

	return -1;
}

static int
isa_ide_controller_inw(void *_cpssp, unsigned short *valp, uint32_t port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->port_0_7 == port) {
		sig_ide_bus_inw(cpssp->sig_ide_bus, 0, valp);
		return 0;
	}

	return -1;
}

static int
isa_ide_controller_outb(void *_cpssp, unsigned char val, uint32_t port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->port_0_7 < port && port < cpssp->port_0_7 + 8) {
		sig_ide_bus_outw(cpssp->sig_ide_bus,
				port - cpssp->port_0_7, val);
		return 0;
	} else if (cpssp->port_8 == port) {
		sig_ide_bus_outw(cpssp->sig_ide_bus,
				port - cpssp->port_8 + 8, val);
		return 0;
	}

	return -1;
}

static int
isa_ide_controller_outw(void *_cpssp, unsigned short val, uint32_t port)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (cpssp->port_0_7 == port) {
		sig_ide_bus_outw(cpssp->sig_ide_bus, 0, val);
		return 0;
	}

	return -1;
}

void *
chip_gen_idectrl_create(
	const char *name,
	const char *irq,
	const char *ioaddr_1,
	const char *ioaddr_8,
	struct sig_manage *port_manage,
	struct sig_isa_conn *port_isa,
	struct sig_ide_bus *port_ide
)
{
	static const struct sig_isa_bus_funcs isa_bus_funcs = {
		.inb = isa_ide_controller_inb,
		.inw = isa_ide_controller_inw,
		.outb = isa_ide_controller_outb,
		.outw = isa_ide_controller_outw,
	};
	static const struct sig_ide_bus_funcs ide_bus_funcs = {
		.irq = isa_ide_controller_irq_set,
		.dmarq_set = isa_ide_controller_dmarq_set,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	/*
	 * Config
	 */
	cpssp->irq = strtoul(irq, NULL, 0);
	cpssp->port_0_7 = strtoul(ioaddr_8, NULL, 0);
	cpssp->port_8 = strtoul(ioaddr_1, NULL, 0);

	/*
	 * Signals
	 */
	cpssp->sig_p5V = port_isa->p5V;
	/* Connect to sig_p5V - FIXME */
	cpssp->sig_n_reset = port_isa->n_reset;
	/* Connect to sig_n_reset - FIXME */

	cpssp->sig_isa_bus = port_isa->main;
	sig_isa_bus_connect(port_isa->main, cpssp, &isa_bus_funcs);
	switch (cpssp->irq) {
	case 0x9: cpssp->sig_isa_bus_irq = port_isa->int9; break;
	case 0xa: cpssp->sig_isa_bus_irq = port_isa->int10; break;
	case 0xb: cpssp->sig_isa_bus_irq = port_isa->int11; break;
	case 0xc: cpssp->sig_isa_bus_irq = port_isa->int12; break;
	default: assert(0); /* Shouldn't happen. */
	}
	sig_std_logic_connect_out(cpssp->sig_isa_bus_irq, cpssp, SIG_STD_LOGIC_L);

	cpssp->sig_ide_bus = port_ide;
	sig_ide_bus_connect(port_ide, cpssp, &ide_bus_funcs);

	/*
	 * State
	 */
	/* Nothing... */

	return cpssp;
}

void
chip_gen_idectrl_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
chip_gen_idectrl_suspend(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
chip_gen_idectrl_resume(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}
