/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"
#include <assert.h>
#include <inttypes.h>
#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>

#include "system.h"
#include "conv_zero.h"
#include "glue-storage.h"
#include "glue-shm.h"
#include "glue-suspend.h"
#include "umutil.h"

#define INCLUDE
#include "arch_i2c_slave.c"
#undef INCLUDE

#include "chip_fujitsu_mb85rc256v.h"

#define COMP_(x) chip_fujitsu_mb85rc256v_ ## x

struct cpssp {
	struct sig_std_logic *port_sda;
	unsigned int state_vdd;
	unsigned int state_a0;
	unsigned int state_a1;
	unsigned int state_a2;

	struct media *media;

	unsigned int counter;
	unsigned int phase;

#define STATE
#define NAME		i2c_slave
#define NAME_(x)	i2c_slave_ ## x
#include "arch_i2c_slave.c"
#undef NAME_
#undef NAME
#undef STATE
};

static bool
COMP_(ack_addr)(struct cpssp *cpssp, unsigned char addr)
{
	if (((addr >> 4) & 0xf) == 0b1010
	 && ((addr >> 3) & 0x1) == cpssp->state_a2
	 && ((addr >> 2) & 0x1) == cpssp->state_a1
	 && ((addr >> 1) & 0x1) == cpssp->state_a0) {
		/* Selected */
		return true;
	}

	return false;
}

static void
COMP_(read_byte)(struct cpssp *cpssp, unsigned char *valp)
{
	int ret;

	switch (cpssp->phase) {
	case 0:
		*valp = (cpssp->counter >> 8) & 0xff;
		cpssp->phase++;
		break;
	case 1:
		*valp = (cpssp->counter >> 0) & 0xff;
		cpssp->phase++;
		break;
	default:
		assert(cpssp->counter < 32*1024);
		ret = storage_read(cpssp->media, valp, 1, cpssp->counter);
		assert(ret == 1);
		cpssp->counter++;
		cpssp->counter &= (32*1024 - 1);
		assert(cpssp->counter < 32*1024);
		break;
	}
}

static bool
COMP_(write_byte)(struct cpssp *cpssp, unsigned char val)
{
	int ret;

	switch (cpssp->phase) {
	case 0:
		cpssp->counter &= ~(0xff << 8);
		cpssp->counter |= val << 8;
		cpssp->phase++;
		break;
	case 1:
		cpssp->counter &= ~(0xff << 0);
		cpssp->counter |= val << 0;
		cpssp->phase++;
		break;
	default:
		ret = storage_write(cpssp->media, &val, 1, cpssp->counter);
		assert(ret == 1);
		cpssp->counter++;
		break;
	}
	cpssp->counter &= 32*1024 - 1;
	assert(cpssp->counter < 32*1024);

	return true;
}

static void
COMP_(stop_transaction)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->phase = 0;
}

static int
i2c_slave_ack_addr(struct cpssp *cpssp, uint8_t val)
{
	return COMP_(ack_addr)(cpssp, val);
}

static void
i2c_slave_read_byte(struct cpssp *cpssp, uint8_t *valp)
{
	COMP_(read_byte)(cpssp, valp);
}

static void
i2c_slave_write_byte(struct cpssp *cpssp, uint8_t val)
{
	COMP_(write_byte)(cpssp, val);
}

static void
i2c_slave_stop_transaction(struct cpssp *cpssp)
{
	COMP_(stop_transaction)(cpssp);
}

static void
i2c_slave_sda_out(struct cpssp *cpssp, unsigned int val)
{
	sig_std_logic_set(cpssp->port_sda, cpssp, val);
}

#define BEHAVIOR
#define NAME		i2c_slave
#define NAME_(x)	i2c_slave_ ## x
#include "arch_i2c_slave.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

static void
COMP_(sda_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	i2c_slave_sda_in(cpssp, val);
}

static void
COMP_(scl_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	i2c_slave_scl_in(cpssp, val);
}

static void
COMP_(a0_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_a0 = val;
}

static void
COMP_(a1_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_a1 = val;
}

static void
COMP_(a2_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_a2 = val;
}

static void
COMP_(vdd_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_vdd = val;

	if (val) {
		/* Power-on reset. */
		cpssp->phase = 0;
	}
}

void *
COMP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_std_logic *port_a0,
	struct sig_std_logic *port_a1,
	struct sig_std_logic *port_a2,
	struct sig_std_logic *port_vss,
	struct sig_std_logic *port_sda,
	struct sig_std_logic *port_scl,
	struct sig_std_logic *port_wp,
	struct sig_std_logic *port_vdd
)
{
	static const struct sig_std_logic_funcs a0_funcs = {
		.boolean_or_set = COMP_(a0_set),
	};
	static const struct sig_std_logic_funcs a1_funcs = {
		.boolean_or_set = COMP_(a1_set),
	};
	static const struct sig_std_logic_funcs a2_funcs = {
		.boolean_or_set = COMP_(a2_set),
	};
#if 0
	static const struct sig_std_logic_funcs vss_funcs = {
		.boolean_or_set = COMP_(vss_set),
	};
#endif
	static const struct sig_std_logic_funcs sda_funcs = {
		.std_logic_set = COMP_(sda_set),
	};
	static const struct sig_std_logic_funcs scl_funcs = {
		.std_logic_set = COMP_(scl_set),
	};
#if 0
	static const struct sig_std_logic_funcs wp_funcs = {
		.boolean_or_set = COMP_(wp_set),
	};
#endif
	static const struct sig_std_logic_funcs vdd_funcs = {
		.boolean_or_set = COMP_(vdd_set),
	};
	struct cpssp *cpssp;

	system_name_push(name);

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	cpssp->media = storage_create(system_path(), 32*1024,
			NULL,
			conv_zero_open, conv_zero_close, conv_zero_read);
	assert(cpssp->media);

	i2c_slave_create(cpssp);

	/* Out */
	cpssp->port_sda = port_sda;
	sig_std_logic_connect_out(port_sda, cpssp, SIG_STD_LOGIC_Z);

	/* In */
	cpssp->state_a0 = 0;
	sig_std_logic_connect_in(port_a0, cpssp, &a0_funcs);
	cpssp->state_a1 = 0;
	sig_std_logic_connect_in(port_a1, cpssp, &a1_funcs);
	cpssp->state_a2 = 0;
	sig_std_logic_connect_in(port_a2, cpssp, &a2_funcs);

	sig_std_logic_connect_in(port_sda, cpssp, &sda_funcs);
	sig_std_logic_connect_in(port_scl, cpssp, &scl_funcs);

	cpssp->state_vdd = 0;
	sig_std_logic_connect_in(port_vdd, cpssp, &vdd_funcs);

	system_name_pop();

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	int ret;

	i2c_slave_destroy(cpssp);

	ret = storage_destroy(cpssp->media);

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
COMP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	struct media *savemedia = cpssp->media;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
	
	cpssp->media = savemedia;
}
