/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>

#include "glue.h"

#include "chip_atmel_atmega32.h"

#define CHIP_(x)	chip_atmel_atmega32_ ## x

#define INCLUDE
#include "arch_atmel_cpu.c"
#include "arch_atmel_adc.c"
#include "arch_atmel_gpio.c"
#include "arch_atmel_int.c"
#include "arch_atmel_timer16.c"
#undef INCLUDE

#define STATE
struct cpssp {
	struct process process;

	int state_vcc;
	int state_nreset;

	unsigned int state_a0;
	unsigned int state_gpioa_0;
	unsigned int state_adc_0;
	unsigned int state_a1;
	unsigned int state_gpioa_1;
	unsigned int state_adc_1;
	unsigned int state_a2;
	unsigned int state_gpioa_2;
	unsigned int state_adc_2;
	unsigned int state_a3;
	unsigned int state_gpioa_3;
	unsigned int state_adc_3;
	unsigned int state_a4;
	unsigned int state_gpioa_4;
	unsigned int state_adc_4;
	unsigned int state_a5;
	unsigned int state_gpioa_5;
	unsigned int state_adc_5;
	unsigned int state_a6;
	unsigned int state_gpioa_6;
	unsigned int state_adc_6;
	unsigned int state_a7;
	unsigned int state_gpioa_7;
	unsigned int state_adc_7;

	unsigned int state_b0;
	unsigned int state_gpiob_0;
	unsigned int state_b1;
	unsigned int state_gpiob_1;
	unsigned int state_b2;
	unsigned int state_gpiob_2;
	unsigned int state_int2__;
	unsigned int state_b3;
	unsigned int state_gpiob_3;
	unsigned int state_b4;
	unsigned int state_gpiob_4;
	unsigned int state_b5;
	unsigned int state_gpiob_5;
	unsigned int state_b6;
	unsigned int state_gpiob_6;
	unsigned int state_b7;
	unsigned int state_gpiob_7;

	unsigned int state_c0;
	unsigned int state_gpioc_0;
	unsigned int state_c1;
	unsigned int state_gpioc_1;
	unsigned int state_c2;
	unsigned int state_gpioc_2;
	unsigned int state_c3;
	unsigned int state_gpioc_3;
	unsigned int state_c4;
	unsigned int state_gpioc_4;
	unsigned int state_c5;
	unsigned int state_gpioc_5;
	unsigned int state_c6;
	unsigned int state_gpioc_6;
	unsigned int state_c7;
	unsigned int state_gpioc_7;

	unsigned int state_d0;
	unsigned int state_gpiod_0;
	unsigned int state_d1;
	unsigned int state_gpiod_1;
	unsigned int state_d2;
	unsigned int state_gpiod_2;
	unsigned int state_int0__;
	unsigned int state_d3;
	unsigned int state_gpiod_3;
	unsigned int state_int1__;
	unsigned int state_d4;
	unsigned int state_gpiod_4;
	unsigned int state_d5;
	unsigned int state_gpiod_5;
	unsigned int state_d6;
	unsigned int state_gpiod_6;
	unsigned int state_d7;
	unsigned int state_gpiod_7;

	struct sig_std_logic *port_a0;
	struct sig_std_logic *port_a1;
	struct sig_std_logic *port_a2;
	struct sig_std_logic *port_a3;
	struct sig_std_logic *port_a4;
	struct sig_std_logic *port_a5;
	struct sig_std_logic *port_a6;
	struct sig_std_logic *port_a7;

	struct sig_std_logic *port_b0;
	struct sig_std_logic *port_b1;
	struct sig_std_logic *port_b2;
	struct sig_std_logic *port_b3;
	struct sig_std_logic *port_b4;
	struct sig_std_logic *port_b5;
	struct sig_std_logic *port_b6;
	struct sig_std_logic *port_b7;

	struct sig_std_logic *port_c0;
	struct sig_std_logic *port_c1;
	struct sig_std_logic *port_c2;
	struct sig_std_logic *port_c3;
	struct sig_std_logic *port_c4;
	struct sig_std_logic *port_c5;
	struct sig_std_logic *port_c6;
	struct sig_std_logic *port_c7;

	struct sig_std_logic *port_d0;
	struct sig_std_logic *port_d1;
	struct sig_std_logic *port_d2;
	struct sig_std_logic *port_d3;
	struct sig_std_logic *port_d4;
	struct sig_std_logic *port_d5;
	struct sig_std_logic *port_d6;
	struct sig_std_logic *port_d7;

	uint16_t code[32*1024 / sizeof(uint16_t) + 1];
	uint8_t data[0x800];

	uint8_t io[256];

#define NAME		cpu
#define NAME_(x)	cpu_ ## x
#include "arch_atmel_cpu.c"
#undef NAME_
#undef NAME
#define NAME		adc
#define NAME_(x)	adc_ ## x
#include "arch_atmel_adc.c"
#undef NAME_
#undef NAME
#define NAME		gpioa
#define NAME_(x)	gpioa_ ## x
#include "arch_atmel_gpio.c"
#undef NAME_
#undef NAME
#define NAME		gpiob
#define NAME_(x)	gpiob_ ## x
#include "arch_atmel_gpio.c"
#undef NAME_
#undef NAME
#define NAME		gpioc
#define NAME_(x)	gpioc_ ## x
#include "arch_atmel_gpio.c"
#undef NAME_
#undef NAME
#define NAME		gpiod
#define NAME_(x)	gpiod_ ## x
#include "arch_atmel_gpio.c"
#undef NAME_
#undef NAME
#define NAME		int0
#define NAME_(x)	int0_ ## x
#include "arch_atmel_int.c"
#undef NAME_
#undef NAME
#define NAME		int1
#define NAME_(x)	int1_ ## x
#include "arch_atmel_int.c"
#undef NAME_
#undef NAME
#define NAME		int2
#define NAME_(x)	int2_ ## x
#include "arch_atmel_int.c"
#undef NAME_
#undef NAME
#define NAME		timer1
#define NAME_(x)	timer1_ ## x
#include "arch_atmel_timer16.c"
#undef NAME_
#undef NAME
};
#undef STATE

/* cpu */
/*forward*/ static void
cpu_irqN_set(struct cpssp *cpssp, int n, unsigned int val);

/* adc */
/*forward*/ static void
adc_mux_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
adc_mux_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
adc_sra_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
adc_sra_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
adc_h_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
adc_h_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
adc_l_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
adc_l_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
adc_0_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc_1_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc_2_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc_3_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc_4_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc_5_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc_6_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
adc_7_in_set(struct cpssp *cpssp, unsigned int val);

/* gpioa */
/*forward*/ static void
gpioa_port_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpioa_port_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpioa_ddr_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpioa_ddr_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpioa_pin_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpioa_pin_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpioa_0_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioa_1_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioa_2_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioa_3_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioa_4_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioa_5_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioa_6_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioa_7_in_set(struct cpssp *cpssp, unsigned int val);

/* gpiob */
/*forward*/ static void
gpiob_port_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpiob_port_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpiob_ddr_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpiob_ddr_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpiob_pin_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpiob_pin_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpiob_0_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiob_1_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiob_2_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiob_3_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiob_4_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiob_5_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiob_6_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiob_7_in_set(struct cpssp *cpssp, unsigned int val);

/* gpioc */
/*forward*/ static void
gpioc_port_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpioc_port_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpioc_ddr_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpioc_ddr_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpioc_pin_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpioc_pin_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpioc_0_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioc_1_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioc_2_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioc_3_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioc_4_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioc_5_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioc_6_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpioc_7_in_set(struct cpssp *cpssp, unsigned int val);

/* gpiod */
/*forward*/ static void
gpiod_port_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpiod_port_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpiod_ddr_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpiod_ddr_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpiod_pin_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
gpiod_pin_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
gpiod_0_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiod_1_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiod_2_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiod_3_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiod_4_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiod_5_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiod_6_in_set(struct cpssp *cpssp, unsigned int val);
/*forward*/ static void
gpiod_7_in_set(struct cpssp *cpssp, unsigned int val);

/* int0 */
/*forward*/ static void
int0_f_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
int0_f_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
int0_ie_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
int0_ie_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
int0_sc_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
int0_sc_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
int0_ack(struct cpssp *cpssp);
/*forward*/ static void
int0___in_set(struct cpssp *cpssp, unsigned int val);

/* int1 */
/*forward*/ static void
int1_f_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
int1_f_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
int1_ie_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
int1_ie_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
int1_sc_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
int1_sc_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
int1_ack(struct cpssp *cpssp);
/*forward*/ static void
int1___in_set(struct cpssp *cpssp, unsigned int val);

/* int2 */
/*forward*/ static void
int2_f_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
int2_f_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
int2_ie_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
int2_ie_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
int2_sc_set(struct cpssp *cpssp, uint8_t val);
/*forward*/ static void
int2_sc_get(struct cpssp *cpssp, uint8_t *valp);
/*forward*/ static void
int2_ack(struct cpssp *cpssp);
/*forward*/ static void
int2___in_set(struct cpssp *cpssp, unsigned int val);

/* timer1 */
/*forward*/ static void
timer1_reg_set(struct cpssp *cpssp, uint8_t port, uint8_t val);
/*forward*/ static void
timer1_reg_get(struct cpssp *cpssp, uint8_t port, uint8_t *valp);
/*forward*/ static void
timer1_ack(struct cpssp *cpssp, unsigned int n);

#define RESOLVE_IO_IO(port, comp0, sig0) \
static void \
CHIP_(port_ ## port ## _resolve)(struct cpssp *cpssp) \
{ \
	unsigned int val; \
	\
	val = cpssp->state_ ## port; \
	val = sig_std_logic_resolve(val, cpssp->state_ ## comp0 ## _ ## sig0); \
	\
	sig_std_logic_set(cpssp->port_ ## port, cpssp, val); \
	comp0 ## _ ## sig0 ## _in_set(cpssp, val); \
} \
\
static void \
CHIP_(port_ ## port ## _out_set)(void *_cpssp, unsigned int val) \
{ \
	struct cpssp *cpssp = _cpssp; \
	\
	cpssp->state_ ## port = val; \
	\
	CHIP_(port_ ## port ## _resolve)(cpssp); \
} \
\
static void \
comp0 ## _ ## sig0 ## _out_set(struct cpssp *cpssp, unsigned int val) \
{ \
	cpssp->state_ ## comp0 ## _ ## sig0 = val; \
	\
	CHIP_(port_ ## port ## _resolve)(cpssp); \
}

#define RESOLVE_IO_IO_IN(port, comp0, sig0, comp1, sig1) \
static void \
CHIP_(port_ ## port ## _resolve)(struct cpssp *cpssp) \
{ \
	unsigned int val; \
	\
	val = cpssp->state_ ## port; \
	val = sig_std_logic_resolve(val, cpssp->state_ ## comp0 ## _ ## sig0); \
	\
	sig_std_logic_set(cpssp->port_ ## port, cpssp, val); \
	comp0 ## _ ## sig0 ## _in_set(cpssp, val); \
	comp1 ## _ ## sig1 ## _in_set(cpssp, val); \
} \
\
static void \
CHIP_(port_ ## port ## _out_set)(void *_cpssp, unsigned int val) \
{ \
	struct cpssp *cpssp = _cpssp; \
	\
	cpssp->state_ ## port = val; \
	\
	CHIP_(port_ ## port ## _resolve)(cpssp); \
} \
\
static void \
comp0 ## _ ## sig0 ## _out_set(struct cpssp *cpssp, unsigned int val) \
{ \
	cpssp->state_ ## comp0 ## _ ## sig0 = val; \
	\
	CHIP_(port_ ## port ## _resolve)(cpssp); \
}

RESOLVE_IO_IO_IN(a0, gpioa, 0, adc, 0)
RESOLVE_IO_IO_IN(a1, gpioa, 1, adc, 1)
RESOLVE_IO_IO_IN(a2, gpioa, 2, adc, 2)
RESOLVE_IO_IO_IN(a3, gpioa, 3, adc, 3)
RESOLVE_IO_IO_IN(a4, gpioa, 4, adc, 4)
RESOLVE_IO_IO_IN(a5, gpioa, 5, adc, 5)
RESOLVE_IO_IO_IN(a6, gpioa, 6, adc, 6)
RESOLVE_IO_IO_IN(a7, gpioa, 7, adc, 7)

RESOLVE_IO_IO(b0, gpiob, 0)
RESOLVE_IO_IO(b1, gpiob, 1)
RESOLVE_IO_IO_IN(b2, gpiob, 2, int2, _)
RESOLVE_IO_IO(b3, gpiob, 3)
RESOLVE_IO_IO(b4, gpiob, 4)
RESOLVE_IO_IO(b5, gpiob, 5)
RESOLVE_IO_IO(b6, gpiob, 6)
RESOLVE_IO_IO(b7, gpiob, 7)

RESOLVE_IO_IO(c0, gpioc, 0)
RESOLVE_IO_IO(c1, gpioc, 1)
RESOLVE_IO_IO(c2, gpioc, 2)
RESOLVE_IO_IO(c3, gpioc, 3)
RESOLVE_IO_IO(c4, gpioc, 4)
RESOLVE_IO_IO(c5, gpioc, 5)
RESOLVE_IO_IO(c6, gpioc, 6)
RESOLVE_IO_IO(c7, gpioc, 7)

RESOLVE_IO_IO(d0, gpiod, 0)
RESOLVE_IO_IO(d1, gpiod, 1)
RESOLVE_IO_IO_IN(d2, gpiod, 2, int0, _)
RESOLVE_IO_IO_IN(d3, gpiod, 3, int1, _)
RESOLVE_IO_IO(d4, gpiod, 4)
RESOLVE_IO_IO(d5, gpiod, 5)
RESOLVE_IO_IO(d6, gpiod, 6)
RESOLVE_IO_IO(d7, gpiod, 7)

/*
 * Callbacks for cpu.
 */
static void
cpu_fetch(struct cpssp *cpssp, uint16_t addr, uint16_t *valp)
{
	assert(addr < sizeof(cpssp->code) / sizeof(cpssp->code[0]));

	*valp = cpssp->code[addr];
}

static void
cpu_ram_st(struct cpssp *cpssp, uint16_t addr, uint8_t val)
{
	assert(addr < sizeof(cpssp->data) / sizeof(cpssp->data[0]));

	cpssp->data[addr] = val;
}

static void
cpu_ram_ld(struct cpssp *cpssp, uint16_t addr, uint8_t *valp)
{
	assert(addr < sizeof(cpssp->data) / sizeof(cpssp->data[0]));

	*valp = cpssp->data[addr];
}

/*
 * See table at page 327.
 */
static void
cpu_io_st(struct cpssp *cpssp, uint8_t addr, uint8_t val)
{
	assert(/* 0 <= addr && */ addr < 0x40);

	switch (addr) {
	case 0x3f:
	case 0x3e:
	case 0x3d:
		assert(0); /* Core internal. Mustn't happen. */
	/* FIXME */
	case 0x3b: /* GICR */
		int1_ie_set(cpssp, (val >> 7) & 1);
		int0_ie_set(cpssp, (val >> 6) & 1);
		int2_ie_set(cpssp, (val >> 5) & 1);
		/* Bit 4-2: Reserved. */
		/* FIXME */
		goto def; /* FIXME */
	case 0x3a: /* GIFR */
		int1_f_set(cpssp, (val >> 7) & 1);
		int0_f_set(cpssp, (val >> 6) & 1);
		int2_f_set(cpssp, (val >> 5) & 1);
		/* Bit 4-0: Reserved. */
		break;
	case 0x39: /* TIMSK */
		/* FIXME */
		timer1_reg_set(cpssp, 11, (val >> 2) & 0xf);
		/* FIXME */
		goto def; /* FIXME */
	case 0x38: /* TIFR */
		/* FIXME */
		timer1_reg_set(cpssp, 10, (val >> 2) & 0xf);
		/* FIXME */
		goto def; /* FIXME */
	/* FIXME */
	case 0x35: /* MCUCR */
		/* FIXME */
		int1_sc_set(cpssp, (val >> 2) & 3);
		int0_sc_set(cpssp, (val >> 0) & 3);
		goto def; /* FIXME */
	case 0x34: /* MCUCSR */
		/* FIXME */
		int2_sc_set(cpssp, (1 << 1) | ((val >> 6) & 1));
		/* FIXME */
		goto def;
	/* FIXME */
	case 0x2f: /* TCCR1A */
	case 0x2e: /* TCCR1B */
	case 0x2d: /* TCNT1H */
	case 0x2c: /* TCNT1L */
	case 0x2b: /* OCR1AH */
	case 0x2a: /* OCR1AL */
	case 0x29: /* OCR1BH */
	case 0x28: /* OCR1BL */
	case 0x27: /* ICR1H */
	case 0x26: /* ICR1L */
		timer1_reg_set(cpssp, addr - 0x26, val);
		break;
	case 0x1b: /* PORTA */
		gpioa_port_set(cpssp, val);
		break;
	case 0x1a: /* DDRA */
		gpioa_ddr_set(cpssp, val);
		break;
	case 0x19: /* PINA */
		gpioa_pin_set(cpssp, val);
		break;
	case 0x18: /* PORTB */
		gpiob_port_set(cpssp, val);
		break;
	case 0x17: /* DDRB */
		gpiob_ddr_set(cpssp, val);
		break;
	case 0x16: /* PINB */
		gpiob_pin_set(cpssp, val);
		break;
	case 0x15: /* PORTC */
		gpioc_port_set(cpssp, val);
		break;
	case 0x14: /* DDRC */
		gpioc_ddr_set(cpssp, val);
		break;
	case 0x13: /* PINC */
		gpioc_pin_set(cpssp, val);
		break;
	case 0x12: /* PORTD */
		gpiod_port_set(cpssp, val);
		break;
	case 0x11: /* DDRD */
		gpiod_ddr_set(cpssp, val);
		break;
	case 0x10: /* PIND */
		gpiod_pin_set(cpssp, val);
		break;
	/* FIXME */
	case 0x07: /* ADCMUX */
		adc_mux_set(cpssp, val);
		break;
	case 0x06: /* ADCSRA */
		adc_sra_set(cpssp, val);
		break;
	case 0x05: /* ADCH */
		adc_h_set(cpssp, val);
		break;
	case 0x04: /* ADCL */
		adc_l_set(cpssp, val);
		break;
	/* FIXME */
	default:
	def:    ;
		fprintf(stderr, "WARNING: %s: addr=0x%02x\n",
				__FUNCTION__, addr);
		break;
	}
}

/*
 * See table at page 327.
 */
static void
cpu_io_ld(struct cpssp *cpssp, uint8_t addr, uint8_t *valp)
{
	uint8_t val;

	assert(/* 0 <= addr && */ addr < 0x40);

	switch (addr) {
	case 0x3f:
	case 0x3e:
	case 0x3d:
		assert(0); /* Core internal. Mustn't happen. */
	/* FIXME */
	case 0x3b: /* GICR */
		*valp = 0;
		int1_ie_get(cpssp, &val);
		*valp |= val << 7;
		int0_ie_get(cpssp, &val);
		*valp |= val << 6;
		int2_ie_get(cpssp, &val);
		*valp |= val << 5;
		/* Bit 4-2: Reserved. */
		/* FIXME */
		goto def; /* FIXME */
	case 0x3a: /* GIFR */
		*valp = 0;
		int1_f_get(cpssp, &val);
		*valp |= val << 7;
		int0_f_get(cpssp, &val);
		*valp |= val << 6;
		int2_f_get(cpssp, &val);
		*valp |= val << 5;
		/* Bit 4-0: Reserved. */
		break;
	case 0x39: /* TIMSK */
		*valp = 0;
		/* FIXME */
		timer1_reg_get(cpssp, 11, &val);
		*valp |= val << 2;
		/* FIXME */
		goto def; /* FIXME */
	case 0x38: /* TIFR */
		*valp = 0;
		/* FIXME */
		timer1_reg_get(cpssp, 10, &val);
		*valp |= val << 2;
		/* FIXME */
		goto def; /* FIXME */
	/* FIXME */
	case 0x35: /* MCUCR */
		*valp = 0;
		/* FIXME */
		int1_sc_get(cpssp, &val);
		*valp |= val << 2;
		int0_sc_get(cpssp, &val);
		*valp |= val << 0;
		goto def; /* FIXME */
	case 0x34: /* MCUCSR */
		*valp = 0;
		/* FIXME */
		int2_sc_get(cpssp, &val);
		assert((val >> 1) & 1);
		*valp |= (val & 1) << 6;
		/* FIXME */
		goto def;
	/* FIXME */
	case 0x2f: /* TCCR1A */
	case 0x2e: /* TCCR1B */
	case 0x2d: /* TCNT1H */
	case 0x2c: /* TCNT1L */
	case 0x2b: /* OCR1AH */
	case 0x2a: /* OCR1AL */
	case 0x29: /* OCR1BH */
	case 0x28: /* OCR1BL */
	case 0x27: /* ICR1H */
	case 0x26: /* ICR1L */
		timer1_reg_get(cpssp, addr - 0x26, valp);
		break;
	/* FIXME */
	case 0x1b: /* PORTA */
		gpioa_port_get(cpssp, valp);
		break;
	case 0x1a: /* DDRA */
		gpioa_ddr_get(cpssp, valp);
		break;
	case 0x19: /* PINA */
		gpioa_pin_get(cpssp, valp);
		break;
	case 0x18: /* PORTB */
		gpiob_port_get(cpssp, valp);
		break;
	case 0x17: /* DDRB */
		gpiob_ddr_get(cpssp, valp);
		break;
	case 0x16: /* PINB */
		gpiob_pin_get(cpssp, valp);
		break;
	case 0x15: /* PORTC */
		gpioc_port_get(cpssp, valp);
		break;
	case 0x14: /* DDRC */
		gpioc_ddr_get(cpssp, valp);
		break;
	case 0x13: /* PINC */
		gpioc_pin_get(cpssp, valp);
		break;
	case 0x12: /* PORTD */
		gpiod_port_get(cpssp, valp);
		break;
	case 0x11: /* DDRD */
		gpiod_ddr_get(cpssp, valp);
		break;
	case 0x10: /* PIND */
		gpiod_pin_get(cpssp, valp);
		break;
	/* FIXME */
	case 0x07: /* ADCMUX */
		adc_mux_get(cpssp, valp);
		break;
	case 0x06: /* ADCSRA */
		adc_sra_get(cpssp, valp);
		break;
	case 0x05: /* ADCH */
		adc_h_get(cpssp, valp);
		break;
	case 0x04: /* ADCL */
		adc_l_get(cpssp, valp);
		break;
	/* FIXME */
	default:
	def:    ;
		fprintf(stderr, "WARNING: %s: addr=0x%02x\n",
				__FUNCTION__, addr);
		break;
	}
}

static void
cpu_irq_ack(struct cpssp *cpssp, unsigned int n)
{
	/* Interrupt Vectors (page 44) */
	switch (n) {
	case 1:
		int0_ack(cpssp);
		break;
	case 2:
		int1_ack(cpssp);
		break;
	case 3:
		int2_ack(cpssp);
		break;
	case 4:
	case 5:
		goto todo;
	case 6:
	case 7:
	case 8:
	case 9:
		timer1_ack(cpssp, n - 6);
		break;
	default:
	todo:	;
		fprintf(stderr, "%s: n=%u\n", __FUNCTION__, n);
		assert(0);
	}
}

/*
 * Callbacks for int0.
 */
static void
int0_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* Interrupt Vectors (page 44) */
	cpu_irqN_set(cpssp, 1, val);
}

/*
 * Callbacks for int1.
 */
static void
int1_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* Interrupt Vectors (page 44) */
	cpu_irqN_set(cpssp, 2, val);
}

/*
 * Callbacks for int0.
 */
static void
int2_irq_set(struct cpssp *cpssp, unsigned int val)
{
	/* Interrupt Vectors (page 44) */
	cpu_irqN_set(cpssp, 3, val);
}

/*
 * Callbacks for timer1.
 */
static void
timer1_irq_set(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(0 <= n && n < 4);

	/* Interrupt Vectors (page 44) */
	cpu_irqN_set(cpssp, 6 + n, val);
}

#define BEHAVIOR
#define NAME		cpu
#define NAME_(x)	cpu_ ## x
#include "arch_atmel_cpu.c"
#undef NAME_
#undef NAME
#define NAME		adc
#define NAME_(x)	adc_ ## x
#include "arch_atmel_adc.c"
#undef NAME_
#undef NAME
#define NAME		gpioa
#define NAME_(x)	gpioa_ ## x
#include "arch_atmel_gpio.c"
#undef NAME_
#undef NAME
#define NAME		gpiob
#define NAME_(x)	gpiob_ ## x
#include "arch_atmel_gpio.c"
#undef NAME_
#undef NAME
#define NAME		gpioc
#define NAME_(x)	gpioc_ ## x
#include "arch_atmel_gpio.c"
#undef NAME_
#undef NAME
#define NAME		gpiod
#define NAME_(x)	gpiod_ ## x
#include "arch_atmel_gpio.c"
#undef NAME_
#undef NAME
#define NAME		int0
#define NAME_(x)	int0_ ## x
#include "arch_atmel_int.c"
#undef NAME_
#undef NAME
#define NAME		int1
#define NAME_(x)	int1_ ## x
#include "arch_atmel_int.c"
#undef NAME_
#undef NAME
#define NAME		int2
#define NAME_(x)	int2_ ## x
#include "arch_atmel_int.c"
#undef NAME_
#undef NAME
#define NAME		timer1
#define NAME_(x)	timer1_ ## x
#include "arch_atmel_timer16.c"
#undef NAME_
#undef NAME
#undef BEHAVIOR

static void
CHIP_(reset)(struct cpssp *cpssp)
{
	cpu_reset(cpssp);
}

static void
CHIP_(step)(struct cpssp *cpssp)
{
	cpu_step(cpssp);
	timer1_step(cpssp);
}

static void
CHIP_(vcc_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_vcc = val;
}

static void
CHIP_(nreset_set)(void *_cpssp, unsigned int nval)
{
	struct cpssp *cpssp = _cpssp;

	cpssp->state_nreset = nval;
}

static void __attribute__((__noreturn__))
CHIP_(clk)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

again:	;
	while (cpssp->process.inst_cnt < cpssp->process.inst_limit) {
		if (unlikely(! cpssp->state_vcc)) {
			/* Nothing to do... */
#if 0
		} else if (unlikely(! cpssp->state_nreset)) {
			CHIP_(reset)(cpssp);
#endif
		} else {
			CHIP_(step)(cpssp);
		}
		cpssp->process.inst_cnt += 1;
	}

	sched_to_scheduler();

	goto again;
}

void *
CHIP_(create)(
	const char *name,
	struct sig_manage *manage,
	struct sig_std_logic *port_b0,
	struct sig_std_logic *port_b1,
	struct sig_std_logic *port_b2,
	struct sig_std_logic *port_b3,
	struct sig_std_logic *port_b4,
	struct sig_std_logic *port_b5,
	struct sig_std_logic *port_b6,
	struct sig_std_logic *port_b7,
	struct sig_std_logic *port_nreset,
	struct sig_std_logic *port_vcc,
	struct sig_std_logic *port_gnd0,
	struct sig_std_logic *port_xtal2,
	struct sig_std_logic *port_xtal1,
	struct sig_std_logic *port_d0,
	struct sig_std_logic *port_d1,
	struct sig_std_logic *port_d2,
	struct sig_std_logic *port_d3,
	struct sig_std_logic *port_d4,
	struct sig_std_logic *port_d5,
	struct sig_std_logic *port_d6,
	struct sig_std_logic *port_d7,
	struct sig_std_logic *port_c0,
	struct sig_std_logic *port_c1,
	struct sig_std_logic *port_c2,
	struct sig_std_logic *port_c3,
	struct sig_std_logic *port_c4,
	struct sig_std_logic *port_c5,
	struct sig_std_logic *port_c6,
	struct sig_std_logic *port_c7,
	struct sig_std_logic *port_avcc,
	struct sig_std_logic *port_gnd1,
	struct sig_std_logic *port_aref,
	struct sig_std_logic *port_a7,
	struct sig_std_logic *port_a6,
	struct sig_std_logic *port_a5,
	struct sig_std_logic *port_a4,
	struct sig_std_logic *port_a3,
	struct sig_std_logic *port_a2,
	struct sig_std_logic *port_a1,
	struct sig_std_logic *port_a0
)
{
	static const struct sig_std_logic_funcs port_vcc_funcs = {
		.boolean_or_set = CHIP_(vcc_set),
	};
	static const struct sig_std_logic_funcs port_nreset_funcs = {
		.boolean_or_set = CHIP_(nreset_set),
	};
	static const struct sig_std_logic_funcs port_a0_funcs = {
		.set_ext = CHIP_(port_a0_out_set),
	};
	static const struct sig_std_logic_funcs port_a1_funcs = {
		.set_ext = CHIP_(port_a1_out_set),
	};
	static const struct sig_std_logic_funcs port_a2_funcs = {
		.set_ext = CHIP_(port_a2_out_set),
	};
	static const struct sig_std_logic_funcs port_a3_funcs = {
		.set_ext = CHIP_(port_a3_out_set),
	};
	static const struct sig_std_logic_funcs port_a4_funcs = {
		.set_ext = CHIP_(port_a4_out_set),
	};
	static const struct sig_std_logic_funcs port_a5_funcs = {
		.set_ext = CHIP_(port_a5_out_set),
	};
	static const struct sig_std_logic_funcs port_a6_funcs = {
		.set_ext = CHIP_(port_a6_out_set),
	};
	static const struct sig_std_logic_funcs port_a7_funcs = {
		.set_ext = CHIP_(port_a7_out_set),
	};
	static const struct sig_std_logic_funcs port_b0_funcs = {
		.set_ext = CHIP_(port_b0_out_set),
	};
	static const struct sig_std_logic_funcs port_b1_funcs = {
		.set_ext = CHIP_(port_b1_out_set),
	};
	static const struct sig_std_logic_funcs port_b2_funcs = {
		.set_ext = CHIP_(port_b2_out_set),
	};
	static const struct sig_std_logic_funcs port_b3_funcs = {
		.set_ext = CHIP_(port_b3_out_set),
	};
	static const struct sig_std_logic_funcs port_b4_funcs = {
		.set_ext = CHIP_(port_b4_out_set),
	};
	static const struct sig_std_logic_funcs port_b5_funcs = {
		.set_ext = CHIP_(port_b5_out_set),
	};
	static const struct sig_std_logic_funcs port_b6_funcs = {
		.set_ext = CHIP_(port_b6_out_set),
	};
	static const struct sig_std_logic_funcs port_b7_funcs = {
		.set_ext = CHIP_(port_b7_out_set),
	};
	static const struct sig_std_logic_funcs port_c0_funcs = {
		.set_ext = CHIP_(port_c0_out_set),
	};
	static const struct sig_std_logic_funcs port_c1_funcs = {
		.set_ext = CHIP_(port_c1_out_set),
	};
	static const struct sig_std_logic_funcs port_c2_funcs = {
		.set_ext = CHIP_(port_c2_out_set),
	};
	static const struct sig_std_logic_funcs port_c3_funcs = {
		.set_ext = CHIP_(port_c3_out_set),
	};
	static const struct sig_std_logic_funcs port_c4_funcs = {
		.set_ext = CHIP_(port_c4_out_set),
	};
	static const struct sig_std_logic_funcs port_c5_funcs = {
		.set_ext = CHIP_(port_c5_out_set),
	};
	static const struct sig_std_logic_funcs port_c6_funcs = {
		.set_ext = CHIP_(port_c6_out_set),
	};
	static const struct sig_std_logic_funcs port_c7_funcs = {
		.set_ext = CHIP_(port_c7_out_set),
	};
	static const struct sig_std_logic_funcs port_d0_funcs = {
		.set_ext = CHIP_(port_d0_out_set),
	};
	static const struct sig_std_logic_funcs port_d1_funcs = {
		.set_ext = CHIP_(port_d1_out_set),
	};
	static const struct sig_std_logic_funcs port_d2_funcs = {
		.set_ext = CHIP_(port_d2_out_set),
	};
	static const struct sig_std_logic_funcs port_d3_funcs = {
		.set_ext = CHIP_(port_d3_out_set),
	};
	static const struct sig_std_logic_funcs port_d4_funcs = {
		.set_ext = CHIP_(port_d4_out_set),
	};
	static const struct sig_std_logic_funcs port_d5_funcs = {
		.set_ext = CHIP_(port_d5_out_set),
	};
	static const struct sig_std_logic_funcs port_d6_funcs = {
		.set_ext = CHIP_(port_d6_out_set),
	};
	static const struct sig_std_logic_funcs port_d7_funcs = {
		.set_ext = CHIP_(port_d7_out_set),
	};
	struct cpssp *cpssp;
	int fd;
	int ret;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	fd = open("boardtest.bin", O_RDONLY);
	assert(0 <= fd);

	ret = read(fd, cpssp->code, sizeof(cpssp->code));
	assert(0 <= ret);
	assert(0 < ret);
	assert(ret < sizeof(cpssp->code));

	ret = close(fd);
	assert(0 <= ret);

	cpssp->state_vcc = 0;
	cpssp->state_nreset = 1;

	cpssp->state_a0 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioa_0 = SIG_STD_LOGIC_Z;
	cpssp->state_adc_0 = SIG_STD_LOGIC_Z;
	cpssp->state_a1 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioa_1 = SIG_STD_LOGIC_Z;
	cpssp->state_adc_1 = SIG_STD_LOGIC_Z;
	cpssp->state_a2 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioa_2 = SIG_STD_LOGIC_Z;
	cpssp->state_adc_2 = SIG_STD_LOGIC_Z;
	cpssp->state_a3 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioa_3 = SIG_STD_LOGIC_Z;
	cpssp->state_adc_3 = SIG_STD_LOGIC_Z;
	cpssp->state_a4 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioa_4 = SIG_STD_LOGIC_Z;
	cpssp->state_adc_4 = SIG_STD_LOGIC_Z;
	cpssp->state_a5 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioa_5 = SIG_STD_LOGIC_Z;
	cpssp->state_adc_5 = SIG_STD_LOGIC_Z;
	cpssp->state_a6 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioa_6 = SIG_STD_LOGIC_Z;
	cpssp->state_adc_6 = SIG_STD_LOGIC_Z;
	cpssp->state_a7 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioa_7 = SIG_STD_LOGIC_Z;
	cpssp->state_adc_7 = SIG_STD_LOGIC_Z;

	cpssp->state_b0 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiob_0 = SIG_STD_LOGIC_Z;
	cpssp->state_b1 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiob_1 = SIG_STD_LOGIC_Z;
	cpssp->state_b2 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiob_2 = SIG_STD_LOGIC_Z;
	cpssp->state_int2__ = SIG_STD_LOGIC_Z;
	cpssp->state_b3 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiob_3 = SIG_STD_LOGIC_Z;
	cpssp->state_b4 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiob_4 = SIG_STD_LOGIC_Z;
	cpssp->state_b5 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiob_5 = SIG_STD_LOGIC_Z;
	cpssp->state_b6 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiob_6 = SIG_STD_LOGIC_Z;
	cpssp->state_b7 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiob_7 = SIG_STD_LOGIC_Z;

	cpssp->state_c0 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioc_0 = SIG_STD_LOGIC_Z;
	cpssp->state_c1 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioc_1 = SIG_STD_LOGIC_Z;
	cpssp->state_c2 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioc_2 = SIG_STD_LOGIC_Z;
	cpssp->state_c3 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioc_3 = SIG_STD_LOGIC_Z;
	cpssp->state_c4 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioc_4 = SIG_STD_LOGIC_Z;
	cpssp->state_c5 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioc_5 = SIG_STD_LOGIC_Z;
	cpssp->state_c6 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioc_6 = SIG_STD_LOGIC_Z;
	cpssp->state_c7 = SIG_STD_LOGIC_Z;
	cpssp->state_gpioc_7 = SIG_STD_LOGIC_Z;

	cpssp->state_d0 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiod_0 = SIG_STD_LOGIC_Z;
	cpssp->state_d1 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiod_1 = SIG_STD_LOGIC_Z;
	cpssp->state_d2 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiod_2 = SIG_STD_LOGIC_Z;
	cpssp->state_int0__ = SIG_STD_LOGIC_Z;
	cpssp->state_d3 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiod_3 = SIG_STD_LOGIC_Z;
	cpssp->state_int1__ = SIG_STD_LOGIC_Z;
	cpssp->state_d4 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiod_4 = SIG_STD_LOGIC_Z;
	cpssp->state_d5 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiod_5 = SIG_STD_LOGIC_Z;
	cpssp->state_d6 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiod_6 = SIG_STD_LOGIC_Z;
	cpssp->state_d7 = SIG_STD_LOGIC_Z;
	cpssp->state_gpiod_7 = SIG_STD_LOGIC_Z;

	cpu_create(cpssp);
	adc_create(cpssp);
	gpioa_create(cpssp);
	gpiob_create(cpssp);
	gpioc_create(cpssp);
	gpiod_create(cpssp);
	int0_create(cpssp);
	int1_create(cpssp);
	int2_create(cpssp);
	timer1_create(cpssp);

	sig_std_logic_connect_in(port_vcc, cpssp, &port_vcc_funcs);
	sig_std_logic_connect_in(port_nreset, cpssp, &port_nreset_funcs);

	cpssp->port_a0 = port_a0;
	sig_std_logic_connect_out(port_a0, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_a0, cpssp, &port_a0_funcs);
	cpssp->port_a1 = port_a1;
	sig_std_logic_connect_out(port_a1, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_a1, cpssp, &port_a1_funcs);
	cpssp->port_a2 = port_a2;
	sig_std_logic_connect_out(port_a2, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_a2, cpssp, &port_a2_funcs);
	cpssp->port_a3 = port_a3;
	sig_std_logic_connect_out(port_a3, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_a3, cpssp, &port_a3_funcs);
	cpssp->port_a4 = port_a4;
	sig_std_logic_connect_out(port_a4, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_a4, cpssp, &port_a4_funcs);
	cpssp->port_a5 = port_a5;
	sig_std_logic_connect_out(port_a5, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_a5, cpssp, &port_a5_funcs);
	cpssp->port_a6 = port_a6;
	sig_std_logic_connect_out(port_a6, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_a6, cpssp, &port_a6_funcs);
	cpssp->port_a7 = port_a7;
	sig_std_logic_connect_out(port_a7, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_a7, cpssp, &port_a7_funcs);

	cpssp->port_b0 = port_b0;
	sig_std_logic_connect_out(port_b0, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_b0, cpssp, &port_b0_funcs);
	cpssp->port_b1 = port_b1;
	sig_std_logic_connect_out(port_b1, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_b1, cpssp, &port_b1_funcs);
	cpssp->port_b2 = port_b2;
	sig_std_logic_connect_out(port_b2, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_b2, cpssp, &port_b2_funcs);
	cpssp->port_b3 = port_b3;
	sig_std_logic_connect_out(port_b3, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_b3, cpssp, &port_b3_funcs);
	cpssp->port_b4 = port_b4;
	sig_std_logic_connect_out(port_b4, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_b4, cpssp, &port_b4_funcs);
	cpssp->port_b5 = port_b5;
	sig_std_logic_connect_out(port_b5, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_b5, cpssp, &port_b5_funcs);
	cpssp->port_b6 = port_b6;
	sig_std_logic_connect_out(port_b6, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_b6, cpssp, &port_b6_funcs);
	cpssp->port_b7 = port_b7;
	sig_std_logic_connect_out(port_b7, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_b7, cpssp, &port_b7_funcs);

	cpssp->port_c0 = port_c0;
	sig_std_logic_connect_out(port_c0, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_c0, cpssp, &port_c0_funcs);
	cpssp->port_c1 = port_c1;
	sig_std_logic_connect_out(port_c1, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_c1, cpssp, &port_c1_funcs);
	cpssp->port_c2 = port_c2;
	sig_std_logic_connect_out(port_c2, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_c2, cpssp, &port_c2_funcs);
	cpssp->port_c3 = port_c3;
	sig_std_logic_connect_out(port_c3, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_c3, cpssp, &port_c3_funcs);
	cpssp->port_c4 = port_c4;
	sig_std_logic_connect_out(port_c4, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_c4, cpssp, &port_c4_funcs);
	cpssp->port_c5 = port_c5;
	sig_std_logic_connect_out(port_c5, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_c5, cpssp, &port_c5_funcs);
	cpssp->port_c6 = port_c6;
	sig_std_logic_connect_out(port_c6, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_c6, cpssp, &port_c6_funcs);
	cpssp->port_c7 = port_c7;
	sig_std_logic_connect_out(port_c7, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_c7, cpssp, &port_c7_funcs);

	cpssp->port_d0 = port_d0;
	sig_std_logic_connect_out(port_d0, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_d0, cpssp, &port_d0_funcs);
	cpssp->port_d1 = port_d1;
	sig_std_logic_connect_out(port_d1, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_d1, cpssp, &port_d1_funcs);
	cpssp->port_d2 = port_d2;
	sig_std_logic_connect_out(port_d2, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_d2, cpssp, &port_d2_funcs);
	cpssp->port_d3 = port_d3;
	sig_std_logic_connect_out(port_d3, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_d3, cpssp, &port_d3_funcs);
	cpssp->port_d4 = port_d4;
	sig_std_logic_connect_out(port_d4, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_d4, cpssp, &port_d4_funcs);
	cpssp->port_d5 = port_d5;
	sig_std_logic_connect_out(port_d5, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_d5, cpssp, &port_d5_funcs);
	cpssp->port_d6 = port_d6;
	sig_std_logic_connect_out(port_d6, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_d6, cpssp, &port_d6_funcs);
	cpssp->port_d7 = port_d7;
	sig_std_logic_connect_out(port_d7, cpssp, SIG_STD_LOGIC_Z);
	sig_std_logic_connect_in(port_d7, cpssp, &port_d7_funcs);

	CHIP_(reset)(cpssp); /* FIXME */

	cpssp->process.inst_hz = 8*1000*1000;

	sched_process_init(&cpssp->process, CHIP_(clk), cpssp);

	return cpssp;
}

void
CHIP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	timer1_destroy(cpssp);
	int2_destroy(cpssp);
	int1_destroy(cpssp);
	int0_destroy(cpssp);
	gpiod_destroy(cpssp);
	gpioc_destroy(cpssp);
	gpiob_destroy(cpssp);
	gpioa_destroy(cpssp);
	adc_destroy(cpssp);
	cpu_destroy(cpssp);

	shm_free(cpssp);
}

void
CHIP_(suspend)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
CHIP_(resume)(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}

