/*
 * Copyright (C) 2003-2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef STATE
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT
struct {
	/* 9-24 */
	uint8_t e;	/* MTRR Enable */
	uint8_t fe;	/* Fixed Range Registers Enable */
	uint8_t type;	/* Default Memory Type */

	/* 9-25 */
	uint8_t fix64k[8];	/* Memory Type 0x00000-0x7ffff */
	uint8_t fix16k[16];	/* Memory Type 0x80000-0xbffff */
	uint8_t fix4k[64];	/* Memory Type 0xc0000-0xfffff */

	/* 9-26 */
	struct {
		uint8_t valid;
		uint8_t type;
		uint64_t base;
		uint64_t mask;
	} var[8];
} NAME_(mtrr);
#endif
#endif /* STATE */

#ifdef BEHAVIOR
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT

#include "arch_gen_cpu_x86_mtrr.h"

enum NAME_(mtrr_mem_type) {
	NAME_(MTRR_UC) = 0,
	NAME_(MTRR_WC) = 1,
	NAME_(MTRR_WT) = 4,
	NAME_(MTRR_WP) = 5,
	NAME_(MTRR_WB) = 6,
};

static uint8_t
NAME_(mtrr_type)(struct cpssp *cpssp, Paddr paddr)
{
	/* What about overlapping regions? FIXME */
	int i;

	if (! cpssp->NAME_(mtrr).e) {
		return NAME_(MTRR_UC);
	}

	if (cpssp->NAME_(mtrr).fe) {
		if (paddr < 0x80000) {
			return cpssp->NAME_(mtrr).fix64k[(paddr - 0x00000) / 0x10000];
		} else if (paddr < 0xc0000) {
			return cpssp->NAME_(mtrr).fix16k[(paddr - 0x80000) / 0x4000];
		} else if (paddr < 0x100000) {
			return cpssp->NAME_(mtrr).fix4k[(paddr - 0xc0000) / 0x1000];
		}
	}

	for (i = 0; i < 8; i++) {
		if (cpssp->NAME_(mtrr).var[i].valid
		 && cpssp->NAME_(mtrr).var[i].base
				== (paddr & cpssp->NAME_(mtrr).var[i].mask)) {
			return cpssp->NAME_(mtrr).var[i].type;
		}
	}

	/* Default memory type. */
	return cpssp->NAME_(mtrr).type;
}

int
NAME_(mtrr_rdmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t *valp)
{
	int o;
	int i;

	switch (ecx) {
	case MSR_IA32_MTRRCAP: /* 0xfe */
		/* 9-23 */
		*valp = (1 << 10) /* Write combining supported. */
		      | (1 << 8)  /* Fixed range regs supported. */
		      | (8 << 0); /* 8 Variable range regs supported. */
		return 1;

	case MSR_IA32_MTRR_DEF_TYPE: /* 0x2ff */
		/* 9-24 */
		*valp = (cpssp->NAME_(mtrr).e << 11)
			| (cpssp->NAME_(mtrr).fe << 10)
			| (cpssp->NAME_(mtrr).type << 0);
		return 1;

	case MSR_IA32_MTRR_FIX64K_00000: /* 0x250 */
		/* 9-25 */
		o = 0;
		*valp = 0;
		for (i = 0; i < 8; i++) {
			*valp |= (uint64_t) cpssp->NAME_(mtrr).fix64k[o + i] << (i*8);
		}
		return 1;

	case MSR_IA32_MTRR_FIX16K_80000: /* 0x258 */
	case MSR_IA32_MTRR_FIX16K_A0000: /* 0x259 */
		/* 9-25 */
		o = (ecx & 1) * 8;
		*valp = 0;
		for (i = 0; i < 8; i++) {
			*valp |= (uint64_t) cpssp->NAME_(mtrr).fix16k[o + i] << (i*8);
		}
		return 1;

	case MSR_IA32_MTRR_FIX4K_C0000: /* 0x268 */
	case MSR_IA32_MTRR_FIX4K_C8000: /* 0x269 */
	case MSR_IA32_MTRR_FIX4K_D0000: /* 0x26a */
	case MSR_IA32_MTRR_FIX4K_D8000: /* 0x26b */
	case MSR_IA32_MTRR_FIX4K_E0000: /* 0x26c */
	case MSR_IA32_MTRR_FIX4K_E8000: /* 0x26d */
	case MSR_IA32_MTRR_FIX4K_F0000: /* 0x26e */
	case MSR_IA32_MTRR_FIX4K_F8000: /* 0x26f */
		/* 9-25 */
		o = (ecx & 3) * 8;
		*valp = 0;
		for (i = 0; i < 8; i++) {
			*valp |= (uint64_t) cpssp->NAME_(mtrr).fix4k[o + i] << (i*8);
		}
		return 1;

	case MSR_IA32_MTRR_PHYSBASE0: /* 0x200 */
	case MSR_IA32_MTRR_PHYSBASE1: /* 0x202 */
	case MSR_IA32_MTRR_PHYSBASE2: /* 0x204 */
	case MSR_IA32_MTRR_PHYSBASE3: /* 0x206 */
	case MSR_IA32_MTRR_PHYSBASE4: /* 0x208 */
	case MSR_IA32_MTRR_PHYSBASE5: /* 0x20a */
	case MSR_IA32_MTRR_PHYSBASE6: /* 0x20c */
	case MSR_IA32_MTRR_PHYSBASE7: /* 0x20e */
		/* 9-26 */
		o = (ecx >> 1) & 3;
		*valp = (cpssp->NAME_(mtrr).var[o].base << (12-12))
			| (cpssp->NAME_(mtrr).var[o].type << 0);
		return 1;

	case MSR_IA32_MTRR_PHYSMASK0: /* 0x201 */
	case MSR_IA32_MTRR_PHYSMASK1: /* 0x203 */
	case MSR_IA32_MTRR_PHYSMASK2: /* 0x205 */
	case MSR_IA32_MTRR_PHYSMASK3: /* 0x207 */
	case MSR_IA32_MTRR_PHYSMASK4: /* 0x209 */
	case MSR_IA32_MTRR_PHYSMASK5: /* 0x20b */
	case MSR_IA32_MTRR_PHYSMASK6: /* 0x20d */
	case MSR_IA32_MTRR_PHYSMASK7: /* 0x20f */
		/* 9-26 */
		o = (ecx >> 1) & 3;
		*valp = (cpssp->NAME_(mtrr).var[o].mask << (12-12))
			| (cpssp->NAME_(mtrr).var[o].valid << 11);
		return 1;
	}
	return 0;
}

int
NAME_(mtrr_wrmsr)(struct cpssp *cpssp, uint32_t ecx, uint64_t val)
{
	int o;
	int i;

	switch (ecx) {
	case MSR_IA32_MTRRCAP: /* 0xfe */
		/* 9-23 */
		/* Read-only Register */
		return 0;

	case MSR_IA32_MTRR_DEF_TYPE: /* 0x2ff */
		/* 9-24 */
		if ((val & 0xfffffffffffff300ULL) != 0) {
			/* Reserved bits set. */
			return 0;
		}
		switch ((int) (val >> 0) & 0xff) {
		case NAME_(MTRR_UC):
		case NAME_(MTRR_WC):
		case NAME_(MTRR_WT):
		case NAME_(MTRR_WP):
		case NAME_(MTRR_WB):
			break;
		default:
			/* Bad default memory type. */
			return 0;
		}
		cpssp->NAME_(mtrr).e = (val >> 11) & 1;
		cpssp->NAME_(mtrr).fe = (val >> 10) & 1;
		cpssp->NAME_(mtrr).type = (val >> 0) & 0xff;
		break;

	case MSR_IA32_MTRR_FIX64K_00000: /* 0x250 */
		/* 9-25 */
		o = 0;
		/* Check types. */
		for (i = 0; i < 8; i++) {
			switch ((val >> (i*8)) & 0xff) {
			case NAME_(MTRR_UC):
			case NAME_(MTRR_WC):
			case NAME_(MTRR_WT):
			case NAME_(MTRR_WP):
			case NAME_(MTRR_WB):
				break;
			default:
				/* Bad memory type. */
				return 0;
			}
		}
		for (i = 0; i < 8; i++) {
			cpssp->NAME_(mtrr).fix64k[o + i] = (val >> (i*8)) & 0xff;
		}
		return 0;

	case MSR_IA32_MTRR_FIX16K_80000: /* 0x258 */
	case MSR_IA32_MTRR_FIX16K_A0000: /* 0x259 */
		/* 9-25 */
		o = (ecx & 1) * 8;
		/* Check types. */
		for (i = 0; i < 8; i++) {
			switch ((val >> (i*8)) & 0xff) {
			case NAME_(MTRR_UC):
			case NAME_(MTRR_WC):
			case NAME_(MTRR_WT):
			case NAME_(MTRR_WP):
			case NAME_(MTRR_WB):
				break;
			default:
				/* Bad memory type. */
				return 0;
			}
		}
		for (i = 0; i < 8; i++) {
			cpssp->NAME_(mtrr).fix16k[o + i] = (val >> (i*8)) & 0xff;
		}
		return 0;

	case MSR_IA32_MTRR_FIX4K_C0000: /* 0x268 */
	case MSR_IA32_MTRR_FIX4K_C8000: /* 0x269 */
	case MSR_IA32_MTRR_FIX4K_D0000: /* 0x26a */
	case MSR_IA32_MTRR_FIX4K_D8000: /* 0x26b */
	case MSR_IA32_MTRR_FIX4K_E0000: /* 0x26c */
	case MSR_IA32_MTRR_FIX4K_E8000: /* 0x26d */
	case MSR_IA32_MTRR_FIX4K_F0000: /* 0x26e */
	case MSR_IA32_MTRR_FIX4K_F8000: /* 0x26f */
		/* 9-25 */
		o = (ecx & 3) * 8;
		/* Check types. */
		for (i = 0; i < 8; i++) {
			switch ((val >> (i*8)) & 0xff) {
			case NAME_(MTRR_UC):
			case NAME_(MTRR_WC):
			case NAME_(MTRR_WT):
			case NAME_(MTRR_WP):
			case NAME_(MTRR_WB):
				break;
			default:
				/* Bad memory type. */
				return 0;
			}
		}
		for (i = 0; i < 8; i++) {
			cpssp->NAME_(mtrr).fix4k[o + i] = (val >> (i*8)) & 0xff;
		}
		return 0;

	case MSR_IA32_MTRR_PHYSBASE0: /* 0x200 */
	case MSR_IA32_MTRR_PHYSBASE1: /* 0x202 */
	case MSR_IA32_MTRR_PHYSBASE2: /* 0x204 */
	case MSR_IA32_MTRR_PHYSBASE3: /* 0x206 */
	case MSR_IA32_MTRR_PHYSBASE4: /* 0x208 */
	case MSR_IA32_MTRR_PHYSBASE5: /* 0x20a */
	case MSR_IA32_MTRR_PHYSBASE6: /* 0x20c */
	case MSR_IA32_MTRR_PHYSBASE7: /* 0x20e */
		/* 9-26 */
		o = (ecx >> 1) & 3;
		/* Check reserved bits. */
		if ((val >> CONFIG_CPU_PHYS_BITS)
		 || (val & 0xf00)) {
			/* Reserved bits set. */
			return 0;
		}
		/* Check memory type. */
		switch ((int) (val >> 0) & 0xff) {
		case NAME_(MTRR_UC):
		case NAME_(MTRR_WC):
		case NAME_(MTRR_WT):
		case NAME_(MTRR_WP):
		case NAME_(MTRR_WB):
			break;
		default:
			/* Bad memory type. */
			return 0;
		}
		cpssp->NAME_(mtrr).var[o].base = val & 0xffffff000ULL;
		cpssp->NAME_(mtrr).var[o].type = (val >> 0) & 0xff;
		return 1;

	case MSR_IA32_MTRR_PHYSMASK0: /* 0x201 */
	case MSR_IA32_MTRR_PHYSMASK1: /* 0x203 */
	case MSR_IA32_MTRR_PHYSMASK2: /* 0x205 */
	case MSR_IA32_MTRR_PHYSMASK3: /* 0x207 */
	case MSR_IA32_MTRR_PHYSMASK4: /* 0x209 */
	case MSR_IA32_MTRR_PHYSMASK5: /* 0x20b */
	case MSR_IA32_MTRR_PHYSMASK6: /* 0x20d */
	case MSR_IA32_MTRR_PHYSMASK7: /* 0x20f */
		/* 9-26 */
		o = (ecx >> 1) & 3;
		/* Check reserved bits. */
		if ((val >> CONFIG_CPU_PHYS_BITS)
		 || (val & 0x7ff)) {
			/* Reserved bits set. */
			return 0;
		}
		cpssp->NAME_(mtrr).var[o].mask = val & 0xffffff000ULL;
		cpssp->NAME_(mtrr).var[o].valid = (val >> 11) & 1;
		return 1;
	}
	return 0;
}

void
NAME_(mtrr_reset)(struct cpssp *cpssp)
{
	cpssp->NAME_(mtrr).e = 0;
	cpssp->NAME_(mtrr).fe = 0;
}

static void
NAME_(mtrr_create)(struct cpssp *cpssp)
{
}

static void
NAME_(mtrr_destroy)(struct cpssp *cpssp)
{
}
#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT */

#endif /* BEHAVIOR */
