/*
 * Derived from QEMU sources.
 *  
 *  Copyright (c) 2005-2014 FAUmachine Team.
 *  Copyright (c) 2003 Fabrice Bellard.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 */

/* FIXME */
#if defined(CONFIG_CPU_TLBD_SIZE)
/* MMU with TLB for all sizes. */
#define SETS	CONFIG_CPU_TLBD_ASSOC
#define LINES	(CONFIG_CPU_TLBD_SIZE / SETS)

#elif defined(CONFIG_CPU_TLBD4M_SIZE) && defined(CONFIG_CPU_TLBD4K_SIZE)
/* MMU with distinct TLBs for 4K/4M sizes. */
#define SETS4M	CONFIG_CPU_TLBD4M_ASSOC
#define LINES4M	(CONFIG_CPU_TLBD4M_SIZE / SETS4M)
#define SETS4K	CONFIG_CPU_TLBD4K_ASSOC
#define LINES4K	(CONFIG_CPU_TLBD4K_SIZE / SETS4K)

#else
/* No MMU - FIXME */
#define SETS4M	4
#define LINES4M	4
#endif

#ifdef STATE
struct {
#ifdef LINES
	struct {
		Vaddr vaddr;
		Paddr paddr;
		Paddr psize;
		uint8_t gflag;
		uint8_t wflag;
		uint8_t uflag;
		uint8_t type; /* UC, WB, ... */
	} set[SETS];
	LRU_DECL(SETS, lru);
#endif
#ifdef LINES4M
	struct {
		struct {
			Vaddr vaddr;
			Paddr paddr;
			uint8_t gflag;
			uint8_t wflag;
			uint8_t uflag;
			uint8_t type; /* UC, WB, ... */
		} set[SETS4M];
		LRU_DECL(SETS4M, lru);
	} tlb4M[LINES4M];
#endif
#ifdef LINES4K
	struct {
		struct {
			Vaddr vaddr;
			Paddr paddr;
			uint8_t gflag;
			uint8_t wflag;
			uint8_t uflag;
			uint8_t type; /* UC, WB, ... */
		} set[SETS4K];
		LRU_DECL(SETS4K, lru);
	} tlb4K[LINES4K];
#endif
} NAME_(mmud);
#endif

#ifdef BEHAVIOR
#define PHYS_ADDR_MASK 0xfffff000

static uint32_t
NAME_(mmud_mrl)(struct cpssp *cpssp, Paddr pa)
{
	Data val;

	NAME_(a20gate_mr)(cpssp, pa & ~(sizeof(Data) - 1),
	0b1111 << (pa & (sizeof(Data) - 1)), &val);

	return (val >> ((pa & (sizeof(Data) - 1)) * 8)) & 0xffffffff;
}

static void
NAME_(mmud_mwl)(struct cpssp *cpssp, Paddr pa, uint32_t val)
{
	Data valX;

	assert(! (pa & 3));

	valX = ((Data) val) << ((pa & (sizeof(Data) - 1)) * 8);

	NAME_(a20gate_mw)(cpssp, pa & ~(pa & (sizeof(Data) - 1)),
			0b1111 << (pa & (sizeof(Data) - 1)), valX);
}

static uint64_t
NAME_(mmud_mrq)(struct cpssp *cpssp, Paddr pa)
{
	uint32_t val1;
	uint32_t val2;

	assert(! (pa & 3));
	val1 = NAME_(mmud_mrl)(cpssp, pa + 0);
	val2 = NAME_(mmud_mrl)(cpssp, pa + 4);

	return (((uint64_t) val2) << 32) | val1;
}

static int
NAME_(mmud_check)(struct cpssp *cpssp, uint32_t ptep, int want_wflag, int want_uflag)
{
	if (want_uflag) {
		if (! (ptep & PG_USER_MASK))
			return 1;
		if (want_wflag
		 && ! (ptep & PG_RW_MASK))
			return 1;
	} else {
		if ((cpssp->cr[0] & CPU_CR0_WP_MASK)
		 && want_wflag
		 && !(ptep & PG_RW_MASK))
			return 1;
	}
	return 0;
}

static void
NAME_(mmud_touch)(struct cpssp *cpssp, Paddr pte_addr, uint32_t pte, int want_wflag)
{
	int is_dirty;

	is_dirty = want_wflag && ! (pte & PG_DIRTY_MASK);
	if (! (pte & PG_ACCESSED_MASK)
	 || is_dirty) {
		pte |= PG_ACCESSED_MASK;
		if (is_dirty)
			pte |= PG_DIRTY_MASK;
		NAME_(mmud_mwl)(cpssp, pte_addr, pte);
	}
}

static uint8_t
NAME_(mmud_cdwr_to_type)(uint32_t ptep)
{
	static const uint8_t type[] = {
		/* 00: WB */	0b110,
		/* 01: WT */	0b100,
		/* 10: WC */	0b001,
		/* 11: UC */	0b000,
	};
	unsigned int i;

	i = ((ptep >> PG_PWT_BIT) & 1) << 0
		| ((ptep >> PG_PCD_BIT) & 1) << 1;
	return type[i];
}

/*
 * Return value:
 * 0 = nothing more to do.
 * 1 = generate page-fault.
 */
static int
NAME_(__mmud_fault)(
	struct cpssp *cpssp,
	Vaddr addr,
	int want_wflag,
	int want_uflag,
	Paddr *paddrp,
	Paddr *psizep,
	uint8_t *gflagp,
	uint8_t *typep
)
{
	uint32_t pde_addr;
	uint32_t pdpe_addr;
	uint32_t pte_addr;
	uint64_t pte;
	uint64_t ptep;
	int error_code;
	int gflag;
	int uflag;
	int wflag;
	int dflag;
	int page_size;
	unsigned long paddr;

	if (! (cpssp->cr[0] & CPU_CR0_PG_MASK)) {
		pte = addr & ~0x3fffff;
		uflag = 0;
		wflag = 1;
		dflag = 1;
		page_size = 1024*4096;
		ptep = (0 << PG_GLOBAL_BIT)
			| (0 << PG_PCD_BIT)
			| (0 << PG_PWT_BIT)
			| (1 << PG_USER_BIT)
			| (1 << PG_RW_BIT)
			| (1 << PG_PRESENT_BIT);
		goto do_mapping;
	}

	if (cpssp->cr[4] & CPU_CR4_PAE_MASK) { /* FIXME fox the whole if part needs reworking */
		uint64_t pde, pdpe;

		/* XXX: we only use 32 bit physical addresses */
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
		if (cpssp->hflags & HF_LMA_MASK) {
			uint32_t pml4e_addr;
			uint64_t pml4e;
			int32_t sext;
			
			/* test virtual address sign extension */
			sext = (int64_t) addr >> 47;
			if (sext != 0 && sext != -1) {
				cpssp->error_code = 0;
				cpssp->exception_index = CPU_FAULT_GP;
				return 1;
			}
			
			pml4e_addr = ((cpssp->cr[3] & ~0xfff) + (((addr >> 39) & 0x1ff) << 3));
			pml4e = NAME_(mmud_mrq)(cpssp, pml4e_addr);
			if (!(pml4e & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			if (!(cpssp->efer & MSR_EFER_NXE) && (pml4e & PG_NX_MASK)) {
				error_code = PG_ERROR_RSVD_MASK;
				goto do_fault;
			}
			NAME_(mmud_touch)(cpssp, pml4e_addr, pml4e, 0);
			ptep = pml4e ^ PG_NX_MASK;
			pdpe_addr = ((pml4e & PHYS_ADDR_MASK) + (((addr >> 30) & 0x1ff) << 3));
			pdpe = NAME_(mmud_mrq)(cpssp, pdpe_addr);
			if (!(pdpe & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			if (!(cpssp->efer & MSR_EFER_NXE) && (pdpe & PG_NX_MASK)) {
				error_code = PG_ERROR_RSVD_MASK;
				goto do_fault;
			}
			ptep &= pdpe ^ PG_NX_MASK;
			NAME_(mmud_touch)(cpssp, pdpe_addr, pdpe, 0);
		} else
#endif /* CONFIG_CPU_LM_SUPPORT */
		{
			/* XXX: load them when cr3 is loaded ? */
			pdpe_addr = ((cpssp->cr[3] & ~0x1f) + ((addr >> 30) << 3));
			pdpe = NAME_(mmud_mrq)(cpssp, pdpe_addr);
			if (!(pdpe & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			ptep = PG_NX_MASK | PG_USER_MASK | PG_RW_MASK;
		}

		pde_addr = ((pdpe & PHYS_ADDR_MASK) + (((addr >> 21) & 0x1ff) << 3));
		pde = NAME_(mmud_mrq)(cpssp, pde_addr);
		if (!(pde & PG_PRESENT_MASK)) {
			error_code = 0;
			goto do_fault;
		}
		if (!(cpssp->efer & MSR_EFER_NXE) && (pde & PG_NX_MASK)) {
			error_code = PG_ERROR_RSVD_MASK;
			goto do_fault;
		}
		ptep &= pde ^ PG_NX_MASK;
		if (pde & PG_PSE_MASK) {
			/* 2 MB page */
			page_size = 2048 * 1024;
			ptep ^= PG_NX_MASK;
#if 0 /* FIXME? fox - what is iswrite == 2? */
			if ((ptep & PG_NX_MASK) && is_write1 == 2)
				goto do_fault_protect;
#endif
			if (NAME_(mmud_check)(cpssp, ptep, want_wflag, want_uflag)) {
				goto do_fault_protect;
			}
			NAME_(mmud_touch)(cpssp, pde_addr, pde, want_wflag);
			/* align to page_size */
			pte = pde & ((PHYS_ADDR_MASK & ~(page_size - 1)) | 0xfff); 
		} else {
			/* 4 KB page */
			NAME_(mmud_touch)(cpssp, pde_addr, pde, 0);
			pte_addr = ((pde & PHYS_ADDR_MASK) + (((addr >> 12) & 0x1ff) << 3));
			pte = NAME_(mmud_mrq)(cpssp, pte_addr);
			if (!(pte & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			if (!(cpssp->efer & MSR_EFER_NXE) && (pte & PG_NX_MASK)) {
				error_code = PG_ERROR_RSVD_MASK;
				goto do_fault;
			}
			/* combine pde and pte nx, user and rw protections */
			ptep &= pte ^ PG_NX_MASK;
			ptep ^= PG_NX_MASK;
#if 0 /* FIXME? fox */
			if ((ptep & PG_NX_MASK) && is_write1 == 2)
				goto do_fault_protect;
#endif
			if (NAME_(mmud_check)(cpssp, ptep, want_wflag, want_uflag)) {
				goto do_fault_protect;
			}
			NAME_(mmud_touch)(cpssp, pte_addr, pte, want_wflag);
			page_size = 4096;
			pte = pte & (PHYS_ADDR_MASK | 0xfff);
		}
	} else { /* End of if that needs FIXME fox */
		uint32_t pde;

		/* Page directory entry. */
		pde_addr = ((cpssp->cr[3] & ~0xfff) + ((addr >> 20) & ~3));
		pde = NAME_(mmud_mrl)(cpssp, pde_addr);
		if (! (pde & PG_PRESENT_MASK)) {
			error_code = 0;
			goto do_fault;
		}
		/* If PSE bit is set, then we use a 4MB page. */
		if ((pde & PG_PSE_MASK)
		 && (cpssp->cr[4] & CPU_CR4_PSE_MASK)) {
			if (NAME_(mmud_check)(cpssp, pde, want_wflag, want_uflag)) {
				goto do_fault_protect;
			}
			NAME_(mmud_touch)(cpssp, pde_addr, pde, want_wflag);
	
			pte = pde & ~0x003ff000; /* align to 4MB */
			ptep = pte;
			page_size = 4096 * 1024;
	
		} else {
			NAME_(mmud_touch)(cpssp, pde_addr, pde, 0);
	
			/* Page table entry. */
			pte_addr = ((pde & ~0xfff) + ((addr >> 10) & 0xffc));
			pte = NAME_(mmud_mrl)(cpssp, pte_addr);
			if (! (pte & PG_PRESENT_MASK)) {
				error_code = 0;
				goto do_fault;
			}
			/* Combine pde and pte user and rw protections. */
			ptep = pte & pde;
			if (NAME_(mmud_check)(cpssp, ptep, want_wflag, want_uflag)) {
				goto do_fault_protect;
			}
			NAME_(mmud_touch)(cpssp, pte_addr, pte, want_wflag);
	
			page_size = 4096;
		}
	}

do_mapping:
	paddr = pte & TARGET_PAGE_MASK;
	gflag = (ptep >> PG_GLOBAL_BIT) & (cpssp->cr[4] >> CPU_CR4_PGE_SHIFT) & 1;

	*paddrp = paddr;
	*psizep = page_size;
	*gflagp = gflag;
	*typep = NAME_(mmud_cdwr_to_type)(ptep);
	return 0;

do_fault_protect:
	error_code = PG_ERROR_P_MASK;
do_fault:
	cpssp->cr[2] = addr;
	cpssp->error_code = error_code;
	cpssp->error_code |= want_wflag << PG_ERROR_W_BIT;
	cpssp->error_code |= want_uflag << PG_ERROR_U_BIT;
	return 1;
}

#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT
static uint8_t
NAME_(mmud_combine_type)(uint8_t pat, uint8_t mtrr)
{
	/* 9-16 */
	uint8_t comb[8][8] = {
	/*        UC,WC,--,--,WT,WP,WB,UC- */
	/*UC:0*/ { 0, 0,-0,-0, 0, 0, 0, 0 },
	/*WC:1*/ { 0, 1,-0,-0,-0,-0, 1, 1 },
	/*--:2*/ {-0,-0,-0,-0,-0,-0,-0,-0 },
	/*--:3*/ {-0,-0,-0,-0,-0,-0,-0,-0 },
	/*WT:4*/ { 0, 1,-0,-0, 4,-0, 4, 0 },
	/*WP:5*/ { 0, 1,-0,-0,-0, 5, 5,-0 },
	/*WB:6*/ { 0, 1,-0,-0, 4, 5, 6, 0 },
	/*UC-:7*/{-0,-0,-0,-0,-0,-0,-0,-0 },
	};

	return comb[mtrr][pat];
}
#endif

#ifdef LINES4M
static int
NAME_(mmud_hash4M)(Vaddr vaddr)
{
	return (vaddr / 1024 / 4096) % LINES4M;
}
#endif /* LINES4M */

#ifdef LINES4K
static int
NAME_(mmud_hash4K)(Vaddr vaddr)
{
	return (vaddr / 4096) % LINES4K;
}
#endif /* LINES4K */

static int
NAME_(_mmud_fault)(
	struct cpssp *cpssp,
	Vaddr vaddr,
	int want_wflag,
	int want_uflag,
	Paddr *paddrp,
	Paddr *psizep
)
{
#if defined(LINES4M) || defined(LINES4K)
	int line;
#endif
	int set;
	Paddr paddr;
	Paddr psize;
	uint8_t gflag;
	uint8_t type;

#ifdef LINES
	/* Search for page in common-TLB. */
	for (set = 0; set < SETS; set++) {
		if ((vaddr & ~(cpssp->NAME_(mmud).set[set].psize - 1))
				== cpssp->NAME_(mmud).set[set].vaddr) {
			if (cpssp->NAME_(mmud).set[set].wflag < want_wflag
			 || cpssp->NAME_(mmud).set[set].uflag < want_uflag) {
				if (NAME_(__mmud_fault)(cpssp, vaddr, want_wflag, want_uflag,
						&paddr, &psize, &gflag, &type)) {
					return 1;
				}
				cpssp->NAME_(mmud).set[set].wflag = want_wflag;
				cpssp->NAME_(mmud).set[set].uflag = want_uflag;
			}
		found:	;
			NAME_(lru_use)(SETS, cpssp->NAME_(mmud).lru, set);
			*paddrp = cpssp->NAME_(mmud).set[set].paddr;
			*psizep = cpssp->NAME_(mmud).set[set].psize;
			return 0;
		}
	}
#endif

#ifdef LINES4M
	/* Search for page in 4M-page-TLB. */
	line = NAME_(mmud_hash4M)(vaddr);
	for (set = 0; set < SETS4M; set++) {
		if ((vaddr & ~0x3fffff) == cpssp->NAME_(mmud).tlb4M[line].set[set].vaddr) {
			if (cpssp->NAME_(mmud).tlb4M[line].set[set].wflag < want_wflag
			 || cpssp->NAME_(mmud).tlb4M[line].set[set].uflag < want_uflag) {
				if (NAME_(__mmud_fault)(cpssp, vaddr, want_wflag, want_uflag,
						&paddr, &psize, &gflag, &type)) {
					return 1;
				}
				cpssp->NAME_(mmud).tlb4M[line].set[set].wflag = want_wflag;
				cpssp->NAME_(mmud).tlb4M[line].set[set].uflag = want_uflag;
			}
		found4M:;
			NAME_(lru_use)(SETS4M, cpssp->NAME_(mmud).tlb4M[line].lru, set);
			*paddrp = cpssp->NAME_(mmud).tlb4M[line].set[set].paddr;
			*psizep = 1024*4096;
			return 0;
		}
	}
#endif /* LINES4M */

#ifdef LINES4K
	/* Search for page in 4K-page-TLB. */
	line = NAME_(mmud_hash4K)(vaddr);
	for (set = 0; set < SETS4K; set++) {
		if ((vaddr & ~0xfff) == cpssp->NAME_(mmud).tlb4K[line].set[set].vaddr) {
			if (cpssp->NAME_(mmud).tlb4K[line].set[set].wflag < want_wflag
			 || cpssp->NAME_(mmud).tlb4K[line].set[set].uflag < want_uflag) {
				if (NAME_(__mmud_fault)(cpssp, vaddr, want_wflag, want_uflag,
						&paddr, &psize, &gflag, &type)) {
					return 1;
				}
				cpssp->NAME_(mmud).tlb4K[line].set[set].wflag = want_wflag;
				cpssp->NAME_(mmud).tlb4K[line].set[set].uflag = want_uflag;
			}
		found4K:;
			NAME_(lru_use)(SETS4K, cpssp->NAME_(mmud).tlb4K[line].lru, set);
			*paddrp = cpssp->NAME_(mmud).tlb4K[line].set[set].paddr;
			*psizep = 4096;
			return 0;
		}
	}
#endif /* LINES4K */

	/* Not found. */
	if (NAME_(__mmud_fault)(cpssp, vaddr, want_wflag, want_uflag,
			&paddr, &psize, &gflag, &type)) {
		return 1;
	}
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT
	type = NAME_(mmud_combine_type)(type, NAME_(mtrr_type)(cpssp, paddr));
#endif

#ifdef LINES
	if (psize == 1024*4096
	 || psize ==  512*4096
	 || psize ==    1*4096) {
		set = NAME_(lru_oldest)(SETS, cpssp->NAME_(mmud).lru);
		cpssp->NAME_(mmud).set[set].vaddr = (vaddr & ~0xfff);
		cpssp->NAME_(mmud).set[set].paddr = paddr;
		cpssp->NAME_(mmud).set[set].psize = psize;
		cpssp->NAME_(mmud).set[set].gflag = gflag;
		cpssp->NAME_(mmud).set[set].wflag = want_wflag;
		cpssp->NAME_(mmud).set[set].uflag = want_uflag;
		cpssp->NAME_(mmud).set[set].type = type;
		goto found;
	}
#endif /* LINES */
#ifdef LINES4M
	if (psize == 1024*4096) {
		line = NAME_(mmud_hash4M)(vaddr);
		set = NAME_(lru_oldest)(SETS4M, cpssp->NAME_(mmud).tlb4M[line].lru);
		cpssp->NAME_(mmud).tlb4M[line].set[set].vaddr = (vaddr & ~0xfff);
		cpssp->NAME_(mmud).tlb4M[line].set[set].paddr = paddr;
		cpssp->NAME_(mmud).tlb4M[line].set[set].gflag = gflag;
		cpssp->NAME_(mmud).tlb4M[line].set[set].wflag = want_wflag;
		cpssp->NAME_(mmud).tlb4M[line].set[set].uflag = want_uflag;
		cpssp->NAME_(mmud).tlb4M[line].set[set].type = type;
		goto found4M;

	} else
#endif /* LINES4M */
#ifdef LINES4K
	if (psize == 4096) {
		line = NAME_(mmui_hash4K)(vaddr);
		set = NAME_(lru_oldest)(SETS4K, cpssp->NAME_(mmud).tlb4K[line].lru);
		cpssp->NAME_(mmud).tlb4K[line].set[set].vaddr = (vaddr & ~0xfff);
		cpssp->NAME_(mmud).tlb4K[line].set[set].paddr = paddr;
		cpssp->NAME_(mmud).tlb4K[line].set[set].gflag = gflag;
		cpssp->NAME_(mmud).tlb4K[line].set[set].wflag = want_wflag;
		cpssp->NAME_(mmud).tlb4K[line].set[set].uflag = want_uflag;
		cpssp->NAME_(mmud).tlb4K[line].set[set].type = type;
		goto found4K;

	} else
#endif /* LINES4K */
	{
		assert(0);
	}
}

/*
 * Return value:
 * 0 = nothing more to do.
 * 1 = generate page-fault.
 */
static int
NAME_(mmud_map)(
	struct cpssp *cpssp,
	Vaddr addr,
	int want_wflag,
	int want_uflag,
	Paddr *paddrp
)
{
	Paddr paddr;
	Paddr psize;
	int ret;

	ret = NAME_(_mmud_fault)(cpssp, addr, want_wflag, want_uflag, &paddr, &psize);
	if (! ret) {
		paddr |= addr & (psize - 1);
		*paddrp = paddr;
	}

	return ret;
}

void
NAME_(mmud_invlpg)(struct cpssp *cpssp, Vaddr vaddr)
{
#if defined(LINES4M) || defined(LINES4K)
	int line;
#endif
	int set;

#ifdef LINES
	for (set = 0; set < SETS; set++) {
		if ((vaddr & ~(cpssp->NAME_(mmud).set[set].psize - 1))
				== cpssp->NAME_(mmud).set[set].vaddr) {
			cpssp->NAME_(mmud).set[set].vaddr = (Vaddr) -1; /* Invalid */
		}
	}
#endif
#ifdef LINES4M
	line = NAME_(mmud_hash4M)(vaddr);
	for (set = 0; set < SETS4M; set++) {
		if ((vaddr & ~0x3fffff)
				== cpssp->NAME_(mmud).tlb4M[line].set[set].vaddr) {
			cpssp->NAME_(mmud).tlb4M[line].set[set].vaddr = (Vaddr) -1; /* Invalid */
		}
	}
#endif
#ifdef LINES4K
	line = NAME_(mmud_hash4K)(vaddr);
	for (set = 0; set < SETS4K; set++) {
		if ((vaddr & ~0xfff)
				== cpssp->NAME_(mmud).tlb4K[line].set[set].vaddr) {
			cpssp->NAME_(mmud).tlb4K[line].set[set].vaddr = (Vaddr) -1; /* Invalid */
		}
	}
#endif

	NAME_(cache1i_invlpg)(cpssp, vaddr);
	NAME_(cache1d_invlpg)(cpssp, vaddr);
}

void
NAME_(mmud_flush_all)(struct cpssp *cpssp, int flush_global)
{
#if defined(LINES4M) || defined(LINES4K)
	int line;
#endif
	int set;

#ifdef LINES
	{
		for (set = 0; set < SETS; set++) {
			if (flush_global
			 || ! cpssp->NAME_(mmud).set[set].gflag) {
				cpssp->NAME_(mmud).set[set].vaddr
						= (Vaddr) -1; /* Invalid */
			}
		}
	}
#endif
#ifdef LINES4M
	for (line = 0; line < LINES4M; line++) {
		for (set = 0; set < SETS4M; set++) {
			if (flush_global
			 || ! cpssp->NAME_(mmud).tlb4M[line].set[set].gflag) {
				cpssp->NAME_(mmud).tlb4M[line].set[set].vaddr
						= (Vaddr) -1; /* Invalid */
			}
		}
	}
#endif /* LINES4M */
#ifdef LINES4K
	for (line = 0; line < LINES4K; line++) {
		for (set = 0; set < SETS4K; set++) {
			if (flush_global
			 || ! cpssp->NAME_(mmud).tlb4K[line].set[set].gflag) {
				cpssp->NAME_(mmud).tlb4K[line].set[set].vaddr
						= (Vaddr) -1; /* Invalid */
			}
		}
	}
#endif /* LINES4K */

	NAME_(cache1i_flush_all)(cpssp, flush_global);
	NAME_(cache1d_flush_all)(cpssp, flush_global);
}

void
NAME_(mmud_unmap)(struct cpssp *cpssp, unsigned long pa, unsigned long len)
{
	NAME_(cache1d_flush_all)(cpssp, 1);
}

void
NAME_(mmud_reset)(struct cpssp *cpssp)
{
#if defined(LINES4M) || defined(LINES4K)
	int line;
#endif
	int set;

#ifdef LINES
	{
		for (set = 0; set < SETS; set++) {
			cpssp->NAME_(mmud).set[set].vaddr
					= (Vaddr) -1; /* Invalid */
		}
		NAME_(lru_reset)(SETS, cpssp->NAME_(mmud).lru);
	}
#endif
#ifdef LINES4M
	for (line = 0; line < LINES4M; line++) {
		for (set = 0; set < SETS4M; set++) {
			cpssp->NAME_(mmud).tlb4M[line].set[set].vaddr
					= (Vaddr) -1; /* Invalid */
		}
		NAME_(lru_reset)(SETS4M, cpssp->NAME_(mmud).tlb4M[line].lru);
	}
#endif /* LINES4M */
#ifdef LINES4K
	for (line = 0; line < LINES4K; line++) {
		for (set = 0; set < SETS4K; set++) {
			cpssp->NAME_(mmud).tlb4K[line].set[set].vaddr
					= (Vaddr) -1; /* Invalid */
		}
		NAME_(lru_reset)(SETS4K, cpssp->NAME_(mmud).tlb4K[line].lru);
	}
#endif /* LINES4K */
}

static void
NAME_(mmud_create)(struct cpssp *cpssp)
{
}

static void
NAME_(mmud_destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef LINES
#undef SETS
#undef LINES4M
#undef SETS4M
#undef LINES4K
#undef SETS4K
