/*
 * Copyright (C) 2007-2012 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef STATE

struct {
	struct {
		struct {
			uint32_t page;
			uint32_t entry;
		} cache[4];
		uint32_t lru;
	} mmu_tlb[8];
	uint32_t cr0_pg;
	uint32_t cr3_pdb;
} mmu;

#endif /* STATE */

#ifdef BEHAVIOR

static int
NAME_(check)(uint32_t entry, bool wflag, bool uflag)
{
	return (((entry >> 0) & 1)
		&& (!(wflag & uflag) || (entry >> 1) & 1)
		&& (!uflag || (entry >> 2) & 1));
}


static uint32_t
NAME_(virt_entry)(struct cpssp *cpssp, uint32_t addr, bool wflag, bool uflag)
{
	uint32_t _index;
	uint32_t paddr;
	uint32_t entry0;
	uint32_t entry1;
	bool check;
	bool accessed;
	bool dirty;


	/* Page-Directory Entry */
	_index = (addr >> 22) & 0x3ff;
	/* why cr3 << 12? requieres to >> 12 when mov to cr3, seems unnecessary */
	/* change to not << 12 */
	paddr = cpssp->NAME.cr3_pdb | (_index << 2);

	/*
	 * Read-Modify-Write Cycle:
	 * Set "accessed" bit only if page is present.
	 */
	NAME_(mr)(cpssp, paddr, 0xf, &entry0);
	check = NAME_(check)(entry0, wflag, uflag);
	accessed = (entry0 >> 5) & 1;
	accessed |= check;
	entry0 |= accessed << 5;
	NAME_(mw)(cpssp, paddr, 0xf, entry0);

	if (! ((entry0 >> 0) & 1)) {
		/* Page-table not present. */
		return 0;
	}

	/* Page-Table Entry */
	_index = (addr >> 12) & 0x3ff;
	paddr = (entry0 & 0xfffff000) | (_index << 2);

	/*
	 * Read-Modify-Write Cycle:
	 * Set "accessed" bit only if page is present.
	 * Set "dirty" bit only if page is present and write access.
	 */
	NAME_(mr)(cpssp, paddr, 0xf, &entry1);
	check = NAME_(check)(entry1, wflag, uflag);
	accessed = (entry1 >> 5) & 1;
	accessed |= check;
	entry1 |= accessed << 5; /* Set "accessed" bit. */
	dirty = (entry1 >> 6) & 1;
	dirty |= wflag & check;
	entry1 |= dirty << 6; /* Set "dirty" bit. */
	NAME_(mw)(cpssp, paddr, 0xf, entry1);

	if (! ((entry1 >> 0) & 1)) {
		/* Page not present. */
		return 0;
	}

	return entry1 & (0xffffffff8 | (entry0 & 0x7));
}


static uint32_t
cpu_virt_addr(struct cpssp *cpssp, uint32_t addr, int wflag, int uflag, uint8_t *errp)
{
	uint32_t page;
	uint32_t offset;
	unsigned int c0;
	unsigned int c1;
	uint32_t entry;

	page = addr >> 12;
	offset = addr & 0xfff;

	c0 = page & 0x7;

	if (cpssp->NAME.mmu_tlb[c0].cache[0].page == page) {
		c1 = 0;
	} else if (cpssp->NAME.mmu_tlb[c0].cache[1].page == page) {
		c1 = 1;
	} else if (cpssp->NAME.mmu_tlb[c0].cache[2].page == page) {
		c1 = 2;
	} else if (cpssp->NAME.mmu_tlb[c0].cache[3].page == page) {
		c1 = 3;
	} else {
		c1 = cpssp->NAME.mmu_tlb[c0].lru;
		cpssp->NAME.mmu_tlb[c0].lru = (cpssp->NAME.mmu_tlb[c0].lru + 1) & 3;
	}

	entry = cpssp->NAME.mmu_tlb[c0].cache[c1].entry;
	if (cpssp->NAME.mmu_tlb[c0].cache[c1].page != page
	 || (wflag && !((entry >> 6) & 1))) {
		/* No hit or "modified" bit not set. */
		entry = NAME_(virt_entry)(cpssp, addr, wflag, uflag);
		cpssp->NAME.mmu_tlb[c0].cache[c1].page = page;
		cpssp->NAME.mmu_tlb[c0].cache[c1].entry = entry;
	}

	/* Check. */
	if (! NAME_(check)(entry, wflag, uflag)) {
		*errp = (1 << 3) | (uflag << 2) | (wflag << 1) | (entry & 1);
		return 0;
	}

	*errp = 0;
	return (entry & 0xfffff000) | offset;
}


static void
NAME_(tlb_flush)(struct cpssp *cpssp)
{
	unsigned int c0;
	unsigned int c1;

	for (c0 = 0; c0 < 8; c0++) {
		for (c1 = 0; c1 < 4; c1++) {
			cpssp->NAME.mmu_tlb[c0].cache[c1].page = -1;
		}
	}
}


static void
NAME_(load_cr3)(struct cpssp *cpssp, uint32_t val)
{
	/* TODO use / save flags in lower 12 bits */
	cpssp->NAME.cr3_pdb = val & 0xfffff000;
	NAME_(tlb_flush)(cpssp);
}


/**********************************************************************/

static void
NAME_(vmr)(
	struct cpssp *cpssp,
	uint32_t vaddr,
	unsigned int bs,
	bool user,
	uint32_t *valp,
	uint8_t *errp,
	bool w_test
)
{
	uint32_t paddr;

	if (cpssp->NAME.cr0_pg) {
		paddr = cpu_virt_addr(cpssp, vaddr, w_test, user, errp);
	} else {
		paddr = vaddr;
		*errp = 0;
	}

	if (! *errp) {
		NAME_(mr)(cpssp, paddr, bs, valp);
	}
}

static void
NAME_(vmx)(
	struct cpssp *cpssp,
	uint32_t vaddr,
	unsigned int bs,
	bool user,
	uint32_t *valp,
	uint8_t *errp
)
{
	uint32_t paddr;

	if (cpssp->NAME.cr0_pg) {
		paddr = cpu_virt_addr(cpssp, vaddr, 0, user, errp);
	} else {
		paddr = vaddr;
		*errp = 0;
	}

	if (! *errp) {
		NAME_(mx)(cpssp, paddr, bs, valp);
	}
}

static void
NAME_(vmw)(
	struct cpssp *cpssp,
	uint32_t vaddr,
	unsigned int bs,
	bool user,
	uint32_t val,
	uint8_t *errp
)
{
	uint32_t paddr;

	if (cpssp->NAME.cr0_pg) {
		paddr = cpu_virt_addr(cpssp, vaddr, 1, user, errp);
	} else {
		paddr = vaddr;
		*errp = 0;
	}

	if (! *errp) {
		NAME_(mw)(cpssp, paddr, bs, val);
	}
}

#endif /* BEHAVIOR */
