/*
 * Copyright (C) 2007-2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */


#define LJMP_MODE	1
#define RET_MODE	2
#define EXIT_MODE	3

/* CONFIG */
#define _DEBUG_LVL_		1
#define _DEBUG_LOG_		0
#define _LOG_START_		0x90000000000L
#define _SEG_LIMIT_CHECK_	0
#define _EXCEPION_MODE_		LJMP_MODE


/* Some macros for debugging */

#define RED(x)			"\E[0;31m" #x "\E[0m"
#define GREEN(x)		"\E[0;32m" #x "\E[0m"
#define YELLOW(x)		"\E[0;33m" #x "\E[0m"
#define BLUE(x)			"\E[0;34m" #x "\E[0m"
#define MAGENTA(x)		"\E[0;35m" #x "\E[0m"
#define CYAN(x)			"\E[0;36m" #x "\E[0m"


#define _ERROR(...)		fprintf(stderr, "[" RED(ERROR) "] " __VA_ARGS__);	\


#if	(_DEBUG_LVL_ == 1)
#define _DEBUG1(...)		fprintf(stderr, "[" MAGENTA(DEBUG) "] " __VA_ARGS__);
#define _DEBUG2(...)		;

#elif	(_DEBUG_LVL_ == 2)
#define _DEBUG1(...)		fprintf(stderr, "[" MAGENTA(DEBUG) "] " __VA_ARGS__);
#define _DEBUG2(...)		fprintf(stderr, "[" MAGENTA(DEBUG) "] " __VA_ARGS__);
#define _DEBUG2(...)		;

#else
#define _DEBUG1(...)		;
#define _DEBUG2(...)		;
#endif

#if	(_DEBUG_LOG_ == 1)
#define _DEBUG_L(...)		fprintf(stderr, "[" GREEN(LOG) "] " __VA_ARGS__);
#else
#define _DEBUG_L(...)		;
#endif



#ifdef STATE

struct {
	/* state */
	uint8_t state_halt;
	uint8_t sti;

		/* regs */
	uint32_t gp_regs[8];
	uint32_t eip;
	uint64_t tsc;

		/* seg regs */
	struct {
		/* visible part */
		uint16_t selector;

		/* hidden part */
		uint32_t base;
		uint32_t limit;
		uint8_t g_flag;
		uint8_t db_flag;
		uint8_t avl_flag;
		uint8_t p_flag;
		uint8_t dpl_flag;
		uint8_t cd_flag;
		uint8_t ce_flag;
		uint8_t rw_flag;
		uint8_t a_flag;
	} seg_regs[10];

		/* cr */
	uint32_t cr[7];
			/* cr0 */
	uint8_t pe_flag;
	uint8_t pg_flag;

		/* eflags */
	uint32_t e_sf;
	uint32_t e_zf;
	uint32_t e_of;
	uint32_t e_cf;
	uint32_t e_pf;
	uint32_t e_af;

	uint32_t e_tf;
	uint32_t e_if;
	uint32_t e_df;

	int8_t direction;

	/* tmp vars */
		/* mode */
	uint8_t pre_op;
	uint8_t pre_addr;
	uint8_t pre_repe_repne;

		/* addr */
	uint8_t reg_opex;
	uint8_t rm;
	uint8_t mod;
	uint8_t seg[2];
	uint8_t seg_override;
	uint32_t addr;

		/* tmp regs */
	uint32_t res;
	uint32_t op1;
	uint32_t op2;

		/* exception */
	uint32_t exception_num;
	uint32_t error_code;
	uint32_t eip_save;
	uint32_t esp_save;
#if (_EXCEPION_MODE_ == LJMP_MODE)
	jmp_buf exception_buf;
#endif

	/* debug */
#if	(_DEBUG_LVL_ > 0)
#endif
} NAME;

#endif /* STATE */


// TODO is this correct?
#define SEG_ES		0x0
#define SEG_CS		0x1
#define SEG_SS		0x2
#define SEG_DS		0x3
#define SEG_FS		0x4
#define SEG_GS		0x5
#define SEG_GDT		0x6
#define SEG_LDT		0x7
#define SEG_IDT		0x8
#define SEG_TR		0x9

#define E_AX		0x0
#define E_CX		0x1
#define E_DX		0x2
#define E_BX		0x3
#define E_SP		0x4
#define E_BP		0x5
#define E_SI		0x6
#define E_DI		0x7

#define AL		0x0
#define CL		0x1
#define DL		0x2
#define BL		0x3
#define AH		0x4
#define CH		0x5
#define DH		0x6
#define BH		0x7

#define _DE()		EXCEPTION(0)
#define _BP()		EXCEPTION(3)
#define _OF()		EXCEPTION(4)
#define _UD()		EXCEPTION(6)
#define _NM()		EXCEPTION(7)
#define _DF()		EXCEPTION(8)
#define _TS()		EXCEPTION(10)
#define _NP()		EXCEPTION(11)
#define _SS()		EXCEPTION(12)
#define _GP()		EXCEPTION(13)
#define _PF()		EXCEPTION(14)
#define _MF()		EXCEPTION(16)


#define EFLAGS_TF	(0x1 << 8)
#define EFLAGS_IF	(0x1 << 9)
#define EFLAGS_DF	(0x1 << 10)


#define CPU_CPL(a)	(a->NAME.seg_regs[SEG_CS].dpl_flag)
#define DESC_DPL(a)	((a[1] >> 12) & 0x3)

#define GET_REG(a)	(a & 0x07)
#define GET_OPC(a)	(a >> 3)


#if (_EXCEPION_MODE_ == LJMP_MODE)
#define THROW_EXCEPTION		longjmp(cpssp->NAME.exception_buf, 1);
#elif (_EXCEPION_MODE_ == RET_MODE)
#define THROW_EXCEPTION		return;
#elif (_EXCEPION_MODE_ == EXIT_MODE)
#define THROW_EXCEPTION		{_ERROR("EXCEPTION %d\n", cpssp->NAME.exception_num); exit(1);}
#endif

#define __DF()			{_ERROR("DOUBLE FAULT: not implemented -> ABORT!\n"); assert(0);}
#define EXCEPTION(a)		{if (cpssp->NAME.exception_num != 0) __DF(); cpssp->NAME.exception_num = a; cpssp->state_event |= 1; THROW_EXCEPTION}


#ifdef BEHAVIOR


/* ################ REG ACCESS ############################################# */

static uint8_t
NAME_(get_reg8)(struct cpssp *cpssp, uint8_t reg_num)
{
	uint8_t bit;
	uint8_t reg;
	uint8_t res;
	reg = reg_num & 0x3;
	bit = (reg_num & 0x4) << 1;
	res = (cpssp->NAME.gp_regs[reg] >> bit);
	return res;
}


static void
NAME_(set_reg8)(struct cpssp *cpssp, uint8_t reg_num, uint8_t val)
{
	uint8_t bit;
	uint8_t reg;
	uint32_t tmp;
	uint32_t mask;
	uint32_t val32;

	reg = reg_num & 0x3;
	bit = (reg_num & 0x4) << 1;
	tmp = cpssp->NAME.gp_regs[reg];
	mask = ~(0xff << bit);
	tmp &= mask;
	val32 = val;
	tmp |= (val32 << bit);
	cpssp->NAME.gp_regs[reg] = tmp;
}


static uint16_t
NAME_(get_reg16)(struct cpssp *cpssp, uint8_t reg_num)
{
	return cpssp->NAME.gp_regs[reg_num];
}


static void
NAME_(set_reg16)(struct cpssp *cpssp, uint8_t reg_num, uint16_t val)
{
	uint32_t tmp;

	tmp = cpssp->NAME.gp_regs[reg_num];
	tmp &= 0xffff0000;
	tmp |= val;
	cpssp->NAME.gp_regs[reg_num] = tmp;
}


static uint32_t
NAME_(get_reg32)(struct cpssp *cpssp, uint8_t reg_num)
{
	return cpssp->NAME.gp_regs[reg_num];
}


static void
NAME_(set_reg32)(struct cpssp *cpssp, uint8_t reg_num, uint32_t val)
{
	cpssp->NAME.gp_regs[reg_num] = val;
}


/* ################ SEG FUNCS ############################################## */

static uint32_t
NAME_(get_lin_addr)(struct cpssp *cpssp, uint32_t addr, uint8_t seg, uint8_t data_size)
{
	uint32_t lin_addr;

#if (_SEG_LIMIT_CHECK_ > 0)
	if ((addr + (data_size - 1)) > cpssp->NAME.seg_regs[seg].limit) {
		// TODO EXCEPTION
		_ERROR("Segmentation violation. Offset " RED(0x%08x) " with limit " YELLOW(0x%08x) " of segment " YELLOW(0x%01x) "\n", addr, cpssp->NAME.seg_regs[seg].limit, seg);
		uint32_t base = cpssp->NAME.seg_regs[SEG_CS].base;
		_DEBUG1("[" BLUE(0x%016x) "] @ADDR: " YELLOW(0x%08x:0x%08x - 0x%08x) "\n", cpssp->NAME.tsc, base, cpssp->NAME.eip_save, base+cpssp->NAME.eip_save);
		_DEBUG1("pe-flag: 0x%01x\n", cpssp->NAME.pe_flag);
	//	exit(1);
	}
#endif
	lin_addr = (addr + cpssp->NAME.seg_regs[seg].base);
	_DEBUG2("[MEM] Read / Write lin_addr " GREEN(0x%08x) "\n", lin_addr);
	return lin_addr;
}


/* ################ BUS FUNCS ############################################## */

static uint8_t
NAME_(smx8)(struct cpssp *cpssp, uint32_t addr, uint8_t seg)
{
	uint8_t ret;
	uint8_t err;
	uint32_t lin_addr;

	lin_addr = NAME_(get_lin_addr)(cpssp, addr, seg, 1);
	NAME_(umxb)(cpssp, lin_addr, 0, &ret, &err);
	if (err) {
		_PF();
	}
	return ret;
}


static uint8_t
NAME_(smr8)(struct cpssp *cpssp, uint32_t addr, uint8_t seg, bool w_test)
{
	uint8_t ret;
	uint8_t err;
	uint32_t lin_addr;

	lin_addr = NAME_(get_lin_addr)(cpssp, addr, seg, 1);
	NAME_(umrb)(cpssp, lin_addr, 0, &ret, &err, w_test);
	if (err) {
		_PF();
	}
	return ret;
}


static void
NAME_(smw8)(struct cpssp *cpssp, uint32_t addr, uint8_t seg, uint8_t val)
{
	uint8_t err;
	uint32_t lin_addr;

	lin_addr = NAME_(get_lin_addr)(cpssp, addr, seg, 1);
	NAME_(umwb)(cpssp, lin_addr, 0, val, &err);
	if (err) {
		_PF();
	}
}


static uint16_t
NAME_(smx16)(struct cpssp *cpssp, uint32_t addr, uint8_t seg)
{
	uint16_t ret;
	uint8_t err;
	uint32_t lin_addr;

	lin_addr = NAME_(get_lin_addr)(cpssp, addr, seg, 2);
	NAME_(umxw)(cpssp, lin_addr, 0, &ret, &err);
	if (err) {
		_PF();
	}
	return ret;
}


static uint16_t
NAME_(smr16)(struct cpssp *cpssp, uint32_t addr, uint8_t seg, bool w_test)
{
	uint16_t ret;
	uint8_t err;
	uint32_t lin_addr;

	lin_addr = NAME_(get_lin_addr)(cpssp, addr, seg, 2);
	NAME_(umrw)(cpssp, lin_addr, 0, &ret, &err, w_test);
	if (err) {
		_PF();
	}
	return ret;
}


static void
NAME_(smw16)(struct cpssp *cpssp, uint32_t addr, uint8_t seg, uint16_t val)
{
	uint8_t err;
	uint32_t lin_addr;

	lin_addr = NAME_(get_lin_addr)(cpssp, addr, seg, 2);
	NAME_(umww)(cpssp, lin_addr, 0, val, &err);
	if (err) {
		_PF();
	}
}


static uint32_t
NAME_(smx32)(struct cpssp *cpssp, uint32_t addr, uint8_t seg)
{
	uint32_t ret;
	uint8_t err;
	uint32_t lin_addr;

	lin_addr = NAME_(get_lin_addr)(cpssp, addr, seg, 4);
	NAME_(umxd)(cpssp, lin_addr, 0, &ret, &err);
	if (err) {
		_PF();
	}
	return ret;
}


static uint32_t
NAME_(smr32)(struct cpssp *cpssp, uint32_t addr, uint8_t seg, bool w_test)
{
	uint32_t ret;
	uint8_t err;
	uint32_t lin_addr;

	lin_addr = NAME_(get_lin_addr)(cpssp, addr, seg, 4);
	NAME_(umrd)(cpssp, lin_addr, 0, &ret, &err, w_test);
	if (err) {
		_PF();
	}
	return ret;
}


static void
NAME_(smw32)(struct cpssp *cpssp, uint32_t addr, uint8_t seg, uint32_t val)
{
	uint8_t err;
	uint32_t lin_addr;

	lin_addr = NAME_(get_lin_addr)(cpssp, addr, seg, 4);
	NAME_(umwd)(cpssp, lin_addr, 0, val, &err);
	if (err) {
		_PF();
	}
}


static void
NAME_(out8)(struct cpssp *cpssp)
{
	if ((uint16_t) cpssp->NAME.op1 == 0xffff) {
		fprintf(stderr, "%c", (char) cpssp->NAME.op2);
	}
	NAME_(uiowb)(cpssp, cpssp->NAME.op1, cpssp->NAME.op2); // TODO
}


static void
NAME_(in8)(struct cpssp *cpssp)
{
	uint8_t ret;

	NAME_(uiorb)(cpssp, cpssp->NAME.op1, &ret); // TODO
	cpssp->NAME.res = ret;
}


static void
NAME_(out16_32)(struct cpssp *cpssp, bool op32)
{
	if (likely(op32)) {
		NAME_(uiowd)(cpssp, cpssp->NAME.op1, cpssp->NAME.op2); // TODO
	} else {
		NAME_(uioww)(cpssp, cpssp->NAME.op1, cpssp->NAME.op2); // TODO
	}
}


static void
NAME_(in16_32)(struct cpssp *cpssp, bool op32)
{
	uint16_t tmp;
	uint32_t ret;

	if (likely(op32)) {
		NAME_(uiord)(cpssp, cpssp->NAME.op1, &ret); // TODO
	} else {
		NAME_(uiorw)(cpssp, cpssp->NAME.op1, &tmp); // TODO
		ret = tmp;
	}
	cpssp->NAME.res = ret;
}


/* ################ SEG FUNCS ############################################## */

static void
NAME_(load_seg_reg)(struct cpssp *cpssp, uint8_t seg, uint16_t selector, uint32_t base, uint32_t limit, uint16_t flags)
{
	uint8_t granularity;
	uint8_t shift;
	uint16_t or;

	cpssp->NAME.seg_regs[seg].selector = selector;
	cpssp->NAME.seg_regs[seg].base = base;

	granularity = (flags >> 15) & 0x1;
	cpssp->NAME.seg_regs[seg].g_flag = granularity;
	cpssp->NAME.seg_regs[seg].db_flag = (flags >> 14) & 0x1;
	cpssp->NAME.seg_regs[seg].avl_flag = (flags >> 13) & 0x1;
	cpssp->NAME.seg_regs[seg].p_flag = (flags >> 7) & 0x1;
	cpssp->NAME.seg_regs[seg].dpl_flag = (flags >> 5) & 0x3;
	cpssp->NAME.seg_regs[seg].cd_flag = (flags >> 3) & 0x1;
	cpssp->NAME.seg_regs[seg].ce_flag = (flags >> 2) & 0x1;
	cpssp->NAME.seg_regs[seg].rw_flag = (flags >> 1) & 0x1;
	cpssp->NAME.seg_regs[seg].a_flag = flags & 0x1;

	shift = granularity * 12;
	or = granularity * 0xfff;
	// TODO expand-down b-flag...
	cpssp->NAME.seg_regs[seg].limit = (limit << shift) | or;
	if (cpssp->NAME.seg_regs[seg].cd_flag == 0) {
		if (cpssp->NAME.seg_regs[seg].ce_flag == 1) {
			_ERROR("TODO: expand-down\n");
			exit(1);
		}
	}

	_DEBUG2("SEG: " GREEN(0x%02x) "\n", seg);
	_DEBUG2("SELEC: " GREEN(0x%08x) "\n", selector);
	_DEBUG2("BASE: " GREEN(0x%08x) "\n", base);
	_DEBUG2("LIMIT: " GREEN(0x%08x) "\n", cpssp->NAME.seg_regs[seg].limit);
	_DEBUG2("FLAGS: " GREEN(0x%04x) "\n", flags);
}


static void
NAME_(get_seg_entry)(struct cpssp *cpssp, uint8_t seg, uint16_t selector)
{
	uint8_t entry[8];
	uint8_t desc_type;
	uint8_t cd;
	uint8_t seg_cs;
	uint16_t offset;
	uint16_t flags;
	uint32_t base;
	uint32_t limit;
	uint32_t *entp;

	// TODO clean up segmentation in general, remove 10000000 helper funcs and foo
	if (unlikely(!cpssp->NAME.pe_flag)) {
		/* REAL-ADDRESS-MODE */

		base = selector << 4;
		limit = 0xffff;
		flags = 0;
		NAME_(load_seg_reg)(cpssp, seg, selector, base, limit, flags);
		return;
	}

		/* PROTECTED-MODE */

	offset = selector & ~0x07;
	seg_cs = seg == SEG_CS;
	if (unlikely(!offset && (seg_cs || (SEG_SS == seg)))) {
		_GP();
	}

	// TODO seg? check for mem-err
	entp = (uint32_t*) entry;
	entp[0] = NAME_(smr32)(cpssp, offset, SEG_GDT, 0);
	entp[1] = NAME_(smr32)(cpssp, offset+4, SEG_GDT, 0);

	base = entry[2];
	base |= entry[3] << 8;
	base |= entry[4] << 16;
	base |= entry[7] << 24;
	limit = entry[0];
	limit |= entry[1] << 8;
	limit |= (entry[6] & 0x0f) << 16;
	flags = entry[5];
	flags |= (entry[6] & 0xf0) << 8;

	// TODO check for code / data / task / ... seg and perform option / exit
	desc_type = (flags >> 4) & 0x1;
	if (likely(desc_type)) {
		/* Code / Data Segment */
		cd = (flags >> 3) & 0x1;
		if ((cd ^ seg_cs)) {
			_ERROR("Attemp to load Data Segment to CS register or Code Segment to reg other than CS!\n");
			exit(1);
		}
		NAME_(load_seg_reg)(cpssp, seg, selector, base, limit, flags);
	} else {
		/* System Segment */
		/* FIXME do it properly */
		if (SEG_LDT == seg) {
			desc_type = flags & 0xf;
			if (desc_type == 2) {
				NAME_(load_seg_reg)(cpssp, seg, selector, base, limit, flags);
				return;
			}
			_GP();
		}

		_ERROR("System Segment!\n");
		exit(1);
	}
}


static void
NAME_(check_seg_over)(struct cpssp *cpssp)
{
	cpssp->NAME.seg[0] = cpssp->NAME.seg[cpssp->NAME.seg_override];
}


/* ################ OPERATION MODE AND CPU CONTROL ######################### */

static bool
NAME_(addr32_mode)(struct cpssp *cpssp)
{
	// TODO this correct?
	return cpssp->NAME.seg_regs[SEG_CS].db_flag ^ cpssp->NAME.pre_addr;
}


static bool
NAME_(stack32_mode)(struct cpssp *cpssp)
{
	// TODO this correct? or !db_flag?
	return cpssp->NAME.seg_regs[SEG_SS].db_flag;
}


static bool
NAME_(op32_mode)(struct cpssp *cpssp)
{
	// Intel Manual 3-18 Vol. 1: default op-size is also determined by d-flag of cs-segment
	return cpssp->NAME.seg_regs[SEG_CS].db_flag ^ cpssp->NAME.pre_op;
}


static void
NAME_(load_cr)(struct cpssp *cpssp, uint8_t reg_num, uint32_t val)
{
	switch (reg_num) {
/* CR0 */
	case 0x0:
		cpssp->NAME.pe_flag = val & 0x1;
		//_DEBUG1("CR1:PE = " BLUE(0x%01x) "\n", cpssp->NAME.pe_flag);
		cpssp->NAME.pg_flag = (val >> 31) & 0x1;
		cpssp->mmu.cr0_pg = cpssp->NAME.pg_flag;
		break;

	case 0x3:
		NAME_(load_cr3)(cpssp, val);
		break;

	default:
		_ERROR("invalid cr: cr" RED(%01d) "\n", reg_num);
		_ERROR("cr0: %08x\n", cpssp->NAME.cr[0]);
		exit(1);
	}
	cpssp->NAME.cr[reg_num] = val;
}


static void
NAME_(change_if)(struct cpssp *cpssp, uint32_t val)
{
	// TODO only after sti 1 more inst befor in enabled? or also after pop eflags
	cpssp->NAME.e_if = val;
	cpssp->state_event |= val;
}


static void
NAME_(change_df)(struct cpssp *cpssp, uint32_t val)
{
	cpssp->NAME.e_df = val;
	cpssp->NAME.direction = 1 - (2 * (cpssp->NAME.e_df >> 10));
}


/*
 * EFLAGS
 * ||31      22|  21|   20|   19|  18|  17|  16||
 * ||[RESERVED]| ID | VIP | VIF | AC | VM | RF ||
 *
 * || 15|  14|13  12|  11|  10|   9|   8|   7|   6|  5|   4|  3|   2|  1|   0||
 * || 0 | NT | IOPL | OF | DF | IF | TF | SF | ZF | 0 | AF | 0 | PF | 1 | CF ||
 */
static uint32_t
NAME_(get_eflags)(struct cpssp *cpssp)
{
	uint32_t eflags;

	eflags = 0x00000002;
	eflags |= cpssp->NAME.e_cf;
	eflags |= (cpssp->NAME.e_pf << 2);
	eflags |= cpssp->NAME.e_af;
	eflags |= (cpssp->NAME.e_zf << 6);
	eflags |= (cpssp->NAME.e_sf << 7);
	eflags |= (cpssp->NAME.e_of << 11);
	eflags |= cpssp->NAME.e_tf;
	eflags |= cpssp->NAME.e_if;
	eflags |= cpssp->NAME.e_df;
	return eflags;
}


static void
NAME_(load_eflags)(struct cpssp *cpssp, uint32_t eflags, bool op32)
{
	// TODO test for reserved bits?
	cpssp->NAME.e_cf = eflags & 0x1;
	cpssp->NAME.e_pf = (eflags >> 2) & 0x1;
	cpssp->NAME.e_af = eflags & (0x1 << 4);
	cpssp->NAME.e_zf = (eflags >> 6) & 0x1;
	cpssp->NAME.e_sf = (eflags >> 7) & 0x1;
	cpssp->NAME.e_of = (eflags >> 11) & 0x1;
	cpssp->NAME.e_tf = eflags & EFLAGS_TF;
	NAME_(change_if)(cpssp, (eflags & EFLAGS_IF));
	NAME_(change_df)(cpssp, (eflags & EFLAGS_DF));

	if (likely(op32)) {
		// TODO eflgas 16-31
	}
}


/* ################ SIGN EXTENSION ######################################### */

static uint16_t
NAME_(sign_ext_8to16)(int8_t val)
{
	return val;
}


static uint32_t
NAME_(sign_ext_8to32)(int8_t val)
{
	return val;
}


static uint32_t
NAME_(sign_ext_16to32)(int16_t val)
{
	return val;
}


/* ################ LOAD AND STORE OPERANDS ################################ */

static void
NAME_(load_op1_sreg)(struct cpssp *cpssp, uint8_t seg)
{
	cpssp->NAME.op1 = cpssp->NAME.seg_regs[seg].selector;
}


static void
NAME_(store_res_sreg)(struct cpssp *cpssp, uint8_t seg)
{
	NAME_(get_seg_entry)(cpssp, seg, cpssp->NAME.res);
}


static void
NAME_(load_op1_eflags)(struct cpssp *cpssp, bool op32)
{
	uint8_t s;
	uint32_t tmp;

	s = 16*(!op32);
	tmp = NAME_(get_eflags)(cpssp);
	tmp <<= s;
	tmp >>= s;
	cpssp->NAME.op1 = tmp;
}


static void
NAME_(store_res_eflags)(struct cpssp *cpssp, bool op32)
{
	NAME_(load_eflags)(cpssp, cpssp->NAME.res, op32);
}


static void
NAME_(uload_op1_reg8)(struct cpssp *cpssp, uint8_t reg_num)
{
	cpssp->NAME.op1 = NAME_(get_reg8)(cpssp, reg_num);
}


static void
NAME_(uload_op1_reg16)(struct cpssp *cpssp, uint8_t reg_num)
{
	cpssp->NAME.op1 = NAME_(get_reg16)(cpssp, reg_num);
}


static void
NAME_(uload_op1_reg32)(struct cpssp *cpssp, uint8_t reg_num)
{
	cpssp->NAME.op1 = NAME_(get_reg32)(cpssp, reg_num);
}


static void
NAME_(uload_op1_reg16_32)(struct cpssp *cpssp, uint8_t reg_num, bool op32)
{
	uint8_t s;
	uint32_t tmp;

	s = 16*(!op32);
	tmp = NAME_(get_reg32)(cpssp, reg_num);
	tmp <<= s;
	tmp >>= s;
	cpssp->NAME.op1 = tmp;
}


static void
NAME_(uload_op1_mem8)(struct cpssp *cpssp, bool w_test)
{
	cpssp->NAME.op1 = NAME_(smr8)(cpssp, cpssp->NAME.addr, cpssp->NAME.seg[0], w_test);
}


static void
NAME_(uload_op1_mem16)(struct cpssp *cpssp, bool w_test)
{
	cpssp->NAME.op1 = NAME_(smr16)(cpssp, cpssp->NAME.addr, cpssp->NAME.seg[0], w_test);
}


static void
NAME_(uload_op1_mem32)(struct cpssp *cpssp, bool w_test)
{
	cpssp->NAME.op1 = NAME_(smr32)(cpssp, cpssp->NAME.addr, cpssp->NAME.seg[0], w_test);
}


static void
NAME_(uload_op1_mem16_32)(struct cpssp *cpssp, bool op32, bool w_test)
{
	if (likely(op32)) {
		NAME_(uload_op1_mem32)(cpssp, w_test);
	} else {
		NAME_(uload_op1_mem16)(cpssp, w_test);
	}
}


static void
NAME_(uload_op1_imm8)(struct cpssp *cpssp)
{
	cpssp->NAME.op1 = NAME_(smx8)(cpssp, cpssp->NAME.eip, SEG_CS);
	cpssp->NAME.eip++;
}


static void
NAME_(uload_op1_imm16)(struct cpssp *cpssp)
{
	cpssp->NAME.op1 = NAME_(smx16)(cpssp, cpssp->NAME.eip, SEG_CS);
	cpssp->NAME.eip += 2;
}


static void
NAME_(uload_op1_imm32)(struct cpssp *cpssp)
{
	cpssp->NAME.op1 = NAME_(smx32)(cpssp, cpssp->NAME.eip, SEG_CS);
	cpssp->NAME.eip += 4;
}


static void
NAME_(uload_op1_imm16_32)(struct cpssp *cpssp, bool op32)
{
	if (likely(op32)) {
		NAME_(uload_op1_imm32)(cpssp);
	} else {
		NAME_(uload_op1_imm16)(cpssp);
	}
}


static void
NAME_(uload_op2_reg8)(struct cpssp *cpssp, uint8_t reg_num)
{
	cpssp->NAME.op2 = NAME_(get_reg8)(cpssp, reg_num);
}


static void
NAME_(uload_op2_reg16)(struct cpssp *cpssp, uint8_t reg_num)
{
	cpssp->NAME.op2 = NAME_(get_reg16)(cpssp, reg_num);
}


static void
NAME_(uload_op2_reg32)(struct cpssp *cpssp, uint8_t reg_num)
{
	cpssp->NAME.op2 = NAME_(get_reg32)(cpssp, reg_num);
}


static void
NAME_(uload_op2_reg16_32)(struct cpssp *cpssp, uint8_t reg_num, bool op32)
{
	uint8_t s;
	uint32_t tmp;

	s = 16*(!op32);
	tmp = NAME_(get_reg32)(cpssp, reg_num);
	tmp <<= s;
	tmp >>= s;
	cpssp->NAME.op2 = tmp;
}


static void
NAME_(uload_op2_mem8)(struct cpssp *cpssp, bool w_test)
{
	cpssp->NAME.op2 = NAME_(smr8)(cpssp, cpssp->NAME.addr, cpssp->NAME.seg[0], w_test);
}


static void
NAME_(uload_op2_mem16)(struct cpssp *cpssp, bool w_test)
{
	cpssp->NAME.op2 = NAME_(smr16)(cpssp, cpssp->NAME.addr, cpssp->NAME.seg[0], w_test);
}


static void
NAME_(uload_op2_mem32)(struct cpssp *cpssp, bool w_test)
{
	cpssp->NAME.op2 = NAME_(smr32)(cpssp, cpssp->NAME.addr, cpssp->NAME.seg[0], w_test);
}


static void
NAME_(uload_op2_mem16_32)(struct cpssp *cpssp, bool op32, bool w_test)
{
	if (likely(op32)) {
		NAME_(uload_op2_mem32)(cpssp, w_test);
	} else {
		NAME_(uload_op2_mem16)(cpssp, w_test);
	}
}


static void
NAME_(uload_op2_imm8)(struct cpssp *cpssp)
{
	cpssp->NAME.op2 = NAME_(smx8)(cpssp, cpssp->NAME.eip, SEG_CS);
	cpssp->NAME.eip++;
}


static void
NAME_(uload_op2_imm16)(struct cpssp *cpssp)
{
	cpssp->NAME.op2 = NAME_(smx16)(cpssp, cpssp->NAME.eip, SEG_CS);
	cpssp->NAME.eip += 2;
}


static void
NAME_(uload_op2_imm32)(struct cpssp *cpssp)
{
	cpssp->NAME.op2 = NAME_(smx32)(cpssp, cpssp->NAME.eip, SEG_CS);
	cpssp->NAME.eip += 4;
}


static void
NAME_(uload_op2_imm16_32)(struct cpssp *cpssp, bool op32)
{
	if (likely(op32)) {
		NAME_(uload_op2_imm32)(cpssp);
	} else {
		NAME_(uload_op2_imm16)(cpssp);
	}
}


static void
NAME_(uload_op_m16_16_32)(struct cpssp *cpssp, bool op32 /* FIXME really never w_test==1? bool w_test */)
{
	if (likely(op32)) {
		NAME_(uload_op1_mem32)(cpssp, 0);
		cpssp->NAME.addr += 4;
	} else {
		NAME_(uload_op1_mem16)(cpssp, 0);
		cpssp->NAME.addr += 2;
	}
	NAME_(uload_op2_mem16)(cpssp, 0);
}


static void
NAME_(sload_op1_reg8)(struct cpssp *cpssp, uint8_t reg_num)
{
	NAME_(uload_op1_reg8)(cpssp, reg_num);
	cpssp->NAME.op1 = NAME_(sign_ext_8to32)(cpssp->NAME.op1);
}


static void
NAME_(sload_op1_reg16)(struct cpssp *cpssp, uint8_t reg_num)
{
	NAME_(uload_op1_reg16)(cpssp, reg_num);
	cpssp->NAME.op1 = NAME_(sign_ext_16to32)(cpssp->NAME.op1);
}


static void
NAME_(sload_op1_reg16_32)(struct cpssp *cpssp, uint8_t reg_num, bool op32)
{
	uint8_t s;
	int32_t tmp;

	s = 16*(!op32);
	tmp = NAME_(get_reg32)(cpssp, reg_num);
	tmp <<= s;
	tmp >>= s;
	cpssp->NAME.op1 = tmp;
}


static void
NAME_(sload_op1_mem8)(struct cpssp *cpssp, bool w_test)
{
	NAME_(uload_op1_mem8)(cpssp, w_test);
	cpssp->NAME.op1 = NAME_(sign_ext_8to32)(cpssp->NAME.op1);
}


static void
NAME_(sload_op1_mem16_32)(struct cpssp *cpssp, bool op32, bool w_test)
{
	if (likely(op32)) {
		NAME_(uload_op1_mem32)(cpssp, w_test);
	} else {
		NAME_(uload_op1_mem16)(cpssp, w_test);
		cpssp->NAME.op1 = NAME_(sign_ext_16to32)(cpssp->NAME.op1);
	}
}


static void
NAME_(sload_op1_imm8)(struct cpssp *cpssp)
{
	NAME_(uload_op1_imm8)(cpssp);
	cpssp->NAME.op1 = NAME_(sign_ext_8to32)(cpssp->NAME.op1);
}


static void
NAME_(sload_op1_imm16_32)(struct cpssp *cpssp, bool op32)
{
	if (likely(op32)) {
		NAME_(uload_op1_imm32)(cpssp);
	} else {
		NAME_(uload_op1_imm16)(cpssp);
		cpssp->NAME.op1 = NAME_(sign_ext_16to32)(cpssp->NAME.op1);
	}
}


static void
NAME_(sload_op2_reg8)(struct cpssp *cpssp, uint8_t reg_num)
{
	NAME_(uload_op2_reg8)(cpssp, reg_num);
	cpssp->NAME.op2 = NAME_(sign_ext_8to32)(cpssp->NAME.op2);
}


static void
NAME_(sload_op2_reg16_32)(struct cpssp *cpssp, uint8_t reg_num, bool op32)
{
	uint8_t s;
	int32_t tmp;

	s = 16*(!op32);
	tmp = NAME_(get_reg32)(cpssp, reg_num);
	tmp <<= s;
	tmp >>= s;
	cpssp->NAME.op2 = tmp;
}


static void
NAME_(sload_op2_mem8)(struct cpssp *cpssp, bool w_test)
{
	NAME_(uload_op2_mem8)(cpssp, w_test);
	cpssp->NAME.op2 = NAME_(sign_ext_8to32)(cpssp->NAME.op2);
}


static void
NAME_(sload_op2_mem16_32)(struct cpssp *cpssp, bool op32, bool w_test)
{
	if (likely(op32)) {
		NAME_(uload_op2_mem32)(cpssp, w_test);
	} else {
		NAME_(uload_op2_mem16)(cpssp, w_test);
		cpssp->NAME.op2 = NAME_(sign_ext_16to32)(cpssp->NAME.op2);
	}
}


static void
NAME_(sload_op2_imm8)(struct cpssp *cpssp)
{
	NAME_(uload_op2_imm8)(cpssp);
	cpssp->NAME.op2 = NAME_(sign_ext_8to32)(cpssp->NAME.op2);
}


static void
NAME_(sload_op2_imm16_32)(struct cpssp *cpssp, bool op32)
{
	if (likely(op32)) {
		NAME_(uload_op2_imm32)(cpssp);
	} else {
		NAME_(uload_op2_imm16)(cpssp);
		cpssp->NAME.op2 = NAME_(sign_ext_16to32)(cpssp->NAME.op2);
	}
}


static void
NAME_(store_res_reg8)(struct cpssp *cpssp, uint8_t reg_num)
{
	NAME_(set_reg8)(cpssp, reg_num, cpssp->NAME.res);
}


static void
NAME_(store_res_reg16)(struct cpssp *cpssp, uint8_t reg_num)
{
	NAME_(set_reg16)(cpssp, reg_num, cpssp->NAME.res);
}


static void
NAME_(store_res_reg32)(struct cpssp *cpssp, uint8_t reg_num)
{
	NAME_(set_reg32)(cpssp, reg_num, cpssp->NAME.res);
}


static void
NAME_(store_res_reg16_32)(struct cpssp *cpssp, uint8_t reg_num, bool op32)
{
	if (likely(op32)) {
		NAME_(store_res_reg32)(cpssp, reg_num);
	} else {
		NAME_(store_res_reg16)(cpssp, reg_num);
	}
}


static void
NAME_(store_res_mem8)(struct cpssp *cpssp)
{
	NAME_(smw8)(cpssp, cpssp->NAME.addr, cpssp->NAME.seg[0], cpssp->NAME.res);
}


static void
NAME_(store_res_mem16)(struct cpssp *cpssp)
{
	NAME_(smw16)(cpssp, cpssp->NAME.addr, cpssp->NAME.seg[0], cpssp->NAME.res);
}


static void
NAME_(store_res_mem32)(struct cpssp *cpssp)
{
	NAME_(smw32)(cpssp, cpssp->NAME.addr, cpssp->NAME.seg[0], cpssp->NAME.res);
}


static void
NAME_(store_res_mem16_32)(struct cpssp *cpssp, bool op32)
{
	if (likely(op32)) {
		NAME_(store_res_mem32)(cpssp);
	} else {
		NAME_(store_res_mem16)(cpssp);
	}
}


static void
NAME_(push)(struct cpssp *cpssp, bool op32)
{
	bool stack32;
	uint16_t addr16;
	uint32_t addr32;

	stack32 = NAME_(stack32_mode)(cpssp);

	if (likely(stack32)) {
		addr32 = NAME_(get_reg32)(cpssp, E_SP);
		if (likely(op32)) {
			addr32 -= 4;
			NAME_(smw32)(cpssp, addr32, SEG_SS, cpssp->NAME.op1);
		} else {
			addr32 -= 2;
			NAME_(smw16)(cpssp, addr32, SEG_SS, cpssp->NAME.op1);
		}
		NAME_(set_reg32)(cpssp, E_SP, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_SP);
		if (likely(op32)) {
			addr16 -= 4;
			NAME_(smw32)(cpssp, addr16, SEG_SS, cpssp->NAME.op1);
		} else {
			addr16 -= 2;
			NAME_(smw16)(cpssp, addr16, SEG_SS, cpssp->NAME.op1);
		}
		NAME_(set_reg16)(cpssp, E_SP, addr16);
	}
}

static void
NAME_(pop)(struct cpssp *cpssp, bool op32)
{
	bool stack32;
	uint16_t addr16;
	uint32_t addr32;

	stack32 = NAME_(stack32_mode)(cpssp);

	if (likely(stack32)) {
		addr32 = NAME_(get_reg32)(cpssp, E_SP);
		if (likely(op32)) {
			cpssp->NAME.res = NAME_(smr32)(cpssp, addr32, SEG_SS, 0);
			addr32 += 4;
		} else {
			cpssp->NAME.res = NAME_(smr16)(cpssp, addr32, SEG_SS, 0);
			addr32 += 2;
		}
		NAME_(set_reg32)(cpssp, E_SP, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_SP);
		if (likely(op32)) {
			cpssp->NAME.res = NAME_(smr32)(cpssp, addr16, SEG_SS, 0);
			addr16 += 4;
		} else {
			cpssp->NAME.res = NAME_(smr16)(cpssp, addr16, SEG_SS, 0);
			addr16 += 2;
		}
		NAME_(set_reg16)(cpssp, E_SP, addr16);
	}
}


/* ################ FLAGS ################################################## */

#if 0
/**
  * wikipedia:
  * "The Auxiliary flag is set (to 1) if there is a carry from
  * the low nibble (lowest four bits) to the high nibble (upper four bits),
  * or a borrow from the high nibble to the low nibble, in the low-order 8-bit
  * portion of an addition or subtraction operation. Otherwise, if no such carry
  * or borrow occurs, the flag is cleared (reset to 0)."
  * auxiliary bit: 4
  */
static void
NAME_(det_af)(struct cpssp *cpssp)
{
	// TODO only for add?
	cpssp->NAME.e_af = ((cpssp->NAME.op1 ^ cpssp->NAME.op2) ^ cpssp->NAME.res) & 0x10;
}


/**
  * wikipedia: "OF is set when the MSB is changed by adding two numbers with the same sign"
  *
  * overflow bit: 11
  */
static void
NAME_(det_of_add)(struct cpssp *cpssp)
{
//	cpssp->NAME.of = (((~(op1 ^ op2)) & (res ^ op1)) & 0x80000000) >> 20;
}


/**
  * wikipedia: "OF is set when the MSB is changed by substracting two numbers with opposite signs"
  *
  * overflow bit: 11
  */
static void
NAME_(det_of_sub)(struct cpssp *cpssp)
{
//	cpssp->NAME.of = (((op1 ^ op2) & (res ^ op1)) & 0x80000000) >> 20;
}


/**
  * The carry flag is set if the addition of two numbers causes a carry
  * out of the most significant (leftmost) bits added.
  *
  * carry bit: 0
  */
static void
NAME_(det_cf_add)(struct cpssp *cpssp)
{
	cpssp->NAME.e_cf = cpssp->NAME.res < cpssp->NAME.op1;
}


/**
  * The carry (borrow) flag is also set if the subtraction of two numbers
  * requires a borrow into the most significant (leftmost) bits subtracted.
  *
  * carry bit: 0
  */
static void
NAME_(det_cf_sub)(struct cpssp *cpssp)
{
	cpssp->NAME.e_cf = cpssp->NAME.op1 < cpssp->NAME.op2;
}
#endif


/**
  * wikipedia:
  * "The Auxiliary flag is set (to 1) if there is a carry from
  * the low nibble (lowest four bits) to the high nibble (upper four bits),
  * or a borrow from the high nibble to the low nibble, in the low-order 8-bit
  * portion of an addition or subtraction operation. Otherwise, if no such carry
  * or borrow occurs, the flag is cleared (reset to 0)."
  * auxiliary bit: 4
  */
static void
NAME_(det_af)(struct cpssp *cpssp)
{
	cpssp->NAME.e_af = (cpssp->NAME.op1 ^ cpssp->NAME.op2 ^ cpssp->NAME.res) & 0x10;
}


/**
  * Set if the least-significant byte of the result
  * contains an even number of 1 bits; cleared otherwise.
  *
  * parity bit: 2
  */
static void
NAME_(det_pf)(struct cpssp *cpssp)
{
	uint8_t lsb = cpssp->NAME.res;
	lsb ^= (lsb >> 4);
	lsb ^= (lsb >> 2);
	lsb ^= (lsb >> 1);
	lsb = (~lsb & 0x01);
	cpssp->NAME.e_pf = lsb;
}


/**
  * Set equal to the most-significant bit of the result, which is the sign bit of a signed
  * integer. (0 indicates a positive value and 1 indicates a negative value.)
  *
  * sign bit: 7
  */
static void
NAME_(det_sf)(struct cpssp *cpssp)
{
	cpssp->NAME.e_sf = cpssp->NAME.res >> 31;
}


static void
NAME_(det_sf_s)(struct cpssp *cpssp, uint8_t op_size)
{
	cpssp->NAME.e_sf = (cpssp->NAME.res >> (op_size - 1)) & 0x1;
}


/**
  * Set if the result is zero; cleared otherwise.
  *
  * zero bit: 6
  */
static void
NAME_(det_zf)(struct cpssp *cpssp)
{
	cpssp->NAME.e_zf = !cpssp->NAME.res;
}


/* ################ INSTRUCTIONS ########################################### */

static void
NAME_(and)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.op1 & cpssp->NAME.op2;

	cpssp->NAME.e_cf = 0;
	cpssp->NAME.e_of = 0;
	/* XXX af undefined */
	cpssp->NAME.e_af = 0;
	NAME_(det_pf)(cpssp);
	NAME_(det_sf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(or)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.op1 | cpssp->NAME.op2;

	cpssp->NAME.e_cf = 0;
	cpssp->NAME.e_of = 0;
	/* XXX af undefined */
	NAME_(det_af)(cpssp);
	NAME_(det_pf)(cpssp);
	NAME_(det_sf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(xor)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.op1 ^ cpssp->NAME.op2;

	cpssp->NAME.e_cf = 0;
	cpssp->NAME.e_of = 0;
	/* XXX af undefined */
	NAME_(det_af)(cpssp);
	NAME_(det_pf)(cpssp);
	NAME_(det_sf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(add)(struct cpssp *cpssp, uint8_t op_size)
{
	uint32_t af32;

	cpssp->NAME.res = cpssp->NAME.op1 + cpssp->NAME.op2;

	af32 = cpssp->NAME.res ^ cpssp->NAME.op1 ^ cpssp->NAME.op2;
	cpssp->NAME.e_cf = cpssp->NAME.res < cpssp->NAME.op1;
	cpssp->NAME.e_af = af32 & 0x10;
	cpssp->NAME.e_of = (cpssp->NAME.e_cf ^ (af32 >> (op_size - 1))) & 0x1;
	NAME_(det_sf_s)(cpssp, op_size);
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(adc)(struct cpssp *cpssp, uint8_t op_size)
{
	uint32_t af32;

	cpssp->NAME.res = cpssp->NAME.op1 + cpssp->NAME.op2 + cpssp->NAME.e_cf;

	af32 = cpssp->NAME.res ^ cpssp->NAME.op1 ^ cpssp->NAME.op2;
	// TODO best way to do?
	cpssp->NAME.e_cf = (cpssp->NAME.res < cpssp->NAME.op1) | (cpssp->NAME.res < cpssp->NAME.op2);
	cpssp->NAME.e_af = af32 & 0x10;
	cpssp->NAME.e_of = (cpssp->NAME.e_cf ^ (af32 >> (op_size - 1))) & 0x1;
	NAME_(det_sf_s)(cpssp, op_size);
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(sub)(struct cpssp *cpssp, uint8_t op_size)
{
	uint32_t af32;

	cpssp->NAME.res = cpssp->NAME.op1 - cpssp->NAME.op2;

	af32 = cpssp->NAME.res ^ cpssp->NAME.op1 ^ cpssp->NAME.op2;
	cpssp->NAME.e_cf = cpssp->NAME.op1 < cpssp->NAME.op2;
	cpssp->NAME.e_af = af32 & 0x10;
	cpssp->NAME.e_of = (cpssp->NAME.e_cf ^ (af32 >> (op_size - 1))) & 0x1;
	NAME_(det_sf_s)(cpssp, op_size);
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(sbb)(struct cpssp *cpssp, uint8_t op_size)
{
	uint32_t af32;

	cpssp->NAME.res = cpssp->NAME.op1 - cpssp->NAME.op2 - cpssp->NAME.e_cf;

	af32 = cpssp->NAME.res ^ cpssp->NAME.op1 ^ cpssp->NAME.op2;
	// TODO best way to do?
	cpssp->NAME.e_cf = (cpssp->NAME.op1 < cpssp->NAME.op2) | ((cpssp->NAME.op1 == cpssp->NAME.op2) & cpssp->NAME.e_cf);
	cpssp->NAME.e_af = af32 & 0x10;
	cpssp->NAME.e_of = (cpssp->NAME.e_cf ^ (af32 >> (op_size - 1))) & 0x1;
	NAME_(det_sf_s)(cpssp, op_size);
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(cmp)(struct cpssp *cpssp, uint8_t op_size)
{
	NAME_(sub)(cpssp, op_size);
}


static void
NAME_(test)(struct cpssp *cpssp)
{
	NAME_(and)(cpssp);
}


static void
NAME_(neg)(struct cpssp *cpssp, uint8_t op_size)
{
	uint32_t af32;

	cpssp->NAME.res = -cpssp->NAME.op1;

	// TODO correct if sub works
	// TODO is that correct?
	af32 = (cpssp->NAME.res ^ cpssp->NAME.op1);
	cpssp->NAME.e_cf = (cpssp->NAME.op1 != 0);
	cpssp->NAME.e_af = af32 & 0x10;
	cpssp->NAME.e_of = (cpssp->NAME.e_cf ^ (af32 >> (op_size - 1))) & 0x1;
	NAME_(det_sf_s)(cpssp, op_size);
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(not)(struct cpssp *cpssp)
{
	cpssp->NAME.res = ~cpssp->NAME.op1;
}


static void
NAME_(inc)(struct cpssp *cpssp, uint8_t op_size)
{
	uint32_t af32;

	cpssp->NAME.res = cpssp->NAME.op1 + 1;

	af32 = cpssp->NAME.res ^ cpssp->NAME.op1;
	cpssp->NAME.e_af = af32 & 0x10;
	cpssp->NAME.e_of = ((~cpssp->NAME.op1 & af32) >> (op_size - 1)) & 0x1;
	NAME_(det_sf_s)(cpssp, op_size);
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(dec)(struct cpssp *cpssp, uint8_t op_size)
{
	uint32_t af32;

	cpssp->NAME.res = cpssp->NAME.op1 - 1;

	af32 = cpssp->NAME.res ^ cpssp->NAME.op1;
	cpssp->NAME.e_af = af32 & 0x10;
	cpssp->NAME.e_of = ((cpssp->NAME.op1 & af32) >> (op_size - 1)) & 0x1;
	NAME_(det_sf_s)(cpssp, op_size);
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
}

// TODO simplify
static void
NAME_(sal_shl)(struct cpssp *cpssp, uint8_t op_size)
{
	uint8_t flags[2];
	uint8_t s1;
	uint8_t s2;

	cpssp->NAME.op2 &= 0x1f;
	cpssp->NAME.res = cpssp->NAME.op1 << cpssp->NAME.op2;

	if (!cpssp->NAME.op2)
		return;
	s1 = cpssp->NAME.op2 - 1;
	s2 = op_size - 1;
	flags[1] = cpssp->NAME.e_cf;
	flags[0] = ((cpssp->NAME.op1 << s1) >> s2) & 0x1;
	cpssp->NAME.e_cf = flags[!cpssp->NAME.op2];
	cpssp->NAME.e_sf = cpssp->NAME.res >> s2;
	flags[0] = cpssp->NAME.e_of;
	flags[1] = 1 ^ (cpssp->NAME.e_sf & cpssp->NAME.e_cf);
	cpssp->NAME.e_of = flags[!s1];
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
	/* undefined */
	cpssp->NAME.e_af = 0;
}


static void
NAME_(sar)(struct cpssp *cpssp, uint8_t op_size)
{
	uint8_t flags[2];
	uint8_t s1;
	uint8_t s2;

	s2 = 32 - op_size;
	cpssp->NAME.op1 <<= s2;
	cpssp->NAME.op1 = ((int32_t) cpssp->NAME.op1) >> s2;
	cpssp->NAME.op2 &= 0x1f;
	cpssp->NAME.res = ((int32_t) cpssp->NAME.op1) >> cpssp->NAME.op2;

	if (!cpssp->NAME.op2)
		return;
	s1 = cpssp->NAME.op2 - 1;
	flags[1] = cpssp->NAME.e_cf;
	flags[0] = (((int32_t) cpssp->NAME.op1) >> s1) & 0x1;
	cpssp->NAME.e_cf = flags[!cpssp->NAME.op2];
	cpssp->NAME.e_sf = cpssp->NAME.res >> (op_size - 1);
	flags[0] = cpssp->NAME.e_of;
	flags[1] = 0;
	cpssp->NAME.e_of = flags[!s1];
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
	/* undefined */
	cpssp->NAME.e_af = 0;
}


static void
NAME_(shr)(struct cpssp *cpssp, uint8_t op_size)
{
	uint8_t flags[2];
	uint8_t s;

	cpssp->NAME.op2 &= 0x1f;
	cpssp->NAME.res = cpssp->NAME.op1 >> cpssp->NAME.op2;

	if (!cpssp->NAME.op2)
		return;
	s = cpssp->NAME.op2 - 1;
	flags[1] = cpssp->NAME.e_cf;
	flags[0] = (cpssp->NAME.op1 >> s) & 0x1;
	cpssp->NAME.e_cf = flags[!cpssp->NAME.op2];
	cpssp->NAME.e_sf = cpssp->NAME.res >> (op_size - 1);
	flags[0] = cpssp->NAME.e_of;
	flags[1] = cpssp->NAME.e_sf;
	cpssp->NAME.e_of = flags[!s];
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
	/* undefined */
	cpssp->NAME.e_af = 0;
}


static void
NAME_(shrd)(struct cpssp *cpssp, uint8_t op_size)
{
	uint8_t flags[2];
	uint8_t s;
	uint32_t b;

//	cpssp->NAME.op1 &= 0x1f;	Intel manual: "If the count operand is CL, the shift count is the logical AND of CL and a count mask."
	b = cpssp->NAME.res;
	cpssp->NAME.res >>= cpssp->NAME.op1;
	cpssp->NAME.res |= (cpssp->NAME.op2 << (op_size - cpssp->NAME.op1));

	if (!cpssp->NAME.op1)
		return;
	s = cpssp->NAME.op1 - 1;
	cpssp->NAME.e_cf = (b >> s) & 0x1;
	cpssp->NAME.e_sf = cpssp->NAME.res >> (op_size - 1);
	flags[0] = cpssp->NAME.e_of;
	flags[1] = (cpssp->NAME.res ^ b) >> (op_size - 1);
	cpssp->NAME.e_of = flags[!s];
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
	/* undefined */
	cpssp->NAME.e_af = 0;
}


static void
NAME_(rol)(struct cpssp *cpssp, uint8_t op_size)
{
	uint8_t flags[2];
	uint8_t s;

	s = op_size - 1;
	cpssp->NAME.op2 &= s;
	cpssp->NAME.res = cpssp->NAME.op1 << cpssp->NAME.op2;
	cpssp->NAME.res |= (cpssp->NAME.op1 >> (op_size - cpssp->NAME.op2));

	flags[1] = cpssp->NAME.e_cf;
	flags[0] = cpssp->NAME.res & 0x1;
	cpssp->NAME.e_cf = flags[!cpssp->NAME.op2];
	flags[0] = cpssp->NAME.e_of;
	flags[1] = (cpssp->NAME.res >> s) ^ cpssp->NAME.e_cf;
	cpssp->NAME.e_of = flags[(cpssp->NAME.op2 == 1)];
}


static void
NAME_(ror)(struct cpssp *cpssp, uint8_t op_size)
{
	uint8_t flags[2];
	uint8_t s;

	s = op_size - 1;
	cpssp->NAME.op2 &= s;
	cpssp->NAME.res = cpssp->NAME.op1 >> cpssp->NAME.op2;
	cpssp->NAME.res |= (cpssp->NAME.op1 << (op_size - cpssp->NAME.op2));

	flags[1] = cpssp->NAME.e_cf;
	flags[0] = cpssp->NAME.res >> s;
	cpssp->NAME.e_cf = flags[!cpssp->NAME.op2];
	flags[0] = cpssp->NAME.e_of;
	flags[1] = ((cpssp->NAME.res >> (op_size - 2)) & 0x1) ^ cpssp->NAME.e_cf;
	cpssp->NAME.e_of = flags[(cpssp->NAME.op2 == 1)];
}


static void
NAME_(rcl)(struct cpssp *cpssp, uint8_t op_size)
{
	uint8_t flags[2];
	uint8_t s;
	uint8_t n;
	uint32_t tmp;

	cpssp->NAME.op2 &= 0x1f;
	cpssp->NAME.op2 %= (op_size + 1);
	n = !cpssp->NAME.op2;
	s = cpssp->NAME.op2 - 1;
	cpssp->NAME.res = cpssp->NAME.op1 << cpssp->NAME.op2;
	cpssp->NAME.res |= ((cpssp->NAME.e_cf & !n) << s);
	tmp = cpssp->NAME.op1 >> (op_size - cpssp->NAME.op2);
	cpssp->NAME.res |= (tmp >> 1);

	flags[1] = cpssp->NAME.e_cf;
	flags[0] = tmp & 0x1;
	cpssp->NAME.e_cf = flags[n];
	flags[0] = cpssp->NAME.e_of;
	flags[1] = (cpssp->NAME.res >> (op_size -1)) ^ cpssp->NAME.e_cf;
	cpssp->NAME.e_of = flags[!s];
}


static void
NAME_(rcr)(struct cpssp *cpssp, uint8_t op_size)
{
	uint8_t flags[2];
	uint8_t s;
	uint8_t n;
	uint8_t o;

	cpssp->NAME.op2 &= 0x1f;
	cpssp->NAME.op2 %= (op_size + 1);
	n = !cpssp->NAME.op2;
	s = op_size - cpssp->NAME.op2;
	o = cpssp->NAME.op2 - 1;
	cpssp->NAME.res = cpssp->NAME.op1 >> cpssp->NAME.op2;
	cpssp->NAME.res |= (cpssp->NAME.e_cf & !n) << s;
	cpssp->NAME.res |= ((cpssp->NAME.op1 << s) << 1);

	flags[0] = cpssp->NAME.e_of;
	flags[1] = (cpssp->NAME.op1 >> (op_size - 1)) ^ cpssp->NAME.e_cf;
	cpssp->NAME.e_of = flags[!o];
	flags[1] = cpssp->NAME.e_cf;
	flags[0] = (cpssp->NAME.op1 >> o) & 0x1;
	cpssp->NAME.e_cf = flags[n];
}


static void
NAME_(cwd)(struct cpssp *cpssp, bool op32)
{
	int32_t tmp;

	// TODO
	tmp = cpssp->NAME.op1;
	tmp >>= 31;
	cpssp->NAME.res = tmp;
}


static void
NAME_(mul8)(struct cpssp *cpssp)
{
	uint32_t tmp;
	bool flag;

	cpssp->NAME.res = cpssp->NAME.op1 * cpssp->NAME.op2;

	tmp = (uint8_t) cpssp->NAME.res;
	flag = tmp != cpssp->NAME.res;
	cpssp->NAME.e_cf = flag;
	cpssp->NAME.e_of = flag;
	/* undefined */
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
	NAME_(det_sf_s)(cpssp, 8);
	NAME_(det_af)(cpssp);
}


static void
NAME_(mul16_32)(struct cpssp *cpssp, bool op32)
{
	uint64_t tmp1;
	uint64_t tmp2[2];
	bool flag[2];

	tmp1 = cpssp->NAME.op1 * cpssp->NAME.op2;

	tmp2[0] = (uint16_t) tmp1;
	flag[0] = tmp1 != tmp2[0];
	tmp2[1] = (uint32_t) tmp1;
	flag[1] = tmp1 != tmp2[1];
	cpssp->NAME.e_cf = flag[op32];
	cpssp->NAME.e_of = flag[op32];
	cpssp->NAME.res = tmp1;
	cpssp->NAME.op1 = tmp1 >> (16 + (16*op32));
	/* undefined */
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
	NAME_(det_sf_s)(cpssp, 16 << op32);
	NAME_(det_af)(cpssp);
}


static void
NAME_(imul8)(struct cpssp *cpssp)
{
	int32_t tmp1;
	int32_t tmp2;
	bool flag;

	tmp1 = ((int32_t) cpssp->NAME.op1) * ((int32_t) cpssp->NAME.op2);

	tmp2 = (uint8_t) tmp1;
	tmp2 = (int8_t) tmp2;
	flag = tmp1 != tmp2;
	cpssp->NAME.e_cf = flag;
	cpssp->NAME.e_of = flag;
	cpssp->NAME.res = tmp1;
	NAME_(det_sf)(cpssp);
	/* undefined */
	cpssp->NAME.e_af = 0;
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(imul16_32)(struct cpssp *cpssp, bool op32)
{
	int64_t tmp1;
	int64_t tmp2[2];
	bool flag[2];

	tmp1 = ((int32_t) cpssp->NAME.op1) * ((int32_t) cpssp->NAME.op2);

	tmp2[0] = (uint16_t) tmp1;
	tmp2[0] = (int16_t) tmp2[0];
	flag[0] = tmp1 != tmp2[0];
	tmp2[1] = (uint32_t) tmp1;
	tmp2[1] = (int32_t) tmp2[1];
	flag[1] = tmp1 != tmp2[1];
	cpssp->NAME.e_cf = flag[op32];
	cpssp->NAME.e_of = flag[op32];
	cpssp->NAME.res = tmp1;
	cpssp->NAME.op1 = tmp1 >> (16 + (16*op32));
	NAME_(det_sf)(cpssp);
	/* undefined */
	cpssp->NAME.e_af = 0;
	NAME_(det_pf)(cpssp);
	NAME_(det_zf)(cpssp);
}


static void
NAME_(div8)(struct cpssp *cpssp)
{
	if (!cpssp->NAME.op2) {
		_DE();
	}

	cpssp->NAME.res = cpssp->NAME.op1 / cpssp->NAME.op2;
	cpssp->NAME.op1 = cpssp->NAME.op1 % cpssp->NAME.op2;

	if (cpssp->NAME.res > 0xff) {
		_DE();
	}
	/* undefined */
	cpssp->NAME.e_pf = 0;
	cpssp->NAME.e_zf = 0;
	//NAME_(det_pf)(cpssp);
	//NAME_(det_zf)(cpssp);
}


static void
NAME_(div16_32)(struct cpssp *cpssp, bool op32)
{
	uint64_t d;
	uint64_t r1;
	uint64_t r2;

	if (!cpssp->NAME.op2) {
		_DE();
	}

	d = cpssp->NAME.res;
	d <<= (16 + (16 * op32));
	d |= cpssp->NAME.op1;
	r1 = d / cpssp->NAME.op2;
	r2 = d % cpssp->NAME.op2;

	if (r1 > (0xffff | (0xffffffff * op32))) {
		_DE();
	}

	cpssp->NAME.res = r1;
	cpssp->NAME.op1 = r2;
	/* undefined */
	cpssp->NAME.e_pf = 0;
	cpssp->NAME.e_zf = 0;
	//NAME_(det_pf)(cpssp);
	//NAME_(det_zf)(cpssp);
}


static void
NAME_(idiv8)(struct cpssp *cpssp)
{
	int64_t r1;
	int64_t r2;
	bool err;

	if (!cpssp->NAME.op2) {
		_DE();
	}

	r1 = ((int32_t) cpssp->NAME.op1) / ((int32_t) cpssp->NAME.op2);
	r2 = ((int32_t) cpssp->NAME.op1) % ((int32_t) cpssp->NAME.op2);

	err = (r1 > ((int8_t) 0x7f)) | (r1 < ((int8_t) 0x80));
	if (err) {
		_DE();
	}

	cpssp->NAME.res = r1;
	cpssp->NAME.op1 = r2;
}


static void
NAME_(idiv16_32)(struct cpssp *cpssp, bool op32)
{
	int64_t d;
	int64_t r1;
	int64_t r2;
	bool err[2];

	if (!cpssp->NAME.op2) {
		_DE();
	}

	d = cpssp->NAME.res;
	d <<= (16 + (16 * op32));
	d |= cpssp->NAME.op1;
	r1 = d / ((int32_t) cpssp->NAME.op2);
	r2 = d % ((int32_t) cpssp->NAME.op2);

	err[0] = (r1 > ((int16_t) 0x7fff)) | (r1 < ((int16_t) 0x8000));
	err[1] = (r1 > ((int32_t) 0x7fffffff)) | (r1 < ((int32_t) 0x80000000));
	if (err[op32]) {
		_DE();
	}

	cpssp->NAME.res = r1;
	cpssp->NAME.op1 = r2;
}


static void
NAME_(alu_bit)(struct cpssp *cpssp, uint8_t op_size)
{
	switch (cpssp->NAME.reg_opex) {
/* ROL */
	case 0x0:
		NAME_(rol)(cpssp, op_size);
		break;

/* ROR */
	case 0x1:
		NAME_(ror)(cpssp, op_size);
		break;

/* RCL */
	case 0x2:
		NAME_(rcl)(cpssp, op_size);
		break;

/* RCR */
	case 0x3:
		NAME_(rcr)(cpssp, op_size);
		break;

/* SAL/SHL */
	case 0x4:
		NAME_(sal_shl)(cpssp, op_size);
		break;

/* SHR */
	case 0x5:
		NAME_(shr)(cpssp, op_size);
		break;

/* SAR */
	case 0x7:
		NAME_(sar)(cpssp, op_size);
		break;

	default:
		_ERROR("Unknown opcode in " YELLOW('alu_bit')
				" with opcode-extension " RED(0x%01x) "\n", cpssp->NAME.reg_opex);
		exit(1);
		break;
	}
}


static void
NAME_(alu_arith)(struct cpssp *cpssp, uint8_t op, uint8_t op_size)
{
	switch (op) {
/* ADD */
	case 0x0:
		NAME_(add)(cpssp, op_size);
		break;

/* OR */
	case 0x1:
		NAME_(or)(cpssp);
		break;

/* ADC */
	case 0x2:
		NAME_(adc)(cpssp, op_size);
		break;

/* SBB */
	case 0x3:
		NAME_(sbb)(cpssp, op_size);
		break;

/* AND */
	case 0x4:
		NAME_(and)(cpssp);
		break;

/* SUB */
	case 0x5:
		NAME_(sub)(cpssp, op_size);
		break;

/* XOR */
	case 0x6:
		NAME_(xor)(cpssp);
		break;

/* CMP */
	case 0x7:
		NAME_(cmp)(cpssp, op_size);
		// TODO really ok to write back (ro-page / segment what ever....)
		cpssp->NAME.res = cpssp->NAME.op1;
		break;

	default:
		_ERROR("Unknown opcode in " YELLOW('alu_arith')
				" with opcode-extension " RED(0x%01x) "\n", cpssp->NAME.reg_opex);
		exit(1);
		break;
	}
}


static void
NAME_(mov)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.op2;
}


static void
NAME_(jmpa)(struct cpssp *cpssp)
{
	// TODO check for seg_limit
	cpssp->NAME.eip = cpssp->NAME.op1;
	if (!NAME_(addr32_mode)(cpssp)) {
		cpssp->NAME.eip &= 0x0000ffff;
	}
}


static void
NAME_(jmp)(struct cpssp *cpssp)
{
	// TODO check for seg_limit
	// TODO clear upper 2 byte if 16-bit mode?
//	uint32_t base = cpssp->NAME.seg_regs[SEG_CS].base;
//	_DEBUG1("[" BLUE(0x%016x) "] JMP: " YELLOW(0x%08x:0x%08x - 0x%08x) "\n", cpssp->NAME.tsc, base, cpssp->NAME.eip, base+cpssp->NAME.eip);
	bool addr32;
	uint32_t mask;

	cpssp->NAME.eip += cpssp->NAME.op1;
	addr32 = NAME_(addr32_mode)(cpssp);
	mask = 0x0000ffff | (0xffffffff * addr32);
	cpssp->NAME.eip &= mask;
//	base = cpssp->NAME.seg_regs[SEG_CS].base;
//	_DEBUG1("\t\t\t  --> " YELLOW(0x%08x:0x%08x - 0x%08x) "\n", base, cpssp->NAME.eip, base+cpssp->NAME.eip);
}


static void
NAME_(setcc)(struct cpssp *cpssp, bool not)
{
	cpssp->NAME.res ^= not;
}


static void
NAME_(jcc)(struct cpssp *cpssp, bool not)
{
	uint32_t offset[2];

	offset[0] = 0;
	offset[1] = cpssp->NAME.op1;
	cpssp->NAME.op1 = offset[cpssp->NAME.res ^ not];
	NAME_(jmp)(cpssp);
}


static void
NAME_(loop)(struct cpssp *cpssp)
{
	uint16_t cx;
	uint32_t ecx;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		ecx = NAME_(get_reg32)(cpssp, E_CX);
		ecx--;
		NAME_(set_reg32)(cpssp, E_CX, ecx);
		if (ecx) {
			NAME_(jmp)(cpssp);
		}
	} else {
		cx = NAME_(get_reg16)(cpssp, E_CX);
		cx--;
		NAME_(set_reg16)(cpssp, E_CX, cx);
		if (cx) {
			NAME_(jmp)(cpssp);
		}
	}
}


static void
NAME_(loopcc)(struct cpssp *cpssp, bool not)
{
	uint16_t cx;
	uint32_t ecx;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		ecx = NAME_(get_reg32)(cpssp, E_CX);
		ecx--;
		NAME_(set_reg32)(cpssp, E_CX, ecx);
		if (ecx && (cpssp->NAME.res ^ not)) {
			NAME_(jmp)(cpssp);
		}
	} else {
		cx = NAME_(get_reg16)(cpssp, E_CX);
		cx--;
		NAME_(set_reg16)(cpssp, E_CX, cx);
		if (cx && (cpssp->NAME.res ^ not)) {
			NAME_(jmp)(cpssp);
		}
	}
}


static void
NAME_(test_ecx)(struct cpssp *cpssp)
{
	cpssp->NAME.res = !cpssp->NAME.op2;
}


static void
NAME_(test_pf)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.e_pf;
}


static void
NAME_(test_of)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.e_of;
}


static void
NAME_(test_sf)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.e_sf;
}


static void
NAME_(test_zf)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.e_zf;
}


static void
NAME_(test_cf)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.e_cf;
}


static void
NAME_(test_na_be)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.e_cf | cpssp->NAME.e_zf;
}


static void
NAME_(test_ng_le)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.e_zf | (cpssp->NAME.e_sf ^ cpssp->NAME.e_of);
}


static void
NAME_(test_l_nge)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.e_sf ^ cpssp->NAME.e_of;
}


static void
NAME_(lgdt)(struct cpssp *cpssp)
{
	NAME_(load_seg_reg)(cpssp, SEG_GDT, 0x0000, cpssp->NAME.op2, cpssp->NAME.op1, 0);
}


static void
NAME_(lidt)(struct cpssp *cpssp)
{
	NAME_(load_seg_reg)(cpssp, SEG_IDT, 0x0000, cpssp->NAME.op2, cpssp->NAME.op1, 0);
}


static void
NAME_(mov_from_cr)(struct cpssp *cpssp)
{
	// TODO priv lvl check and other cr, check range and stuff
	if (cpssp->NAME.reg_opex != 0) {
		_ERROR("invalid cr: cr" RED(%01d) "\n", cpssp->NAME.reg_opex);
		exit(1);
	}
	cpssp->NAME.res = cpssp->NAME.cr[cpssp->NAME.reg_opex];
}


static void
NAME_(mov_to_cr)(struct cpssp *cpssp)
{
	// TODO priv lvl check and other cr, check range and stuff
	NAME_(load_cr)(cpssp, cpssp->NAME.reg_opex, cpssp->NAME.op2);
}


static void
NAME_(mov_from_sreg)(struct cpssp *cpssp)
{
	if (unlikely(cpssp->NAME.reg_opex > 0x5)) {
		_GP();
	}

	NAME_(load_op1_sreg)(cpssp, cpssp->NAME.reg_opex);
	cpssp->NAME.res = cpssp->NAME.op1;
}


static void
NAME_(mov_to_sreg)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.op2;
	NAME_(store_res_sreg)(cpssp, cpssp->NAME.reg_opex);
}


static void
NAME_(ljmp)(struct cpssp *cpssp)
{
	// TODO if jmpa out of new sc don't set new seg... integrate somehow
	cpssp->NAME.res = cpssp->NAME.op2;
	NAME_(store_res_sreg)(cpssp, SEG_CS);
	NAME_(jmpa)(cpssp);
}


static bool
NAME_(rep)(struct cpssp *cpssp)
{
	uint32_t tmp;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		tmp = NAME_(get_reg32)(cpssp, E_CX);
		if (tmp) {
			tmp--;
			NAME_(set_reg32)(cpssp, E_CX, tmp);
			cpssp->NAME.eip = cpssp->NAME.eip_save;
			return 1;
		}
	} else {
		tmp = NAME_(get_reg16)(cpssp, E_CX);
		if (tmp) {
			tmp--;
			NAME_(set_reg16)(cpssp, E_CX, tmp);
			cpssp->NAME.eip = cpssp->NAME.eip_save;
			return 1;
		}
	}
	return 0;
}


static bool
NAME_(rep_z_nz)(struct cpssp *cpssp)
{
	uint32_t tmp;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		tmp = NAME_(get_reg32)(cpssp, E_CX);
		if (tmp) {
			tmp--;
			NAME_(set_reg32)(cpssp, E_CX, tmp);
			return 1;
		}
	} else {
		tmp = NAME_(get_reg16)(cpssp, E_CX);
		if (tmp) {
			tmp--;
			NAME_(set_reg16)(cpssp, E_CX, tmp);
			return 1;
		}
	}
	return 0;
}


static void
NAME_(repn_z)(struct cpssp *cpssp)
{
	uint8_t sel;
	uint32_t eip[2];

	/* repne/repnz  == 0x1 */
	/* rep/repe/repz == 0x2 */
	sel = cpssp->NAME.pre_repe_repne & 0x1;
	sel ^= cpssp->NAME.e_zf;

	eip[0] = cpssp->NAME.eip;
	eip[1] = cpssp->NAME.eip_save;
	cpssp->NAME.eip = eip[sel];
}


static void
NAME_(movs8)(struct cpssp *cpssp)
{
	int8_t next;
	uint16_t addr_src16;
	uint16_t addr_dst16;
	uint32_t addr_src32;
	uint32_t addr_dst32;

	next = 1 * cpssp->NAME.direction;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr_src32 = NAME_(get_reg32)(cpssp, E_SI);
		addr_dst32 = NAME_(get_reg32)(cpssp, E_DI);
		cpssp->NAME.op1 = NAME_(smr8)(cpssp, addr_src32, cpssp->NAME.seg[0], 0);
		NAME_(smw8)(cpssp, addr_dst32, SEG_ES, cpssp->NAME.op1);
		addr_src32 += next;
		addr_dst32 += next;
		NAME_(set_reg32)(cpssp, E_SI, addr_src32);
		NAME_(set_reg32)(cpssp, E_DI, addr_dst32);
	} else {
		addr_src16 = NAME_(get_reg16)(cpssp, E_SI);
		addr_dst16 = NAME_(get_reg16)(cpssp, E_DI);
		cpssp->NAME.op1 = NAME_(smr8)(cpssp, addr_src16, cpssp->NAME.seg[0], 0);
		NAME_(smw8)(cpssp, addr_dst16, SEG_ES, cpssp->NAME.op1);
		addr_src16 += next;
		addr_dst16 += next;
		NAME_(set_reg16)(cpssp, E_SI, addr_src16);
		NAME_(set_reg16)(cpssp, E_DI, addr_dst16);
	}
}


static void
NAME_(movs16_32)(struct cpssp *cpssp, bool op32)
{
	int8_t next;
	uint16_t addr_src16;
	uint16_t addr_dst16;
	uint32_t addr_src32;
	uint32_t addr_dst32;

	next = 2 * (1 + op32) * cpssp->NAME.direction;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr_src32 = NAME_(get_reg32)(cpssp, E_SI);
		addr_dst32 = NAME_(get_reg32)(cpssp, E_DI);
		if (likely(op32)) {
			cpssp->NAME.op1 = NAME_(smr32)(cpssp, addr_src32, cpssp->NAME.seg[0], 0);
			NAME_(smw32)(cpssp, addr_dst32, SEG_ES, cpssp->NAME.op1);
		} else {
			cpssp->NAME.op1 = NAME_(smr16)(cpssp, addr_src32, cpssp->NAME.seg[0], 0);
			NAME_(smw16)(cpssp, addr_dst32, SEG_ES, cpssp->NAME.op1);
		}
		addr_src32 += next;
		addr_dst32 += next;
		NAME_(set_reg32)(cpssp, E_SI, addr_src32);
		NAME_(set_reg32)(cpssp, E_DI, addr_dst32);
	} else {
		addr_src16 = NAME_(get_reg16)(cpssp, E_SI);
		addr_dst16 = NAME_(get_reg16)(cpssp, E_DI);
		if (likely(op32)) {
			cpssp->NAME.op1 = NAME_(smr32)(cpssp, addr_src16, cpssp->NAME.seg[0], 0);
			NAME_(smw32)(cpssp, addr_dst16, SEG_ES, cpssp->NAME.op1);
		} else {
			cpssp->NAME.op1 = NAME_(smr32)(cpssp, addr_src16, cpssp->NAME.seg[0], 0);
			NAME_(smw32)(cpssp, addr_dst16, SEG_ES, cpssp->NAME.op1);
		}
		addr_src16 += next;
		addr_dst16 += next;
		NAME_(set_reg16)(cpssp, E_SI, addr_src16);
		NAME_(set_reg16)(cpssp, E_DI, addr_dst16);
	}
}


static void
NAME_(cmps8)(struct cpssp *cpssp)
{
	int8_t next;
	uint16_t addr_src16;
	uint16_t addr_dst16;
	uint32_t addr_src32;
	uint32_t addr_dst32;

	next = 1 * cpssp->NAME.direction;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr_src32 = NAME_(get_reg32)(cpssp, E_SI);
		addr_dst32 = NAME_(get_reg32)(cpssp, E_DI);
		cpssp->NAME.op1 = NAME_(smr8)(cpssp, addr_src32, cpssp->NAME.seg[0], 0);
		cpssp->NAME.op2 = NAME_(smr8)(cpssp, addr_dst32, SEG_ES, 0);
		NAME_(cmp)(cpssp, 7);
		addr_src32 += next;
		addr_dst32 += next;
		NAME_(set_reg32)(cpssp, E_SI, addr_src32);
		NAME_(set_reg32)(cpssp, E_DI, addr_dst32);
	} else {
		addr_src16 = NAME_(get_reg16)(cpssp, E_SI);
		addr_dst16 = NAME_(get_reg16)(cpssp, E_DI);
		cpssp->NAME.op1 = NAME_(smr8)(cpssp, addr_src16, cpssp->NAME.seg[0], 0);
		cpssp->NAME.op2 = NAME_(smr8)(cpssp, addr_dst16, SEG_ES, 0);
		NAME_(cmp)(cpssp, 7);
		addr_src16 += next;
		addr_dst16 += next;
		NAME_(set_reg16)(cpssp, E_SI, addr_src16);
		NAME_(set_reg16)(cpssp, E_DI, addr_dst16);
	}
}


static void
NAME_(cmps16_32)(struct cpssp *cpssp, bool op32)
{
	int8_t next;
	uint16_t addr_src16;
	uint16_t addr_dst16;
	uint32_t addr_src32;
	uint32_t addr_dst32;

	next = 2 * (1 + op32) * cpssp->NAME.direction;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr_src32 = NAME_(get_reg32)(cpssp, E_SI);
		addr_dst32 = NAME_(get_reg32)(cpssp, E_DI);
		if (likely(op32)) {
			cpssp->NAME.op1 = NAME_(smr32)(cpssp, addr_src32, cpssp->NAME.seg[0], 0);
			cpssp->NAME.op2 = NAME_(smr32)(cpssp, addr_dst32, SEG_ES, 0);
			NAME_(cmp)(cpssp, 32);
		} else {
			cpssp->NAME.op1 = NAME_(smr16)(cpssp, addr_src32, cpssp->NAME.seg[0], 0);
			cpssp->NAME.op2 = NAME_(smr16)(cpssp, addr_dst32, SEG_ES, 0);
			NAME_(cmp)(cpssp, 16);
		}
		addr_src32 += next;
		addr_dst32 += next;
		NAME_(set_reg32)(cpssp, E_SI, addr_src32);
		NAME_(set_reg32)(cpssp, E_DI, addr_dst32);
	} else {
		addr_src16 = NAME_(get_reg16)(cpssp, E_SI);
		addr_dst16 = NAME_(get_reg16)(cpssp, E_DI);
		if (likely(op32)) {
			cpssp->NAME.op1 = NAME_(smr32)(cpssp, addr_src16, cpssp->NAME.seg[0], 0);
			cpssp->NAME.op2 = NAME_(smr32)(cpssp, addr_dst16, SEG_ES, 0);
			NAME_(cmp)(cpssp, 32);
		} else {
			cpssp->NAME.op1 = NAME_(smr32)(cpssp, addr_src16, cpssp->NAME.seg[0], 0);
			cpssp->NAME.op2 = NAME_(smr32)(cpssp, addr_dst16, SEG_ES, 0);
			NAME_(cmp)(cpssp, 16);
		}
		addr_src16 += next;
		addr_dst16 += next;
		NAME_(set_reg16)(cpssp, E_SI, addr_src16);
		NAME_(set_reg16)(cpssp, E_DI, addr_dst16);
	}
}


static void
NAME_(stos8)(struct cpssp *cpssp)
{
	int8_t next;
	uint16_t addr16;
	uint32_t addr32;

	next = 1 * cpssp->NAME.direction;

	cpssp->NAME.op1 = NAME_(get_reg8)(cpssp, AL);
	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr32 = NAME_(get_reg32)(cpssp, E_DI);
		NAME_(smw8)(cpssp, addr32, SEG_ES, cpssp->NAME.op1);
		addr32 += next;
		NAME_(set_reg32)(cpssp, E_DI, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_DI);
		NAME_(smw8)(cpssp, addr16, SEG_ES, cpssp->NAME.op1);
		addr16 += next;
		NAME_(set_reg16)(cpssp, E_DI, addr16);
	}
}


static void
NAME_(stos16_32)(struct cpssp *cpssp, bool op32)
{
	int8_t next;
	uint16_t addr16;
	uint32_t addr32;

	next = 2 * (1 + op32) * cpssp->NAME.direction;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr32 = NAME_(get_reg32)(cpssp, E_DI);
		if (likely(op32)) {
			cpssp->NAME.op1 = NAME_(get_reg32)(cpssp, E_AX);
			NAME_(smw32)(cpssp, addr32, SEG_ES, cpssp->NAME.op1);
		} else {
			cpssp->NAME.op1 = NAME_(get_reg16)(cpssp, E_AX);
			NAME_(smw16)(cpssp, addr32, SEG_ES, cpssp->NAME.op1);
		}
		addr32 += next;
		NAME_(set_reg32)(cpssp, E_DI, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_DI);
		if (likely(op32)) {
			cpssp->NAME.op1 = NAME_(get_reg32)(cpssp, E_AX);
			NAME_(smw32)(cpssp, addr16, SEG_ES, cpssp->NAME.op1);
		} else {
			cpssp->NAME.op1 = NAME_(get_reg16)(cpssp, E_AX);
			NAME_(smw16)(cpssp, addr16, SEG_ES, cpssp->NAME.op1);
		}
		addr16 += next;
		NAME_(set_reg16)(cpssp, E_DI, addr16);
	}
}


static void
NAME_(scas8)(struct cpssp *cpssp)
{
	int8_t next;
	uint16_t addr16;
	uint32_t addr32;

	next = 1 * cpssp->NAME.direction;

	cpssp->NAME.op1 = NAME_(get_reg8)(cpssp, AL);
	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr32 = NAME_(get_reg32)(cpssp, E_DI);
		cpssp->NAME.op2 = NAME_(smr8)(cpssp, addr32, SEG_ES, 0);
		NAME_(cmp)(cpssp, 7);
		addr32 += next;
		NAME_(set_reg32)(cpssp, E_DI, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_DI);
		cpssp->NAME.op2 = NAME_(smr8)(cpssp, addr16, SEG_ES, 0);
		NAME_(cmp)(cpssp, 7);
		addr16 += next;
		NAME_(set_reg16)(cpssp, E_DI, addr16);
	}
}


static void
NAME_(scas16_32)(struct cpssp *cpssp, bool op32)
{
	int8_t next;
	uint16_t addr16;
	uint32_t addr32;

	next = 2 * (1 + op32) * cpssp->NAME.direction;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr32 = NAME_(get_reg32)(cpssp, E_DI);
		if (likely(op32)) {
			cpssp->NAME.op1 = NAME_(get_reg32)(cpssp, E_AX);
			cpssp->NAME.op2 = NAME_(smr32)(cpssp, addr32, SEG_ES, 0);
			NAME_(cmp)(cpssp, 32);
		} else {
			cpssp->NAME.op1 = NAME_(get_reg16)(cpssp, E_AX);
			cpssp->NAME.op2 = NAME_(smr16)(cpssp, addr32, SEG_ES, 0);
			NAME_(cmp)(cpssp, 16);
		}
		addr32 += next;
		NAME_(set_reg32)(cpssp, E_DI, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_DI);
		if (likely(op32)) {
			cpssp->NAME.op1 = NAME_(get_reg32)(cpssp, E_AX);
			cpssp->NAME.op2 = NAME_(smr32)(cpssp, addr16, SEG_ES, 0);
			NAME_(cmp)(cpssp, 32);
		} else {
			cpssp->NAME.op1 = NAME_(get_reg16)(cpssp, E_AX);
			cpssp->NAME.op2 = NAME_(smr16)(cpssp, addr16, SEG_ES, 0);
			NAME_(cmp)(cpssp, 16);
		}
		addr16 += next;
		NAME_(set_reg16)(cpssp, E_DI, addr16);
	}
}


static void
NAME_(lods8)(struct cpssp *cpssp)
{
	int8_t next;
	uint16_t addr16;
	uint32_t addr32;

	next = 1 * cpssp->NAME.direction;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr32 = NAME_(get_reg32)(cpssp, E_SI);
		cpssp->NAME.res = NAME_(smr8)(cpssp, addr32, cpssp->NAME.seg[0], 0);
		addr32 += next;
		NAME_(set_reg32)(cpssp, E_SI, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_SI);
		cpssp->NAME.res = NAME_(smr8)(cpssp, addr16, cpssp->NAME.seg[0], 0);
		addr16 += next;
		NAME_(set_reg16)(cpssp, E_SI, addr16);
	}
	NAME_(set_reg8)(cpssp, AL, cpssp->NAME.res);
}


static void
NAME_(lods16_32)(struct cpssp *cpssp, bool op32)
{
	int8_t next;
	uint16_t addr16;
	uint32_t addr32;

	next = 2 * (1 + op32) * cpssp->NAME.direction;

	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr32 = NAME_(get_reg32)(cpssp, E_SI);
		if (likely(op32)) {
			cpssp->NAME.res = NAME_(smr32)(cpssp, addr32, cpssp->NAME.seg[0], 0);
			NAME_(set_reg32)(cpssp, E_AX, cpssp->NAME.res);
		} else {
			cpssp->NAME.res = NAME_(smr16)(cpssp, addr32, cpssp->NAME.seg[0], 0);
			NAME_(set_reg16)(cpssp, E_AX, cpssp->NAME.res);
		}
		addr32 += next;
		NAME_(set_reg32)(cpssp, E_SI, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_SI);
		if (likely(op32)) {
			cpssp->NAME.res = NAME_(smr32)(cpssp, addr16, cpssp->NAME.seg[0], 0);
			NAME_(set_reg32)(cpssp, E_AX, cpssp->NAME.res);
		} else {
			cpssp->NAME.res = NAME_(smr16)(cpssp, addr16, cpssp->NAME.seg[0], 0);
			NAME_(set_reg16)(cpssp, E_AX, cpssp->NAME.res);
		}
		addr16 += next;
		NAME_(set_reg16)(cpssp, E_SI, addr16);
}
}


static void
NAME_(ins8)(struct cpssp *cpssp)
{
	int8_t next;
	uint16_t addr16;
	uint32_t addr32;

	next = 1 * cpssp->NAME.direction;

	cpssp->NAME.op1 = NAME_(get_reg16)(cpssp, E_DX);
	NAME_(in8)(cpssp);
	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr32 = NAME_(get_reg32)(cpssp, E_DI);
		NAME_(smw8)(cpssp, addr32, SEG_ES, cpssp->NAME.res);
		addr32 += next;
		NAME_(set_reg32)(cpssp, E_DI, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_DI);
		NAME_(smw8)(cpssp, addr16, SEG_ES, cpssp->NAME.res);
		addr16 += next;
		NAME_(set_reg16)(cpssp, E_DI, addr16);
	}
}


static void
NAME_(ins16_32)(struct cpssp *cpssp, bool op32)
{
	int8_t next;
	uint16_t addr16;
	uint32_t addr32;

	next = 2 * (1 + op32) * cpssp->NAME.direction;

	cpssp->NAME.op1 = NAME_(get_reg16)(cpssp, E_DX);
	NAME_(in16_32)(cpssp, op32);
	if (likely(NAME_(addr32_mode)(cpssp))) {
		addr32 = NAME_(get_reg32)(cpssp, E_DI);
		if (likely(op32)) {
			NAME_(smw32)(cpssp, addr32, SEG_ES, cpssp->NAME.res);
		} else {
			NAME_(smw16)(cpssp, addr32, SEG_ES, cpssp->NAME.res);
		}
		addr32 += next;
		NAME_(set_reg32)(cpssp, E_DI, addr32);
	} else {
		addr16 = NAME_(get_reg16)(cpssp, E_DI);
		if (likely(op32)) {
			NAME_(smw32)(cpssp, addr16, SEG_ES, cpssp->NAME.res);
		} else {
			NAME_(smw16)(cpssp, addr16, SEG_ES, cpssp->NAME.res);
		}
		addr16 += next;
		NAME_(set_reg16)(cpssp, E_DI, addr16);
	}
}


static void
NAME_(calla)(struct cpssp *cpssp, bool op32)
{
	uint32_t offset;

	offset = cpssp->NAME.op1;
	cpssp->NAME.op1 = cpssp->NAME.eip;
	NAME_(push)(cpssp, op32);
	cpssp->NAME.eip = offset;
}


static void
NAME_(call)(struct cpssp *cpssp, bool op32)
{
	uint32_t offset;

	offset = cpssp->NAME.op1;
	cpssp->NAME.op1 = cpssp->NAME.eip;
	NAME_(push)(cpssp, op32);
	cpssp->NAME.eip += offset;
	// TODO clear upper 2 byte if 16-bit mode?
}


static void
NAME_(fcall)(struct cpssp *cpssp, bool op32)
{
	uint32_t offset;

	offset = cpssp->NAME.op1;
	cpssp->NAME.res = cpssp->NAME.op2;

	cpssp->NAME.op1 = cpssp->NAME.seg_regs[SEG_CS].selector;
	NAME_(push)(cpssp, op32);
	cpssp->NAME.op1 = cpssp->NAME.eip;
	NAME_(push)(cpssp, op32);

	// TODO IF DEST[31:16] is not zero THEN #GP(0); FI; ??? ask
	NAME_(store_res_sreg)(cpssp, SEG_CS);
	cpssp->NAME.eip = offset;
}


static void
NAME_(intn)(struct cpssp *cpssp, uint8_t int_num, uint8_t swi)
{
	uint8_t type;
	uint8_t properties;
	uint16_t vec;
	uint16_t selector;
	uint32_t offset;
	uint32_t entry[2];

	if (unlikely(!cpssp->NAME.pe_flag)) {
		/* REAL-ADDRESS-MODE */

		// TODO exception or actuall smr32?
		vec = int_num << 2;
		if ((vec + 3) > cpssp->NAME.seg_regs[SEG_IDT].limit) {
			_GP();
		}
		entry[0] = NAME_(smr32)(cpssp, vec, SEG_IDT, 0);
		offset = entry[0] & 0xffff;
		cpssp->NAME.res = entry[0] >> 16;

		// TODO really push16?
		NAME_(load_op1_eflags)(cpssp, 0);
		NAME_(push)(cpssp, 0);
		cpssp->NAME.op1 = cpssp->NAME.seg_regs[SEG_CS].selector;
		NAME_(push)(cpssp, 0);
		cpssp->NAME.op1 = cpssp->NAME.eip;
		NAME_(push)(cpssp, 0);

		NAME_(change_if)(cpssp, 0);
		cpssp->NAME.e_tf = 0;
		NAME_(store_res_sreg)(cpssp, SEG_CS);
		cpssp->NAME.eip = offset;
		return;
	}

	/* PROTECTED-MODE */

	vec = int_num << 3;
	// TODO check if limit is tested correct
	if ((vec + 7) > cpssp->NAME.seg_regs[SEG_IDT].limit) {
		_DEBUG1("%lx\n", cpssp->NAME.tsc);
		_GP();
	}

	entry[0] = NAME_(smr32)(cpssp, vec, SEG_IDT, 0);
	entry[1] = NAME_(smr32)(cpssp, vec+4, SEG_IDT, 0);

	properties = entry[1] >> 8;
	type = properties & 0x1f;
	if (likely(type  == 0x6)) {
		/* Interrupt / Trap Gate */

		if (!(properties & 0x80)) {
			_NP();
		}
		if (swi) {
			if (DESC_DPL(entry) < CPU_CPL(cpssp))
				_GP();
		}

		offset = entry[0] & 0xffff;
		offset |= entry[1] & ~0xffff;
		selector = entry[0] & ~0xffff;

		_ERROR("finish that\n");
		exit(1);

	} else if (type == 0x5) {
		/* Task Gate */

		if (!(properties & 0x80)) {
			_NP();
		}
		if (swi) {
			if (DESC_DPL(entry) < CPU_CPL(cpssp))
				_GP();
		}

		_ERROR("TODO Task Gate int\n");
		exit(1);

	} else {
		/* invalide descriptor type */
		_GP();
	}
}


static void
NAME_(ret)(struct cpssp *cpssp, bool op32)
{
	NAME_(pop)(cpssp, op32);
	cpssp->NAME.eip = cpssp->NAME.res;
}


static void
NAME_(fret)(struct cpssp *cpssp, bool op32)
{
	uint32_t offset;

	NAME_(pop)(cpssp, op32);
	offset = cpssp->NAME.res;
	NAME_(pop)(cpssp, op32);

	NAME_(store_res_sreg)(cpssp, SEG_CS);
	cpssp->NAME.eip = offset;
}


static void
NAME_(iret)(struct cpssp *cpssp, bool op32)
{
	if (unlikely(!cpssp->NAME.pe_flag)) {
		/* REAL-ADDRESS-MODE */

		if (op32) {
			_ERROR("TODO iret16 with op32");
			exit(1);
		}

		// TODO really push16?
		NAME_(pop)(cpssp, 0);
		cpssp->NAME.eip = cpssp->NAME.res;
		NAME_(pop)(cpssp, 0);
		NAME_(store_res_sreg)(cpssp, SEG_CS);
		NAME_(pop)(cpssp, 0);
		NAME_(store_res_eflags)(cpssp, 0);
		return;
	}

	/* PROTECTED-MODE */

	_ERROR("TODO iret32\n");
	exit(1);
}


static void
NAME_(lea)(struct cpssp *cpssp)
{
	cpssp->NAME.res = cpssp->NAME.addr;
}


static void
NAME_(xchg)(struct cpssp *cpssp)
{
	NAME_(mov)(cpssp);
}


static void
NAME_(cli)(struct cpssp *cpssp)
{
	if (unlikely(!cpssp->NAME.pe_flag)) {
		/* REAL-ADDRESS-MODE */

		NAME_(change_if)(cpssp, 0);
		return;
	}

	/* PROTECTED-MODE */

	// FIXME priv.../
	NAME_(change_if)(cpssp, 0);
}


static void
NAME_(sti)(struct cpssp *cpssp)
{
	if (unlikely(!cpssp->NAME.pe_flag)) {
		/* REAL-ADDRESS-MODE */

		cpssp->NAME.sti = 1;
		cpssp->state_event = 1;
		return;
	}

	/* PROTECTED-MODE */

	// FIXME priv.../
	cpssp->NAME.sti = 1;
	cpssp->state_event = 1;
}


static void
NAME_(clc)(struct cpssp *cpssp)
{
	cpssp->NAME.e_cf = 0;
}


static void
NAME_(stc)(struct cpssp *cpssp)
{
	cpssp->NAME.e_cf = 1;
}


static void
NAME_(cld)(struct cpssp *cpssp)
{
	NAME_(change_df)(cpssp, 0);
}


static void
NAME_(std)(struct cpssp *cpssp)
{
	NAME_(change_df)(cpssp, EFLAGS_DF);
}


static void
NAME_(hlt)(struct cpssp *cpssp)
{
	if (unlikely(!cpssp->NAME.pe_flag)) {
		/* REAL-ADDRESS-MODE */

		cpssp->state_event = 1;
		cpssp->NAME.state_halt = 1;
		return;
	}

	/* PROTECTED-MODE */

	// FIXME priv...
	cpssp->state_event = 1;
	cpssp->NAME.state_halt = 1;
}


static void
NAME_(xlat)(struct cpssp *cpssp, uint8_t addr32)
{
	uint32_t addr;

	// TODO addr16/32 because of overflow to start at addr 0?
	addr = NAME_(get_reg32)(cpssp, E_BX);
	addr &= (0x0000ffff | (0xffffffff * addr32));
	addr += NAME_(get_reg8)(cpssp, AL);
	cpssp->NAME.res = NAME_(smr8)(cpssp, addr, cpssp->NAME.seg[0], 0);
}


static void
NAME_(l_s)(struct cpssp *cpssp, uint8_t seg)
{
	cpssp->NAME.res = cpssp->NAME.op1;
	if (unlikely(!cpssp->NAME.pe_flag)) {
		/* REAL-ADDRESS-MODE */

		NAME_(get_seg_entry)(cpssp, seg, cpssp->NAME.op2);
		return;
	}

	/* PROTECTED-MODE */

	// FIXME priv....
	NAME_(get_seg_entry)(cpssp, seg, cpssp->NAME.op2);
}


static void
NAME_(lmsw)(struct cpssp *cpssp)
{
	uint32_t tmp;

	// TODO check pirv...
	tmp = cpssp->NAME.cr[0];
	tmp &= 0xfffffff0;
	tmp |= (cpssp->NAME.op1 & 0xf);
	NAME_(load_cr)(cpssp, 0, tmp);
}

/* ################ DEBUG ################################################## */

static inline __attribute__((__always_inline__)) void
NAME_(dump)(struct cpssp *cpssp)
{
		fprintf(stderr, " %08lx %08lx %08lx %08lx %08lx %08lx %08lx %08lx",
				(uint32_t) NAME_(get_reg32)(cpssp, E_AX),
				(uint32_t) NAME_(get_reg32)(cpssp, E_BX),
				(uint32_t) NAME_(get_reg32)(cpssp, E_CX),
				(uint32_t) NAME_(get_reg32)(cpssp, E_DX),
				(uint32_t) NAME_(get_reg32)(cpssp, E_BP),
				(uint32_t) NAME_(get_reg32)(cpssp, E_SP),
				(uint32_t) NAME_(get_reg32)(cpssp, E_DI),
				(uint32_t) NAME_(get_reg32)(cpssp, E_SI));

		fprintf(stderr, " %08lx", NAME_(get_eflags)(cpssp));

		fprintf(stderr, "\n");
		fflush(stderr);
}


/* ################ MODRM ################################################## */

static void
NAME_(get_modrm)(struct cpssp *cpssp)
{
	uint8_t modrm;

	modrm = NAME_(smx8)(cpssp, cpssp->NAME.eip, SEG_CS);
	cpssp->NAME.eip++;

	cpssp->NAME.mod = (modrm >> 6) & 0x03;
	cpssp->NAME.reg_opex = (modrm >> 3) & 0x07;
	cpssp->NAME.rm = modrm & 0x07;
}


static void
NAME_(get_addr)(struct cpssp *cpssp)
{
	uint8_t scale;
	uint8_t index;
	uint8_t base;
	uint8_t seg[2];
	uint16_t r16_1;
	uint16_t r16_2;
	uint16_t addr16;
	uint16_t disp16;
	uint32_t addr32;
	uint32_t disp32;
	uint32_t sib;
	uint32_t base_val;
	uint32_t index_val[2];

#if	(_DEBUG_LVL_ > 0)
	if (0x3 == cpssp->NAME.mod) {
		_ERROR("mod == 3\n");
		exit(1);
	}
#endif
	if (likely(NAME_(addr32_mode)(cpssp))) {
		/* 32bit addressing mode */

		seg[0] = SEG_DS;
		if (0x4 == cpssp->NAME.rm) {
			/* SIB-byte */
			sib = NAME_(smx8)(cpssp, cpssp->NAME.eip, SEG_CS);
			cpssp->NAME.eip++;
			scale = (sib >> 6) & 0x03;
			index = (sib >> 3) & 0x07;
			base = sib & 0x07;

			base_val = NAME_(get_reg32)(cpssp, base);
			index_val[0] = NAME_(get_reg32)(cpssp, index);
			index_val[(0x4 != index)] = 0;


			switch (cpssp->NAME.mod) {
			case 0x0:
				if (0x5 == base) {
					base_val = 0;
					disp32 = NAME_(smx32)(cpssp, cpssp->NAME.eip, SEG_CS);
					cpssp->NAME.eip += 4;
				} else {
					disp32 = 0;
					seg[(0x4 != base)] = SEG_SS; // TODO ask again, intel manual useless again
				}
				break;
			case 0x1:
				disp32 = NAME_(sign_ext_8to32)(NAME_(smx8)(cpssp, cpssp->NAME.eip, SEG_CS));
				cpssp->NAME.eip++;
				seg[((0x4 != base) & (0x5 != base))] = SEG_SS; // TODO ask again, intel manual useless again
				break;
			case 0x2:
				disp32 = NAME_(smx32)(cpssp, cpssp->NAME.eip, SEG_CS);
				cpssp->NAME.eip += 4;
				seg[((0x4 != base) & (0x5 != base))] = SEG_SS; // TODO ask again, intel manual useless again
				break;
			default:	/* should not be reachable */
				assert(0);
			}

			addr32 = base_val + (index_val[0] << scale);

		} else {
			/* normall addressing */
			addr32 = NAME_(get_reg32)(cpssp, cpssp->NAME.rm);

			switch (cpssp->NAME.mod) {
			case 0x0:
				if (0x5 == cpssp->NAME.rm) {
					addr32 = 0;
					disp32 = NAME_(smx32)(cpssp, cpssp->NAME.eip, SEG_CS);
					cpssp->NAME.eip += 4;
				} else {
					disp32 = 0;
				}
				break;
			case 0x1:
				disp32 = NAME_(sign_ext_8to32)(NAME_(smx8)(cpssp, cpssp->NAME.eip, SEG_CS));
				cpssp->NAME.eip++;
				seg[(0x5 != cpssp->NAME.rm)] = SEG_SS;
				break;
			case 0x2:
				disp32 = NAME_(smx32)(cpssp, cpssp->NAME.eip, SEG_CS);
				cpssp->NAME.eip += 4;
				seg[(0x5 != cpssp->NAME.rm)] = SEG_SS;
				break;
			default:	/* should not be reachable */
				assert(0);
			}
		}

		addr32 += disp32;
		cpssp->NAME.addr = addr32;
		cpssp->NAME.seg[0] = seg[0];

	} else {
		/* 16bit addressing mode */

		cpssp->NAME.seg[0] = SEG_DS;
		// TODO stattdessen arrays machen und get_reg16(array[rm]); "Abbildungstabellen"
		switch (cpssp->NAME.rm) {
		case 0x0:
			r16_1 = NAME_(get_reg16)(cpssp, E_BX);
			r16_2 = NAME_(get_reg16)(cpssp, E_SI);
			break;
		case 0x1:
			r16_1 = NAME_(get_reg16)(cpssp, E_BX);
			r16_2 = NAME_(get_reg16)(cpssp, E_DI);
			break;
		case 0x2:
			r16_1 = NAME_(get_reg16)(cpssp, E_BP);
			r16_2 = NAME_(get_reg16)(cpssp, E_SI);
			cpssp->NAME.seg[0] = SEG_SS;
			break;
		case 0x3:
			r16_1 = NAME_(get_reg16)(cpssp, E_BP);
			r16_2 = NAME_(get_reg16)(cpssp, E_DI);
			cpssp->NAME.seg[0] = SEG_SS;
			break;
		case 0x4:
			r16_1 = NAME_(get_reg16)(cpssp, E_SI);
			r16_2 = 0;
			break;
		case 0x5:
			r16_1 = NAME_(get_reg16)(cpssp, E_DI);
			r16_2 = 0;
			break;
		case 0x6:
			r16_1 = NAME_(get_reg16)(cpssp, E_BP);
			r16_2 = 0;
			cpssp->NAME.seg[0] = SEG_SS;
			break;
		case 0x7:
			r16_1 = NAME_(get_reg16)(cpssp, E_BX);
			r16_2 = 0;
			break;
		default:	/* should not be reachable */
			assert(0);
		}

		switch (cpssp->NAME.mod) {
		case 0x0:
			if (0x6 == cpssp->NAME.rm) {
				disp16 = NAME_(smx16)(cpssp, cpssp->NAME.eip, SEG_CS);
				cpssp->NAME.eip += 2;
				cpssp->NAME.seg[0] = SEG_DS;
				r16_1 = 0;
				r16_2 = 0;
			} else {
				disp16 = 0;
			}
			break;
		case 0x1:
			disp16 = NAME_(sign_ext_8to16)(NAME_(smx8)(cpssp, cpssp->NAME.eip, SEG_CS));
			cpssp->NAME.eip++;
			break;
		case 0x2:
			disp16 = NAME_(smx16)(cpssp, cpssp->NAME.eip, SEG_CS);
			cpssp->NAME.eip += 2;
			break;
		default:	/* should not be reachable */
			assert(0);
		}

		addr16 = disp16 + r16_1 + r16_2;
		cpssp->NAME.addr = addr16;
	}

	NAME_(check_seg_over)(cpssp);
}


/* ################ CPU_RESET ############################################## */

static void
NAME_(cpu_reset)(struct cpssp *cpssp)
{
	//_DEBUG1("Reset: CPU\n");
	// TODO reset other vars and regs too
	cpssp->NAME.eip = 0x0000fff0;
	cpssp->NAME.tsc = 0x0;
	cpssp->NAME.sti = 0;
	cpssp->NAME.state_halt = 0;

	NAME_(load_seg_reg)(cpssp, SEG_CS, 0xf000, 0xffff0000, 0xffff, 0);

	NAME_(load_seg_reg)(cpssp, SEG_SS, 0x0000, 0x00000000, 0xffff, 0);
	NAME_(load_seg_reg)(cpssp, SEG_DS, 0x0000, 0x00000000, 0xffff, 0);
	NAME_(load_seg_reg)(cpssp, SEG_ES, 0x0000, 0x00000000, 0xffff, 0);
	NAME_(load_seg_reg)(cpssp, SEG_FS, 0x0000, 0x00000000, 0xffff, 0);
	NAME_(load_seg_reg)(cpssp, SEG_GS, 0x0000, 0x00000000, 0xffff, 0);

	NAME_(load_seg_reg)(cpssp, SEG_GDT, 0x0000, 0x00000000, 0xffff, 0);
	NAME_(load_seg_reg)(cpssp, SEG_LDT, 0x0000, 0x00000000, 0xffff, 0);
	NAME_(load_seg_reg)(cpssp, SEG_IDT, 0x0000, 0x00000000, 0xffff, 0);
	NAME_(load_seg_reg)(cpssp, SEG_TR, 0x0000, 0x00000000, 0xffff, 0);

	NAME_(load_eflags)(cpssp, 0x00000002, 1);
	NAME_(load_cr)(cpssp, 0, 0x60000010);
	NAME_(set_reg32)(cpssp, E_DX, 0x00000600);
}


/* ################ CPU_STEP ############################################### */

static void
NAME_(step)(struct cpssp *cpssp)
{
	bool op32;
	uint8_t vec;
	uint8_t opcode;
#if (_DEBUG_LVL_ > 1)
	uint32_t base = cpssp->NAME.seg_regs[SEG_CS].base;
	static uint32_t log = 0;
#endif


	cpssp->NAME.eip_save = cpssp->NAME.eip;
	cpssp->NAME.esp_save = NAME_(get_reg32)(cpssp, E_SP);

#if (_EXCEPION_MODE_ == LJMP_MODE)
	if (unlikely(setjmp(cpssp->NAME.exception_buf))) {
		cpssp->NAME.eip = cpssp->NAME.eip_save;
		NAME_(set_reg32)(cpssp, E_SP, cpssp->NAME.esp_save);
		return;
	}
#endif

	// TODO find out order for events to test for
	if (unlikely(cpssp->state_event)) {
		if (!cpssp->state_power) {
			return;
		} else if (!cpssp->state_n_reset) {
			NAME_(cpu_reset)(cpssp);
			return;
		} else if (cpssp->NAME.sti) {
			// TODO
			cpssp->NAME.e_if = EFLAGS_IF;
			cpssp->NAME.sti = 0;
		} else if (cpssp->state_nmi_pending) {
			vec = NAME_(ack)(cpssp);
			NAME_(intn)(cpssp, vec, 0);
			cpssp->NAME.state_halt = 0;
			return;
		} else if (cpssp->state_interrupt_pending && cpssp->NAME.e_if) {
			vec = NAME_(ack)(cpssp);
#if (_DEBUG_LVL_ > 1)
			if (log) {
				fprintf(stderr, "0: Interrupt 0x%02x at %08llx (%08llx:%08llx)",
						vec,
						base+cpssp->NAME.eip_save,
						base,
						cpssp->NAME.eip_save);
				NAME_(dump)(cpssp);
			}
#endif
			NAME_(intn)(cpssp, vec, 0);
			cpssp->NAME.state_halt = 0;
			return;
		} else if (cpssp->NAME.exception_num) {
#if (_DEBUG_LVL_ > 1)
			if (log) {
				fprintf(stderr, "0: Interrupt 0x%02x at %08llx (%08llx:%08llx)", /* FIXME not interrupt, not vec... */
						vec,
						base+cpssp->NAME.eip_save,
						base,
						cpssp->NAME.eip_save);
				NAME_(dump)(cpssp);
			}
#endif
			NAME_(intn)(cpssp, cpssp->NAME.exception_num, 0);
			cpssp->NAME.exception_num = 0;
			// TODO error code
			return;
		} else if (cpssp->NAME.state_halt) {
			return;
		} else {
			// TODO like this?
			cpssp->state_event = 0;
		}
	}

	cpssp->NAME.tsc++;


#if (_DEBUG_LVL_ > 1)
	//if ((cpssp->NAME.eip_save + cpssp->NAME.seg_regs[SEG_CS].base) == 0x30A3E) {
	if ((cpssp->NAME.eip_save + cpssp->NAME.seg_regs[SEG_CS].base) == 0xc010019d) {
	//if ((cpssp->NAME.eip_save + cpssp->NAME.seg_regs[SEG_CS].base) == 0x30A3E) {
	//if ((cpssp->NAME.eip_save + cpssp->NAME.seg_regs[SEG_CS].base) == 0x30c30) {
	//if ((cpssp->NAME.eip_save + cpssp->NAME.seg_regs[SEG_CS].base) == 0x7c00) {
		log = 1;
	}
	if (log) {
		log++;
		if (log == 1000100) exit(1);
		//if ((cpssp->NAME.eip_save + cpssp->NAME.seg_regs[SEG_CS].base) == 0x30A3E) {
		//	exit(1);
		//}
	}
	if (log) {
		fprintf(stderr, "0: Executing at %08lx (%08lx:%08lx)",
				base+cpssp->NAME.eip_save,
				base,
				cpssp->NAME.eip_save);

		NAME_(dump)(cpssp);
	}
#endif

FETCH_OPCODE:
	opcode = NAME_(smx8)(cpssp, cpssp->NAME.eip, SEG_CS);
	cpssp->NAME.eip++;

	switch (opcode) {
/* PREFIX */
	case 0x66:	/* op */
		cpssp->NAME.pre_op = 1;
		goto FETCH_OPCODE;

	case 0x67:	/* addr */
		cpssp->NAME.pre_addr = 1;
		goto FETCH_OPCODE;

	case 0x2e:	/* CS */
		cpssp->NAME.seg_override = 1;
		cpssp->NAME.seg[1] = SEG_CS;
		goto FETCH_OPCODE;

	case 0x3e:	/* DS */
		cpssp->NAME.seg_override = 1;
		cpssp->NAME.seg[1] = SEG_DS;
		goto FETCH_OPCODE;

	case 0x26:	/* ES */
		cpssp->NAME.seg_override = 1;
		cpssp->NAME.seg[1] = SEG_ES;
		goto FETCH_OPCODE;

	case 0x36:	/* SS */
		cpssp->NAME.seg_override = 1;
		cpssp->NAME.seg[1] = SEG_SS;
		goto FETCH_OPCODE;

	case 0x64:	/* FS */
		cpssp->NAME.seg_override = 1;
		cpssp->NAME.seg[1] = SEG_FS;
		goto FETCH_OPCODE;

	case 0x65:	/* GS */
		cpssp->NAME.seg_override = 1;
		cpssp->NAME.seg[1] = SEG_GS;
		goto FETCH_OPCODE;

	case 0xf2:	/* repne/repnz */
		cpssp->NAME.pre_repe_repne = 0x1;
		goto FETCH_OPCODE;

	case 0xf3:	/* rep/repe/repz */
		cpssp->NAME.pre_repe_repne = 0x2;
		goto FETCH_OPCODE;


/* ADD/ADC/AND/XOR/OR/SBB/SUB/CMP */
	case 0x00:	/* add/adc/and/xor/or/sbb/sub/cmp rm8, r8 */
	case 0x10:
	case 0x20:
	case 0x30:
	case 0x08:
	case 0x18:
	case 0x28:
	case 0x38:
		NAME_(get_modrm)(cpssp);
		NAME_(sload_op2_reg8)(cpssp, cpssp->NAME.reg_opex);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op1_reg8)(cpssp, cpssp->NAME.rm);
			NAME_(alu_arith)(cpssp, GET_OPC(opcode), 8);
			NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op1_mem8)(cpssp, 1);
			NAME_(alu_arith)(cpssp, GET_OPC(opcode), 8);
			NAME_(store_res_mem8)(cpssp);
		}
		break;

	case 0x01:	/* add/adc/and/xor/or/sbb/sub/cmp rm16/32, r16/32 */
	case 0x11:
	case 0x21:
	case 0x31:
	case 0x09:
	case 0x19:
	case 0x29:
	case 0x39:
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		NAME_(sload_op2_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			NAME_(alu_arith)(cpssp, GET_OPC(opcode), 16 << op32);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op1_mem16_32)(cpssp, op32, 1);
			NAME_(alu_arith)(cpssp, GET_OPC(opcode), 16 << op32);
			NAME_(store_res_mem16_32)(cpssp, op32);
		}
		break;

	case 0x02:	/* add/adc/and/xor/or/sbb/sub/cmp r8, rm8 */
	case 0x12:
	case 0x22:
	case 0x32:
	case 0x0a:
	case 0x1a:
	case 0x2a:
	case 0x3a:
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op2_reg8)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op2_mem8)(cpssp, 0);
		}
		NAME_(sload_op1_reg8)(cpssp, cpssp->NAME.reg_opex);
		NAME_(alu_arith)(cpssp, GET_OPC(opcode), 8);
		NAME_(store_res_reg8)(cpssp, cpssp->NAME.reg_opex);
		break;

	case 0x03:	/* add/adc/and/xor/or/sbb/sub/cmp r16/32, rm16/32 */
	case 0x13:
	case 0x23:
	case 0x33:
	case 0x0b:
	case 0x1b:
	case 0x2b:
	case 0x3b:
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op2_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op2_mem16_32)(cpssp, op32, 0);
		}
		NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		NAME_(alu_arith)(cpssp, GET_OPC(opcode), 16 << op32);
		NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		break;

	case 0x04:	/* add/adc/and/xor/or/sbb/sub/cmp al, imm8 */
	case 0x14:
	case 0x24:
	case 0x34:
	case 0x0c:
	case 0x1c:
	case 0x2c:
	case 0x3c:
		NAME_(sload_op1_reg8)(cpssp, AL);
		NAME_(sload_op2_imm8)(cpssp);
		NAME_(alu_arith)(cpssp, GET_OPC(opcode), 8);
		NAME_(store_res_reg8)(cpssp, AL);
		break;

	case 0x05:	/* add/adc/and/xor/or/sbb/sub/cmp e/ax, imm16/32 */
	case 0x15:
	case 0x25:
	case 0x35:
	case 0x0d:
	case 0x1d:
	case 0x2d:
	case 0x3d:
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(sload_op1_reg16_32)(cpssp, E_AX, op32);
		NAME_(sload_op2_imm16_32)(cpssp, op32);
		NAME_(alu_arith)(cpssp, GET_OPC(opcode), 16 << op32);
		NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
		break;

	case 0x80:	/* add/adc/and/xor/or/sbb/sub/cmp rm8, imm8 */
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op2_imm8)(cpssp);
			NAME_(sload_op1_reg8)(cpssp, cpssp->NAME.rm);
			NAME_(alu_arith)(cpssp, cpssp->NAME.reg_opex, 8);
			NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op2_imm8)(cpssp);
			NAME_(sload_op1_mem8)(cpssp, 1);
			NAME_(alu_arith)(cpssp, cpssp->NAME.reg_opex, 8);
			NAME_(store_res_mem8)(cpssp);
		}
		break;

	case 0x81:	/* add/adc/and/xor/or/sbb/sub/cmp rm16/32, imm16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op2_imm16_32)(cpssp, op32);
			NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			NAME_(alu_arith)(cpssp, cpssp->NAME.reg_opex, 16 << op32);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op2_imm16_32)(cpssp, op32);
			NAME_(sload_op1_mem16_32)(cpssp, op32, 1);
			NAME_(alu_arith)(cpssp, cpssp->NAME.reg_opex, 16 << op32);
			NAME_(store_res_mem16_32)(cpssp, op32);
		}
		break;

	case 0x83:	/* add/adc/and/xor/or/sbb/sub/cmp rm16/32, imm8 se */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op2_imm8)(cpssp);
			NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			NAME_(alu_arith)(cpssp, cpssp->NAME.reg_opex, 16 << op32);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op2_imm8)(cpssp);
			NAME_(sload_op1_mem16_32)(cpssp, op32, 1);
			NAME_(alu_arith)(cpssp, cpssp->NAME.reg_opex, 16 << op32);
			NAME_(store_res_mem16_32)(cpssp, op32);
		}
		break;


/* INC */
	case 0x40:	/* inc r16/32 */
	case 0x41:
	case 0x42:
	case 0x43:
	case 0x44:
	case 0x45:
	case 0x46:
	case 0x47:
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(sload_op1_reg16_32)(cpssp, GET_REG(opcode), op32);
		NAME_(inc)(cpssp, 16 << op32);
		NAME_(store_res_reg16_32)(cpssp, GET_REG(opcode), op32);
		break;


/* DEC */
	case 0x48:	/* dec r16/32 */
	case 0x49:
	case 0x4a:
	case 0x4b:
	case 0x4c:
	case 0x4d:
	case 0x4e:
	case 0x4f:
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(sload_op1_reg16_32)(cpssp, GET_REG(opcode), op32);
		NAME_(dec)(cpssp, 16 << op32);
		NAME_(store_res_reg16_32)(cpssp, GET_REG(opcode), op32);
		break;


/* IMUL */
	case 0x6b:	/* imul r16/32, rm16/32, imm8 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op2_imm8)(cpssp);
			NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op2_imm8)(cpssp);
			NAME_(sload_op1_mem16_32)(cpssp, op32, 0);
		}
		NAME_(imul16_32)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		break;

	case 0x69:	/* imul r16/32, rm16/32, imm16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op2_imm16_32)(cpssp, op32);
			NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op2_imm16_32)(cpssp, op32);
			NAME_(sload_op1_mem16_32)(cpssp, op32, 0);
		}
		NAME_(imul16_32)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		break;


/* PUSH */
		// TODO in realmode trip-fault when E_SP == 1?
	case 0x0e:	/* push CS */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(load_op1_sreg)(cpssp, SEG_CS);
		NAME_(push)(cpssp, op32);
		break;

	case 0x16:	/* push SS */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(load_op1_sreg)(cpssp, SEG_SS);
		NAME_(push)(cpssp, op32);
		break;

	case 0x1e:	/* push DS */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(load_op1_sreg)(cpssp, SEG_DS);
		NAME_(push)(cpssp, op32);
		break;

	case 0x06:	/* push ES */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(load_op1_sreg)(cpssp, SEG_ES);
		NAME_(push)(cpssp, op32);
		break;

	case 0x50:	/* push reg16/32 */
	case 0x51:
	case 0x52:
	case 0x53:
	case 0x54:
	case 0x55:
	case 0x56:
	case 0x57:
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op1_reg16_32)(cpssp, GET_REG(opcode), op32);
		NAME_(push)(cpssp, op32);
		break;

	case 0x60:	/* pusha/pushad */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op2_reg16_32)(cpssp, E_SP, op32);
		NAME_(uload_op1_reg16_32)(cpssp, E_AX, op32);
		NAME_(push)(cpssp, op32);
		NAME_(uload_op1_reg16_32)(cpssp, E_CX, op32);
		NAME_(push)(cpssp, op32);
		NAME_(uload_op1_reg16_32)(cpssp, E_DX, op32);
		NAME_(push)(cpssp, op32);
		NAME_(uload_op1_reg16_32)(cpssp, E_BX, op32);
		NAME_(push)(cpssp, op32);
		cpssp->NAME.op1 = cpssp->NAME.op2;
		NAME_(push)(cpssp, op32);
		NAME_(uload_op1_reg16_32)(cpssp, E_BP, op32);
		NAME_(push)(cpssp, op32);
		NAME_(uload_op1_reg16_32)(cpssp, E_SI, op32);
		NAME_(push)(cpssp, op32);
		NAME_(uload_op1_reg16_32)(cpssp, E_DI, op32);
		NAME_(push)(cpssp, op32);
		break;

	case 0x6a:	/* push imm8 se */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(push)(cpssp, op32);
		break;

	case 0x68:	/* push imm16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op1_imm16_32)(cpssp, op32);
		NAME_(push)(cpssp, op32);
		break;

	case 0x9c:	/* pushf/pushfd */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(load_op1_eflags)(cpssp,op32);
		NAME_(push)(cpssp, op32);
		break;


/* POP */
	case 0x1f:	/* pop DS */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_sreg)(cpssp, SEG_DS);
		break;

	case 0x07:	/* pop ES */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_sreg)(cpssp, SEG_ES);
		break;

	case 0x17:	/* pop SS */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_sreg)(cpssp, SEG_SS);
		break;

	case 0x58:	/* pop reg16/32 */
	case 0x59:
	case 0x5a:
	case 0x5b:
	case 0x5c:
	case 0x5d:
	case 0x5e:
	case 0x5f:
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, GET_REG(opcode), op32);
		break;

	case 0x61:	/* popa/popad */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_DI, op32);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_SI, op32);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_BP, op32);
		NAME_(uload_op1_reg16_32)(cpssp, E_SP, op32);
		cpssp->NAME.res = cpssp->NAME.op1 + 2 + (2 * op32);
		NAME_(store_res_reg16_32)(cpssp, E_SP, op32);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_BX, op32);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_DX, op32);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_CX, op32);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
		break;

	case 0x9d:	/* popf/popfd */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_eflags)(cpssp, op32);
		break;


/* JMP */
	case 0xe9:	/* jmp rel16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(sload_op1_imm16_32)(cpssp, op32);
		NAME_(jmp)(cpssp);
		break;

	case 0xea:	/* ljmp ptr16:16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op1_imm16_32)(cpssp, op32);
		NAME_(uload_op2_imm16)(cpssp);
		NAME_(ljmp)(cpssp);
		break;

	case 0xeb:	/* jmp rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(jmp)(cpssp);
		break;


/* Jcc */
	case 0xe3:	/* jcxz/jecxz rel8 */
		op32 = NAME_(addr32_mode)(cpssp);
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(uload_op2_reg16_32)(cpssp, E_CX, op32);
		NAME_(test_ecx)(cpssp);
		NAME_(jcc)(cpssp, 0);
		break;

	case 0x70:	/* jo rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_of)(cpssp);
		NAME_(jcc)(cpssp, 0);
		break;

	case 0x7a:	/* jp/jpe rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_pf)(cpssp);
		NAME_(jcc)(cpssp, 0);
		break;

	case 0x71:	/* jno rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_of)(cpssp);
		NAME_(jcc)(cpssp, 1);
		break;

	case 0x7b:	/* jnp/jpo rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_pf)(cpssp);
		NAME_(jcc)(cpssp, 1);
		break;

	case 0x78:	/* js rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_sf)(cpssp);
		NAME_(jcc)(cpssp, 0);
		break;

	case 0x79:	/* jns rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_sf)(cpssp);
		NAME_(jcc)(cpssp, 1);
		break;

	case 0x77:	/* ja/jnbe rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_na_be)(cpssp);
		NAME_(jcc)(cpssp, 1);
		break;

	case 0x73:	/* jae/jnb/jnc rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_cf)(cpssp);
		NAME_(jcc)(cpssp, 1);
		break;

	case 0x72:	/* jb/jc/jnae rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_cf)(cpssp);
		NAME_(jcc)(cpssp, 0);
		break;

	case 0x76:	/* jbe/jna rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_na_be)(cpssp);
		NAME_(jcc)(cpssp, 0);
		break;

	case 0x74:	/* je/jz rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_zf)(cpssp);
		NAME_(jcc)(cpssp, 0);
		break;

	case 0x7f:	/* jg/jnle rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_ng_le)(cpssp);
		NAME_(jcc)(cpssp, 1);
		break;

	case 0x7d:	/* jge/jnl rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_l_nge)(cpssp);
		NAME_(jcc)(cpssp, 1);
		break;

	case 0x7c:	/* jl/jnge rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_l_nge)(cpssp);
		NAME_(jcc)(cpssp, 0);
		break;

	case 0x7e:	/* jle/jng rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_ng_le)(cpssp);
		NAME_(jcc)(cpssp, 0);
		break;

	case 0x75:	/* jne/jnz rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_zf)(cpssp);
		NAME_(jcc)(cpssp, 1);
		break;


/* TEST */
	case 0x84:	/* test rm8, r8 */
		NAME_(get_modrm)(cpssp);
		NAME_(sload_op2_reg8)(cpssp, cpssp->NAME.reg_opex);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op1_reg8)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op1_mem8)(cpssp, 0);
		}
		NAME_(test)(cpssp);
		break;

	case 0x85:	/* test rm16/32, r16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		NAME_(sload_op2_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op1_mem16_32)(cpssp, op32, 0);
		}
		NAME_(test)(cpssp);
		break;

	case 0xa8:	/* test al, imm8 */
		NAME_(sload_op1_reg8)(cpssp, AL);
		NAME_(sload_op2_imm8)(cpssp);
		NAME_(test)(cpssp);
		break;

	case 0xa9:	/* test e/ax, imm16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(sload_op1_reg16_32)(cpssp, E_AX, op32);
		NAME_(sload_op2_imm16_32)(cpssp, op32);
		NAME_(test)(cpssp);
		break;


/* LEA */
	case 0x8d:	/* lea r16/32, m */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		NAME_(get_addr)(cpssp);
		NAME_(lea)(cpssp);
		NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		break;


/* XCHG */
	case 0x86:	/* xchg rm8, r8 / xchg r8, rm8 */
		NAME_(get_modrm)(cpssp);
		NAME_(uload_op2_reg8)(cpssp, cpssp->NAME.reg_opex);
		NAME_(xchg)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(uload_op2_reg8)(cpssp, cpssp->NAME.rm);
			NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			NAME_(xchg)(cpssp);
			NAME_(store_res_reg8)(cpssp, cpssp->NAME.reg_opex);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(uload_op2_mem8)(cpssp, 1);
			NAME_(store_res_mem8)(cpssp);
			NAME_(xchg)(cpssp);
			NAME_(store_res_reg8)(cpssp, cpssp->NAME.reg_opex);
		}
		break;

	case 0x87:	/* xchg rm16/32, r16/32 / xchg r16/32, rm16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		NAME_(uload_op2_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		NAME_(xchg)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(uload_op2_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			NAME_(xchg)(cpssp);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(uload_op2_mem16_32)(cpssp, op32, 1);
			NAME_(store_res_mem16_32)(cpssp, op32);
			NAME_(xchg)(cpssp);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		}
		break;

	case 0x90:	/* xchg e/ax, r16/32 / xchg r16/32, e/ax */
	case 0x91:
	case 0x92:
	case 0x93:
	case 0x94:
	case 0x95:
	case 0x96:
	case 0x97:
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op2_reg16_32)(cpssp, E_AX, op32);
		NAME_(xchg)(cpssp);
		NAME_(uload_op2_reg16_32)(cpssp, GET_REG(opcode), op32);
		NAME_(store_res_reg16_32)(cpssp, GET_REG(opcode), op32);
		NAME_(xchg)(cpssp);
		NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
		break;


/* MODRM-OPCEX */
	case 0x8f:
		NAME_(get_modrm)(cpssp);
		switch (cpssp->NAME.reg_opex) {
/* POP */
		case 0x0:	/* pop rm16_32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(pop)(cpssp, op32);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem16_32)(cpssp, op32);
				// TODO test before pop or else save esp
			}
			break;


		default:
			_ERROR("Unknown opcode " RED(0x%02x) "@" YELLOW(0x%08x) " with opcode-extension " RED(0x%01x) "\n", opcode,
					cpssp->NAME.eip_save, cpssp->NAME.reg_opex);
			exit(1);
		}
		break;


/* MOV */	// TODO
	case 0x88:	/* mov rm8, r8 */
		NAME_(get_modrm)(cpssp);
		NAME_(sload_op2_reg8)(cpssp, cpssp->NAME.reg_opex);
		NAME_(mov)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(store_res_mem8)(cpssp);
		}
		break;

	case 0x89:	/* mov rm16/32, r16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		NAME_(sload_op2_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		NAME_(mov)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(store_res_mem16_32)(cpssp, op32);
		}
		break;

	case 0x8a:	/* mov r8, rm8 */
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(sload_op2_reg8)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op2_mem8)(cpssp, 0);
		}
		NAME_(mov)(cpssp);
		NAME_(store_res_reg8)(cpssp, cpssp->NAME.reg_opex);
		break;

	case 0x8b:	/* mov r16/32, rm16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			//TODO uload
			NAME_(sload_op2_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(sload_op2_mem16_32)(cpssp, op32, 0);
		}
		NAME_(mov)(cpssp);
		NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		break;

	case 0x8c:	/* mov rm16, Sreg */
		NAME_(get_modrm)(cpssp);
		NAME_(mov_from_sreg)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(store_res_reg16)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(store_res_mem16)(cpssp);
		}
		break;

		// TODO sti bei ss
	case 0x8e:	/* mov Sreg, rm16 */
		NAME_(get_modrm)(cpssp);
		if (unlikely((cpssp->NAME.reg_opex > 0x5) || (SEG_CS == cpssp->NAME.reg_opex))) {
			// TODO GP
			_ERROR("invalid seg_reg: " RED(%01d) "\n", cpssp->NAME.reg_opex);
			exit(1);
		}
		if (0x3 == cpssp->NAME.mod) {
			NAME_(uload_op2_reg16)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(uload_op2_mem16)(cpssp, 0);
		}
		NAME_(mov_to_sreg)(cpssp);
		break;

		// TODO what op-size what addr-size, intel manual makes no sense... ask!
	case 0xa0:	/* mov al, moffs8 */
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		//NAME_(uload_op1_imm8)(cpssp);
		NAME_(uload_op1_imm16_32)(cpssp, NAME_(addr32_mode)(cpssp));
		cpssp->NAME.addr = cpssp->NAME.op1;
		NAME_(uload_op2_mem8)(cpssp, 0);
		NAME_(mov)(cpssp);
		NAME_(store_res_reg8)(cpssp, AL);
		break;

	case 0xa1:	/* mov e/ax, moffs16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		NAME_(uload_op1_imm16_32)(cpssp, NAME_(addr32_mode)(cpssp));
		cpssp->NAME.addr = cpssp->NAME.op1;
		NAME_(uload_op2_mem16_32)(cpssp, op32, 0);
		NAME_(mov)(cpssp);
		NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
		break;

	case 0xa2:	/* mov moffs8, al */
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		//NAME_(uload_op1_imm8)(cpssp);
		NAME_(uload_op1_imm16_32)(cpssp, NAME_(addr32_mode)(cpssp));
		cpssp->NAME.addr = cpssp->NAME.op1;
		NAME_(uload_op2_reg8)(cpssp, AL);
		NAME_(mov)(cpssp);
		NAME_(store_res_mem8)(cpssp);
		break;

	case 0xa3:	/* mov moffs16/32, e/ax */
		op32 = NAME_(op32_mode)(cpssp);
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		NAME_(uload_op1_imm16_32)(cpssp, NAME_(addr32_mode)(cpssp));
		cpssp->NAME.addr = cpssp->NAME.op1;
		NAME_(uload_op2_reg16_32)(cpssp, E_AX, op32);
		NAME_(mov)(cpssp);
		NAME_(store_res_mem16_32)(cpssp, op32);
		break;

	case 0xb0:	/* mov r8, imm8 */
	case 0xb1:
	case 0xb2:
	case 0xb3:
	case 0xb4:
	case 0xb5:
	case 0xb6:
	case 0xb7:
		NAME_(uload_op2_imm8)(cpssp);
		NAME_(mov)(cpssp);
		NAME_(store_res_reg8)(cpssp, GET_REG(opcode));
		break;

	case 0xb8:	/* mov r16/32, imm16/32 */
	case 0xb9:
	case 0xba:
	case 0xbb:
	case 0xbc:
	case 0xbd:
	case 0xbe:
	case 0xbf:
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(sload_op2_imm16_32)(cpssp, op32);
		NAME_(mov)(cpssp);
		NAME_(store_res_reg16_32)(cpssp, GET_REG(opcode), op32);
		break;


/* CBW/CWDE */
	case 0x98:	/* cbw/cwde */
		op32 = NAME_(op32_mode)(cpssp);
		if (likely(op32)) {
			NAME_(sload_op1_reg16)(cpssp, E_AX);
		} else {
			NAME_(sload_op1_reg8)(cpssp, AL);
		}
		cpssp->NAME.res = cpssp->NAME.op1;
		NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
		break;


/* CWD/CDQ */
	case 0x99:	/* cwd/cdq */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(sload_op1_reg16_32)(cpssp, E_AX, op32);
		NAME_(cwd)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_DX, op32);
		break;


/* CALL */
	case 0x9a:	/* call ptr16:16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op1_imm16_32)(cpssp, op32);
		NAME_(uload_op2_imm16)(cpssp);
		NAME_(fcall)(cpssp, op32);
		break;

	case 0xe8:	/* call rel16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(sload_op1_imm16_32)(cpssp, op32);
		NAME_(call)(cpssp, op32);
		break;


/* RET */
	case 0xc3:	/* ret */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(ret)(cpssp, op32);
		break;

	case 0xcb:	/* ret */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(fret)(cpssp, op32);
		break;


/* MOVS */
		// TODO how often? has to be fast?
	case 0xa4:	/* movsb / movs m8 */
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		if (0x2 == cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep)(cpssp)) {
				break;
			}
		}
		NAME_(movs8)(cpssp);
		break;

	case 0xa5:	/* movsw/movsd / movs m16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		if (0x2 == cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep)(cpssp)) {
				break;
			}
		}
		NAME_(movs16_32)(cpssp, op32);
		break;


/* CMPS */
	case 0xa6:	/* cmpsb / cmps m8, m8 */
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		if (cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep_z_nz)(cpssp)) {
				break;
			}
			NAME_(cmps8)(cpssp);
			NAME_(repn_z)(cpssp);
		}
		NAME_(cmps8)(cpssp);
		break;

	case 0xa7:	/* cmpsw/cmpsd / cmps m16/32, m16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		if (cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep_z_nz)(cpssp)) {
				break;
			}
			NAME_(cmps16_32)(cpssp, op32);
			NAME_(repn_z)(cpssp);
		}
		NAME_(cmps16_32)(cpssp, op32);
		break;


/* STOS */
	case 0xaa:	/* stos m8 */
		if (0x2 == cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep)(cpssp)) {
				break;
			}
		}
		NAME_(stos8)(cpssp);
		break;

	case 0xab:	/* stos m16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		if (0x2 == cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep)(cpssp)) {
				break;
			}
		}
		NAME_(stos16_32)(cpssp, op32);
		break;


/* LODS */
	case 0xac:	/* lods m8 */
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		if (0x2 == cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep)(cpssp)) {
				break;
			}
		}
		NAME_(lods8)(cpssp);
		break;

	case 0xad:	/* lods m16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		if (0x2 == cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep)(cpssp)) {
				break;
			}
		}
		NAME_(lods16_32)(cpssp, op32);
		break;


/* SCAS */
	case 0xae:	/* scasb / scas m8 */
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		if (cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep_z_nz)(cpssp)) {
				break;
			}
			NAME_(scas8)(cpssp);
			NAME_(repn_z)(cpssp);
		}
		NAME_(scas8)(cpssp);
		break;

	case 0xaf:	/* scasw/scasd / scas m16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		if (cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep_z_nz)(cpssp)) {
				break;
			}
			NAME_(scas16_32)(cpssp, op32);
			NAME_(repn_z)(cpssp);
		}
		NAME_(scas16_32)(cpssp, op32);
		break;


/* INS */
	case 0x6c:	/* ins m8 */
		if (0x2 == cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep)(cpssp)) {
				break;
			}
		}
		NAME_(ins8)(cpssp);
		break;

	case 0x6d:	/* ins m16/32 */
		op32 = NAME_(op32_mode)(cpssp);
		if (0x2 == cpssp->NAME.pre_repe_repne) {
			if (!NAME_(rep)(cpssp)) {
				break;
			}
		}
		NAME_(ins16_32)(cpssp, op32);
		break;


/* SHIFT */
	case 0xc0:	/* sal/shl/sar/shr rm8, imm8 */
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(uload_op2_imm8)(cpssp);
			NAME_(uload_op1_reg8)(cpssp, cpssp->NAME.rm);
			NAME_(alu_bit)(cpssp, 8);
			NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(uload_op2_imm8)(cpssp);
			NAME_(uload_op1_mem8)(cpssp, 1);
			NAME_(alu_bit)(cpssp, 8);
			NAME_(store_res_mem8)(cpssp);
		}
		break;

	case 0xc1:	/* sal/shl/sar/shr rm16/32, imm8 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(uload_op2_imm8)(cpssp);
			NAME_(uload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			NAME_(alu_bit)(cpssp, 16 << op32);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(uload_op2_imm8)(cpssp);
			NAME_(uload_op1_mem16_32)(cpssp, op32, 1);
			NAME_(alu_bit)(cpssp, 16 << op32);
			NAME_(store_res_mem16_32)(cpssp, op32);
		}
		break;

	case 0xd0:	/* sal/shl/sar/shr rm8, 1 */
		NAME_(get_modrm)(cpssp);
		cpssp->NAME.op2 = 1;
		if (0x3 == cpssp->NAME.mod) {
			NAME_(uload_op1_reg8)(cpssp, cpssp->NAME.rm);
			NAME_(alu_bit)(cpssp, 8);
			NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(uload_op1_mem8)(cpssp, 1);
			NAME_(alu_bit)(cpssp, 8);
			NAME_(store_res_mem8)(cpssp);
		}
		break;

	case 0xd1:	/* sal/shl/sar/shr rm16/32, 1 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		cpssp->NAME.op2 = 1;
		if (0x3 == cpssp->NAME.mod) {
			NAME_(uload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			NAME_(alu_bit)(cpssp, 16 << op32);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(uload_op1_mem16_32)(cpssp, op32, 1);
			NAME_(alu_bit)(cpssp, 16 << op32);
			NAME_(store_res_mem16_32)(cpssp, op32);
		}
		break;

	case 0xd2:	/* sal/shl/sar/shr rm8, cl */
		NAME_(get_modrm)(cpssp);
		NAME_(uload_op2_reg8)(cpssp, CL);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(uload_op1_reg8)(cpssp, cpssp->NAME.rm);
			NAME_(alu_bit)(cpssp, 8);
			NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(uload_op1_mem8)(cpssp, 1);
			NAME_(alu_bit)(cpssp, 8);
			NAME_(store_res_mem8)(cpssp);
		}
		break;

	case 0xd3:	/* sal/shl/sar/shr rm16/32, cl */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		NAME_(uload_op2_reg8)(cpssp, CL);
		if (0x3 == cpssp->NAME.mod) {
			NAME_(uload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			NAME_(alu_bit)(cpssp, 16 << op32);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
		} else {
			NAME_(get_addr)(cpssp);
			NAME_(uload_op1_mem16_32)(cpssp, op32, 1);
			NAME_(alu_bit)(cpssp, 16 << op32);
			NAME_(store_res_mem16_32)(cpssp, op32);
		}
		break;


/* MODRM-OPCEX */
	case 0xc6:
		NAME_(get_modrm)(cpssp);
		switch (cpssp->NAME.reg_opex) {
/* MOV */	// TODO
		case 0x0:	/* mov rm8, imm8 */
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op2_imm8)(cpssp);
				NAME_(mov)(cpssp);
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op2_imm8)(cpssp);
				NAME_(mov)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;


		default:
			_ERROR("Unknown opcode " RED(0x%02x) "@" YELLOW(0x%08x) " with opcode-extension " RED(0x%01x) "\n", opcode,
					cpssp->NAME.eip_save, cpssp->NAME.reg_opex);
			exit(1);
		}
		break;


/* MODRM-OPCEX */
	case 0xc7:
		NAME_(get_modrm)(cpssp);
		switch (cpssp->NAME.reg_opex) {
/* MOV */	// TODO
		case 0x0:	/* mov rm16/32, imm16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op2_imm16_32)(cpssp, op32);
				NAME_(mov)(cpssp);
				NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op2_imm16_32)(cpssp, op32);
				NAME_(mov)(cpssp);
				NAME_(store_res_mem16_32)(cpssp, op32);
			}
			break;


		default:
			_ERROR("Unknown opcode " RED(0x%02x) "@" YELLOW(0x%08x) " with opcode-extension " RED(0x%01x) "\n", opcode,
					cpssp->NAME.eip_save, cpssp->NAME.reg_opex);
			exit(1);
		}
		break;


/* LEAVE */
	case 0xc9:	/* leave */
		op32 = NAME_(stack32_mode)(cpssp);
		NAME_(uload_op2_reg16_32)(cpssp, E_BP, op32);
		NAME_(mov)(cpssp);
		NAME_(store_res_reg16_32)(cpssp, E_SP, op32);
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(pop)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_BP, op32);
		break;


/* INT */
	case 0xcc:	/* int3 */
		NAME_(intn)(cpssp, 3, 1);
		break;

	case 0xcd:	/* int imm8 */
		NAME_(uload_op1_imm8)(cpssp);
#if (_DEBUG_LVL_ > 1)
		if (log) {
			fprintf(stderr, "0: Exception 0x%02x at %08llx (%08llx:%08llx)",
					cpssp->NAME.op1,
					base+cpssp->NAME.eip_save,
					base,
					cpssp->NAME.eip_save);
			NAME_(dump)(cpssp);
		}
#endif
		NAME_(intn)(cpssp, cpssp->NAME.op1, 1);
		break;

	case 0xce:	/* into */
		if (cpssp->NAME.e_of) {
			NAME_(intn)(cpssp, 4, 1);
		}
		break;


/* IRET */
	case 0xcf:	/* iret */
		op32 = NAME_(op32_mode)(cpssp);
#if (_DEBUG_LVL_ > 1)
		if (log) {
			fprintf(stderr, "0: iret at %08lx (%08lx:%08lx)",
					base+cpssp->NAME.eip_save,
					base,
					cpssp->NAME.eip_save);
			NAME_(dump)(cpssp);
		}
#endif
		NAME_(iret)(cpssp, op32);
		break;


/* LDS/LES */
	case 0xc4:	/* les r16/32, m16:16/32 */
		op32 = NAME_(addr32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		NAME_(get_addr)(cpssp);
		NAME_(uload_op_m16_16_32)(cpssp, op32);
		NAME_(l_s)(cpssp, SEG_ES);
		NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		break;

	case 0xc5:	/* lds r16/32, m16:16/32 */
		op32 = NAME_(addr32_mode)(cpssp);
		NAME_(get_modrm)(cpssp);
		NAME_(get_addr)(cpssp);
		NAME_(uload_op_m16_16_32)(cpssp, op32);
		NAME_(l_s)(cpssp, SEG_DS);
		NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
		break;


/* XLAT */
	case 0xd7:	/* xlat/xlatb */
		op32 = NAME_(addr32_mode)(cpssp);
		cpssp->NAME.seg[0] = SEG_DS;
		NAME_(check_seg_over)(cpssp);
		NAME_(xlat)(cpssp, op32);
		NAME_(store_res_reg8)(cpssp, AL);
		break;


/* FPU */
		/* FIXME we don't have a FPU so ignore it... */
	case 0xd8:	/* ESC */
	case 0xd9:
	case 0xda:
	case 0xdb:
	case 0xdc:
	case 0xdd:
	case 0xde:
	case 0xdf:
		NAME_(get_modrm)(cpssp);
		if (0x3 != cpssp->NAME.mod) {
			NAME_(get_addr)(cpssp);
		}
		_ERROR("[ESC] Not implemented! Ignored.\n")
		break;

	case 0x9b:	/* wait / fwait */
		_ERROR("[WAIT/FWAIT] Not implemented! Ignored.\n")
		break;


/* LOOP/LOOPcc */
	case 0xe0:	/* loopne/loopnz rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_zf)(cpssp);
		NAME_(loopcc)(cpssp, 1);
		break;

	case 0xe1:	/* loope/loopz rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(test_zf)(cpssp);
		NAME_(loopcc)(cpssp, 0);
		break;

	case 0xe2:	/* loop rel8 */
		NAME_(sload_op1_imm8)(cpssp);
		NAME_(loop)(cpssp);
		break;


/* IN */
	case 0xe4:	/* in al, imm8 */
		NAME_(uload_op1_imm8)(cpssp);
		NAME_(in8)(cpssp);
		NAME_(store_res_reg8)(cpssp, AL);
		break;

	case 0xe5:	/* in e/ax, imm8 */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op1_imm8)(cpssp);
		NAME_(in16_32)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
		break;

	case 0xec:	/* in al, dx */
		NAME_(uload_op1_reg16)(cpssp, E_DX);
		NAME_(in8)(cpssp);
		NAME_(store_res_reg8)(cpssp, AL);
		break;

	case 0xed:	/* in e/ax, dx */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op1_reg16)(cpssp, E_DX);
		NAME_(in16_32)(cpssp, op32);
		NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
		break;


/* OUT */
	case 0xe6:	/* out imm8, al */
		NAME_(uload_op1_imm8)(cpssp);
		NAME_(uload_op2_reg8)(cpssp, AL);
		NAME_(out8)(cpssp);
		break;

	case 0xe7:	/* out imm8, e/ax */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op1_imm8)(cpssp);
		NAME_(uload_op2_reg32)(cpssp, E_AX);
		NAME_(out16_32)(cpssp, op32);
		break;

	case 0xee:	/* out dx, al */
		NAME_(uload_op1_reg16)(cpssp, E_DX);
		NAME_(uload_op2_reg8)(cpssp, AL);
		NAME_(out8)(cpssp);
		break;

	case 0xef:	/* out dx, e/ax */
		op32 = NAME_(op32_mode)(cpssp);
		NAME_(uload_op1_reg16)(cpssp, E_DX);
		NAME_(uload_op2_reg32)(cpssp, E_AX);
		NAME_(out16_32)(cpssp, op32);
		break;


/* MODRM-OPCEX */
	case 0xf6:
		NAME_(get_modrm)(cpssp);
		switch (cpssp->NAME.reg_opex) {
/* TEST */
		case 0x0:	/* test rm8, imm8 */
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op2_imm8)(cpssp);
				NAME_(sload_op1_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op2_imm8)(cpssp);
				NAME_(sload_op1_mem8)(cpssp, 0);
			}
			NAME_(test)(cpssp);
			break;

/* NOT */
		case 0x2:	/* not rm8 */
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op1_reg8)(cpssp, cpssp->NAME.rm);
				NAME_(not)(cpssp);
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op1_mem8)(cpssp, 1);
				NAME_(not)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

/* NEG */
		case 0x3:	/* neg rm8 */
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op1_reg8)(cpssp, cpssp->NAME.rm);
				NAME_(neg)(cpssp, 8);
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op1_mem8)(cpssp, 1);
				NAME_(neg)(cpssp, 8);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

/* MUL */
		case 0x4:	/* mul rm8 (AX <- AL * rm8) */
			NAME_(uload_op1_reg8)(cpssp, AL);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op2_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op2_mem8)(cpssp, 0);
			}
			NAME_(mul8)(cpssp);
			NAME_(store_res_reg16)(cpssp, E_AX);
			break;

/* IMUL */
		case 0x5:	/* imul rm8 (AX <- AL * rm8) */
			NAME_(sload_op1_reg8)(cpssp, AL);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op2_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op2_mem8)(cpssp, 0);
			}
			NAME_(imul8)(cpssp);
			NAME_(store_res_reg16)(cpssp, E_AX);
			break;

/* DIV */
		case 0x6:	/* div rm8 (AL <- AX / rm8, AH <- AX % rm8) */
			NAME_(uload_op1_reg16)(cpssp, E_AX);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op2_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op2_mem8)(cpssp, 0);
			}
			NAME_(div8)(cpssp);
			NAME_(store_res_reg8)(cpssp, AL);
			cpssp->NAME.res = cpssp->NAME.op1;
			NAME_(store_res_reg8)(cpssp, AH);
			break;

/* IDIV */
		case 0x7:	/* idiv rm8 (AL <- AX / rm8, AH <- AX % rm8) */
			NAME_(sload_op1_reg16)(cpssp, E_AX);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op2_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op2_mem8)(cpssp, 0);
			}
			NAME_(idiv8)(cpssp);
			NAME_(store_res_reg8)(cpssp, AL);
			cpssp->NAME.res = cpssp->NAME.op1;
			NAME_(store_res_reg8)(cpssp, AH);
			break;

		default:
			_ERROR("Unknown opcode " RED(0x%02x) "@" YELLOW(0x%08x) " with opcode-extension " RED(0x%01x) "\n", opcode,
					cpssp->NAME.eip_save, cpssp->NAME.reg_opex);
			exit(1);
		}
		break;


/* MODRM-OPCEX */
	case 0xf7:
		NAME_(get_modrm)(cpssp);
		switch (cpssp->NAME.reg_opex) {
/* TEST */
		case 0x0:	/* test rm16/32, imm16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op2_imm16_32)(cpssp, op32);
				NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op2_imm16_32)(cpssp, op32);
				NAME_(sload_op1_mem16_32)(cpssp, op32, 0);
			}
			NAME_(test)(cpssp);
			break;

/* NOT */
		case 0x2:	/* not rm8 */
			op32 = NAME_(op32_mode)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
				NAME_(not)(cpssp);
				NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op1_mem16_32)(cpssp, op32, 1);
				NAME_(not)(cpssp);
				NAME_(store_res_mem16_32)(cpssp, op32);
			}
			break;

/* NEG */
		case 0x3:	/* neg rm8 */
			op32 = NAME_(op32_mode)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
				NAME_(neg)(cpssp, 16 << op32);
				NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op1_mem16_32)(cpssp, op32, 1);
				NAME_(neg)(cpssp, 16 << op32);
				NAME_(store_res_mem16_32)(cpssp, op32);
			}
			break;

/* MUL */			/* mul rm16 (DX:AX <- AX * rm16) */
		case 0x4:	/* mul rm32 (EDX:EAX <- EAX * rm32) */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(uload_op1_reg16_32)(cpssp, E_AX, op32);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op2_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op2_mem16_32)(cpssp, op32, 0);
			}
			NAME_(mul16_32)(cpssp, op32);
			NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
			cpssp->NAME.res = cpssp->NAME.op1;
			NAME_(store_res_reg16_32)(cpssp, E_DX, op32);
			break;

/* IMUL */			/* imul rm16 (DX:AX <- AX * rm16) */
		case 0x5:	/* imul rm32 (EDX:EAX <- EAX * rm32) */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_reg16_32)(cpssp, E_AX, op32);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op2_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op2_mem16_32)(cpssp, op32, 0);
			}
			NAME_(imul16_32)(cpssp, op32);
			NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
			cpssp->NAME.res = cpssp->NAME.op1;
			NAME_(store_res_reg16_32)(cpssp, E_DX, op32);
			break;

/* DIV */			/* div rm16 (AX <- DX:AX / rm16, DX <- DX:AX % rm16) */
		case 0x6:	/* div rm32 (EAX <- EDX:EAX / rm32, EDX <- EDX:EAX % rm32) */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(uload_op1_reg16_32)(cpssp, E_DX, op32);
			cpssp->NAME.res = cpssp->NAME.op1;
			NAME_(uload_op1_reg16_32)(cpssp, E_AX, op32);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op2_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op2_mem16_32)(cpssp, op32, 0);
			}
			NAME_(div16_32)(cpssp, op32);
			NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
			cpssp->NAME.res = cpssp->NAME.op1;
			NAME_(store_res_reg16_32)(cpssp, E_DX, op32);
			break;

/* IDIV */			/* idiv rm16 (AX <- DX:AX / rm16, DX <- DX:AX % rm16) */
		case 0x7:	/* idiv rm32 (EAX <- EDX:EAX / rm32, EDX <- EDX:EAX % rm32) */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_reg16_32)(cpssp, E_DX, op32);
			cpssp->NAME.res = cpssp->NAME.op1;
			NAME_(sload_op1_reg16_32)(cpssp, E_AX, op32);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op2_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op2_mem16_32)(cpssp, op32, 0);
			}
			NAME_(idiv16_32)(cpssp, op32);
			NAME_(store_res_reg16_32)(cpssp, E_AX, op32);
			cpssp->NAME.res = cpssp->NAME.op1;
			NAME_(store_res_reg16_32)(cpssp, E_DX, op32);
			break;

		default:
			_ERROR("Unknown opcode " RED(0x%02x) "@" YELLOW(0x%08x) " with opcode-extension " RED(0x%01x) "\n", opcode,
					cpssp->NAME.eip_save, cpssp->NAME.reg_opex);
			exit(1);
		}
		break;


/* HLT */
	case 0xf4:	/* hlt */
		NAME_(hlt)(cpssp);
		break;


/* CLC */
	case 0xf8:	/* clc */
		NAME_(clc)(cpssp);
		break;


/* STI */
	case 0xf9:	/* stc */
		NAME_(stc)(cpssp);
		break;


/* CLI */
	case 0xfa:	/* cli */
		NAME_(cli)(cpssp);
		break;


/* STI */
	case 0xfb:	/* sti */
		NAME_(sti)(cpssp);
		break;


/* CLD */
	case 0xfc:	/* cld */
		NAME_(cld)(cpssp);
		break;


/* STD */
	case 0xfd:	/* std */
		NAME_(std)(cpssp);
		break;


/* MODRM-OPCEX */
	case 0xfe:
		NAME_(get_modrm)(cpssp);
		switch (cpssp->NAME.reg_opex) {
/* INC */
		case 0x0:	/* inc rm8 */
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op1_reg8)(cpssp, cpssp->NAME.rm);
				NAME_(inc)(cpssp, 8);
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op1_mem8)(cpssp, 1);
				NAME_(inc)(cpssp, 8);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

/* DEC */
		case 0x1:	/* dec rm8 */
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op1_reg8)(cpssp, cpssp->NAME.rm);
				NAME_(dec)(cpssp, 8);
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op1_mem8)(cpssp, 1);
				NAME_(dec)(cpssp, 8);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		default:
			_ERROR("Unknown opcode " RED(0x%02x) "@" YELLOW(0x%08x) " with opcode-extension " RED(0x%01x) "\n", opcode,
					cpssp->NAME.eip_save, cpssp->NAME.reg_opex);
			exit(1);
		}
		break;


/* MODRM-OPCEX */
	case 0xff:
		NAME_(get_modrm)(cpssp);
		switch (cpssp->NAME.reg_opex) {
/* INC */
		case 0x0:	/* inc rm16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
				NAME_(inc)(cpssp, 16 << op32);
				NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op1_mem16_32)(cpssp, op32, 1);
				NAME_(inc)(cpssp, 16 << op32);
				NAME_(store_res_mem16_32)(cpssp, op32);
			}
			break;

/* DEC */
		case 0x1:	/* dec rm16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
				NAME_(dec)(cpssp, 16 << op32);
				NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op1_mem16_32)(cpssp, op32, 1);
				NAME_(dec)(cpssp, 16 << op32);
				NAME_(store_res_mem16_32)(cpssp, op32);
			}
			break;

/* CALL */
		case 0x2:	/* call rm16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op1_mem16_32)(cpssp, op32, 0);
			}
			NAME_(calla)(cpssp, op32);
			break;

		case 0x3:	/* call m16:16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(get_addr)(cpssp);
			NAME_(uload_op_m16_16_32)(cpssp, op32);
			NAME_(fcall)(cpssp, op32);
			break;

/* JMP */
		case 0x4:	/* jmp rm16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op1_mem16_32)(cpssp, op32, 0);
			}
			NAME_(jmpa)(cpssp);
			break;

		case 0x5:	/* ljmp m16:16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(get_addr)(cpssp);
			NAME_(uload_op_m16_16_32)(cpssp, op32);
			NAME_(ljmp)(cpssp);
			break;

/* PUSH */
		case 0x6:	/* push rm16_32 */
			op32 = NAME_(op32_mode)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op1_mem16_32)(cpssp, op32, 0);
			}
			NAME_(push)(cpssp, op32);
			break;

		default:
			_ERROR("Unknown opcode " RED(0x%02x) "@" YELLOW(0x%08x) " with opcode-extension " RED(0x%01x) "\n", opcode,
					cpssp->NAME.eip_save, cpssp->NAME.reg_opex);
			exit(1);
		}
		break;


/* 2-BYTE OPCODE */
	case 0x0f:
		opcode = NAME_(smx8)(cpssp, cpssp->NAME.eip, SEG_CS);
		cpssp->NAME.eip++;
		if (cpssp->NAME.tsc > (_LOG_START_ - 100)) {
			_DEBUG_L("Exec 2-byte opcode " GREEN(0x0f%02x) "@" YELLOW(0x%08x) "\n", opcode, cpssp->NAME.eip-2);
		}
		switch (opcode) {
/* MODRM-OPCEX */
		case 0x00:
			NAME_(get_modrm)(cpssp);
			switch (cpssp->NAME.reg_opex) {
/* LLDT */
			case 0x2:	/* lldt m16 */
				// TODO implement properly
				if (0x3 == cpssp->NAME.mod) {
					NAME_(uload_op2_reg16)(cpssp, cpssp->NAME.rm);
				} else {
					NAME_(get_addr)(cpssp);
					NAME_(uload_op2_mem16)(cpssp, 0);
				}
				if ((cpssp->NAME.op2 & 0xfffc) == 0) {
					NAME_(load_seg_reg)(cpssp, SEG_LDT, 0x0000, 0x00000000, 0x0000, 0);
				} else {
					NAME_(get_seg_entry)(cpssp, SEG_LDT, cpssp->NAME.op2);
				}
				break;

/* LTR */			// TODO
			case 0x3:	/* ltr rm16 */
				_ERROR("TODO ltr\n");
				exit(1);
				break;

			default:
				_ERROR("Unknown 2-byte opcode " RED(0x0f%02x) "@" YELLOW(0x%08x)
						" with opcode-extension " RED(0x%01x) "\n", opcode, cpssp->NAME.eip_save, cpssp->NAME.reg_opex);
				exit(1);
			}
			break;

/* MODRM-OPCEX */
		case 0x01:
			NAME_(get_modrm)(cpssp);
			switch (cpssp->NAME.reg_opex) {
/* LGDT */			// TODO
			case 0x2:	/* lgdt m16&m32 */
				// TODO different op/addre size ->
				NAME_(get_addr)(cpssp);
				NAME_(uload_op1_mem16)(cpssp, 0);
				cpssp->NAME.addr += 2;		// TODO ugly?
				NAME_(uload_op2_mem32)(cpssp, 0);
				NAME_(lgdt)(cpssp);
				break;

/* LIDT */
			case 0x3:	/*lidt m16&m32 */
				NAME_(get_addr)(cpssp);
				NAME_(uload_op1_mem16)(cpssp, 0);
				cpssp->NAME.addr += 2;		// TODO ugly?
				NAME_(uload_op2_mem32)(cpssp, 0);
				// TODO check if 16/32bit mode
				NAME_(lidt)(cpssp);
				break;

/* LMSW */
			case 0x6:	/* lmsw rm16 */
				if (0x3 == cpssp->NAME.mod) {
					NAME_(uload_op1_reg16)(cpssp, cpssp->NAME.rm);
				} else {
					NAME_(get_addr)(cpssp);
					NAME_(uload_op1_mem16)(cpssp, 0);
				}
				NAME_(lmsw)(cpssp);
				break;

			default:
				_ERROR("Unknown 2-byte opcode " RED(0x0f%02x) "@" YELLOW(0x%08x)
						" with opcode-extension " RED(0x%01x) "\n", opcode, cpssp->NAME.eip_save, cpssp->NAME.reg_opex);
				exit(1);
			}
			break;

/* CLTS */
		case 0x06:	/* clts */
			// FIXME implement me!
			_ERROR("[CLTS] Not implemented! Ignored.\n");
			break;

/* UD2 */
		case 0x0b:	/* ud2 */
			_UD();
			break;

/* MOV CR */		// TODO
		case 0x20:	/* mov r32, CR0-CR7 */
			NAME_(get_modrm)(cpssp);
			NAME_(mov_from_cr)(cpssp);
			NAME_(store_res_reg32)(cpssp, cpssp->NAME.rm);
			break;

		case 0x22:	/* mov CR0-CR7, r32 */
			NAME_(get_modrm)(cpssp);
			NAME_(uload_op2_reg32)(cpssp, cpssp->NAME.rm);
			NAME_(mov_to_cr)(cpssp);
			break;

/* Jcc */
		case 0x80:	/* jo rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_of)(cpssp);
			NAME_(jcc)(cpssp, 0);
			break;

		case 0x8a:	/* jp/jpe rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_pf)(cpssp);
			NAME_(jcc)(cpssp, 0);
			break;

		case 0x81:	/* jno rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_of)(cpssp);
			NAME_(jcc)(cpssp, 1);
			break;

		case 0x8b:	/* jnp/jpo rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_pf)(cpssp);
			NAME_(jcc)(cpssp, 1);
			break;

		case 0x88:	/* js rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_sf)(cpssp);
			NAME_(jcc)(cpssp, 0);
			break;

		case 0x89:	/* jns rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_sf)(cpssp);
			NAME_(jcc)(cpssp, 1);
			break;

		case 0x87:	/* ja/jnbe rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_na_be)(cpssp);
			NAME_(jcc)(cpssp, 1);
			break;

		case 0x83:	/* jae/jnb/jnc rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_cf)(cpssp);
			NAME_(jcc)(cpssp, 1);
			break;

		case 0x82:	/* jb/jc/jnae rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_cf)(cpssp);
			NAME_(jcc)(cpssp, 0);
			break;

		case 0x86:	/* jbe/jna rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_na_be)(cpssp);
			NAME_(jcc)(cpssp, 0);
			break;

		case 0x84:	/* je/jz rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_zf)(cpssp);
			NAME_(jcc)(cpssp, 0);
			break;

		case 0x8f:	/* jg/jnle rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_ng_le)(cpssp);
			NAME_(jcc)(cpssp, 1);
			break;

		case 0x8d:	/* jge/jnl rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_l_nge)(cpssp);
			NAME_(jcc)(cpssp, 1);
			break;

		case 0x8c:	/* jl/jnge rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_l_nge)(cpssp);
			NAME_(jcc)(cpssp, 0);
			break;

		case 0x8e:	/* jle/jng rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_ng_le)(cpssp);
			NAME_(jcc)(cpssp, 0);
			break;

		case 0x85:	/* jne/jnz rel16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(sload_op1_imm16_32)(cpssp, op32);
			NAME_(test_zf)(cpssp);
			NAME_(jcc)(cpssp, 1);
			break;

/* SETcc */
		case 0x90:	/* seto rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_of)(cpssp);
			NAME_(setcc)(cpssp, 0);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x9a:	/* setp/setpe rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_pf)(cpssp);
			NAME_(setcc)(cpssp, 0);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x91:	/* setno rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_of)(cpssp);
			NAME_(setcc)(cpssp, 1);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x9b:	/* setnp/setpo rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_pf)(cpssp);
			NAME_(setcc)(cpssp, 1);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x98:	/* sets rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_sf)(cpssp);
			NAME_(setcc)(cpssp, 0);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x99:	/* setns rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_sf)(cpssp);
			NAME_(setcc)(cpssp, 1);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x97:	/* seta/setnbe rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_na_be)(cpssp);
			NAME_(setcc)(cpssp, 1);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x93:	/* setae/setnb/setnc rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_cf)(cpssp);
			NAME_(setcc)(cpssp, 1);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x92:	/* setb/setc/setnae rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_cf)(cpssp);
			NAME_(setcc)(cpssp, 0);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x96:	/* setbe/setna rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_na_be)(cpssp);
			NAME_(setcc)(cpssp, 0);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x94:	/* sete/setz rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_zf)(cpssp);
			NAME_(setcc)(cpssp, 0);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x9f:	/* setg/setnle rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_ng_le)(cpssp);
			NAME_(setcc)(cpssp, 1);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x9d:	/* setge/setnl rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_l_nge)(cpssp);
			NAME_(setcc)(cpssp, 1);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x9c:	/* setl/setnge rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_l_nge)(cpssp);
			NAME_(setcc)(cpssp, 0);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x9e:	/* setle/setng rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_ng_le)(cpssp);
			NAME_(setcc)(cpssp, 0);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

		case 0x95:	/* setne/setnz rel16/32 */
			NAME_(get_modrm)(cpssp);
			NAME_(test_zf)(cpssp);
			NAME_(setcc)(cpssp, 1);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(store_res_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(store_res_mem8)(cpssp);
			}
			break;

/* PUSH */
		case 0xa0:	/* push FS */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(load_op1_sreg)(cpssp, SEG_FS);
			NAME_(push)(cpssp, op32);
			break;

		case 0xa8:	/* push GS */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(load_op1_sreg)(cpssp, SEG_GS);
			NAME_(push)(cpssp, op32);
			break;

/* POP */
		case 0xa1:	/* pop FS */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(pop)(cpssp, op32);
			NAME_(store_res_sreg)(cpssp, SEG_FS);
			break;

		case 0xa9:	/* pop GS */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(pop)(cpssp, op32);
			NAME_(store_res_sreg)(cpssp, SEG_GS);
			break;

/* SHRD */
		case 0xac:	/* shrd rm16/32, r16/32, imm8 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(get_modrm)(cpssp);
			NAME_(uload_op2_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
				cpssp->NAME.res = cpssp->NAME.op1;
				NAME_(uload_op1_imm8)(cpssp);
				NAME_(shrd)(cpssp, op32);
				NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op1_mem16_32)(cpssp, op32, 1);
				cpssp->NAME.res = cpssp->NAME.op1;
				NAME_(uload_op1_imm8)(cpssp);
				NAME_(shrd)(cpssp, op32);
				NAME_(store_res_mem16_32)(cpssp, op32);
			}
			break;

		case 0xad:	/* shrd rm16/32, r16/32, cl */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(get_modrm)(cpssp);
			NAME_(uload_op2_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op1_reg16_32)(cpssp, cpssp->NAME.rm, op32);
				cpssp->NAME.res = cpssp->NAME.op1;
				NAME_(uload_op1_reg8)(cpssp, CL);
				cpssp->NAME.op1 &= 0x1f;
				NAME_(shrd)(cpssp, op32);
				NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op1_mem16_32)(cpssp, op32, 1);
				cpssp->NAME.res = cpssp->NAME.op1;
				NAME_(uload_op1_reg8)(cpssp, CL);
				cpssp->NAME.op1 &= 0x1f;
				NAME_(shrd)(cpssp, op32);
				NAME_(store_res_mem16_32)(cpssp, op32);
			}
			break;

/* IMUL */
		case 0xaf:	/* imul r16/32, rm16/32 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(get_modrm)(cpssp);
			NAME_(sload_op1_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op2_reg16_32)(cpssp, cpssp->NAME.rm, op32);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op2_mem16_32)(cpssp, op32, 0);
			}
			NAME_(imul16_32)(cpssp, op32);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
			break;

/* LSS/LGS/LFS */
		case 0xb2:	/* lss r16/32, m16:16/32 */
			op32 = NAME_(addr32_mode)(cpssp);
			NAME_(get_modrm)(cpssp);
			NAME_(get_addr)(cpssp);
			NAME_(uload_op_m16_16_32)(cpssp, op32);
			NAME_(l_s)(cpssp, SEG_SS);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
			break;

		case 0xb4:	/* lfs r16/32, m16:16/32 */
			op32 = NAME_(addr32_mode)(cpssp);
			NAME_(get_modrm)(cpssp);
			NAME_(get_addr)(cpssp);
			NAME_(uload_op_m16_16_32)(cpssp, op32);
			NAME_(l_s)(cpssp, SEG_FS);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
			break;

		case 0xb5:	/* lgs r16/32, m16:16/32 */
			op32 = NAME_(addr32_mode)(cpssp);
			NAME_(get_modrm)(cpssp);
			NAME_(get_addr)(cpssp);
			NAME_(uload_op_m16_16_32)(cpssp, op32);
			NAME_(l_s)(cpssp, SEG_GS);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
			break;

/* MOV */
		case 0xb6:	/* movzx r16/32, rm8 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(get_modrm)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op2_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op2_mem8)(cpssp, 0);
			}
			NAME_(mov)(cpssp);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
			break;

		case 0xb7:	/* movzx r32, rm16 */
			NAME_(get_modrm)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(uload_op2_reg16)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(uload_op2_mem16)(cpssp, 0);
			}
			NAME_(mov)(cpssp);
			NAME_(store_res_reg32)(cpssp, cpssp->NAME.reg_opex);
			break;

		case 0xbe:	/* movsx r16/32, rm8 */
			op32 = NAME_(op32_mode)(cpssp);
			NAME_(get_modrm)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op2_reg8)(cpssp, cpssp->NAME.rm);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op2_mem8)(cpssp, 0);
			}
			NAME_(mov)(cpssp);
			NAME_(store_res_reg16_32)(cpssp, cpssp->NAME.reg_opex, op32);
			break;

		case 0xbf:	/* movsx r32, rm16 */
			NAME_(get_modrm)(cpssp);
			if (0x3 == cpssp->NAME.mod) {
				NAME_(sload_op2_reg16_32)(cpssp, cpssp->NAME.rm, 0);
			} else {
				NAME_(get_addr)(cpssp);
				NAME_(sload_op2_mem16_32)(cpssp, 0, 0);
			}
			NAME_(mov)(cpssp);
			NAME_(store_res_reg32)(cpssp, cpssp->NAME.reg_opex);
			break;

		default:
			_ERROR("Unknown 2-byte opcode " RED(0x0f%02x) "@" YELLOW(0x%08x) "\n", opcode, cpssp->NAME.eip_save);
			exit(1);
			break;
		}
		break;


	default:
		_ERROR("[" BLUE(0x%016x) "] Unknown opcode " RED(0x%02x) "@" YELLOW(0x%08x:0x%08x - 0x%08x) "\n", cpssp->NAME.tsc, opcode,
				cpssp->NAME.seg_regs[SEG_CS].base, cpssp->NAME.eip_save, cpssp->NAME.eip_save + cpssp->NAME.seg_regs[SEG_CS].base);
		exit(1);
		break;
	}

	/* reset prefix */	// TODO only if needed?
	cpssp->NAME.pre_op = 0;
	cpssp->NAME.pre_addr = 0;
	cpssp->NAME.pre_repe_repne = 0;
	cpssp->NAME.seg_override = 0;
}

#endif /* BEHAVIOR */
