/*
 * Parts derived from QEMU sources.
 * Modified for FAUmachine by Volkmar Sieh.
 * 
 *  Copyright (c) 2002-2014 FAUmachine Team.
 *  Copyright (c) 2003-2005 Fabrice Bellard.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 */

#define FAST

/* Set this to '1' to trace execution. */
#define DEBUG_CONTROL_BIOS		0
#define DEBUG_CONTROL_FLOW		0
#define DEBUG_CONTROL_FLOW_FLAGS	0
#define DEBUG_CONTROL_FLOW_REGS		0
#define DEBUG_CONTROL_FLOW_SREGS	0

/* Set to '1' to debug SMM entry. */
#define DEBUG_SMM_ENTRY			0

/*
 * 8-3
 */
static void
NAME_(core_do_reset)(struct cpssp *cpssp, int reset_flag)
{
	int i;

	if (reset_flag) {
		NAME_(apic_reset)(cpssp);
		NAME_(cache1i_reset)(cpssp);
		NAME_(cache1d_reset)(cpssp);
		NAME_(mmui_reset)(cpssp);
		NAME_(mmud_reset)(cpssp);
		NAME_(a20gate_reset)(cpssp);
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT
		NAME_(mtrr_reset)(cpssp);
#endif
		NAME_(cache2_reset)(cpssp);
	}

	cpssp->hflags = 0;
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if (! cpssp->NAME_(apic).bsp) {
		cpssp->hflags |= HF_WAITING_FOR_STARTUP_MASK;
	}
#endif

	/*
	 * must do it in this sequence:
	 * - first initialize real-mode (cr0)
	 * - then initialize segment registers
	 */
	cpssp->eflags = 0x00000002; /* reserved bit */
	cpssp->cc_src = cpssp->eflags
		& (CC_O | CC_S | CC_Z | CC_A | CC_P | CC_C);
	cpssp->df = 1 - (2 * ((cpssp->eflags >> 10) & 1));
	cpssp->cc_op = CC_OP_EFLAGS;
	cpssp->eflags
		&= ~(CPU_DF_MASK | CC_O | CC_S | CC_Z | CC_A | CC_P | CC_C);

	NAME_(update_cr0)(cpssp, 0x60000010);	/* write through */
						/* caching disabled */
						/* reserved bit */
	/* NAME_(update_cr2)(cpssp, 0x00000000); */
	NAME_(update_cr3)(cpssp, 0x00000000);
	NAME_(update_cr4)(cpssp, 0x00000000);

	cpssp->exception_index = -1;
	if (reset_flag) {
		cpssp->interrupt_request = 0;
	}

	/* code segment register */
#if CONFIG_CPU <= 80286
	cpu_x86_load_seg_cache(cpssp, R_CS, 0xf000, 0x000f0000, 0xffff, 0);
#else
	cpu_x86_load_seg_cache(cpssp, R_CS, 0xf000, 0xffff0000, 0xffff, 0);
#endif

	/* instruction pointer */
	cpssp->eip = 0x0000fff0;

	/* other segment registers */
	cpu_x86_load_seg_cache(cpssp, R_DS, 0, 0, 0xffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_ES, 0, 0, 0xffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_SS, 0, 0, 0xffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_FS, 0, 0, 0xffff, 0);
	cpu_x86_load_seg_cache(cpssp, R_GS, 0, 0, 0xffff, 0);

	/* edx - depends on processor version - FIXME VOSSI */
	cpssp->regs[R_EDX] = 0x00000600; /* indicate P6 processor */

	/* eax - should contain BIST code - FIXME VOSSI */
	cpssp->regs[R_EAX] = 0x00000000;

	/* other registers */
	cpssp->regs[R_EBX] = 0x00000000;
	cpssp->regs[R_ECX] = 0x00000000;
	cpssp->regs[R_ESI] = 0x00000000;
	cpssp->regs[R_EDI] = 0x00000000;
	cpssp->regs[R_EBP] = 0x00000000;
	cpssp->regs[R_ESP] = 0x00000000;

	if (reset_flag) {
		/* FPU init */
		cpssp->fpstt = 0;
		cpssp->fpus = 0;
		cpssp->fpuc = 0x37f;
		for (i = 0; i < 8; i++) {
			cpssp->fptags[i] = 1;
		}
		cpssp->mxcsr = 0;
		memset(cpssp->xmm_regs, 0, sizeof(cpssp->xmm_regs));
		memset(&cpssp->xmm_t0, 0, sizeof(cpssp->xmm_t0));
		memset(&cpssp->mmx_t0, 0, sizeof(cpssp->mmx_t0));
	}

	/* gdtr, idtr */
	cpssp->gdt.base = 0x00000000;
	cpssp->gdt.limit = 0xffff;
	cpssp->idt.base = 0x00000000;
	cpssp->idt.limit = 0xffff;

	/* ldtr, task register */
	cpssp->ldt.base = 0x00000000;
	cpssp->ldt.limit = 0xffff;
	cpssp->ldt.flags = DESC_P_MASK;

	cpssp->tr.base = 0x00000000;
	cpssp->tr.limit = 0xffff;
	cpssp->tr.flags = DESC_P_MASK;

	/* dr0 - dr3 */
	cpssp->dr[0] = 0;
	cpssp->dr[1] = 0;
	cpssp->dr[2] = 0;
	cpssp->dr[3] = 0;

	/* dr6 */
	cpssp->dr[6] = 0;

	/* dr7 */
	cpssp->dr[7] = 0;

	/* time-stamp counter */
	/* FIXME */

	/* perf. counters and event select */
	/* FIXME VOSSI */

	/* all other MSRs */
#if CONFIG_CPU >= 80486 && CONFIG_CPU_SEP_SUPPORT
	cpssp->sysenter_cs = 0;
	cpssp->sysenter_esp = 0;
	cpssp->sysenter_eip = 0;
#endif
	cpssp->efer = 0;
	cpssp->star = 0;
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT
	cpssp->lstar = 0;
	cpssp->cstar = 0;
	cpssp->fmask = 0;
	cpssp->kernelgsbase = 0;
#endif
	memset(cpssp->msr, 0, sizeof(cpssp->msr));

#if 80486 <= CONFIG_CPU && CONFIG_CPU_PAT_SUPPORT
	/* PAT Table */
	cpssp->pat[0] = 6; /* Write back */
	cpssp->pat[1] = 4; /* Write through */
	cpssp->pat[2] = 7; /* Uncached */
	cpssp->pat[3] = 0; /* Uncacheable */
	cpssp->pat[4] = 6; /* Write back */
	cpssp->pat[5] = 4; /* Write through */
	cpssp->pat[6] = 7; /* Uncached */
	cpssp->pat[7] = 0; /* Uncacheable */
#endif

	/* machine-check architecture */
	/* FIXME VOSSI */

	/* System Management */
	if (reset_flag) {
		cpssp->smbase = 0x30000;
	}
	cpssp->smm = 0;

	/* APIC */
	/* FIXME VOSSI */
}

static void
NAME_(core_reset)(struct cpssp *cpssp)
{
	NAME_(core_do_reset)(cpssp, 1);
}

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
static void
NAME_(core_init)(struct cpssp *cpssp)
{
	NAME_(core_do_reset)(cpssp, 0);
}
#endif

/*
 * Signal an interruption. It is executed in the main CPU loop.
 * is_int is TRUE if coming from the int instruction. next_eip is the
 * EIP value AFTER the interrupt instruction. It is only relevant if
 * is_int is TRUE.
 */
void __attribute__((__noreturn__))
NAME_(raise_interrupt)(int intno, int is_int, int error_code, int next_eip_addend)
{
	env->exception_index = intno;
	env->error_code = error_code;
	env->exception_is_int = is_int;
	env->exception_next_eip = env->eip + next_eip_addend;
	longjmp(env->jmp_env, 1);
}

/* same as raise_exception_err, but do not restore global registers */
void __attribute__((__noreturn__))
NAME_(raise_exception_err_norestore)(int exception_index, int error_code)
{
	env->exception_index = exception_index;
	env->error_code = error_code;
	env->exception_is_int = 0;
	env->exception_next_eip = 0;
	longjmp(env->jmp_env, 1);
}

/* shortcuts to generate exceptions */

void __attribute__((__noreturn__))
NAME_(raise_exception_err)(int exception_index, int error_code)
{
	NAME_(raise_interrupt)(exception_index, 0, error_code, 0);
}

void __attribute__((__noreturn__))
NAME_(raise_exception)(int exception_index)
{
	NAME_(raise_interrupt)(exception_index, 0, 0, 0);
}

#if DEBUG_CONTROL_BIOS || DEBUG_CONTROL_FLOW
static void
NAME_(dump)(void)
{
#if DEBUG_CONTROL_BIOS
	fprintf(stderr, " AX=%04lx BX=%04lx CX=%04lx DX=%04lx BP=%04lx SP=%04lx DI=%04lx SI=%04lx",
			(unsigned long) EAX & 0xffff,
			(unsigned long) EBX & 0xffff,
			(unsigned long) ECX & 0xffff,
			(unsigned long) EDX & 0xffff,
			(unsigned long) EBP & 0xffff,
			(unsigned long) ESP & 0xffff,
			(unsigned long) EDI & 0xffff,
			(unsigned long) ESI & 0xffff);
	fprintf(stderr, " CS=%04x SS=%04x DS=%04x ES=%04x",
			(unsigned int) env->segs[R_CS].selector,
			(unsigned int) env->segs[R_SS].selector,
			(unsigned int) env->segs[R_DS].selector,
			(unsigned int) env->segs[R_ES].selector);
#endif
#if DEBUG_CONTROL_FLOW_REGS
	fprintf(stderr, " %08lx %08lx %08lx %08lx %08lx %08lx %08lx %08lx",
			(unsigned long) EAX, (unsigned long) EBX,
			(unsigned long) ECX, (unsigned long) EDX,
			(unsigned long) EBP, (unsigned long) ESP,
			(unsigned long) EDI, (unsigned long) ESI);
#endif
#if DEBUG_CONTROL_FLOW_SREGS
	fprintf(stderr, " %04x %04x %04x %04x %04x %04x",
			(unsigned int) env->segs[R_CS].selector,
			(unsigned int) env->segs[R_SS].selector,
			(unsigned int) env->segs[R_DS].selector,
			(unsigned int) env->segs[R_ES].selector,
			(unsigned int) env->segs[R_FS].selector,
			(unsigned int) env->segs[R_GS].selector);
#endif
#if DEBUG_CONTROL_FLOW_FLAGS
	fprintf(stderr, " %08lx", compute_eflags());
#endif
	fprintf(stderr, "\n");
}
#endif /* DEBUG_CONTROL_BIOS || DEBUG_CONTROL_FLOW */

void __attribute__((__noreturn__))
NAME_(step)(void *_cpssp)
{
	env = (struct cpssp *) _cpssp;

again:	;
	/* prepare setjmp context for exception handling */
	setjmp(env->jmp_env);
	env->link_tb = 0;

	while (env->process.inst_cnt < env->process.inst_limit) {
		env->current_tb = (TranslationBlock *) 0;

		if (unlikely(env->state_ext != STATE_POWER)) {
			if (! (env->state_ext & STATE_POWER)) {
#if DEBUG_CONTROL_FLOW
				if (2 <= DEBUG_CONTROL_FLOW
				 || loglevel) {
					fprintf(stderr, "%d: Power-off\n", env->id);
				}
	#endif
				if (env->process.inst_cnt < env->process.inst_limit) {
					env->process.inst_cnt = env->process.inst_limit;
				}
			} else if (env->state_ext & (STATE_RESET | STATE_N_RESET_ACK)) {
#if DEBUG_CONTROL_FLOW
				if (2 <= DEBUG_CONTROL_FLOW
				 || loglevel) {
					fprintf(stderr, "%d: Reset\n", env->id);
				}
#endif
				env->state_ext &= ~STATE_N_RESET_ACK;
				NAME_(core_reset)(env);
				if (env->process.inst_cnt < env->process.inst_limit) {
					env->process.inst_cnt = env->process.inst_limit;
				}
#if 80386 < CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
			} else if (env->state_ext & (STATE_INIT | STATE_N_INIT_ACK)) {
#if DEBUG_CONTROL_FLOW
				if (2 <= DEBUG_CONTROL_FLOW
				 || loglevel) {
					fprintf(stderr, "%d: Init\n", env->id);
				}
#endif
				env->state_ext &= ~STATE_N_INIT_ACK;
				NAME_(core_init)(env);
				if (env->process.inst_cnt < env->process.inst_limit) {
					env->process.inst_cnt = env->process.inst_limit;
				}
#endif /* 80386 < CONFIG_CPU */
			} else {
				assert(0); /* Mustn't happen. */
			}
			break;

#if 80386 < CONFIG_CPU
		} else if (unlikely(env->hflags & HF_WAITING_FOR_STARTUP_MASK)) {
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
				fprintf(stderr, "%d: Waiting for Startup-IPI at %08llx (%08llx:%08llx)",
					env->id,
					(unsigned long long) env->eip
					+ (unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->eip);
				NAME_(dump)();
			}
#endif
			if (env->state_startup) {
				/* Leave WAITING_FOR_STARTUP state. */
				env->state_startup = 0;
				env->hflags &= ~HF_WAITING_FOR_STARTUP_MASK;

				/* Load code segment register. */
				cpu_x86_load_seg_cache(env, R_CS,
					(uint16_t) env->state_startup_vec << 8, /* Selector */
					(uint32_t) env->state_startup_vec << 12,/* Base */
					0xffff,                 /* Limit */
					0);                     /* Flags */

				/* Load instruction pointer. */
				env->eip = 0x00000000;
			}

			if (env->process.inst_cnt < env->process.inst_limit) {
				env->process.inst_cnt = env->process.inst_limit;
			}
			break;
#endif /* 80386 < CONFIG_CPU */

		} else if (unlikely(0 <= env->exception_index)) {
			/*
			 * If an exception is pending, we execute it here.
			 * Simulate a real cpu exception. On i386, it can
			 * trigger new exceptions, but we do not handle
			 * double or triple faults yet.
			 */
#if DEBUG_CONTROL_BIOS
			if (2 <= DEBUG_CONTROL_BIOS
			 || loglevel) {
				fprintf(stderr, "int 0x%02x at %08llx (%08llx:%08llx)",
					env->exception_index,
					(unsigned long long) EIP
					+ (unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->segs[R_CS].base,
					(unsigned long long) EIP);
				NAME_(dump)();
			}
#endif
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
				fprintf(stderr, "%d: Exception 0x%02x at %08llx (%08llx:%08llx)",
					env->id,
					env->exception_index,
					(unsigned long long) EIP
					+ (unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->segs[R_CS].base,
					(unsigned long long) EIP);
				NAME_(dump)();
			}
#endif
			env->process.inst_cnt += 100;

			NAME_(do_exception)();
			env->link_tb = 0;

#if 80386 <= CONFIG_CPU
		} else if (unlikely((env->interrupt_request & CPU_INTERRUPT_SMI)
			&& ! env->smm
			&& ! (env->hflags & HF_INHIBIT_IRQ_MASK))) {
			env->hflags &= ~HF_HALTED_MASK;

			/* signal "acknowledge" */
			NAME_(apic_smi_ack)(env);

#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
				fprintf(stderr, "%d: SMM-Interrupt at %08llx (%08llx:%08llx)",
					env->id,
					(unsigned long long) env->eip
					+ (unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->eip);
				NAME_(dump)();
			}
#endif

			env->process.inst_cnt += 100;

			/* execute SMI handler: start system management mode */
			NAME_(do_smm)();
			/* FIXME knilch: improve performance */
			env->link_tb = 0;
#endif /* 80386 <= CONFIG_CPU */

		} else if (unlikely((env->interrupt_request & CPU_INTERRUPT_IRQ)
			&& (env->eflags & CPU_IF_MASK)
			&& ! (env->hflags & HF_INHIBIT_IRQ_MASK))) {
			/*
			 * If hardware interrupt pending, we execute it.
			 */
			env->hflags &= ~HF_HALTED_MASK;

			env->exception_index = NAME_(apic_irq_ack)(env);
			env->exception_is_int = 0;
			env->error_code = 0;
			env->exception_next_eip = 0;

#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
				fprintf(stderr, "%d: Interrupt 0x%02x at %08llx (%08llx:%08llx)",
					env->id,
					env->exception_index,
					(unsigned long long) env->eip
					+ (unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->eip);
				NAME_(dump)();
			}
#endif

			env->process.inst_cnt += 100;

			NAME_(do_interrupt)();
			env->link_tb = 0;

		} else if (unlikely(env->hflags & HF_HALTED_MASK)) {
			/*
			 * Do nothing...
			 */
#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
				fprintf(stderr, "%d: Halting at %08llx (%08llx:%08llx)",
					env->id,
					(unsigned long long) env->eip
					+ (unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->eip);
				NAME_(dump)();
			}
#endif

			if (env->process.inst_cnt < env->process.inst_limit) {
				env->process.inst_cnt = env->process.inst_limit;
			}
			break;

		} else {
			/*
			 * Execute the generated code.
			 */
			TranslationBlock *tb;
			target_ulong cs_base;
			target_ulong pc;
			unsigned int flags;

#if DEBUG_CONTROL_FLOW
			if (2 <= DEBUG_CONTROL_FLOW
			 || loglevel) {
				fprintf(stderr, "%d: Executing at %08llx (%08llx:%08llx)",
					env->id,
					(unsigned long long) env->eip
					+ (unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->segs[R_CS].base,
					(unsigned long long) env->eip);
				NAME_(dump)();
			}
#endif

			/*
			 * We record a subset of the CPU state. It will
			 * always be the same before a given translated block
			 * is executed.
			 */
			flags = env->hflags;
			flags |= (env->eflags & (CPU_IOPL_MASK | CPU_TF_MASK | CPU_VM_MASK));
			cs_base = env->segs[R_CS].base;
			pc = cs_base + env->eip;

			tb = NAME_(tb_get)(pc, cs_base, flags, env->link_tb);

			env->link_tb = 0;
			env->current_tb = tb;
#if CONFIG_CPU >= 80486 && CONFIG_CPU_LM_SUPPORT && defined(__i386__)
			asm volatile ("subl $0x04,%%esp\n"
					"push %%ebx\n"
					"push %%esi\n"
					"push %%edi\n"
					"call *%0\n"
					"pop %%edi\n"
					"pop %%esi\n"
					"pop %%ebx\n"
					"addl $0x04,%%esp\n"
					: : "r" ((void (*)(void)) tb->tc_ptr) : "ebx", "esi", "edi");
#else
			((void (*)(void)) tb->tc_ptr)();
#endif
		}
	}

	sched_to_scheduler();

	goto again;
}
