/*
 * $Id: arch_gen_cpu_x86_code_fast.c,v 1.9 2014/02/11 10:52:16 sieh Exp $
 *
 * Derived from QEMU sources.
 * Modified for FAUmachine by Volkmar Sieh.
 *
 *  Copyright (c) 2007-2009 FAUmachine Team.
 *  Copyright (c) 2003 Fabrice Bellard.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 */


/*
 * Add a new TLB entry. At most one entry for a given virtual address
 * is permitted.
 */
static void
tlb_set_page(
	Vaddr vaddr,
	Paddr paddr,
	int rwx,
	int is_user,
	Haddr haddr
)
{
	TranslationBlock *first_tb;
	unsigned int index_;
	Vaddr address;
	Haddr addend;
	Paddr phys_addend;

	assert(! (vaddr & (CACHE_LINE_SIZE - 1)));
	assert(! (paddr & (CACHE_LINE_SIZE - 1)));

	/* NOTE: we also allocate the page at this stage */
	first_tb = NAME_(tb_find_alloc)(paddr);

	index_ = (vaddr / CACHE_LINE_SIZE) & (CPU_TLB_SIZE - 1);

	phys_addend = paddr - vaddr;

	switch (rwx) {
	case 0:
		/*
		 * Memory Read
		 */
		if (! haddr) {
			/* I/O Case */
			address = vaddr | IO_MEM_IO;
			addend = NULL;
		} else {
			/* Standard Memory */
			address = vaddr | IO_MEM_RAM;
			addend = haddr - vaddr;
		}
		env->tlb_read[is_user][index_].address = address;
		env->tlb_read[is_user][index_].host_addend = addend;
		env->tlb_read[is_user][index_].phys_addend = phys_addend;
		break;

	case 1:
		/*
		 * Memory Write
		 */
		if (! haddr) {
			if (first_tb) {
				address = vaddr | IO_MEM_IO_CODE;
			} else {
				address = vaddr | IO_MEM_IO;
			}
			addend = NULL;
		} else {
			if (first_tb) {
				address = vaddr | IO_MEM_CODE;
			} else {
				address = vaddr | IO_MEM_RAM;
			}
			addend = haddr - vaddr;
		}
		env->tlb_write[is_user][index_].address = address;
		env->tlb_write[is_user][index_].host_addend = addend;
		env->tlb_write[is_user][index_].phys_addend = phys_addend;
		break;

	case 2:
		/*
		 * Memory Execute
		 */
		if (! haddr) {
			/* I/O Case */
			address = vaddr | IO_MEM_IO;
			addend = NULL;
		} else {
			/* Standard Memory */
			address = vaddr | IO_MEM_RAM;
			addend = haddr - vaddr;
		}
		env->tlb_code[is_user][index_].address = address;
		env->tlb_code[is_user][index_].host_addend = addend;
		env->tlb_code[is_user][index_].phys_addend = phys_addend;
		break;

	default:
		assert(0); /* Mustn't happen. */
	}
}

/*
 * Try to fill the TLB and return an exception if error. If retaddr is
 * NULL, it means that the function was called in C code (i.e. not
 * from generated code or from helper.c)
 */
/* XXX: fix it to restore all registers */
void
NAME_(tlb_fill)(Vaddr addr, int rwx, int is_user, void *retaddr)
{
	Paddr paddr;
	Haddr haddr;
	TranslationBlock *tb;
	int ret;
	unsigned long pc;

	switch (rwx) {
	case 0:
		/* Read */
		ret = NAME_(cache1d_map_r)(env, addr, is_user, &paddr, &haddr);
		break;
	case 1:
		/* Write */
		ret = NAME_(cache1d_map_w)(env, addr, is_user, &paddr, &haddr);
		break;
	case 2:
		/* Execute */
		ret = NAME_(cache1i_map_x)(env, addr, is_user, &paddr, &haddr);
		break;
	default:
		assert(0);
	}
	if (ret) {
		if (retaddr) {
			/* now we have a real cpu fault */
			pc = (unsigned long)retaddr;
			tb = NAME_(tb_find_pc)(pc);
			if (tb) {
				/*
				 * The PC is inside the translated code. It
				 * means that we have a virtual CPU fault
				 */
				NAME_(restore_state)(tb, pc);
			}
		}
		if (retaddr)
			NAME_(raise_exception_err)(CPU_FAULT_PF,
					env->error_code);
		else
			NAME_(raise_exception_err_norestore)(CPU_FAULT_PF,
					env->error_code);
	} else {
		tlb_set_page(addr & ~(CACHE_LINE_SIZE - 1),
				paddr & ~(CACHE_LINE_SIZE - 1), rwx, is_user,
				haddr ? haddr - (paddr & (CACHE_LINE_SIZE - 1)) : NULL);
	}
}

#define MMUSUFFIX _mmu
#define GETPC() (__builtin_return_address(0))

#define SHIFT 0
#include "arch_gen_cpu_x86_code_template.c"

#define SHIFT 1
#include "arch_gen_cpu_x86_code_template.c"

#define SHIFT 2
#include "arch_gen_cpu_x86_code_template.c"

#define SHIFT 3
#include "arch_gen_cpu_x86_code_template.c"

#undef GETPC
#undef MMUSUFFIX
