/*
 * Derived from QEMU sources.
 *
 *  Copyright (c) 2005-2014 FAUmachine Team.
 *  Copyright (c) 2003 Fabrice Bellard.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301
 * USA
 */

static inline void
cpu_tlb_protect1(CPUTLBEntry *tlb_entry, Paddr pa)
{
	Paddr tlbpa;

	if (tlb_entry->address & TLB_INVALID_MASK) {
		return;
	}
	tlbpa = (tlb_entry->address & CACHE_LINE_MASK) + tlb_entry->phys_addend;
	if (tlbpa < pa
	 || pa + TARGET_PAGE_SIZE <= tlbpa) {
		/* Different address in cache. */
		return;
	}

	switch (tlb_entry->address & ~CACHE_LINE_MASK) {
	case IO_MEM_CODE:
	case IO_MEM_IO_CODE:
		/* Already protected. */
		break;
	case 0: /* RAM */
		tlb_entry->address
			= (tlb_entry->address & CACHE_LINE_MASK)
			| IO_MEM_CODE;
		break;
	case IO_MEM_IO:
		tlb_entry->address
			= (tlb_entry->address & CACHE_LINE_MASK)
			| IO_MEM_IO_CODE;
		break;
	default:
		assert(0);
	}
}

/*
 * Update the TLBs so that writes to code in the physical page 'pa'
 * can be detected.
 */
void
NAME_(tlb_protect)(struct cpssp *cpssp, Paddr pa)
{
	int i;

	pa &= TARGET_PAGE_MASK;
	for (i = 0; i < CPU_TLB_SIZE; i++) {
		cpu_tlb_protect1(&cpssp->tlb_write[0][i], pa);
		cpu_tlb_protect1(&cpssp->tlb_write[1][i], pa);
	}
}

static inline void
cpu_tlb_unprotect1(CPUTLBEntry *tlb_entry, Paddr pa)
{
	Paddr tlbpa;

	if (tlb_entry->address & TLB_INVALID_MASK) {
		return;
	}
	tlbpa = (tlb_entry->address & CACHE_LINE_MASK) + tlb_entry->phys_addend;
	if (tlbpa < pa
	 || pa + TARGET_PAGE_SIZE <= tlbpa) {
		/* Different address in cache. */
		return;
	}

	switch (tlb_entry->address & ~CACHE_LINE_MASK) {
	case IO_MEM_RAM:
	case IO_MEM_IO:
		/* Already unprotected. */
		break;
	case IO_MEM_CODE:
		tlb_entry->address = (tlb_entry->address & CACHE_LINE_MASK)
			| IO_MEM_RAM;
		break;
	case IO_MEM_IO_CODE:
		tlb_entry->address = (tlb_entry->address & CACHE_LINE_MASK)
			| IO_MEM_IO;
		break;
	default:
		assert(0);
	}
}

/*
 * Update the TLB so that writes in physical page 'pa' are no longer
 * tested self modifying code.
 */
void
NAME_(tlb_unprotect)(struct cpssp *cpssp, Paddr pa)
{
	int i;

	pa &= TARGET_PAGE_MASK;
	for (i = 0; i < CPU_TLB_SIZE; i++) {
		cpu_tlb_unprotect1(&cpssp->tlb_write[0][i], pa);
		cpu_tlb_unprotect1(&cpssp->tlb_write[1][i], pa);
	}
}

Paddr
NAME_(tlb_virt_to_phys)(Vaddr va)
{
	unsigned int hash;
	unsigned int is_user;

	hash = (va / CACHE_LINE_SIZE) & (CPU_TLB_SIZE - 1);
	is_user = ((env->hflags & HF_CPL_MASK) == 3);
	if (__builtin_expect(env->tlb_code[is_user][hash].address
			!= (va & CACHE_LINE_MASK), 0)) {
		(void) ldub_code(va);
	}

	return va + env->tlb_code[is_user][hash].phys_addend;
}

static inline void
tlb_flush_entry(CPUTLBEntry *tlb_entry, Vaddr addr)
{
	if (addr == (tlb_entry->address & (CACHE_LINE_MASK | TLB_INVALID_MASK)))
		tlb_entry->address = -1;
}

void
NAME_(code_invlpg)(struct cpssp *cpssp, Vaddr addr)
{
	int i, j;
	TranslationBlock *tb;
		
	/*
	 * Must reset current TB so that interrupts cannot modify the
	 * links while we are modifying them.
	 */
	cpssp->current_tb = NULL;
			
	addr &= TARGET_PAGE_MASK;
	for (j = 0; j < TARGET_PAGE_SIZE / CACHE_LINE_SIZE; j++) {
		i = (addr / CACHE_LINE_SIZE) & (CPU_TLB_SIZE - 1);
		tlb_flush_entry(&cpssp->tlb_read[0][i], addr);
		tlb_flush_entry(&cpssp->tlb_write[0][i], addr);
		tlb_flush_entry(&cpssp->tlb_code[0][i], addr);
		tlb_flush_entry(&cpssp->tlb_read[1][i], addr);
		tlb_flush_entry(&cpssp->tlb_write[1][i], addr);
		tlb_flush_entry(&cpssp->tlb_code[1][i], addr);
		addr += CACHE_LINE_SIZE;
	}
		
	for (i = 0; i < TB_JMP_CACHE_SIZE; i++) {
		tb = cpssp->tb_jmp_cache[i];
		if (tb
		 && ((tb->pc & TARGET_PAGE_MASK) == addr
		  || ((tb->pc + tb->size - 1) & TARGET_PAGE_MASK) == addr)) {
			cpssp->tb_jmp_cache[i] = NULL;
		}
	}
}

void
NAME_(code_flush_all)(struct cpssp *cpssp, int flush_global)
{
	int i;

	/*
	 * Must reset current TB so that interrupts cannot modify the
	 * links while we are modifying them.
	 */
	cpssp->current_tb = NULL;

	for (i = 0; i < CPU_TLB_SIZE; i++) {
		cpssp->tlb_read[0][i].address = -1;
		cpssp->tlb_write[0][i].address = -1;
		cpssp->tlb_code[0][i].address = -1;
		cpssp->tlb_read[1][i].address = -1;
		cpssp->tlb_write[1][i].address = -1;
		cpssp->tlb_code[1][i].address = -1;
	}

	memset(cpssp->tb_jmp_cache, 0, sizeof(cpssp->tb_jmp_cache));
}

static void
NAME_(mw_code_b)(struct cpssp *cpssp, Paddr pa, uint32_t val)
{
	NAME_(tb_invalidate_phys_page_fast)(cpssp, pa, 1);
	NAME_(mw_data_b)(cpssp, pa, val);
}

static void
NAME_(mw_code_w)(struct cpssp *cpssp, Paddr pa, uint32_t val)
{
	NAME_(tb_invalidate_phys_page_fast)(cpssp, pa, 2);
	NAME_(mw_data_w)(cpssp, pa, val);
}

static void
NAME_(mw_code_l)(struct cpssp *cpssp, Paddr pa, uint32_t val)
{
	NAME_(tb_invalidate_phys_page_fast)(cpssp, pa, 4);
	NAME_(mw_data_l)(cpssp, pa, val);
}

CPUReadMemoryFunc *const NAME_(io_mem_read)[IO_MEM_NB_ENTRIES][4] = {
	[IO_MEM_RAM >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_CODE >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_IO >> IO_MEM_SHIFT] = {
		NAME_(mr_data_b), NAME_(mr_data_w), NAME_(mr_data_l)
	},
	[IO_MEM_IO_CODE >> IO_MEM_SHIFT] = {
		NAME_(mr_data_b), NAME_(mr_data_w), NAME_(mr_data_l)
	},
};
CPUWriteMemoryFunc *const NAME_(io_mem_write)[IO_MEM_NB_ENTRIES][4] = {
	[IO_MEM_RAM >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_CODE >> IO_MEM_SHIFT] = {
		NAME_(mw_code_b), NAME_(mw_code_w), NAME_(mw_code_l)
	},
	[IO_MEM_IO >> IO_MEM_SHIFT] = {
		NAME_(mw_data_b), NAME_(mw_data_w), NAME_(mw_data_l)
	},
	[IO_MEM_IO_CODE >> IO_MEM_SHIFT] = {
		NAME_(mw_code_b), NAME_(mw_code_w), NAME_(mw_code_l)
	},
};
CPUReadMemoryFunc *const NAME_(io_mem_code)[IO_MEM_NB_ENTRIES][4] = {
	[IO_MEM_RAM >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_CODE >> IO_MEM_SHIFT] = {
		NULL, NULL, NULL /* Not used. */
	},
	[IO_MEM_IO >> IO_MEM_SHIFT] = {
		NAME_(mx_code_b), NAME_(mx_code_w), NAME_(mx_code_l)
	},
	[IO_MEM_IO_CODE >> IO_MEM_SHIFT] = {
		NAME_(mx_code_b), NAME_(mx_code_w), NAME_(mx_code_l)
	},
};

#define MMUSUFFIX _cmmu
#define GETPC() ((void *) 0)
#define CODE_ACCESS

#define SHIFT 0
#include "arch_gen_cpu_x86_code_template.c"

#define SHIFT 1
#include "arch_gen_cpu_x86_code_template.c"

#define SHIFT 2
#include "arch_gen_cpu_x86_code_template.c"

#define SHIFT 3
#include "arch_gen_cpu_x86_code_template.c"

#undef GETPC
#undef MMUSUFFIX
