/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define SETS	(CONFIG_CPU_L2_ASSOC)
#define LINES	(CONFIG_CPU_L2_SIZE / CACHE_LINE_SIZE / SETS)

#define COUNT	0

#ifdef STATE
struct {
#ifdef CONFIG_CPU_L2_SIZE
#if COUNT
	int hit;
	int miss;
#endif
	struct {
		struct NAME_(cache2_entry) {
			Paddr paddr;
			Haddr raddr;
			Haddr waddr;
			Haddr xaddr;
#if defined(CONFIG_CPU_SOCKET_ISA)
			int (*rcf)(void *, uint32_t, unsigned int, uint32_t *);
			void *rcs;
			int (*wcf)(void *, uint32_t, unsigned int, uint32_t);
			void *wcs;
			int (*xcf)(void *, uint32_t, unsigned int, uint32_t *);
			void *xcs;
#elif defined(CONFIG_CPU_SOCKET_HOST) \
   || defined(CONFIG_CPU_SOCKET_SLOT1)
			int (*rcf)(void *, uint32_t, unsigned int, uint32_t *);
			void *rcs;
			int (*wcf)(void *, uint32_t, unsigned int, uint32_t);
			void *wcs;
			int (*xcf)(void *, uint32_t, unsigned int, uint32_t *);
			void *xcs;
#elif defined(CONFIG_CPU_SOCKET_775)
			int (*rcf)(void *, uint64_t, unsigned int, uint64_t *);
			void *rcs;
			int (*wcf)(void *, uint64_t, unsigned int, uint64_t);
			void *wcs;
			int (*xcf)(void *, uint64_t, unsigned int, uint64_t *);
			void *xcs;
#else
#error "Unknown socket."
#endif
		} set[SETS];
		LRU_DECL(SETS, lru);
	} line[LINES];
#endif /* CONFIG_CPU_L2_SIZE */
} NAME_(cache2);
#endif /* STATE */

#ifdef BEHAVIOR
#include "arch_gen_cpu_x86_cache2.h"

#ifdef CONFIG_CPU_L2_SIZE
static struct NAME_(cache2_entry) *
NAME_(cache2_line)(struct cpssp *cpssp, Paddr paddr)
{
	int line;
	int set;
	
	line = (paddr / CACHE_LINE_SIZE) % LINES;
	for (set = 0; ; set++) {
		struct NAME_(cache2_entry) *entry;

		if (set == SETS) {
			/* Miss */
#if COUNT
			cpssp->NAME_(cache2).miss++;
#endif
			set = NAME_(lru_oldest)(SETS, cpssp->NAME_(cache2).line[line].lru);
			NAME_(lru_use)(SETS, cpssp->NAME_(cache2).line[line].lru, set);
			entry = &cpssp->NAME_(cache2).line[line].set[set];
			entry->paddr = paddr;
			entry->raddr = NULL;
			entry->waddr = NULL;
			entry->xaddr = NULL;
			return entry;
		}
		entry = &cpssp->NAME_(cache2).line[line].set[set];
		if (entry->paddr == paddr) {
			/* Hit */
#if COUNT
			cpssp->NAME_(cache2).hit++;
#endif
			NAME_(lru_use)(SETS, cpssp->NAME_(cache2).line[line].lru, set);
			return entry;
		}
	}
}
#endif /* CONFIG_CPU_L2_SIZE */

static void
NAME_(cache2_map_r)(struct cpssp *cpssp, Paddr paddr, char **haddrp)
{
	int off;

	off = paddr & ~CACHE_LINE_MASK;
	paddr &= CACHE_LINE_MASK;

#ifdef CONFIG_CPU_L2_SIZE
	struct NAME_(cache2_entry) *entry;
	entry = NAME_(cache2_line)(cpssp, paddr);

	if (unlikely(! entry->raddr)) {
		Haddr haddr;

		NAME_(apic_map_r)(cpssp, paddr, &haddr);
		if (! haddr) {
			*haddrp = NULL;
			return;
		}
		entry->raddr = haddr;
	}

	*haddrp = entry->raddr + off;

#else /* CONFIG_CPU_L2_SIZE */
	Haddr haddr;

	NAME_(apic_map_r)(cpssp, paddr, &haddr);
	if (! haddr) {
		*haddrp = NULL;
		return;
	}

	*haddrp = haddr + off;
#endif /* CONFIG_CPU_L2_SIZE */
}

static void
NAME_(cache2_map_w)(struct cpssp *cpssp, Paddr paddr, char **haddrp)
{
	int off;

	off = paddr & ~CACHE_LINE_MASK;
	paddr &= CACHE_LINE_MASK;

#ifdef CONFIG_CPU_L2_SIZE
	struct NAME_(cache2_entry) *entry;
	entry = NAME_(cache2_line)(cpssp, paddr);

	if (unlikely(! entry->waddr)) {
		Haddr haddr;

		NAME_(apic_map_w)(cpssp, paddr, &haddr);
		if (! haddr) {
			*haddrp = NULL;
			return;
		}
		entry->waddr = haddr;
	}

	*haddrp = entry->waddr + off;

#else /* CONFIG_CPU_L2_SIZE */
	Haddr haddr;

	NAME_(apic_map_w)(cpssp, paddr, &haddr);
	if (! haddr) {
		*haddrp = NULL;
		return;
	}

	*haddrp = haddr + off;
#endif /* CONFIG_CPU_L2_SIZE */
}

static void
NAME_(cache2_map_x)(struct cpssp *cpssp, Paddr paddr, char **haddrp)
{
	int off;

	off = paddr & ~CACHE_LINE_MASK;
	paddr &= CACHE_LINE_MASK;

#ifdef CONFIG_CPU_L2_SIZE
	struct NAME_(cache2_entry) *entry;
	entry = NAME_(cache2_line)(cpssp, paddr);

	if (unlikely(! entry->xaddr)) {
		Haddr haddr;

		NAME_(apic_map_x)(cpssp, paddr, &haddr);
		if (! haddr) {
			*haddrp = NULL;
			return;
		}
		entry->xaddr = haddr;
	}

	*haddrp = entry->xaddr + off;

#else /* CONFIG_CPU_L2_SIZE */
	Haddr haddr;

	NAME_(apic_map_x)(cpssp, paddr, &haddr);
	if (! haddr) {
		*haddrp = NULL;
		return;
	}

	*haddrp = haddr + off;
#endif /* CONFIG_CPU_L2_SIZE */
}

static void
NAME_(cache2_mr)(struct cpssp *cpssp, Paddr addr, unsigned int bs, Data *valp)
{
	return NAME_(apic_mr)(cpssp, addr, bs, valp);
}

static void
NAME_(cache2_mw)(struct cpssp *cpssp, Paddr addr, unsigned int bs, Data val)
{
	return NAME_(apic_mw)(cpssp, addr, bs, val);
}

static void
NAME_(cache2_mx)(struct cpssp *cpssp, Paddr addr, unsigned int bs, Data *valp)
{
	return NAME_(apic_mx)(cpssp, addr, bs, valp);
}

void
NAME_(cache2_unmap)(struct cpssp *cpssp, unsigned long pa, unsigned long len)
{
#ifdef CONFIG_CPU_L2_SIZE
	int line;
	int set;

	for (line = 0; line < LINES; line++) {
		for (set = 0; set < SETS; set++) {
			cpssp->NAME_(cache2).line[line].set[set].paddr = (Paddr) -1; /* Invalid */
		}
	}
#endif

	NAME_(mmui_unmap)(cpssp, pa, len);
	NAME_(mmud_unmap)(cpssp, pa, len);
}

void
NAME_(cache2_reset)(struct cpssp *cpssp)
{
#ifdef CONFIG_CPU_L2_SIZE
	int line;
	int set;

	for (line = 0; line < LINES; line++) {
		for (set = 0; set < SETS; set++) {
			cpssp->NAME_(cache2).line[line].set[set].paddr = (Paddr) -1; /* Invalid */
		}
		NAME_(lru_reset)(SETS, cpssp->NAME_(cache2).line[line].lru);
	}
#endif
}

static void
NAME_(cache2_create)(struct cpssp *cpssp)
{
#ifdef CONFIG_CPU_L2_SIZE
#if COUNT
	cpssp->NAME_(cache2).hit = 0;
	cpssp->NAME_(cache2).miss = 0;
#endif
#endif
}

static void
NAME_(cache2_destroy)(struct cpssp *cpssp)
{
#ifdef CONFIG_CPU_L2_SIZE
#if COUNT
	fprintf(stderr, "%s: hit=%d miss=%d\n", __FUNCTION__,
			cpssp->NAME_(cache2).hit,
			cpssp->NAME_(cache2).miss);
#endif
#endif
}

#endif /* BEHAVIOR */

#undef COUNT

#undef LINES
#undef SETS
