/*
 * Copyright (C) 2007-2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef STATE
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT

#define APIC_ID         0x020
#define APIC_LVR        0x030
#define APIC_TASKPRI    0x080
#define APIC_ARBPRI     0x090
#define APIC_PROCPRI    0x0a0
#define APIC_EOI        0x0b0
#define APIC_LDR        0x0d0
#define APIC_DFR        0x0e0
#define APIC_SPIV       0x0f0
#define APIC_ISR        0x100
#define APIC_TMR        0x180
#define APIC_IRR        0x200
#define APIC_ESR        0x280
#define APIC_ICR        0x300
#define APIC_ICR2       0x310
#define APIC_LVTT       0x320
#define APIC_LVTPC      0x340
#define APIC_LVT0       0x350
#define APIC_LVT1       0x360
#define APIC_LVTERR     0x370
#define APIC_TMICT      0x380
#define APIC_TMCCT      0x390
#define APIC_TDCR       0x3e0

struct local_apic {
/*020*/ /* APIC ID Register */
#if 1   /* Pentium and P6 family */
        unsigned int phys_apic_id : 4;
#elif 0 /* Pentium 4 and Xeon */
        unsigned int phys_apic_id : 8;
#endif

/*080*/ /* Task Priority Register */
        unsigned int tpr : 8;

/*090*/ /* Arbitration Priority Register */
        unsigned int apr : 8;

/*0A0*/ /* Processor Priority Register */
        unsigned int ppr : 8;

/*0D0*/ /* Logical Destination Register */
        unsigned int ldr : 8;

/*0E0*/ /* Destination Format Register */
        unsigned int dfr_model : 4;

/*0F0*/ struct { /* Spurious Interrupt Vector Register */
                unsigned int spurious_vector : 8;
                unsigned int apic_enabled : 1;
                unsigned int focus_cpu : 1;
        } svr;

/*100*/ /* In Service Register */
        uint32_t isr[8];

/*180*/ /* Trigger Mode Register */
        uint32_t tmr[8];

/*200*/ /* Interrupt Request Register */
        uint32_t irr[8];

/*280*/ /* Error Status Register */
        unsigned int send_cs_error : 1;
        unsigned int receive_cs_error : 1;
        unsigned int send_accept_error : 1;
        unsigned int receive_accept_error : 1;
        unsigned int send_illegal_vector : 1;
        unsigned int receive_illegal_vector : 1;
        unsigned int illegal_register_address : 1;

/*300*/ struct { /* Interrupt Command Register 1 */
                unsigned int vector : 8;
                unsigned int delivery_mode : 3;
                unsigned int destination_mode : 1;
                unsigned int delivery_status : 1;
                unsigned int level : 1;
                unsigned int trigger : 1;
                unsigned int shorthand : 2;
        } icr1;

/*310*/ struct { /* Interrupt Command Register 2 */
                unsigned int destination : 8;
        } icr2;

/*320*/ struct { /* LVT - Timer */
                unsigned int vector : 8;
                unsigned int delivery_status : 1;
                unsigned int mask : 1;
                unsigned int timer_mode : 1;
        } lvt_timer;

/*340*/ struct { /* LVT - Performance Counter */
                unsigned int vector : 8;
                unsigned int delivery_mode : 3;
                unsigned int delivery_status : 1;
                unsigned int mask : 1;
        } lvt_pc;

/*350*/
/*360*/ struct { /* LVT - LINT0/1 */
                unsigned int vector : 8;
                unsigned int delivery_mode : 3;
                unsigned int delivery_status : 1;
                unsigned int polarity : 1;
                unsigned int remote_irr : 1;
                unsigned int trigger : 1;
                unsigned int mask : 1;
        } lvt_lint[2];

/*370*/ struct { /* LVT - Error */
                unsigned int vector : 8;
                unsigned int delivery_status : 1;
                unsigned int mask : 1;
        } lvt_error;

/*380*/ /* Timer Initial Count Register */
        uint32_t timer_icr;

/*390*/ /* Timer Current Count Register */
        uint32_t timer_ccr;

/*3E0*/ /* Timer Divide Configuration Register */
        unsigned int timer_dcr : 3;

        unsigned long long timer_event;
        int timer_running;

        int extint_pending;
        uint8_t extint_pri;
        int nmi_pending;
        int smi_pending;

        uint64_t base;
        int apic_enable;
        int bsp;

        unsigned long tsc_to_bus;
} NAME_(apic);

#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT */
#endif /* STATE */

#ifdef BEHAVIOR
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT

#define DEBUG_CONTROL_FLOW	0

#define CPU_APIC_VERSION	1
#define CPU_APIC_MAX_LVT	4

/*forward*/ static void
NAME_(apic_msg0_receive)(
	struct cpssp *cpssp,
	unsigned int destination_mode,
	unsigned int delivery_mode,
	unsigned int level,
	unsigned int trigger_mode,
	uint8_t vector,
	uint8_t destination
);

static int
NAME_(apic_bit_find)(uint32_t *v, unsigned int vsize)
{
	int i;
	int j;

	for (i = (vsize / 32) - 1; ; i--) {
		uint32_t m;

		if (i < 0) {
			return -1;
		}
		m = v[i];
		if (m == 0) {
			continue;
		}
		for (j = 31; ; j--) {
			if (m & (1 << j)) {
				return i * 32 + j;
			}
		}
	}
}

static void
NAME_(apic_bit_set)(uint32_t *v, unsigned int bitno)
{
	v[bitno / 32] |= 1 << (bitno & 0x1f);
}

static void
NAME_(apic_bit_clr)(uint32_t *v, unsigned int bitno)
{
	v[bitno / 32] &= ~(1 << (bitno & 0x1f));
}

static int
NAME_(apic_bit_test)(uint32_t *v, unsigned int bitno)
{
	return v[bitno / 32] >> (bitno & 0x1f) & 1;
}

static inline int
NAME_(apic_extint_pri)(struct cpssp *cpssp)
{
	if (cpssp->NAME_(apic).extint_pending) {
		return cpssp->NAME_(apic).extint_pri;
	} else {
		return -1;
	}
}

static inline int
NAME_(apic_smp_pri)(struct cpssp *cpssp)
{
	int smp_irr;
	int smp_isr;
	int smp_ppr;

	smp_irr = NAME_(apic_bit_find)(cpssp->NAME_(apic).irr, 256);
	smp_isr = NAME_(apic_bit_find)(cpssp->NAME_(apic).isr, 256);
	smp_ppr = cpssp->NAME_(apic).tpr;

	if (0 <= smp_irr
	 && smp_isr < smp_irr
	 && smp_ppr <= smp_irr) {
		return smp_irr;
	} else {
		return -1;
	}
}

static void
NAME_(apic_irq_update)(struct cpssp *cpssp)
{
	int extint_pri;
	int smp_pri;

	extint_pri = NAME_(apic_extint_pri)(cpssp);
	smp_pri = NAME_(apic_smp_pri)(cpssp);

	if (0 <= extint_pri
	 || (cpssp->NAME_(apic).apic_enable
	  && cpssp->NAME_(apic).svr.apic_enabled
	  && 0 <= smp_pri)) {
		cpssp->interrupt_request |= CPU_INTERRUPT_IRQ;
	} else {
		cpssp->interrupt_request &= ~CPU_INTERRUPT_IRQ;
	}
}

static void
NAME_(apic_nmi_update)(struct cpssp *cpssp)
{
	if (cpssp->NAME_(apic).nmi_pending) {
		cpssp->interrupt_request |= CPU_INTERRUPT_NMI;
	} else {
		cpssp->interrupt_request &= ~CPU_INTERRUPT_NMI;
	}
}

static void
NAME_(apic_smi_update)(struct cpssp *cpssp)
{
	if (cpssp->NAME_(apic).smi_pending) {
		cpssp->interrupt_request |= CPU_INTERRUPT_SMI;
	} else {
		cpssp->interrupt_request &= ~CPU_INTERRUPT_SMI;
	}
}

int
NAME_(apic_irq_ack)(struct cpssp *cpssp)
{
	int extint_pri;
	int smp_pri;
	uint8_t vec;

	/* Get external interrupt priority. */
	extint_pri = NAME_(apic_extint_pri)(cpssp);

	/* Get SMP interrupt priority */
	smp_pri = NAME_(apic_smp_pri)(cpssp);

	if (extint_pri < 0
	 && smp_pri < 0) {
		/*
		 * Spurious interrupt.
		 */
		vec = cpssp->NAME_(apic).svr.spurious_vector;

	} else if (extint_pri < smp_pri) {
		/*
		 * SMP with high priority.
		 */
		vec = smp_pri;

		NAME_(apic_bit_clr)(cpssp->NAME_(apic).irr, vec);
		NAME_(apic_bit_set)(cpssp->NAME_(apic).isr, vec);
		if (NAME_(apic_bit_test)(cpssp->NAME_(apic).tmr, vec)) {
			NAME_(apic_bit_clr)(cpssp->NAME_(apic).tmr, vec);
			/* Send EOI to all IOAPICs. */
			sig_icc_bus_eoi(cpssp->icc_bus, cpssp, vec);
		}

	} else { assert(smp_pri <= extint_pri);
		/*
		 * EXTINT with high priority.
		 */
		cpssp->NAME_(apic).extint_pending = 0;

		NAME_(ack)(cpssp, &vec);
	}

	NAME_(apic_irq_update)(cpssp);

	return vec;
}

void
NAME_(apic_nmi_ack)(struct cpssp *cpssp)
{
	cpssp->NAME_(apic).nmi_pending = 0;
	NAME_(apic_nmi_update)(cpssp);
}

void
NAME_(apic_smi_ack)(struct cpssp *cpssp)
{
	cpssp->NAME_(apic).smi_pending = 0;
	NAME_(apic_smi_update)(cpssp);
}

static int
NAME_(apic_affected)(
	struct cpssp *cpssp,
	unsigned int destination_mode,
	uint8_t destination
)
{
	if (destination_mode == 0) {
		/*
		 * Physical Destination
		 */
		/* High order bits are ignored. */
		destination &= 0x0f;

		if (destination == cpssp->NAME_(apic).phys_apic_id
		 || destination == 0xf) {
			return 1;
		} else {
			return 0;
		}
	} else {
		/*
		 * Logical Destination
		 */
		switch (cpssp->NAME_(apic).dfr_model) {
		case 0x0: /* Cluster Model */
			if ((destination >> 4) == (cpssp->NAME_(apic).ldr >> 4)
			 || (destination >> 4) == 0xf) {
				return (destination & cpssp->NAME_(apic).ldr & 0xf) ? 1 : 0;
			} else {
				return 0;
			}
		case 0xf: /* Flat Model */
			return (destination & cpssp->NAME_(apic).ldr) != 0;
		default:
			assert(0);
		}
	}
}

static void
NAME_(apic_deliver_irq_local)(
	struct cpssp *cpssp,
	unsigned int delivery_mode,
	unsigned int level,
	unsigned int trigger_mode,
	uint8_t vector
)
{
	if (2 <= loglevel + DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: delivery_mode=%d, level=%d, trigger_mode=%d, vector=0x%02x\n",
				__FUNCTION__,
				delivery_mode,
				level, trigger_mode, vector);
	}

	switch (delivery_mode) {
	case 0: /* FIXED */
		NAME_(apic_bit_set)(cpssp->NAME_(apic).irr, vector);
		if (trigger_mode) {
			NAME_(apic_bit_set)(cpssp->NAME_(apic).tmr, vector);
		} else {
			NAME_(apic_bit_clr)(cpssp->NAME_(apic).tmr, vector);
		}
		NAME_(apic_irq_update)(cpssp);
		break;

	case 1: /* Lowest Priority */
		/* Same as FIXED for now. FIXME */
		NAME_(apic_bit_set)(cpssp->NAME_(apic).irr, vector);
		if (trigger_mode) {
			NAME_(apic_bit_set)(cpssp->NAME_(apic).tmr, vector);
		} else {
			NAME_(apic_bit_clr)(cpssp->NAME_(apic).tmr, vector);
		}
		NAME_(apic_irq_update)(cpssp);
		break;

	case 2: /* SMI */
		cpssp->NAME_(apic).smi_pending = 1;
		NAME_(apic_smi_update)(cpssp);
		break;

	case 4: /* NMI */
		cpssp->NAME_(apic).nmi_pending = 1;
		NAME_(apic_nmi_update)(cpssp);
		break;

	case 5: /* INIT */
		if (level == 0) {
			/* Set arbitration ID to value of APIC ID. */
			/* Nothing to do, yet... */
		} else {
			NAME_(core_init_by_apic)(cpssp);
		}
		break;

	case 6: /* STARTUP */
		NAME_(core_startup)(cpssp, vector);
		break;

	case 7: /* EXTINT */
		cpssp->NAME_(apic).extint_pending = 1;
		cpssp->NAME_(apic).extint_pri = vector;
		NAME_(apic_irq_update)(cpssp);
		break;

	default:
		fprintf(stderr, "delivery_mode=%d\n", delivery_mode);
		assert(0); /* Cannot happen. */
	}
}

static void
NAME_(apic_deliver_eoi_local)(struct cpssp *cpssp, uint8_t vec)
{
	/* FIXME VOSSI */
}

static void
NAME_(apic_deliver_eoi)(struct cpssp *cpssp, uint8_t vec)
{
	NAME_(apic_deliver_eoi_local)(cpssp, vec);
	/* FIXME VOSSI */
}

static void
NAME_(apic_lintX_set)(struct cpssp *cpssp, unsigned int nr, unsigned int val)
{
	if (2 <= loglevel + DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: lint%d set to %d\n", __FUNCTION__,
				nr, val);
	}

	if (cpssp->NAME_(apic).apic_enable
	 && cpssp->NAME_(apic).svr.apic_enabled) {
		/*
		 * APIC enabled.
		 */
		val ^= cpssp->NAME_(apic).lvt_lint[nr].polarity;
		val &= ! cpssp->NAME_(apic).lvt_lint[nr].mask;

		if (val) {
			NAME_(apic_deliver_irq_local)(cpssp,
				cpssp->NAME_(apic).lvt_lint[nr].delivery_mode,
				1, /* Level */
				cpssp->NAME_(apic).lvt_lint[nr].trigger,
				cpssp->NAME_(apic).lvt_lint[nr].vector);
		}
	} else {
		/*
		 * APIC disabled.
		 */
		if (val) {
			switch (nr) {
			case 0: /* IRQ */
				NAME_(apic_deliver_irq_local)(cpssp,
						7, 1, 0, 0x00);
				break;
			case 1: /* NMI */
				NAME_(apic_deliver_irq_local)(cpssp,
						4, 1, 1, 0x00);
				break;
			default:
				assert(0);
			}
		}
	}
}

static void
NAME_(apic_lint0_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(apic_lintX_set)(cpssp, 0, val);
}

static void
NAME_(apic_lint1_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(apic_lintX_set)(cpssp, 1, val);
}

static void
NAME_(apic_smi_set)(struct cpssp *cpssp, unsigned int val)
{
	cpssp->NAME_(apic).smi_pending = val;
	NAME_(apic_smi_update)(cpssp);
}

static void
NAME_(apic_eoi_receive)(struct cpssp *cpssp, uint8_t vector)
{
	fixme();
}

static void
NAME_(apic_msg0_receive)(
	struct cpssp *cpssp,
	unsigned int destination_mode,
	unsigned int delivery_mode,
	unsigned int level,
	unsigned int trigger_mode,
	uint8_t vector,
	uint8_t destination
)
{
	if (2 <= loglevel + DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: destination_mode=%d, delivery_mode=%d, level=%d, trigger_mode=%d, vector=0x%02x, destination=0x%02x\n",
				__FUNCTION__,
				destination_mode, delivery_mode,
				level, trigger_mode, vector, destination);
	}

	if (NAME_(apic_affected)(cpssp, destination_mode, destination)) {
		NAME_(apic_deliver_irq_local)(cpssp, delivery_mode, level,
				trigger_mode, vector);
	}
}

static void
NAME_(apic_status0_receive)(struct cpssp *cpssp, unsigned int status)
{
	fixme();
}

static void
NAME_(apic_msg1_receive)(struct cpssp *cpssp, uint8_t processor_priority)
{
	fixme();
}

static void
NAME_(apic_status1_receive)(struct cpssp *cpssp, unsigned int status)
{
	fixme();
}

#if 0
static unsigned int
NAME_(apic_timer_base)(struct cpssp *css)
{
	unsigned int value;
	unsigned int mask;
	unsigned int base;

	value = css->NAME_(apic).timer_dcr;

	mask = (value & 0x3) | ((value & 0x8) >> 1);

	switch (mask) {
	case 0: base = 2; break;
	case 1: base = 4; break;
	case 2: base = 8; break;
	case 3: base = 16; break;
	case 4: base = 32; break;
	case 5: base = 64; break;
	case 6: base = 128; break;
	case 7: base = 1; break;
	default: assert(0); /* Cannot happen. */
	}

	return base;
}
#endif

/*forward*/ static void
NAME_(apic_timer_event)(void *_cpssp);

static void
NAME_(apic_timer_update)(struct cpssp *cpssp)
{
	unsigned long long tsc;
	unsigned long long ticks;
	int irq;

	tsc = time_virt() - cpssp->NAME_(apic).timer_event;
	ticks = (tsc / cpssp->NAME_(apic).tsc_to_bus) >> cpssp->NAME_(apic).timer_dcr;
	cpssp->NAME_(apic).timer_event += (ticks << cpssp->NAME_(apic).timer_dcr) * cpssp->NAME_(apic).tsc_to_bus;

	irq = 0;
	while (cpssp->NAME_(apic).timer_ccr != 0
	    && 0 < ticks) {
		if (ticks < cpssp->NAME_(apic).timer_ccr) {
			cpssp->NAME_(apic).timer_ccr -= ticks;
			ticks = 0;
		} else {
			ticks -= cpssp->NAME_(apic).timer_ccr;
			cpssp->NAME_(apic).timer_ccr = 0;
			if (cpssp->NAME_(apic).lvt_timer.timer_mode) {
				/* Periodic Timer */
				cpssp->NAME_(apic).timer_ccr = cpssp->NAME_(apic).timer_icr;
			}
			irq = 1;
		}
	}
	if (irq
	 && ! cpssp->NAME_(apic).lvt_timer.mask) {
		NAME_(apic_deliver_irq_local)(
				cpssp,
				0, /* Delivery Mode FIXED */
				1, /* Level */
				0, /* Edge Triggered */
				cpssp->NAME_(apic).lvt_timer.vector);
	}
}

static void
NAME_(apic_timer_stop)(struct cpssp *cpssp)
{
	if (cpssp->NAME_(apic).timer_running) {
		int ret;

		ret = time_call_delete(NAME_(apic_timer_event), cpssp);
		assert(ret == 0);

		cpssp->NAME_(apic).timer_running = 0;
	}
}

static void
NAME_(apic_timer_start)(struct cpssp *cpssp)
{
	if (cpssp->NAME_(apic).timer_ccr != 0) {
		unsigned long long ticks;
		unsigned long long tsc;

		ticks = cpssp->NAME_(apic).timer_ccr << cpssp->NAME_(apic).timer_dcr;
		tsc = cpssp->NAME_(apic).timer_event + ticks * cpssp->NAME_(apic).tsc_to_bus;
		time_call_at(tsc, NAME_(apic_timer_event), cpssp);

		cpssp->NAME_(apic).timer_running = 1;
	}
}

static void
NAME_(apic_timer_event)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	/* Timer already stopped by time code. */
	cpssp->NAME_(apic).timer_running = 0;

	NAME_(apic_timer_update)(cpssp);

	NAME_(apic_timer_start)(cpssp);
}

static uint32_t
NAME_(_apic_read)(struct cpssp *cpssp, uint32_t reg)
{
	unsigned long val;

	switch (reg) {
	case APIC_ID: /* 0x020 */
		val = cpssp->NAME_(apic).phys_apic_id << 24;
		break;

	case APIC_LVR: /* 0x030 */
		val = CPU_APIC_MAX_LVT << 16;
		val |= 1 << 4; /* Internal APIC */
		val |= CPU_APIC_VERSION << 0;
		break;

	case APIC_TASKPRI: /* 0x080 */
		val = cpssp->NAME_(apic).tpr;
		break;

	case APIC_ARBPRI: /* 0x090 */
		val = 0; /* FIXME VOSSI */
		break;

	case APIC_PROCPRI: /* 0x0a0 */
		val = 0; /* FIXME VOSSI */
		break;

	case APIC_EOI: /* 0x0b0 */
		/* Write-only. */
		val = 0;
		break;

	case APIC_LDR: /* 0x0d0 */
		val = cpssp->NAME_(apic).ldr << 24;
		break;

	case APIC_DFR: /* 0x0e0 */
		val = cpssp->NAME_(apic).dfr_model << 28;
		val |= 0x0fffffff;
		break;

	case APIC_SPIV: /* 0x0f0 */
		val = cpssp->NAME_(apic).svr.focus_cpu << 9;
		val |= cpssp->NAME_(apic).svr.apic_enabled << 8;
		val |= cpssp->NAME_(apic).svr.spurious_vector << 0;
		break;

	case APIC_ISR + 0x00: /* 0x100 */
	case APIC_ISR + 0x10:
	case APIC_ISR + 0x20:
	case APIC_ISR + 0x30:
	case APIC_ISR + 0x40:
	case APIC_ISR + 0x50:
	case APIC_ISR + 0x60:
	case APIC_ISR + 0x70:
		val = cpssp->NAME_(apic).isr[(reg >> 4) & 7];
		break;

	case APIC_TMR + 0x00: /* 0x180 */
	case APIC_TMR + 0x10:
	case APIC_TMR + 0x20:
	case APIC_TMR + 0x30:
	case APIC_TMR + 0x40:
	case APIC_TMR + 0x50:
	case APIC_TMR + 0x60:
	case APIC_TMR + 0x70:
		val = cpssp->NAME_(apic).tmr[(reg >> 4) & 7];
		break;

	case APIC_IRR + 0x00: /* 0x200 */
	case APIC_IRR + 0x10:
	case APIC_IRR + 0x20:
	case APIC_IRR + 0x30:
	case APIC_IRR + 0x40:
	case APIC_IRR + 0x50:
	case APIC_IRR + 0x60:
	case APIC_IRR + 0x70:
		val = cpssp->NAME_(apic).irr[(reg >> 4) & 7];
		break;

	case APIC_ESR: /* 0x280 */
		val = 0;
		val |= cpssp->NAME_(apic).illegal_register_address << 7;
		val |= cpssp->NAME_(apic).receive_illegal_vector << 6;
		val |= cpssp->NAME_(apic).send_illegal_vector << 5;
		/* Bit 4: reserved */
		val |= cpssp->NAME_(apic).receive_accept_error << 3;
		val |= cpssp->NAME_(apic).send_accept_error << 2;
		val |= cpssp->NAME_(apic).receive_cs_error << 1;
		val |= cpssp->NAME_(apic).send_cs_error << 0;
		break;

	case APIC_ICR: /* 0x300 */
		val = 0;
		val |= cpssp->NAME_(apic).icr1.shorthand << 18;
		val |= cpssp->NAME_(apic).icr1.trigger << 15;
		val |= cpssp->NAME_(apic).icr1.level << 14;
		val |= cpssp->NAME_(apic).icr1.delivery_status << 12;
		val |= cpssp->NAME_(apic).icr1.destination_mode << 11;
		val |= cpssp->NAME_(apic).icr1.delivery_mode << 8;
		val |= cpssp->NAME_(apic).icr1.vector << 0;
		break;

	case APIC_ICR2: /* 0x310 */
		val = 0;
		val |= cpssp->NAME_(apic).icr2.destination << 24;
		break;

	case APIC_LVTT: /* 0x320 */
		val = 0;
		val |= cpssp->NAME_(apic).lvt_timer.timer_mode << 17;
		val |= cpssp->NAME_(apic).lvt_timer.mask << 16;
		val |= cpssp->NAME_(apic).lvt_timer.delivery_status << 12;
		val |= cpssp->NAME_(apic).lvt_timer.vector << 0;
		break;

#if 4 <= CPU_APIC_MAX_LVT
	case APIC_LVTPC: /* 0x340 */
		val = 0;
		val |= cpssp->NAME_(apic).lvt_pc.mask << 16;
		val |= cpssp->NAME_(apic).lvt_pc.delivery_status << 12;
		val |= cpssp->NAME_(apic).lvt_pc.delivery_mode << 8;
		val |= cpssp->NAME_(apic).lvt_pc.vector << 0;
		break;
#endif
	case APIC_LVT0: /* 0x350 */
		val = 0;
		val |= cpssp->NAME_(apic).lvt_lint[0].mask << 16;
		val |= cpssp->NAME_(apic).lvt_lint[0].trigger << 15;
		val |= cpssp->NAME_(apic).lvt_lint[0].remote_irr << 14;
		val |= cpssp->NAME_(apic).lvt_lint[0].polarity << 13;
		val |= cpssp->NAME_(apic).lvt_lint[0].delivery_status << 12;
		val |= cpssp->NAME_(apic).lvt_lint[0].delivery_mode << 8;
		val |= cpssp->NAME_(apic).lvt_lint[0].vector << 0;
		break;

	case APIC_LVT1: /* 0x360 */
		val = 0;
		val |= cpssp->NAME_(apic).lvt_lint[1].mask << 16;
		val |= cpssp->NAME_(apic).lvt_lint[1].trigger << 15;
		val |= cpssp->NAME_(apic).lvt_lint[1].remote_irr << 14;
		val |= cpssp->NAME_(apic).lvt_lint[1].polarity << 13;
		val |= cpssp->NAME_(apic).lvt_lint[1].delivery_status << 12;
		val |= cpssp->NAME_(apic).lvt_lint[1].delivery_mode << 8;
		val |= cpssp->NAME_(apic).lvt_lint[1].vector << 0;
		break;

#if 3 <= CPU_APIC_MAX_LVT
	case APIC_LVTERR: /* 0x370 */
		val = 0;
		val |= cpssp->NAME_(apic).lvt_error.mask << 16;
		val |= cpssp->NAME_(apic).lvt_error.delivery_status << 12;
		val |= cpssp->NAME_(apic).lvt_error.vector << 0;
		break;
#endif

	case APIC_TMICT: /* 0x380 */
		val = cpssp->NAME_(apic).timer_icr;
		break;

	case APIC_TMCCT: /* 0x390 */
		NAME_(apic_timer_stop)(cpssp);
		NAME_(apic_timer_update)(cpssp);
		val = cpssp->NAME_(apic).timer_ccr;
		NAME_(apic_timer_start)(cpssp);
		break;

	case APIC_TDCR: /* 0x3e0 */
		val = 0;
		val |= (((cpssp->NAME_(apic).timer_dcr - 1) >> 2) & 1) << 3;
		val |= (((cpssp->NAME_(apic).timer_dcr - 1) >> 0) & 3) << 0;
		break;

	default:
		fprintf(stderr, "WARNING: Reading from APIC register 0x%03x\n",
				(unsigned int) reg);
		val = 0;
		break;
	}

	if (2 <= loglevel + DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "APIC: Reading 0x%08lx from register 0x%03x\n",
				(unsigned long) val, (unsigned int) reg);
	}

	return val;
}

static void
NAME_(_apic_write)(struct cpssp *cpssp, uint32_t reg, uint32_t val)
{
	int vec;
	unsigned int destination_mode;
	uint8_t destination;

	if (2 <= loglevel + DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "APIC: Writing 0x%08lx to register 0x%03x\n",
				(unsigned long) val, (unsigned int) reg);
	}

	switch (reg) {
	case APIC_ID: /* 0x020 */
#if 1	/* Pentium and P6 family */
		cpssp->NAME_(apic).phys_apic_id = (val >> 24) & 0xf;
#elif 0	/* Pentium 4 and Xeon */
		cpssp->NAME_(apic).phys_apic_id = (val >> 24) & 0xff;
#endif
		break;

	case APIC_LVR: /* 0x030 */
		/* Read-only. */
		break;

	case APIC_TASKPRI: /* 0x080 */
		cpssp->NAME_(apic).tpr = (val >> 0) & 0xff;
		break;

	case APIC_ARBPRI: /* 0x090 */
		/* Read-only. */
		break;

	case APIC_PROCPRI: /* 0x0a0 */
		/* Read-only. */
		break;

	case APIC_EOI: /* 0x0b0 */
		vec = NAME_(apic_bit_find)(cpssp->NAME_(apic).isr, 256);
		if (0 <= vec) {
			NAME_(apic_bit_clr)(cpssp->NAME_(apic).isr, vec);
			if (NAME_(apic_bit_test)(cpssp->NAME_(apic).tmr, vec)) {
				NAME_(apic_deliver_eoi)(cpssp, vec);
			}
		}

		NAME_(apic_irq_update)(cpssp);
		break;

	case APIC_LDR: /* 0x0d0 */
		cpssp->NAME_(apic).ldr = (val >> 24) & 0xff;
		break;

	case APIC_DFR: /* 0x0e0 */
		cpssp->NAME_(apic).dfr_model = (val >> 28) & 0xf;
		break;

	case APIC_SPIV: /* 0x0f0 */
		val |= 0xf; /* Bit 3-0: 1111; read-only */
		cpssp->NAME_(apic).svr.focus_cpu = (val >> 9) & 1;
		cpssp->NAME_(apic).svr.apic_enabled = (val >> 8) & 1;
		cpssp->NAME_(apic).svr.spurious_vector = (val >> 0) & 0xff;
		break;

	case APIC_ISR + 0x00: /* 0x100 */
	case APIC_ISR + 0x10:
	case APIC_ISR + 0x20:
	case APIC_ISR + 0x30:
	case APIC_ISR + 0x40:
	case APIC_ISR + 0x50:
	case APIC_ISR + 0x60:
	case APIC_ISR + 0x70:
		/* Read-only. */
		break;

	case APIC_TMR + 0x00: /* 0x180 */
	case APIC_TMR + 0x10:
	case APIC_TMR + 0x20:
	case APIC_TMR + 0x30:
	case APIC_TMR + 0x40:
	case APIC_TMR + 0x50:
	case APIC_TMR + 0x60:
	case APIC_TMR + 0x70:
		/* Read-only. */
		break;

	case APIC_IRR + 0x00: /* 0x200 */
	case APIC_IRR + 0x10:
	case APIC_IRR + 0x20:
	case APIC_IRR + 0x30:
	case APIC_IRR + 0x40:
	case APIC_IRR + 0x50:
	case APIC_IRR + 0x60:
	case APIC_IRR + 0x70:
		/* Read-only. */
		break;

	case APIC_ESR: /* 0x280 */
		/* Read-only. */
		break;

	case APIC_ICR: /* 0x300 */
		cpssp->NAME_(apic).icr1.shorthand = (val >> 18) & 3;
		cpssp->NAME_(apic).icr1.trigger = (val >> 15) & 1;
		cpssp->NAME_(apic).icr1.level = (val >> 14) & 1;
		/* Writing to ICR1 always triggers an interrupt */
		cpssp->NAME_(apic).icr1.destination_mode = (val >> 11) & 1;
		cpssp->NAME_(apic).icr1.delivery_mode = (val >> 8) & 7;
		cpssp->NAME_(apic).icr1.vector = (val >> 0) & 0xff;

		switch (cpssp->NAME_(apic).icr1.shorthand) {
		case 0: /* No Shorthand */
			destination_mode = cpssp->NAME_(apic).icr1.destination_mode;
			destination = cpssp->NAME_(apic).icr2.destination;
			break;
		case 1: /* Self */
		case 2: /* All Including Self */
		case 3: /* All Excluding Self */
			destination_mode = 0;
			destination = 0xff;
			break;
		default:
			assert(0); /* Cannot happen. */
		}

		cpssp->NAME_(apic).icr1.delivery_status = 1;

		if (cpssp->NAME_(apic).icr1.shorthand != 3) {
			/* Send IRQ to own APIC. */
			NAME_(apic_msg0_receive)(
					cpssp,
					destination_mode,
					cpssp->NAME_(apic).icr1.delivery_mode, 
					cpssp->NAME_(apic).icr1.level, 
					cpssp->NAME_(apic).icr1.trigger, 
					cpssp->NAME_(apic).icr1.vector,
					destination);
		}
		if (cpssp->NAME_(apic).icr1.shorthand != 1) {
			/* Send IRQ to other APICs. */
			sig_icc_bus_msg0(cpssp->icc_bus, cpssp,
					destination_mode,
					cpssp->NAME_(apic).icr1.delivery_mode, 
					cpssp->NAME_(apic).icr1.level, 
					cpssp->NAME_(apic).icr1.trigger, 
					cpssp->NAME_(apic).icr1.vector,
					destination);
		}

		cpssp->NAME_(apic).icr1.delivery_status = 0;
		break;

	case APIC_ICR2: /* 0x310 */
		cpssp->NAME_(apic).icr2.destination = (val >> 24) & 0xff;
		break;

	case APIC_LVTT: /* 0x320 */
		cpssp->NAME_(apic).lvt_timer.timer_mode = (val >> 17) & 1;
		cpssp->NAME_(apic).lvt_timer.mask = (val >> 16) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME_(apic).lvt_timer.vector = (val >> 0) & 0xff;
		break;

#if 4 <= CPU_APIC_MAX_LVT
	case APIC_LVTPC: /* 0x340 */
		cpssp->NAME_(apic).lvt_pc.mask = (val >> 16) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME_(apic).lvt_pc.delivery_mode = (val >> 8) & 7;
		cpssp->NAME_(apic).lvt_pc.vector = (val >> 0) & 0xff;
		break;
#endif

	case APIC_LVT0: /* 0x350 */
		cpssp->NAME_(apic).lvt_lint[0].mask = (val >> 16) & 1;
		cpssp->NAME_(apic).lvt_lint[0].trigger = (val >> 15) & 1;
		cpssp->NAME_(apic).lvt_lint[0].remote_irr = (val >> 14) & 1;
		cpssp->NAME_(apic).lvt_lint[0].polarity = (val >> 13) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME_(apic).lvt_lint[0].delivery_mode = (val >> 8) & 7;
		cpssp->NAME_(apic).lvt_lint[0].vector = (val >> 0) & 0xff;
		break;

	case APIC_LVT1: /* 0x360 */
		cpssp->NAME_(apic).lvt_lint[1].mask = (val >> 16) & 1;
		cpssp->NAME_(apic).lvt_lint[1].trigger = (val >> 15) & 1;
		cpssp->NAME_(apic).lvt_lint[1].remote_irr = (val >> 14) & 1;
		cpssp->NAME_(apic).lvt_lint[1].polarity = (val >> 13) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME_(apic).lvt_lint[1].delivery_mode = (val >> 8) & 7;
		cpssp->NAME_(apic).lvt_lint[1].vector = (val >> 0) & 0xff;
		break;

#if 3 <= CPU_APIC_MAX_LVT
	case APIC_LVTERR: /* 0x370 */
		cpssp->NAME_(apic).lvt_error.mask = (val >> 16) & 1;
		/* Bit 12 (delivery status) is read only! */
		cpssp->NAME_(apic).lvt_error.vector = (val >> 0) & 0xff;
		break;
#endif

	case APIC_TMICT: /* 0x380 */
		/* Writing to TMICT starts the timer */
		NAME_(apic_timer_stop)(cpssp);
		cpssp->NAME_(apic).timer_event = time_virt();
		cpssp->NAME_(apic).timer_icr = val;
		cpssp->NAME_(apic).timer_ccr = val;
		NAME_(apic_timer_start)(cpssp);
		break;

	case APIC_TMCCT: /* 0x390 */
		/* Read-only. */
		break;

	case APIC_TDCR: /* 0x3e0 */
		/*
		 * 0000 -> 1
		 * 0001 -> 2
		 * 0010 -> 3
		 * 0011 -> 4
		 * 1000 -> 5
		 * 1001 -> 6
		 * 1010 -> 7
		 * 1011 -> 0
		 */
		val |= (val >> 1) & 4;
		val += 1;
		val &= 7;
		cpssp->NAME_(apic).timer_dcr = val;
		break;

	default:
		assert(0); /* FIXME VOSSI */
	}
}

void
NAME_(apic_base_msr_set)(struct cpssp *cpssp, uint64_t val)
{
	if (2 <= loglevel + DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "APIC: Writing 0x%08lx to msr\n",
				(unsigned long) val);
	}

	if (cpssp->NAME_(apic).apic_enable) {
		/* Invalidate old mapping. */
		NAME_(cache2_unmap)(cpssp, cpssp->NAME_(apic).base, 0x1000);
	}

	cpssp->NAME_(apic).base = val & ~0xfff;
	cpssp->NAME_(apic).apic_enable = (val >> 11) & 1;

	if (cpssp->NAME_(apic).apic_enable) {
		/* Invalidate old mapping. */
		NAME_(cache2_unmap)(cpssp, cpssp->NAME_(apic).base, 0x1000);
	}
}

uint64_t
NAME_(apic_base_msr_get)(struct cpssp *cpssp)
{
	uint64_t val;

	val = cpssp->NAME_(apic).base
		| (cpssp->NAME_(apic).apic_enable << 11)
		| (cpssp->NAME_(apic).bsp << 8);

	if (2 <= loglevel + DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "APIC: Reading 0x%08lx from msr\n",
				(unsigned long) val);
	}

	return val;
}

void
NAME_(set_apic_tpr)(struct cpssp *cpssp, uint8_t val)
{
	if (2 <= loglevel + DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "APIC: Writing 0x%02x to tpr\n",
				(unsigned int) val);
	}

	cpssp->NAME_(apic).tpr = (val & 0x0f) << 4;
	NAME_(apic_irq_update)(cpssp);
}

uint8_t
NAME_(get_apic_tpr)(struct cpssp *cpssp)
{
	uint8_t val;
	
	val = cpssp->NAME_(apic).tpr >> 4;

	if (2 <= loglevel + DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "APIC: Reading 0x%02x from tpr\n",
				(unsigned int) val);
	}

	return val;
}

#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT */

#if defined(CONFIG_CPU_SOCKET_ISA)
/* 16 Bit Data Bus */

static void
NAME_(apic_mr)(struct cpssp *cpssp, Paddr pa, unsigned int bs, uint16_t *valp)
{
	/* No APIC */
	NAME_(mr)(cpssp, pa, bs, valp);
}

static void
NAME_(apic_mw)(struct cpssp *cpssp, Paddr pa, unsigned int bs, uint16_t val)
{
	/* No APIC */
	NAME_(mw)(cpssp, pa, bs, val);
}

static void
NAME_(apic_mx)(struct cpssp *cpssp, Paddr pa, unsigned int bs, uint16_t *valp)
{
	NAME_(mx)(cpssp, pa, bs, valp);
}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
   || defined(CONFIG_CPU_SOCKET_SLOT1)
/* 32 Bit Data Bus */

static void
NAME_(apic_mr)(struct cpssp *cpssp, Paddr pa, unsigned int bs, uint32_t *valp)
{
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if ((pa & ~0xff0) == cpssp->NAME_(apic).base
	 && bs == 0b1111) {
		*valp = NAME_(_apic_read)(cpssp, pa & 0xff0);
		return;
	}
#endif
	NAME_(mr)(cpssp, pa, bs, valp);
}

static void
NAME_(apic_mw)(struct cpssp *cpssp, Paddr pa, unsigned int bs, uint32_t val)
{
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if ((pa & ~0xff0) == cpssp->NAME_(apic).base
	 && bs == 0b1111) {
		NAME_(_apic_write)(cpssp, pa & 0xff0, val);
		return;
	}
#endif
	NAME_(mw)(cpssp, pa, bs, val);
}

static void
NAME_(apic_mx)(struct cpssp *cpssp, Paddr pa, unsigned int bs, uint32_t *valp)
{
	NAME_(mx)(cpssp, pa, bs, valp);
}

#elif defined(CONFIG_CPU_SOCKET_775)
/* 64 Bit Data Bus */

static void
NAME_(apic_mr)(struct cpssp *cpssp, Paddr pa, unsigned int bs, uint64_t *valp)
{
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if ((pa & ~0xff0) == cpssp->NAME_(apic).base
	 && bs == 0b1111) {
		*valp = (uint64_t) NAME_(_apic_read)(cpssp, pa & 0xff0);
		return;
	}
#endif
	NAME_(mr)(cpssp, pa, bs, valp);
}

static void
NAME_(apic_mw)(struct cpssp *cpssp, Paddr pa, unsigned int bs, uint64_t val)
{
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if ((pa & ~0xff0) == cpssp->NAME_(apic).base
	 && bs == 0b1111) {
		NAME_(_apic_write)(cpssp, pa & 0xff0, (uint32_t) val);
		return;
	}
#endif
	NAME_(mw)(cpssp, pa, bs, val);
}

static void
NAME_(apic_mx)(struct cpssp *cpssp, Paddr pa, unsigned int bs, uint64_t *valp)
{
	NAME_(mx)(cpssp, pa, bs, valp);
}

#else
#error "Unknown socket."
#endif

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
static int
NAME_(apic_map)(struct cpssp *cpssp, unsigned long pa)
{
	if ((cpssp->NAME_(apic).base & ~0xfff) == (pa & ~0xfff)) {
		return 0;
	} else {
		return 1;
	}
}
#endif

static int
NAME_(apic_map_r)(struct cpssp *cpssp, Paddr paddr, char **haddrp)
{
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if (! NAME_(apic_map)(cpssp, paddr)) {
		*haddrp = NULL;
		return 0;
	}
#endif
	return NAME_(map_r)(cpssp, paddr, haddrp);
}

static int
NAME_(apic_map_w)(struct cpssp *cpssp, Paddr paddr, char **haddrp)
{
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if (! NAME_(apic_map)(cpssp, paddr)) {
		*haddrp = NULL;
		return 0;
	}
#endif
	return NAME_(map_w)(cpssp, paddr, haddrp);
}

static int
NAME_(apic_map_x)(struct cpssp *cpssp, Paddr paddr, char **haddrp)
{
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	if (! NAME_(apic_map)(cpssp, paddr)) {
		*haddrp = NULL;
		return 0;
	}
#endif
	return NAME_(map_x)(cpssp, paddr, haddrp);
}

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT

void
NAME_(apic_reset)(struct cpssp *cpssp)
{
	cpssp->NAME_(apic).phys_apic_id = cpssp->apic_arbitration_id;
	cpssp->NAME_(apic).dfr_model = 0xF;
	cpssp->NAME_(apic).svr.spurious_vector = 0xff;
	cpssp->NAME_(apic).svr.apic_enabled = 0;
	cpssp->NAME_(apic).svr.focus_cpu = 1;
	cpssp->NAME_(apic).lvt_timer.timer_mode = 0;
	cpssp->NAME_(apic).lvt_timer.mask = 1;
	cpssp->NAME_(apic).lvt_timer.vector = 0x00;
	cpssp->NAME_(apic).lvt_pc.mask = 1;
	cpssp->NAME_(apic).lvt_pc.delivery_mode = 0;
	cpssp->NAME_(apic).lvt_pc.vector = 0x00;
	cpssp->NAME_(apic).lvt_lint[0].mask = 1;
	cpssp->NAME_(apic).lvt_lint[0].trigger = 0;
	cpssp->NAME_(apic).lvt_lint[0].polarity = 0;
	cpssp->NAME_(apic).lvt_lint[0].delivery_mode = 0;
	cpssp->NAME_(apic).lvt_lint[0].vector = 0x00;
	cpssp->NAME_(apic).lvt_lint[1].mask = 1;
	cpssp->NAME_(apic).lvt_lint[1].trigger = 0;
	cpssp->NAME_(apic).lvt_lint[1].polarity = 0;
	cpssp->NAME_(apic).lvt_lint[1].delivery_mode = 0;
	cpssp->NAME_(apic).lvt_lint[1].vector = 0x00;
	cpssp->NAME_(apic).lvt_error.mask = 1;
	cpssp->NAME_(apic).lvt_error.vector = 0x00;
	cpssp->NAME_(apic).base = 0xfee00000;
	cpssp->NAME_(apic).apic_enable = 1;

	cpssp->NAME_(apic).timer_event = 0;
	cpssp->NAME_(apic).timer_running = 0;

	cpssp->NAME_(apic).tsc_to_bus = TIME_HZ / SIG_HOST_BUS_HZ;
}

void
NAME_(apic_init)(struct cpssp *cpssp)
{
}

void
NAME_(apic_create)(struct cpssp *cpssp)
{
	static int count = 0;

	cpssp->NAME_(apic).bsp = 1;
	count++;
}

void
NAME_(apic_destroy)(struct cpssp *cpssp)
{
	/* Don't use cpssp! See cpu.c -- FIXME */
}

#else /* 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT */

static void
NAME_(apic_lint0_set)(struct cpssp *cpssp, unsigned int val)
{
	if (val) {
		cpssp->interrupt_request |= CPU_INTERRUPT_IRQ;
	} else {
		cpssp->interrupt_request &= ~CPU_INTERRUPT_IRQ;
	}
}

static void
NAME_(apic_lint1_set)(struct cpssp *cpssp, unsigned int val)
{
	if (val) {
		cpssp->interrupt_request |= CPU_INTERRUPT_NMI;
	}
}

#if 80386 <= CONFIG_CPU
static void
NAME_(apic_smi_set)(struct cpssp *cpssp, unsigned int val)
{
	if (val) {
		cpssp->interrupt_request |= CPU_INTERRUPT_SMI;
	}
}
#endif

int
NAME_(apic_irq_ack)(struct cpssp *cpssp)
{
	uint8_t vec;

	NAME_(ack)(cpssp, &vec);

	return vec;
}

void
NAME_(apic_nmi_ack)(struct cpssp *cpssp)
{
	cpssp->interrupt_request &= ~CPU_INTERRUPT_NMI;
}

#if 80386 <= CONFIG_CPU
void
NAME_(apic_smi_ack)(struct cpssp *cpssp)
{
	cpssp->interrupt_request &= ~CPU_INTERRUPT_SMI;
}
#endif

void
NAME_(apic_reset)(struct cpssp *cpssp) { }
void
NAME_(apic_init)(struct cpssp *cpssp) { }
void
NAME_(apic_create)(struct cpssp *cpssp) { }
void
NAME_(apic_destroy)(struct cpssp *cpssp) { }

#undef DEBUG_CONTROL_FLOW

#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT */
#endif /* BEHAVIOR */
