/*
 * Copyright (C) 2007-2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_IO	0

uint8_t
NAME_(inb)(struct cpssp *cpssp, uint16_t port)
{
	uint8_t val8;

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	uint16_t val16;

	NAME_(ior)(cpssp, port & ~1, 1 << (port & 1), &val16);
	val8 = (val16 >> ((port & 1) * 8)) & 0xff;

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val32;

	NAME_(ior)(cpssp, port & ~3, 1 << (port & 3), &val32);
	val8 = (val32 >> ((port & 3) * 8)) & 0xff;

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

	NAME_(ior)(cpssp, port & ~7, 1 << (port & 7), &val64);
	val8 = (val64 >> ((port & 7) * 8)) & 0xff;

#else
#error "Unknown socket."
#endif

	if (2 <= DEBUG_IO + loglevel) {
		fprintf(stderr, "%s: 0x%04x -> 0x%02x\n", __FUNCTION__,
				port, val8);
	}
	return val8;
}

uint16_t
NAME_(inw)(struct cpssp *cpssp, uint16_t port)
{
	uint16_t val16;

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	if (0 < (port & 1)) {
		uint16_t val0;
		uint16_t val1;

		NAME_(ior)(cpssp, (port & ~1) + 0, (0x3 << (port & 1)) & 0x3, &val0);
		NAME_(ior)(cpssp, (port & ~1) + 2, (0x3 << (port & 1)) >> 2, &val1);
		val16 = val0 | (val1 << 8);

	} else {
		NAME_(ior)(cpssp, port, 0x3, &val16);
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	if (2 < (port & 3)) {
		uint32_t val0;
		uint32_t val1;

		NAME_(ior)(cpssp, (port & ~3) + 0, (0x3 << (port & 3)) & 0xf, &val0);
		NAME_(ior)(cpssp, (port & ~3) + 4, (0x3 << (port & 3)) >> 4, &val1);
		val16 = (val0 >> ((port & 3) * 8))
		      | (val1 << ((4 - (port & 3)) * 8));

	} else {
		uint32_t val32;

		NAME_(ior)(cpssp, port & ~3, 3 << (port & 3), &val32);
		val16 = (val32 >> ((port & 3) * 8)) & 0xffff;
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	if (6 < (port & 7)) {
		uint64_t val0;
		uint64_t val1;

		NAME_(ior)(cpssp, (port & ~7) + 0, (0x3 << (port & 7)) & 0xff, &val0);
		NAME_(ior)(cpssp, (port & ~7) + 8, (0x3 << (port & 7)) >> 8, &val1);
		val16 = (val0 >> ((port & 7) * 8))
		      | (val1 << ((8 - (port & 7)) * 8));

	} else {
		uint64_t val64;

		NAME_(ior)(cpssp, port & ~7, 3 << (port & 7), &val64);
		val16 = (val64 >> ((port & 7) * 8)) & 0xffff;
	}
#else
#error "Unknown socket."
#endif

	if (2 <= DEBUG_IO + loglevel) {
		fprintf(stderr, "%s: 0x%04x -> 0x%04x\n", __FUNCTION__,
				port, val16);
	}
	return val16;
}

#if 80386 <= CONFIG_CPU
uint32_t
NAME_(inl)(struct cpssp *cpssp, uint16_t port)
{
	uint32_t val32;

#if defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	if (0 < (port & 3)) {
		uint32_t val0;
		uint32_t val1;

		NAME_(ior)(cpssp, (port & ~3) + 0, (0xf << (port & 3)) & 0xf, &val0);
		NAME_(ior)(cpssp, (port & ~3) + 4, (0xf << (port & 3)) >> 4, &val1);
		val32 = (val0 >> ((port & 3) * 8))
		      | (val1 << ((4 - (port & 3)) * 8));

	} else {
		NAME_(ior)(cpssp, port, 0xf, &val32);
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	if (4 < (port & 7)) {
		uint64_t val0;
		uint64_t val1;

		NAME_(ior)(cpssp, (port & ~7) + 0, (0xf << (port & 7)) & 0xff, &val0);
		NAME_(ior)(cpssp, (port & ~7) + 8, (0xf << (port & 7)) >> 8, &val1);
		val32 = (val0 >> ((port & 7) * 8))
		      | (val1 << ((8 - (port & 7)) * 8));

	} else {
		uint64_t val64;

		NAME_(ior)(cpssp, port & ~7, 0xf << (port & 7), &val64);
		val32 = (val64 >> ((port & 7) * 8)) & 0xffffffff;
	}
#else
#error "Unknown socket."
#endif

	if (2 <= DEBUG_IO + loglevel) {
		fprintf(stderr, "%s: 0x%04x -> 0x%08x\n", __FUNCTION__,
				port, val32);
	}
	return val32;
}
#endif /* 80386 <= CONFIG_CPU */

void
NAME_(outb)(struct cpssp *cpssp, uint8_t value, uint16_t port)
{
	if (2 <= DEBUG_IO + loglevel) {
		fprintf(stderr, "%s: 0x%04x <- 0x%02x\n", __FUNCTION__,
				port, value);
	}

#if DEBUG_BIOS_POST_CODE
	if (port == 0x0080
	 || port == 0x0300) {
		/* BIOS Post code. */
		fprintf(stderr, "BIOS: Post code 0x%02x.\n", value);

	}
#endif
	if (port == 0xffff) {
		/* System BIOS / VGA BIOS output port. */
		fprintf(stderr, "%c", value);
		return;
	}

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	uint16_t val16;

	val16 = value << ((port & 1) * 8);
	NAME_(iow)(cpssp, port & ~1, 1 << (port & 1), val16);

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val32;

	val32 = value << ((port & 3) * 8);
	NAME_(iow)(cpssp, port & ~3, 1 << (port & 3), val32);

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

	val64 = (uint64_t) value << ((port & 7) * 8);
	NAME_(iow)(cpssp, port & ~7, 1 << (port & 7), val64);
#else
#error "Unknown socket."
#endif
}

void
NAME_(outw)(struct cpssp *cpssp, uint16_t value, uint16_t port)
{
	if (2 <= DEBUG_IO + loglevel) {
		fprintf(stderr, "%s: 0x%04x <- 0x%04x\n", __FUNCTION__,
				port, value);
	}

#if DEBUG_BIOS_POST_CODE
	if (port == 0x0080
	 || port == 0x0300) {
		/* BIOS Post code. */
		fprintf(stderr, "BIOS: Post code 0x%04x.\n", value);
	}
#endif

#if defined(CONFIG_CPU_SOCKET_ISA)
	/* 16 Bit Data Bus */
	if (0 < (port & 1)) {
		unsigned char value0 = (value >> 0) & 0xff;
		unsigned char value8 = (value >> 8) & 0xff;

		fprintf(stderr, "%s: WARNING: outw->outb port 0x%04x "
				"value 0x%x\n", __FUNCTION__, port, value);

		NAME_(outb)(cpssp, value0, port + 0);
		NAME_(outb)(cpssp, value8, port + 1);

	} else {
		NAME_(iow)(cpssp, port, 3, value);
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	uint32_t val32;

	if (2 < (port & 3)) {
		unsigned char value0 = (value >> 0) & 0xff;
		unsigned char value8 = (value >> 8) & 0xff;

		fprintf(stderr, "%s: WARNING: outw->outb port 0x%04x "
				"value 0x%x\n", __FUNCTION__, port, value);

		NAME_(outb)(cpssp, value0, port + 0);
		NAME_(outb)(cpssp, value8, port + 1);

	} else {
		val32 = value << ((port & 3) * 8);
		NAME_(iow)(cpssp, port & ~3, 3 << (port & 3), val32);
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	uint64_t val64;

	if (6 < (port & 7)) {
		unsigned char value0 = (value >> 0) & 0xff;
		unsigned char value8 = (value >> 8) & 0xff;

		fprintf(stderr, "%s: WARNING: outw->outb port 0x%04x "
				"value 0x%x\n", __FUNCTION__, port, value);

		NAME_(outb)(cpssp, value0, port + 0);
		NAME_(outb)(cpssp, value8, port + 1);

	} else {
		val64 = (uint64_t) value << ((port & 7) * 8);
		NAME_(iow)(cpssp, port & ~7, 3 << (port & 7), val64);
	}
#else
#error "Unknown socket."
#endif
}

#if 80386 <= CONFIG_CPU
void
NAME_(outl)(struct cpssp *cpssp, uint32_t value, uint16_t port)
{
	if (2 <= DEBUG_IO + loglevel) {
		fprintf(stderr, "%s: 0x%04x <- 0x%08x\n", __FUNCTION__,
				port, value);
	}

#if DEBUG_BIOS_POST_CODE
	if (port == 0x0080
	 || port == 0x0300) {
		/* BIOS Post code. */
		fprintf(stderr, "BIOS: Post code 0x%08x.\n", value);
	}
#endif

#if defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	/* 32 Bit Data Bus */
	if (0 < (port & 3)) {
		uint32_t val0;
		uint32_t val1;

		fprintf(stderr, "%s: WARNING: outl->outw port 0x%04x "
			"value 0x%08lx\n", __FUNCTION__, port, (long)value);

		val0 = value << ((port & 3) * 8);
		val1 = value << ((4 - (port & 3)) * 8);

		NAME_(iow)(cpssp, port & ~3, (0xf << (port & 3)) & 0xf, val0);
		NAME_(iow)(cpssp, (port & ~3) + 4, 0xf >> (4 - (port & 3)), val1);

	} else {
		NAME_(iow)(cpssp, port & ~3, 0xf, value);
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	/* 64 Bit Data Bus */
	if (4 < (port & 7)) {
		uint64_t val0;
		uint64_t val1;

		fprintf(stderr, "%s: WARNING: outl->outw port 0x%04x "
			"value 0x%08lx\n", __FUNCTION__, port, (long)value);

		val0 = (uint64_t) value << ((port & 7) * 8);
		val1 = (uint64_t) value << ((8 - (port & 7)) * 8);

		NAME_(iow)(cpssp, port & ~7, (0xf << (port & 7)) & 0xff, val0);
		NAME_(iow)(cpssp, (port & ~7) + 8, 0xf >> (8 - (port & 7)), val1);

	} else {
		uint64_t val64;

		val64 = (uint64_t) value << ((port & 7) * 8);
		NAME_(iow)(cpssp, port & ~7, 0xf << (port & 7), val64);
	}
#else
#error "Unknown socket."
#endif
}
#endif /* 80386 <= CONFIG_CPU */

uint32_t
NAME_(mr_data_b)(struct cpssp *cpssp, Paddr pa)
{
	Data val;

	assert(! (pa & 0));

	NAME_(a20gate_mr)(cpssp, pa & ~(sizeof(Data) - 1),
			0b1 << (pa & (sizeof(Data) - 1)), &val);

	return (val >> ((pa & (sizeof(Data) - 1)) * 8)) & 0xff;
}

uint32_t
NAME_(mr_data_w)(struct cpssp *cpssp, Paddr pa)
{
	Data val;

	assert(! (pa & 1));

	NAME_(a20gate_mr)(cpssp, pa & ~(sizeof(Data) - 1),
			0b11 << (pa & (sizeof(Data) - 1)), &val);

	return (val >> ((pa & (sizeof(Data) - 1)) * 8)) & 0xffff;
}

uint32_t
NAME_(mr_data_l)(struct cpssp *cpssp, Paddr pa)
{
	Data val;

	assert(! (pa & 3));

	NAME_(a20gate_mr)(cpssp, pa & ~(sizeof(Data) - 1),
			0b1111 << (pa & (sizeof(Data) - 1)), &val);

	return (val >> ((pa & (sizeof(Data) - 1)) * 8)) & 0xffffffff;
}

uint64_t
NAME_(mr_data_q)(struct cpssp *cpssp, Paddr pa)
{
	if (sizeof(Data) < sizeof(uint64_t)) {
		uint32_t v1;
		uint32_t v2;

		v1 = NAME_(mr_data_l)(env, pa + 0);
		v2 = NAME_(mr_data_l)(env, pa + 4);

		return v1 | ((uint64_t) v2 << 32);
	} else {
		Data val;

		assert(! (pa & 7));

		NAME_(a20gate_mr)(cpssp, pa & ~(sizeof(Data) - 1),
				0b11111111 << (pa & (sizeof(Data) - 1)), &val);

		return (val >> ((pa & (sizeof(Data) - 1)) * 8)) & 0xffffffffffffffffULL;
	}
}

void
NAME_(mw_data_b)(struct cpssp *cpssp, Paddr pa, uint32_t val)
{
	Data valX;

	assert(! (pa & 0));

	valX = ((Data) val) << ((pa & (sizeof(Data) - 1)) * 8);

	NAME_(a20gate_mw)(cpssp, pa & ~(pa & (sizeof(Data) - 1)),
			0b1 << (pa & (sizeof(Data) - 1)), valX);
}

void
NAME_(mw_data_w)(struct cpssp *cpssp, Paddr pa, uint32_t val)
{
	Data valX;

	assert(! (pa & 1));

	valX = ((Data) val) << ((pa & (sizeof(Data) - 1)) * 8);

	NAME_(a20gate_mw)(cpssp, pa & ~(pa & (sizeof(Data) - 1)),
			0b11 << (pa & (sizeof(Data) - 1)), valX);
}

void
NAME_(mw_data_l)(struct cpssp *cpssp, Paddr pa, uint32_t val)
{
	Data valX;

	assert(! (pa & 3));

	valX = ((Data) val) << ((pa & (sizeof(Data) - 1)) * 8);

	NAME_(a20gate_mw)(cpssp, pa & ~(pa & (sizeof(Data) - 1)),
			0b1111 << (pa & (sizeof(Data) - 1)), valX);
}

void
NAME_(mw_data_q)(struct cpssp *cpssp, Paddr pa, uint64_t val)
{
	if (sizeof(Data) < sizeof(val)) {
		uint32_t v1;
		uint32_t v2;

		v1 = (uint32_t) val;
		v2 = val >> 32;
		NAME_(mw_data_l)(env, pa + 0, v1); 
		NAME_(mw_data_l)(env, pa + 4, v2);
	} else {
		Data valX;

		assert(! (pa & 7));

		valX = ((Data) val) << ((pa & (sizeof(Data) - 1)) * 8);

		NAME_(a20gate_mw)(cpssp, pa & ~(pa & (sizeof(Data) - 1)),
				0b1111111 << (pa & (sizeof(Data) - 1)), valX);
	}
}

uint32_t
NAME_(mx_code_b)(struct cpssp *cpssp, Paddr pa)
{
	Data val;

	assert(! (pa & 0));

	NAME_(a20gate_mx)(cpssp, pa & ~(sizeof(Data) - 1),
			0b1 << (pa & (sizeof(Data) - 1)), &val);

	return (val >> ((pa & (sizeof(Data) - 1)) * 8)) & 0xff;
}

uint32_t
NAME_(mx_code_w)(struct cpssp *cpssp, Paddr pa)
{
	Data val;

	assert(! (pa & 0));

	NAME_(a20gate_mx)(cpssp, pa & ~(sizeof(Data) - 1),
			0b11 << (pa & (sizeof(Data) - 1)), &val);

	return (val >> ((pa & (sizeof(Data) - 1)) * 8)) & 0xffff;
}

uint32_t
NAME_(mx_code_l)(struct cpssp *cpssp, Paddr pa)
{
	Data val;

	assert(! (pa & 0));

	NAME_(a20gate_mx)(cpssp, pa & ~(sizeof(Data) - 1),
			0b1111 << (pa & (sizeof(Data) - 1)), &val);

	return (val >> ((pa & (sizeof(Data) - 1)) * 8)) & 0xffffffff;
}
