/*
 * Copyright (C) 2007-2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_BIOS_POST_CODE	0

#include "config.h"
#include "compiler.h"

#include <assert.h>
#include "fixme.h"
#include <stdio.h>
#include <string.h>

#include "system.h"
#include "glue-shm.h"
#include "glue-suspend.h"

#include "exec-all.h"
#include "arch_gen_lru.h"
#include "arch_gen_cpu_x86_state.h"
#include "arch_gen_cpu_x86_core.h"
#include "arch_gen_cpu_x86_apic.h"
#include "arch_gen_cpu_x86_cache2.h"
#include "arch_gen_cpu_x86_mtrr.h"
#include "arch_gen_cpu_x86_a20gate.h"
#include "arch_gen_cpu_x86_mmui.h"
#include "arch_gen_cpu_x86_mmud.h"
#include "arch_gen_cpu_x86_cache1i.h"
#include "arch_gen_cpu_x86_cache1d.h"
#include "arch_gen_cpu_x86_align.h"
#include "arch_gen_cpu_x86_code.h"
#include "arch_gen_cpu_x86_reg.h"
#include "arch_gen_cpu_x86_seg.h"
#include "qemu/cpu_jit.h"

#define COMP SNAME

#ifdef CONFIG_CPU_SOCKET_ISA
static void
NAME_(ior)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint16_t *valp)
{
	uint8_t val8;

	switch (bs) {
	case 0x1:
		sig_isa_bus_inb(cpssp->port_bus, cpssp, &val8, port + 0);
		*valp = val8 << 0;
		break;
	case 0x2:
		sig_isa_bus_inb(cpssp->port_bus, cpssp, &val8, port + 1);
		*valp = val8 << 8;
		break;
	case 0x3:
		sig_isa_bus_inw(cpssp->port_bus, cpssp, valp, port + 0);
		break;
	}
}

static void
NAME_(iow)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint16_t val)
{
	switch (bs) {
	case 0x1:
		sig_isa_bus_outb(cpssp->port_bus, cpssp, val >> 0, port + 0);
		break;
	case 0x2:
		sig_isa_bus_outb(cpssp->port_bus, cpssp, val >> 8, port + 1);
		break;
	case 0x3:
		sig_isa_bus_outw(cpssp->port_bus, cpssp, val, port + 0);
		break;
	}
}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
static void
NAME_(ior)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint32_t *valp)
{
	*valp = -1;
	if (unlikely(sig_host_bus_ior(cpssp->port_bus, cpssp, port, bs, valp) != 0)) {
		sig_host_bus_addr_type(cpssp->port_bus, cpssp,
				0, port, SIG_HOST_BUS_IOR);
		/* delay... */
		sig_host_bus_read_data(cpssp->port_bus, cpssp,
				bs, valp);
	}
}

static void
NAME_(iow)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_iow(cpssp->port_bus, cpssp, port, bs, val) != 0)) {
		sig_host_bus_addr_type(cpssp->port_bus, cpssp,
				0, port, SIG_HOST_BUS_IOW);
		/* delay... */
		sig_host_bus_write_data(cpssp->port_bus, cpssp,
				bs, val);
	}
}

#elif defined(CONFIG_CPU_SOCKET_775)
static void
NAME_(ior)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint64_t *valp)
{
	*valp = -1;
	if (unlikely(sig_host775_bus_ior(cpssp->port_bus, cpssp, port, bs, valp) != 0)) {
		sig_host775_bus_addr_type(cpssp->port_bus, cpssp,
				0, port, SIG_HOST775_BUS_IOR);
		/* delay... */
		sig_host775_bus_read_data(cpssp->port_bus, cpssp,
				bs, valp);
	}
}

static void
NAME_(iow)(struct cpssp *cpssp, uint16_t port, unsigned int bs, uint64_t val)
{
	if (unlikely(sig_host775_bus_iow(cpssp->port_bus, cpssp, port, bs, val) != 0)) {
		sig_host775_bus_addr_type(cpssp->port_bus, cpssp,
				0, port, SIG_HOST775_BUS_IOW);
		/* delay... */
		sig_host775_bus_write_data(cpssp->port_bus, cpssp,
				bs, val);
	}
}
#else
#error "Unknown socket."
#endif

#if defined(CONFIG_CPU_SOCKET_ISA)
void
NAME_(mr)(struct cpssp *cpssp, Paddr addr, unsigned int bs, uint16_t *valp)
{
	uint8_t val8;

	switch (bs) {
	case 0x1:
		sig_isa_bus_readb(cpssp->port_bus, cpssp, addr + 0, &val8);
		*valp = val8 << 0;
		break;
	case 0x2:
		sig_isa_bus_readb(cpssp->port_bus, cpssp, addr + 1, &val8);
		*valp = val8 << 8;
		break;
	case 0x3:
		sig_isa_bus_readw(cpssp->port_bus, cpssp, addr + 0, valp);
		break;
	}
}

static void
NAME_(mw)(struct cpssp *cpssp, Paddr addr, unsigned int bs, uint16_t val)
{
	switch (bs) {
	case 0x1:
		sig_isa_bus_writeb(cpssp->port_bus, cpssp, addr + 0, val >> 0);
		break;
	case 0x2:
		sig_isa_bus_writeb(cpssp->port_bus, cpssp, addr + 1, val >> 8);
		break;
	case 0x3:
		sig_isa_bus_writew(cpssp->port_bus, cpssp, addr + 0, val >> 0);
		break;
	}
}

static void
NAME_(mx)(struct cpssp *cpssp, Paddr addr, unsigned int bs, uint16_t *valp)
{
	NAME_(mr)(cpssp, addr, bs, valp);
}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
void
NAME_(mr)(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t *valp)
{
	if (unlikely(sig_host_bus_mr(cpssp->port_bus, cpssp,
			cpssp->smm, addr, bs, valp) != 0)) {
		sig_host_bus_addr_type(cpssp->port_bus, cpssp,
				cpssp->smm, addr, SIG_HOST_BUS_MR);
		/* delay */
		sig_host_bus_read_data(cpssp->port_bus, cpssp,
				bs, valp);
	}
}

static void
NAME_(mw)(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t val)
{
	if (unlikely(sig_host_bus_mw(cpssp->port_bus, cpssp,
			cpssp->smm, addr, bs, val) != 0)) {
		sig_host_bus_addr_type(cpssp->port_bus, cpssp,
				cpssp->smm, addr, SIG_HOST_BUS_MW);
		/* delay */
		sig_host_bus_write_data(cpssp->port_bus, cpssp,
				bs, val);
	}
}

static void
NAME_(mx)(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint32_t *valp)
{
	if (unlikely(sig_host_bus_mr(cpssp->port_bus, cpssp,
			cpssp->smm, addr, bs, valp) != 0)) {
		sig_host_bus_addr_type(cpssp->port_bus, cpssp,
				cpssp->smm, addr, SIG_HOST_BUS_MR);
		/* delay */
		sig_host_bus_read_data(cpssp->port_bus, cpssp,
				bs, valp);
	}
}

#elif defined(CONFIG_CPU_SOCKET_775)
void
NAME_(mr)(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint64_t *valp)
{
	if (unlikely(sig_host775_bus_mr(cpssp->port_bus, cpssp,
			cpssp->smm, addr, bs, valp) != 0)) {
		sig_host775_bus_addr_type(cpssp->port_bus, cpssp,
				cpssp->smm, addr, SIG_HOST775_BUS_MR);
		/* delay */
		sig_host775_bus_read_data(cpssp->port_bus, cpssp,
				bs, valp);
	}
}

static void
NAME_(mw)(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint64_t val)
{
	if (unlikely(sig_host775_bus_mw(cpssp->port_bus, cpssp,
			cpssp->smm, addr, bs, val) != 0)) {
		sig_host775_bus_addr_type(cpssp->port_bus, cpssp,
				cpssp->smm, addr, SIG_HOST775_BUS_MW);
		/* delay */
		sig_host775_bus_write_data(cpssp->port_bus, cpssp,
				bs, val);
	}
}

static void
NAME_(mx)(struct cpssp *cpssp, unsigned long addr, unsigned int bs, uint64_t *valp)
{
	if (unlikely(sig_host775_bus_mr(cpssp->port_bus, cpssp,
			cpssp->smm, addr, bs, valp) != 0)) {
		sig_host775_bus_addr_type(cpssp->port_bus, cpssp,
				cpssp->smm, addr, SIG_HOST775_BUS_MR);
		/* delay */
		sig_host775_bus_read_data(cpssp->port_bus, cpssp,
				bs, valp);
	}
}
#else
#error "Unknown socket."
#endif

static int
NAME_(map_r)(struct cpssp *cpssp, Paddr paddr, char **haddr_p)
{
#if defined(CONFIG_CPU_SOCKET_ISA)
	int (*cf)(void *, uint32_t, unsigned int, uint32_t *);
	void *cs;

	/* Try isa bus components. */
	if (! sig_isa_bus_map_r(cpssp->port_bus, cpssp, paddr & ~0xfff,
			&cf, &cs, haddr_p)) {
		if (*haddr_p) {
			*haddr_p += paddr & 0xfff;
		}
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	int (*cf)(void *, uint32_t, unsigned int, uint32_t *);
	void *cs;

	/* Try host bus components. */
	if (! sig_host_bus_map_r_check(cpssp->port_bus, cpssp, cpssp->smm, paddr)
	 && ! sig_host_bus_map_r(cpssp->port_bus, cpssp, cpssp->smm, paddr & ~0xfff,
	 		&cf, &cs, haddr_p)) {
		if (*haddr_p) {
			*haddr_p += paddr & 0xfff;
		}
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	int (*cf)(void *, uint64_t, unsigned int, uint64_t *);
	void *cs;

	/* Try host bus components. */
	if (! sig_host775_bus_map_r_check(cpssp->port_bus, cpssp, cpssp->smm, paddr)
	 && ! sig_host775_bus_map_r(cpssp->port_bus, cpssp, cpssp->smm, paddr & ~0xfff,
	 		&cf, &cs, haddr_p)) {
		if (*haddr_p) {
			*haddr_p += paddr & 0xfff;
		}
		return 0;
	}
#else
#error "Unknown socket."
#endif

	/* Region unknown => not mapped. */
	*haddr_p = NULL;
	return 1;
}

static int
NAME_(map_w)(struct cpssp *cpssp, Paddr paddr, char **haddr_p)
{
#if defined(CONFIG_CPU_SOCKET_ISA)
	int (*cf)(void *, uint32_t, unsigned int, uint32_t);
	void *cs;

	/* Try isa bus components. */
	if (! sig_isa_bus_map_w(cpssp->port_bus, cpssp, paddr & ~0xfff,
			&cf, &cs, haddr_p)) {
		if (*haddr_p) {
			*haddr_p += paddr & 0xfff;
		}
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	int (*cf)(void *, uint32_t, unsigned int, uint32_t);
	void *cs;

	/* Try host bus components. */
	if (! sig_host_bus_map_w_check(cpssp->port_bus, cpssp, cpssp->smm, paddr)
	 && ! sig_host_bus_map_w(cpssp->port_bus, cpssp, cpssp->smm, paddr & ~0xfff,
			&cf, &cs, haddr_p)) {
		if (*haddr_p) {
			*haddr_p += paddr & 0xfff;
		}
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	int (*cf)(void *, uint64_t, unsigned int, uint64_t);
	void *cs;

	/* Try host bus components. */
	if (! sig_host775_bus_map_w_check(cpssp->port_bus, cpssp, cpssp->smm, paddr)
	 && ! sig_host775_bus_map_w(cpssp->port_bus, cpssp, cpssp->smm, paddr & ~0xfff,
			&cf, &cs, haddr_p)) {
		if (*haddr_p) {
			*haddr_p += paddr & 0xfff;
		}
		return 0;
	}
#else
#error "Unknown socket."
#endif

	/* Region unknown => not mapped. */
	*haddr_p = NULL;
	return 1;
}

static int
NAME_(map_x)(struct cpssp *cpssp, Paddr paddr, char **haddr_p)
{
#if defined(CONFIG_CPU_SOCKET_ISA)
	int (*cf)(void *, uint32_t, unsigned int, uint32_t *);
	void *cs;

	/* Try isa bus components. */
	if (! sig_isa_bus_map_r(cpssp->port_bus, cpssp, paddr & ~0xfff,
			&cf, &cs, haddr_p)) {
		if (*haddr_p) {
			*haddr_p += paddr & 0xfff;
		}
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	int (*cf)(void *, uint32_t, unsigned int, uint32_t *);
	void *cs;

	/* Try host bus components. */
	if (! sig_host_bus_map_x_check(cpssp->port_bus, cpssp, cpssp->smm, paddr)
	 && ! sig_host_bus_map_x(cpssp->port_bus, cpssp, cpssp->smm, paddr & ~0xfff,
			&cf, &cs, haddr_p)) {
		if (*haddr_p) {
			*haddr_p += paddr & 0xfff;
		}
		return 0;
	}

#elif defined(CONFIG_CPU_SOCKET_775)
	int (*cf)(void *, uint64_t, unsigned int, uint64_t *);
	void *cs;

	/* Try host bus components. */
	if (! sig_host775_bus_map_x_check(cpssp->port_bus, cpssp, cpssp->smm, paddr)
	 && ! sig_host775_bus_map_x(cpssp->port_bus, cpssp, cpssp->smm, paddr & ~0xfff,
			&cf, &cs, haddr_p)) {
		if (*haddr_p) {
			*haddr_p += paddr & 0xfff;
		}
		return 0;
	}
#else
#error "Unknown socket."
#endif

	/* Region unknown => not mapped. */
	*haddr_p = NULL;
	return 1;
}

static void
NAME_(ack)(struct cpssp *cpssp, uint8_t *vecp)
{
#if defined(CONFIG_CPU_SOCKET_ISA)
	sig_isa_bus_inta_addr(cpssp->port_bus, cpssp);
	sig_isa_bus_inta_data(cpssp->port_bus, cpssp, vecp);

#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	sig_host_bus_inta_addr(cpssp->port_bus, cpssp);
	sig_host_bus_inta_data(cpssp->port_bus, cpssp, vecp);

#elif defined(CONFIG_CPU_SOCKET_775)
	sig_host775_bus_inta_addr(cpssp->port_bus, cpssp);
	sig_host775_bus_inta_data(cpssp->port_bus, cpssp, vecp);
#else
#error "Unknown socket."
#endif
}

#define BEHAVIOR
#include "arch_gen_lru.c"
#include "arch_gen_cpu_x86_apic.c"
#include "arch_gen_cpu_x86_cache2.c"
#include "arch_gen_cpu_x86_mtrr.c"
#include "arch_gen_cpu_x86_a20gate.c"
#include "arch_gen_cpu_x86_mmui.c"
#include "arch_gen_cpu_x86_mmud.c"
#include "arch_gen_cpu_x86_cache1i.c"
#include "arch_gen_cpu_x86_cache1d.c"
#include "arch_gen_cpu_x86_align.c"
#include "arch_gen_cpu_x86_code.c"
#include "arch_gen_cpu_x86_reg.c"
#include "arch_gen_cpu_x86_core.c"
#undef BEHAVIOR

#if 80386 <= CONFIG_CPU
static void
NAME_(a20m_set)(void *_cpssp, unsigned int a20_state)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(a20gate_a20m_set)(cpssp, a20_state);
}
#endif /* 80386 <= CONFIG_CPU */

#if defined(CONFIG_CPU_SOCKET_ISA)
static int
NAME_(map_r_check)(void *_cpssp, uint32_t pa)
{
	// fprintf(stderr, "%s\n", __FUNCTION__);
	return 0; /* Always ok for now. */
}

static int
NAME_(map_w_check)(void *_cpssp, uint32_t pa)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(tb_invalidate_phys_page)(cpssp, pa);

	return 0; /* Always ok for now. */
}

static void
NAME_(unmap)(void *_cpssp, uint32_t pa, uint32_t len)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(cache2_unmap)(cpssp, pa, len);
}

#elif defined(CONFIG_CPU_SOCKET_HOST) \
   || defined(CONFIG_CPU_SOCKET_SLOT1)
static int
NAME_(map_r_check)(void *_cpssp, unsigned int smm, uint32_t pa)
{
	// fprintf(stderr, "%s\n", __FUNCTION__);
	return 0; /* Always ok for now. */
}

static int
NAME_(map_w_check)(void *_cpssp, unsigned int smm, uint32_t pa)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(tb_invalidate_phys_page)(cpssp, pa);

	return 0; /* Always ok for now. */
}

static int
NAME_(map_x_check)(void *_cpssp, unsigned int smm, uint32_t pa)
{
	// fprintf(stderr, "%s\n", __FUNCTION__);
	return 0; /* Always ok for now. */
}

static void
NAME_(unmap)(void *_cpssp, uint32_t pa, uint32_t len)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(cache2_unmap)(cpssp, pa, len);
}

#elif defined(CONFIG_CPU_SOCKET_775)
static int
NAME_(map_r_check)(void *_cpssp, unsigned int smm, uint64_t pa)
{
	return 0; /* Always ok for now. */
}

static int
NAME_(map_w_check)(void *_cpssp, unsigned int smm, uint64_t pa)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(tb_invalidate_phys_page)(cpssp, pa);

	return 0; /* Always ok for now. */
}

static int
NAME_(map_x_check)(void *_cpssp, unsigned int smm, uint64_t pa)
{
	return 0; /* Always ok for now. */
}

static void
NAME_(unmap)(void *_cpssp, uint64_t pa, uint64_t len)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(cache2_unmap)(cpssp, pa, len);
}

#else
#error "Unknown socket."
#endif

static void
NAME_(lint0_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(val == 0 || val == 1);

	NAME_(apic_lint0_set)(cpssp, val);
}

static void
NAME_(lint1_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(val == 0 || val == 1);

	NAME_(apic_lint1_set)(cpssp, val);
}

#if 80386 <= CONFIG_CPU
static void
NAME_(smi_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(val == 0 || val == 1);

	NAME_(apic_smi_set)(cpssp, val);
}
#endif /* 80386 <= CONFIG_CPU */

#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
static void
NAME_(eoi_receive)(void *_cpssp, uint8_t vector)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(apic_eoi_receive)(cpssp, vector);
}

static void
NAME_(msg0_receive)(
	void *_cpssp,
	unsigned int destination_mode,
	unsigned int delivery_mode,
	unsigned int level,
	unsigned int trigger_mode,
	uint8_t vector,
	uint8_t destination
)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(apic_msg0_receive)(cpssp, destination_mode, delivery_mode, level,
			trigger_mode, vector, destination);
}

static void
NAME_(status0_receive)(void *_cpssp, unsigned int status)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(apic_status0_receive)(cpssp, status);
}

static void
NAME_(msg1_receive)(void *_cpssp, uint8_t processor_priority)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(apic_msg1_receive)(cpssp, processor_priority);
}

static void
NAME_(status1_receive)(void *_cpssp, unsigned int status)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(apic_status1_receive)(cpssp, status);
}
#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT */

static void
NAME_(power_set)(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(core_power_set)(cpssp, val);
}

static void
NAME_(n_reset_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(core_n_reset_set)(cpssp, n_val);
}

#if 80386 < CONFIG_CPU
static void
NAME_(n_init_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(core_n_init_set)(cpssp, n_val);
}
#endif /* 80386 <= CONFIG_CPU */

static void
NAME_(n_ignne_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(core_n_ignne_set)(cpssp, n_val);
}

void *
NAME_(create)(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_power,
	struct sig_std_logic *port_n_reset,
#if 80386 < CONFIG_CPU
	struct sig_std_logic *port_n_init,
#endif
	struct sig_std_logic *port_irq,
	struct sig_std_logic *port_nmi,
#if 80386 <= CONFIG_CPU
	struct sig_std_logic *port_smi,
#endif
	struct sig_std_logic *port_n_ferr,
	struct sig_std_logic *port_n_ignne,
#if 80386 <= CONFIG_CPU
	struct sig_std_logic *port_a20,
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	struct sig_icc_bus *port_icc,
#endif
#if defined(CONFIG_CPU_SOCKET_ISA)
	struct sig_isa_bus *port_bus
#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	struct sig_host_bus *port_bus
#elif defined(CONFIG_CPU_SOCKET_775)
	struct sig_host775_bus *port_bus
#else
#error "Unknown socket."
#endif
)
{
	/* APIC */
	static const struct sig_std_logic_funcs lint0_func = {
		.boolean_or_set = NAME_(lint0_set),
	};
	static const struct sig_std_logic_funcs lint1_func = {
		.boolean_or_set = NAME_(lint1_set),
	};
#if 80386 <= CONFIG_CPU
	static const struct sig_std_logic_funcs smi_func = {
		.boolean_or_set = NAME_(smi_set),
	};
#endif /* 80386 <= CONFIG_CPU */
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	static const struct sig_icc_bus_funcs icc_funcs = {
		.eoi = NAME_(eoi_receive),
		.msg0 = NAME_(msg0_receive),
		.status0 = NAME_(status0_receive),
		.msg1 = NAME_(msg1_receive),
		.status1 = NAME_(status1_receive),
	};
#endif /* 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT */

	/* Core */
	static const struct sig_manage_funcs manage_funcs = {
		.connect = NAME_(connect),
		.disconnect = NAME_(disconnect),
	};
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = NAME_(power_set),
	};
	static const struct sig_std_logic_funcs n_reset_funcs = {
		.boolean_or_set = NAME_(n_reset_set),
	};
#if 80386 < CONFIG_CPU
	static const struct sig_std_logic_funcs n_init_funcs = {
		.boolean_or_set = NAME_(n_init_set),
	};
#endif
	static const struct sig_std_logic_funcs n_ignne_funcs = {
		.boolean_or_set = NAME_(n_ignne_set),
	};

#if 80386 <= CONFIG_CPU
	/* MMU */
	static const struct sig_std_logic_funcs a20_func = {
		.boolean_or_set = NAME_(a20m_set),
	};
#endif

	/* I/O */
#if defined(CONFIG_CPU_SOCKET_ISA)
	static const struct sig_isa_bus_funcs main_funcs = {
		.map_r_check = NAME_(map_r_check),
		.map_w_check = NAME_(map_w_check),
		.unmap = NAME_(unmap),
	};
#elif defined(CONFIG_CPU_SOCKET_HOST) \
   || defined(CONFIG_CPU_SOCKET_SLOT1)
	static const struct sig_host_bus_funcs main_funcs = {
		.map_r_check = NAME_(map_r_check),
		.map_w_check = NAME_(map_w_check),
		.map_x_check = NAME_(map_x_check),
		.unmap = NAME_(unmap),
	};
#elif defined(CONFIG_CPU_SOCKET_775)
	static const struct sig_host775_bus_funcs main_funcs = {
		.map_r_check = NAME_(map_r_check),
		.map_w_check = NAME_(map_w_check),
		.map_x_check = NAME_(map_x_check),
		.unmap = NAME_(unmap),
	};
#endif
	static unsigned int nr = 0;
	struct cpssp *cpssp;

	cpssp = shm_palloc(sizeof(*cpssp), SHM_CODE);
	assert(cpssp);

	system_name_push(name);
	cpssp->id = nr++;
	fprintf(stderr, "%s: %d\n", system_path(), nr);

	/* Configuration */
	/* FIXME */
	cpssp->apic_cluster_id = 1;
	cpssp->apic_arbitration_id = cpssp->id;

	cpssp->process.inst_hz = CONFIG_CPU_FREQ;

	sig_manage_connect(port_manage, cpssp, &manage_funcs);

	NAME_(apic_create)(cpssp);

	cpssp->sig_power = port_power;
	cpssp->sig_n_reset = port_n_reset;
#if 80386 < CONFIG_CPU
	cpssp->sig_n_init = port_n_init;
#endif
	cpssp->sig_irq = port_irq;
	cpssp->sig_nmi = port_nmi;
#if 80386 <= CONFIG_CPU
	cpssp->sig_smi = port_smi;
#endif
	cpssp->sig_n_ferr = port_n_ferr;
	cpssp->sig_n_ignne = port_n_ignne;
#if 80386 <= CONFIG_CPU
	cpssp->sig_a20 = port_a20;
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	cpssp->icc_bus = port_icc;
#endif
	cpssp->port_bus = port_bus;

	NAME_(jit_buffer_init)(cpssp);
	NAME_(jit_compile_init)(cpssp);
	NAME_(apic_init)(cpssp);
	NAME_(cache2_create)(cpssp);
	NAME_(a20gate_create)(cpssp);
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT
	NAME_(mtrr_create)(cpssp);
#endif
	NAME_(mmui_create)(cpssp);
	NAME_(mmud_create)(cpssp);
	NAME_(cache1i_create)(cpssp);
	NAME_(cache1d_create)(cpssp);

	/* Output signals. */
	sig_std_logic_connect_out(cpssp->sig_n_ferr, cpssp, SIG_STD_LOGIC_L);

	/* Cycle signals. */
#if defined(CONFIG_CPU_SOCKET_ISA)
	sig_isa_bus_connect(cpssp->port_bus, cpssp, &main_funcs);
#elif defined(CONFIG_CPU_SOCKET_HOST) \
	|| defined(CONFIG_CPU_SOCKET_SLOT1)
	sig_host_bus_connect(cpssp->port_bus, cpssp, &main_funcs);
#elif defined(CONFIG_CPU_SOCKET_775)
	sig_host775_bus_connect(cpssp->port_bus, cpssp, &main_funcs);
#else
#error "Unknown socket."
#endif
#if 80486 <= CONFIG_CPU && CONFIG_CPU_APIC_SUPPORT
	sig_icc_bus_connect(cpssp->icc_bus, cpssp, &icc_funcs);
#endif

	/* Input signals. */
	sig_std_logic_connect_in(cpssp->sig_power, cpssp, &power_funcs);
	sig_std_logic_connect_in(cpssp->sig_n_reset, cpssp, &n_reset_funcs);
#if 80386 < CONFIG_CPU /* FIXME */
	sig_std_logic_connect_in(cpssp->sig_n_init, cpssp, &n_init_funcs);
#endif

	sig_std_logic_connect_in(cpssp->sig_irq, cpssp, &lint0_func);
	sig_std_logic_connect_in(cpssp->sig_nmi, cpssp, &lint1_func);
#if 80386 <= CONFIG_CPU
	sig_std_logic_connect_in(cpssp->sig_smi, cpssp, &smi_func);
#endif

	sig_std_logic_connect_in(cpssp->sig_n_ignne, cpssp, &n_ignne_funcs);

#if 80386 <= CONFIG_CPU
	sig_std_logic_connect_in(cpssp->sig_a20, cpssp, &a20_func);
#endif

	sched_process_init(&cpssp->process, NAME_(step), cpssp);

	system_name_pop();

	return cpssp;
}

void
NAME_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(cache1i_destroy)(cpssp);
	NAME_(cache1d_destroy)(cpssp);
	NAME_(mmui_destroy)(cpssp);
	NAME_(mmud_destroy)(cpssp);
	NAME_(a20gate_destroy)(cpssp);
#if 80486 <= CONFIG_CPU && CONFIG_CPU_MTRR_SUPPORT
	NAME_(mtrr_destroy)(cpssp);
#endif
	NAME_(cache2_destroy)(cpssp);
	NAME_(apic_destroy)(cpssp);

	shm_pfree(cpssp);
}

void
NAME_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
NAME_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fComp);
}

#undef DEBUG_BIOS_POST_CODE
