/* $Id: arch_usb_controller.c,v 1.16 2013-06-11 06:45:26 vrsieh Exp $ 
 *
 * Copyright (C) 2005-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* This is an implementation of the Intel UHCI. */

#define USB_NUM_PORTS	2		/* UHCI root hub has 2 ports */

#define USB_SENDBUFFER_SIZE 0x500
#define USB_NUM_VISITED_QHS 10

#ifdef STATE

struct {
	/*
	 * Config Space
	 */
	uint8_t io_enable;
	uint8_t master_enable;
	uint8_t master_latency_timer;
	uint32_t base; 
	uint8_t interrupt_line;

	/*
	 * I/O Space
	 */
	/* USB Command Register (16 Bit, R/W, WORD writeable only) */
	uint16_t usbcmd;

	/* USB Status (16 Bit, R/WC) */
	uint16_t usbsts;

	/* USB Interrupt Enable (16 Bit, R/W) */
	uint16_t usbintr;

	/* Frame Number (16 Bit, R/W, WORD writeable only, 15:11 reserved) */
	uint16_t frnum;

	/* Frame List Base Address (32 Bit, R/W, 11:0 reserved) */
	uint32_t flbaseadd;

	/* Start Of Frame Modify (8 Bit, R/W, 7 reserved) */
	uint8_t sofmod;

	/* Port Status And Control (16 Bit, R/W, WORD writeable only) */
	uint16_t portsc[USB_NUM_PORTS];

	/* time of the most recent timer event */
	unsigned long long tsc_passed;
	/* time between timer events when USBCMD_RUN_STOP is STOP */
	unsigned long long tsc_step_idle;
	/* time between timer events when USBCMD_RUN_STOP is RUN */
	unsigned long long tsc_step_busy;

	unsigned char global_reset;	/* currently in global reset mode? */

	struct {
		int speed;
		char reset;
	} portstate[USB_NUM_PORTS];

	/* name -- for debugging purposes */
	char name[10];

	/* schedule state */
	char working;	/* currently working on a schedule */

	char timeout_interrupt;	/* trigger interrupt on timeout/crc error */
	char ioc;	/* trigger interrupt on completion at frame end */
	char spd;	/* trigger interrupt on short packet detection at frame end */
	char babble_interrupt, stalled_interrupt, data_buffer_error_interrupt,
	     bit_stuff_error_interrupt, hc_process_error_interrupt,
	     host_system_error_interrupt;	/* non-maskable interrupts */

	/* schedule list traversal state, cf. UHCI, Revision 1.1, chapter 3.4.2 "Transfer Queueing" */
	/* are we in a queue context? */
	char q_context;

	/* Pointer to QH */
	uint32_t qhp;

	/* QH */
	struct {
		uint32_t qhlp;
		char qh_q, qh_t;
		uint32_t qelp;
		char qe_vf, qe_q, qe_t;
	} qh;

	/* Pointer to TD */
	uint32_t tdp;

	/* TD */
	struct {
		uint32_t td[4];	/* content to avoid re-read */

		/* TD link pointer: points to next TD to be executed */
		uint32_t tdlp;
		char td_vf, td_q, td_t;

		unsigned char c_err, actlen, endpt, addr, pid;
		unsigned int maxlen;
	} td;

	unsigned tds_sent;
	
	unsigned char sendbuffer[USB_SENDBUFFER_SIZE];

	/*
	 * count queue headers: bandwidth reclamation leads to the last QH
	 * pointing back in the queue, in effect creating an endless loop; real
	 * hardware terminates the frame if the 1ms time frame is over, we try
	 * to emulate this by counting QHs.
	 *
	 * This is only a fallback if the visited_qhs (see below) mechanism
	 * fails in case the guest OS has some weird QH arrangement.
	 */
	unsigned long qh_count;

	/*
	 * Store the first USB_NUM_VISITED_QHS addresses to QH structures to
	 * avoid looping in memory at all. As we execute multiple TDs at once
	 * regardless of Vf (width/breadth first), the looping mechanism is
	 * useless anyways.
	 */
	uint32_t visited_qhs[USB_NUM_VISITED_QHS];

	/* number of scheduled time_events */
	unsigned int timer_scheduled;
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

#include "std-pci.h"
#include "lib_usb.h" /* FIXME */

/*----------------------------DEBUG-------------------------------------------*/
/* binary OR these in USB_DEBUGMASK */
#define USB_DEBUG_WARNINGS		(1 << 0)
#define USB_DEBUG_CONFSPACE		(1 << 1)
#define USB_DEBUG_IO			(1 << 2)
#define USB_DEBUG_CIM			(1 << 3)
#define USB_DEBUG_DUMP_PACKETS		(1 << 4)
#define USB_DEBUG_FRAME_LIST		(1 << 5)
#define USB_DEBUG_DUMP_TDS		(1 << 6)
#define USB_DEBUG_IRQ			(1 << 7)
#define USB_DEBUG_TIMEOUTS		(1 << 8)

#define USB_DEBUGMASK 0x0000

#define USB_LOG_TYPE	"USB"
/*----------------------------------------------------------------------------*/

/* The following definitions are adopted from the chipset manual */

/* USB Host Controller PCI Configuration Registers */
#define C82371AB_USB_LEGSUP	0xC0	/* 16 bits */
#define C82371AB_USB_MISCSUP	0xFF	/*  8 bits */

/* USB Host Controller IO Space Registers */
#define C82371AB_USB_USBCMD	0x00	/* 16 bits */
#define C82371AB_USB_USBSTS	0x02	/* 16 bits */
#define C82371AB_USB_USBINTR	0x04	/* 16 bits */
#define C82371AB_USB_FRNUM	0x06	/* 16 bits */
#define C82371AB_USB_FLBASEADD	0x08	/* 32 bits */
#define C82371AB_USB_SOFMOD	0x0C	/*  8 bits */
#define C82371AB_USB_PORTSC0	0x10	/* 16 bits */
#define C82371AB_USB_PORTSC1	0x12	/* 16 bits */

/* Possible values for USBCMD */
#define USBCMD_RESERVED			0xFF00
#define USBCMD_MAX_PACKET		(1<<7)
#define USBCMD_CONFIGURE_FLAG		(1<<6)
#define USBCMD_SOFTWARE_DEBUG		(1<<5)
#define USBCMD_FORCE_GLOBAL_RESUME	(1<<4)
#define USBCMD_ENTER_GLOBAL_SUSPEND	(1<<3)
#define USBCMD_GLOBAL_RESET		(1<<2)
#define USBCMD_HOST_CONTROLLER_RESET	(1<<1)
#define USBCMD_RUN_STOP			(1<<0)

/* Possible values for USBSTS */
#define USBSTS_RESERVED			0xFFC0
#define USBSTS_HC_HALTED		(1<<5)
#define USBSTS_HC_PROCESS_ERROR		(1<<4)
#define USBSTS_HOST_SYSTEM_ERROR	(1<<3)
#define USBSTS_RESUME_DETECT		(1<<2)
#define USBSTS_ERROR_INTERRUPT		(1<<1)
#define USBSTS_INTERRUPT		(1<<0)

/* Possible values for USBINTR */
#define USBINTR_RESERVED		0xFFF0
#define USBINTR_SPIE			(1<<3)
#define USBINTR_IOC			(1<<2)
#define USBINTR_RIE			(1<<1)
#define USBINTR_TIE			(1<<0)

/* Possible values for FRNUM */
#define FRNUM_RESERVED			0xF800
#define FRNUM_FRNUM			(~FRNUM_RESERVED)
#define FRNUM_FL_INDEX			0x03FF

/* Possible values for FLBASEADD */
#define FLBASEADD_FLBASEADD		0xFFFFF000UL
#define FLBASEADD_RESERVED		(~FLBASEADD_FLBASEADD)

/* Possible values for SOFMOD */
#define SOFMOD_RESERVED			(1<<7)
#define SOFMOD_SOF_TIMING		0x7F

/* Possible values for PORTSC0/1 */
#define PORTSC_RESERVED_CLEAR		0xE000	/* write as 0 */
#define PORTSC_SUSPEND			(1<<12)	/* R/W */
#define PORTSC_OVERCURRENT_INDICATOR_CHANGE	(1<<11)	/* R/WC */
#define PORTSC_OVERCURRENT_INDICATOR	(1<<10)	/* RO */
#define PORTSC_PORT_RESET		(1<<9)	/* R/W */
#define PORTSC_LOWSPEED_DEVICE_ATTACHED	(1<<8)	/* RO */
#define PORTSC_RESERVED_SET		(1<<7)	/* RO, read as 1 */
#define PORTSC_RESUME_DETECT		(1<<6)	/* R/W */
#define PORTSC_LINE_STATUS_DMINUS	(1<<5)	/* RO */
#define PORTSC_LINE_STATUS_DPLUS	(1<<4)	/* RO */
#define PORTSC_PORT_ENABLE_DISABLE_CHANGE	(1<<3)	/* R/WC */
#define PORTSC_PORT_ENABLE		(1<<2)	/* R/W */
#define PORTSC_CONNECT_STATUS_CHANGE	(1<<1)	/* R/WC */
#define PORTSC_CURRENT_CONNECT_STATUS	(1<<0)	/* RO */

#define SZ_USB_IOSPACE		32	/* I/O space size */

#define USB_TIMER_FREQ_IDLE	10	/* used when USBCMD_RUN_STOP is STOP */
#define USB_TIMER_FREQ_BUSY	100	/* used when USBCMD_RUN_STOP is RUN, should be 1000 */

/* in-memory data structures (cf. UHCI, Rev. 1.1, Section 3 "Data Structures") */
/* Frame List Pointer */
#define USB_MEM_FLP_FLP		0xFFFFFFF0UL	/* Frame List Pointer (FLP) */
#define USB_MEM_FLP_RESERVED	0x0000000CUL	/* write as 0 */
#define USB_MEM_FLP_QH		(1<<1)		/* QH/TD Select (Q) */
#define USB_MEM_FLP_TERMINATE	(1<<0)		/* TERMINATE (T) */

/* Transfer Descriptor */
/* TD Link Pointer */
#define USB_MEM_TD_0_LP		0xFFFFFFF0UL	/* Link Pointer (LP) */
#define USB_MEM_TD_0_RESERVED	(1<<3)		/* write as 0 */
#define USB_MEM_TD_0_VF		(1<<2)		/* Depth/Breadth Select (Vf) */
#define USB_MEM_TD_0_QH		(1<<1)		/* QH/TD Select (Q) */
#define USB_MEM_TD_0_TERMINATE	(1<<0)		/* TERMINATE (T) */
/* TD Control and Status */
#define USB_MEM_TD_1_RESERVED	0xC001F800UL	/* bits 31,30,16,15:11 */
#define USB_MEM_TD_1_SPD	(1<<29)		/* Short Packet Detect (SPD) */
#define USB_MEM_TD_1_C_ERR	(1<<28|1<<27)	/* Error Counter (ERR_C) */
#define USB_MEM_TD_1_C_ERR_SHIFT	27
#define USB_MEM_TD_1_LS		(1<<26)		/* Low Speed Device (LS) */
#define USB_MEM_TD_1_IOS	(1<<25)		/* Isochronous Select (IOS) */
#define USB_MEM_TD_1_IOC	(1<<24)		/* Interrupt on Complete (IOC) */
#define USB_MEM_TD_1_STATUS	0x00FF0000	/* Status */
#define USB_MEM_TD_1_STATUS_ACTIVE		(1<<23)	/* Status Active */
#define USB_MEM_TD_1_STATUS_STALLED		(1<<22)	/* Status Stalled */
#define USB_MEM_TD_1_STATUS_DATA_BUFFER_ERROR	(1<<21)	/* Status Data Buffer Error */
#define USB_MEM_TD_1_STATUS_BABBLE_DETECTED	(1<<20)	/* Status Babble Detected */
#define USB_MEM_TD_1_STATUS_NAK_RECEIVED	(1<<19)	/* Status NAK Received */
#define USB_MEM_TD_1_STATUS_CRC_TIMEOUT_ERROR	(1<<18)	/* Status CRC/Time Out Error */
#define USB_MEM_TD_1_STATUS_BITSTUFF_ERROR	(1<<17)	/* Status Bitstuff Error */
#define USB_MEM_TD_1_ACTLEN		0x000007FF	/* Actual Length (ActLen) */
/* TD Token */
#define USB_MEM_TD_2_MAXLEN		0xFFE00000	/* Maximum Length (MaxLen) (31:21) */
#define USB_MEM_TD_2_MAXLEN_SHIFT	21
#define USB_MEM_TD_2_MAXLEN_WRAP	0x7FF		/* wrap values at this maximum value */
#define USB_MEM_TD_2_MAXLEN_ILLEGAL	0x500		/* everything >= 0x500 is illegal */
#define USB_MEM_TD_2_RESERVED	(1<<20)		/* Reserved */
#define USB_MEM_TD_2_D		(1<<19)		/* Data Toggle (D) */
#define USB_MEM_TD_2_ENDPT	0x00078000UL	/* Endpoint (EndPt) (18:15) */
#define USB_MEM_TD_2_ENDPT_SHIFT	15
#define USB_MEM_TD_2_DEV_ADDR	0x00007F00UL	/* Device Address (14:8) */
#define USB_MEM_TD_2_DEV_ADDR_SHIFT	8
#define USB_MEM_TD_2_PID	0x000000FFUL	/* Packet Identification (PID) (7:0) */
/* TD Buffer Pointer -- extends over whole DWORD 3 */
/* DWORDS 4-7 are reserved for use by software */

/* Queue Head (QH) */
/* Queue Head Link Pointer */
#define USB_MEM_QH_0_QHLP	0xFFFFFFF0UL	/* Queue Head Link Pointer (QHLP) (31:4) */
#define USB_MEM_QH_0_RESERVED	0x0000000CUL	/* write as 0 (3:2) */
#define USB_MEM_QH_0_QH		(1<<1)		/* QH/TD Select (Q) */
#define USB_MEM_QH_0_TERMINATE	(1<<0)		/* TERMINATE (T) */
/* Queue Element Link Pointer */
#define USB_MEM_QH_1_QELP	0xFFFFFFF0UL	/* Queue Element Link Pointer (QELP) (31:4) */
#define USB_MEM_QH_1_RESERVED	(1<<3)		/* write as 0 */
#define USB_MEM_QH_1_VF		(1<<2)		/* reserved for copied TD VF */
#define USB_MEM_QH_1_QH		(1<<1)		/* QH/TD Select (Q) */
#define USB_MEM_QH_1_TERMINATE	(1<<0)		/* TERMINATE (T) */

/* parameter 2 of NAME_(advance_queue)() */
#define USB_QUEUE_NO_ADVANCE	0
#define USB_QUEUE_ADVANCE	1

/* see arch_usb_controller.h -> qh_count */
#define USB_MAX_QH_PER_FRAME	30

#define USB_MAX_TDS_AT_ONCE	1

/*
 * how many frames to skip in case no TD was successful in one frame
 */
#define USB_FRAME_HOP_COUNT	10


/* forward declarations */
static void
NAME_(connect_device)(struct cpssp *cpssp, int usb_portnr);
static void 
NAME_(packet_timeout)(struct cpssp *cpssp);
static void
NAME_(timer_event)(void *datap);

static void
NAME_(set_irq)(struct cpssp *cpssp, int val)
{
#if USB_DEBUGMASK & USB_DEBUG_IRQ
	if (val) {
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: IRQ line 1, USBSTS is %02X\n",
			__FUNCTION__, cpssp->NAME.usbsts);
	} else {
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: IRQ line 0\n", __FUNCTION__);
	}
#endif
	NAME_(irqrq_out_set)(cpssp, val);
}

static void
NAME_(soft_reset)(struct cpssp *cpssp)
{
	int i;

	/* Initialize I/O Space Registers */
	cpssp->NAME.usbcmd		= 0x0;
	cpssp->NAME.usbsts		= 0x0;
	cpssp->NAME.usbintr		= 0x0;
	cpssp->NAME.frnum			= 0x0;
	cpssp->NAME.flbaseadd		= 0x0;
	cpssp->NAME.sofmod		= 0x40;
	for (i = 0; i < USB_NUM_PORTS; i++) {
		cpssp->NAME.portsc[i]	= 0x80;
	}

	cpssp->NAME.global_reset = 0;
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* Initialize Config Space Registers */
	cpssp->NAME.io_enable = 0;
	cpssp->NAME.master_enable = 0;
	cpssp->NAME.master_latency_timer = 0x00;
	cpssp->NAME.base = 0x00000000;
	cpssp->NAME.interrupt_line = 0;

	NAME_(soft_reset)(cpssp);
}

/*----------------------------I/O interface-----------------------------------*/
static void
NAME_(_inw_core)(struct cpssp *cpssp, uint16_t port, uint16_t *valp)
{
	switch (port) {
	case C82371AB_USB_USBCMD:	/* USB Command */
		*valp = cpssp->NAME.usbcmd;
		break;

	case C82371AB_USB_USBSTS:	/* USB Status */
		*valp = cpssp->NAME.usbsts;
		break;

	case C82371AB_USB_USBINTR:	/* USB Interrupt Enable */
		*valp = cpssp->NAME.usbintr;
		break;

	case C82371AB_USB_FRNUM:	/* Frame Number */
		*valp = cpssp->NAME.frnum;
		break;

	case C82371AB_USB_FLBASEADD:	/* Frame List Base Address (Low) */
		*valp = (cpssp->NAME.flbaseadd >> 0) & 0x0000ffff;
		break;

	case C82371AB_USB_FLBASEADD + 2:/* Frame List Base Address (High) */
		*valp = (cpssp->NAME.flbaseadd >> 16) & 0x0000ffff;
		break;

	case C82371AB_USB_SOFMOD:	/* Start Of Frame Modify */
		*valp = cpssp->NAME.sofmod;
		break;

	case C82371AB_USB_PORTSC0:	/* USB Port Status And Control */
	case C82371AB_USB_PORTSC1:
		*valp = cpssp->NAME.portsc[(port - C82371AB_USB_PORTSC0) >> 1];
		break;

	case C82371AB_USB_PORTSC1 + 2:	/* non-existent port 2 */
		*valp = 0xff7f;	/* bit 7 = 0 makes linux kernel stop
					 * scanning for more ports */
		break;

	default:
		/*
		 * Reading from unknown I/O address
		 */
		*valp = 0xff7f;
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
			"reading from I/O offset 0x%02x\n", port);
#endif
		break;
	}
}

static void
NAME_(_outb_core)(struct cpssp *cpssp, uint16_t port, uint8_t val)
{
	switch (port) {
	case C82371AB_USB_USBCMD:	/* USB Command (R/W), WORD writeable only */
	case C82371AB_USB_USBCMD + 1:
		goto word_writeable_only;

	case C82371AB_USB_USBSTS:	/* USB Status (R/WC) (Bits 0-7) */
		/* FIXME sihoschi: Bits 15-6 are reserved */
		cpssp->NAME.usbsts &= ~val;

		/* FIXME: Linux Kernel sources tell this:
		 * 'Contrary to the UHCI specification, the "HC Halted" status
		 * bit is persistent: it is RO, not R/WC.'
		 */

		/* are all interrupt reasons cleared? */
		if ((cpssp->NAME.usbsts & 0x1F) == 0) {
			NAME_(set_irq)(cpssp, 0);
		}
		break;

	case C82371AB_USB_USBSTS + 1:	/* USB Status (R/WC) (Bits 8-15) */
		/* reserved, nothing to do */
		break;

	case C82371AB_USB_USBINTR:	/* USB Interrupt Enable (Bits 0-7) */
		/* FIXME sihoschi: Bits 15-4 are reserved */
		cpssp->NAME.usbintr = val;
		break;

	case C82371AB_USB_USBINTR + 1:	/* USB Interrupt Enable (Bits 8-15) */
		/* reserved, nothing to do */
		break;

	case C82371AB_USB_FRNUM:	/* Frame Number (Bits 0-7) */
	case C82371AB_USB_FRNUM + 1:	/* Frame Number (Bits 8-15) */
		goto word_writeable_only;

	case C82371AB_USB_FLBASEADD:	/* Frame List Base Address (Bits 0-7) */
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		if (val != 0x00) {
			faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
				"%s: port=0x%02x val=0x%02x "
				"although reserved and zero\n",
				__FUNCTION__, port, val);
		}
#endif
		val = 0x0;
		cpssp->NAME.flbaseadd &= 0xFFFFFF00UL;
		cpssp->NAME.flbaseadd |= val;
		break;

	case C82371AB_USB_FLBASEADD + 1:/* Frame List Base Address (Bits 8-15) */
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		if (val & 0x07) {
			faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
				"%s: port=0x%02x val=0x%02x "
				"although reserved and zero\n",
				__FUNCTION__, port, val);
		}
#endif
		val &= ~0x07;
		cpssp->NAME.flbaseadd &= 0xFFFF00FFUL;
		cpssp->NAME.flbaseadd |= val << 8;
		break;

	case C82371AB_USB_FLBASEADD + 2:/* Frame List Base Address (Bits 16-23) */
		cpssp->NAME.flbaseadd &= 0xFF00FFFFUL;
		cpssp->NAME.flbaseadd |= val << 16;
		break;

	case C82371AB_USB_FLBASEADD + 3:/* Frame List Base Address (Bits 24-31) */
		cpssp->NAME.flbaseadd &= 0x00FFFFFFUL;
		cpssp->NAME.flbaseadd |= val << 24;
		break;

	case C82371AB_USB_SOFMOD:	/* Start Of Frame Modify */
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		if (val & SOFMOD_RESERVED) {
			faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
				"%s: port=0x%02x val=0x%02x "
				"although reserved and zero\n",
				__FUNCTION__, port, val);
		}
#endif
		val &= SOFMOD_SOF_TIMING;
		cpssp->NAME.sofmod = val;
		break;

	case C82371AB_USB_PORTSC0:	/* USB Port Status And Control */
	case C82371AB_USB_PORTSC0 + 1:
	case C82371AB_USB_PORTSC1:
	case C82371AB_USB_PORTSC1 + 1:
		goto word_writeable_only;

	word_writeable_only:
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
			"writing to I/O offset 0x%02x (word writeable only)\n",
			port);
#endif
		break;

	default:
		/*
		 * Writing to unknown I/O address
		 */
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
			"writing to I/O offset 0x%02x\n", port);
#endif
		break;
	}
}

static void
NAME_(_outw_core)(struct cpssp *cpssp, uint16_t port, uint16_t val)
{
	uint16_t old_val;
	uint16_t mask;
	unsigned int usb_portnr;

	switch (port) {
	case C82371AB_USB_USBCMD:	/* USB Command (R/W), WORD writeable only */
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		if (val & USBCMD_RESERVED) {
			faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
				"%s: port=0x%02x val=0x%04x "
				"although reserved and zero\n",
				__FUNCTION__, port, val);
		}
#endif
		cpssp->NAME.usbcmd = val;

		/* HCReset */
		if (val & USBCMD_HOST_CONTROLLER_RESET) {
			/* this implicitly clears the HCReset bit */
			NAME_(soft_reset)(cpssp);

			/* TODO: terminate transactions in progress? */

			/* connect status change  and  port enable/disable
			 * change  of every port is set now */
			for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
				cpssp->NAME.portsc[usb_portnr] |=
					  PORTSC_PORT_ENABLE_DISABLE_CHANGE
					| PORTSC_CONNECT_STATUS_CHANGE;
				/* FIXME: reconnect devices "after 64 bit-times"? */
				NAME_(connect_device)(cpssp, usb_portnr);
			}
		}

		/* GReset... */
		if (val & USBCMD_GLOBAL_RESET) {		/* ... was SET */
			cpssp->NAME.global_reset = 1;
		} else if (cpssp->NAME.global_reset) {	/* ... was CLEARED */
			cpssp->NAME.global_reset = 0;
			/* reset after the guaranteed 10ms minimum global_reset delay */
			NAME_(soft_reset)(cpssp);

			/* reset everyone downstream */
			for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
				if (usb_portnr == 0) {
					NAME_(0_reset_set)(cpssp, 1);
				} else { assert(usb_portnr == 1);
					NAME_(1_reset_set)(cpssp, 1);
				}
				NAME_(connect_device)(cpssp, usb_portnr);
			}
		}

		/* Run/Stop */
		if (val & USBCMD_RUN_STOP) {			/* run */
			/* clear corresponding status bit */
			cpssp->NAME.usbsts &= ~USBSTS_HC_HALTED;
			/* TODO: maybe immediately start a new frame?
			 * problem: locking... */
		} else {					/* stop */
			/* set corresponding status bit */
			cpssp->NAME.usbsts |= USBSTS_HC_HALTED;

			if (0 < cpssp->NAME.tds_sent) {
				NAME_(packet_timeout)(cpssp);
				cpssp->NAME.tds_sent = 0;
			}
		}

#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		/* FIXME: SWDEBUG completely ignored at the moment */
		if (val & USBCMD_SOFTWARE_DEBUG) {
			faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE,
				SNAME,
				"unimplemented USBCMD bit set: %04x\n",
				val);
		}
#endif
		break;

	case C82371AB_USB_FRNUM:	/* Frame Number */
		/* "This register cannot be written unless the Host Controller
		 * is in the STOPPED state as indicated by the HCHalted bit
		 * (USBSTS register). A write to this register while the
		 * Run/Stop bit is set (USBCMD register) is ignored." */
		if (! (cpssp->NAME.usbcmd & USBCMD_RUN_STOP)
		 && cpssp->NAME.usbsts & USBSTS_HC_HALTED) {
			cpssp->NAME.frnum = val & FRNUM_FRNUM;
		} else {
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
			faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE,
				SNAME,
				"%s: port=0x%02x (FRNUM) val=0x%04x "
				"but currently not writeable\n",
				__FUNCTION__, port, val);
#endif
		}
		break;

	case C82371AB_USB_PORTSC0:	/* USB Port Status And Control */
	case C82371AB_USB_PORTSC1:
		usb_portnr = (port - C82371AB_USB_PORTSC0) >> 1;
		old_val = cpssp->NAME.portsc[usb_portnr];

		/* R/W bits */
		mask = PORTSC_SUSPEND | PORTSC_PORT_RESET
		     | PORTSC_RESUME_DETECT | PORTSC_PORT_ENABLE;
		old_val &= ~mask;
		old_val |= val & mask;

		/* R/WC bits */
		mask = PORTSC_OVERCURRENT_INDICATOR_CHANGE
		     | PORTSC_PORT_ENABLE_DISABLE_CHANGE
		     | PORTSC_CONNECT_STATUS_CHANGE;
		old_val &= ~(val & mask);

		cpssp->NAME.portsc[usb_portnr] = old_val;

		/* PORT_RESET... */
		/* FIXME: docs are ambiguous here? */
		if (old_val & PORTSC_PORT_RESET) { /* ... was SET */
			cpssp->NAME.portstate[usb_portnr].reset = 1;
			cpssp->NAME.portsc[usb_portnr] &=
				~(PORTSC_LOWSPEED_DEVICE_ATTACHED
				| PORTSC_LINE_STATUS_DMINUS
				| PORTSC_LINE_STATUS_DPLUS
				| PORTSC_PORT_ENABLE_DISABLE_CHANGE
				| PORTSC_PORT_ENABLE
				| PORTSC_CONNECT_STATUS_CHANGE
				| PORTSC_CURRENT_CONNECT_STATUS);
		} else if (cpssp->NAME.portstate[usb_portnr].reset) { /* ... was CLEARED */
			cpssp->NAME.portstate[usb_portnr].reset = 0;

			/* reset everyone on this port */
			if (usb_portnr == 0) {
				NAME_(0_reset_set)(cpssp, 1);
			} else { assert(usb_portnr == 1);
				NAME_(1_reset_set)(cpssp, 1);
			}
			NAME_(connect_device)(cpssp, usb_portnr);
		}
		break;

	default:
		NAME_(_outb_core)(cpssp, port + 0, (val >> 0) & 0x00ff);
		NAME_(_outb_core)(cpssp, port + 1, (val >> 8) & 0x00ff);
		break;
	}
}

static int
NAME_(ior)(struct cpssp *cpssp, uint32_t port, unsigned int bs, uint32_t *valp)
{
	uint16_t val16;

	if (cpssp->NAME.io_enable
	 && cpssp->NAME.base == (port & ~0x1f)) {
		port &= 0x1f;

		*valp = 0;
		if (((bs >> 0) & 1)
		 || ((bs >> 1) & 1)) {
			NAME_(_inw_core)(cpssp, port + 0, &val16);
			*valp |= val16 << 0;
		}
		if (((bs >> 2) & 1)
		 || ((bs >> 3) & 1)) {
			NAME_(_inw_core)(cpssp, port + 2, &val16);
			*valp |= val16 << 16;
		}
		return 0;
	}
	return 1;
}

static int
NAME_(iow)(struct cpssp *cpssp, uint32_t port, unsigned int bs, uint32_t val)
{
	if (cpssp->NAME.io_enable
	 && cpssp->NAME.base == (port & ~0x1f)) {
		port &= 0x1f;

		/* FIXME */
		switch (bs) {
		case 0x1:
			NAME_(_outb_core)(cpssp, port + 0, val >> 0);
			break;
		case 0x2:
			NAME_(_outb_core)(cpssp, port + 1, val >> 8);
			break;
		case 0x4:
			NAME_(_outb_core)(cpssp, port + 2, val >> 16);
			break;
		case 0x8:
			NAME_(_outb_core)(cpssp, port + 3, val >> 24);
			break;
		case 0x3:
			NAME_(_outw_core)(cpssp, port + 0, val >> 0);
			break;
		case 0xc:
			NAME_(_outw_core)(cpssp, port + 2, val >> 16);
			break;
		case 0xf:
			NAME_(_outw_core)(cpssp, port + 0, val >> 0);
			NAME_(_outw_core)(cpssp, port + 2, val >> 16);
			break;
		default:
			assert(0);
		}
		return 0;
	}
	return 1;
}

static int
NAME_(ior_info)(
	struct cpssp *cpssp,
	uint32_t port,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp
)
{
	return 1; /* FIXME */
}

static int
NAME_(iow_info)(
	struct cpssp *cpssp,
	uint32_t port,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp
)
{
	return 1; /* FIXME */
}

/* ---------------------Configuration Interface (PCI bus)---------------*/

static void
NAME_(c0r)(struct cpssp *cpssp, uint8_t addr, unsigned int bs, uint32_t *valp)
{
	*valp = 0x00000000;
	switch (addr) {
	case 0x00:
		/* Vendor Identification Register */
		/* 6.1.1 */
		*valp |= 0x8086 << 0;

		/* Device Identification Register */
		/* 6.1.2 */
		*valp |= DEVICE_ID << 16;
		break;

	case 0x04:
		/* PCI Command Register */
		/* 6.1.3 */
		*valp |= cpssp->NAME.io_enable << 0;
		*valp |= 0b0 << 1; /* Memory Space Enable */
		*valp |= cpssp->NAME.master_enable << 2;
		*valp |= 0b0 << 3; /* Special Cycle Enable */
		*valp |= 0b0 << 4; /* Memory Write and Invalidate Enable */
		*valp |= 0b0000 << 5; /* Reserved */
		*valp |= 0b0 << 9; /* Fast Back to Back Enable */
		*valp |= 0b000000 << 10; /* Reserved */

		/* PCI Status Register */
		/* 6.1.4 */
		*valp |= 0b0000000 << (0 + 16); /* Reserved */
		*valp |= 0b1 << (7 + 16); /* Fast Back to Back Capable */
		*valp |= 0b0 << (8 + 16); /* Data Parity Detected */
		*valp |= 0b01 << (9 + 16); /* DEVSEL Timing Status */
		*valp |= 0b0 << (11 + 16); /* Signaled Target-Abort Status */
		*valp |= 0b0 << (12 + 16); /* Received Target-Abort Status */
		*valp |= 0b0 << (13 + 16); /* Master-Abort Status */
		*valp |= 0b0 << (14 + 16); /* SERR# Status */
		*valp |= 0b0 << (15 + 16); /* Detected Parity */
		break;

	case 0x08:
		/* Revision Identification Register */
		/* 6.1.5 */
		*valp |= REVISION_ID << 0;

		/* Class Code Register */
		/* 6.1.6 */
		*valp |= 0x00 << (0 + 8); /* UHCI */
		*valp |= 0x03 << (8 + 8); /* USB Controller */
		*valp |= 0x0c << (16 + 8); /* Serial Bus Controller */
		break;

	case 0x0c:
		*valp |= 0x00 << 0; /* Reserved */

		/* Master Latency Timer Register */
		/* 6.1.7 */
		*valp |= cpssp->NAME.master_latency_timer << 8;

		/* Header Type Register */
		/* 6.1.8 */
		*valp |= 0b0000000 << (0 + 16);
		*valp |= MULTIFUNC << (7 + 16);

		*valp |= 0x00 << 24; /* Reserved */
		break;

	case 0x10:
	case 0x14:
	case 0x18:
	case 0x1c:
		goto reserved;

	case 0x20:
		/* USB I/O Space Base Address Register */
		/* 6.1.13 */
		*valp |= 0b1 << 0;
		*valp |= 0b0000 << 1; /* Reserved */
		*valp |= cpssp->NAME.base << (5-5);

		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x24:
	case 0x28:
	case 0x2c:
	case 0x30:
	case 0x34:
	case 0x38:
		goto reserved;

	case 0x3c:
		/* Interrupt Line Register */
		/* 6.1.9 */
		*valp |= cpssp->NAME.interrupt_line << 0;
		
		/* Interrupt Pin Register */
		/* 6.1.10 */
		*valp |= INTERRUPT_PIN << 8;

		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0x40:
	case 0x44:
	case 0x48:
	case 0x4c:
	case 0x50:
	case 0x54:
	case 0x58:
	case 0x5c:
		goto reserved;

	case 0x60:
		/* Serial Bus Release Number Register */
		/* 6.1.11 */
		*valp |= 0x10 << 0; /* Version 1.0 */

		*valp |= 0x000000 << 8; /* Reserved */
		break;

	case 0x64: case 0x68: case 0x6c:
	case 0x70: case 0x74: case 0x78: case 0x7c:
	case 0x80: case 0x84: case 0x88: case 0x8c:
	case 0x90: case 0x94: case 0x98: case 0x9c:
	case 0xa0: case 0xa4: case 0xa8: case 0xac:
	case 0xb0: case 0xb4: case 0xb8: case 0xbc:
		goto reserved;

	case 0xc0:
		/* Legacy Support Register */
		/* 6.1.12 */
		/* FIXME */
		fprintf(stderr, "WARNING: %s: reading legacy support register.\n",
				__FUNCTION__);

		*valp |= 0x0000 << 16; /* Reserved */
		break;

	case 0xc4: case 0xc8: case 0xcc:
	case 0xd0: case 0xd4: case 0xd8: case 0xdc:
	case 0xe0: case 0xe4: case 0xe8: case 0xec:
	case 0xf0: case 0xf4: case 0xf8: case 0xfc:
		/* MISCSUP Register must be handled elsewhere! */
		goto reserved;

	reserved:
		*valp |= 0x00000000 << 0; /* Reserved */
		break;

	default:
		assert(0); /* Cannot happen. */
	}
}

static void
NAME_(c0w)(struct cpssp *cpssp, uint8_t addr, unsigned int bs, uint32_t val)
{
	switch (addr) {
	case 0x00:
		/* Vendor Identification Register */
		/* 6.1.1 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		/* Device Identification Register */
		/* 6.1.2 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x04:
		/* PCI Command Register */
		/* 6.1.3 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.io_enable = (val >> 0) & 1;
			/* Bit 1: Hardwired to '0'. */
			cpssp->NAME.master_enable = (val >> 2) & 1;
			/* Bit 3-7: Hardwired to '0'. */
		}
		if ((bs >> 1) & 1) {
			/* Hardwired to '0'. */
		}

		/* PCI Status Register */
		/* 6.1.4 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x08:
		/* Revision Identification Register */
		/* 6.1.5 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}

		/* Class Code Register */
		/* 6.1.6 */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		break;

	case 0x0c:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}

		/* Master Latency Timer Register */
		/* 6.1.7 */
		if ((bs >> 1) & 1) {
			cpssp->NAME.master_latency_timer = (val >> 8) & 0xf0;
		}

		/* Header Type Register */
		/* 6.1.8 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}

		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x10:
	case 0x14:
	case 0x18:
	case 0x1c:
		goto reserved;

	case 0x20:
		/* USB I/O Space Base Address Register */
		/* 6.1.13 */
		if ((bs >> 0) & 1) {
			/* Bit 0-4: Read-only */
			cpssp->NAME.base &= ~(0xe0 << 0);
			cpssp->NAME.base |= val & (0xe0 << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.base &= ~(0xff << 8);
			cpssp->NAME.base |= val & (0xff << 8);
		}

		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x24:
	case 0x28:
	case 0x2c:
	case 0x30:
	case 0x34:
	case 0x38:
		goto reserved;

	case 0x3c:
		/* Interrupt Line Register */
		/* 6.1.9 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.interrupt_line = (val >> 0) & 0xff;
		}
		
		/* Interrupt Pin Register */
		/* 6.1.10 */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x40: case 0x44: case 0x48: case 0x4c:
	case 0x50: case 0x54: case 0x58: case 0x5c:
		goto reserved;

	case 0x60:
		/* Serial Bus Release Number Register */
		/* 6.1.11 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}

		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0x64: case 0x68: case 0x6c:
	case 0x70: case 0x74: case 0x78: case 0x7c:
	case 0x80: case 0x84: case 0x88: case 0x8c:
	case 0x90: case 0x94: case 0x98: case 0x9c:
	case 0xa0: case 0xa4: case 0xa8: case 0xac:
	case 0xb0: case 0xb4: case 0xb8: case 0xbc:
		goto reserved;

	case 0xc0:
		/* Legacy Support Register */
		/* 6.1.12 */
		fprintf(stderr, "WARNING: %s: writing legacy support register.\n",
				__FUNCTION__);
		if ((bs >> 0) & 1) {
			/* FIXME */
		}
		if ((bs >> 1) & 1) {
			/* FIXME */
		}

		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	case 0xc4: case 0xc8: case 0xcc:
	case 0xd0: case 0xd4: case 0xd8: case 0xdc:
	case 0xe0: case 0xe4: case 0xe8: case 0xec:
	case 0xf0: case 0xf4: case 0xf8: case 0xfc:
		/* MISCSUP Register must be handled elsewhere! */
		goto reserved;

	reserved:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		break;

	default:
		assert(0); /* Cannot happen. */
	}
}

/*----------------------------Auxiliary Functions-----------------------------*/

/* set I/O registers accoording to device attached or not */
static void
NAME_(connect_device)(struct cpssp *cpssp, int usb_portnr)
{
	int current_connect_status = cpssp->NAME.portsc[usb_portnr] & PORTSC_CURRENT_CONNECT_STATUS;
	char signal_global_resume = 0;

	if (cpssp->NAME.portstate[usb_portnr].speed != USB_SPEED_UNCONNECTED) {
		if (! current_connect_status) {
			cpssp->NAME.portsc[usb_portnr] |=
				  PORTSC_CONNECT_STATUS_CHANGE
				| PORTSC_CURRENT_CONNECT_STATUS;
			signal_global_resume = 1;
		}

		if (cpssp->NAME.portstate[usb_portnr].speed == USB_SPEED_FULL) { /* full speed device */
			/* cf. USB Spec. 1.1, chapter 7.1.5, "Device Speed Identification"  */
			cpssp->NAME.portsc[usb_portnr] |=
				  PORTSC_LINE_STATUS_DPLUS;
			cpssp->NAME.portsc[usb_portnr] &=
				~(PORTSC_LINE_STATUS_DMINUS
				| PORTSC_LOWSPEED_DEVICE_ATTACHED);
		} else { /* low speed device */
			cpssp->NAME.portsc[usb_portnr] |=
				  PORTSC_LOWSPEED_DEVICE_ATTACHED
				| PORTSC_LINE_STATUS_DMINUS;
			cpssp->NAME.portsc[usb_portnr] &=
				~PORTSC_LINE_STATUS_DPLUS;
		}
	} else { /* no device attached */
		if (current_connect_status) {
			cpssp->NAME.portsc[usb_portnr] |=
				PORTSC_CONNECT_STATUS_CHANGE;
			signal_global_resume = 1;
		}
		cpssp->NAME.portsc[usb_portnr] &=
			~(PORTSC_LOWSPEED_DEVICE_ATTACHED
			| PORTSC_LINE_STATUS_DMINUS
			| PORTSC_LINE_STATUS_DPLUS
			| PORTSC_PORT_ENABLE_DISABLE_CHANGE
			| PORTSC_PORT_ENABLE
			| PORTSC_CURRENT_CONNECT_STATUS);
	}

	/* if Resume Interrupt Enabled and in Global Suspend Mode
	 * and bits 1, 3 or 6 of PORTSC were set, signal interrupt */
	if (cpssp->NAME.usbcmd & USBCMD_ENTER_GLOBAL_SUSPEND
	 && signal_global_resume) {
		cpssp->NAME.usbcmd |= USBCMD_FORCE_GLOBAL_RESUME;
		cpssp->NAME.usbsts |= USBSTS_RESUME_DETECT;

		if (cpssp->NAME.usbintr & USBINTR_RIE) {
			NAME_(set_irq)(cpssp, 1);
		}
	}
}

/* send a USB token packet over CIM */
static void
NAME_(send_usb_token)(
	struct cpssp *cpssp,
	unsigned char pid,
	unsigned char addr,
	unsigned char endp
)
{
	int usb_portnr;

#if USB_DEBUGMASK & USB_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: sending %s packet to %02d:%02d\n",
		__FUNCTION__, cim_usb_debug_pid2str(pid), addr, endp);
#endif

	for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
		/* port disabled or in selective suspend mode? */
		if (! (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)
		 || (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE
		  && cpssp->NAME.portsc[usb_portnr] & PORTSC_SUSPEND)) {
			continue;
		}

		if (usb_portnr == 0) {
			NAME_(0_send_token)(cpssp, pid, addr, endp);
		} else { assert(usb_portnr == 1);
			NAME_(1_send_token)(cpssp, pid, addr, endp);
		}
	}
}

/* send a USB start-of-frame packet over CIM */
static __attribute__ ((unused)) void
NAME_(send_usb_sof)(
	struct cpssp *cpssp,
	unsigned short frame_num
)
{
	int usb_portnr;

#if USB_DEBUGMASK & USB_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: sending %s packet (frame %04d)\n",
		__FUNCTION__, cim_usb_debug_pid2str(USB_PID_SOF), frame_num);
#endif

	for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
		/* port disabled or in selective suspend mode? */
		if (!(cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)
		 || (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE
		  && cpssp->NAME.portsc[usb_portnr] & PORTSC_SUSPEND)) {
			continue;
		}

		if (usb_portnr == 0) {
			NAME_(0_send_sof)(cpssp, frame_num);
		} else { assert(usb_portnr == 1);
			NAME_(1_send_sof)(cpssp, frame_num);
		}
	}
}

/* send a USB data packet over CIM */
static void
NAME_(send_usb_data)(
	struct cpssp *cpssp,
	unsigned char pid,
	unsigned char *data,
	unsigned int length
)
{
	int usb_portnr;

#if USB_DEBUGMASK & USB_DEBUG_DUMP_PACKETS
	int i;

	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: sending %s packet (length %d)",
		__FUNCTION__, cim_usb_debug_pid2str(pid), length);

	if (0 < length) {
		faum_cont(FAUM_LOG_DEBUG, " ( ");

		for (i = 0; i < length && i < 8; i++) {
			faum_cont(FAUM_LOG_DEBUG, "%02X ", data[i]);
		}

		if (i != length) {
			faum_cont(FAUM_LOG_DEBUG, "... ");
		}

		faum_cont(FAUM_LOG_DEBUG, ")");
	}

	faum_cont(FAUM_LOG_DEBUG, "\n");
#endif

	for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
		/* port disabled or in selective suspend mode? */
		if (!(cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)
		 || (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE
		  && cpssp->NAME.portsc[usb_portnr] & PORTSC_SUSPEND)) {
			continue;
		}

		if (usb_portnr == 0) {
			NAME_(0_send_data)(cpssp, pid, data, length, 0);
		} else { assert(usb_portnr == 1);
			NAME_(1_send_data)(cpssp, pid, data, length, 0);
		}
	}
}

/* send a USB handshake packet over CIM */
static void
NAME_(send_usb_handshake)(struct cpssp *cpssp, unsigned char pid)
{
	int usb_portnr;

#if USB_DEBUGMASK & USB_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: sending %s packet\n",
		__FUNCTION__, cim_usb_debug_pid2str(pid));
#endif

	for (usb_portnr = 0; usb_portnr < USB_NUM_PORTS; usb_portnr++) {
		/* port disabled or in selective suspend mode? */
		if (!(cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)
		 || (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE
		  && cpssp->NAME.portsc[usb_portnr] & PORTSC_SUSPEND)) {
			continue;
		}

		if (usb_portnr == 0) {
			NAME_(0_send_handshake)(cpssp, pid);
		} else { assert(usb_portnr == 1);
			NAME_(1_send_handshake)(cpssp, pid);
		}
	}
}

/*----------------------In-memory data structure related----------------------*/

/* some forward declarations */
static void
NAME_(work_frame)(struct cpssp *cpssp);
static void
NAME_(finish_frame)(struct cpssp *cpssp);

static void
NAME_(read_td)(struct cpssp *cpssp)
{
	uint32_t addr = cpssp->NAME.tdp;

	/* fetch TD from memory (4 DWORDs) */
	NAME_(mem_read)(cpssp, addr +  0, 0xf, &cpssp->NAME.td.td[0]);
	NAME_(mem_read)(cpssp, addr +  4, 0xf, &cpssp->NAME.td.td[1]);
	NAME_(mem_read)(cpssp, addr +  8, 0xf, &cpssp->NAME.td.td[2]);
	NAME_(mem_read)(cpssp, addr + 12, 0xf, &cpssp->NAME.td.td[3]);

	cpssp->NAME.td.tdlp = cpssp->NAME.td.td[0] & USB_MEM_TD_0_LP;
	cpssp->NAME.td.td_vf = cpssp->NAME.td.td[0] & USB_MEM_TD_0_VF;
	cpssp->NAME.td.td_q = cpssp->NAME.td.td[0] & USB_MEM_TD_0_QH;
	cpssp->NAME.td.td_t = cpssp->NAME.td.td[0] & USB_MEM_TD_0_TERMINATE;

#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
	/* check reserved bits */
	if (cpssp->NAME.td.td[0] & USB_MEM_TD_0_RESERVED) {
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
				"%s: reserved bits in TD[0] set: %08lX\n",
				__FUNCTION__,
				(unsigned long) (cpssp->NAME.td.td[0] & USB_MEM_TD_0_RESERVED));
	}
	if (cpssp->NAME.td.td[1] & USB_MEM_TD_1_RESERVED) {
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
				"%s: reserved bits in TD[1] set: %08lX\n",
				__FUNCTION__,
				cpssp->NAME.td.td[1] & USB_MEM_TD_1_RESERVED);
	}
#endif

	/* TD inactive? */
	if (!(cpssp->NAME.td.td[1] & USB_MEM_TD_1_STATUS_ACTIVE)) {
		return;
	}

#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
	if (cpssp->NAME.td.td[2] & USB_MEM_TD_2_RESERVED) {
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
			"%s: reserved bits in TD[2] set: %08lX\n",
			__FUNCTION__,
			(unsigned long) (cpssp->NAME.td.td[2] & USB_MEM_TD_2_RESERVED));
	}
#endif

	cpssp->NAME.td.c_err = (cpssp->NAME.td.td[1] & USB_MEM_TD_1_C_ERR)
			>> USB_MEM_TD_1_C_ERR_SHIFT;
	cpssp->NAME.td.actlen = (cpssp->NAME.td.td[1] + 1) & USB_MEM_TD_1_ACTLEN;
	cpssp->NAME.td.maxlen = (((cpssp->NAME.td.td[2] & USB_MEM_TD_2_MAXLEN)
		>> USB_MEM_TD_2_MAXLEN_SHIFT) + 1) & USB_MEM_TD_2_MAXLEN_WRAP;

	cpssp->NAME.td.endpt = (cpssp->NAME.td.td[2] & USB_MEM_TD_2_ENDPT)
		>> USB_MEM_TD_2_ENDPT_SHIFT;
	cpssp->NAME.td.addr = (cpssp->NAME.td.td[2] & USB_MEM_TD_2_DEV_ADDR)
		>> USB_MEM_TD_2_DEV_ADDR_SHIFT;
	cpssp->NAME.td.pid = cpssp->NAME.td.td[2] & USB_MEM_TD_2_PID;
}

static void
NAME_(read_qh)(struct cpssp *cpssp)
{
	int qh_nr;

#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: reading QH @%08lX (#%lu in this frame)\n",
		__FUNCTION__, (unsigned long) cpssp->NAME.qhp,
		cpssp->NAME.qh_count);
#endif

	/*
	 * QH to be read is at address cpssp->NAME.qhp
	 * Did we already visit it in this frame?
	 */
	for (qh_nr = 0; qh_nr < cpssp->NAME.qh_count; qh_nr++) {
		if (cpssp->NAME.visited_qhs[qh_nr] != cpssp->NAME.qhp) {
			continue;
		}
		
		/*
		 * We have already visited this QH in this frame, loop detected;
		 * finish frame.
		 */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: QH has already been visited this frame (was QH #%d), aborting list traversal\n",
			__FUNCTION__, qh_nr);
#endif
		NAME_(finish_frame)(cpssp);
		return;
	}

	/* Store QH address for loop detection. */
	if (cpssp->NAME.qh_count < USB_NUM_VISITED_QHS) {
		cpssp->NAME.visited_qhs[cpssp->NAME.qh_count] = cpssp->NAME.qhp;
	}

	cpssp->NAME.qh_count++;

	NAME_(mem_read)(cpssp, cpssp->NAME.qhp + 0, 0xf, &cpssp->NAME.qh.qhlp);
	NAME_(mem_read)(cpssp, cpssp->NAME.qhp + 4, 0xf, &cpssp->NAME.qh.qelp);

#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
	if (cpssp->NAME.qh.qhlp & USB_MEM_QH_0_RESERVED) {
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
			"%s: reserved bits in QHLP set\n", __FUNCTION__);
	}
#endif
	cpssp->NAME.qh.qh_q = cpssp->NAME.qh.qhlp & USB_MEM_QH_0_QH;
	cpssp->NAME.qh.qh_t = cpssp->NAME.qh.qhlp & USB_MEM_QH_0_TERMINATE;
	cpssp->NAME.qh.qhlp &= USB_MEM_QH_0_QHLP;

#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
	if (cpssp->NAME.qh.qelp & USB_MEM_QH_1_RESERVED) {
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
			"%s: reserved bits in QELP set\n", __FUNCTION__);
	}
#endif
	cpssp->NAME.qh.qe_vf = cpssp->NAME.qh.qelp & USB_MEM_QH_1_VF;
	cpssp->NAME.qh.qe_q = cpssp->NAME.qh.qelp & USB_MEM_QH_1_QH;
	cpssp->NAME.qh.qe_t = cpssp->NAME.qh.qelp & USB_MEM_QH_1_TERMINATE;
	cpssp->NAME.qh.qelp &= USB_MEM_QH_1_QELP;

	cpssp->NAME.q_context = 1;

	/*
	 * make all further code independent of whether we are executing the
	 * first or any subsequent TD in this queue; needed for executing
	 * multiple transactions at once
	 */
	cpssp->NAME.td.td_vf = cpssp->NAME.qh.qe_vf;
	cpssp->NAME.td.td_q = cpssp->NAME.qh.qe_q;
	cpssp->NAME.td.td_t = cpssp->NAME.qh.qe_t;
	cpssp->NAME.td.tdlp = cpssp->NAME.qh.qelp;
}

static void
NAME_(advance_queue)(struct cpssp *cpssp, int advance)
{
	assert(cpssp->NAME.tds_sent == 0);

	if (cpssp->NAME.q_context) {
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: in a queue context (QH @%08lX): ",
			__FUNCTION__, (unsigned long) cpssp->NAME.qhp);
#endif
		if (advance == USB_QUEUE_ADVANCE) {
			/* tdlp -> qh.qelp (in memory, too) */
			cpssp->NAME.qh.qelp = cpssp->NAME.td.tdlp;
			cpssp->NAME.qh.qe_vf = cpssp->NAME.td.td_vf;
			cpssp->NAME.qh.qe_q = cpssp->NAME.td.td_q;
			cpssp->NAME.qh.qe_t = cpssp->NAME.td.td_t;

			NAME_(mem_write)(cpssp, cpssp->NAME.qhp + 4,
				0xf, cpssp->NAME.td.td[0]);

#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_cont(FAUM_LOG_DEBUG, "queue advanced; ");
#endif
		}

		/*
		 * We are ignoring td_vf completely here! FIXME
		 */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
		faum_cont(FAUM_LOG_DEBUG, "all necessary answers here, leaving queue\n");
#endif
			
		/* go to the right */
		if (cpssp->NAME.qh.qh_t) {
			/* nothing comes next */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_cont(FAUM_LOG_DEBUG, "nothing to the right\n");
#endif
			/* finish frame */
			NAME_(finish_frame)(cpssp);

		} else if (cpssp->NAME.qh.qh_q) {
			/* QH comes next */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_cont(FAUM_LOG_DEBUG, "QH @%08lX to the right\n",
				(unsigned long) cpssp->NAME.qh.qhlp);
#endif
			cpssp->NAME.qhp = cpssp->NAME.qh.qhlp;
			NAME_(read_qh)(cpssp);

		} else {
			/* TD comes next */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_cont(FAUM_LOG_DEBUG, "TD @%08lX to the right, leaving queue context\n",
				(unsigned long) cpssp->NAME.qh.qhlp);
#endif
			cpssp->NAME.td.tdlp = cpssp->NAME.qh.qhlp;
			/* this is missing in the specs! */
			cpssp->NAME.q_context = 0;
		}

	} else { /* not a queue context */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: not in a queue context: ",
			__FUNCTION__);
#endif
		if (cpssp->NAME.td.td_t) {
			/* nothing comes next */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_cont(FAUM_LOG_DEBUG, "nothing next\n");
#endif
			/* finish frame */
			NAME_(finish_frame)(cpssp);

		} else if (cpssp->NAME.td.td_q) {
			/* QH is next */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_cont(FAUM_LOG_DEBUG, "QH @%08lX next\n",
				(unsigned long) cpssp->NAME.td.tdlp);
#endif
			cpssp->NAME.qhp = cpssp->NAME.td.tdlp;
			NAME_(read_qh)(cpssp);

		} else {
			/* TD is next */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_cont(FAUM_LOG_DEBUG, "TD @%08lX next\n",
				(unsigned long) cpssp->NAME.td.tdlp);
#endif
		}
	}
}

/* handle packet timeouts that may occur on device disconnection */
static void
NAME_(packet_timeout)(struct cpssp *cpssp)
{
	uint32_t *tdctrl;
	
#if USB_DEBUGMASK & USB_DEBUG_TIMEOUTS
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: packet timeout counter exceeds maximum, signaling timeout\n",
		__FUNCTION__);
#endif
	
	tdctrl = &cpssp->NAME.td.td[1];

	/*
	 * - signal a timeout error in the TD
	 * - mark TD inactive
	 * - signal STALLED
	 * - set error count to zero (we do not retry even if error counter was >1)
	 */
	*tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_C_ERR);
	*tdctrl |= USB_MEM_TD_1_STATUS_STALLED | USB_MEM_TD_1_STATUS_CRC_TIMEOUT_ERROR;

	/* write TD control and status back to memory */
	NAME_(mem_write)(cpssp, cpssp->NAME.tdp + 4, 0xf, *tdctrl);

	/* generate interrupt if ordered to do so */
	cpssp->NAME.timeout_interrupt = 1;

	/* we are not waiting for an answer anymore */
	cpssp->NAME.tds_sent = 0;

	/* do not advance the current queue */
	NAME_(advance_queue)(cpssp, USB_QUEUE_NO_ADVANCE);
}

/* handle incoming USB linestate packet */
static void
NAME_(handle_packet_linestate)(
	struct cpssp *cpssp,
	int usb_portnr,
	int usb_speed
)
{
	cpssp->NAME.portstate[usb_portnr].speed = usb_speed;
	NAME_(connect_device)(cpssp, usb_portnr);
}

/* handle incoming USB data packet */
static void
NAME_(handle_packet_usb_data)(
	struct cpssp *cpssp,
	int usb_portnr,
	unsigned char pid,
	unsigned int length,    /* not contained in real packets */
	unsigned char *data,
	unsigned short crc16
)
{
#if USB_DEBUGMASK & USB_DEBUG_DUMP_PACKETS
	int j;
#endif
	uint32_t *tdctrl;
	int advance_queue;

	/* only process USB token packets if powered and port enabled */
	if (! cpssp->state_power
	 || ! (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)) {
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE,
			SNAME,
			"ignoring incoming USB packets on port %d -- I am not powered, or port is disabled\n",
			usb_portnr);
		return;
	}

	tdctrl = &cpssp->NAME.td.td[1];

#if USB_DEBUGMASK & USB_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: received %s packet (length %d) on port %d",
		__FUNCTION__, cim_usb_debug_pid2str(pid), length, usb_portnr);

	if (0 < length) {
		faum_cont(FAUM_LOG_DEBUG, " ( ");

		for (j = 0; j < length && j < 8; j++) {
			faum_cont(FAUM_LOG_DEBUG, "%02X ", data[j]);
		}

		if (j != length) {
			faum_cont(FAUM_LOG_DEBUG, "... ");
		}

		faum_cont(FAUM_LOG_DEBUG, ")");
	}

	faum_cont(FAUM_LOG_DEBUG, "\n");
#endif

	if (cpssp->NAME.td.pid != USB_PID_IN) {
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
			"%s: %s packet received on port %d although TD PID is %s\n",
			__FUNCTION__, cim_usb_debug_pid2str(pid), usb_portnr,
			cim_usb_debug_pid2str(cpssp->NAME.td.pid));
#endif
		return;
	}

	/* c_err: must only be decremented for errors we do not simulate */
	/* data toggle correct? */
	if (((cpssp->NAME.td.td[2] & USB_MEM_TD_2_D)
	   && pid != USB_PID_DATA1)
	 || (!(cpssp->NAME.td.td[2] & USB_MEM_TD_2_D)
	   && pid != USB_PID_DATA0)) {
		/*
		 * Data toggle incorrect.
		 */
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
			"%s: %s packet received on port %d although %s expected\n",
			__FUNCTION__, cim_usb_debug_pid2str(pid),
			usb_portnr, (cpssp->NAME.td.td[2] & USB_MEM_TD_2_D) ?
					cim_usb_debug_pid2str(USB_PID_DATA1) :
					cim_usb_debug_pid2str(USB_PID_DATA0));
#endif
		advance_queue = USB_QUEUE_NO_ADVANCE;

		NAME_(send_usb_handshake)(cpssp, USB_PID_ACK);

	} else {
		/*
		 * Data toggle correct.
		 */
		if (cpssp->NAME.td.maxlen < length) {
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
				"%s: oversized %s packet received (%d bytes, maxlen = %d)\n",
				__FUNCTION__, cim_usb_debug_pid2str(pid),
				length, cpssp->NAME.td.maxlen);
#endif
			/* FIXME: is this the correct behaviour? */
			/* this would be "babble" on real hardware */
			length = cpssp->NAME.td.maxlen;
		}

		/* short packet detected? */
		if (*tdctrl & USB_MEM_TD_1_SPD
		 && cpssp->NAME.q_context
		 && length < cpssp->NAME.td.maxlen) {
			cpssp->NAME.spd = 1;
			advance_queue = USB_QUEUE_NO_ADVANCE;
		} else {
			advance_queue = USB_QUEUE_ADVANCE;
		}

		/* IOC? */
		if (*tdctrl & USB_MEM_TD_1_IOC) {
			cpssp->NAME.ioc = 1;
		}

		/* put data in memory */
		if (0 < length) {
			uint32_t addr;
			unsigned int i;

			addr = cpssp->NAME.td.td[3];
			i = 0;
			while (i < length) {
				unsigned int count;
				unsigned int bs;
				uint32_t val;

				count = length - i;
				if (4 < (addr & 3) + count) {
					count = 4 - (addr & 3);
				}
				bs = ((1 << count) - 1) << (addr & 3);

				val = 0;
				if ((bs >> 0) & 1) {
					val |= data[i++] << 0;
				}
				if ((bs >> 1) & 1) {
					val |= data[i++] << 8;
				}
				if ((bs >> 2) & 1) {
					val |= data[i++] << 16;
				}
				if ((bs >> 3) & 1) {
					val |= data[i++] << 24;
				}

				NAME_(mem_write)(cpssp, addr & ~3, bs, val);

				addr += count;
			}
		}

		*tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_ACTLEN);

		/*
		 * store data length:
		 * this also takes care of the special 0 encoding
		 */
		*tdctrl |= (length - 1) & USB_MEM_TD_1_ACTLEN;

		/* write TD control and status back to memory */
		NAME_(mem_write)(cpssp, cpssp->NAME.tdp + 4, 0xf, *tdctrl);

		NAME_(send_usb_handshake)(cpssp, USB_PID_ACK);
	}
	
	/* one answer less pending */
	cpssp->NAME.tds_sent--;
	assert(cpssp->NAME.tds_sent == 0);

	NAME_(advance_queue)(cpssp, advance_queue);

	NAME_(work_frame)(cpssp);
}

/* handle incoming USB handshake packet */
static void
NAME_(handle_packet_usb_handshake)(
	struct cpssp *cpssp,
	int usb_portnr,
	unsigned char pid
)
{
	uint32_t *tdctrl;

	int advance_queue;

	/* only process USB token packets if powered and port enabled */
	if (! cpssp->state_power
	 || ! (cpssp->NAME.portsc[usb_portnr] & PORTSC_PORT_ENABLE)) {
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE,
			SNAME,
			"ignoring incoming USB packets on port %d -- I am not powered, or port is disabled\n",
			usb_portnr);
		return;
	}

	tdctrl = &cpssp->NAME.td.td[1];

#if USB_DEBUGMASK & USB_DEBUG_DUMP_PACKETS
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: %s handshake packet received on port %d\n",
		__FUNCTION__, cim_usb_debug_pid2str(pid), usb_portnr);
#endif

	/* c_err: must only be decremented for errors we do not simulate */
	/* maybe timeout? */
	if (pid == USB_PID_ACK) {
		/* FIXME: correct behaviour? */

		advance_queue = USB_QUEUE_ADVANCE;
		*tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_ACTLEN);
		*tdctrl |= (cpssp->NAME.td.maxlen - 1) & USB_MEM_TD_1_ACTLEN;

		if (*tdctrl & USB_MEM_TD_1_IOC) {
			cpssp->NAME.ioc = 1;
		}

	} else if (pid == USB_PID_NAK) {
		/* FIXME: correct behaviour? */

		advance_queue = USB_QUEUE_NO_ADVANCE;
		/* *tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_ACTLEN); */
		*tdctrl &= ~USB_MEM_TD_1_ACTLEN;
		*tdctrl |= USB_MEM_TD_1_STATUS_NAK_RECEIVED;

		/* if this was an IN transaction, we didn't receive anything */
		if (cpssp->NAME.td.pid == USB_PID_IN) {
			*tdctrl |= USB_MEM_TD_1_ACTLEN;		/* encoded 0 */
		} else {
			*tdctrl |= (cpssp->NAME.td.maxlen - 1) & USB_MEM_TD_1_ACTLEN;
		}

		/*
		 * If a NAK handshake is received from a SETUP transaction, a
		 * Time Out Error will also be reported.
		 */
		if (cpssp->NAME.td.pid == USB_PID_SETUP) {
			*tdctrl |= USB_MEM_TD_1_STATUS_CRC_TIMEOUT_ERROR;
		}
	} else if (pid == USB_PID_STALL) {
		/* FIXME: correct behaviour? */

		advance_queue = USB_QUEUE_NO_ADVANCE;
		*tdctrl &= ~(USB_MEM_TD_1_STATUS | USB_MEM_TD_1_ACTLEN);
		*tdctrl |= USB_MEM_TD_1_STATUS_STALLED;

		/* if this was an IN transaction, we didn't receive anything */
		if (cpssp->NAME.td.pid == USB_PID_IN) {
			*tdctrl |= USB_MEM_TD_1_ACTLEN;		/* encoded 0 */
		} else {
			*tdctrl |= (cpssp->NAME.td.maxlen - 1) & USB_MEM_TD_1_ACTLEN;
		}

		/*
		 * If a STALL handshake is received from a SETUP transaction, a
		 * Time Out Error will also be reported.
		 */
		if (cpssp->NAME.td.pid == USB_PID_SETUP) {
			*tdctrl |= USB_MEM_TD_1_STATUS_CRC_TIMEOUT_ERROR;
		}

		cpssp->NAME.stalled_interrupt = 1;
	} else {
		/* there are no other handshake types, programming error! */
		faum_log(FAUM_LOG_ERROR, USB_LOG_TYPE, SNAME,
			"%s: ignoring unknown handshake!\n",
			__FUNCTION__);
		return;
	}

	/* write TD control and status back to memory */
	NAME_(mem_write)(cpssp, cpssp->NAME.tdp + 4, 0xf, *tdctrl);

	/* one answer less pending */
	cpssp->NAME.tds_sent--;

	NAME_(advance_queue)(cpssp, advance_queue);

	NAME_(work_frame)(cpssp);
}

static void
NAME_(usb0_speed_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_linestate)(cpssp, 0, val);
}

static void
NAME_(usb0_recv_token)(
	void *_cpssp,
	int pid,
	int addr,
	int endp
)
{
	fixme();
}

static void
NAME_(usb0_recv_sof)(
	void *_cpssp,
	int frame_num
)
{
	fixme();
}

static void
NAME_(usb0_recv_data)(
	void *_cpssp,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_usb_data)(cpssp, 0, pid, length, data, crc16);
}

static void
NAME_(usb0_recv_handshake)(void *_cpssp, int pid)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_usb_handshake)(cpssp, 0, pid);
}

static void
NAME_(usb1_speed_set)(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_linestate)(cpssp, 1, val);
}

static void
NAME_(usb1_recv_token)(
	void *_cpssp,
	int pid,
	int addr,
	int endp
)
{
	fixme();
}

static void
NAME_(usb1_recv_sof)(
	void *_cpssp,
	int frame_num
)
{
	fixme();
}

static void
NAME_(usb1_recv_data)(
	void *_cpssp,
	int pid,
	unsigned int length,
	uint8_t *data,
	uint16_t crc16
)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_usb_data)(cpssp, 1, pid, length, data, crc16);
}

static void
NAME_(usb1_recv_handshake)(void *_cpssp, int pid)
{
	struct cpssp *cpssp = _cpssp;

	NAME_(handle_packet_usb_handshake)(cpssp, 1, pid);
}

static void
NAME_(debug_dump_td)(struct cpssp *cpssp)
{
#if USB_DEBUGMASK & USB_DEBUG_DUMP_TDS
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: raw TD:\n%08lX\n%08lX\n%08lX\n%08lX\n",
		__FUNCTION__,
		(unsigned long) cpssp->NAME.td.td[0],
		(unsigned long) cpssp->NAME.td.td[1],
		(unsigned long) cpssp->NAME.td.td[2],
		(unsigned long) cpssp->NAME.td.td[3]);

	faum_cont(FAUM_LOG_DEBUG, "-----------------------------------------------------------\n");
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: information from TD @%08lx:\n"
		"1) LP: %08lX %s %s %s\n"
		"2) %s C_ERR=%d %s %s %s ACTLEN=%d\n"
		"   STATUS: %s %s %s %s %s %s %s\n"
		"3) MAXLEN=%4d DATA=%d ENDPT=%2d ADDRESS=%2d PID=%02X (%s)\n"
		"4) BUFFER POINTER: %08lX\n",
		__FUNCTION__,
		(unsigned long) cpssp->NAME.tdp,

		(unsigned long) cpssp->NAME.td.tdlp,
		cpssp->NAME.td.td_vf ? "VF" : "vf",
		cpssp->NAME.td.td_q ? "Q" : "q",
		cpssp->NAME.td.td_t ? "T" : "t",

		cpssp->NAME.td.td[1] & USB_MEM_TD_1_SPD ? "SPD" : "spd",
		cpssp->NAME.td.c_err,
		cpssp->NAME.td.td[1] & USB_MEM_TD_1_LS ? "LS" : "ls",
		cpssp->NAME.td.td[1] & USB_MEM_TD_1_IOS ? "IOS" : "ios",
		cpssp->NAME.td.td[1] & USB_MEM_TD_1_IOC ? "IOC" : "ioc",
		cpssp->NAME.td.actlen,

		cpssp->NAME.td.td[1] & USB_MEM_TD_1_STATUS_ACTIVE ? "ACTIVE" : "active",
		cpssp->NAME.td.td[1] & USB_MEM_TD_1_STATUS_STALLED ? "STALLED" : "stalled",
		cpssp->NAME.td.td[1] & USB_MEM_TD_1_STATUS_DATA_BUFFER_ERROR ? "DATABUF" : "databuf",
		cpssp->NAME.td.td[1] & USB_MEM_TD_1_STATUS_BABBLE_DETECTED ? "BABBLE" : "babble",
		cpssp->NAME.td.td[1] & USB_MEM_TD_1_STATUS_NAK_RECEIVED ? "NAK" : "nak",
		cpssp->NAME.td.td[1] & USB_MEM_TD_1_STATUS_CRC_TIMEOUT_ERROR ? "TIMEOUT" : "timeout",
		cpssp->NAME.td.td[1] & USB_MEM_TD_1_STATUS_BITSTUFF_ERROR ? "BITSTUFF" : "bitstuff",

		cpssp->NAME.td.maxlen,
		cpssp->NAME.td.td[2] & USB_MEM_TD_2_D ? 1 : 0,
		cpssp->NAME.td.endpt,
		cpssp->NAME.td.addr,
		cpssp->NAME.td.pid,
		cim_usb_debug_pid2str(cpssp->NAME.td.pid),

		(unsigned long) cpssp->NAME.td.td[3]);
	faum_cont(FAUM_LOG_DEBUG, "-----------------------------------------------------------\n");
#endif
}

static void
NAME_(execute_td)(struct cpssp *cpssp)
{
	/* TD inactive? */
	if (!(cpssp->NAME.td.td[1] & USB_MEM_TD_1_STATUS_ACTIVE)) {
		/* IOC set? */
		if (cpssp->NAME.td.td[1] & USB_MEM_TD_1_IOC) {
			cpssp->NAME.ioc = 1;
		}

		NAME_(advance_queue)(cpssp, USB_QUEUE_NO_ADVANCE);
		return;
	}

	if (USB_MEM_TD_2_MAXLEN_ILLEGAL <= cpssp->NAME.td.maxlen) {
#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
				"%s: maxlen is 0x%X >= 0x%X, signaling HC process error\n",
				__FUNCTION__, cpssp->NAME.td.maxlen, USB_MEM_TD_2_MAXLEN_ILLEGAL);
#endif
		cpssp->NAME.hc_process_error_interrupt = 1;
		NAME_(finish_frame)(cpssp);
		return;
	}

	/* check PID validity */
	if (cpssp->NAME.td.pid != USB_PID_SETUP
	 && cpssp->NAME.td.pid != USB_PID_IN
	 && cpssp->NAME.td.pid != USB_PID_OUT) {
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: erroneous PID %02X found, signaling HC process error\n",
			__FUNCTION__, cpssp->NAME.td.pid);
#endif
		cpssp->NAME.hc_process_error_interrupt = 1;
		NAME_(finish_frame)(cpssp);
		return;
	}

	NAME_(debug_dump_td)(cpssp);

	/* now we are sure we'll really execute this TD */

	cpssp->NAME.tds_sent++;

	/*
	 * In IN transfers the function is the sender, in
	 * OUT and SETUP transfers the host sends.
	 */
	if (cpssp->NAME.td.pid == USB_PID_SETUP
	 || cpssp->NAME.td.pid == USB_PID_OUT) {
		if (0 < cpssp->NAME.td.maxlen) {
			/* fetch data from memory */
			uint32_t addr;
			unsigned int i;

			addr = cpssp->NAME.td.td[3];
			i = 0;
			while (i < cpssp->NAME.td.maxlen) {
				unsigned int count;
				unsigned int bs;
				uint32_t val;

				count = cpssp->NAME.td.maxlen - i;
				if (4 < (addr & 3) + count) {
					count = 4 - (addr & 3);
				}
				bs = ((1 << count) - 1) << (addr & 3);

				NAME_(mem_read)(cpssp, addr & ~3, bs, &val);

				if ((bs >> 0) & 1) {
					cpssp->NAME.sendbuffer[i++] = val >> 0;
				}
				if ((bs >> 1) & 1) {
					cpssp->NAME.sendbuffer[i++] = val >> 8;
				}
				if ((bs >> 2) & 1) {
					cpssp->NAME.sendbuffer[i++] = val >> 16;
				}
				if ((bs >> 3) & 1) {
					cpssp->NAME.sendbuffer[i++] = val >> 24;
				}

				addr += count;
			}
		}

		/* Send Token */
		NAME_(send_usb_token)(cpssp, cpssp->NAME.td.pid,
				cpssp->NAME.td.addr, cpssp->NAME.td.endpt);

		/* Send Data Packet */
		NAME_(send_usb_data)(cpssp,
				cpssp->NAME.td.td[2] & USB_MEM_TD_2_D ?
						USB_PID_DATA1 : USB_PID_DATA0,
				cpssp->NAME.sendbuffer,
				cpssp->NAME.td.maxlen);
	} else {
		/* Send Token */
		NAME_(send_usb_token)(cpssp, cpssp->NAME.td.pid,
				cpssp->NAME.td.addr, cpssp->NAME.td.endpt);
	}

	/* isochroneous TD and SETUP/OUT? */
	if (cpssp->NAME.td.td[1] & USB_MEM_TD_1_IOS
	 && (cpssp->NAME.td.pid == USB_PID_SETUP
	  || cpssp->NAME.td.pid == USB_PID_OUT)) {
		/* mark inactive, do not wait for an answer */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: TD with IOS set!\n", __FUNCTION__);
#endif
		NAME_(handle_packet_usb_handshake)(cpssp, 0, USB_PID_ACK);
	}
}

static void
NAME_(start_frame)(struct cpssp *cpssp)
{
	uint32_t flp;

#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: starting a new frame (frnum %04d)\n",
		__FUNCTION__, cpssp->NAME.frnum);
#endif

	cpssp->NAME.working = 1;
	cpssp->NAME.timeout_interrupt =
		cpssp->NAME.ioc =
		cpssp->NAME.spd =
		cpssp->NAME.babble_interrupt =
		cpssp->NAME.stalled_interrupt =
		cpssp->NAME.data_buffer_error_interrupt =
		cpssp->NAME.bit_stuff_error_interrupt =
		cpssp->NAME.hc_process_error_interrupt =
		cpssp->NAME.host_system_error_interrupt = 0;

	cpssp->NAME.qh_count = 0;

	/* TODO: issue SOF? */

	/* read frame list pointer from frame list */
	NAME_(mem_read)(cpssp, cpssp->NAME.flbaseadd +
		((cpssp->NAME.frnum & FRNUM_FL_INDEX) << 2),
		0xf, &flp);

#if USB_DEBUGMASK & USB_DEBUG_WARNINGS
	if (flp & USB_MEM_FLP_RESERVED) {
		faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
			"%s: reserved bits in frame list entry set\n", __FUNCTION__);
	}
#endif

	/* initialize frame list traversal state */
	if (flp & USB_MEM_FLP_TERMINATE) {
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: frame list entry with T bit set\n", __FUNCTION__);
#endif
		NAME_(finish_frame)(cpssp);
		return;
	} else if (flp & USB_MEM_FLP_QH) {	/* first structure is a QH */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: first structure in frame is a QH\n", __FUNCTION__);
#endif
		cpssp->NAME.qhp = flp & USB_MEM_FLP_FLP;
		NAME_(read_qh)(cpssp);
	} else { /* first structure is a TD */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
		faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
			"%s: first structure in frame is a TD\n", __FUNCTION__);
#endif
		cpssp->NAME.td.tdlp = flp & USB_MEM_FLP_FLP;
		cpssp->NAME.q_context = 0;
	}

	/* we haven't done any communications up to now, so we're not waiting
	 * for packets already */
	cpssp->NAME.tds_sent = 0;

	NAME_(work_frame)(cpssp);
}

static void
NAME_(work_frame)(struct cpssp *cpssp)
{
	if (cpssp->NAME.tds_sent) {
		/* Wait for answer first... */
		return;
	}
 
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: entering loop\n",
		__FUNCTION__);
#endif

	do {
		/*
		 * cf. UHCI Spec., Rev. 1.1, chapter 3.4.2 "Transfer Queueing",
		 * "USB Schedule List Traversal Decision Table" (incomplete!)
		 * and "USB Linked List Traversal State Diagram"
		 */

		/* states #1, #2, #3 */
		if (!cpssp->NAME.q_context) {
			/*
			 * Not in a queue context.
			 */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
				"%s: not in a queue context, executing next TD @%08lX\n",
				__FUNCTION__, (unsigned long) cpssp->NAME.td.tdlp);
#endif

			/* execute TD pointed at by cpssp->NAME.td.tdlp */
			cpssp->NAME.tdp = cpssp->NAME.td.tdlp;
			NAME_(read_td)(cpssp);
			NAME_(execute_td)(cpssp);

		} else {
			/*
			 * Queue context, states #4-#12
			 */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
				"%s: in a queue context (QH @%08lX): ",
				__FUNCTION__, (unsigned long) cpssp->NAME.qhp);
#endif

			/* states #4, #5, #6, #9, #11 */
			if (!cpssp->NAME.td.td_t
			 && !cpssp->NAME.td.td_q) {
				/* TD below */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
				faum_cont(FAUM_LOG_DEBUG, "TD @%08lX below\n",
					(unsigned long) cpssp->NAME.td.tdlp);
#endif
				/* execute TD pointed at by cpssp->NAME.td.tdlp */
				cpssp->NAME.tdp = cpssp->NAME.td.tdlp;
				NAME_(read_td)(cpssp);
				NAME_(execute_td)(cpssp);
				/* we are probably waiting for answers now */

			/* state #8 */
			} else if (!cpssp->NAME.td.td_t
				&& cpssp->NAME.td.td_q) {
				/* QH below */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
				faum_cont(FAUM_LOG_DEBUG, "QH @%08lX below\n",
					(unsigned long) cpssp->NAME.qh.qelp);
#endif
				cpssp->NAME.qhp = cpssp->NAME.qh.qelp;
				NAME_(read_qh)(cpssp);
				/* we are not yet waiting for answers, loop! */

			/* states #7, #10, #12 */
			} else { /* td.td_t is set */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
				faum_cont(FAUM_LOG_DEBUG, "no (more) entries in this queue, ");
#endif

				/* state #10 */
				if (cpssp->NAME.qh.qh_t) {
					/* QH with nothing to the right */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
					faum_cont(FAUM_LOG_DEBUG, "nothing to the right\n");
#endif
					/* end of frame */
					NAME_(finish_frame)(cpssp);
					/* we are done for this frame, leaving the loop */

				/* state #7 */
				} else if (!cpssp->NAME.qh.qh_q) {
					/* QH with TD to the right */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
					faum_cont(FAUM_LOG_DEBUG, "TD @%08lX to the right\n",
						(unsigned long) cpssp->NAME.qh.qhlp);
#endif
					cpssp->NAME.td.tdlp = cpssp->NAME.qh.qhlp;
					/* leaving queue context! specs are buggy here! */
					cpssp->NAME.q_context = 0;
					/* we are not yet waiting for answers, loop! */

				/* state #12 */
				} else if (cpssp->NAME.qh.qh_q) {
					/* QH with QH to the right */
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
					faum_cont(FAUM_LOG_DEBUG, "QH @%08lX to the right\n",
						(unsigned long) cpssp->NAME.qh.qhlp);
#endif
					cpssp->NAME.qhp = cpssp->NAME.qh.qhlp;
					NAME_(read_qh)(cpssp);
					/* we are not yet waiting for answers, loop! */
				}
			}
		}

		if (cpssp->NAME.tds_sent == 0
		 && USB_MAX_QH_PER_FRAME <= cpssp->NAME.qh_count) {
#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
			/*
			 * If the simulation hits this point, two constants
			 * might need to be adjusted:
			 * - USB_MAX_QH_PER_FRAME might need to be increased in
			 *   case the in-memory QH chain gets too long (too
			 *   many devices attached, too many endpoints per
			 *   device)
			 * - USB_NUM_VISITED_QHS might need to be increased in
			 *   case the QH chain loops back to a QH too late in
			 *   the list (too many INT queues, too many low-speed
			 *   CTRL or BULK queues) (unlikely)
			 */
			faum_log(FAUM_LOG_WARNING, USB_LOG_TYPE, SNAME,
				"%s: forcing frame termination, qh_count exceeded limit -- this is only a fallback mechanism!\n",
				__FUNCTION__);
#endif
			NAME_(finish_frame)(cpssp);
		}
	} while (cpssp->NAME.tds_sent == 0
	      && cpssp->NAME.working);

#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: no more work this time\n",
		__FUNCTION__);
#endif
}

static void
NAME_(finish_frame)(struct cpssp *cpssp)
{
	int do_interrupt = 0;

#if USB_DEBUGMASK & USB_DEBUG_FRAME_LIST
	faum_log(FAUM_LOG_DEBUG, USB_LOG_TYPE, SNAME,
		"%s: finishing frame\n",
		__FUNCTION__);
#endif

	cpssp->NAME.working = 0;
	assert(cpssp->NAME.tds_sent == 0);

	/* generate interrupt? */
	/* cf. UHCI Spec. Rev. 1.1, chapter 4, "Interrupts" */
	if (cpssp->NAME.timeout_interrupt) {
		if (cpssp->NAME.usbintr & USBINTR_TIE) {
			do_interrupt = 1;
		}
		cpssp->NAME.usbsts |= USBSTS_ERROR_INTERRUPT;
	}
	if (cpssp->NAME.ioc) {
		if (cpssp->NAME.usbintr & USBINTR_IOC) {
			do_interrupt = 1;
		}
		cpssp->NAME.usbsts |= USBSTS_INTERRUPT;
	}
	if (cpssp->NAME.spd) {
		if (cpssp->NAME.usbintr & USBINTR_SPIE) {
			do_interrupt = 1;
		}
		cpssp->NAME.usbsts |= USBSTS_INTERRUPT;
	}
	/* TODO: resume received missing */
	if (cpssp->NAME.babble_interrupt
	 || cpssp->NAME.stalled_interrupt
	 || cpssp->NAME.data_buffer_error_interrupt
	 || cpssp->NAME.bit_stuff_error_interrupt
	 || cpssp->NAME.hc_process_error_interrupt
	 || cpssp->NAME.host_system_error_interrupt) {
		do_interrupt = 1; /* non-maskable */
		if (cpssp->NAME.babble_interrupt
	 	 || cpssp->NAME.stalled_interrupt
		 || cpssp->NAME.data_buffer_error_interrupt
		 || cpssp->NAME.bit_stuff_error_interrupt) {
			cpssp->NAME.usbsts |= USBSTS_ERROR_INTERRUPT;
		}
		if (cpssp->NAME.hc_process_error_interrupt) {
			cpssp->NAME.usbsts |= USBSTS_HC_PROCESS_ERROR;
			cpssp->NAME.usbsts |= USBSTS_HC_HALTED;
			cpssp->NAME.usbcmd &= ~USBCMD_RUN_STOP;
		}
		if (cpssp->NAME.host_system_error_interrupt) {
			cpssp->NAME.usbsts |= USBSTS_HC_HALTED;
			cpssp->NAME.usbcmd &= ~USBCMD_RUN_STOP;
		}
	}

	if (do_interrupt) {
		NAME_(set_irq)(cpssp, 1);
	}

#if 1
	cpssp->NAME.tsc_passed += cpssp->NAME.tsc_step_busy;
#else
	cpssp->NAME.tsc_passed += TIME_HZ / 10;
#endif
	cpssp->NAME.frnum++;
	cpssp->NAME.frnum &= FRNUM_FRNUM;

	/* re-schedule timer handler for next frame */
	cpssp->NAME.timer_scheduled++;
	time_call_at(cpssp->NAME.tsc_passed + cpssp->NAME.tsc_step_busy,
			NAME_(timer_event), cpssp);
}

static void
NAME_(timer_event)(void *datap)
{
	struct cpssp *cpssp = datap;

	/* not scheduled anymore */
	cpssp->NAME.timer_scheduled--;

	if (! cpssp->state_power) {		/* not powered */
		goto schedule_idle;
	}

	/* check BME (bus master enable), check schedule flag, check HCHalted */
	if (cpssp->NAME.master_enable
	 && cpssp->NAME.usbcmd & USBCMD_RUN_STOP
	 && !(cpssp->NAME.usbsts & USBSTS_HC_HALTED)) {
		if (!cpssp->NAME.working) {
			/* start a new frame */
			NAME_(start_frame)(cpssp);
		} else { /* we are working on a frame */
			NAME_(work_frame)(cpssp);
		}
	} else { /* stopped or HCHalted */
		cpssp->NAME.working = 0;
		goto schedule_idle;
	}

	/*
	 * schedule in any case: external devices might not respond and we
	 * won't be woken up anymore if we don't additionally schedule with
	 * time again; NAME_(work_frame)() handles timeouts (which
	 * may occur on device disconnection for instance).
	 */

schedule_idle:
	if (! cpssp->NAME.timer_scheduled) {
		cpssp->NAME.timer_scheduled++;
		cpssp->NAME.tsc_passed += cpssp->NAME.tsc_step_idle;
		time_call_at(cpssp->NAME.tsc_passed + cpssp->NAME.tsc_step_idle,
				NAME_(timer_event), cpssp);
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	cpssp->NAME.timer_scheduled = 0;

	/* schedule timer event handler */
	cpssp->NAME.tsc_passed = time_virt();
	cpssp->NAME.tsc_step_idle = TIME_HZ / USB_TIMER_FREQ_IDLE;
	cpssp->NAME.tsc_step_busy = TIME_HZ / USB_TIMER_FREQ_BUSY;
	time_call_at(cpssp->NAME.tsc_passed + cpssp->NAME.tsc_step_idle,
			NAME_(timer_event), cpssp);
	cpssp->NAME.timer_scheduled++;
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
