/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* General-purpose timers (TIM) */

/*
 * For most of the comments, infos, ... see:
 * ST - RM0383 - Reference Manual - STM32F411xC/E advanced ARM-based
 * 32-bit MCUs
 */

#define DEBUG_CONTROL_FLOW	1

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	uint16_t prescaler_count;

	/* control register 1 (TIMx_CR1), 13.4.1, 342/836 */
	uint8_t ckd;
	uint8_t arpe;
	uint8_t cms;
	uint8_t dir;
	uint8_t opm;
	uint8_t urs;
	uint8_t udis;
	uint8_t cen;

	/* control register 2 (TIMx_CR1), 13.4.2, 344/836 */
	uint8_t ti1s;
	uint8_t mms;
	uint8_t ccds;

	/* event generation register (TIMx_CR1), 13.4.6, 351/836 */
	/* Nothing... */

	/* capture/compare mode register 1-2 (TIMx_CCMRy), 13.4.7-8, 352/836 */
	uint8_t occe[4];
	uint8_t ocm[4];
	uint8_t ocpe[4];
	uint8_t ocfe[4];
	uint8_t ccs[4];

	/* counter (TIMx_CNT), 13.4.10, 358/836 */
	uint32_t cnt; /* 16 bit? FIXME */

	/* prescaler (TIMx_PSC), 13.4.11, 358/836 */
	uint16_t psc;
	uint16_t psc_shadow;

	/* auto-reload register (TIMx_ARR), 13.4.12, 358/836 */
	uint16_t arr;
	uint16_t arr_shadow;

	uint32_t ccr[4]; /* 16 bit? FIXME */
	uint32_t ccr_shadow[4]; /* 16 bit? FIXME */

	uint32_t reg[0x1000 >> 2];
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

static void
NAME_(update_event)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s\n", __FUNCTION__);
	}

	if (! cpssp->NAME.udis) {
		unsigned int n;

		cpssp->NAME.psc_shadow = cpssp->NAME.psc;
		cpssp->NAME.arr_shadow = cpssp->NAME.arr;
		for (n = 0; n < 4; n++) {
			cpssp->NAME.ccr_shadow[n] = cpssp->NAME.ccr[n];
		}

		NAME_(irq_set)(cpssp, 1);
		NAME_(irq_set)(cpssp, 0);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int n;

	addr &= 0x3ff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		/* control register 1 (TIMx_CR1), 13.4.1, 342/836 */
		assert((bs & 0b0011) == 0b0011);

		/* Bit 15-10: Reserved */
		cpssp->NAME.ckd = (val >> 8) & 3;
		cpssp->NAME.arpe = (val >> 7) & 1;
		cpssp->NAME.cms = (val >> 5) & 3;
		cpssp->NAME.dir = (val >> 4) & 1;
		cpssp->NAME.opm = (val >> 3) & 1;
		cpssp->NAME.urs = (val >> 2) & 1;
		cpssp->NAME.udis = (val >> 1) & 1;
		cpssp->NAME.cen = (val >> 0) & 1;
		break;
	case 0x04:
		/* control register 2 (TIMx_CR1), 13.4.2, 344/836 */
		assert((bs & 0b0011) == 0b0011);

		/* Bit 15-8: Reserved */
		cpssp->NAME.ti1s = (val >> 7) & 1;
		cpssp->NAME.mms = (val >> 4) & 7;
		cpssp->NAME.ccds = (val >> 3) & 1;
		/* Bit 2-0: Reserved */
		break;
	case 0x08:
		assert(0); /* FIXME */
		break;
	case 0x0c:
		assert(0); /* FIXME */
		break;
	case 0x10:
		assert(0); /* FIXME */
		break;
	case 0x14:
		/* event generation register (TIMx_CR1), 13.4.6, 351/836 */
		/* Bit 15-7: Reserved */
		if ((val >> 6) & 1) {
			assert(0); /* FIXME */
		}
		/* Bit 5: Reserved */
		if ((val >> 4) & 1) {
			assert(0); /* FIXME */
		}
		if ((val >> 3) & 1) {
			assert(0); /* FIXME */
		}
		if ((val >> 2) & 1) {
			assert(0); /* FIXME */
		}
		if ((val >> 1) & 1) {
			assert(0); /* FIXME */
		}
		if ((val >> 0) & 1) {
			NAME_(update_event)(cpssp);
		}
		break;
	case 0x18:
	case 0x1c:
		/* capture/compare mode register 1-2 (TIMx_CCMRy), 13.4.7-8, 352/836 */
		n = (addr - 0x18) >> 1;

		cpssp->NAME.occe[n + 1] = (val >> 15) & 1;
		cpssp->NAME.ocm[n + 1] = (val >> 12) & 7;
		cpssp->NAME.ocpe[n + 1] = (val >> 11) & 1;
		cpssp->NAME.ocfe[n + 1] = (val >> 10) & 1;
		cpssp->NAME.ccs[n + 1] = (val >> 8) & 3;

		cpssp->NAME.occe[n + 0] = (val >> 7) & 1;
		cpssp->NAME.ocm[n + 0] = (val >> 4) & 7;
		cpssp->NAME.ocpe[n + 0] = (val >> 3) & 1;
		cpssp->NAME.ocfe[n + 0] = (val >> 2) & 1;
		cpssp->NAME.ccs[n + 0] = (val >> 0) & 3;
		break;
	case 0x20:
		/* capture/compare enable register (TIMx_CCER), 13.4.9, 356/836 */
		goto warn;
	case 0x24:
		/* counter (TIMx_CNT), 13.4.10, 358/836 */
		cpssp->NAME.cnt = val;
		break;
	case 0x28:
		/* prescaler (TIMx_PSC), 13.4.11, 358/836 */
		cpssp->NAME.psc = (val >> 0) & 0xffff;
		break;
	case 0x2c:
		/* auto-reload register (TIMx_ARR), 13.4.12, 358/836 */
		cpssp->NAME.arr = (val >> 0) & 0xffff;
		if (! cpssp->NAME.arpe) {
			cpssp->NAME.arr_shadow = (val >> 0) & 0xffff;
		}
		break;
	case 0x30:
		goto undefined;
	case 0x34:
	case 0x38:
	case 0x3c:
	case 0x40:
		/* capture/compare register 1-4 (TIMx_CCR1-4), 13.4.13-16, 358/836 */
		n = (addr - 0x34) >> 4;

		cpssp->NAME.ccr[n] = val;
		break;
	default:
	undefined:;
	warn:	;
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);

		cpssp->NAME.reg[addr >> 2] = val;
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int n;

	addr &= 0x3ff;

	switch (addr) {
	case 0x00:
		/* control register 1 (TIMx_CR1), 13.4.1, 342/836 */
		*valp = 0;
		/* Bit 15-10: Reserved */
		*valp |= cpssp->NAME.ckd << 8;
		*valp |= cpssp->NAME.arpe << 7;
		*valp |= cpssp->NAME.cms << 5;
		*valp |= cpssp->NAME.dir << 4;
		*valp |= cpssp->NAME.opm << 3;
		*valp |= cpssp->NAME.urs << 2;
		*valp |= cpssp->NAME.udis << 1;
		*valp |= cpssp->NAME.cen << 0;
		break;
	case 0x04:
		/* control register 2 (TIMx_CR1), 13.4.2, 344/836 */
		*valp = 0;
		/* Bit 15-8: Reserved */
		*valp |= cpssp->NAME.ti1s << 7;
		*valp |= cpssp->NAME.mms << 4;
		*valp |= cpssp->NAME.ccds << 3;
		/* Bit 2-0: Reserved */
		break;
	case 0x08:
		assert(0); /* FIXME */
		break;
	case 0x0c:
		assert(0); /* FIXME */
		break;
	case 0x10:
		assert(0); /* FIXME */
		break;
	case 0x14:
		/* event generation register (TIMx_CR1), 13.4.6, 351/836 */
		goto write_only;
	case 0x18:
	case 0x1c:
		/* capture/compare mode register 1-2 (TIMx_CCMRy), 13.4.7-8, 352/836 */
		n = (addr - 0x18) >> 1;

		*valp = 0;
		*valp |= cpssp->NAME.occe[n + 1] << 15;
		*valp |= cpssp->NAME.ocm[n + 1] << 12;
		*valp |= cpssp->NAME.ocpe[n + 1] << 11;
		*valp |= cpssp->NAME.ocfe[n + 1] << 10;
		*valp |= cpssp->NAME.ccs[n + 1] << 8;

		*valp |= cpssp->NAME.occe[n + 0] << 7;
		*valp |= cpssp->NAME.ocm[n + 0] << 4;
		*valp |= cpssp->NAME.ocpe[n + 0] << 3;
		*valp |= cpssp->NAME.ocfe[n + 0] << 2;
		*valp |= cpssp->NAME.ccs[n + 0] << 0;
		break;
	case 0x20:
		/* capture/compare enable register (TIMx_CCER), 13.4.9, 356/836 */
		*valp = 0;
		goto warn;
	case 0x24:
		/* counter (TIMx_CNT), 13.4.10, 358/836 */
		*valp = 0;
		*valp |= cpssp->NAME.cnt << 0;
		break;
	case 0x28:
		/* prescaler (TIMx_PSC), 13.4.11, 358/836 */
		*valp = 0;
		*valp |= cpssp->NAME.psc << 0;
		break;
	case 0x2c:
		/* auto-reload register (TIMx_ARR), 13.4.12, 358/836 */
		*valp = 0;
		*valp |= cpssp->NAME.arr << 0;
		break;
	case 0x30:
		goto undefined;
	case 0x34:
	case 0x38:
	case 0x3c:
	case 0x40:
		/* capture/compare register 1-4 (TIMx_CCR1-4), 13.4.13-16, 358/836 */
		n = (addr - 0x34) >> 2;

		*valp = 0;
		*valp |= cpssp->NAME.ccr[n] << 0;
		break;
	default:
	undefined:;
	write_only:;
	warn:	;
		*valp = 0;
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(tick2)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s cnt=0x%04x\n", __FUNCTION__,
				cpssp->NAME.cnt);
	}

	if (cpssp->NAME.dir) {
		/* Downcounter */
		if (cpssp->NAME.cnt == 0) {
			NAME_(update_event)(cpssp);
			cpssp->NAME.cnt = cpssp->NAME.arr_shadow;
		} else {
			cpssp->NAME.cnt--;
			if (cpssp->NAME.cnt == 0
			 && cpssp->NAME.cms != 0b00) {
				/* Interrupt? - FIXME */
				NAME_(update_event)(cpssp);
				cpssp->NAME.dir = 0;
			}
		}
	} else {
		/* Upcounter */
		if (cpssp->NAME.cnt == cpssp->NAME.arr_shadow) {
			NAME_(update_event)(cpssp);
			cpssp->NAME.cnt = 0;
		} else {
			cpssp->NAME.cnt++;
			if (cpssp->NAME.cnt == cpssp->NAME.arr_shadow
			 && cpssp->NAME.cms != 0b00) {
				/* Interrupt? - FIXME */
				NAME_(update_event)(cpssp);
				cpssp->NAME.dir = 1;
			}
		}
	}
}

static void
NAME_(tick)(struct cpssp *cpssp)
{
	if (cpssp->NAME.cen) {
		if (cpssp->NAME.prescaler_count == cpssp->NAME.psc) {
			cpssp->NAME.prescaler_count = 0;
			NAME_(tick2)(cpssp);
		} else {
			cpssp->NAME.prescaler_count++;
		}
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int n;

	cpssp->NAME.prescaler_count = 0;

	/* control register 1 (TIMx_CR1), 13.4.1, 342/836 */
	cpssp->NAME.ckd = 0;
	cpssp->NAME.arpe = 0;
	cpssp->NAME.cms = 0;
	cpssp->NAME.dir = 0;
	cpssp->NAME.opm = 0;
	cpssp->NAME.urs = 0;
	cpssp->NAME.udis = 0;
	cpssp->NAME.cen = 0;

	/* control register 2 (TIMx_CR1), 13.4.2, 344/836 */
	cpssp->NAME.ti1s = 0;
	cpssp->NAME.mms = 0;
	cpssp->NAME.ccds = 0;

	/* event generation register (TIMx_CR1), 13.4.6, 351/836 */
	/* Nothing... */

	/* capture/compare mode register 1-2 (TIMx_CCMRy), 13.4.7-8, 352/836 */
	for (n = 0; n < 4; n++) {
		cpssp->NAME.occe[n] = 0;
		cpssp->NAME.ocm[n] = 0;
		cpssp->NAME.ocpe[n] = 0;
		cpssp->NAME.ocfe[n] = 0;
		cpssp->NAME.ccs[n] = 0;
	}

	/* counter (TIMx_CNT), 13.4.10, 358/836 */
	cpssp->NAME.cnt = 0x0000;

	/* prescaler (TIMx_PSC), 13.4.11, 358/836 */
	cpssp->NAME.psc = 0x0000;
	cpssp->NAME.psc_shadow = 0x0000;

	/* auto-reload register (TIMx_ARR), 13.4.12, 358/836 */
	cpssp->NAME.arr = 0x0000;
	cpssp->NAME.arr_shadow = 0x0000;

	/* capture/compare register 1-4 (TIMx_CCR1-4), 13.4.13-16, 358/836 */
	for (n = 0; n < 4; n++) {
		cpssp->NAME.ccr[n] = 0;
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
