/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* Reset and clock control (RCC) */

/*
 * For most of the comments, infos, ... see:
 * ST - RM0383 - Reference Manual - STM32F411xC/E advanced ARM-based
 * 32-bit MCUs
 */

#define DEBUG_CONTROL_FLOW	1

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	/* Clock control register (RCC_CR) */
	uint8_t plli2son;
	uint8_t pllon;
	uint8_t csson;
	uint8_t hsebyp;
	uint8_t hseon;
	uint8_t hsical;
	uint8_t hsitrim;
	uint8_t hsion;

	/* PLL configuration register (RCC_PLLCFGR) */
	uint8_t pllq;
	uint8_t pllsrc;
	uint8_t pllp;
	uint16_t plln;
	uint8_t pllm;

	/* Clock configuration register (RCC_CFGR) */
	uint8_t mco2;
	uint8_t mco2pre;
	uint8_t mco1pre;
	uint8_t i2sscr;
	uint8_t mco1;
	uint8_t rtcpre;
	uint8_t ppre2;
	uint8_t ppre1;
	uint8_t hpre;
	uint8_t sws;
	uint8_t sw;

	/* Clock interrupt register (RCC_CIR) */
	uint8_t plli2srdyie;
	uint8_t pllrdyie;
	uint8_t hserdyie;
	uint8_t hsirdyie;
	uint8_t lserdyie;
	uint8_t lsirdyie;
	uint8_t cssf;
	uint8_t plli2srdyf;
	uint8_t pllrdyf;
	uint8_t hserdyf;
	uint8_t hsirdyf;
	uint8_t lserdyf;
	uint8_t lsirdyf;

	/* AHB1 peripheral clock enable register (RCC_AHB1ENR) */
	uint8_t dma2en;
	uint8_t dma1en;
	uint8_t crcen;
	uint8_t gpiohen;
	uint8_t gpioeen;
	uint8_t gpioden;
	uint8_t gpiocen;
	uint8_t gpioben;
	uint8_t gpioaen;

	/* APB1 peripheral clock enable register (RCC_APB1ENR) */
	uint8_t pwren;
	uint8_t i2c3en;
	uint8_t i2c2en;
	uint8_t i2c1en;
	uint8_t usart2en;
	uint8_t spi3en;
	uint8_t spi2en;
	uint8_t wwdgen;
	uint8_t tim5en;
	uint8_t tim4en;
	uint8_t tim3en;
	uint8_t tim2en;

	/* APB2 peripheral clock enable register (RCC_APB1ENR) */
	uint8_t spi5en;
	uint8_t tim11en;
	uint8_t tim10en;
	uint8_t tim9en;
	uint8_t syscfgen;
	uint8_t spi4en;
	uint8_t spi1en;
	uint8_t sdioen;
	uint8_t adc1en;
	uint8_t usart6en;
	uint8_t usart1en;
	uint8_t tim1en;

	/* PLLI2S configuration register (RCC_PLLI2SCFGR) */
	uint8_t plli2sr;
	uint16_t plli2sn;
	uint8_t plli2sm;

	uint32_t reg[0x1000 >> 2];
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

static void
NAME_(irq_update)(struct cpssp *cpssp)
{
	int irq;

	irq = cpssp->NAME.cssf;
	irq |= cpssp->NAME.plli2srdyf & cpssp->NAME.plli2srdyie;
	irq |= cpssp->NAME.pllrdyf & cpssp->NAME.plli2srdyie;
	irq |= cpssp->NAME.hserdyf & cpssp->NAME.hserdyie;
	irq |= cpssp->NAME.hsirdyf & cpssp->NAME.hsirdyie;
	irq |= cpssp->NAME.lserdyf & cpssp->NAME.lserdyie;
	irq |= cpssp->NAME.lsirdyf & cpssp->NAME.lsirdyie;

	// NAME_(irq_out_set)(cpssp, irq);
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x3ff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		/* Clock control register (RCC_CR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-28: Reserved */
			/* Bit 27: Read-only */
			cpssp->NAME.plli2son = (val >> 26) & 1;
			/* Bit 25: Read-only */
			cpssp->NAME.pllon = (val >> 24) & 1;
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-20: Reserved */
			cpssp->NAME.csson = (val >> 19) & 1;
			cpssp->NAME.hsebyp = (val >> 18) & 1;
			/* Bit 17: Read-only */
			cpssp->NAME.hseon = (val >> 16) & 1;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.hsical = (val >> 8) & 0xff;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.hsitrim = (val >> 3) & 0x1f;
			/* Bit 2: Reserved */
			/* Bit 1: Read-only */
			cpssp->NAME.hsion = (val >> 0) & 1;
		}
		break;
	case 0x004:
		/* PLL configuration register (RCC_PLLCFGR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-28: Reserved */
			cpssp->NAME.pllq = (val >> 24) & 0xf;
		}
		if ((bs >> 2) & 1) {
			/* Bit 23: Reserved */
			cpssp->NAME.pllsrc = (val >> 22) & 1;
			/* Bit 21-18: Reserved */
			cpssp->NAME.pllp = (val >> 16) & 0x3;
		}
		if ((bs >> 1) & 1) {
			/* Bit 15: Reserved */
			cpssp->NAME.plln &= ~(0x7f << 2);
			cpssp->NAME.plln |= ((val >> 8) & 0x7f) << 2;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.plln &= ~3;
			cpssp->NAME.plln |= (val >> 6) & 0x3;
			cpssp->NAME.pllm = (val >> 0) & 0x3f;
		}
		break;
	case 0x008:
		/* Clock configuration register (RCC_CFGR) */
		if ((bs >> 3) & 1) {
			cpssp->NAME.mco2 = (val >> 30) & 0x3;
			cpssp->NAME.mco2pre = (val >> 27) & 0x7;
			cpssp->NAME.mco1pre = (val >> 24) & 0x7;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.i2sscr = (val >> 23) & 1;
			cpssp->NAME.mco1 = (val >> 21) & 0x3;
			cpssp->NAME.rtcpre = (val >> 16) & 0x1f;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.ppre2 = (val >> 13) & 0x7;
			cpssp->NAME.ppre1 = (val >> 10) & 0x7;
			/* Bit 9-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.hpre = (val >> 4) & 0xf;
			/* Bit 3-2: Read-only */
			cpssp->NAME.sw = (val >> 0) & 0x3;
		}
		break;
	case 0x00c:
		/* Clock interrupt register (RCC_CIR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.cssf &= ~((val >> 23) & 1);
			/* Bit 22: Reserved */
			cpssp->NAME.plli2srdyf &= ~((val >> 21) & 1);
			cpssp->NAME.pllrdyf &= ~((val >> 20) & 1);
			cpssp->NAME.hserdyf &= ~((val >> 19) & 1);
			cpssp->NAME.hsirdyf &= ~((val >> 18) & 1);
			cpssp->NAME.lserdyf &= ~((val >> 17) & 1);
			cpssp->NAME.lsirdyf &= ~((val >> 16) & 1);
		}
		if ((bs >> 1) & 1) {
			/* Bit 15-14: Reserved */
			cpssp->NAME.plli2srdyie = (val >> 13) & 1;
			cpssp->NAME.pllrdyie = (val >> 12) & 1;
			cpssp->NAME.hserdyie = (val >> 11) & 1;
			cpssp->NAME.hsirdyie = (val >> 10) & 1;
			cpssp->NAME.lserdyie = (val >> 9) & 1;
			cpssp->NAME.lsirdyie = (val >> 8) & 1;
		}
		if ((bs >> 0) & 1) {
			/* Bit 7: Read-only */
			/* Bit 6: Reserved */
			/* Bit 5-0: Read-only */
		}
		NAME_(irq_update)(cpssp);
		break;
	case 0x030:
		/* AHB1 peripheral clock enable register (RCC_AHB1ENR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 23: Reserved */
			cpssp->NAME.dma2en = (val >> 22) & 1;
			cpssp->NAME.dma1en = (val >> 21) & 1;
			/* Bit 20-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Bit 15-13: Reserved */
			cpssp->NAME.crcen = (val >> 12) & 1;
			/* Bit 11-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.gpiohen = (val >> 7) & 1;
			/* Bit 6-5: Reserved */
			cpssp->NAME.gpioeen = (val >> 4) & 1;
			cpssp->NAME.gpioden = (val >> 3) & 1;
			cpssp->NAME.gpiocen = (val >> 2) & 1;
			cpssp->NAME.gpioben = (val >> 1) & 1;
			cpssp->NAME.gpioaen = (val >> 0) & 1;
		}
		break;
	case 0x040:
		/* APB1 peripheral clock enable register (RCC_APB1ENR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-29: Reserved */
			cpssp->NAME.pwren = (val >> 28) & 1;
			/* Bit 27-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.i2c3en = (val >> 23) & 1;
			cpssp->NAME.i2c2en = (val >> 22) & 1;
			cpssp->NAME.i2c1en = (val >> 21) & 1;
			/* Bit 20-18: Reserved */
			cpssp->NAME.usart2en = (val >> 17) & 1;
			/* Bit 16: Reserved */
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.spi3en = (val >> 15) & 1;
			cpssp->NAME.spi2en = (val >> 14) & 1;
			/* Bit 13-12: Reserved */
			cpssp->NAME.wwdgen = (val >> 11) & 1;
			/* Bit 10-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			/* Bit 7-4: Reserved */
			cpssp->NAME.tim5en = (val >> 3) & 1;
			cpssp->NAME.tim4en = (val >> 2) & 1;
			cpssp->NAME.tim3en = (val >> 1) & 1;
			cpssp->NAME.tim2en = (val >> 0) & 1;
		}
		break;
	case 0x044:
		/* APB2 peripheral clock enable register (RCC_APB1ENR) */
		if ((bs >> 3) & 1) {
			/* Bit 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-21: Reserved */
			cpssp->NAME.spi5en = (val >> 20) & 1;
			/* Bit 19: Reserved */
			cpssp->NAME.tim11en = (val >> 18) & 1;
			cpssp->NAME.tim10en = (val >> 17) & 1;
			cpssp->NAME.tim9en = (val >> 16) & 1;
		}
		if ((bs >> 1) & 1) {
			/* Bit 15: Reserved */
			cpssp->NAME.syscfgen = (val >> 14) & 1;
			cpssp->NAME.spi4en = (val >> 13) & 1;
			cpssp->NAME.spi1en = (val >> 12) & 1;
			cpssp->NAME.sdioen = (val >> 11) & 1;
			/* Bit 10-9: Reserved */
			cpssp->NAME.adc1en = (val >> 8) & 1;
		}
		if ((bs >> 0) & 1) {
			/* Bit 7-6: Reserved */
			cpssp->NAME.usart6en = (val >> 5) & 1;
			cpssp->NAME.usart1en = (val >> 4) & 1;
			/* Bit 3-1: Reserved */
			cpssp->NAME.tim1en = (val >> 0) & 1;
		}
		break;
	case 0x84:
		/* PLLI2S configuration register (RCC_PLLI2SCFGR) */
		if ((bs >> 3) & 1) {
			/* Bit 31: Reserved */
			cpssp->NAME.plli2sr = (val >> 28) & 0x7;
			/* Bit 27-14: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Bit 15: Reserved */
			cpssp->NAME.plli2sn &= ~(0x7f << 2);
			cpssp->NAME.plli2sn |= ((val >> 8) & 0x7f) << 2;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.plli2sn &= ~3;
			cpssp->NAME.plli2sn |= (val >> 6) & 3;
			cpssp->NAME.plli2sm = (val >> 0) & 0x3f;
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);

		cpssp->NAME.reg[addr >> 2] = val;
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x3ff;

	switch (addr) {
	case 0x000:
		/* Clock control register (RCC_CR) */
		*valp = 0;

		/* Bit 31-28: Reserved */
		*valp |= 1 << 27; /* pllrdy */
		*valp |= cpssp->NAME.plli2son << 26;
		*valp |= 1 << 25; /* pllrdy */
		*valp |= cpssp->NAME.pllon << 24;

		/* Bit 23-20: Reserved */
		*valp |= cpssp->NAME.csson << 19;
		*valp |= cpssp->NAME.hsebyp << 18;
		*valp |= 1 << 17; /* hserdy */
		*valp |= cpssp->NAME.hseon << 16;

		*valp |= cpssp->NAME.hsical << 8;

		*valp |= cpssp->NAME.hsitrim << 3;
		/* Bit 2: Reserved */
		*valp |= 1 << 1; /* hsirdy */
		*valp |= cpssp->NAME.hsion << 0;
		break;
	case 0x004:
		/* PLL configuration register (RCC_PLLCFGR) */
		*valp = 0;

		/* Bit 31-28: Reserved */
		*valp |= cpssp->NAME.pllq << 24;

		/* Bit 23: Reserved */
		*valp |= cpssp->NAME.pllsrc << 22;
		/* Bit 21-18: Reserved */
		*valp |= cpssp->NAME.pllp << 16;

		/* Bit 15: Reserved */
		*valp |= cpssp->NAME.plln << 6;

		*valp |= cpssp->NAME.pllm << 0;
		break;
	case 0x008:
		/* Clock configuration register (RCC_CFGR) */
		*valp = 0;

		*valp |= cpssp->NAME.mco2 << 30;
		*valp |= cpssp->NAME.mco2pre << 27;
		*valp |= cpssp->NAME.mco1pre << 24;

		*valp |= cpssp->NAME.i2sscr << 23;
		*valp |= cpssp->NAME.mco1 << 21;
		*valp |= cpssp->NAME.rtcpre << 16;

		*valp |= cpssp->NAME.ppre2 << 13;
		*valp |= cpssp->NAME.ppre1 << 10;
		/* Bit 9-8: Reserved */

		*valp |= cpssp->NAME.hpre << 4;
		*valp |= cpssp->NAME.sw << 2; /* sws */
		*valp |= cpssp->NAME.sw << 0;
		break;
	case 0x00c:
		/* Clock interrupt register (RCC_CIR) */
		*valp = 0;

		/* Bit 31-24: Reserved */

		/* Bit 23: Write-only */
		/* Bit 22: Reserved */
		/* Bit 21-16: Write-only */

		/* Bit 15-14: Reserved */
		*valp |= cpssp->NAME.plli2srdyie << 13;
		*valp |= cpssp->NAME.pllrdyie << 12;
		*valp |= cpssp->NAME.hserdyie << 11;
		*valp |= cpssp->NAME.hsirdyie << 10;
		*valp |= cpssp->NAME.lserdyie << 9;
		*valp |= cpssp->NAME.lsirdyie << 8;

		*valp |= cpssp->NAME.cssf << 7;
		/* Bit 6: Reserved */
		*valp |= cpssp->NAME.plli2srdyf << 5;
		*valp |= cpssp->NAME.pllrdyf << 4;
		*valp |= cpssp->NAME.hserdyf << 3;
		*valp |= cpssp->NAME.hsirdyf << 2;
		*valp |= cpssp->NAME.lserdyf << 1;
		*valp |= cpssp->NAME.lsirdyf << 0;
		break;
	case 0x030:
		/* AHB1 peripheral clock enable register (RCC_AHB1ENR) */
		*valp = 0;

		/* Bit 31-24: Reserved */

		/* Bit 23: Reserved */
		*valp |= cpssp->NAME.dma2en << 22;
		*valp |= cpssp->NAME.dma1en << 21;
		/* Bit 20-16: Reserved */

		/* Bit 15-13: Reserved */
		*valp |= cpssp->NAME.crcen << 12;
		/* Bit 11-8: Reserved */

		*valp |= cpssp->NAME.gpiohen << 7;
		/* Bit 15-13: Reserved */
		*valp |= cpssp->NAME.gpioeen << 4;
		*valp |= cpssp->NAME.gpioden << 3;
		*valp |= cpssp->NAME.gpiocen << 2;
		*valp |= cpssp->NAME.gpioben << 1;
		*valp |= cpssp->NAME.gpioaen << 0;
		break;
	case 0x040:
		/* APB1 peripheral clock enable register (RCC_APB1ENR) */
		*valp = 0;

		/* Bit 31-29: Reserved */
		*valp |= cpssp->NAME.pwren << 28;
		/* Bit 27-24: Reserved */

		*valp |= cpssp->NAME.i2c3en << 23;
		*valp |= cpssp->NAME.i2c2en << 22;
		*valp |= cpssp->NAME.i2c1en << 21;
		/* Bit 20-18: Reserved */
		*valp |= cpssp->NAME.usart2en << 17;
		/* Bit 16: Reserved */

		*valp |= cpssp->NAME.spi3en << 15;
		*valp |= cpssp->NAME.spi2en << 14;
		/* Bit 13-12: Reserved */
		*valp |= cpssp->NAME.wwdgen << 11;
		/* Bit 10-8: Reserved */

		/* Bit 7-4: Reserved */
		*valp |= cpssp->NAME.tim5en << 3;
		*valp |= cpssp->NAME.tim4en << 2;
		*valp |= cpssp->NAME.tim3en << 1;
		*valp |= cpssp->NAME.tim2en << 0;
		break;
	case 0x044:
		/* APB2 peripheral clock enable register (RCC_APB1ENR) */
		*valp = 0;

		/* Bit 31-24: Reserved */

		/* Bit 23-21: Reserved */
		*valp |= cpssp->NAME.spi5en << 20;
		/* Bit 19: Reserved */
		*valp |= cpssp->NAME.tim11en << 18;
		*valp |= cpssp->NAME.tim10en << 17;
		*valp |= cpssp->NAME.tim9en << 16;

		/* Bit 15: Reserved */
		*valp |= cpssp->NAME.syscfgen << 14;
		*valp |= cpssp->NAME.spi4en << 13;
		*valp |= cpssp->NAME.spi1en << 12;
		*valp |= cpssp->NAME.sdioen << 11;
		/* Bit 10-9: Reserved */
		*valp |= cpssp->NAME.adc1en << 8;

		/* Bit 7-6: Reserved */
		*valp |= cpssp->NAME.usart6en << 5;
		*valp |= cpssp->NAME.usart1en << 4;
		/* Bit 3-1: Reserved */
		*valp |= cpssp->NAME.tim1en << 0;
		break;
	case 0x84:
		/* PLLI2S configuration register (RCC_PLLI2SCFGR) */
		*valp = 0;

		/* Bit 31: Reserved */
		*valp |= cpssp->NAME.plli2sr << 28;
		/* Bit 27-24: Reserved */

		/* Bit 23-16: Reserved */

		/* Bit 15: Reserved */
		*valp |= cpssp->NAME.plli2sn << 6;

		*valp |= cpssp->NAME.plli2sm << 0;
		break;
	default:
		*valp = cpssp->NAME.reg[addr >> 2];

		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* Clock control register (RCC_CR) */
	cpssp->NAME.plli2son = 0;
	cpssp->NAME.pllon = 0;
	cpssp->NAME.hsebyp = 0;
	cpssp->NAME.hseon = 0;
	cpssp->NAME.hsical = 0x00;
	cpssp->NAME.hsitrim = 0x10;
	cpssp->NAME.hsion = 1;

	/* PLL configuration register (RCC_PLLCFGR) */
	cpssp->NAME.pllq = 0x4;
	cpssp->NAME.pllsrc = 0;
	cpssp->NAME.pllp = 0x0;
	cpssp->NAME.plln = 0x30 << 2;
	cpssp->NAME.pllm = 0x10;

	/* Clock configuration register (RCC_CFGR) */
	cpssp->NAME.mco2 = 0x0;
	cpssp->NAME.mco2pre = 0x0;
	cpssp->NAME.mco1pre = 0x0;
	cpssp->NAME.i2sscr = 0;
	cpssp->NAME.mco1 = 0x0;
	cpssp->NAME.rtcpre = 0x00;
	cpssp->NAME.ppre2 = 0x0;
	cpssp->NAME.ppre1 = 0x0;
	cpssp->NAME.hpre = 0x0;
	cpssp->NAME.sw = 0x0;

	/* Clock interrupt register (RCC_CIR) */
	cpssp->NAME.plli2srdyie = 0;
	cpssp->NAME.pllrdyie = 0;
	cpssp->NAME.hserdyie = 0;
	cpssp->NAME.hsirdyie = 0;
	cpssp->NAME.lserdyie = 0;
	cpssp->NAME.lsirdyie = 0;
	cpssp->NAME.cssf = 0;
	cpssp->NAME.plli2srdyf = 0;
	cpssp->NAME.pllrdyf = 0;
	cpssp->NAME.hserdyf = 0;
	cpssp->NAME.hsirdyf = 0;
	cpssp->NAME.lserdyf = 0;
	cpssp->NAME.lsirdyf = 0;

	/* AHB1 peripheral clock enable register (RCC_AHB1ENR) */
	cpssp->NAME.dma2en = 0;
	cpssp->NAME.dma1en = 0;
	cpssp->NAME.crcen = 0;
	cpssp->NAME.gpiohen = 0;
	cpssp->NAME.gpioeen = 0;
	cpssp->NAME.gpioden = 0;
	cpssp->NAME.gpiocen = 0;
	cpssp->NAME.gpioben = 0;
	cpssp->NAME.gpioaen = 0;

	/* APB1 peripheral clock enable register (RCC_APB1ENR) */
	cpssp->NAME.pwren = 0;
	cpssp->NAME.i2c3en = 0;
	cpssp->NAME.i2c2en = 0;
	cpssp->NAME.i2c1en = 0;
	cpssp->NAME.usart2en = 0;
	cpssp->NAME.spi3en = 0;
	cpssp->NAME.spi2en = 0;
	cpssp->NAME.wwdgen = 0;
	cpssp->NAME.tim5en = 0;
	cpssp->NAME.tim4en = 0;
	cpssp->NAME.tim3en = 0;
	cpssp->NAME.tim2en = 0;

	/* APB2 peripheral clock enable register (RCC_APB1ENR) */
	cpssp->NAME.spi5en = 0;
	cpssp->NAME.tim11en = 0;
	cpssp->NAME.tim10en = 0;
	cpssp->NAME.tim9en = 0;
	cpssp->NAME.syscfgen = 0;
	cpssp->NAME.spi4en = 0;
	cpssp->NAME.spi1en = 0;
	cpssp->NAME.adc1en = 0;
	cpssp->NAME.usart6en = 0;
	cpssp->NAME.usart1en = 0;
	cpssp->NAME.tim1en = 0;

	/* PLLI2S configuration register (RCC_PLLI2SCFGR) */
	cpssp->NAME.plli2sr = 2;
	cpssp->NAME.plli2sn = 3 << 6;
	cpssp->NAME.plli2sm = 0;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
