/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* Inter-integrated circuit interface (I2C) */

/*
 * For most of the comments, infos, ... see:
 * ST - RM0383 - Reference Manual - STM32F411xC/E advanced ARM-based
 * 32-bit MCUs
 */

#define DEBUG_CONTROL_FLOW	1

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	unsigned int state_sda;

	/* Control register 1 (I2C_CR1), 483/836 */
	uint8_t swrst;
	uint8_t alert;
	uint8_t pec;
	uint8_t pos;
	uint8_t ack;
	uint8_t stop;
	uint8_t start;
	uint8_t nostretch;
	uint8_t engc;
	uint8_t enpec;
	uint8_t enarp;
	uint8_t smbtype;
	uint8_t smbus;
	uint8_t pe;

	/* Control register 2 (I2C_CR2), 485/836 */
	uint8_t last;
	uint8_t dmaen;
	uint8_t itbufen;
	uint8_t itevten;
	uint8_t iterren;
	uint8_t freq;

	/* Own address register 1 (I2C_OAR1), 487/836 */
	uint8_t admode;
	uint16_t add;

	/* Own address register 2 (I2C_OAR2), 487/836 */
	uint8_t add2;
	uint8_t endual;

	/* Data register (I2C_DR), 488/836 */

	/* Status register 1 (I2C_SR1), 488/836 */
	uint8_t smbalert;
	uint8_t timeout;
	uint8_t pecerr;
	uint8_t ovr;
	uint8_t af;
	uint8_t arlo;
	uint8_t berr;
	uint8_t txe;
	uint8_t rxne;
	uint8_t stopf;
	uint8_t add10;
	uint8_t btf;
	uint8_t addr;
	uint8_t sb;

	/* Status register 2 (I2C_SR2), 492/836 */
	uint8_t pec_sr;
	uint8_t dualf;
	uint8_t smbhost;
	uint8_t smbdefault;
	uint8_t gencall;
	uint8_t tra;
	uint8_t busy;
	uint8_t msl;

	/* Clock control register (I2C_CCR), 493/836 */
	uint8_t fs;
	uint8_t duty;
	uint16_t ccr;

	/* TRISE register (I2C_TRISE), 494/836 */
	uint8_t trise;

	uint32_t reg[0x1000 >> 2];
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* Control register 1 (I2C_CR1), 483/836 */
	cpssp->NAME.swrst = 0;
	cpssp->NAME.alert = 0;
	cpssp->NAME.pec = 0;
	cpssp->NAME.pos = 0;
	cpssp->NAME.ack = 0;
	cpssp->NAME.stop = 0;
	cpssp->NAME.start = 0;
	cpssp->NAME.nostretch = 0;
	cpssp->NAME.engc = 0;
	cpssp->NAME.enpec = 0;
	cpssp->NAME.enarp = 0;
	cpssp->NAME.smbtype = 0;
	cpssp->NAME.smbus = 0;
	cpssp->NAME.pe = 0;

	/* Control register 2 (I2C_CR2), 485/836 */
	cpssp->NAME.last = 0;
	cpssp->NAME.dmaen = 0;
	cpssp->NAME.itbufen = 0;
	cpssp->NAME.itevten = 0;
	cpssp->NAME.iterren = 0;
	cpssp->NAME.freq = 0;

	/* Own address register 1 (I2C_OAR1), 487/836 */
	cpssp->NAME.admode = 0;
	cpssp->NAME.add = 0;

	/* Own address register 2 (I2C_OAR2), 487/836 */
	cpssp->NAME.add2 = 0;
	cpssp->NAME.endual = 0;

	/* Data register (I2C_DR), 488/836 */

	/* Status register 1 (I2C_SR1), 488/836 */
	cpssp->NAME.smbalert = 0;
	cpssp->NAME.timeout = 0;
	cpssp->NAME.pecerr = 0;
	cpssp->NAME.ovr = 0;
	cpssp->NAME.af = 0;
	cpssp->NAME.arlo = 0;
	cpssp->NAME.berr = 0;
	cpssp->NAME.txe = 0;
	cpssp->NAME.rxne = 0;
	cpssp->NAME.stopf = 0;
	cpssp->NAME.add10 = 0;
	cpssp->NAME.btf = 0;
	cpssp->NAME.addr = 0;
	cpssp->NAME.sb = 0;

	/* Status register 2 (I2C_SR2), 492/836 */
	cpssp->NAME.pec_sr = 0;
	cpssp->NAME.dualf = 0;
	cpssp->NAME.smbhost = 0;
	cpssp->NAME.smbdefault = 0;
	cpssp->NAME.gencall = 0;
	cpssp->NAME.tra = 0;
	cpssp->NAME.busy = 0;
	cpssp->NAME.msl = 0;

	/* Clock control register (I2C_CCR), 493/836 */
	cpssp->NAME.fs = 0;
	cpssp->NAME.duty = 0;
	cpssp->NAME.ccr = 0;

	/* TRISE register (I2C_TRISE), 494/836 */
	cpssp->NAME.trise = 0x2;

	NAME_(scl_out_set)(cpssp, SIG_STD_LOGIC_1);
	NAME_(sda_out_set)(cpssp, SIG_STD_LOGIC_1);
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	assert((bs & 0b11) == 0b11);

	addr &= 0x3ff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		/* Control register 1 (I2C_CR1), 483/836 */
		cpssp->NAME.swrst = (val >> 15) & 1;
		/* Bit 14: Reserved */
		cpssp->NAME.alert = (val >> 13) & 1;
		cpssp->NAME.pec = (val >> 12) & 1;
		cpssp->NAME.pos = (val >> 11) & 1;
		cpssp->NAME.ack = (val >> 10) & 1;
		cpssp->NAME.stop = (val >> 9) & 1;
		cpssp->NAME.start = (val >> 8) & 1;
		cpssp->NAME.nostretch = (val >> 7) & 1;
		cpssp->NAME.engc = (val >> 6) & 1;
		cpssp->NAME.enpec = (val >> 5) & 1;
		cpssp->NAME.enarp = (val >> 4) & 1;
		cpssp->NAME.smbtype = (val >> 3) & 1;
		/* Bit 2: Reserved */
		cpssp->NAME.smbus = (val >> 1) & 1;
		cpssp->NAME.pe = (val >> 0) & 1;

		/* FIXME */
		if (cpssp->NAME.start) {
			cpssp->NAME.msl = 1; /* Master Mode */
			cpssp->NAME.sb = 1; /* Start Bit Generated */
			cpssp->NAME.busy = 1;
			cpssp->NAME.txe = 1;
		}
		break;
	case 0x04:
		/* Control register 2 (I2C_CR2), 485/836 */
		/* Bit 15-13: Reserved */
		cpssp->NAME.last = (val >> 12) & 1;
		cpssp->NAME.dmaen = (val >> 11) & 1;
		cpssp->NAME.itbufen = (val >> 10) & 1;
		cpssp->NAME.itevten = (val >> 9) & 1;
		cpssp->NAME.iterren = (val >> 8) & 1;
		/* Bit 7-6: Reserved */
		cpssp->NAME.freq = (val >> 0) & 0x3f;
		break;
	case 0x08:
		/* Own address register 1 (I2C_OAR1), 487/836 */
		cpssp->NAME.admode = (val >> 15) & 1;
		/* Bit 14-10: Reserved */
		cpssp->NAME.add = (val >> 0) & 0x3ff;
		break;
	case 0x0c:
		/* Own address register 2 (I2C_OAR2), 487/836 */
		/* Bit 15-8: Reserved */
		cpssp->NAME.add2 = (val >> 1) & 0x7f;
		cpssp->NAME.endual = (val >> 0) & 1;
		break;
	case 0x10:
		/* Data register (I2C_DR), 488/836 */
		/* FIXME */
		cpssp->NAME.addr = 1;
		cpssp->NAME.tra = 1;
		cpssp->NAME.btf = 1;
		cpssp->NAME.rxne = 1;
		break;
	case 0x14:
		/* Status register 1 (I2C_SR1), 488/836 */
		cpssp->NAME.smbalert &= (val >> 15) & 1;
		cpssp->NAME.timeout &= (val >> 14) & 1;
		/* Bit 13: Reserved */
		cpssp->NAME.pecerr &= (val >> 12) & 1;
		cpssp->NAME.ovr &= (val >> 11) & 1;
		cpssp->NAME.af &= (val >> 10) & 1;
		cpssp->NAME.arlo &= (val >> 9) & 1;
		cpssp->NAME.berr &= (val >> 8) & 1;
		/* Bit 7-6: Read-only */
		/* Bit 5: Reserved */
		/* Bit 4-0: Read-only */
		break;
	case 0x18:
		/* Status register 2 (I2C_SR2), 492/836 */
		cpssp->NAME.pec_sr = (val >> 8) & 0xff;
		cpssp->NAME.dualf = (val >> 7) & 1;
		cpssp->NAME.smbhost = (val >> 6) & 1;
		cpssp->NAME.smbdefault = (val >> 5) & 1;
		cpssp->NAME.gencall = (val >> 4) & 1;
		/* Bit 3: Reserved */
		cpssp->NAME.tra = (val >> 2) & 1;
		cpssp->NAME.busy = (val >> 1) & 1;
		cpssp->NAME.msl = (val >> 0) & 1;
		break;
	case 0x1c:
		/* Clock control register (I2C_CCR), 493/836 */
		cpssp->NAME.fs = (val >> 15) & 1;
		cpssp->NAME.duty = (val >> 14) & 1;
		/* Bit 13-12: Reserved */
		cpssp->NAME.ccr = (val >> 0) & 0xfff;
		break;
	case 0x20:
		/* TRISE register (I2C_TRISE), 494/836 */
		/* Bit 15-6: Reserved */
		cpssp->NAME.trise = (val >> 0) & 0x3f;
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);

		cpssp->NAME.reg[addr >> 2] = val;
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	assert((bs & 0b11) == 0b11);

	addr &= 0x3ff;

	switch (addr) {
	case 0x00:
		/* Control register 1 (I2C_CR1), 483/836 */
		*valp = 0;
		*valp |= cpssp->NAME.swrst << 15;
		/* Bit 14: Reserved */
		*valp |= cpssp->NAME.alert << 13;
		*valp |= cpssp->NAME.pec << 12;
		*valp |= cpssp->NAME.pos << 11;
		*valp |= cpssp->NAME.ack << 10;
		*valp |= cpssp->NAME.stop << 9;
		*valp |= cpssp->NAME.start << 8;
		*valp |= cpssp->NAME.nostretch << 7;
		*valp |= cpssp->NAME.engc << 6;
		*valp |= cpssp->NAME.enpec << 5;
		*valp |= cpssp->NAME.enarp << 4;
		*valp |= cpssp->NAME.smbtype << 3;
		/* Bit 2: Reserved */
		*valp |= cpssp->NAME.smbus << 1;
		*valp |= cpssp->NAME.pe << 0;
		break;
	case 0x04:
		/* Control register 2 (I2C_CR2), 485/836 */
		*valp = 0;
		/* Bit 15-13: Reserved */
		*valp |= cpssp->NAME.last << 12;
		*valp |= cpssp->NAME.dmaen << 11;
		*valp |= cpssp->NAME.itbufen << 10;
		*valp |= cpssp->NAME.itevten << 9;
		*valp |= cpssp->NAME.iterren << 8;
		/* Bit 7-6: Reserved */
		*valp |= cpssp->NAME.freq << 0;
		break;
	case 0x08:
		/* Own address register 1 (I2C_OAR1), 487/836 */
		*valp = 0;
		*valp |= cpssp->NAME.admode << 15;
		/* Bit 14-10: Reserved */
		*valp |= cpssp->NAME.add << 0;
		break;
	case 0x0c:
		/* Own address register 2 (I2C_OAR2), 487/836 */
		*valp = 0;
		/* Bit 15-8: Reserved */
		*valp |= cpssp->NAME.add2 << 1;
		*valp |= cpssp->NAME.endual << 0;
		break;
	case 0x10:
		/* Data register (I2C_DR), 488/836 */
		/* FIXME */
		*valp = 0;
		cpssp->NAME.stop = 0;
		break;
	case 0x14:
		/* Status register 1 (I2C_SR1), 488/836 */
		*valp = 0;
		*valp |= cpssp->NAME.smbalert << 15;
		*valp |= cpssp->NAME.timeout << 14;
		/* Bit 13: Reserved */
		*valp |= cpssp->NAME.pecerr << 12;
		*valp |= cpssp->NAME.ovr << 11;
		*valp |= cpssp->NAME.af << 10;
		*valp |= cpssp->NAME.arlo << 9;
		*valp |= cpssp->NAME.berr << 8;
		*valp |= cpssp->NAME.txe << 7;
		*valp |= cpssp->NAME.rxne << 6;
		/* Bit 5: Reserved */
		*valp |= cpssp->NAME.stopf << 4;
		*valp |= cpssp->NAME.add10 << 3;
		*valp |= cpssp->NAME.btf << 2;
		*valp |= cpssp->NAME.addr << 1;
		*valp |= cpssp->NAME.sb << 0;
		break;
	case 0x18:
		/* Status register 2 (I2C_SR2), 492/836 */
		*valp = 0;
		*valp |= cpssp->NAME.pec_sr << 8;
		*valp |= cpssp->NAME.dualf << 7;
		*valp |= cpssp->NAME.smbhost << 6;
		*valp |= cpssp->NAME.smbdefault << 5;
		*valp |= cpssp->NAME.gencall << 4;
		/* Bit 3: Reserved */
		*valp |= cpssp->NAME.tra << 2;
		*valp |= cpssp->NAME.busy << 1;
		*valp |= cpssp->NAME.msl << 0;
		break;
	case 0x1c:
		/* Clock control register (I2C_CCR), 493/836 */
		*valp = 0;
		*valp |= cpssp->NAME.fs << 15;
		*valp |= cpssp->NAME.duty << 14;
		/* Bit 13-12: Reserved */
		*valp |= cpssp->NAME.ccr << 0;
		break;
	case 0x20:
		/* TRISE register (I2C_TRISE), 494/836 */
		*valp = 0;
		/* Bit 15-6: Reserved */
		*valp |= cpssp->NAME.trise;
		break;
	default:
		*valp = cpssp->NAME.reg[addr >> 2];

		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(scl_in_set)(struct cpssp *cpssp, unsigned int val)
{
	/* Nothing to do... */
}

static void
NAME_(sda_in_set)(struct cpssp *cpssp, unsigned int val)
{
	switch (val) {
	case SIG_STD_LOGIC_0: val = 0; break;
	case SIG_STD_LOGIC_H: val = 1; break;
	default: val = 0; /* FIXME */ break;
	}

	cpssp->NAME.state_sda = val;
}

static void
NAME_(smba_in_set)(struct cpssp *cpssp, unsigned int val)
{
	/* FIXME */
}

static void
NAME_(clk)(struct cpssp *cpssp)
{
	if (cpssp->NAME.swrst
	 || ! cpssp->NAME.pe) {
		return;
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
