/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* General Purpose I/O (GPIO) */

/*
 * For most of the comments, infos, ... see:
 * ST - RM0383 - Reference Manual - STM32F411xC/E advanced ARM-based
 * 32-bit MCUs
 */

#define DEBUG_CONTROL_FLOW	1

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	unsigned int state_ain[16];
	uint16_t state_left_in[16];
	unsigned int state_right_out[16];
	unsigned int state_right_in[16];

	/* port mode register (GPIOx_MODER), 8.4.1, 153/836 */
	uint8_t moder[16];

	/* port output type register (GPIOx_OTYPER), 8.4.2, 154/836 */
	uint16_t otyper;

	/* port output speed register (GPIOx_OSPEEDR), 8.4.3, 155/836 */
	uint8_t ospeedr[16];

	/* port pull-up/pull-down register (GPIOx_PUPDR), 8.4.4, 156/836 */
	uint16_t pupdr_down;
	uint16_t pupdr_up;

	/* port input data register (GPIOx_IDR), 8.4.5, 156/836 */
	uint16_t idr;

	/* port output data register (GPIOx_ODR), 8.4.6, 157/836 */
	uint16_t odr;

	/* alternate function low/high register (GPIOx_AFRL/GPIOx_AFRH), 8.4.9/8.4.10, 159/836 */
	uint8_t afr[16];

	uint32_t reg[0x1000 >> 2];
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

static void
NAME_(rightN_resolve)(struct cpssp *cpssp, int n)
{
	unsigned int val;

	if ((cpssp->NAME.pupdr_down >> n) & 1) {
		val = SIG_STD_LOGIC_L;
	} else if ((cpssp->NAME.pupdr_up >> n) & 1) {
		val = SIG_STD_LOGIC_H;
	} else {
		val = SIG_STD_LOGIC_Z;
	}

	val = sig_std_logic_resolve(val, cpssp->NAME.state_right_out[n]);
	val = sig_std_logic_resolve(val, cpssp->NAME.state_right_in[n]);

	NAME_(rightN_out)(cpssp, n, val);

	NAME_(aN_out)(cpssp, n, val);

	/* Schmitt trigger */
	if (cpssp->NAME.moder[n] == 0b11) {
		/* Analog */
		val = 0;
	} else {
		val = 1000 < SIG_mV(val);
	}

	NAME_(leftNM_out_set)(cpssp, n, cpssp->NAME.afr[n], val);

	cpssp->NAME.idr &= ~(1 << n);
	cpssp->NAME.idr |= val << n;
}

static void
NAME_(output_update)(struct cpssp *cpssp)
{
	unsigned int n;
	unsigned int val;

	for (n = 0; n < 16; n++) {
		switch (cpssp->NAME.moder[n]) {
		case 0b00:
			/* Input */
			val = SIG_STD_LOGIC_Z;
			break;
		case 0b01:
			/* Output */
			val = (cpssp->NAME.odr >> n) & 1;
			goto out;
		case 0b10:
			/* Alternate function output */
			val = (cpssp->NAME.state_left_in[cpssp->NAME.afr[n]] >> n) & 1;
		out:	;
			if ((cpssp->NAME.otyper >> n) & 1) {
				/* Open-Drain */
				val = val ? SIG_STD_LOGIC_Z : SIG_STD_LOGIC_0;
			} else {
				/* Push/Pull */
				val = val ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0;
			}
			break;
		case 0b11:
			/* Analog mode */
			val = cpssp->NAME.state_ain[n];
			break;
		default: assert(0); /* Mustn't happen. */
		}

		cpssp->NAME.state_right_out[n] = val;
		NAME_(rightN_resolve)(cpssp, n);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int n;

	addr &= 0x3ff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	/*
	 * CPIO register map, 8.4.11, 160/836
	 */
	switch (addr) {
	case 0x00:
		/* port mode register (GPIOx_MODER), 8.4.1, 153/836 */
		if ((bs >> 3) & 1) {
			for (n = 12; n < 16; n++) {
				cpssp->NAME.moder[n] = (val >> 2 * n) & 0b11;
			}
		}
		if ((bs >> 2) & 1) {
			for (n = 8; n < 12; n++) {
				cpssp->NAME.moder[n] = (val >> 2 * n) & 0b11;
			}
		}
		if ((bs >> 1) & 1) {
			for (n = 4; n < 8; n++) {
				cpssp->NAME.moder[n] = (val >> 2 * n) & 0b11;
			}
		}
		if ((bs >> 0) & 1) {
			for (n = 0; n < 4; n++) {
				cpssp->NAME.moder[n] = (val >> 2 * n) & 0b11;
			}
		}
		NAME_(output_update)(cpssp);
		break;

	case 0x04:
		/* port output type register (GPIOx_OTYPER), 8.4.2, 154/836 */
		if ((bs >> 3) & 1) {
			/* Bit 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.otyper &= ~(0xff << 8);
			cpssp->NAME.otyper |= ((val >> 8) & 0xff) << 8;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.otyper &= ~(0xff << 0);
			cpssp->NAME.otyper |= ((val >> 0) & 0xff) << 0;
		}
		NAME_(output_update)(cpssp);
		break;

	case 0x08:
		/* port output speed register (GPIOx_OSPEEDR), 8.4.3, 155/836 */
		if ((bs >> 3) & 1) {
			for (n = 12; n < 16; n++) {
				cpssp->NAME.ospeedr[n] = (val >> 2 * n) & 0b11;
			}
		}
		if ((bs >> 2) & 1) {
			for (n = 8; n < 12; n++) {
				cpssp->NAME.ospeedr[n] = (val >> 2 * n) & 0b11;
			}
		}
		if ((bs >> 1) & 1) {
			for (n = 4; n < 8; n++) {
				cpssp->NAME.ospeedr[n] = (val >> 2 * n) & 0b11;
			}
		}
		if ((bs >> 0) & 1) {
			for (n = 0; n < 4; n++) {
				cpssp->NAME.ospeedr[n] = (val >> 2 * n) & 0b11;
			}
		}
		break;
	
	case 0x0c:
		/* port pull-up/pull-down register (GPIOx_PUPDR), 8.4.4, 156/836 */
		if ((bs >> 3) & 1) {
			for (n = 12; n < 16; n++) {
				cpssp->NAME.pupdr_down &= ~(1 << n);
				cpssp->NAME.pupdr_down |= (val >> (2 * n + 1)) & 1;
				cpssp->NAME.pupdr_up &= ~(1 << n);
				cpssp->NAME.pupdr_up |= (val >> (2 * n + 0)) & 1;
			}
		}
		if ((bs >> 2) & 1) {
			for (n = 8; n < 12; n++) {
				cpssp->NAME.pupdr_down &= ~(1 << n);
				cpssp->NAME.pupdr_down |= (val >> (2 * n + 1)) & 1;
				cpssp->NAME.pupdr_up &= ~(1 << n);
				cpssp->NAME.pupdr_up |= (val >> (2 * n + 0)) & 1;
			}
		}
		if ((bs >> 1) & 1) {
			for (n = 4; n < 8; n++) {
				cpssp->NAME.pupdr_down &= ~(1 << n);
				cpssp->NAME.pupdr_down |= (val >> (2 * n + 1)) & 1;
				cpssp->NAME.pupdr_up &= ~(1 << n);
				cpssp->NAME.pupdr_up |= (val >> (2 * n + 0)) & 1;
			}
		}
		if ((bs >> 0) & 1) {
			for (n = 0; n < 4; n++) {
				cpssp->NAME.pupdr_down &= ~(1 << n);
				cpssp->NAME.pupdr_down |= (val >> (2 * n + 1)) & 1;
				cpssp->NAME.pupdr_up &= ~(1 << n);
				cpssp->NAME.pupdr_up |= (val >> (2 * n + 0)) & 1;
			}
		}
		NAME_(output_update)(cpssp);
		break;
	
	case 0x10:
		/* port input data register (GPIOx_IDR), 8.4.5, 156/836 */
		/* Bit 31-16: Reserved */
		/* Bit 15-0: Read-only */
		goto read_only;
	
	case 0x14:
		/* port output data register (GPIOx_ODR), 8.4.6, 157/836 */
		if ((bs >> 3) & 1) {
			/* Bit 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.odr &= ~(0xff << 8);
			cpssp->NAME.odr |= ((val >> 8) & 0xff) << 8;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.odr &= ~(0xff << 0);
			cpssp->NAME.odr |= ((val >> 0) & 0xff) << 0;
		}

		NAME_(output_update)(cpssp);
		break;

	case 0x18:
		/* port bit set/reset register (GPIOx_BSRR), 8.4.7, 157/836 */
		if ((bs >> 3) & 1) {
			for (n = 8; n < 16; n++) {
				cpssp->NAME.odr &= ~(((val >> (n + 16)) & 1) << n);
			}
		}
		if ((bs >> 2) & 1) {
			for (n = 0; n < 8; n++) {
				cpssp->NAME.odr &= ~(((val >> (n + 16)) & 1) << n);
			}
		}
		if ((bs >> 1) & 1) {
			for (n = 8; n < 16; n++) {
				cpssp->NAME.odr |= ((val >> n) & 1) << n;
			}
		}
		if ((bs >> 0) & 1) {
			for (n = 0; n < 8; n++) {
				cpssp->NAME.odr |= ((val >> n) & 1) << n;
			}
		}
		NAME_(output_update)(cpssp);
		break;

	case 0x1c:
		/* port configuration lock register (GPIOx_LCKR), 8.4.8, 157/836 */
		goto todo;

	case 0x20:
		/* alternate function low register (GPIOx_AFRL), 8.4.9, 159/836 */
		goto todo;

	case 0x24:
		/* alternate function high register (GPIOx_AFRH), 8.4.10, 159/836 */
		goto todo;

	default:
	todo:	;
	read_only:;
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);

		cpssp->NAME.reg[addr >> 2] = val;
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int n;

	addr &= 0x3ff;

	/*
	 * CPIO register map, 8.4.11, 160/836
	 */
	switch (addr) {
	case 0x00:
		/* port mode register (GPIOx_MODER), 8.4.1, 153/836 */
		*valp = 0;
		for (n = 0; n < 16; n++) {
			*valp |= cpssp->NAME.moder[n] << 2 * n;
		}
		break;

	case 0x04:
		/* port output type register (GPIOx_OTYPER), 8.4.2, 154/836 */
		/* Bit 31-16: Reserved */
		*valp = cpssp->NAME.otyper << 0;
		break;

	case 0x08:
		/* port output speed register (GPIOx_OSPEEDR), 8.4.3, 155/836 */
		*valp = 0;
		for (n = 0; n < 16; n++) {
			*valp |= cpssp->NAME.ospeedr[n] << 2 * n;
		}
		break;

	case 0x0c:
		/* port pull-up/pull-down register (GPIOx_PUPDR), 8.4.4, 156/836 */
		*valp = 0;
		for (n = 0; n < 16; n++) {
			*valp |= ((cpssp->NAME.pupdr_down >> n) & 1) << (2 * n + 1);
			*valp |= ((cpssp->NAME.pupdr_up >> n) & 1) << (2 * n + 0);
		}
		break;

	case 0x10:
		/* port input data register (GPIOx_IDR), 8.4.5, 156/836 */
		*valp = 0;
		/* Bit 31-16: Reserved */
		*valp |= cpssp->NAME.idr;
		break;

	case 0x14:
		/* port output data register (GPIOx_ODR), 8.4.6, 157/836 */
		*valp = 0;
		/* Bit 31-16: Reserved */
		*valp |= cpssp->NAME.odr;
		break;

	case 0x18:
		/* port bit set/reset register (GPIOx_BSRR), 8.4.7, 157/836 */
		/* Bit 31-0: write-only */
		goto write_only;

	case 0x1c:
		/* port configuration lock register (GPIOx_LCKR), 8.4.8, 157/836 */
		goto todo;

	case 0x20:
		/* alternate function low register (GPIOx_AFRL), 8.4.9, 159/836 */
		goto todo;

	case 0x24:
		/* alternate function high register (GPIOx_AFRH), 8.4.10, 159/836 */
		goto todo;
		
	default:
	todo:	;
	write_only:;
		*valp = cpssp->NAME.reg[addr >> 2];

		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(leftNM_in_set)(struct cpssp *cpssp, int n, int alt, unsigned int val)
{
	/* FIXME */
}

static void
NAME_(rightN_in_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	cpssp->NAME.state_right_in[n] = val;
	NAME_(rightN_resolve)(cpssp, n);
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int n;

	/* port mode register (GPIOx_MODER), 8.4.1, 153/836 */
	for (n = 0; n < 16; n++) {
		cpssp->NAME.moder[n] = 0b00; /* Input */
	}

	/* port output type register (GPIOx_OTYPER), 8.4.2, 154/836 */
	cpssp->NAME.otyper = 0x0000;

	/* port output speed register (GPIOx_OSPEEDR), 8.4.3, 155/836 */
	for (n = 0; n < 16; n++) {
		cpssp->NAME.ospeedr[n] = 0b00; /* Low speed */
	}

	/* port pull-up/pull-down register (GPIOx_PUPDR), 8.4.4, 156/836 */
	cpssp->NAME.pupdr_down = 0x0000;
	cpssp->NAME.pupdr_up = 0x0000;

	/* port input data register (GPIOx_IDR), 8.4.5, 156/836 */
	/* Input */

	/* port output data register (GPIOx_ODR), 8.4.6, 157/836 */
	cpssp->NAME.odr = 0x0000;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	unsigned int alt;
	unsigned int n;

	for (n = 0; n < 16; n++) {
		cpssp->NAME.state_ain[n] = SIG_STD_LOGIC_0;
	}
	for (alt = 0; alt < 16; alt++) {
		for (n = 0; n < 16; n++) {
			cpssp->NAME.state_left_in[alt] &= ~(1 << n);
		}
	}
	for (n = 0; n < 16; n++) {
		cpssp->NAME.state_right_out[n] = SIG_STD_LOGIC_Z;
	}
	for (n = 0; n < 16; n++) {
		cpssp->NAME.state_right_in[n] = SIG_STD_LOGIC_Z;
	}
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
