/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For infos look at:
 * "KL02 Sub-Family Reference Manual for 48 MHz devices 32 pin package
 * Reference Manual".
 *
 * "31: Universal Asynchronous Receiver/Transmitter (UART)"
 */

#define DEBUG	0

#ifdef INCLUDE

#endif /* INCLUDE */

#ifdef STATE

struct {
	/* 31.2.1/31.2.2 */
	uint8_t lbkdie;
	uint8_t rxedgie;
	uint8_t sbns;
	uint16_t sbr;

	/* 31.2.3 */
	uint8_t loops;
	uint8_t dozeen;
	uint8_t rsrc;
	uint8_t m;
	uint8_t wake;
	uint8_t ilt;
	uint8_t pe;
	uint8_t pt;

	/* 31.2.4 */
	uint8_t tie;
	uint8_t tcie;
	uint8_t rie;
	uint8_t ilie;
	uint8_t te;
	uint8_t re;
	uint8_t rwu;
	uint8_t sbk;

	/* 31.2.5 */
	uint8_t tdre;
	uint8_t tc;
	uint8_t rdrf;
	uint8_t idle;
	uint8_t or;
	uint8_t nf;
	uint8_t fe;
	uint8_t pf;

	/* 31.2.6 */
	uint8_t lbkdif;
	uint8_t rxedgif;
	uint8_t msbf;
	uint8_t rxinv;
	uint8_t rwuid;
	uint8_t brk13;
	uint8_t lbkde;
	uint8_t raf;

	/* 31.2.7 */
	uint8_t r8;
	uint8_t r9;
	uint8_t t9;
	uint8_t t8;
	uint8_t txdir;
	uint8_t txinv;
	uint8_t orie;
	uint8_t neie;
	uint8_t feie;
	uint8_t peie;

	/* 31.2.8 */
	uint8_t rx, tx;

	/* 31.2.9*/
	uint8_t ma1;

	/* 31.2.10 */
	uint8_t ma2;

	/* 31.2.11 */
	uint8_t maen1;
	uint8_t maen2;
	uint8_t m10;
	uint8_t osr;

	/* UART Control Register 5 (UARTx_C5) */
#if CONFIG_DMA
	uint8_t tdmae;
	uint8_t rdmae;
#endif
	uint8_t bothedge;
	uint8_t resyncdis;

	uint16_t sbr_count;
	uint8_t osr_count;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(update_irq)(struct cpssp *cpssp)
{
	NAME_(irq_set)(cpssp, 0); /* FIXME */
}

#if CONFIG_DMA
static void
NAME_(update_dma)(struct cpssp *cpssp)
{
	NAME_(dma_receive)(cpssp, 0); /* FIXME */
	NAME_(dma_transmit)(cpssp, 0); /* FIXME */
}
#endif

static void
NAME_(send)(struct cpssp *cpssp)
{
	/* FIXME */
	fprintf(stderr, "%c", cpssp->NAME.tx);
	NAME_(tx_out_set)(cpssp, SIG_STD_LOGIC_0);
	NAME_(tx_out_set)(cpssp, SIG_STD_LOGIC_1);
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000:
		if ((bs >> 0) & 1) {
			/* Baud Rate Register High */
			/* 31.2.1 */
			*valp |= cpssp->NAME.lbkdie << (7+0);
			*valp |= cpssp->NAME.rxedgie << (6+0);
			*valp |= cpssp->NAME.sbns << (5+0);
			*valp |= ((cpssp->NAME.sbr >> 8) & 0x1f) << (0+0);
		}
		if ((bs >> 1) & 1) {
			/* Baud Rate Register Low */
			/* 31.2.2 */
			*valp |= ((cpssp->NAME.sbr >> 0) & 0xff) << (0+8);
		}
		if ((bs >> 2) & 1) {
			/* Control Register 1 */
			/* 31.2.3 */
			*valp |= cpssp->NAME.loops << (7+16);
			*valp |= cpssp->NAME.dozeen << (6+16);
			*valp |= cpssp->NAME.rsrc << (5+16);
			*valp |= cpssp->NAME.m << (4+16);
			*valp |= cpssp->NAME.wake << (3+16);
			*valp |= cpssp->NAME.ilt << (2+16);
			*valp |= cpssp->NAME.pe << (1+16);
			*valp |= cpssp->NAME.pt << (0+16);
		}
		if ((bs >> 3) & 1) {
			/* Control Register 2 */
			/* 31.2.4 */
			*valp |= cpssp->NAME.tie << (7+24);
			*valp |= cpssp->NAME.tcie << (6+24);
			*valp |= cpssp->NAME.rie << (5+24);
			*valp |= cpssp->NAME.ilie << (4+24);
			*valp |= cpssp->NAME.te << (3+24);
			*valp |= cpssp->NAME.re << (2+24);
			*valp |= cpssp->NAME.rwu << (1+24);
			*valp |= cpssp->NAME.sbk << (0+24);
		}
		break;
	case 0x004:
		if ((bs >> 0) & 1) {
			/* Status Register 1 */
			/* 31.2.5 */
			*valp |= cpssp->NAME.tdre << (7+0);
			*valp |= cpssp->NAME.tc << (6+0);
			*valp |= cpssp->NAME.rdrf << (5+0);
			*valp |= cpssp->NAME.idle << (4+0);
			*valp |= cpssp->NAME.or << (3+0);
			*valp |= cpssp->NAME.nf << (2+0);
			*valp |= cpssp->NAME.fe << (1+0);
			*valp |= cpssp->NAME.pf << (0+0);

			cpssp->NAME.tc = 1;
		}
		if ((bs >> 1) & 1) {
			/* Status Register 2 */
			/* 31.2.6 */
			*valp |= cpssp->NAME.lbkdif << (7+8);
			*valp |= cpssp->NAME.rxedgif << (6+8);
			*valp |= cpssp->NAME.msbf << (5+8);
			*valp |= cpssp->NAME.rxinv << (4+8);
			*valp |= cpssp->NAME.rwuid << (3+8);
			*valp |= cpssp->NAME.brk13 << (2+8);
			*valp |= cpssp->NAME.lbkde << (1+8);
			*valp |= cpssp->NAME.raf << (0+8);
		}
		if ((bs >> 2) & 1) {
			/* Control Register 3 */
			/* 31.2.7 */
			*valp |= cpssp->NAME.r8 << (7+16);
			*valp |= cpssp->NAME.r9 << (6+16);
			*valp |= cpssp->NAME.txdir << (5+16);
			*valp |= cpssp->NAME.txinv << (4+16);
			*valp |= cpssp->NAME.orie << (3+16);
			*valp |= cpssp->NAME.neie << (2+16);
			*valp |= cpssp->NAME.feie << (1+16);
			*valp |= cpssp->NAME.peie << (0+16);
		}
		if ((bs >> 3) & 1) {
			/* Data Register */
			/* 31.2.8 */
			/* FIXME */
			*valp |= cpssp->NAME.rx << (0+24);
		}
		break;
	case 0x008:
		if ((bs >> 0) & 1) {
			/* Match Address Registers 1 */
			/* 31.2.9 */
			*valp |= cpssp->NAME.ma1 << (0+0);
		}
		if ((bs >> 1) & 1) {
			/* Match Address Registers 2 */
			/* 31.2.10 */
			*valp |= cpssp->NAME.ma2 << (0+8);
		}
		if ((bs >> 2) & 1) {
			/* UART Control Register 4 (UARTx_C4) */
			*valp |= cpssp->NAME.maen1 << (7+16);
			*valp |= cpssp->NAME.maen2 << (6+16);
			*valp |= cpssp->NAME.m10 << (5+16);
			*valp |= cpssp->NAME.osr << (0+16);
		}
		if ((bs >> 3) & 1) {
			/* UART Control Register 5 (UARTx_C5) */
#if CONFIG_DMA
			*valp |= cpssp->NAME.tdmae << (7+24);
			/* 6: Reserved */
			*valp |= cpssp->NAME.rdmae << (5+24);
#else
			/* 7-5: Reserved */
#endif
			/* 4-2: Reserved */
			*valp |= cpssp->NAME.bothedge << (1+24);
			*valp |= cpssp->NAME.resyncdis << (0+24);
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		*valp = 0;
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x1000 - 1;

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		if ((bs >> 0) & 1) {
			/* Baud Rate Register High */
			/* 31.2.1 */
			cpssp->NAME.lbkdie = (val >> (7+0)) & 1;
			cpssp->NAME.rxedgie = (val >> (6+0)) & 1;
			cpssp->NAME.sbns = (val >> (5+0)) & 1;
			cpssp->NAME.sbr &= ~(0x1f << 8);
			cpssp->NAME.sbr |= ((val >> (0+0)) & 0x1f) << 8;
		}
		if ((bs >> 1) & 1) {
			/* Baud Rate Register Low */
			/* 31.2.2 */
			cpssp->NAME.sbr &= ~(0xff << 0);
			cpssp->NAME.sbr |= ((val >> (0+8)) & 0xff) << 0;
		}
		if ((bs >> 2) & 1) {
			/* Control Register 1 */
			/* 31.2.3 */
			cpssp->NAME.loops = (val >> (7+16)) & 1;
			cpssp->NAME.dozeen = (val >> (6+16)) & 1;
			cpssp->NAME.rsrc = (val >> (5+16)) & 1;
			cpssp->NAME.m = (val >> (4+16)) & 1;
			cpssp->NAME.wake = (val >> (3+16)) & 1;
			cpssp->NAME.ilt = (val >> (2+16)) & 1;
			cpssp->NAME.pe = (val >> (1+16)) & 1;
			cpssp->NAME.pt = (val >> (0+16)) & 1;
		}
		if ((bs >> 3) & 1) {
			/* Control Register 2 */
			/* 31.2.4 */
			cpssp->NAME.tie = (val >> (7+24)) & 1;
			cpssp->NAME.tcie = (val >> (6+24)) & 1;
			cpssp->NAME.rie = (val >> (5+24)) & 1;
			cpssp->NAME.ilie = (val >> (4+24)) & 1;
			cpssp->NAME.te = (val >> (3+24)) & 1;
			cpssp->NAME.re = (val >> (2+24)) & 1;
			cpssp->NAME.rwu = (val >> (1+24)) & 1;
			cpssp->NAME.sbk = (val >> (0+24)) & 1;
		}
		break;
	case 0x004:
		if ((bs >> 0) & 1) {
			/* Status Register 1 */
			/* 31.2.5 */
			/* 7-5: Read-only */
			cpssp->NAME.idle &= ~((val >> (4+0)) & 1);
			cpssp->NAME.or &= ~((val >> (3+0)) & 1);
			cpssp->NAME.nf &= ~((val >> (2+0)) & 1);
			cpssp->NAME.fe &= ~((val >> (1+0)) & 1);
			cpssp->NAME.pf &= ~((val >> (0+0)) & 1);
		}
		if ((bs >> 1) & 1) {
			/* Status Register 2 */
			/* 31.2.6 */
			cpssp->NAME.lbkdif &= ~((val >> (7+8)) & 1);
			cpssp->NAME.rxedgif &= ~((val >> (6+8)) & 1);
			cpssp->NAME.msbf = (val >> (5+8)) & 1;
			cpssp->NAME.rxinv = (val >> (4+8)) & 1;
			cpssp->NAME.rwuid = (val >> (3+8)) & 1;
			cpssp->NAME.brk13 = (val >> (2+8)) & 1;
			cpssp->NAME.lbkde = (val >> (1+8)) & 1;
			/* 0: Read-only */
		}
		if ((bs >> 2) & 1) {
			/* Control Register 3 */
			/* 31.2.7 */
			cpssp->NAME.t9 = (val >> (7+16)) & 1;
			cpssp->NAME.t8 = (val >> (6+16)) & 1;
			cpssp->NAME.txdir = (val >> (5+16)) & 1;
			cpssp->NAME.txinv = (val >> (4+16)) & 1;
			cpssp->NAME.orie = (val >> (3+16)) & 1;
			cpssp->NAME.neie = (val >> (2+16)) & 1;
			cpssp->NAME.feie = (val >> (1+16)) & 1;
			cpssp->NAME.peie = (val >> (0+16)) & 1;
		}
		if ((bs >> 3) & 1) {
			/* Data Register */
			/* 31.2.8 */
			cpssp->NAME.tx = (val >> (0+24)) & 0xff;
			cpssp->NAME.tc = 0;
			NAME_(send)(cpssp);
		}
		break;
	case 0x008:
		if ((bs >> 0) & 1) {
			/* Match Address Registers 1 */
			/* 31.2.9 */
			cpssp->NAME.ma1 = (val >> (0+0)) & 0xff;
		}
		if ((bs >> 1) & 1) {
			/* Match Address Registers 2 */
			/* 31.2.10 */
			cpssp->NAME.ma2 = (val >> (0+8)) & 0xff;
		}
		if ((bs >> 2) & 1) {
			/* UART Control Register 4 (UARTx_C4) */
			/* 31.2.11 */
			cpssp->NAME.maen1 = (val >> (7+16)) & 1;
			cpssp->NAME.maen2 = (val >> (6+16)) & 1;
			cpssp->NAME.m10 = (val >> (5+16)) & 1;
			cpssp->NAME.osr = (val >> (0+16)) & 1;
		}
		if ((bs >> 3) & 1) {
			/* UART Control Register 5 (UARTx_C5) */
#if CONFIG_DMA
			cpssp->NAME.tdmae = (val >> (7+24)) & 1;
			/* 6: Reserved */
			cpssp->NAME.rdmae = (val >> (5+24)) & 1;
#else
			/* 7-5: Reserved */
#endif
			/* 4-2: Reserved */
			cpssp->NAME.bothedge = (val >> (1+24)) & 1;
			cpssp->NAME.resyncdis = (val >> (0+24)) & 1;
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(_clock)(struct cpssp *cpssp)
{
	/* Send/receive one bit. */
	/* FIXME */
}

static void
NAME_(clock)(struct cpssp *cpssp)
{
	if (cpssp->NAME.sbr < cpssp->NAME.sbr_count) { /* Correct? FIXME */
		cpssp->NAME.sbr_count = 0;

		if (cpssp->NAME.osr == cpssp->NAME.osr_count) {
			cpssp->NAME.osr_count = 0;

			NAME_(_clock)(cpssp);
		} else {
			cpssp->NAME.osr_count++;
		}
	} else {
		cpssp->NAME.sbr_count++;
	}
}

static void
NAME_(rx_in_set)(struct cpssp *cpssp, unsigned int val)
{
#if CONFIG_DMA
	/* FIXME */
	if (0) NAME_(update_dma)(cpssp);
#endif
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* 31.2.1/31.2.2 */
	cpssp->NAME.lbkdie = 0;
	cpssp->NAME.rxedgie = 0;
	cpssp->NAME.sbns = 0;
	cpssp->NAME.sbr = 0b0000000000100;

	/* 31.2.3 */
	cpssp->NAME.loops = 0;
	cpssp->NAME.dozeen = 0;
	cpssp->NAME.rsrc = 0;
	cpssp->NAME.m = 0;
	cpssp->NAME.wake = 0;
	cpssp->NAME.ilt = 0;
	cpssp->NAME.pe = 0;
	cpssp->NAME.pt = 0;

	/* 31.2.4 */
	cpssp->NAME.tie = 0;
	cpssp->NAME.tcie = 0;
	cpssp->NAME.rie = 0;
	cpssp->NAME.ilie = 0;
	cpssp->NAME.te = 0;
	cpssp->NAME.re = 0;
	cpssp->NAME.rwu = 0;
	cpssp->NAME.sbk = 0;

	/* 31.2.5 */
	cpssp->NAME.tdre = 1;
	cpssp->NAME.tc = 1;
	cpssp->NAME.rdrf = 0;
	cpssp->NAME.idle = 0;
	cpssp->NAME.or = 0;
	cpssp->NAME.nf = 0;
	cpssp->NAME.fe = 0;
	cpssp->NAME.pf = 0;

	/* 31.2.6 */
	cpssp->NAME.lbkdif = 0;
	cpssp->NAME.rxedgif = 0;
	cpssp->NAME.msbf = 0;
	cpssp->NAME.rxinv = 0;
	cpssp->NAME.rwuid = 0;
	cpssp->NAME.brk13 = 0;
	cpssp->NAME.lbkde = 0;
	cpssp->NAME.raf = 0;

	/* 31.2.7 */
	cpssp->NAME.r8 = 0;
	cpssp->NAME.r9 = 0;
	cpssp->NAME.t9 = 0;
	cpssp->NAME.t8 = 0;
	cpssp->NAME.txdir = 0;
	cpssp->NAME.txinv = 0;
	cpssp->NAME.orie = 0;
	cpssp->NAME.neie = 0;
	cpssp->NAME.feie = 0;
	cpssp->NAME.peie = 0;
	cpssp->NAME.raf = 0;

	/* 31.2.8 */
	cpssp->NAME.rx = 0;
	cpssp->NAME.tx = 0;

	/* 31.2.9 */
	cpssp->NAME.ma1 = 0;

	/* 31.2.10 */
	cpssp->NAME.ma2 = 0;

	/* UART Control Register 4 (UARTx_C4) */
	cpssp->NAME.maen1 = 0;
	cpssp->NAME.maen2 = 0;
	cpssp->NAME.m10 = 0;
	cpssp->NAME.osr = 0b01111;

	/* UART Control Register 5 (UARTx_C5) */
#if CONFIG_DMA
	cpssp->NAME.tdmae = 0;
	cpssp->NAME.rdmae = 0;
#endif
	cpssp->NAME.bothedge = 0;
	cpssp->NAME.resyncdis = 0;

	NAME_(update_irq)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	NAME_(reset)(cpssp);
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG
