/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For infos look at:
 * "KL02 Sub-Family Reference Manual for 48 MHz devices 32 pin package
 * Reference Manual".
 *
 * "13: System Mode Controller (SMC)"
 */

#define DEBUG	0

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */

#ifdef STATE

struct {
	enum { NAME_(RUN), NAME_(WAIT), NAME_(STOP) } state;
	unsigned char gate0_state;
	unsigned char gate1_state;
	unsigned char gate2_state;
	unsigned char gate3_state;
	unsigned char power0_state;
	unsigned char power1_state;
	unsigned char power2_state;
	unsigned char power3_state;

	/* Power Mode Protection Register (SMC_PMPROT) */
	uint8_t avlp;
	uint8_t avlls;

	/* Power Mode Control Register (SMC_PMCTRL) */
	uint8_t runm;
	uint8_t stopa;
	uint8_t stopm;

	/* Stop Control Register (SMC_STOPCTRL) */
	uint8_t pstopo;
	uint8_t porpo;
	uint8_t vllsm;

	/* Power Mode Status Register (SMC_PMSTAT) */
	uint8_t pmstat;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(update_state)(struct cpssp *cpssp)
{
	enum {
		RUN,
		WAIT,
		STOP,
		PSTOP1,
		PSTOP2,
		VLPR,
		VLPW,
		VLPS,
		LLS,
		VLLS0,
		VLLS1,
		VLLS3,
	} state;
	static const struct {
		unsigned char g0; /* MCGIRCLK */
		unsigned char g1; /* Core/System Clock */
		unsigned char g2; /* Bus Clock */
		unsigned char g3; /* OSCERCLK */
		unsigned char v0;
		unsigned char v1;
		unsigned char v2;
		unsigned char v3;
	} table[] = {
	[RUN]    = { 1, 1, 1, 1,  1, 1, 1, 1 },
	[WAIT]   = { 1, 1, 1, 1,  1, 1, 1, 1 },
	[STOP]   = { 1, 0, 0, 1,  1, 1, 1, 1 },
	[PSTOP1] = { 1, 0, 0, 1,  1, 1, 1, 1 },
	[PSTOP2] = { 1, 0, 1, 1,  1, 1, 1, 1 },
	[VLPR]   = { 1, 1, 1, 1,  1, 1, 1, 1 },
	[VLPW]   = { 1, 1, 1, 1,  1, 1, 1, 1 },
	[VLPS]   = { 1, 0, 0, 1,  1, 1, 1, 1 },
	[LLS]    = { 1, 0, 0, 1,  1, 1, 1, 1 },
	[VLLS0]  = { 1, 0, 0, 1,  1, 1, 1, 1 },
	[VLLS1]  = { 1, 0, 0, 1,  1, 1, 1, 1 },
	[VLLS3]  = { 1, 0, 0, 1,  1, 1, 1, 1 },
	};

	switch (cpssp->NAME.state) {
	case NAME_(RUN):
		switch (cpssp->NAME.runm) {
		case 0b00:
			/* RUN */
			cpssp->NAME.pmstat = 0b0000001;
			state = RUN;
			break;

		case 0b10:
			/* VLPR */
			cpssp->NAME.pmstat = 0b0000100;
			state = VLPR;
			break;

		case 0b01:
		case 0b11:
			/* Reserved */
			assert(0); /* FIXME */
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		break;

	case NAME_(WAIT):
		switch (cpssp->NAME.runm) {
		case 0b00:
			/* WAIT */
			cpssp->NAME.pmstat = 0b0000000; /* Correct? FIXME */
			state = WAIT;
			break;

		case 0b10:
			/* VLPW */
			cpssp->NAME.pmstat = 0b0001000;
			state = VLPW;
			break;

		case 0b01:
		case 0b11:
			/* Reserved */
			assert(0); /* FIXME */
			break;
		default:
			assert(0); /* Cannot happen. */
		}
		break;

	case NAME_(STOP):
		switch (cpssp->NAME.stopm) {
		case 0b000:
			if (cpssp->NAME.runm == 0b10) {
				/* Special case. */
				goto vlps;
			}

			cpssp->NAME.pmstat = 0b0000010;

			switch (cpssp->NAME.pstopo) {
			case 0b00:
				/* STOP */
				state = STOP;
				break;
			case 0b01:
				/* PSTOP1 */
				state = PSTOP1;
				break;
			case 0b10:
				/* PSTOP2 */
				state = PSTOP2;
				break;
			case 0b11:
				/* Reserved */
				assert(0); /* FIXME */
				break;
			default:
				assert(0); /* Cannot happen. */
			}
			break;
		case 0b010:
		vlps:	;
			/* VLPS */
			cpssp->NAME.pmstat = 0b0010000;
			state = VLPS;
			break;

		case 0b011:
			/* LLS */
			cpssp->NAME.pmstat = 0b0100000;
			state = LLS;
			break;

		case 0b100:
			/* VLLSx */
			cpssp->NAME.pmstat = 0b1000000;

			switch (cpssp->NAME.vllsm) {
			case 0b000:
				/* VLLS0 */
				state = VLLS0;
				break;
			case 0b001:
				/* VLLS1 */
				state = VLLS1;
				break;
			case 0b011:
				/* VLLS3 */
				state = VLLS3;
				break;
			case 0b010:
			case 0b100:
			case 0b101:
			case 0b110:
			case 0b111:
				/* Reserved */
				assert(0); /* FIXME */
				break;
			default:
				assert(0); /* Cannot happen. */
			}
			break;

		case 0b001:
		case 0b101:
		case 0b110:
		case 0b111:
			/* Reserved */
			assert(0); /* FIXME */
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
		break;

	default:
		assert(0); /* Cannot happen. */
	}
	
	cpssp->NAME.gate0_state = table[state].g0;
	cpssp->NAME.gate1_state = table[state].g1;
	cpssp->NAME.gate2_state = table[state].g2;
	cpssp->NAME.gate3_state = table[state].g3;

	NAME_(power0_set)(cpssp, table[state].v0);
	NAME_(power1_set)(cpssp, table[state].v1);
	NAME_(power2_set)(cpssp, table[state].v2);
	NAME_(power3_set)(cpssp, table[state].v3);

	NAME_(runm_set)(cpssp, cpssp->NAME.runm);
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000:
		if ((bs >> 0) & 1) {
			/* Power Mode Protection Register (SMC_PMPROT) */
			/* 7-6: Reserved */
			*valp |= cpssp->NAME.avlp << (5+0);
			/* 4-2: Reserved */
			*valp |= cpssp->NAME.avlls << (1+0);
			/* 0: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Power Mode Control Register (SMC_PMCTRL) */
			/* 13.3.2 */
			/* 7: Reserved */
			*valp |= cpssp->NAME.runm << (5+8);
			/* 4: Reserved */
			*valp |= cpssp->NAME.stopa << (3+8);
			*valp |= cpssp->NAME.stopm << (0+8);
		}
		if ((bs >> 2) & 1) {
			/* Stop Control Register (SMC_STOPCTRL) */
			/* 13.3.3 */
			*valp |= cpssp->NAME.pstopo << (6+16);
			*valp |= cpssp->NAME.porpo << (5+16);
			/* 4-3: Reserved */
			*valp |= cpssp->NAME.vllsm << (0+16);
		}
		if ((bs >> 3) & 1) {
			/* Power Mode Status Register (SMC_PMSTAT) */
			/* 13.3.4 */
			/* 7: Reserved */
			*valp |= cpssp->NAME.pmstat << (0+24);
		}
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		*valp = 0;
		assert(0); /* FIXME */
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	uint8_t old_runm, new_runm;

	addr &= 0x1000 - 1;

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		if ((bs >> 0) & 1) {
			/* Power Mode Protection Register (SMC_PMPROT) */
			/* 13.3.1 */
			/* 7-6: Reserved */
			cpssp->NAME.avlp = (val >> (5+0)) & 1;
			/* 4-2: Reserved */
			cpssp->NAME.avlls = (val >> (1+0)) & 1;
			/* 0: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Power Mode Control Register (SMC_PMCTRL) */
			/* 13.3.2 */
			/* 7: Reserved */
			old_runm = cpssp->NAME.runm;
			cpssp->NAME.runm = (val >> (5+8)) & 0b11;
			cpssp->NAME.runm &= cpssp->NAME.avlp << 1;
			new_runm = cpssp->NAME.runm;
			/* 4: Reserved */
			/* 3: Read-only */
			cpssp->NAME.stopm = (val >> (0+8)) & 0b111;

			if (new_runm != old_runm) {
				NAME_(update_state)(cpssp);
			}
		}
		if ((bs >> 2) & 1) {
			/* Stop Control Register (SMC_STOPCTRL) */
			/* 13.3.3 */
			cpssp->NAME.pstopo = (val >> (6+16)) & 0b11;
			cpssp->NAME.porpo = (val >> (5+16)) & 1;
			/* 4-3: Reserved */
			cpssp->NAME.vllsm = (val >> (0+16)) & 0b111;

			/* NAME_(update_state)(cpssp); We're in run mode... */
		}
		if ((bs >> 3) & 1) {
			/* Power Mode Status Register (SMC_PMSTAT) */
			/* 13.3.4 */
			/* Read-only */
		}
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(stop)(struct cpssp *cpssp, int sleepdeep)
{
	cpssp->NAME.state = sleepdeep ? NAME_(STOP) : NAME_(WAIT);
	NAME_(update_state)(cpssp);
}

static void
NAME_(start)(struct cpssp *cpssp)
{
	if (cpssp->NAME.state == NAME_(RUN)) {
		return;
	}

	cpssp->NAME.state = NAME_(RUN);
	NAME_(update_state)(cpssp);

	if ((cpssp->NAME.stopm >> 2) & 1) {
		NAME_(reset_out)(cpssp);
	}
}

static void
NAME_(system_clk_in)(struct cpssp *cpssp)
{
	if (cpssp->NAME.gate1_state) {
		NAME_(system_clk_out)(cpssp);
	}
}

static void
NAME_(bus_clk_in)(struct cpssp *cpssp)
{
	if (cpssp->NAME.gate2_state) {
		NAME_(bus_clk_out)(cpssp);
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* Power Mode Protection Register (SMC_PMPROT) */
	cpssp->NAME.avlp = 0;
	cpssp->NAME.avlls = 0;

	/* Power Mode Control Register (SMC_PMCTRL) */
	cpssp->NAME.runm = 0;
	cpssp->NAME.stopa = 0;
	cpssp->NAME.stopm = 0;

	/* Stop Control Register (SMC_STOPCTRL) */
	cpssp->NAME.pstopo = 0;
	cpssp->NAME.porpo = 0;
	cpssp->NAME.vllsm = 0b011; /* VLLS3 */

	/* Power Mode Status Register (SMC_PMSTAT) */
	cpssp->NAME.pmstat = 0b0000001; /* RUN */

	cpssp->NAME.state = NAME_(RUN);
	NAME_(update_state)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG
