/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For infos look at:
 * "KL02 Sub-Family Reference Manual for 48 MHz devices 32 pin package
 * Reference Manual".
 *
 * "12: System Integration Module (SIM)"
 */

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */

#ifdef STATE

struct {
	unsigned int count_outdiv1;
	unsigned int count_outdiv4;
#if defined(KL02Z)
	/* No PLL */
#elif defined(KL46Z)
	unsigned int count_pllclk;
#else
#error
#endif
	/* Service COP Register (SIM_SRVCOP) */
	unsigned int count_precop;
	unsigned int count_cop;

	/* System Options Register 2 (SIM_SOPT2) */
	uint8_t uart0src;
	uint8_t tpmsrc;
#if defined(KL02Z)
	/* No USB */
	/* No PLL */
	/* No ? */
	/* No RTC */
#elif defined(KL46Z)
	uint8_t usbsrc;
	uint8_t pllfllsel;
	uint8_t clkoutsel;
	uint8_t rtcclkoutsel;
#else
#error
#endif

	/* System Options Register 4 (SIM_SOPT4) */
	uint8_t tpm1clksel;
	uint8_t tpm0clksel;
	uint8_t tpm1ch0src;

	/* System Options Register 5 (SIM_SOPT5) */
	uint8_t uart0ode;
	uint8_t uart0rxsrc;
	uint8_t uart0txsrc;

	/* System Options Register 7 (SIM_SOPT7) */
	uint8_t adc0alttrgen;
	uint8_t adc0pretrgsel;
	uint8_t adc0trgsel;

	/* System Clock Gating Control Register 4 (SIM_SCGC4) */
	uint8_t spi0;
	uint8_t cmp;
	uint8_t uart0;
	uint8_t i2c1;
	uint8_t i2c0;

	/* System Clock Gating Control Register 5 (SIM_SCGC5) */
	uint8_t portb;
	uint8_t porta;
	uint8_t lptmr;

	/* System Clock Gating Control Register 6 (SIM_SCGC6) */
#if defined(KL02Z)
	/* No DAC0 */
#elif defined(KL46Z)
	uint8_t dac0;
#else
#error
#endif
	uint8_t adc0;
	uint8_t tpm1;
	uint8_t tpm0;
#if defined(KL02Z)
	/* No PIT */
#elif defined(KL46Z)
	uint8_t pit;
#else
#error
#endif
	uint8_t ftf;

	/* System Clock Gating Control Register 7 (SIM_SCGC7) */
	uint8_t dma;

	/* System Clock Divider Register 1 (SIM_CLKDIV1) */
	uint8_t outdiv1;
	uint8_t outdiv4;

	/* Unique Identification Register Mid-High (SIM_UIDMH) */
	uint16_t uidh;
	/* Unique Identification Register Mid-Low (SIM_UIDML) */
	uint32_t uidm;
	/* Unique Identification Register Low (SIM_UIDL) */
	uint32_t uidl;

	/* COP Control Register (SIM_COPC) */
	uint8_t copt;
	uint8_t copclks;
	uint8_t copw;

	/* Service COP Register (SIM_SRVCOP) */
	uint8_t cop_state;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(7ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	assert(bs == 0b1111);

	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000:
		/* System Options Register 1 (SIM_SOPT1) */
		if ((bs >> 3) & 1) {
			/* FIXME */
		}
		if ((bs >> 2) & 1) {
			/* FIXME */
		}
		if ((bs >> 1) & 1) {
			/* FIXME */
		}
		if ((bs >> 0) & 1) {
			/* FIXME */
		}
		break;

	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%04x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(7st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	assert(bs == 0b1111);

	addr &= 0x1000 - 1;

	switch (addr) {
	case 0x000:
		/* System Options Register 1 */
		if ((bs >> 3) & 1) {
			/* FIXME */
		}
		if ((bs >> 2) & 1) {
			/* FIXME */
		}
		if ((bs >> 1) & 1) {
			/* FIXME */
		}
		if ((bs >> 0) & 1) {
			/* FIXME */
		}
		break;

	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%04x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(8ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	assert(bs == 0b1111);

	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x004:
		/* System Options Register 2 (SIM_SOPT2) */
		if ((bs >> 3) & 1) {
			/* 31-28: reserved */
			*valp |= cpssp->NAME.uart0src << 26;
			*valp |= cpssp->NAME.tpmsrc << 24;
		}
		if ((bs >> 2) & 1) {
#if defined(KL02Z)
			/* 23-16: reserved */
#elif defined(KL46Z)
			/* 23-19: reserved */
			*valp |= cpssp->NAME.usbsrc << 18;
			/* 17: reserved */
			*valp |= cpssp->NAME.pllfllsel << 16;
#else
#error
#endif
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
#if defined(KL02Z)
			/* 7-0: reserved */
#elif defined(KL46Z)
			*valp |= cpssp->NAME.clkoutsel << 5;
			*valp |= cpssp->NAME.rtcclkoutsel << 4;
			/* 3-0: reserved */
#else
#error
#endif
		}
		break;

	case 0x00c:
		/* System Options Register 4 (SIM_SOPT4) */
		if ((bs >> 3) & 1) {
			/* 31-26: reserved */
			*valp |= cpssp->NAME.tpm1clksel << 25;
			*valp |= cpssp->NAME.tpm0clksel << 24;
		}
		if ((bs >> 2) & 1) {
			/* 23-19: reserved */
			*valp |= cpssp->NAME.tpm1ch0src << 18;
			/* 17-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-0: reserved */
		}
		break;

	case 0x010:
		/* System Options Register 5 (SIM_SOPT5) */
		if ((bs >> 3) & 1) {
			/* 31-24: reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-17: reserved */
			*valp |= cpssp->NAME.uart0ode << 16;
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-3: reserved */
			*valp |= cpssp->NAME.uart0rxsrc << 2;
			/* 1: reserved */
			*valp |= cpssp->NAME.uart0txsrc << 0;
		}
		break;

	case 0x018:
		/* System Options Register 7 (SIM_SOPT7) */
		if ((bs >> 3) & 1) {
			/* 31-24: reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			*valp |= cpssp->NAME.adc0alttrgen << 7;
			/* 6-5: reserved */
			*valp |= cpssp->NAME.adc0pretrgsel << 4;
			*valp |= cpssp->NAME.adc0trgsel << 0;
		}
		break;

	case 0x024:
		/* System Device Identification Register (SIM_SDID) */
		if ((bs >> 3) & 1) {
			*valp |= 0 << 28; /* KL0x Family */
			*valp |= 2 << 24; /* KLx2 Sub-Family */
		}
		if ((bs >> 2) & 1) {
			*valp |= 1 << 20; /* Kinetis Series ID */
			*valp |= 3 << 16; /* SRAM Size (4KB) */
		}
		if ((bs >> 1) & 1) {
			*valp |= 1 << 12; /* Device Revision Number */
			*valp |= 6 << 8; /* Device Die Number */
		}
		if ((bs >> 0) & 1) {
			*valp |= 0 << 7; /* Device Die Number */
			/* 6-4; Reserved */
			*valp |= 2 << 0; /* Pincount; 32-pin */
		}
		break;

	case 0x034:
		/* System Clock Gating Control Register 4 (SIM_SCGC4) */
		if ((bs >> 3) & 1) {
			*valp |= 0b111 << 29; /* reserved */
			/* 28-24; reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23; reserved */
			*valp |= cpssp->NAME.spi0 << 22;
			/* 21-20; reserved */
			*valp |= cpssp->NAME.cmp << 19;
			/* 18-16; reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-11; reserved */
			*valp |= cpssp->NAME.uart0 << 10;
			/* 9-8; reserved */
		}
		if ((bs >> 0) & 1) {
#if defined(KL02Z)
			/* 7: reserved */
#elif defined(KL46Z)
			*valp |= cpssp->NAME.i2c1 << 7;
#else
#error
#endif
			*valp |= cpssp->NAME.i2c0 << 6;
			*valp |= 0b11 << 4; /* reserved */
			/* 3-0; reserved */
		}
		break;

	case 0x038:
		/* System Clock Gating Control Register 5 (SIM_SCGC5) */
		if ((bs >> 3) & 1) {
			/* 31-24: reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-11: reserved */
			*valp |= cpssp->NAME.portb << 10;
			*valp |= cpssp->NAME.porta << 9;
			*valp |= 1 << 8; /* 8: reserved */
		}
		if ((bs >> 0) & 1) {
			*valp |= 1 << 7; /* 7: reserved */
			/* 6-1: reserved */
			*valp |= cpssp->NAME.lptmr << 0;
		}
		break;

	case 0x03c:
		/* System Clock Gating Control Register 6 (SIM_SCGC6) */
		if ((bs >> 3) & 1) {
#if defined(KL02Z)
			/* 31: reserved */
#elif defined(KL46Z)
			*valp |= cpssp->NAME.dac0 << 31;
#else
#error
#endif
			/* 30-28: reserved */
			*valp |= cpssp->NAME.adc0 << 27;
			/* 26: reserved */
			*valp |= cpssp->NAME.tpm1 << 25;
			*valp |= cpssp->NAME.tpm0 << 24;
		}
		if ((bs >> 2) & 1) {
#if defined(KL02Z)
			/* 23: reserved */
#elif defined(KL46Z)
			*valp |= cpssp->NAME.pit << 23;
#else
#error
#endif
			/* 22-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-1: reserved */
			*valp |= cpssp->NAME.ftf << 0;
		}
		break;

	case 0x040:
		/* System Clock Gating Control Register 7 (SIM_SCGC7) */
		if ((bs >> 3) & 1) {
			/* 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-9: Reserved */
			*valp |= cpssp->NAME.dma << 8;
		}
		if ((bs >> 0) & 1) {
			/* 7-0: Reserved */
		}
		break;

	case 0x044:
		/* System Clock Divider Register 1 (SIM_CLKDIV1) */
		if ((bs >> 3) & 1) {
			*valp |= cpssp->NAME.outdiv1 << 28;
			/* 27-24: reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-19: reserved */
			*valp |= cpssp->NAME.outdiv4 << 16;
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-0: reserved */
		}
		break;

	case 0x058:
		/* Unique Identification Register Mid-High (SIM_UIDMH) */
		/* 31-16: reserved */
		*valp |= cpssp->NAME.uidh << 0;
		break;

	case 0x05c:
		/* Unique Identification Register Mid-Low (SIM_UIDML) */
		*valp |= cpssp->NAME.uidm << 0;
		break;

	case 0x060:
		/* Unique Identification Register Low (SIM_UIDL) */
		*valp |= cpssp->NAME.uidl << 0;
		break;

	case 0x100:
		/* COP Control Register (SIM_COPC) */
		/* 31-4: reserved */
		*valp |= cpssp->NAME.copt << 2;
		*valp |= cpssp->NAME.copclks << 1;
		*valp |= cpssp->NAME.copw << 0;
		break;

	case 0x104:
		/* Service COP Register (SIM_SRVCOP) */
		*valp |= 0; /* Write-only register */
		break;

	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%04x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(8st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	assert(bs == 0b1111);

	addr &= 0x1000 - 1;

	switch (addr) {
	case 0x004:
		/* System Options Register 2 (SIM_SOPT2) */
		if ((bs >> 3) & 1) {
			/* 31-28: reserved */
			cpssp->NAME.uart0src = (val >> 26) & 0b11;
			cpssp->NAME.tpmsrc = (val >> 24) & 0b11;
		}
		if ((bs >> 2) & 1) {
#if defined(KL02Z)
			/* 23-16: reserved */
#elif defined(KL46Z)
			/* 23-19: reserved */
			cpssp->NAME.usbsrc = (val >> 18) & 1;
			/* 17: reserved */
			cpssp->NAME.pllfllsel = (val >> 16) & 1;
#else
#error
#endif
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
#if defined(KL02Z)
			/* 7-0: reserved */
#elif defined(KL46Z)
			cpssp->NAME.clkoutsel = (val >> 5) & 0b111;
			cpssp->NAME.rtcclkoutsel = (val >> 4) & 1;
			/* 3-0: reserved */
#else
#error
#endif
		}
		break;

	case 0x00c:
		/* System Options Register 4 (SIM_SOPT4) */
		if ((bs >> 3) & 1) {
			/* 31-26: reserved */
			cpssp->NAME.tpm1clksel = (val >> 25) & 1;
			cpssp->NAME.tpm0clksel = (val >> 24) & 1;
		}
		if ((bs >> 2) & 1) {
			/* 23-19: reserved */
			cpssp->NAME.tpm1ch0src = (val >> 18) & 1;
			/* 17-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-0: reserved */
		}
		break;

	case 0x010:
		/* System Options Register 5 (SIM_SOPT5) */
		if ((bs >> 3) & 1) {
			/* 31-24: reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-17: reserved */
			cpssp->NAME.uart0ode = (val >> 16) & 1;
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-3: reserved */
			cpssp->NAME.uart0rxsrc = (val >> 2) & 1;
			/* 1: reserved */
			cpssp->NAME.uart0txsrc = (val >> 0) & 1;
		}
		break;

	case 0x018:
		/* System Options Register 7 (SIM_SOPT7) */
		if ((bs >> 3) & 1) {
			/* 31-24: reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.adc0alttrgen = (val >> 7) & 1;
			/* 6-5: reserved */
			cpssp->NAME.adc0pretrgsel = (val >> 4) & 1;
			cpssp->NAME.adc0trgsel = (val >> 0) & 0xf;
		}
		break;

	case 0x024:
		/* System Device Identification Register (SIM_SDID) */
		/* Nothing happens. */
		break;

	case 0x034:
		/* System Clock Gating Control Register 4 (SIM_SCGC4) */
		if ((bs >> 3) & 1) {
			/* reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23: reserved */
			cpssp->NAME.spi0 = (val >> 22) & 1;
			/* 21-20: reserved */
			cpssp->NAME.cmp = (val >> 19) & 1;
			/* 18-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-11: reserved */
			cpssp->NAME.uart0 = (val >> 10) & 1;
			/* 9-8: reserved */
		}
		if ((bs >> 0) & 1) {
#if defined(KL02Z)
			/* 7: reserved */
#elif defined(KL46Z)
			cpssp->NAME.i2c1 = (val >> 7) & 1;
#else
#error
#endif
			cpssp->NAME.i2c0 = (val >> 6) & 1;
			/* 5-0: reserved */
		}
		break;

	case 0x038:
		/* System Clock Gating Control Register 5 (SIM_SCGC5) */
		if ((bs >> 3) & 1) {
			/* 31-24: reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-11: reserved */
			cpssp->NAME.portb = (val >> 10) & 1;
			cpssp->NAME.porta = (val >> 9) & 1;
			/* 8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-1: reserved */
			cpssp->NAME.lptmr = (val >> 0) & 1;
		}
		break;

	case 0x03c:
		/* System Clock Gating Control Register 6 (SIM_SCGC6) */
		if ((bs >> 3) & 1) {
#if defined(KL02Z)
			/* 31: reserved */
#elif defined(KL46Z)
			cpssp->NAME.dac0 = (val >> 31) & 1;
#else
#error
#endif
			/* 30-28: reserved */
			cpssp->NAME.adc0 = (val >> 27) & 1;
			/* 26: reserved */
			cpssp->NAME.tpm1 = (val >> 25) & 1;
			cpssp->NAME.tpm0 = (val >> 24) & 1;
		}
		if ((bs >> 2) & 1) {
#if defined(KL02Z)
			/* 23: reserved */
#elif defined(KL46Z)
			cpssp->NAME.pit = (val >> 23) & 1;
#else
#error
#endif
			/* 23-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-1: reserved */
			cpssp->NAME.ftf = (val >> 0) & 1;
		}
		break;
	
	case 0x040:
		/* System Clock Gating Control Register 7 (SIM_SCGC7) */
		if ((bs >> 3) & 1) {
			/* 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-9: Reserved */
			cpssp->NAME.dma = (val >> 8) & 1;
		}
		if ((bs >> 0) & 1) {
			/* 7-0: Reserved */
		}
		break;

	case 0x044:
		/* System Clock Divider Register 1 (SIM_CLKDIV1) */
		if ((bs >> 3) & 1) {
			cpssp->NAME.outdiv1 = (val >> 28) & 0xf;
			/* 27-24: reserved */
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.outdiv4 = (val >> 16) & 0x7;
			/* 23-19: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-0: reserved */
		}
		break;

	case 0x058:
		/* Unique Identification Register Mid-High (SIM_UIDMH) */
		/* Nothing happens. */
		break;

	case 0x05c:
		/* Unique Identification Register Mid-Low (SIM_UIDML) */
		/* Nothing happens. */
		break;

	case 0x060:
		/* Unique Identification Register Low (SIM_UIDL) */
		/* Nothing happens. */
		break;

	case 0x100:
		/* COP Control Register (SIM_COPC) */
		if ((bs >> 3) & 1) {
			/* 31-24: reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-4: reserved */
			cpssp->NAME.copt = (val >> 2) & 0x3;
			cpssp->NAME.copclks = (val >> 1) & 1;
			cpssp->NAME.copw = (val >> 0) & 1;
		}
		break;

	case 0x104:
		/* Service COP Register (SIM_SRVCOP) */
		if ((bs >> 3) & 1) {
			/* 31-24: reserved */
		}
		if ((bs >> 2) & 1) {
			/* 23-16: reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: reserved */
		}
		if ((bs >> 0) & 1) {
			if (cpssp->NAME.cop_state == 0) {
				if (((val >> 0) & 0xff) == 0x55) {
					cpssp->NAME.cop_state = 1;
				} else {
					NAME_(cop_reset)(cpssp);
				}
			} else {
				if (((val >> 0) & 0xff) == 0xaa) {
					cpssp->NAME.cop_state = 0;
					cpssp->NAME.count_precop = 0;
					cpssp->NAME.count_cop = 0;
				} else {
					NAME_(cop_reset)(cpssp);
				}
			}
		}
		break;

	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%04x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(copclk)(struct cpssp *cpssp)
{
	static const unsigned int end[] = { 0, 1 << 5, 1 << 8, 1 << 10 };

	if (cpssp->NAME.copt) {
		if (cpssp->NAME.count_cop == end[cpssp->NAME.copt]) {
			/* Timeout */
			NAME_(cop_reset)(cpssp);
		} else {
			cpssp->NAME.count_cop++;
		}
	}
}

static void
NAME_(lpo)(struct cpssp *cpssp)
{
	if (cpssp->NAME.copclks == 0) {
		NAME_(copclk)(cpssp);
	}
}

static void
NAME_(outclkin)(struct cpssp *cpssp)
{
	if (cpssp->NAME.count_outdiv1 == cpssp->NAME.outdiv1) {
		cpssp->NAME.count_outdiv1 = 0;
		if (1) { /* FIXME */
			NAME_(coreclk)(cpssp);
		}
		if (cpssp->NAME.count_outdiv4 == cpssp->NAME.outdiv4) {
			cpssp->NAME.count_outdiv4 = 0;
			if (1) { /* FIXME */
				if (cpssp->NAME.copclks == 1) {
					/* Bus clock */
					if (cpssp->NAME.count_precop == 0x100) {
						NAME_(copclk)(cpssp);
						cpssp->NAME.count_precop = 0;
					} else {
						cpssp->NAME.count_precop++;
					}
				}
				NAME_(busclk)(cpssp);
			}
		} else {
			cpssp->NAME.count_outdiv4++;
		}
	} else {
		cpssp->NAME.count_outdiv1++;
	}
}

static void
NAME_(fllclkin)(struct cpssp *cpssp)
{
#if defined(KL02Z)
	NAME_(xllclkout)(cpssp);
#elif defined(KL46Z)
	if (cpssp->NAME.pllfllsel == 0) {
		NAME_(xllclkout)(cpssp);
	}
#else
#error
#endif
}

#if defined(KL02Z)
/* No PLL */
#elif defined(KL46Z)
static void
NAME_(pllclkin)(struct cpssp *cpssp)
{
	if (cpssp->NAME.count_pllclk == 1) {
		if (cpssp->NAME.pllfllsel == 1) {
			NAME_(xllclkout)(cpssp);
		}
	}
	cpssp->NAME.count_pllclk ^= 1;
}
#else
#error
#endif

static void
NAME_(tpm_clkinN_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	if (val) {
		/* FIXME */
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.count_outdiv1 = 0;
	cpssp->NAME.count_outdiv4 = 0;
#if defined(KL02Z)
/* No PLL */
#elif defined(KL46Z)
	cpssp->NAME.count_pllclk = 0;
#else
cerror
#endif
	cpssp->NAME.count_precop = 0;
	cpssp->NAME.count_cop = 0;

	/* System Options Register 2 (SIM_SOPT2) */
	cpssp->NAME.uart0src = 0;
	cpssp->NAME.tpmsrc = 0;
#if defined(KL02Z)
	/* No USB */
	/* No PLL */
	/* No ? */
	/* No RTC */
#elif defined(KL46Z)
	cpssp->NAME.usbsrc = 0;
	cpssp->NAME.pllfllsel = 0;
	cpssp->NAME.clkoutsel = 0b000;
	cpssp->NAME.rtcclkoutsel = 0;
#else
#error
#endif

	/* System Options Register 4 (SIM_SOPT4) */
	cpssp->NAME.tpm1clksel = 0;
	cpssp->NAME.tpm0clksel = 0;
	cpssp->NAME.tpm1ch0src = 0;

	/* System Options Register 5 (SIM_SOPT5) */
	cpssp->NAME.uart0ode = 0;
	cpssp->NAME.uart0rxsrc = 0;
	cpssp->NAME.uart0txsrc = 0;

	/* System Options Register 7 (SIM_SOPT7) */
	cpssp->NAME.adc0alttrgen = 0;
	cpssp->NAME.adc0pretrgsel = 0;
	cpssp->NAME.adc0trgsel = 0;

	/* System Clock Gating Control Register 4 (SIM_SCGC4) */
	cpssp->NAME.spi0 = 0;
	cpssp->NAME.cmp = 0;
	cpssp->NAME.uart0 = 0;
#if defined(KL02Z)
	/* No I2C1 */
#elif defined(KL46Z)
	cpssp->NAME.i2c1 = 0;
#else
#error
#endif
	cpssp->NAME.i2c0 = 0;

	/* System Clock Gating Control Register 5 (SIM_SCGC5) */
	cpssp->NAME.portb = 0;
	cpssp->NAME.porta = 0;
	cpssp->NAME.lptmr = 0;

	/* System Clock Gating Control Register 6 (SIM_SCGC6) */
#if defined(KL02Z)
	/* No DAC0 */
#elif defined(KL46Z)
	cpssp->NAME.dac0 = 0;
#else
#error
#endif
	cpssp->NAME.adc0 = 0;
	cpssp->NAME.tpm1 = 0;
	cpssp->NAME.tpm0 = 0;
#if defined(KL02Z)
	/* No PIT */
#elif defined(KL46Z)
	cpssp->NAME.pit = 0;
#else
#error
#endif
	cpssp->NAME.ftf = 1;

	/* System Clock Gating Control Register 7 (SIM_SCGC7) */
	cpssp->NAME.dma = 0;

	/* System Clock Divider Register 1 (SIM_CLKDIV1) */
	cpssp->NAME.outdiv1 = 0; /* FIXME */
	cpssp->NAME.outdiv4 = 1;

	/* COP Control Register (SIM_COPC) */
	cpssp->NAME.copt = 3;
	cpssp->NAME.copclks = 0;
	cpssp->NAME.copw = 0;

	/* Service COP Register (SIM_SRVCOP) */
	cpssp->NAME.cop_state = 0;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	static uint16_t uidh = 0;
	static uint32_t uidm = 0;
	static uint32_t uidl = 1;

	cpssp->NAME.uidh = uidh;
	cpssp->NAME.uidm = uidm;
	cpssp->NAME.uidl = uidl++;
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
