/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For infos look at:
 * "KL02 Sub-Family Reference Manual for 48 MHz devices 32 pin package
 * Reference Manual".
 *
 * "15: Reset Control Module (RCM)"
 */

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */

#ifdef STATE

struct {
	enum {
		NAME_(POR),
		NAME_(PIN),
		NAME_(WDOG),
		NAME_(LOC),
		NAME_(LVD),
		NAME_(WAKEUP),
		NAME_(SACKERR),
		NAME_(MDM_AP),
		NAME_(SW),
		NAME_(LOCKUP),
	} reason;

	/* 15.2.3 */
	uint8_t rstfltss;
	uint8_t rstfltsrw;

	/* 15.2.4 */
	uint8_t rstfltsel;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000:
		if ((bs >> 0) & 1) {
			/* System Reset Status Register 0 */
			/* 15.2.1 */
			*valp |= (cpssp->NAME.reason == NAME_(POR)) << (7+0);
			*valp |= (cpssp->NAME.reason == NAME_(PIN)) << (6+0);
			*valp |= (cpssp->NAME.reason == NAME_(WDOG)) << (5+0);
			/* 4-3: reserved */
			*valp |= (cpssp->NAME.reason == NAME_(LOC)) << (2+0);
			*valp |= ((cpssp->NAME.reason == NAME_(POR))
				| (cpssp->NAME.reason == NAME_(LVD))) << (1+0);
			*valp |= (cpssp->NAME.reason == NAME_(WAKEUP)) << (0+0);
		}
		if ((bs >> 1) & 1) {
			/* System Reset Status Register 1 */
			/* 15.2.2 */
			/* 7-6: reserved */
			*valp |= (cpssp->NAME.reason == NAME_(SACKERR)) << (5+8);
			/* 4: reserved */
			*valp |= (cpssp->NAME.reason == NAME_(MDM_AP)) << (3+8);
			*valp |= (cpssp->NAME.reason == NAME_(SW)) << (2+8);
			*valp |= (cpssp->NAME.reason == NAME_(LOCKUP)) << (2+8);
			/* 0: reserved */
		}
		if ((bs >> 2) & 1) {
			/* reserved */
		}
		if ((bs >> 3) & 1) {
			/* reserved */
		}
		break;

	case 0x004:
		if ((bs >> 0) & 1) {
			/* Reset Pin Filter Control Register */
			/* 15.2.3 */
			/* 7-3: reserved */
			*valp |= cpssp->NAME.rstfltss << (2+0);
			*valp |= cpssp->NAME.rstfltsrw << (0+0);
		}
		if ((bs >> 1) & 1) {
			/* Reset Pin Filter Width Register */
			/* 15.2.4 */
			/* 7-5: reserved */
			*valp |= cpssp->NAME.rstfltsrw << (0+8);
		}
		break;

	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x1000 - 1;

	switch (addr) {
	case 0x000:
		if ((bs >> 0) & 1) {
			/* System Reset Status Register 0 */
			/* 15.2.1 */
			/* read-only */
		}
		if ((bs >> 1) & 1) {
			/* System Reset Status Register 1 */
			/* 15.2.2 */
			/* read-only */
		}
		if ((bs >> 2) & 1) {
			/* reserved */
		}
		if ((bs >> 3) & 1) {
			/* reserved */
		}
		break;

	case 0x004:
		if ((bs >> 0) & 1) {
			/* Reset Pin Filter Control Register */
			/* 15.2.3 */
			/* 7-3: reserved */
			cpssp->NAME.rstfltss = (val >> (2+0)) & 1;
			cpssp->NAME.rstfltsrw = (val >> (0+0)) & 3;
		}
		if ((bs >> 1) & 1) {
			/* Reset Pin Filter Width Register */
			/* 15.2.4 */
			/* 7-5: reserved */
			cpssp->NAME.rstfltsrw = (val >> (0+8)) & 0x1f;
		}
		if ((bs >> 2) & 1) {
			/* reserved */
		}
		if ((bs >> 3) & 1) {
			/* reserved */
		}
		break;

	default:
		/* FIXME */
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(por)(struct cpssp *cpssp)
{
	cpssp->NAME.reason = NAME_(POR);
}

static void
NAME_(wdog)(struct cpssp *cpssp)
{
	cpssp->NAME.reason = NAME_(WDOG);
}

static void
NAME_(wakeup)(struct cpssp *cpssp)
{
	cpssp->NAME.reason = NAME_(WAKEUP);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
