/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */

#ifdef STATE

struct {
	uint8_t mdis;
	uint8_t frz;
	uint32_t lifetime;
	uint32_t ldval[N];
	uint32_t cval[N];
	uint8_t chn[N];
	uint8_t tie[N];
	uint8_t ten[N];
	uint8_t tif[N];
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(update_out)(struct cpssp *cpssp, unsigned int n)
{
	unsigned int val;

	val = cpssp->NAME.cval[n] == 0;
	switch (n) {
	case 0: NAME_(out0_set)(cpssp, val); break;
#if 2 <= N
	case 1: NAME_(out1_set)(cpssp, val); break;
#endif
#if 3 <= N
	case 1: NAME_(out2_set)(cpssp, val); break;
#endif
#if 4 <= N
	case 1: NAME_(out3_set)(cpssp, val); break;
#endif
#if 5 <= N
	case 1: NAME_(out4_set)(cpssp, val); break;
#endif
#if 6 <= N
	case 1: NAME_(out5_set)(cpssp, val); break;
#endif
#if 7 <= N
	case 1: NAME_(out6_set)(cpssp, val); break;
#endif
#if 8 <= N
	case 1: NAME_(out7_set)(cpssp, val); break;
#endif
#if 9 <= N
	case 1: NAME_(out8_set)(cpssp, val); break;
#endif
#if 10 <= N
	case 1: NAME_(out9_set)(cpssp, val); break;
#endif
#if 11 <= N
	case 1: NAME_(out10_set)(cpssp, val); break;
#endif
#if 12 <= N
	case 1: NAME_(out11_set)(cpssp, val); break;
#endif
#if 13 <= N
	case 1: NAME_(out12_set)(cpssp, val); break;
#endif
#if 14 <= N
	case 1: NAME_(out13_set)(cpssp, val); break;
#endif
#if 15 <= N
	case 1: NAME_(out14_set)(cpssp, val); break;
#endif
#if 16 <= N
	case 1: NAME_(out15_set)(cpssp, val); break;
#endif
	default: assert(0); /* Mustn't happen. */
	}
}

static void
NAME_(update_irq)(struct cpssp *cpssp, unsigned int n)
{
	unsigned int irq;

#if 0 /* FIXME */
	irq = cpssp->NAME.tif[n] & cpssp->NAME.tie[n];
	switch (n) {
	case 0: NAME_(irq0_set)(cpssp, irq); break;
#if 2 <= N
	case 1: NAME_(irq1_set)(cpssp, irq); break;
#endif
#if 3 <= N
	case 2: NAME_(irq2_set)(cpssp, irq); break;
#endif
#if 4 <= N
	case 3: NAME_(irq3_set)(cpssp, irq); break;
#endif
#if 5 <= N
	case 4: NAME_(irq4_set)(cpssp, irq); break;
#endif
#if 6 <= N
	case 5: NAME_(irq5_set)(cpssp, irq); break;
#endif
#if 7 <= N
	case 6: NAME_(irq6_set)(cpssp, irq); break;
#endif
#if 8 <= N
	case 7: NAME_(irq7_set)(cpssp, irq); break;
#endif
#if 9 <= N
	case 8: NAME_(irq8_set)(cpssp, irq); break;
#endif
#if 10 <= N
	case 9: NAME_(irq9_set)(cpssp, irq); break;
#endif
#if 11 <= N
	case 10: NAME_(irq10_set)(cpssp, irq); break;
#endif
#if 12 <= N
	case 11: NAME_(irq11_set)(cpssp, irq); break;
#endif
#if 13 <= N
	case 12: NAME_(irq12_set)(cpssp, irq); break;
#endif
#if 14 <= N
	case 13: NAME_(irq13_set)(cpssp, irq); break;
#endif
#if 15 <= N
	case 14: NAME_(irq14_set)(cpssp, irq); break;
#endif
#if 16 <= N
	case 15: NAME_(irq15_set)(cpssp, irq); break;
#endif
	default: assert(0); /* Cannot happen. */
	}
#else
	irq = 0;
	for (n = 0; n < N; n++) {
		irq |= cpssp->NAME.tif[n] & cpssp->NAME.tie[n];
	}
	NAME_(irq_set)(cpssp, irq);
#endif
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int n;

	addr &= 0x1000 - 1;
	n = (addr >> 4) & 0xf;
	*valp = 0;

	switch (addr) {
	case 0x000:
		/* PIT Module Control Register (PIT_MCR) */
		/* 31-3: reserved */
		*valp |= 1 << 2; /* Read-only */
		*valp |= cpssp->NAME.mdis << 1;
		*valp |= cpssp->NAME.frz << 0;
		break;
	case 0x0e0:
		/* PIT Upper Lifetime Register (PIT_LTMR64H) */
		cpssp->NAME.lifetime = cpssp->NAME.cval[0];
		*valp = cpssp->NAME.cval[1];
		break;
	case 0x0e4:
		/* PIT Lower Lifetime Register (PIT_LTMR64L) */
		*valp = cpssp->NAME.lifetime;
		break;
	case 0x100: case 0x110: case 0x120: case 0x130:
	case 0x140: case 0x150: case 0x160: case 0x170:
	case 0x180: case 0x190: case 0x1a0: case 0x1b0:
	case 0x1c0: case 0x1d0: case 0x1e0: case 0x1f0:
		if (N <= n) goto reserved;
		/* Timer Load Value Register (PIT_LDVALn) */
		*valp = cpssp->NAME.ldval[n];
		break;
	case 0x104: case 0x114: case 0x124: case 0x134:
	case 0x144: case 0x154: case 0x164: case 0x174:
	case 0x184: case 0x194: case 0x1a4: case 0x1b4:
	case 0x1c4: case 0x1d4: case 0x1e4: case 0x1f4:
		if (N <= n) goto reserved;
		/* Current Timer Value Register (PIT_CVALn) */
		*valp = cpssp->NAME.cval[n];
		break;
	case 0x108: case 0x118: case 0x128: case 0x138:
	case 0x148: case 0x158: case 0x168: case 0x178:
	case 0x188: case 0x198: case 0x1a8: case 0x1b8:
	case 0x1c8: case 0x1d8: case 0x1e8: case 0x1f8:
		if (N <= n) goto reserved;
		/* Timer Control Register (PIT_TCTRLn) */
		/* 31-3: reserved */
		*valp |= cpssp->NAME.chn[n] << 2;
		*valp |= cpssp->NAME.tie[n] << 1;
		*valp |= cpssp->NAME.ten[n] << 0;
		break;
	case 0x10c: case 0x11c: case 0x12c: case 0x13c:
	case 0x14c: case 0x15c: case 0x16c: case 0x17c:
	case 0x18c: case 0x19c: case 0x1ac: case 0x1bc:
	case 0x1cc: case 0x1dc: case 0x1ec: case 0x1fc:
		if (N <= n) goto reserved;
		/* Timer Flag Register (PIT_TFLGn) */
		/* 31-1: reserved */
		*valp |= cpssp->NAME.tif[n] << 0;
		break;
	default:
	reserved:
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int n;

	addr &= 0x1000 - 1;
	n = (addr >> 4) & 0xf;

	switch (addr) {
	case 0x000:
		/* PIT Module Control Register (PIT_MCR) */
		/* 31-2: reserved */
		cpssp->NAME.mdis = (val >> 1) & 1;
		cpssp->NAME.frz = (val >> 0) & 1;
		break;
	case 0x0e0:
		/* PIT Upper Lifetime Register (PIT_LTMR64H) */
		/* Read-only */
		break;
	case 0x0e4:
		/* PIT Lower Lifetime Register (PIT_LTMR64L) */
		/* Read-only */
		break;
	case 0x100: case 0x110: case 0x120: case 0x130:
	case 0x140: case 0x150: case 0x160: case 0x170:
	case 0x180: case 0x190: case 0x1a0: case 0x1b0:
	case 0x1c0: case 0x1d0: case 0x1e0: case 0x1f0:
		if (N <= n) goto reserved;
		/* Timer Load Value Register (PIT_LDVALn) */
		cpssp->NAME.ldval[n] = val;
		break;
	case 0x104: case 0x114: case 0x124: case 0x134:
	case 0x144: case 0x154: case 0x164: case 0x174:
	case 0x184: case 0x194: case 0x1a4: case 0x1b4:
	case 0x1c4: case 0x1d4: case 0x1e4: case 0x1f4:
		if (N <= n) goto reserved;
		/* Current Timer Value Register (PIT_CVALn) */
		/* Read-only */
		break;
	case 0x108: case 0x118: case 0x128: case 0x138:
	case 0x148: case 0x158: case 0x168: case 0x178:
	case 0x188: case 0x198: case 0x1a8: case 0x1b8:
	case 0x1c8: case 0x1d8: case 0x1e8: case 0x1f8:
		if (N <= n) goto reserved;
		/* Timer Control Register (PIT_TCTRLn) */
		/* 31-3: reserved */
		cpssp->NAME.chn[n] = (val >> 2) & 1;
		assert(! cpssp->NAME.chn[n]); /* FIXME */
		cpssp->NAME.tie[n] = (val >> 1) & 1;
		cpssp->NAME.ten[n] = (val >> 0) & 1;
		NAME_(update_irq)(cpssp, n);
		break;
	case 0x10c: case 0x11c: case 0x12c: case 0x13c:
	case 0x14c: case 0x15c: case 0x16c: case 0x17c:
	case 0x18c: case 0x19c: case 0x1ac: case 0x1bc:
	case 0x1cc: case 0x1dc: case 0x1ec: case 0x1fc:
		if (N <= n) goto reserved;
		/* Timer Flag Register (PIT_TFLGn) */
		/* 31-1: reserved */
		cpssp->NAME.tif[n] &= ~((val >> 0) & 1);
		NAME_(update_irq)(cpssp, n);
		break;
	default:
	reserved:
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
	}
}

/*forward*/ static void
NAME_(dec)(struct cpssp *cpssp, unsigned int n);

static void
NAME_(chn)(struct cpssp *cpssp, unsigned int n)
{
	if (N <= n) return;

	if (cpssp->NAME.ten[n]
	 && cpssp->NAME.chn[n]) {
		NAME_(dec)(cpssp, n);
	}
}

static void
NAME_(dec)(struct cpssp *cpssp, unsigned int n)
{
	if (cpssp->NAME.cval[n] == 0) {
		/* Reload */
		cpssp->NAME.cval[n] = cpssp->NAME.ldval[n];
		NAME_(chn)(cpssp, n + 1);
	} else {
		cpssp->NAME.cval[n]--;
		if (cpssp->NAME.cval[n] == 0) {
			cpssp->NAME.tif[n] = 1;
			NAME_(update_irq)(cpssp, n);
		}
	}
	NAME_(update_out)(cpssp, n);
}

static void
NAME_(clk)(struct cpssp *cpssp)
{
	unsigned int n;

	if (cpssp->NAME.mdis) {
		/* Clock disabled. */
		return;
	}

	for (n = 0; n < N; n++) {
		if (cpssp->NAME.ten[n]
		 && ! cpssp->NAME.chn[n]) {
			NAME_(dec)(cpssp, n);
		}
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int n;

	cpssp->NAME.mdis = 1;
	cpssp->NAME.frz = 0;
	cpssp->NAME.lifetime = 0x00000000;
	for (n = 0; n < N; n++) {
		cpssp->NAME.ldval[N] = 0x00000000;
		cpssp->NAME.cval[N] = 0x00000000;
		cpssp->NAME.chn[N] = 0;
		cpssp->NAME.tie[N] = 0;
		cpssp->NAME.ten[N] = 0;
		cpssp->NAME.tif[N] = 0;
		NAME_(update_out)(cpssp, n);
		NAME_(update_irq)(cpssp, n);
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	assert(N <= 16);
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
