/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For infos look at:
 * "KL02 Sub-Family Reference Manual for 48 MHz devices 32 pin package
 * Reference Manual".
 *
 * "21: Multipurpose Clock Generator (MCG)"
 */

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */

#ifdef STATE

struct {
	unsigned int count_irclk;
	unsigned int count_outclk;
	unsigned int count_fllclk;
	unsigned int count_ffclk;
	unsigned int count_pllclk;

	/* MCG Control 1 Register (MCG_C1) */
	unsigned int clks;
	unsigned int frdiv;
	unsigned int irefs;
	unsigned int irclken;
	unsigned int irefsten;

	/* MCG Control 2 Register (MCG_C2) */
	unsigned int lockre0;
#if defined(KL02Z)
	unsigned int fcftrim;
#elif defined(KL46Z)
	/* reserved */
#else
#error
#endif
	unsigned int range0;
	unsigned int hgo0;
	unsigned int erefs0;
	unsigned int lp;
	unsigned int ircs;

	/* MCG Control 3 Register (MCG_C3) */
	unsigned int sctrim;

	/* MCG Control 4 Register (MCG_C4) */
	unsigned int dmx32;
	unsigned int drst_drs;
	unsigned int fctrim;
	unsigned int scftrim;

	/* MCG Control 5 Register (MCG_C5) */
	unsigned int pllclken0;
	unsigned int pllsten0;
	unsigned int prdiv0;

	/* MCG Control 6 Register (MCG_C6) */
#if defined(KL46Z)
	uint8_t lolie0;
	uint8_t plls;
#else
	/* reserved */
#endif
	uint8_t cme0;
#if defined(KL46Z)
	uint8_t vdiv0;
#else
	/* reserved */
#endif

	/* MCG Status and Control Register (MCG_SC) */
	uint8_t atme;
	uint8_t atms;
	uint8_t atmf;
	uint8_t fltprsrv;
	uint8_t fcrdiv;
	uint8_t locs0;

	/* MCG Auto Trim Compare Value High Register (MCG_ATCVH) */
	uint8_t atcvh;

	/* MCG Auto Trim Compare Value Low Register (MCG_ATCVL) */
	uint8_t atcvl;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

#define DEBUG_CONTROL_FLOW	0

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000:
		if ((bs >> 0) & 1) {
			/* MCG Control 1 Register (MCG_C1) */
			*valp |= cpssp->NAME.clks << (6+0);
			*valp |= cpssp->NAME.frdiv << (3+0);
			*valp |= cpssp->NAME.irefs << (2+0);
			*valp |= cpssp->NAME.irclken << (1+0);
			*valp |= cpssp->NAME.irefsten << (0+0);
		}
		if ((bs >> 1) & 1) {
			/* MCG Control 2 Register (MCG_C2) */
			*valp |= cpssp->NAME.lockre0 << (7+8);
#if defined(KL02Z)
			*valp |= cpssp->NAME.fcftrim << (6+8);
#elif defined(KL46Z)
			/* 6: reserved */
#else
#error
#endif
			*valp |= cpssp->NAME.range0 << (4+8);
			*valp |= cpssp->NAME.hgo0 << (3+8);
			*valp |= cpssp->NAME.erefs0 << (2+8);
			*valp |= cpssp->NAME.lp << (1+8);
			*valp |= cpssp->NAME.ircs << (0+8);
		}
		if ((bs >> 2) & 1) {
			/* MCG Control 3 Register (MCG_C3) */
			*valp |= cpssp->NAME.sctrim << (0+16);
		}
		if ((bs >> 3) & 1) {
			/* MCG Control 4 Register (MCG_C4) */
			*valp |= cpssp->NAME.dmx32 << (7+24);
			*valp |= cpssp->NAME.drst_drs << (5+24);
			*valp |= cpssp->NAME.fctrim << (1+24);
			*valp |= cpssp->NAME.scftrim << (0+24);
		}
		break;

	case 0x004:
		if ((bs >> 0) & 1) {
			/* MCG Control 5 Register (MCG_C5) */
#if defined(KL02Z)
			*valp |= 0x00 << 0;
#elif defined(KL46Z)
			/* 7: reserved */
			*valp |= cpssp->NAME.pllclken0 << (6+0);
			*valp |= cpssp->NAME.pllsten0 << (5+0);
			*valp |= cpssp->NAME.prdiv0 << (0+0);
#else
#error
#endif
		}
		if ((bs >> 1) & 1) {
			/* MCG Control 6 Register (MCG_C6) */
#if defined(KL02Z)
			*valp |= 0 << (6+8); /* 7-6: reserved */
#elif defined(KL46Z)
			*valp |= cpssp->NAME.lolie0 << (7+8);
			*valp |= cpssp->NAME.plls << (6+8);
#else
#error
#endif
			*valp |= cpssp->NAME.cme0 << (5+8);
#if defined(KL02Z)
			*valp |= 0 << (0+8); /* 4-0: reserved */
#elif defined(KL46Z)
			*valp |= cpssp->NAME.vdiv0 << (0+8);
#else
#error
#endif
		}
		if ((bs >> 2) & 1) {
			/* MCG Status Register (MCG_S) */
#if defined(KL02Z)
			*valp |= 0b000 << (5+16); /* 7-5: reserved */
#elif defined(KL46Z)
			*valp |= 0b0 << (7+16); /* Loss of Lock Status */
			*valp |= 0b1 << (6+16); /* Lock Status */
			*valp |= cpssp->NAME.plls << (5+16); /* FIXME */
#else
#error
#endif
			*valp |= cpssp->NAME.irefs << (4+16); /* FIXME */
#if defined(KL02Z)
			*valp |= cpssp->NAME.clks << (2+16); /* FIXME */
#elif defined(KL46Z)
			if (cpssp->NAME.clks == 0
			 && cpssp->NAME.plls) {
				*valp |= 0b11 << (2+16); /* FIXME */
			} else {
				*valp |= cpssp->NAME.clks << (2+16); /* FIXME */
			}
#else
#error
#endif
			*valp |= 1 << (1+16); /* FIXME */
			*valp |= cpssp->NAME.ircs << (0+16); /* FIXME */
		}
		if ((bs >> 3) & 1) {
			/* reserved */
			*valp |= 0x00 << 24;
		}
		break;

	case 0x008:
		if ((bs >> 0) & 1) {
			/* MCG Status and Control Register (MCG_SC) */
			*valp |= cpssp->NAME.atme << (7+0);
			*valp |= cpssp->NAME.atms << (6+0);
			*valp |= cpssp->NAME.atmf << (5+0);
			*valp |= cpssp->NAME.fltprsrv << (4+0);
			*valp |= cpssp->NAME.fcrdiv << (1+0);
			*valp |= cpssp->NAME.locs0 << (0+0);
		}
		if ((bs >> 1) & 1) {
			/* reserved */
		}
		if ((bs >> 2) & 1) {
			/* MCG Auto Trim Compare Value High Register (MCG_ATCVH) */
			*valp |= cpssp->NAME.atcvh << (0+16);
		}
		if ((bs >> 3) & 1) {
			/* MCG Auto Trim Compare Value Low Register (MCG_ATCVL) */
			*valp |= cpssp->NAME.atcvl << (0+24);
		}
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		*valp = 0; /* FIXME */
		break;
	}
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x1000 - 1;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}
	switch (addr) {
	case 0x000:
		if ((bs >> 0) & 1) {
			/* MCG Control 1 Register (MCG_C1) */
			cpssp->NAME.clks = (val >> (6+0)) & 0b11;
			cpssp->NAME.frdiv = (val >> (3+0)) & 0b111;
			cpssp->NAME.irefs = (val >> (2+0)) & 0b1;
			cpssp->NAME.irclken = (val >> (1+0)) & 0b1;
			cpssp->NAME.irefsten = (val >> (0+0)) & 0b1;
		}
		if ((bs >> 1) & 1) {
			/* MCG Control 2 Register (MCG_C2) */
			cpssp->NAME.lockre0 = (val >> (7+8)) & 0b1;
#if defined(KL02Z)
			cpssp->NAME.fcftrim = (val >> (6+8)) & 0b1;
#elif defined(KL46Z)
			/* 6: reserved */
#else
#error
#endif
			cpssp->NAME.range0 = (val >> (4+8)) & 0b11;
			cpssp->NAME.hgo0 = (val >> (3+8)) & 0b1;
			cpssp->NAME.erefs0 = (val >> (2+8)) & 0b1;
			cpssp->NAME.lp = (val >> (1+8)) & 0b1;
			cpssp->NAME.ircs = (val >> (0+8)) & 0b1;
		}
		if ((bs >> 2) & 1) {
			/* MCG Control 3 Register (MCG_C3) */
			cpssp->NAME.sctrim = (val >> (0+16)) & 0b11111111;
		}
		if ((bs >> 3) & 1) {
			/* MCG Control 4 Register (MCG_C4) */
			cpssp->NAME.dmx32 = (val >> (7+24)) & 0b1;
			cpssp->NAME.drst_drs = (val >> (5+24)) & 0b11;
			cpssp->NAME.fctrim = (val >> (1+24)) & 0b1111;
			cpssp->NAME.scftrim = (val >> (0+24)) & 0b1;
		}
		break;

	case 0x004:
		if ((bs >> 0) & 1) {
			/* MCG Control 5 Register (MCG_C5) */
#if defined(KL02Z)
			/* reserved */
#elif defined(KL46Z)
			/* 7: reserved */
			cpssp->NAME.pllclken0 = (val >> 6) & 0b1;
			cpssp->NAME.pllsten0 = (val >> 5) & 0b1;
			cpssp->NAME.prdiv0 = (val >> 0) & 0b11111;
#else
#error
#endif
		}
		if ((bs >> 1) & 1) {
			/* MCG Control 6 Register (MCG_C6) */
#if defined(KL02Z)
			/* 7-6: reserved */
#elif defined(KL46Z)
			cpssp->NAME.lolie0 = (val >> (7+8)) & 1;
			cpssp->NAME.plls = (val >> (6+8)) & 1;
#else
#error
#endif
			cpssp->NAME.cme0 = (val >> (5+8)) & 1;
#if defined(KL02Z)
			/* 4-0: reserved */
#elif defined(KL46Z)
			cpssp->NAME.vdiv0 = (val >> (0+8)) & 0b11111;
#else
#error
#endif
		}
		if ((bs >> 2) & 1) {
			/* reserved */
		}
		if ((bs >> 3) & 1) {
			/* reserved */
		}
		break;

	case 0x008:
		if ((bs >> 0) & 1) {
			/* Status and Control Register */
			/* 21.3.7 */
			cpssp->NAME.atme = (val >> (7+0)) & 1;
			cpssp->NAME.atms = (val >> (6+0)) & 1;
			cpssp->NAME.atmf = 0;
			cpssp->NAME.fltprsrv = (val >> (4+0)) & 1;
			cpssp->NAME.fcrdiv = (val >> (1+0)) & 0x7;
			cpssp->NAME.locs0 = (val >> (0+0)) & 1;
		}
		if ((bs >> 1) & 1) {
			/* reserved */
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.atcvh = (val >> (0+16)) & 0xff;
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.atcvl = (val >> (0+24)) & 0xff;
		}
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

/*
 * Output clock signals.
 */
static void
NAME_(irclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_irclk += mul;
	while (div <= cpssp->NAME.count_irclk) {
		cpssp->NAME.count_irclk -= div;
		NAME_(mcgirclk)(cpssp);
	}
}

static void
NAME_(outclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_outclk += mul;
	while (div <= cpssp->NAME.count_outclk) {
		cpssp->NAME.count_outclk -= div;
		NAME_(mcgoutclk)(cpssp);
	}
}

static void
NAME_(fllclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_fllclk += mul;
	while (div <= cpssp->NAME.count_fllclk) {
		cpssp->NAME.count_fllclk -= div;
		NAME_(mcgfllclk)(cpssp);
	}
}

static void
NAME_(ffclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_ffclk += mul;
	while (div <= cpssp->NAME.count_ffclk) {
		cpssp->NAME.count_ffclk -= div;
		NAME_(mcgffclk)(cpssp);
	}
}

#if defined(KL46Z)
static void
NAME_(pllclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	cpssp->NAME.count_pllclk += mul;
	while (div <= cpssp->NAME.count_pllclk) {
		cpssp->NAME.count_pllclk -= div;
		NAME_(mcgpllclk)(cpssp);
	}
}
#endif

/*
 * Internal clock signals.
 */
static void
NAME_(ircsclk)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	if (cpssp->NAME.irclken && 1) { /* FIXME */
		NAME_(irclk)(cpssp, mul, div);
	}
	if (cpssp->NAME.clks == 1) {
		NAME_(outclk)(cpssp, mul, div);
	}
}

static void
NAME_(fllout)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
#if defined(KL02Z)
	if (cpssp->NAME.clks == 0) {
		NAME_(outclk)(cpssp, mul, div);
	}
#elif defined(KL46Z)
	if (cpssp->NAME.plls == 0
	 && cpssp->NAME.clks == 0) {
		NAME_(outclk)(cpssp, mul, div);
	}
#else
#error
#endif
	NAME_(fllclk)(cpssp, mul, div);
}

static void
NAME_(fllin)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	static const unsigned int f[] = {
		640, 732, 1280, 1464, 1920, 2197, 2560, 2929
	};

	mul *= f[(cpssp->NAME.drst_drs << 1) | (cpssp->NAME.dmx32 << 0)];

	NAME_(fllout)(cpssp, mul, div);
}

static void
NAME_(internal)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	NAME_(fllin)(cpssp, mul, div);

	NAME_(ffclk)(cpssp, mul, div);
}

#if defined(KL02Z)
/* No PLL */

#elif defined(KL46Z)
static void
NAME_(pllout)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	if (cpssp->NAME.plls == 1
	 && cpssp->NAME.clks == 0) {
		NAME_(outclk)(cpssp, mul, div);
	}
	NAME_(pllclk)(cpssp, mul, div);
}

static void
NAME_(pllin)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	NAME_(pllout)(cpssp, mul * (cpssp->NAME.vdiv0 + 24), div);
}
#else
#error
#endif

static void
NAME_(4MHz)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	div <<= cpssp->NAME.fcrdiv;

	if (cpssp->NAME.ircs == 1) {
		NAME_(ircsclk)(cpssp, mul, div);
	}
}

static void
NAME_(32kHz)(struct cpssp *cpssp, unsigned int mul, unsigned int div)
{
	if (cpssp->NAME.ircs == 0) {
		NAME_(ircsclk)(cpssp, mul, div);
	}

	if (cpssp->NAME.irefs == 1) {
		NAME_(internal)(cpssp, mul, div);
	}
}

static void
NAME_(oscclk)(struct cpssp *cpssp)
{
	if (cpssp->NAME.clks == 2) {
		NAME_(outclk)(cpssp, 1, 1);
	}

	if (cpssp->NAME.irefs == 0) {
		unsigned int count_max;

		count_max = 1 << cpssp->NAME.frdiv;
		if (cpssp->NAME.range0 != 0) {
			count_max <<= 5;
		}
		NAME_(internal)(cpssp, 1, count_max);
	}

#if defined(KL02Z)
	/* No PLL */
#elif defined(KL46Z)
	NAME_(pllin)(cpssp, 1, cpssp->NAME.prdiv0 + 1);
#else
#error
#endif
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.count_irclk = 0;
	cpssp->NAME.count_outclk = 0;
	cpssp->NAME.count_fllclk = 0;
	cpssp->NAME.count_ffclk = 0;
#if defined(KL02Z)
	/* No PLL */
#elif defined(KL46Z)
	cpssp->NAME.count_pllclk = 0;
#else
#error
#endif

	/* MCG Control 1 Register (MCG_C1) */
	cpssp->NAME.clks = 0;
	cpssp->NAME.frdiv = 0;
	cpssp->NAME.irefs = 1;
	cpssp->NAME.irclken = 0;
	cpssp->NAME.irefsten = 0;

	/* MCG Control 2 Register (MCG_C2) */
	cpssp->NAME.lockre0 = 1;
#if defined(KL02Z)
	cpssp->NAME.fcftrim = 0;
#elif defined(KL46Z)
	/* No Trim */
#else
#error
#endif
	cpssp->NAME.range0 = 0;
	cpssp->NAME.hgo0 = 0;
	cpssp->NAME.erefs0 = 0;
	cpssp->NAME.lp = 0;
	cpssp->NAME.ircs = 0;

	/* MCG Control 3 Register (MCG_C3) */
	cpssp->NAME.sctrim = 0x12; /* a factory value */

	/* MCG Control 4 Register (MCG_C4) */
	cpssp->NAME.dmx32 = 0;
	cpssp->NAME.drst_drs = 0;
	cpssp->NAME.fctrim = 0;
	cpssp->NAME.scftrim = 0;

#if defined(KL46)
	/* MCG Control 5 Register (MCG_C5) */
	cpssp->NAME.pllclken0 = 0;
	cpssp->NAME.pllsten0 = 0;
	cpssp->NAME.prdiv0 = 0;
#endif

	/* MCG Control 6 Register (MCG_C6) */
#if defined(KL46)
	cpssp->NAME.lolie0 = 0;
	cpssp->NAME.plls = 0;
#else
	/* reserved */
#endif
	cpssp->NAME.cme0 = 0;
#if defined(KL46)
	cpssp->NAME.vdiv0 = 0;
#else
	/* reserved */
#endif

	/* 21.3.6 */
	/* status */

	/* 21.3.7 */
	cpssp->NAME.atme = 0;
	cpssp->NAME.atms = 0;
	cpssp->NAME.atmf = 0;
	cpssp->NAME.fltprsrv = 0;
	cpssp->NAME.fcrdiv = 1;
	cpssp->NAME.locs0 = 0;

	/* 21.3.8 */
	cpssp->NAME.atcvh = 0;

	/* 21.3.9 */
	cpssp->NAME.atcvl = 0;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#undef DEBUG_CONTROL_FLOW

#endif /* BEHAVIOR */
