/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For infos look at:
 * "KL02 Sub-Family Reference Manual for 48 MHz devices 32 pin package
 * Reference Manual".
 *
 * "28: Low-Power Timer (LPTMR)"
 */

#define DEBUG	0

#ifdef INCLUDE

#endif /* INCLUDE */

#ifdef STATE

struct {
	unsigned int prescale_count;

	/* 28.3.1 */
	uint8_t tcf;
	uint8_t tie;
	uint8_t tps;
	uint8_t tpp;
	uint8_t tfc;
	uint8_t tms;
	uint8_t ten;

	/* 28.3.2 */
	uint8_t prescale;
	uint8_t pbyp;
	uint8_t pcs;

	/* 28.3.3 */
	uint16_t compare;

	/* 28.3.4 */
	uint16_t counter;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(_update_irq)(struct cpssp *cpssp)
{
	NAME_(irq_set)(cpssp, cpssp->NAME.tcf & cpssp->NAME.tie);
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	assert(bs == 0b1111);

	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000:
		/* Low Power Timer Control Status Register */
		/* 28.3.1 */
		*valp |= 0x000000 << 8; /* 31-8: Reserved */
		*valp |= cpssp->NAME.tcf << 7;
		*valp |= cpssp->NAME.tie << 6;
		*valp |= cpssp->NAME.tps << 4;
		*valp |= cpssp->NAME.tpp << 3;
		*valp |= cpssp->NAME.tfc << 2;
		*valp |= cpssp->NAME.tms << 1;
		*valp |= cpssp->NAME.ten << 0;
		break;

	case 0x004:
		/* Low Power Timer Prescale Register */
		/* 28.3.2 */
		*valp |= 0x000000 << 7; /* 31-7: Reserved */
		*valp |= cpssp->NAME.prescale << 3;
		*valp |= cpssp->NAME.pbyp << 2;
		*valp |= cpssp->NAME.pcs << 0;
		break;

	case 0x008:
		/* Low Power Timer Compare Register */
		/* 28.3.3 */
		*valp |= 0x0000 << 16; /* 31-16: Reserved */
		*valp |= cpssp->NAME.compare << 0;
		break;

	case 0x00c:
		/* Low Power Timer Counter Register */
		/* 28.3.4 */
		*valp |= 0x0000 << 16; /* 31-16: Reserved */
		*valp |= cpssp->NAME.counter << 0;
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		*valp = 0;
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	assert(bs == 0b1111);

	addr &= 0x1000 - 1;

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x000:
		/* Low Power Timer Control Status Register */
		/* 28.3.1 */
		/* 31-8: Reserved */
		cpssp->NAME.tcf &= ! ((val >> 7) & 0b1); /* Write-1-Clear */
		cpssp->NAME.tie = (val >> 6) & 0b1;
		cpssp->NAME.tps = (val >> 4) & 0b11;
		cpssp->NAME.tpp = (val >> 3) & 0b1;
		cpssp->NAME.tfc = (val >> 2) & 0b1;
		cpssp->NAME.tms = (val >> 1) & 0b1;
		cpssp->NAME.ten = (val >> 0) & 0b1;

		NAME_(_update_irq)(cpssp);
		break;

	case 0x004:
		/* Low Power Timer Prescale Register */
		/* 28.3.2 */
		/* 31-7: Reserved */
		cpssp->NAME.prescale = (val >> 3) & 0b1111;
		cpssp->NAME.pbyp = (val >> 2) & 0b1;
		cpssp->NAME.pcs = (val >> 0) & 0b11;
		break;

	case 0x008:
		/* Low Power Timer Compare Register */
		/* 28.3.3 */
		/* 31-16: Reserved */
		cpssp->NAME.compare = (val >> 0) & 0xffff;
		break;

	case 0x00c:
		/* Low Power Timer Counter Register */
		/* 28.3.4 */
		/* 31-16: Reserved */
		/* 15-0: Read-only */
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(_clk)(struct cpssp *cpssp)
{
	if (cpssp->NAME.ten) {
#if 0
		fprintf(stderr, "%s counter=%d compare=%d\n",
				__FUNCTION__, cpssp->NAME.counter, cpssp->NAME.compare);
#endif
		if (cpssp->NAME.counter == cpssp->NAME.compare) {
			cpssp->NAME.tcf = 1;
			NAME_(_update_irq)(cpssp);
			if (! cpssp->NAME.tfc) {
				cpssp->NAME.counter = 0;
			} else {
				cpssp->NAME.counter++;
			}
		} else {
			cpssp->NAME.counter++;
		}
		cpssp->NAME.counter &= 0xffff;
	}
}

static void
NAME_(_pulse_set)(struct cpssp *cpssp, unsigned int val)
{
	/* FIXME */
}

static void
NAME_(_clkN)(struct cpssp *cpssp, int N)
{
	if (cpssp->NAME.tms == 0
	 && cpssp->NAME.pcs == N) {
		cpssp->NAME.prescale_count++;
		if (cpssp->NAME.pbyp
		 || cpssp->NAME.prescale_count == 1 << (cpssp->NAME.prescale + 1)) {
			NAME_(_clk)(cpssp);
			cpssp->NAME.prescale_count = 0;
		}
	}
}

static void
NAME_(clk0)(struct cpssp *cpssp)
{
	NAME_(_clkN)(cpssp, 0);
}

static void
NAME_(clk1)(struct cpssp *cpssp)
{
	NAME_(_clkN)(cpssp, 1);
}

static void
NAME_(clk2)(struct cpssp *cpssp)
{
	NAME_(_clkN)(cpssp, 2);
}

static void
NAME_(clk3)(struct cpssp *cpssp)
{
	NAME_(_clkN)(cpssp, 3);
}

static void
NAME_(alt_inN_set)(struct cpssp *cpssp, int N, unsigned int val)
{
	if (cpssp->NAME.tps == N) {
		NAME_(_pulse_set)(cpssp, val);
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.prescale_count = 0;

	/* 28.3.1 */
	cpssp->NAME.tcf = 0;
	cpssp->NAME.tie = 0;
	cpssp->NAME.tps = 0;
	cpssp->NAME.tpp = 0;
	cpssp->NAME.tfc = 0;
	cpssp->NAME.tms = 0;
	cpssp->NAME.ten = 0;

	/* 28.3.2 */
	cpssp->NAME.prescale = 0;
	cpssp->NAME.pbyp = 0;
	cpssp->NAME.pcs = 0;

	/* 28.3.3 */
	cpssp->NAME.compare = 0;

	/* 28.3.4 */
	cpssp->NAME.counter = 0;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG
