/*
 * Copyright (C) 2014-2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * Inter-Integrated Circuit (I2C)
 */

#define DEBUG	0

#ifdef INCLUDE

#endif /* INCLUDE */

#ifdef STATE

struct {
	unsigned int state_sda;

	/* State Machine */
	int do_start;
	int do_data;
	int do_rstart;
	int do_stop;
	int count_scl;
	int count_sda;

	/* I2C Address Register 1 (I2Cx_A1) */
	uint8_t a1;

	/* I2C Frequency Divider register (I2Cx_F) */
	uint8_t mult;
	uint8_t mult_count;
	uint8_t icr;

	/* I2C Control Register 1 (I2Cx_C1) */
	uint8_t iicen;
	uint8_t iicie;
	uint8_t mst;
	uint8_t tx;
	uint8_t txak;
	uint8_t wuen;
#if CONFIG_DMA
	uint8_t dmaen;
#endif

	/* I2C Status register (I2Cx_S) */
	uint8_t tcf;
	uint8_t iaas;
	uint8_t busy;
	uint8_t arbl;
	uint8_t ram;
	uint8_t srw;
	uint8_t iicif;
	uint8_t rxak;

	/* I2C Data I/O Register (I2Cx_D) */
	uint8_t data;

	uint8_t fack;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static struct {
	uint16_t scl_div;
	uint16_t sda_hold;
	uint16_t scl_start;
	uint16_t scl_stop;
} NAME_(divtable)[] = {
	[0x00] = { 20, 7, 6, 11 },
	[0x01] = { 22, 7, 7, 12 },
	[0x02] = { 24, 8, 8, 13 },
	[0x03] = { 26, 8, 9, 14 },
	[0x04] = { 28, 9, 10, 15 },
	[0x05] = { 30, 9, 11, 16 },
	[0x06] = { 34, 10, 13, 18 },
	[0x07] = { 40, 10, 16, 21 },
	[0x08] = { 28, 7, 10, 15 },
	[0x09] = { 32, 7, 12, 17 },
	[0x0a] = { 36, 9, 14, 19 },
	[0x0b] = { 40, 9, 16, 21 },
	[0x0c] = { 44, 11, 18, 23 },
	[0x0d] = { 48, 11, 20, 25 },
	[0x0e] = { 56, 13, 24, 29 },
	[0x0f] = { 68, 13, 30, 35 },
	[0x10] = { 48, 9, 18, 25 },
	[0x11] = { 56, 9, 22, 29 },
	[0x12] = { 64, 13, 26, 33 },
	[0x13] = { 72, 13, 30, 37 },

	[0x14] = { 80, 17, 34, 41 },
	[0x15] = { 88, 17, 38, 45 },
	[0x16] = { 104, 21, 46, 53 },
	[0x17] = { 128, 21, 58, 65 },
	[0x18] = { 80, 9, 38, 41 },
	[0x19] = { 96, 9, 46, 49 },
	[0x1a] = { 112, 17, 54, 57 },
	[0x1b] = { 128, 17, 62, 65 },
	[0x1c] = { 144, 25, 70, 73 },
	[0x1d] = { 160, 25, 78, 81 },
	[0x1e] = { 192, 33, 94, 97 },
	[0x1f] = { 240, 33, 118, 121 },

	[0x20] = { 160, 17, 78, 81 },
	[0x21] = { 192, 17, 94, 97 },
	[0x22] = { 224, 33, 110, 113 },
	[0x23] = { 256, 33, 126, 129 },
	[0x24] = { 288, 49, 142, 145 },
	[0x25] = { 320, 49, 158, 161 },
	[0x26] = { 384, 65, 190, 193 },
	[0x27] = { 480, 65, 238, 241 },
	[0x28] = { 320, 33, 158, 161 },
	[0x29] = { 384, 33, 190, 193 },
	[0x2a] = { 448, 65, 222, 225 },
	[0x2b] = { 512, 65, 254, 257 },
	[0x2c] = { 576, 97, 286, 289 },
	[0x2d] = { 640, 97, 318, 321 },
	[0x2e] = { 768, 129, 382, 385 },
	[0x2f] = { 960, 129, 478, 481 },
	[0x30] = { 640, 65, 318, 321 },
	[0x31] = { 768, 65, 382, 385 },
	[0x32] = { 896, 129, 446, 449 },
	[0x33] = { 1024, 129, 510, 513 },

	[0x34] = { 1152, 193, 574, 577 },
	[0x35] = { 1280, 193, 638, 641 },
	[0x36] = { 1536, 257, 766, 769 },
	[0x37] = { 1920, 257, 958, 961 },
	[0x38] = { 1280, 129, 638, 641 },
	[0x39] = { 1536, 129, 766, 769 },
	[0x3a] = { 1792, 257, 894, 897 },
	[0x3b] = { 2048, 257, 1022, 1025 },
	[0x3c] = { 2304, 385, 1150, 1153 },
	[0x3d] = { 2560, 385, 1278, 1281 },
	[0x3e] = { 3072, 513, 1534, 1537 },
	[0x3f] = { 3840, 513, 1918, 1921 },
};

static void
NAME_(update)(struct cpssp *cpssp)
{
	int req;

	req = cpssp->NAME.iicif & cpssp->NAME.iicie;
#if DEBUG
	fprintf(stderr, "%s: irq=%d\n", __FUNCTION__, req);
#endif
	NAME_(irq_set)(cpssp, req);

#if CONFIG_DMA
	req = cpssp->NAME.tcf & cpssp->NAME.dmaen;
#if DEBUG
	fprintf(stderr, "%s: dma=%d\n", __FUNCTION__, req);
#endif
	NAME_(dma_set)(cpssp, req);
#endif
}

static void
NAME_(transfer_begin)(struct cpssp *cpssp)
{
	cpssp->NAME.tcf = 0;

	NAME_(update)(cpssp);
}

static void
NAME_(transfer_end)(struct cpssp *cpssp)
{
	cpssp->NAME.tcf = 1;
#if CONFIG_DMA
	if (! cpssp->NAME.dmaen)
#endif
		cpssp->NAME.iicif |= 1;

	NAME_(update)(cpssp);
}

static void
NAME_(send_start)(struct cpssp *cpssp)
{
	cpssp->NAME.do_start = 1;
}

static void
NAME_(send_restart)(struct cpssp *cpssp)
{
	cpssp->NAME.do_rstart = 1;
}

static void
NAME_(send_stop)(struct cpssp *cpssp)
{
	cpssp->NAME.do_stop = 1;
}

static void
NAME_(send_data)(struct cpssp *cpssp)
{
	cpssp->NAME.do_data = 1;
}

static void
NAME_(recv_data)(struct cpssp *cpssp)
{
	cpssp->NAME.do_data = 1;
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	assert(bs == 0b0001
	    || bs == 0b0010
	    || bs == 0b0100
	    || bs == 0b1000);

	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x00:
		if ((bs >> 0) & 1) {
			/* I2C Address Register 1 (I2Cx_A1) */
			*valp |= cpssp->NAME.a1 << 0;
		}
		if ((bs >> 1) & 1) {
			/* I2C Frequency Divider register (I2Cx_F) */
			*valp |= cpssp->NAME.mult << (6+8);
			*valp |= cpssp->NAME.icr << (0+8);
		}
		if ((bs >> 2) & 1) {
			/* I2C Control Register 1 (I2Cx_C1) */
			*valp |= cpssp->NAME.iicen << (7+16);
			*valp |= cpssp->NAME.iicie << (6+16);
			*valp |= cpssp->NAME.mst << (5+16);
			*valp |= cpssp->NAME.tx << (4+16);
			*valp |= cpssp->NAME.txak << (3+16);
			/* 2: Write-only */
			*valp |= cpssp->NAME.wuen << (1+16);
#if CONFIG_DMA
			*valp |= cpssp->NAME.dmaen << (0+16);
#else
			/* 0: Reserved */
#endif
		}
		if ((bs >> 3) & 1) {
			/* I2C Status register (I2Cx_S) */
			*valp |= cpssp->NAME.tcf << (7+24);
			*valp |= cpssp->NAME.iaas << (6+24);
			*valp |= cpssp->NAME.busy << (5+24);
			*valp |= cpssp->NAME.arbl << (4+24);
			*valp |= cpssp->NAME.ram << (3+24);
			*valp |= cpssp->NAME.srw << (2+24);
			*valp |= cpssp->NAME.iicif << (1+24);
			*valp |= cpssp->NAME.rxak << (0+24);
		}
		break;
	case 0x04:
		if ((bs >> 0) & 1) {
			/* I2C Data I/O Register (I2Cx_D) */
			*valp |= cpssp->NAME.data;
			NAME_(transfer_begin)(cpssp);
			NAME_(recv_data)(cpssp);

			power_consume(cpssp, POWER_i2c_read);
		}
		if ((bs >> 1) & 1) {
			/* I2C Control Register 2 (I2Cx_C2) */
			goto unknown;
		}
		if ((bs >> 2) & 1) {
			/* I2C Programmable Input Glitch Filter register (I2Cx_FLT) */
			goto unknown;
		}
		if ((bs >> 3) & 1) {
			/* I2C Range Address register (I2Cx_RA) */
			goto unknown;
		}
		break;
	case 0x08:
		if ((bs >> 0) & 1) {
			/* I2C SMBus Control and Status register (I2Cx_SMB) */
			goto unknown;
		}
		if ((bs >> 1) & 1) {
			/* I2C Address Register 2 (I2Cx_A2) */
			goto unknown;
		}
		if ((bs >> 2) & 1) {
			/* I2C SCL Low Timeout Register High (I2Cx_SLTH) */
			goto unknown;
		}
		if ((bs >> 3) & 1) {
			/* I2C SCL Low Timeout Register Low (I2Cx_SLTL) */
			goto unknown;
		}
		break;
	default:
	unknown:;
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		*valp = 0;
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	assert(bs == 0b0001
	    || bs == 0b0010
	    || bs == 0b0100
	    || bs == 0b1000);

	addr &= 0x1000 - 1;

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		if ((bs >> 0) & 1) {
			/* I2C Address Register 1 (I2Cx_A1) */
			cpssp->NAME.a1 = (val >> 0) & 0xfe;
		}
		if ((bs >> 1) & 1) {
			/* I2C Frequency Divider register (I2Cx_F) */
			cpssp->NAME.mult = (val >> (6+8)) & 0x3;
			cpssp->NAME.icr = (val >> (0+8)) & 0x3f;
		}
		if ((bs >> 2) & 1) {
			/* I2C Control Register 1 (I2Cx_C1) */
			uint8_t new_mst;
			uint8_t new_rsta;

			cpssp->NAME.iicen = (val >> (7+16)) & 0b1;
			cpssp->NAME.iicie = (val >> (6+16)) & 0b1;

			new_mst = ((val >> (5+16)) & 0b1);
			if (! cpssp->NAME.mst && new_mst) {
				/* Send start. */
				NAME_(send_start)(cpssp);

			} else if (cpssp->NAME.mst && ! new_mst) {
				/* Send stop. */
				NAME_(send_stop)(cpssp);
			}
			cpssp->NAME.mst = new_mst;

			cpssp->NAME.tx = (val >> (4+16)) & 0b1;
			cpssp->NAME.txak = (val >> (3+16)) & 0b1;

			new_rsta = (val >> (2+16)) & 0b1;
			if (new_rsta) {
				/* Send restart. */
				NAME_(send_restart)(cpssp);
			}

			cpssp->NAME.wuen = (val >> (1+16)) & 0b1;
#if CONFIG_DMA
			cpssp->NAME.dmaen = (val >> (0+16)) & 0b1;
#else
			/* 0: Reserved */
#endif
			NAME_(update)(cpssp);
		}
		if ((bs >> 3) & 1) {
			/* I2C Status register (I2Cx_S) */
			/* 7: Read-only */
			cpssp->NAME.iaas = (val >> (6+24)) & 0b1;
			/* 5: Read-only */
			cpssp->NAME.arbl &= ~((val >> (4+24)) & 0b1);
			cpssp->NAME.ram = (val >> (3+24)) & 0b1;
			/* 2: Read-only */
			cpssp->NAME.iicif &= ~((val >> (1+24)) & 0b1);
			/* 0: Read-only */
			NAME_(update)(cpssp);
		}
		break;
	case 0x04:
		if ((bs >> 0) & 1) {
			/* I2C Data I/O Register (I2Cx_D) */
			cpssp->NAME.data = (val >> 0) & 0xff;
			NAME_(transfer_begin)(cpssp);
			NAME_(send_data)(cpssp);

			power_consume(cpssp, POWER_i2c_write);
		}
		if ((bs >> 1) & 1) {
			/* I2C Control Register 2 (I2Cx_C2) */
			goto unknown;
		}
		if ((bs >> 2) & 1) {
			/* I2C Programmable Input Glitch Filter register (I2Cx_FLT) */
			goto unknown;
		}
		if ((bs >> 3) & 1) {
			/* I2C Range Address register (I2Cx_RA) */
			goto unknown;
		}
		break;
	case 0x08:
		if ((bs >> 0) & 1) {
			/* I2C SMBus Control and Status register (I2Cx_SMB) */
			goto unknown;
		}
		if ((bs >> 1) & 1) {
			/* I2C Address Register 2 (I2Cx_A2) */
			goto unknown;
		}
		if ((bs >> 2) & 1) {
			/* I2C SCL Low Timeout Register High (I2Cx_SLTH) */
			goto unknown;
		}
		if ((bs >> 3) & 1) {
			/* I2C SCL Low Timeout Register Low (I2Cx_SLTL) */
			goto unknown;
		}
		break;
	default:
	unknown:;
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(sda_in_set)(struct cpssp *cpssp, unsigned int val)
{
	switch (val) {
	case SIG_STD_LOGIC_0: val = 0; break;
	case SIG_STD_LOGIC_H: val = 1; break;
	default: val = 0; /* FIXME */ break;
	}

	cpssp->NAME.state_sda = val;
}

static void
NAME_(scl_in_set)(struct cpssp *cpssp, unsigned int val)
{
	/* FIXME */
}

static void
NAME_(clk2)(struct cpssp *cpssp)
{
	if (cpssp->NAME.do_stop) {
		/*
		 * Send stop.
		 */
		switch (cpssp->NAME.do_stop) {
		case 1:
			NAME_(scl_out_set)(cpssp, SIG_STD_LOGIC_Z);
			cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_stop;
			cpssp->NAME.do_stop = 2;
			break;
		case 2:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				NAME_(sda_out_set)(cpssp, SIG_STD_LOGIC_Z);
				cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_div / 2; /* FIXME */
				cpssp->NAME.do_stop = 3;
			}
			break;
		case 3:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				cpssp->NAME.do_stop = 0;
			}
			break;
		default:
			assert(0); /* Mustn't happen. */
		}

	} else if (cpssp->NAME.do_rstart) {
		/*
		 * Send repeated start.
		 */
		switch (cpssp->NAME.do_rstart) {
		case 1:
			NAME_(sda_out_set)(cpssp, SIG_STD_LOGIC_Z);
			cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_div / 2; /* FIXME */
			cpssp->NAME.do_rstart = 2;
			break;
		case 2:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				NAME_(scl_out_set)(cpssp, SIG_STD_LOGIC_Z);
				cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_div / 2; /* FIXME */
				cpssp->NAME.do_rstart = 3;
			}
			break;
		case 3:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				NAME_(sda_out_set)(cpssp, SIG_STD_LOGIC_0);
				cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_start;
				cpssp->NAME.do_rstart = 4;
			}
			break;
		case 4:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				NAME_(scl_out_set)(cpssp, SIG_STD_LOGIC_0);
				cpssp->NAME.do_rstart = 0;
			}
			break;
		default:
			assert(0); /* Mustn't happen. */
		}

	} else if (cpssp->NAME.do_start) {
		/*
		 * Send start.
		 */
		switch (cpssp->NAME.do_start) {
		case 1:
			NAME_(sda_out_set)(cpssp, SIG_STD_LOGIC_0);
			cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_start;
			cpssp->NAME.do_start = 2;
			break;
		case 2:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				NAME_(scl_out_set)(cpssp, SIG_STD_LOGIC_0);
				cpssp->NAME.do_start = 0;
			}
			break;
		default:
			assert(0); /* Mustn't happen. */
		}

	} else if (cpssp->NAME.do_data) {
		/*
		 * Send/receive data.
		 */
		switch (cpssp->NAME.do_data) {
		case 1:
			cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_div / 2;
			cpssp->NAME.count_sda = NAME_(divtable)[cpssp->NAME.icr].sda_hold;
			cpssp->NAME.do_data = 2;
			break;
		case 2: case 5: case 8: case 11:
		case 14: case 17: case 20: case 23:
			cpssp->NAME.count_scl--;
			cpssp->NAME.count_sda--;
			if (cpssp->NAME.count_sda == 0) {
				if (cpssp->NAME.tx) {
					int bit = (cpssp->NAME.data >> 7) & 1;
					cpssp->NAME.data <<= 1;

					NAME_(sda_out_set)(cpssp,
						bit ? SIG_STD_LOGIC_Z : SIG_STD_LOGIC_0);
				} else {
					NAME_(sda_out_set)(cpssp, SIG_STD_LOGIC_Z);
				}
				cpssp->NAME.do_data++;
			}
			break;
		case 3: case 6: case 9: case 12:
		case 15: case 18: case 21: case 24:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				NAME_(scl_out_set)(cpssp, SIG_STD_LOGIC_Z);
				cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_div / 2;
				cpssp->NAME.do_data++;
			}
			break;
		case 4: case 7: case 10: case 13:
		case 16: case 19: case 22: case 25:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				if (! cpssp->NAME.tx) {
					int bit = cpssp->NAME.state_sda;
					cpssp->NAME.data <<= 1;
					cpssp->NAME.data |= bit;
				}
				NAME_(scl_out_set)(cpssp, SIG_STD_LOGIC_0);
				cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_div / 2;
				cpssp->NAME.count_sda = NAME_(divtable)[cpssp->NAME.icr].sda_hold;
				cpssp->NAME.do_data++;
			}
			break;
		case 26:
			cpssp->NAME.count_scl--;
			cpssp->NAME.count_sda--;
			if (cpssp->NAME.count_sda == 0) {
				if (cpssp->NAME.tx) {
					/* Receive ACK/NACK. */
					NAME_(sda_out_set)(cpssp, SIG_STD_LOGIC_Z);
				} else {
					/* Send ACK/NACK. */
					NAME_(sda_out_set)(cpssp,
						cpssp->NAME.txak ? SIG_STD_LOGIC_Z : SIG_STD_LOGIC_0);
				}
				cpssp->NAME.do_data++;
			}
			break;
		case 27:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				NAME_(scl_out_set)(cpssp, SIG_STD_LOGIC_Z);
				cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_div / 2;
				cpssp->NAME.do_data++;
			}
			break;
		case 28:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				if (! cpssp->NAME.tx) {
					cpssp->NAME.rxak = cpssp->NAME.state_sda;
				}
				NAME_(scl_out_set)(cpssp, SIG_STD_LOGIC_0);
				cpssp->NAME.count_scl = NAME_(divtable)[cpssp->NAME.icr].scl_div / 2;
				cpssp->NAME.count_sda = NAME_(divtable)[cpssp->NAME.icr].sda_hold;
				cpssp->NAME.do_data++;
			}
			break;
		case 29:
			cpssp->NAME.count_scl--;
			cpssp->NAME.count_sda--;
			if (cpssp->NAME.count_sda == 0) {
				NAME_(sda_out_set)(cpssp, SIG_STD_LOGIC_0);
				cpssp->NAME.do_data++;
			}
			break;
		case 30:
			cpssp->NAME.count_scl--;
			if (cpssp->NAME.count_scl == 0) {
				NAME_(transfer_end)(cpssp);
				cpssp->NAME.do_data = 0;
			}
			break;
		default:
			assert(0); /* Mustn't happen. */
		}
	}
}

static void
NAME_(clk)(struct cpssp *cpssp)
{
	if (cpssp->NAME.iicen) {
		cpssp->NAME.mult_count--;
		if (cpssp->NAME.mult_count == 0) {
			NAME_(clk2)(cpssp);
			cpssp->NAME.mult_count = 1 << cpssp->NAME.mult;
		}
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* I2C Address Register (I2Cx_A1) */
	cpssp->NAME.a1 = 0x00;

	/* I2C Frequency Divider register (I2Cx_F) */
	cpssp->NAME.mult = 0b00;
	cpssp->NAME.mult_count = 0;
	cpssp->NAME.icr = 0b000000;

	/* I2C Control Register 1 (I2Cx_C1) */
	cpssp->NAME.iicen = 0;
	cpssp->NAME.iicie = 0;
	cpssp->NAME.mst = 0;
	cpssp->NAME.tx = 0;
	cpssp->NAME.txak = 0;
	cpssp->NAME.wuen = 0;
#if CONFIG_DMA
	cpssp->NAME.dmaen = 0;
#endif

	/* I2C Status register (I2Cx_S) */
	cpssp->NAME.tcf = 1;
	cpssp->NAME.iaas = 0;
	cpssp->NAME.busy = 0;
	cpssp->NAME.arbl = 0;
	cpssp->NAME.ram = 0;
	cpssp->NAME.srw = 0;
	cpssp->NAME.iicif = 0;
	cpssp->NAME.rxak = 0;

	/* I2C Data I/O Register (I2Cx_D) */
	cpssp->NAME.data = 0;

	cpssp->NAME.fack = 0;

	NAME_(update)(cpssp); /* FIXME */
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	NAME_(reset)(cpssp);
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG
