/*
 * Copyright (C) 2014 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For infos look at:
 * "KL02 Sub-Family Reference Manual for 48 MHz devices 32 pin package
 * Reference Manual".
 *
 * "32: General-Purpose Input/Output (GPIO)"
 */

#define DEBUG	1

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */

#ifdef STATE

struct {
	uint32_t pout;
	uint32_t pin;
	uint32_t pdir;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(out_update)(struct cpssp *cpssp)
{
	int i;
	int val;

	if (DEBUG) {
		fprintf(stderr, "%s: OUT=0x%08lx DIR=0x%08lx\n", __FUNCTION__,
				cpssp->NAME.pout, cpssp->NAME.pdir);
	}

	for (i = 0; i < 32; i++) {
		if ((cpssp->NAME.pdir >> i) & 1) {
			val = (cpssp->NAME.pout >> i) & 1;
			NAME_(outN_set)(cpssp, i, val ? SIG_STD_LOGIC_1 : SIG_STD_LOGIC_0);
		} else {
			NAME_(outN_set)(cpssp, i, SIG_STD_LOGIC_Z);
		}
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	assert(bs == 0b1111);

	addr &= 0x40 - 1;
	*valp = 0;

	switch (addr) {
	case 0x00:
		/* Port Data Output Register */
		/* 32.2.1 */
		*valp = cpssp->NAME.pout;
		break;

	case 0x04:
		/* Port Set Output Register */
		/* 32.2.2 */
		*valp = 0; /* Output only register */
		break;

	case 0x08:
		/* Port Clear Output Register */
		/* 32.2.3 */
		*valp = 0; /* Output only register */
		break;

	case 0x0c:
		/* Port Toggle Output Register */
		/* 32.2.4 */
		*valp = 0; /* Output only register */
		break;

	case 0x10:
		/* Port Data Input Register */
		/* 32.2.5 */
		*valp = cpssp->NAME.pin;
		break;

	case 0x14:
		/* Port Data Direction Register */
		/* 32.2.6 */
		*valp = cpssp->NAME.pdir;
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	assert(bs == 0b1111);

	addr &= 0x40 - 1;

	switch (addr) {
	case 0x00:
		/* Port Data Output Register */
		/* 32.2.1 */
		cpssp->NAME.pout = val;
		NAME_(out_update)(cpssp);
		break;

	case 0x04:
		/* Port Set Output Register */
		/* 32.2.2 */
		cpssp->NAME.pout |= val;
		NAME_(out_update)(cpssp);
		break;

	case 0x08:
		/* Port Clear Output Register */
		/* 32.2.3 */
		cpssp->NAME.pout &= ~val;
		NAME_(out_update)(cpssp);
		break;

	case 0x0c:
		/* Port Toggle Output Register */
		/* 32.2.4 */
		cpssp->NAME.pout ^= val;
		NAME_(out_update)(cpssp);
		break;

	case 0x10:
		/* Port Data Input Register */
		/* 32.2.5 */
		/* Nothing will happen. */
		break;

	case 0x14:
		/* Port Data Direction Register */
		/* 32.2.6 */
		cpssp->NAME.pdir = val;
		NAME_(out_update)(cpssp);
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(inN_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	switch (val) {
	case SIG_STD_LOGIC_0:
	case SIG_STD_LOGIC_L:
		val = 0;
		break;
	case SIG_STD_LOGIC_1:
	case SIG_STD_LOGIC_H:
		val = 1;
		break;
	default:
		val = 0; /* FIXME */
		break;
	}

	cpssp->NAME.pin &= ~(1 << n);
	cpssp->NAME.pin |= val << n;
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.pout = 0x00000000;
	cpssp->NAME.pdir = 0x00000000;
	NAME_(out_update)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG
