/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * DMA Controller
 */

#define DEBUG	0

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */

#ifdef STATE

struct {
	unsigned int request_state[4];

	/* Transfer Buffer */
	uint8_t buf[4][8];
	uint8_t bufhead[4];
	uint8_t buftail[4];
	uint8_t bufsize[4];

	/* DMA Source Address Register (DMA_SARx) */
	uint32_t sar[4];

	/* DMA Destination Address Register (DMA_DARx) */
	uint32_t dar[4];

	/* DMA Status Register / Byte Count Register (DMA_DSR_BCRx) */
	uint8_t ce[4];
	uint8_t bes[4];
	uint8_t bed[4];
	uint8_t req[4];
	uint8_t bsy[4];
	uint8_t done[4];
	uint32_t bcr[4];

	/* DMA Control Register (DMA_DCRx) */
	uint8_t eint[4];
	uint8_t erq[4];
	uint8_t cs[4];
	uint8_t aa[4];
	uint8_t eadreq[4];
	uint8_t sinc[4];
	uint8_t ssize[4];
	uint8_t dinc[4];
	uint8_t dsize[4];
	uint8_t smod[4];
	uint8_t dmod[4];
	uint8_t d_req[4];
	uint8_t linkcc[4];
	uint8_t lch1[4];
	uint8_t lch2[4];
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(update_status)(struct cpssp *cpssp, unsigned int n)
{
	unsigned int irq;

	irq = cpssp->NAME.done[n] & cpssp->NAME.eint[n];

#if DEBUG
	fprintf(stderr, "%s: n=%u, irq=%u\n", __FUNCTION__, n, irq);
#endif

	NAME_(irqN_set)(cpssp, n, irq);
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int n;

	assert(bs == 0b1111);

	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x100:
	case 0x110:
	case 0x120:
	case 0x130:
		/* DMA Source Address Register (DMA_SARx) */
		n = (addr >> 4) & (4 - 1);
		assert(0 <= n && n < 4);

		*valp = cpssp->NAME.sar[n];
		break;

	case 0x104:
	case 0x114:
	case 0x124:
	case 0x134:
		/* DMA Destination Address Register (DMA_DARx) */
		n = (addr >> 4) & (4 - 1);
		assert(0 <= n && n < 4);

		*valp = cpssp->NAME.dar[n];
		break;

	case 0x108:
	case 0x118:
	case 0x128:
	case 0x138:
		/* DMA Status Register / Byte Count Register (DMA_DSR_BCRx) */
		n = (addr >> 4) & (4 - 1);
		assert(0 <= n && n < 4);

		/* 31: Reserved */
		*valp |= cpssp->NAME.ce[n] << 30;
		*valp |= cpssp->NAME.bes[n] << 29;
		*valp |= cpssp->NAME.bed[n] << 28;
		/* 27: Reserved */
		*valp |= cpssp->NAME.req[n] << 26;
		*valp |= cpssp->NAME.bsy[n] << 25;
		*valp |= cpssp->NAME.done[n] << 24;
		*valp |= cpssp->NAME.bcr[n] << 0;
		break;

	case 0x10c:
	case 0x11c:
	case 0x12c:
	case 0x13c:
		/* DMA Control Register (DMA_DCRx) */
		n = (addr >> 4) & (4 - 1);
		assert(0 <= n && n < 4);

		*valp |= cpssp->NAME.eint[n] << 31;
		*valp |= cpssp->NAME.erq[n] << 30;
		*valp |= cpssp->NAME.cs[n] << 29;
		*valp |= cpssp->NAME.aa[n] << 28;
		/* 27-24: Reserved */
		*valp |= cpssp->NAME.eadreq[n] << 23;
		*valp |= cpssp->NAME.sinc[n] << 22;
		*valp |= cpssp->NAME.ssize[n] << 20;
		*valp |= cpssp->NAME.dinc[n] << 19;
		*valp |= cpssp->NAME.dsize[n] << 17;
		/* 16: Write-only */
		*valp |= cpssp->NAME.smod[n] << 12;
		*valp |= cpssp->NAME.dmod[n] << 8;
		*valp |= cpssp->NAME.d_req[n] << 7;
		/* 6: Reserved */
		*valp |= cpssp->NAME.linkcc[n] << 4;
		*valp |= cpssp->NAME.lch1[n] << 2;
		*valp |= cpssp->NAME.lch2[n] << 0;
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x\n",
				__FUNCTION__, addr, bs);
		*valp = 0;
		assert(0); /* FIXME */
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int n;

	assert(bs == 0b1111);

	addr &= 0x1000 - 1;

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x100:
	case 0x110:
	case 0x120:
	case 0x130:
		/* DMA Source Address Register (DMA_SARx) */
		n = (addr >> 4) & (4 - 1);
		assert(0 <= n && n < 4);

		assert((val & 0xfff00000) == 0x00000000
		    || (val & 0xfff00000) == 0x1ff00000
		    || (val & 0xfff00000) == 0x20000000
		    || (val & 0xfff00000) == 0x40000000);

		cpssp->NAME.sar[n] = val;
		break;

	case 0x104:
	case 0x114:
	case 0x124:
	case 0x134:
		/* DMA Destination Address Register (DMA_DARx) */
		n = (addr >> 4) & (4 - 1);
		assert(0 <= n && n < 4);

		assert((val & 0xfff00000) == 0x00000000
		    || (val & 0xfff00000) == 0x1ff00000
		    || (val & 0xfff00000) == 0x20000000
		    || (val & 0xfff00000) == 0x40000000);

		cpssp->NAME.dar[n] = val;
		break;

	case 0x108:
	case 0x118:
	case 0x128:
	case 0x138:
		/* DMA Status Register / Byte Count Register (DMA_DSR_BCRx) */
		n = (addr >> 4) & (4 - 1);
		assert(0 <= n && n < 4);

		assert(((val >> 0) & 0xf00000) == 0x000000);

		/* 31: Reserved */
		cpssp->NAME.ce[n] &= ~((val >> 24) & 1); /* Cleared by done. */
		cpssp->NAME.bes[n] &= ~((val >> 24) & 1); /* Cleared by done. */
		cpssp->NAME.bed[n] &= ~((val >> 24) & 1); /* Cleared by done. */
		/* 27: Reserved */
		/* 26-25: Read-only */
		cpssp->NAME.done[n] &= ~((val >> 24) & 1);
		cpssp->NAME.bcr[n] = (val >> 0) & 0xffffff;

		NAME_(update_status)(cpssp, n);
		break;

	case 0x10c:
	case 0x11c:
	case 0x12c:
	case 0x13c:
		/* DMA Control Register (DMA_DCRx) */
		n = (addr >> 4) & (4 - 1);
		assert(0 <= n && n < 4);

		cpssp->NAME.eint[n] = (val >> 31) & 1;
		cpssp->NAME.erq[n] = (val >> 30) & 1;
		cpssp->NAME.cs[n] = (val >> 29) & 1;
		cpssp->NAME.aa[n] = (val >> 28) & 1;
		/* 27-24: Reserved */
		cpssp->NAME.eadreq[n] = (val >> 23) & 1;
		cpssp->NAME.sinc[n] = (val >> 22) & 1;
		cpssp->NAME.ssize[n] = (val >> 20) & 0b11;
		cpssp->NAME.dinc[n] = (val >> 19) & 1;
		cpssp->NAME.dsize[n] = (val >> 17) & 0b11;

		/* Start */
		if (! cpssp->NAME.erq[n]) {
			cpssp->NAME.bsy[n] |= (val >> 16) & 1;
			cpssp->NAME.req[n] |= (val >> 16) & 1;
		}

		cpssp->NAME.smod[n] = (val >> 12) & 0b1111;
		cpssp->NAME.dmod[n] = (val >> 8) & 0b1111;
		cpssp->NAME.d_req[n] = (val >> 7) & 1;
		/* 6: Reserved */
		cpssp->NAME.linkcc[n] = (val >> 4) & 0b11;
		cpssp->NAME.lch1[n] = (val >> 2) & 0b11;
		cpssp->NAME.lch2[n] = (val >> 0) & 0b11;

		NAME_(update_status)(cpssp, n);
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(requestN_set)(struct cpssp *cpssp, unsigned int n, unsigned int val)
{
#if DEBUG
	fprintf(stderr, "%s: n=%u, val=%u\n", __FUNCTION__, n, val);
#endif
	assert(/* 0 <= n && */ n < 4);

	cpssp->NAME.request_state[n] = val;

	if (cpssp->NAME.erq[n]) {
		cpssp->NAME.bsy[n] |= val;
		cpssp->NAME.req[n] = val;
	}
}

static void
NAME_(clk)(struct cpssp *cpssp)
{
	unsigned int n;

#if 0
	fprintf(stderr, "%s\n", __FUNCTION__);
#endif
	for (n = 0; n < 4; n++) {
		unsigned int ssize;
		unsigned int dsize;

		if (! cpssp->NAME.bsy[n]) continue;
		if (! cpssp->NAME.req[n]) continue;

		ssize = ((cpssp->NAME.ssize[n] - 1) & 0x3) + 1;
		dsize = ((cpssp->NAME.dsize[n] - 1) & 0x3) + 1;

		if (cpssp->NAME.aa[n]) {
			if (ssize < dsize) {
				/* Align on destination size. */
				if (cpssp->NAME.dar[n] & 0x2) {
					if (2 < dsize)
						dsize = 2;
				}
				if (cpssp->NAME.dar[n] & 0x1) {
					if (1 < dsize)
						dsize = 1;
				}
				if (cpssp->NAME.bcr[n] < dsize)
					dsize = cpssp->NAME.bcr[n];
				if (dsize == 3) {
					dsize = 2;
				}
			} else {
				/* Align on source size. */
				if (cpssp->NAME.sar[n] & 0x2) {
					if (2 < ssize)
						ssize = 2;
				}
				if (cpssp->NAME.sar[n] & 0x1) {
					if (1 < ssize)
						ssize = 1;
				}
				if (cpssp->NAME.bcr[n] < ssize)
					ssize = cpssp->NAME.bcr[n];
				if (ssize == 3) {
					ssize = 2;
				}
			}
		}

#if DEBUG
		fprintf(stderr, "%s: ssize=%d, dsize=%d, sar=0x%08x, dar=0x%08x\n",
				__FUNCTION__, ssize, dsize,
				cpssp->NAME.sar[n], cpssp->NAME.dar[n]);
#endif

		assert(ssize == 1 || ssize == 2 || ssize == 4);
		assert(dsize == 1 || dsize == 2 || dsize == 4);

		if ((4 <= ssize && (cpssp->NAME.sar[n] & 0x2))
		 || (2 <= ssize && (cpssp->NAME.sar[n] & 0x1))
		 || cpssp->NAME.bcr[n] < ssize
		 || (4 <= dsize && (cpssp->NAME.dar[n] & 0x2))
		 || (2 <= dsize && (cpssp->NAME.dar[n] & 0x1))
		 || cpssp->NAME.bcr[n] < dsize) {
			/* Configuration error. */
			cpssp->NAME.ce[n] = 1;
			assert(0); /* FIXME */
			NAME_(update_status)(cpssp, n);
		}

		if (dsize <= cpssp->NAME.bufsize[n]) {
			/*
			 * Write data from buffer to bus.
			 */
			uint32_t addr;
			unsigned int bs;
			uint32_t val;
			unsigned int i;

			/* Read bytes from buffer. */
			val = 0;
			for (i = 0; i < dsize; i++) {
				val |= cpssp->NAME.buf[n][cpssp->NAME.buftail[n]] << (i * 8);
				cpssp->NAME.buftail[n] = (cpssp->NAME.buftail[n] + 1) % 8;
				cpssp->NAME.bufsize[n]--;
			}

			/* Write bytes to bus. */
			bs = 0xf >> (4 - dsize);
			bs <<= (cpssp->NAME.dar[n] & 3);
			val <<= (cpssp->NAME.dar[n] & 3) * 8;
			NAME_(out_st)(cpssp, cpssp->NAME.dar[n] & ~3, bs, val);
#if DEBUG
			fprintf(stderr, "%s: wrote addr=0x%08x, bs=0x%x, val=0x%08x\n",
					__FUNCTION__, cpssp->NAME.dar[n] & ~3,
					bs, val);
#endif

			/* Increment destination address. */
			if (cpssp->NAME.dinc[n]) {
				addr = cpssp->NAME.dar[n] + dsize;
			} else {
				addr = cpssp->NAME.dar[n];
			}

			if (cpssp->NAME.dmod[n]) {
				uint32_t mask;

				mask = (1 << (cpssp->NAME.dmod[n] + 3)) - 1;

				cpssp->NAME.dar[n]
					= (cpssp->NAME.dar[n] & ~mask)
					| (addr & mask);
			} else {
				cpssp->NAME.dar[n] = addr;
			}

			cpssp->NAME.bcr[n] -= dsize;
			if (cpssp->NAME.bcr[n] == 0) {
				/* Transfer done. */
				cpssp->NAME.bsy[n] = 0;
				cpssp->NAME.done[n] = 1;
				cpssp->NAME.erq[n] &= ~cpssp->NAME.d_req[n];
				NAME_(update_status)(cpssp, n);
#if 0 /* FIXME */
			} else if (cpssp->NAME.cs[n]
				&& cpssp->NAME.bufsize[n] == 0) {
				/* One transfer done. */
				assert(0); /* FIXME */
				cpssp->NAME.req[n] = cpssp->NAME.request_state[n] & cpssp->NAME.erq[n];
#endif
			}

		} else {
			/*
			 * Read data from bus to buffer.
			 */
			uint32_t addr;
			unsigned int bs;
			uint32_t val;
			unsigned int i;

			/* Read bytes from bus. */
			bs = 0xf >> (4 - ssize);
			bs <<= (cpssp->NAME.sar[n] & 3);
			NAME_(out_ld)(cpssp, cpssp->NAME.sar[n] & ~3, bs, &val);
#if DEBUG
			fprintf(stderr, "%s: read addr=0x%08x, bs=0x%x, val=0x%08x\n",
					__FUNCTION__, cpssp->NAME.sar[n] & ~3,
					bs, val);
#endif
			val >>= (cpssp->NAME.sar[n] & 3) * 8;

			/* Write bytes to buffer. */
			for (i = 0; i < ssize; i++) {
				cpssp->NAME.buf[n][cpssp->NAME.bufhead[n]] = (val >> (i * 8)) & 0xff;
				cpssp->NAME.bufhead[n] = (cpssp->NAME.bufhead[n] + 1) % 8;
				cpssp->NAME.bufsize[n]++;
			}

			/* Increment source address. */
			if (cpssp->NAME.sinc[n]) {
				addr = cpssp->NAME.sar[n] + ssize;
			} else {
				addr = cpssp->NAME.sar[n];
			}

			if (cpssp->NAME.smod[n]) {
				uint32_t mask;

				mask = (1 << (cpssp->NAME.smod[n] + 3)) - 1;

				cpssp->NAME.sar[n]
					= (cpssp->NAME.sar[n] & ~mask)
					| (addr & mask);
			} else {
				cpssp->NAME.sar[n] = addr;
			}
		}
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int n;

	for (n = 0; n < 4; n++) {
		/* Transfer Buffer */
		cpssp->NAME.bufhead[n] = 0;
		cpssp->NAME.buftail[n] = 0;
		cpssp->NAME.bufsize[n] = 0;

		/* DMA Source Address Register (DMA_SARx) */
		cpssp->NAME.sar[n] = 0x00000000;

		/* DMA Destination Address Register (DMA_DARx) */
		cpssp->NAME.dar[n] = 0x00000000;

		/* DMA Status Register / Byte Count Register (DMA_DSR_BCRx) */
		cpssp->NAME.ce[n] = 0;
		cpssp->NAME.bes[n] = 0;
		cpssp->NAME.bed[n] = 0;
		cpssp->NAME.bsy[n] = 0;
		cpssp->NAME.done[n] = 0;
		cpssp->NAME.bcr[n] = 0x000000;

		/* DMA Control Register (DMA_DCRx) */
		cpssp->NAME.eint[n] = 0;
		cpssp->NAME.erq[n] = 0;
		cpssp->NAME.cs[n] = 0;
		cpssp->NAME.aa[n] = 0;
		cpssp->NAME.eadreq[n] = 0;
		cpssp->NAME.sinc[n] = 0;
		cpssp->NAME.ssize[n] = 0;
		cpssp->NAME.dinc[n] = 0;
		cpssp->NAME.dsize[n] = 0;
		cpssp->NAME.smod[n] = 0;
		cpssp->NAME.dmod[n] = 0;
		cpssp->NAME.d_req[n] = 0;
		cpssp->NAME.linkcc[n] = 0;
		cpssp->NAME.lch1[n] = 0;
		cpssp->NAME.lch2[n] = 0;
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	unsigned int n;

	for (n = 0; n < 4; n++) {
		cpssp->NAME.request_state[n] = 0;
	}
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG
