/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifdef INCLUDE

#include <assert.h>
#include <stdio.h>

#endif /* INCLUDE */

#ifdef STATE

struct {
	uint16_t ref_state[2];

	/* DACx_DATnm */
	uint16_t dacdata[2];

	/* DACx_SR */
	uint8_t dacbfrptf;
	uint8_t dacbfrpbf;

	/* DACx_C0 */
	uint8_t dacen;
	uint8_t dacrfs;
	uint8_t dactrgsel;
	uint8_t daclpen;
	uint8_t dacbtien;
	uint8_t dacbbien;

	/* DACx_C1 */
	uint8_t dacdmaen;
	uint8_t dacbfmd;
	uint8_t dacbfen;

	/* DACx_C2 */
	uint8_t dacbfrp;
	uint8_t dacbfup;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(update_out)(struct cpssp *cpssp)
{
	unsigned int out;

	if (! cpssp->NAME.dacen) {
		/*
		 * DAC disabled.
		 */
		out = SIG_STD_LOGIC_Z;

	} else {
		/*
		 * DAC enabled.
		 */
		int idx;
		uint16_t val;
		uint16_t ref;

		idx = cpssp->NAME.dacbfrp & cpssp->NAME.dacbfen;
		val = cpssp->NAME.dacdata[idx];
		ref = cpssp->NAME.ref_state[cpssp->NAME.dacrfs];
		out = SIG_COMB(val * ref / 4096, -1);
	}

	NAME_(out_out_set)(cpssp, out);
}

static void
NAME_(update_irq)(struct cpssp *cpssp)
{
	unsigned int irq;

	cpssp->NAME.dacbfrptf = cpssp->NAME.dacbfrp == 0;
	cpssp->NAME.dacbfrpbf = cpssp->NAME.dacbfrp == cpssp->NAME.dacbfup;

	irq = 0 /* FIXME */
		| (cpssp->NAME.dacbtien & cpssp->NAME.dacbfrptf)
		| (cpssp->NAME.dacbbien & cpssp->NAME.dacbfrpbf);

	NAME_(irq_set)(cpssp, irq);
}

static void
NAME_(trig)(struct cpssp *cpssp)
{
	if (cpssp->NAME.dacbfrp == cpssp->NAME.dacbfup) {
		if (! cpssp->NAME.dacbfmd) {
			/* Normal mode. */
			cpssp->NAME.dacbfrp = 0;
		} else {
			/* One-Time Scan mode. */
			/* Nothing to do... */
		}
	} else {
		cpssp->NAME.dacbfrp ^= 1;
	}
	NAME_(update_out)(cpssp);
	NAME_(update_irq)(cpssp);
}

#if 0
/*
 * PIT Channel 0 is configured as the DAC hardware trigger source.
 */
static void
NAME_(hw_trig)(struct cpssp *cpssp)
{
	if (cpssp->NAME.dactrgsel == 0) {
		NAME_(trig)(cpssp);
	}
}
#endif

static void
NAME_(sw_trig)(struct cpssp *cpssp)
{
	if (cpssp->NAME.dactrgsel == 1) {
		NAME_(trig)(cpssp);
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x000:
		if ((bs >> 3) & 1) {
			/* DAC Data High Register 1 (DACx_DAT1H) */
			*valp |= (cpssp->NAME.dacdata[1] & 0xff00) << 16;
		}
		if ((bs >> 2) & 1) {
			/* DAC Data Low Register 1 (DACx_DAT1L) */
			*valp |= (cpssp->NAME.dacdata[1] & 0x00ff) << 16;
		}
		if ((bs >> 1) & 1) {
			/* DAC Data High Register 0 (DACx_DAT0H) */
			*valp |= (cpssp->NAME.dacdata[0] & 0xff00) << 0;
		}
		if ((bs >> 0) & 1) {
			/* DAC Data Low Register 0 (DACx_DAT0L) */
			*valp |= (cpssp->NAME.dacdata[0] & 0x00ff) << 0;
		}
		break;
	case 0x020:
		assert(bs == 0b0001
		    || bs == 0b0010
		    || bs == 0b0100
		    || bs == 0b1000); /* Do not use 32- or 16-bit accesses. */
		if ((bs >> 3) & 1) {
			/* DAC Control Register 2 (DACx_C2) */
			/* 7-5: reserved */
			*valp |= cpssp->NAME.dacbfrp << (4+24);
			/* 3-1: reserved */
			*valp |= cpssp->NAME.dacbfup << (0+24);
		}
		if ((bs >> 2) & 1) {
			/* DAC Control Register 1 (DACx_C1) */
			*valp |= cpssp->NAME.dacdmaen << (7+16);
			/* 6-3: reserved */
			*valp |= cpssp->NAME.dacbfmd << (2+16);
			/* 1: reserved */
			*valp |= cpssp->NAME.dacbfen << (0+16);
		}
		if ((bs >> 1) & 1) {
			/* DAC Control Register 0 (DACx_C0) */
			*valp |= cpssp->NAME.dacen << (7+8);
			*valp |= cpssp->NAME.dacrfs << (6+8);
			*valp |= cpssp->NAME.dactrgsel << (5+8);
			*valp |= 0 << (4+8); /* Write-only */
			*valp |= cpssp->NAME.dactrgsel << (3+8);
			/* 2: reserved */
			*valp |= cpssp->NAME.dacbtien << (1+8);
			*valp |= cpssp->NAME.dacbbien << (0+8);
		}
		if ((bs >> 0) & 1) {
			/* DAC Status Register (DACx_SR) */
			/* 7-2: reserved */
			*valp |= cpssp->NAME.dacbfrptf << 1;
			*valp |= cpssp->NAME.dacbfrpbf << 0;
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	addr &= 0x1000 - 1;

	switch (addr) {
	case 0x000:
		if ((bs >> 3) & 1) {
			/* DAC Data High Register 1 (DACx_DAT1H) */
			cpssp->NAME.dacdata[1] &= ~0xff00;
			cpssp->NAME.dacdata[1] |= (val >> 16) & 0x0f00;
		}
		if ((bs >> 2) & 1) {
			/* DAC Data Low Register 1 (DACx_DAT1L) */
			cpssp->NAME.dacdata[1] &= ~0x00ff;
			cpssp->NAME.dacdata[1] |= (val >> 16) & 0x00ff;
		}
		if ((bs >> 1) & 1) {
			/* DAC Data High Register 0 (DACx_DAT0H) */
			cpssp->NAME.dacdata[0] &= ~0xff00;
			cpssp->NAME.dacdata[0] |= (val >> 0) & 0x0f00;
		}
		if ((bs >> 0) & 1) {
			/* DAC Data Low Register 0 (DACx_DAT0L) */
			cpssp->NAME.dacdata[0] &= ~0x00ff;
			cpssp->NAME.dacdata[0] |= (val >> 0) & 0x00ff;
		}
		NAME_(update_out)(cpssp);
		break;
	case 0x020:
		assert(bs == 0b0001
		    || bs == 0b0010
		    || bs == 0b0100
		    || bs == 0b1000); /* Do not use 32- or 16-bit accesses. */
		if ((bs >> 3) & 1) {
			/* DAC Control Register 2 (DACx_C2) */
			/* 7-5: reserved */
			cpssp->NAME.dacbfrp = (val >> (4+24)) & 1;
			/* 3-1: reserved */
			cpssp->NAME.dacbfup = (val >> (0+24)) & 1;
		}
		if ((bs >> 2) & 1) {
			/* DAC Control Register 1 (DACx_C1) */
			cpssp->NAME.dacdmaen = (val >> (7+16)) & 1;
			assert(! cpssp->NAME.dacdmaen); /* FIXME */
			/* 6-3: reserved */
			cpssp->NAME.dacbfmd = (val >> (2+16)) & 1;
			/* 1: reserved */
			cpssp->NAME.dacbfen = (val >> (0+16)) & 1;
		}
		if ((bs >> 1) & 1) {
			/* DAC Control Register 0 (DACx_C0) */
			cpssp->NAME.dacen = (val >> (7+8)) & 1;
			cpssp->NAME.dacrfs = (val >> (6+8)) & 1;
			cpssp->NAME.dactrgsel = (val >> (5+8)) & 1;
			if ((val >> (4+8)) & 1) {
				NAME_(sw_trig)(cpssp);
			}
			cpssp->NAME.daclpen = (val >> (3+8)) & 1;
			/* 2: reserved */
			cpssp->NAME.dacbtien = (val >> (1+8)) & 1;
			cpssp->NAME.dacbbien = (val >> (0+8)) & 1;
		}
		if ((bs >> 0) & 1) {
			/* DAC Status Register (DACx_SR) */
			/* 7-2: reserved */
			cpssp->NAME.dacbfrptf &= (val >> 1) & 1;
			cpssp->NAME.dacbfrpbf &= (val >> 0) & 1;
		}
		NAME_(update_out)(cpssp);
		NAME_(update_irq)(cpssp);
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
	}
}

static void
NAME_(refN_set)(struct cpssp *cpssp, unsigned int n, unsigned int val)
{
	cpssp->NAME.ref_state[n] = val;
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	/* DACx_DAT0m */
	cpssp->NAME.dacdata[0] = 0x0000;
	/* DACx_DAT1m */
	cpssp->NAME.dacdata[1] = 0x0000;

	/* DACx_SR */
	/* Done below using update_irq()... */

	/* DACx_C0 */
	cpssp->NAME.dacen = 0;
	cpssp->NAME.dacrfs = 0;
	cpssp->NAME.dactrgsel = 0;
	cpssp->NAME.daclpen = 0;
	cpssp->NAME.dacbtien = 0;
	cpssp->NAME.dacbbien = 0;

	/* DACx_C1 */
	cpssp->NAME.dacdmaen = 0;
	cpssp->NAME.dacbfmd = 0;
	cpssp->NAME.dacbfen = 0;

	/* DACx_C2 */
	cpssp->NAME.dacbfrp = 0;
	cpssp->NAME.dacbfup = 1;

	NAME_(update_out)(cpssp);
	NAME_(update_irq)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
