/*
 * Copyright (C) 2015 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * ADC
 */

#define DEBUG	0

#ifdef INCLUDE

#endif /* INCLUDE */

#ifdef STATE

struct {
	/*
	 * Numbering of vin signals:
	 *
	 * 0..3: DP0 .. DP3 / SE0 .. SE3
	 * 4..7: DM0 .. DM3 / SE4a .. SE7a
	 * 8..23: SE8 .. SE23
	 * 24..17: SE4b .. SE7b
	 */
	unsigned int state_vin[28];

	unsigned int state_ref[2];

	unsigned int state[2];

	/* ADC Status and Control Register 1 (ADC0_SC1A) */
	/* ADC Status and Control Register 1 (ADC0_SC1B) */
	uint8_t coco[2];
	uint8_t aien[2];
	uint8_t diff[2];
	uint8_t adch[2];

	/* ADC Configuration Register 1 (ADC0_CFG1) */
	uint8_t adlpc;
	uint8_t adiv;
	uint8_t adlsmp;
	uint8_t mode;
	uint8_t adiclk;

	/* ADC Configuration Register 2 (ADC0_CFG2) */
	uint8_t muxsel;
	uint8_t adacken;
	uint8_t adhsc;
	uint8_t adlsts;

	/* ADC Data Result Register (ADC0_RA) */
	/* ADC Data Result Register (ADC0_RB) */
	uint16_t r[2];

	/* Compare Value Registers (ADC0_CV1) */
	uint16_t cv1;

	/* Compare Value Registers (ADC0_CV2) */
	uint16_t cv2;

	/* Status and Control Register 2 (ADC0_SC2) */
	uint8_t adtrg;
	uint8_t acfe;
	uint8_t acfgt;
	uint8_t acren;
	uint8_t dmaen;
	uint8_t refsel;

	/* Status and Control Register 3 (ADC0_SC3) */
	uint8_t cal;
	uint8_t calf;
	uint8_t adco;
	uint8_t avge;
	uint8_t avgs;

	/* ADC Offset Correction Register (ADC0_OFS) */
	uint16_t ofs;

	/* ADC Plus-Side Gain Register (ADC0_PG) */
	uint16_t pg;

	/* ADC Minus-Side Gain Register (ADC0_MG) */
	uint16_t mg;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLPD) */
	uint8_t clpd;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLPS) */
	uint8_t clps;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP4) */
	uint16_t clp4;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP3) */
	uint16_t clp3;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP2) */
	uint8_t clp2;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP1) */
	uint8_t clp1;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP0) */
	uint8_t clp0;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLMD) */
	uint8_t clmd;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLMS) */
	uint8_t clms;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM4) */
	uint16_t clm4;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM3) */
	uint16_t clm3;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM2) */
	uint8_t clm2;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM1) */
	uint8_t clm1;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM0) */
	uint8_t clm0;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static void
NAME_(irq_update)(struct cpssp *cpssp)
{
	int irq;

	irq = (cpssp->NAME.coco[0] & cpssp->NAME.aien[0])
			| (cpssp->NAME.coco[1] & cpssp->NAME.aien[1]);

	NAME_(irq_set)(cpssp, irq);
}

static void
NAME_(start)(struct cpssp *cpssp, unsigned int n)
{
	cpssp->NAME.coco[n] = 0;
	NAME_(irq_update)(cpssp);

	if (cpssp->NAME.adch[n] != 0b11111) {
		cpssp->NAME.state[n] = 1;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int n;

	addr &= 0x1000 - 1;
	*valp = 0;

	switch (addr) {
	case 0x00:
		/* ADC Status and Control Register 1 (ADC0_SC1A) */
	case 0x04:
		/* ADC Status and Control Register 1 (ADC0_SC1B) */
		n = (addr >> 2) & 1;

		/* 31-8: reserved */
		*valp |= cpssp->NAME.coco[n] << 7;
		*valp |= cpssp->NAME.aien[n] << 6;
		*valp |= cpssp->NAME.diff[n] << 5;
		*valp |= cpssp->NAME.adch[n] << 0;
		break;

	case 0x08:
		/* ADC Configuration Register 1 (ADC0_CFG1) */
		/* 31-8: reserved */
		*valp |= cpssp->NAME.adlpc << 7;
		*valp |= cpssp->NAME.adiv << 5;
		*valp |= cpssp->NAME.adlsmp << 4;
		*valp |= cpssp->NAME.mode << 2;
		*valp |= cpssp->NAME.adiclk << 0;
		break;

	case 0x0c:
		/* ADC Configuration Register 2 (ADC0_CFG2) */
		/* 31-5: reserved */
		*valp |= cpssp->NAME.muxsel << 4;
		*valp |= cpssp->NAME.adacken << 3;
		*valp |= cpssp->NAME.adhsc << 2;
		*valp |= cpssp->NAME.adlsts << 0;
		break;

	case 0x10:
		/* ADC Data Result Register (ADC0_RA) */
	case 0x14:
		/* ADC Data Result Register (ADC0_RB) */
		n = (addr >> 2) & 1;

		/* 31-16: reserved */
		*valp |= cpssp->NAME.r[n] << 0;

		cpssp->NAME.coco[n] = 0;
		NAME_(irq_update)(cpssp);

		power_consume(cpssp, POWER_adc_read);
		break;

	case 0x18:
		/* Compare Value Registers (ADC0_CV1) */
		/* 31-16: reserved */
		*valp |= cpssp->NAME.cv1 << 0;
		break;

	case 0x1c:
		/* Compare Value Registers (ADC0_CV2) */
		/* 31-16: reserved */
		*valp |= cpssp->NAME.cv2 << 0;
		break;

	case 0x20:
		/* Status and Control Register 2 (ADC0_SC2) */
		/* 31-8: reserved */
		*valp |= ((cpssp->NAME.state[0] != 0) | (cpssp->NAME.state[1] != 0)) << 7;
		*valp |= cpssp->NAME.adtrg << 6;
		*valp |= cpssp->NAME.acfe << 5;
		*valp |= cpssp->NAME.acfgt << 4;
		*valp |= cpssp->NAME.acren << 3;
		*valp |= cpssp->NAME.dmaen << 2;
		*valp |= cpssp->NAME.refsel << 0;
		break;

	case 0x24:
		/* Status and Control Register 3 (ADC0_SC3) */
		/* 31-8: reserved */
		*valp |= cpssp->NAME.calf << 7;
		/* 5-4: reserved */
		*valp |= cpssp->NAME.adco << 3;
		*valp |= cpssp->NAME.avge << 2;
		*valp |= cpssp->NAME.avgs << 0;
		break;

	case 0x28:
		/* ADC Offset Correction Register (ADC0_OFS) */
		/* 31-16: reserved */
		*valp |= cpssp->NAME.ofs << 0;
		break;

	case 0x2c:
		/* ADC Plus-Side Gain Register (ADC0_PG) */
		/* 31-16: reserved */
		*valp |= cpssp->NAME.pg << 0;
		break;

	case 0x30:
		/* ADC Minus-Side Gain Register (ADC0_MG) */
		/* 31-16: reserved */
		*valp |= cpssp->NAME.mg << 0;
		break;

	case 0x34:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLPD) */
		/* 31-6: reserved */
		*valp |= cpssp->NAME.clpd << 0;
		break;

	case 0x38:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLPS) */
		/* 31-6: reserved */
		*valp |= cpssp->NAME.clps << 0;
		break;

	case 0x3c:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP4) */
		/* 31-10: reserved */
		*valp |= cpssp->NAME.clp4 << 0;
		break;

	case 0x40:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP3) */
		/* 31-9: reserved */
		*valp |= cpssp->NAME.clp3 << 0;
		break;

	case 0x44:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP2) */
		/* 31-8: reserved */
		*valp |= cpssp->NAME.clp2 << 0;
		break;

	case 0x48:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP1) */
		/* 31-7: reserved */
		*valp |= cpssp->NAME.clp1 << 0;
		break;

	case 0x4c:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP0) */
		/* 31-6: reserved */
		*valp |= cpssp->NAME.clp0 << 0;
		break;

	case 0x54:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLMD) */
		/* 31-6: reserved */
		*valp |= cpssp->NAME.clmd << 0;
		break;

	case 0x58:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLMS) */
		/* 31-6: reserved */
		*valp |= cpssp->NAME.clms << 0;
		break;

	case 0x5c:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM4) */
		/* 31-10: reserved */
		*valp |= cpssp->NAME.clm4 << 0;
		break;

	case 0x60:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM3) */
		/* 31-9: reserved */
		*valp |= cpssp->NAME.clm3 << 0;
		break;

	case 0x64:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM2) */
		/* 31-8: reserved */
		*valp |= cpssp->NAME.clm2 << 0;
		break;

	case 0x68:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM1) */
		/* 31-7: reserved */
		*valp |= cpssp->NAME.clm1 << 0;
		break;

	case 0x6c:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM0) */
		/* 31-6: reserved */
		*valp |= cpssp->NAME.clm0 << 0;
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%03x bs=0x%x\n",
				__FUNCTION__, addr, bs);
		assert(0); /* FIXME */
		*valp = 0;
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int n;

	addr &= 0x1000 - 1;

	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%03x bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		/* ADC Status and Control Register 1 (ADC0_SC1A) */
	case 0x04:
		/* ADC Status and Control Register 1 (ADC0_SC1B) */
		n = (addr >> 2) & 1;

		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			/* 7: read-only */
			cpssp->NAME.aien[n] = (val >> 6) & 1;
			cpssp->NAME.diff[n] = (val >> 5) & 1;
			cpssp->NAME.adch[n] = (val >> 0) & 0b11111;
			NAME_(start)(cpssp, n);
		}
		break;

	case 0x08:
		/* ADC Configuration Register 1 (ADC0_CFG1) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			cpssp->NAME.adlpc = (val >> 7) & 1;
			cpssp->NAME.adiv = (val >> 5) & 0b11;
			cpssp->NAME.adlsmp = (val >> 4) & 1;
			cpssp->NAME.mode = (val >> 2) & 0b11;
			cpssp->NAME.adiclk = (val >> 0) & 0b11;
		}
		break;

	case 0x0c:
		/* ADC Configuration Register 2 (ADC0_CFG2) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			/* 7-5: reserved */
			cpssp->NAME.muxsel = (val >> 4) & 1;
			cpssp->NAME.adacken = (val >> 3) & 1;
			cpssp->NAME.adhsc = (val >> 2) & 1;
			cpssp->NAME.adlsts = (val >> 0) & 0b11;
		}
		break;
	case 0x10:
		/* ADC Data Result Register (ADC0_RA) */
		/* 31-16: reserved */
		/* 15-0: read-only */
		break;

	case 0x14:
		/* ADC Data Result Register (ADC0_RB) */
		/* 31-16: reserved */
		/* 15-0: read-only */
		break;

	case 0x18:
		/* Compare Value Registers (ADC0_CV1) */
		/* 31-16: reserved */
		if ((bs >> 1) & 1) {
			cpssp->NAME.cv1 &= ~(0xff << 8);
			cpssp->NAME.cv1 |= val & (0xff << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.cv1 &= ~(0xff << 0);
			cpssp->NAME.cv1 |= val & (0xff << 0);
		}
		break;

	case 0x1c:
		/* Compare Value Registers (ADC0_CV2) */
		/* 31-16: reserved */
		if ((bs >> 1) & 1) {
			cpssp->NAME.cv2 &= ~(0xff << 8);
			cpssp->NAME.cv2 |= val & (0xff << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.cv2 &= ~(0xff << 0);
			cpssp->NAME.cv2 |= val & (0xff << 0);
		}
		break;

	case 0x20:
		/* Status and Control Register 2 (ADC0_SC2) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			/* 7: read-only */
			cpssp->NAME.adtrg = (val >> 6) & 1;
			cpssp->NAME.acfe = (val >> 5) & 1;
			cpssp->NAME.acfgt = (val >> 4) & 1;
			cpssp->NAME.acren = (val >> 3) & 1;
			cpssp->NAME.dmaen = (val >> 2) & 1;
			cpssp->NAME.refsel = (val >> 0) & 0b11;
		}
		break;

	case 0x24:
		/* Status and Control Register 3 (ADC0_SC3) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			cpssp->NAME.cal = (val >> 7) & 1;
			cpssp->NAME.calf &= ~((val >> 6) & 1);
			/* 5-4: reserved */
			cpssp->NAME.adco = (val >> 3) & 1;
			cpssp->NAME.avge = (val >> 2) & 1;
			cpssp->NAME.avgs = (val >> 0) & 0b11;
		}
		break;

	case 0x28:
		/* ADC Offset Correction Register (ADC0_OFS) */
		/* 31-16: reserved */
		if ((bs >> 1) & 1) {
			cpssp->NAME.ofs &= ~(0xff << 8);
			cpssp->NAME.ofs |= val & (0xff << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.ofs &= ~(0xff << 0);
			cpssp->NAME.ofs |= val & (0xff << 0);
		}
		break;

	case 0x2c:
		/* ADC Plus-Side Gain Register (ADC0_PG) */
		/* 31-16: reserved */
		if ((bs >> 1) & 1) {
			cpssp->NAME.pg &= ~(0xff << 8);
			cpssp->NAME.pg |= val & (0xff << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.pg &= ~(0xff << 0);
			cpssp->NAME.pg |= val & (0xff << 0);
		}
		break;

	case 0x30:
		/* ADC Minus-Side Gain Register (ADC0_MG) */
		/* 31-16: reserved */
		if ((bs >> 1) & 1) {
			cpssp->NAME.mg &= ~(0xff << 8);
			cpssp->NAME.mg |= val & (0xff << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.mg &= ~(0xff << 0);
			cpssp->NAME.mg |= val & (0xff << 0);
		}
		break;

	case 0x34:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLPD) */
		/* 31-8; reserved */
		if ((bs >> 0) & 1) {
			/* 7-6; reserved */
			cpssp->NAME.clpd = (val >> 0) & 0b111111;
		}
		break;

	case 0x38:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLPS) */
		/* 31-8; reserved */
		if ((bs >> 0) & 1) {
			/* 7-6; reserved */
			cpssp->NAME.clps = (val >> 0) & 0b111111;
		}
		break;

	case 0x3c:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP4) */
		/* 31-16: reserved */
		if ((bs >> 1) & 1) {
			/* 15-10: reserved */
			cpssp->NAME.clp4 &= ~(0x03 << 8);
			cpssp->NAME.clp4 |= val & (0x03 << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.clp4 &= ~(0xff << 0);
			cpssp->NAME.clp4 |= val & (0xff << 0);
		}
		break;

	case 0x40:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP3) */
		/* 31-16: reserved */
		if ((bs >> 1) & 1) {
			/* 15-9: reserved */
			cpssp->NAME.clp3 &= ~(0x01 << 8);
			cpssp->NAME.clp3 |= val & (0x01 << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.clp3 &= ~(0xff << 0);
			cpssp->NAME.clp3 |= val & (0xff << 0);
		}
		break;

	case 0x44:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP2) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			cpssp->NAME.clp2 = val & 0xff;
		}
		break;
	case 0x48:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP1) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			/* 7: reserved */
			cpssp->NAME.clp1 &= ~(0x7f << 0);
			cpssp->NAME.clp1 |= val & (0x7f << 0);
		}
		break;

	case 0x4c:
		/* ADC Plus-Side General Calibration Value Register (ADC0_CLP0) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			/* 7-6: reserved */
			cpssp->NAME.clp0 &= ~(0x3f << 0);
			cpssp->NAME.clp0 |= val & (0x3f << 0);
		}
		break;

	case 0x54:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLMD) */
		/* 31-8; reserved */
		if ((bs >> 0) & 1) {
			/* 7-6; reserved */
			cpssp->NAME.clmd = (val >> 0) & 0b111111;
		}
		break;

	case 0x58:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLMS) */
		/* 31-8; reserved */
		if ((bs >> 0) & 1) {
			/* 7-6; reserved */
			cpssp->NAME.clms = (val >> 0) & 0b111111;
		}
		break;

	case 0x5c:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM4) */
		/* 31-16: reserved */
		if ((bs >> 1) & 1) {
			/* 15-10: reserved */
			cpssp->NAME.clm4 &= ~(0x03 << 8);
			cpssp->NAME.clm4 |= val & (0x03 << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.clm4 &= ~(0xff << 0);
			cpssp->NAME.clm4 |= val & (0xff << 0);
		}
		break;

	case 0x60:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM3) */
		/* 31-16: reserved */
		if ((bs >> 1) & 1) {
			/* 15-9: reserved */
			cpssp->NAME.clm3 &= ~(0x01 << 8);
			cpssp->NAME.clm3 |= val & (0x01 << 8);
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.clm3 &= ~(0xff << 0);
			cpssp->NAME.clm3 |= val & (0xff << 0);
		}
		break;

	case 0x64:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM2) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			cpssp->NAME.clm2 = val & 0xff;
		}
		break;

	case 0x68:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM1) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			/* 7: reserved */
			cpssp->NAME.clm1 &= ~(0x7f << 0);
			cpssp->NAME.clm1 |= val & (0x7f << 0);
		}
		break;

	case 0x6c:
		/* ADC Minus-Side General Calibration Value Register (ADC0_CLM0) */
		/* 31-8: reserved */
		if ((bs >> 0) & 1) {
			/* 7-6: reserved */
			cpssp->NAME.clm0 &= ~(0x3f << 0);
			cpssp->NAME.clm0 |= val & (0x3f << 0);
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx bs=0x%x val=0x%08lx\n",
				__FUNCTION__, addr, bs, val);
		assert(0); /* FIXME */
		break;
	}
}

static void
NAME_(se_inN_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(0 <= n && n < 28);

	cpssp->NAME.state_vin[n] = SIG_mV(val);
}

#if defined(KL46Z)
static void
NAME_(dp_inN_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, n + 0, val);
}

static void
NAME_(dm_inN_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, n + 4, val);
}

static void
NAME_(se4a_in_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, 4, val);
}

static void
NAME_(se4b_in_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, 24, val);
}

static void
NAME_(se5a_in_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, 5, val);
}

static void
NAME_(se5b_in_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, 25, val);
}

static void
NAME_(se6a_in_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, 6, val);
}

static void
NAME_(se6b_in_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, 26, val);
}

static void
NAME_(se7a_in_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, 7, val);
}

static void
NAME_(se7b_in_set)(struct cpssp *cpssp, unsigned int val)
{
	NAME_(se_inN_set)(cpssp, 27, val);
}
#endif

static void
NAME_(refN_set)(struct cpssp *cpssp, int n, unsigned int val)
{
	assert(0 <= n && n < 2);

	cpssp->NAME.state_ref[n] = SIG_mV(val);
}

static void
NAME_(clk)(struct cpssp *cpssp)
{
	unsigned int n;

	if (cpssp->NAME.cal) {
		cpssp->NAME.cal = 0;
		cpssp->NAME.calf = 0;
		cpssp->NAME.coco[0] = 1;
		NAME_(irq_update)(cpssp);
	}
	for (n = 0; n < 2; n++) {
		int32_t valp;
		int32_t valm;
		int32_t res;
		int cmp;

		if (cpssp->NAME.state[n] == 0) continue;

		cpssp->NAME.state[n]++;

		if (cpssp->NAME.state[n] != 30) continue;

		switch (cpssp->NAME.adch[n]) {
		case 0b00000:
		case 0b00001:
		case 0b00010:
		case 0b00011:
			valp = cpssp->NAME.state_vin[cpssp->NAME.adch[n]];
			if (cpssp->NAME.diff[n]) {
				/* Differential */
				/* DAD0 ... DAD3 */
				valm = cpssp->NAME.state_vin[cpssp->NAME.adch[n] + 4];
			} else {
				/* Single-Ended */
				/* SE0 ... SE3 */
				valm = 0;
			}
			break;
		case 0b00100:
		case 0b00101:
		case 0b00110:
		case 0b00111:
			if (cpssp->NAME.diff[n]) {
				/* Differential */
				goto reserved;
			} else {
				/* Single-Ended */
				if (! cpssp->NAME.muxsel) {
					/* SE4a ... SE7a */
					valp = cpssp->NAME.state_vin[cpssp->NAME.adch[n] + 0];
					valm = 0;
				} else {
					/* SE4b ... SE7b */
					valp = cpssp->NAME.state_vin[cpssp->NAME.adch[n] + 20];
					valm = 0;
				}
			}
			break;
		case 0b01000:
		case 0b01001:
		case 0b01010:
		case 0b01011:
		case 0b01100:
		case 0b01101:
		case 0b01110:
		case 0b01111:
		case 0b10000:
		case 0b10001:
		case 0b10010:
		case 0b10011:
		case 0b10100:
		case 0b10101:
		case 0b10110:
		case 0b10111:
			if (cpssp->NAME.diff[n]) {
				/* Differential */
				goto reserved;
			} else {
				/* Single-Ended */
				/* SE8 ... SE23 */
				valp = cpssp->NAME.state_vin[cpssp->NAME.adch[n]];
				valm = 0;
			}
			break;
		case 0b11000:
		case 0b11001:
			goto reserved;
		case 0b11010:
			/* Temp Sensor */
			assert(0); /* FIXME */
			break;
		case 0b11011:
			/* Band Gap */
			assert(0); /* FIXME */
			break;
		case 0b11100:
			goto reserved;
		case 0b11101:
			/* Vrefh */
			valp = cpssp->NAME.state_ref[cpssp->NAME.refsel];
			if (cpssp->NAME.diff[n]) {
				/* Differential */
				valm = -cpssp->NAME.state_ref[cpssp->NAME.refsel];
			} else {
				/* Single-Ended */
				valm = 0;
			}
			break;
		case 0b11110:
			/* Vrefl */
			valp = 0; /* FIXME */
			if (cpssp->NAME.diff[n]) {
				/* Differential */
				goto reserved;
			} else {
				/* Single-Ended */
				valm = 0;
			}
			break;
		default:
		reserved:;
			valp = 0;
			valm = 0;
		}

		res = valp - valm;
		res *= 0x10000;
		res /= cpssp->NAME.state_ref[cpssp->NAME.refsel];

		switch (cpssp->NAME.mode) {
		case 0b00: /* 8-bit */
			res >>= 16 - 8;
			break;
		case 0b01: /* 12-bit */
			res >>= 16 - 12;
			break;
		case 0b10: /* 10-bit */
			res >>= 16 - 10;
			break;
		case 0b11: /* 16-bit */
			res >>= 16 - 16;
			break;
		default:
			assert(0); /* Cannot happen. */
		}

		if (cpssp->NAME.acfe) {
			cmp = res < cpssp->NAME.cv1;
			if (cpssp->NAME.acren) {
				if (cpssp->NAME.cv1 <= cpssp->NAME.cv2) {
					cmp |= cpssp->NAME.cv2 < res;
				} else {
					cmp &= cpssp->NAME.cv2 < res;
				}
			}
			cmp ^= cpssp->NAME.acfgt;
		} else {
			cmp = 1;
		}

		if (cmp) {
			cpssp->NAME.r[n] = res;

			cpssp->NAME.coco[n] = 1;
			NAME_(irq_update)(cpssp);
		}

		if (cpssp->NAME.adco) {
			cpssp->NAME.state[n] = 1;
		} else {
			cpssp->NAME.state[n] = 0;
		}
	}
}

static void
NAME_(clk0)(struct cpssp *cpssp)
{
#if CONFIG_DMA
	if (0) {
		NAME_(dma_set)(cpssp, 0);
	}
#endif
	NAME_(clk)(cpssp);
}

static void
NAME_(clk1)(struct cpssp *cpssp)
{
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	int n;

	for (n = 0; n < 2; n++) {
		cpssp->NAME.state[n] = 0;
	}

	/* ADC Status and Control Register 1 (ADC0_SC1A) */
	/* ADC Status and Control Register 1 (ADC0_SC1B) */
	for (n = 0; n < 2; n++) {
		cpssp->NAME.coco[n] = 0;
		cpssp->NAME.aien[n] = 0;
		cpssp->NAME.diff[n] = 0;
		cpssp->NAME.adch[n] = 0b11111;
	}

	/* ADC Configuration Register 1 (ADC0_CFG1) */
	cpssp->NAME.adlpc = 0;
	cpssp->NAME.adiv = 0b00;
	cpssp->NAME.adlsmp = 0;
	cpssp->NAME.mode = 0b00;
	cpssp->NAME.adiclk = 0b00;

	/* ADC Configuration Register 2 (ADC0_CFG2) */
	cpssp->NAME.muxsel = 0;
	cpssp->NAME.adacken = 0;
	cpssp->NAME.adhsc = 0;
	cpssp->NAME.adlsts = 0b00;

	/* ADC Data Result Register (ADC0_RA) */
	/* ADC Data Result Register (ADC0_RB) */
	for (n = 0; n < 2; n++) {
		cpssp->NAME.r[n] = 0x0000;
	}

	/* Compare Value Registers (ADC0_CV1) */
	cpssp->NAME.cv1 = 0x0000;

	/* Compare Value Registers (ADC0_CV2) */
	cpssp->NAME.cv2 = 0x0000;

	/* Status and Control Register 2 (ADC0_SC2) */
	cpssp->NAME.adtrg = 0;
	cpssp->NAME.acfe = 0;
	cpssp->NAME.acfgt = 0;
	cpssp->NAME.acren = 0;
	cpssp->NAME.dmaen = 0;
	cpssp->NAME.refsel = 0b00;

	/* Status and Control Register 3 (ADC0_SC3) */
	cpssp->NAME.cal = 0;
	cpssp->NAME.calf = 0;
	cpssp->NAME.adco = 0;
	cpssp->NAME.avge = 0;
	cpssp->NAME.avgs = 0;

	/* ADC Offset Correction Register (ADC0_OFS) */
	cpssp->NAME.ofs = 0x0000;

	/* ADC Plus-Side Gain Register (ADC0_PG) */
	cpssp->NAME.pg = 0x0000;

	/* ADC Minus-Side Gain Register (ADC0_MG) */
	cpssp->NAME.mg = 0x0000;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLPD) */
	cpssp->NAME.clpd = 0b000000;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLPS) */
	cpssp->NAME.clps = 0b000000;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP4) */
	cpssp->NAME.clp4 = 0b0000000000;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP3) */
	cpssp->NAME.clp3 = 0b000000000;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP2) */
	cpssp->NAME.clp2 = 0b00000000;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP1) */
	cpssp->NAME.clp1 = 0b0000000;

	/* ADC Plus-Side General Calibration Value Register (ADC0_CLP0) */
	cpssp->NAME.clp0 = 0b000000;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLMD) */
	cpssp->NAME.clmd = 0b000000;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLMS) */
	cpssp->NAME.clms = 0b000000;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM4) */
	cpssp->NAME.clm4 = 0b0000000000;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM3) */
	cpssp->NAME.clm3 = 0b000000000;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM2) */
	cpssp->NAME.clm2 = 0b00000000;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM1) */
	cpssp->NAME.clm1 = 0b0000000;

	/* ADC Minus-Side General Calibration Value Register (ADC0_CLM0) */
	cpssp->NAME.clm0 = 0b000000;

	NAME_(irq_update)(cpssp);
}

static void
NAME_(create)(struct cpssp *cpssp)
{
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG
