/*
 * $Id: arch_intel_pci_to_pci.c,v 1.1 2013-05-13 18:03:35 vrsieh Exp $
 *
 * Copyright (C) 2003-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
00:1e.0 PCI bridge: Intel Corporation 82801 PCI Bridge (rev a2) (prog-if 01 [Subtractive decode])
        Flags: bus master, fast devsel, latency 0
        Bus: primary=00, secondary=03, subordinate=03, sec-latency=32
        I/O behind bridge: 00002000-00002fff
        Memory behind bridge: fc500000-fc5fffff
        Prefetchable memory behind bridge: 00000000f0800000-00000000f08fffff
        Capabilities: [50] Subsystem: Fujitsu Siemens Computers Device 114d

00:1e.0 0604: 8086:244e (rev a2) (prog-if 01 [Subtractive decode])
        Flags: bus master, fast devsel, latency 0
        Bus: primary=00, secondary=03, subordinate=03, sec-latency=32
        I/O behind bridge: 00002000-00002fff
        Memory behind bridge: fc500000-fc5fffff
        Prefetchable memory behind bridge: 00000000f0800000-00000000f08fffff
        Capabilities: [50] Subsystem: 1734:114d
 */

#ifdef STATE

struct {
	int bus_master_enable;
	int mem_enable;
	int io_enable;

	uint8_t primary_bus;
	uint8_t secondary_bus;
	uint8_t subordinate_bus;

	uint16_t io_addr_limit;
	uint16_t io_addr_base;

	uint32_t mem_addr_limit;
	uint32_t mem_addr_base;
	uint64_t pref_mem_addr_limit;
	uint64_t pref_mem_addr_base;

	uint8_t interrupt_line;
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

static int
NAME_(0_in_c0r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	/*
	 * See ICH10, Page 355
	 */
	switch (addr & 0x0fc) {
	case 0x00: /* Vendor ID, Device ID */
		*valp = (0x8086 << 0)
			| (0x244e << 16);
		break;

	case 0x04: /* Command Register, Status Register */
		*valp = (1 << (4+16)) /* Capabilities List exists. */
			| (cpssp->NAME.bus_master_enable << 2)
			| (cpssp->NAME.mem_enable << 1)
			| (cpssp->NAME.io_enable << 0);
		break;

	case 0x08: /* Revision ID, Class Code */
		*valp = (0x06 << 24)	/* Bridge */
			| (0x04 << 16)	/* PCI-to-PCI */
			| (0x01 << 8)	/* Subtractive Decode */
			| (0xa2 << 0);	/* Revision ID */
		break;

	case 0x0c: /* Master Latency, Header Type */
		*valp = (0x00 << 24)	/* ? */
			| (0 << (7+16))	/* No multifunction device */
			| (0x01 << 16)	/* Standard Layout */
			| (0x00 << 8)	/* Latency 0 */
			| (0x00 << 0);	/* ? */
		break;

	case 0x18: /* Primary Bus, Secondary Bus,
			Subordinate Bus, Secondary Bus Latency */
		*valp = (0x20 << 24)	/* Secondary Bus Latency */
			| (cpssp->NAME.subordinate_bus << 16)
			| (cpssp->NAME.secondary_bus << 8)
			| (cpssp->NAME.primary_bus << 8);
		break;

	case 0x1c: /* I/O Base, I/O Limit, Secondary Status */
		*valp = (1 << (7+16))	/* fast back-to-back */
			| ((cpssp->NAME.io_addr_limit >> 12) << 12)
			| (0 << 8)	/* No 32 Bit I/O */
			| ((cpssp->NAME.io_addr_base >> 12) << 4)
			| (0 << 0);	/* No 32 Bit I/O */
		break;

	case 0x20: /* Memory Base, Memory Limit */
		*valp = ((cpssp->NAME.mem_addr_limit >> 20) << 20)
			| ((cpssp->NAME.mem_addr_base >> 20) << 4);
		break;

	case 0x24: /* Pref. Memory Base, Pref. Memory Limit */
		*valp = (((cpssp->NAME.pref_mem_addr_limit & 0xffffffff) >> 20) << 20)
			| (1 << 16)
			| (((cpssp->NAME.pref_mem_addr_base & 0xffffffff) >> 20) << 4)
			| (1 << 0);
		break;

	case 0x28: /* Pref. Memory Base (upper 32 bits) */
		*valp = cpssp->NAME.pref_mem_addr_base >> 32;
		break;

	case 0x2c: /* Pref. Memory Limit (upper 32 bits) */
		*valp = cpssp->NAME.pref_mem_addr_limit >> 32;
		break;

	case 0x34: /* Capabilities Pointer */
		*valp = (0x50 << 0);
		break;

	case 0x3c: /* Interrupt Pin, Interrupt Line, Bridge Control */
		*valp = (0x0000 << 16)	/* FIXME */
			| (0x00 << 8)	/* Interrupt Line */
			| (cpssp->NAME.interrupt_line << 0);
		break;

	case 0x50: /* Capability Register */
		*valp = (0x00 << 8)	/* No further capabilities. */
			| (0x0d << 0);	/* PCI bridge subsystem vendor Cap. */
		break;

	default:
		*valp = 0x00000000;
		break;
	}

#if DEBUG
	fprintf(stderr, "%s: 0x%08x 0x%x -> 0x%08x\n", __FUNCTION__,
			addr, bs, *valp);
#endif

	return 0;
}

static int
NAME_(0_in_c0w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

#if DEBUG
	fprintf(stderr, "%s: 0x%08x 0x%x <- 0x%08x\n", __FUNCTION__,
			addr, bs, val);
#endif

	switch (addr & 0xfc) {
	case 0x00: /* Vendor ID, Device ID */
	case 0x08: /* Revision ID, Class Code */
	case 0x0c: /* Master Latency, Header Type */
	case 0x34: /* Capabilities Pointer */
	case 0x50: /* Capability Register */
	default:
		/* Read-only */
		break;

	case 0x04: /* Command Register, Status Register */
		if ((bs >> 0) & 1) {
			cpssp->NAME.bus_master_enable = (val >> 2) & 1;
			cpssp->NAME.mem_enable = (val >> 1) & 1;
			cpssp->NAME.io_enable = (val >> 0) & 1;
		}
		break;

	case 0x18: /* Primary Bus, Secondary Bus,
			Subordinate Bus, Secondary Bus Latency */
		if ((bs >> 2) & 1) {
			cpssp->NAME.subordinate_bus = (val >> 16) & 0xff;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.secondary_bus = (val >> 8) & 0xff;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.primary_bus = (val >> 0) & 0xff;
		}
		break;

	case 0x1c: /* I/O Base, I/O Limit, Secondary Status */
		if ((bs >> 1) & 1) {
			cpssp->NAME.io_addr_limit = (((val >> 12) & 0xf) << 12)
					| 0xfff;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.io_addr_base = (((val >> 4) & 0xf) << 12)
					| 0x000;
		}
		break;

	case 0x20: /* Memory Base, Memory Limit */
		if ((bs >> 3) & 1) {
			cpssp->NAME.mem_addr_limit &= ~0xff000000;
			cpssp->NAME.mem_addr_limit |= ((val >> 24) & 0xff) << 24;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.mem_addr_limit &= ~0x00f00000;
			cpssp->NAME.mem_addr_limit |= ((val >> 16) & 0xf0) << 16;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.mem_addr_base &= ~0xff000000;
			cpssp->NAME.mem_addr_base |= ((val >>  8) & 0xff) << 24;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.mem_addr_base &= ~0x00f00000;
			cpssp->NAME.mem_addr_base |= ((val >>  0) & 0xf0) << 16;
		}
		break;

	case 0x24: /* Pref. Memory Base, Pref. Memory Limit */
		if ((bs >> 3) & 1) {
			cpssp->NAME.pref_mem_addr_limit &= ~(uint64_t) 0xff000000;
			cpssp->NAME.pref_mem_addr_limit |= ((val >> 24) & 0xff) << 24;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.pref_mem_addr_limit &= ~(uint64_t) 0x00f00000;
			cpssp->NAME.pref_mem_addr_limit |= ((val >> 16) & 0xf0) << 16;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.pref_mem_addr_base &= ~(uint64_t) 0xff000000;
			cpssp->NAME.pref_mem_addr_base |= ((val >> 24) & 0xff) << 24;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.pref_mem_addr_base &= ~(uint64_t) 0x00f00000;
			cpssp->NAME.pref_mem_addr_base |= ((val >> 16) & 0xf0) << 16;
		}
		break;

	case 0x28: /* Pref. Memory Base (upper 32 bits) */
		if ((bs >> 3) & 1) {
			cpssp->NAME.pref_mem_addr_base &= ~((uint64_t) 0xff << 56);
			cpssp->NAME.pref_mem_addr_base |= (uint64_t) ((val >> 24) & 0xff) << 56;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.pref_mem_addr_base &= ~((uint64_t) 0xff << 48);
			cpssp->NAME.pref_mem_addr_base |= (uint64_t) ((val >> 16) & 0xff) << 48;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.pref_mem_addr_base &= ~((uint64_t) 0xff << 40);
			cpssp->NAME.pref_mem_addr_base |= (uint64_t) ((val >> 8) & 0xff) << 40;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.pref_mem_addr_base &= ~((uint64_t) 0xff << 32);
			cpssp->NAME.pref_mem_addr_base |= (uint64_t) ((val >> 0) & 0xff) << 32;
		}
		break;

	case 0x2c: /* Pref. Memory Limit (upper 32 bits) */
		if ((bs >> 3) & 1) {
			cpssp->NAME.pref_mem_addr_limit &= ~((uint64_t) 0xff << 56);
			cpssp->NAME.pref_mem_addr_limit |= (uint64_t) ((val >> 24) & 0xff) << 56;
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.pref_mem_addr_limit &= ~((uint64_t) 0xff << 48);
			cpssp->NAME.pref_mem_addr_limit |= (uint64_t) ((val >> 16) & 0xff) << 48;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.pref_mem_addr_limit &= ~((uint64_t) 0xff << 40);
			cpssp->NAME.pref_mem_addr_limit |= (uint64_t) ((val >> 8) & 0xff) << 40;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.pref_mem_addr_limit &= ~((uint64_t) 0xff << 32);
			cpssp->NAME.pref_mem_addr_limit |= (uint64_t) ((val >> 0) & 0xff) << 32;
		}
		break;

	case 0x3c: /* Interrupt Pin, Interrupt Line, Bridge Control */
		if ((bs >> 0) & 1) {
			cpssp->NAME.interrupt_line = (val >> 0) & 0xff;
		}
		break;
	}

	return 0;
}

static int
NAME_(0_in_c1r)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t bus;
	int ret;

	bus = (addr >> 16) & 0xff;

	if (bus == cpssp->NAME.secondary_bus) {
		/* Relay as type 0 request. */
		uint8_t dev;

		dev = (addr >> 11) & 0x1f;
		if (dev < 21) {
			addr &= 0x7fc;
			assert(! (addr & 3));
			addr |= 1 << (11 + dev);
			assert(! (addr & 3));

			ret = NAME_(1_out_c0r)(cpssp, addr, bs, valp);
		} else {
			ret = -1;
		}

	} else if (cpssp->NAME.secondary_bus < bus
		&& bus <= cpssp->NAME.subordinate_bus) {
		/* Relay as type 1 request. */
		ret = NAME_(1_out_c1r)(cpssp, addr, bs, valp);

	} else {
		/* Not for our bus tree. */
		ret = -1;
	}

	return ret;
}

static int
NAME_(0_in_c1w)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t bus;

	bus = (addr >> 16) & 0xff;

	if (bus == cpssp->NAME.secondary_bus) {
		/* Relay as type 0 request. */
		uint8_t dev;

		dev = (addr >> 11) & 0x1f;
		if (dev < 21) {
			addr &= 0x7fc;
			assert(! (addr & 3));
			addr |= 1 << (11 + dev);
			assert(! (addr & 3));

			return NAME_(1_out_c0w)(cpssp, addr, bs, val);
		} else {
			return -1;
		}

	} else if (cpssp->NAME.secondary_bus < bus
		&& bus <= cpssp->NAME.subordinate_bus) {
		/* Relay as type 1 request. */
		return NAME_(1_out_c1w)(cpssp, addr, bs, val);

	} else {
		/* Not for our bus tree. */
		return -1;
	}
}

static int
NAME_(0_in_ior)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	if (0x03c0 <= addr && addr <= 0x03df) {
		/* Hack! FIXME! */
		goto forward;
	}

	if (! cpssp->NAME.io_enable
	 || addr < cpssp->NAME.io_addr_base
	 || cpssp->NAME.io_addr_limit < addr) {
		return 1;
	}
forward:;
	return NAME_(1_out_ior)(cpssp, addr, bs, valp);
}

static int
NAME_(0_in_iow)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	if (0x03c0 <= addr && addr <= 0x03df) {
		/* Hack! FIXME! */
		goto forward;
	}

	if (! cpssp->NAME.io_enable
	 || addr < cpssp->NAME.io_addr_base
	 || cpssp->NAME.io_addr_limit < addr) {
		return 1;
	}
forward:;
	return NAME_(1_out_iow)(cpssp, addr, bs, val);
}

static int
NAME_(0_in_ior_info)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp
)
{
	if (0x03c0 <= addr && addr <= 0x03df) {
		/* Hack! FIXME! */
		goto forward;
	}

	if (! cpssp->NAME.io_enable) {
		return 1;
	}
forward:;
	return NAME_(1_out_ior_info)(cpssp, addr, bs, cfp, csp);
}

static int
NAME_(0_in_iow_info)(
	struct cpssp *cpssp,
	uint32_t addr,
	unsigned int bs,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp
)
{
	if (0x03c0 <= addr && addr <= 0x03df) {
		/* Hack! FIXME! */
		goto forward;
	}

	if (! cpssp->NAME.io_enable) {
		return 1;
	}
forward:;
	return NAME_(1_out_iow_info)(cpssp, addr, bs, cfp, csp);
}

static int
NAME_(0_in_mr)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	if (0xa0000 <= addr && addr <= 0xbffff) {
		/* Hack! FIXME! */
		goto forward;
	}

	if (! cpssp->NAME.mem_enable
	 || addr < cpssp->NAME.mem_addr_base
	 || cpssp->NAME.mem_addr_limit <= addr) {
		return 1;
	}
forward:;
	return NAME_(1_out_mr)(cpssp, addr, bs, valp);
}

static int
NAME_(1_in_mr)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->NAME.bus_master_enable) {
		fprintf(stderr, "NOTE: %s: bus master disabled!\n", __FUNCTION__);
		return 1;
	}

	return NAME_(0_out_mr)(cpssp, addr, bs, valp);
}

static int
NAME_(0_in_mw)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	if (0xa0000 <= addr && addr <= 0xbffff) {
		/* Hack! FIXME! */
		goto forward;
	}

	if (! cpssp->NAME.mem_enable
	 || addr < cpssp->NAME.mem_addr_base
	 || cpssp->NAME.mem_addr_limit <= addr) {
		return 1;
	}
forward:;
	return NAME_(1_out_mw)(cpssp, addr, bs, val);
}

static int
NAME_(1_in_mw)(void *_cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	struct cpssp *cpssp = _cpssp;

	if (! cpssp->NAME.bus_master_enable) {
		fprintf(stderr, "NOTE: %s: bus master disabled!\n", __FUNCTION__);
		return 1;
	}

	return NAME_(0_out_mw)(cpssp, addr, bs, val);
}

static int
NAME_(0_in_map_r)(
	struct cpssp *cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t *),
	void **csp,
	char **haddr_p)
{
	if (! cpssp->NAME.mem_enable
	 || addr < cpssp->NAME.mem_addr_base
	 || cpssp->NAME.mem_addr_limit <= addr) {
		return 1;
	}
	
	return NAME_(1_out_map_r)(cpssp, addr, cfp, csp, haddr_p);
}

static int
NAME_(0_in_map_w)(
	struct cpssp *cpssp,
	uint32_t addr,
	int (**cfp)(void *, uint32_t, unsigned int, uint32_t),
	void **csp,
	char **haddr_p
)
{
	if (! cpssp->NAME.mem_enable
	 || addr < cpssp->NAME.mem_addr_base
	 || cpssp->NAME.mem_addr_limit <= addr) {
		return 1;
	}
	
	return NAME_(1_out_map_w)(cpssp, addr, cfp, csp, haddr_p);
}

#if 0
static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.bus_master_enable = 0;
	cpssp->NAME.mem_enable = 0;
	cpssp->NAME.io_enable = 0;
}
#endif

static void
NAME_(create)(struct cpssp *cpssp)
{
	memset(&cpssp->NAME, 0, sizeof(cpssp->NAME));
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */
