/*
 * $Id: arch_intel_2e20.c,v 1.2 2013-05-14 07:11:13 vrsieh Exp $
 *
 * Copyright (C) 2003-2013 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * This is an implementation of the Intel Device 0x2e20
 * (Part of Intel 82x45 host controller).
 */

#define DEBUG	0

#ifdef STATE

struct {
	/*
	 * Config
	 */

	/*
	 * State
	 */
	/* PCI Command Register */
	/* 5.1.3 */
	uint8_t serre;
	uint8_t perre;

	/* Subsystem Vendor Identification Register */
	/* 5.1.9 */
	uint16_t svid;

	/* Subsystem Identification Register */
	/* 5.1.10 */
	uint16_t sid;

	/* PCI Express Egress Port Base Address Register */
	/* 5.1.12 */
	uint8_t pxpepbaren;
	uint64_t pxpepbar;

	/* Memory Mapped Register Range Base Address Register */
	/* 5.1.13 */
	uint8_t mchbaren;
	uint64_t mchbar;

	/* PCI Express Register Range Base Address Register */
	/* 5.1.16 */
	uint8_t pciexbaren;
	uint8_t length;
	uint64_t pciexmsk;
	uint64_t pciexbar;

	/* Root Complex Register Range Base Address Register */
	/* 5.1.17 */
	uint8_t dmibaren;
	uint64_t dmibar;

	/* Programable Attribute Map Register */
	/* 5.1.18 - 5.1.24 */
	uint8_t pam0;
	uint8_t pam1[12];

	/* Remap Base Address Register */
	/* 5.1.26 */
	uint64_t remapbase;

	/* Remap Limit Address Register */
	/* 5.1.27 */
	uint64_t remaplmt;

	/* System Management RAM Control Register */
	/* 5.1.28 */
	/* FIXME */
	uint8_t g_smrame;
	uint8_t d_lck;
	uint8_t d_cls;
	uint8_t d_open;

	/* Top of Low Usable DRAM */
	/* 5.1.35 */
	uint32_t tolud;

	/* Scratchpad Data */
	/* 5.1.39 */
	uint32_t skpd;

	/* MCH Registers */
	uint16_t mch_c0drb0;
	uint16_t mch_c0drb1;
	uint16_t mch_c0drb2;
	uint16_t mch_c0drb3;
	uint16_t mch_c1drb0;
	uint16_t mch_c1drb1;
	uint16_t mch_c1drb2;
	uint16_t mch_c1drb3;

	int selected_smm;
	uint64_t selected_addr;
	uint32_t selected_type;

	uint32_t confadd; /* Currently addressed device/register */
} NAME;

#endif /* STATE */

#ifdef BEHAVIOR

/* ----------------------------------------------------------------- */
/* Built-In Functions                                                */
/* ----------------------------------------------------------------- */

enum NAME_(orig) {
	ORIG_HOST_BUS,
	ORIG_PCI_BUS,
};

static int
NAME_(confadd_read)(
	struct cpssp *cpssp,
	uint64_t addr,
	unsigned int *bsp,
	uint64_t *datap
)
{
	if (addr == 0x0cf8
	 && *bsp == 0xf) {
		*datap = cpssp->NAME.confadd;
		*bsp &= ~0xf;
		return 0;
	}
	return 1;
}

static int
NAME_(confadd_write)(
	struct cpssp *cpssp,
	uint64_t addr,
	unsigned int *bsp,
	uint64_t data
)
{
	if (addr == 0x0cf8
	 && *bsp == 0xf) {
		uint32_t odata;

		data &= 0xffffffff;
		data &= ~(1 << 30); /* Clear reserved bits. */
		data &= ~(1 << 29);
		data &= ~(1 << 28);
		data &= ~(1 << 27);
		data &= ~(1 << 26);
		data &= ~(1 << 25);
		data &= ~(1 << 24);
		data &= ~(1 <<  1);
		data &= ~(1 <<  0);

		odata = cpssp->NAME.confadd;
		cpssp->NAME.confadd = data;

		if (((odata >> 31) & 1) != ((data >> 31) & 1)) {
			NAME_(host_out_ior_info_flush)(cpssp, 0x0cf8, 0);
			NAME_(host_out_iow_info_flush)(cpssp, 0x0cf8, 0);
		}
		*bsp &= ~0xf;
		return 0;
	}
	return 1;
}

static void
NAME_(confdata1)(
	struct cpssp *cpssp,
	unsigned int *typep,
	uint64_t *addrp,
	unsigned int *bsp,
	uint64_t *valp
)
{
	if ((*typep == SIG_HOST775_BUS_IOR
	  || *typep == SIG_HOST775_BUS_IOW)
	 && *addrp == 0x0cf8) {
		/*
		 * CONFADD *or* CONFDATA Register
		 *
		 * Intel ISA has no inq/outq instruction. So we access
		 * only one of the CONFADD/CONFDATA registers here.
		 * Access to the CONFADD register will stay locally.
		 * Only access to the CONFDATA register will result
		 * in a config space read/write to the outer world.
		 *
		 * So we forward a type/addr call to the outer world
		 * as if we will access the CONFDATA register in *both*
		 * cases!
		 */
		if (((cpssp->NAME.confadd >> 16) & 0xff) == 0x00) {
			/* Config Space Type 0 */
			uint64_t dev;
			uint32_t reg;

			dev = (cpssp->NAME.confadd >> 11) & 0x1f;
			reg = (cpssp->NAME.confadd >>  0) & 0x7f8;

			if (dev < 64 - 11) {
				dev = (uint64_t) 1 << (11 + dev);
			} else {
				dev = 0;
			}

			*addrp = dev | reg;
			*typep = (*typep == SIG_HOST775_BUS_IOR)
				? SIG_DMI_BUS_C0R : SIG_DMI_BUS_C0W;

		} else {
			/* Config Space Type 1 */
			*addrp = cpssp->NAME.confadd & ~7;
			*typep = (*typep == SIG_HOST775_BUS_IOR)
				? SIG_DMI_BUS_C1R : SIG_DMI_BUS_C1W;
		}

		if (! ((cpssp->NAME.confadd >> 2) & 1)) {
			if (valp) {
				*valp >>= 32;
			}
			*bsp >>= 4;
		}
	}
}

static void
NAME_(confdata2)(
	struct cpssp *cpssp,
	unsigned int type,
	unsigned int bs,
	uint64_t *valp
)
{
	if (type == SIG_DMI_BUS_C0R
	 || type == SIG_DMI_BUS_C1R) {
		if (! ((cpssp->NAME.confadd >> 2) & 1)) {
			if (valp) {
				*valp <<= 32;
			}
		}
	}
}

static int
NAME_(pxpep_mr)(struct cpssp *cpssp, uint64_t addr, uint8_t bs, uint64_t *valp)
{
	if (cpssp->NAME.pxpepbaren
	 && cpssp->NAME.pxpepbar == (addr & ~0xfff)) {
		fprintf(stderr, "WARNING: %s: reading from unimplemented register 0x%" PRIx64 "/0x%x.\n",
				__FUNCTION__, addr, bs);
		*valp = 0;
		return 0;
	} else {
		return 1;
	}
}

static int
NAME_(pxpep_mw)(struct cpssp *cpssp, uint64_t addr, uint8_t bs, uint64_t val)
{
	if (cpssp->NAME.pxpepbaren
	 && cpssp->NAME.pxpepbar == (addr & ~0xfff)) {
		fprintf(stderr, "WARNING: %s: writing to unimplemented register 0x%" PRIx64 "/0x%x.\n",
				__FUNCTION__, addr, bs);
		return 0;
	} else {
		return 1;
	}
}

static int
NAME_(mch_mr)(struct cpssp *cpssp, uint64_t addr, uint8_t bs, uint64_t *valp)
{
	if (! cpssp->NAME.mchbaren
	 || cpssp->NAME.mchbar != (addr & ~0x3fff)) {
		return 1;
	}

	addr &= 0x3fff;
	*valp = 0;
	switch (addr) {
	case 0x200:
		*valp |= (uint64_t) cpssp->NAME.mch_c0drb0 << 0;
		*valp |= (uint64_t) cpssp->NAME.mch_c0drb1 << 16;
		*valp |= (uint64_t) cpssp->NAME.mch_c0drb2 << 32;
		*valp |= (uint64_t) cpssp->NAME.mch_c0drb3 << 48;
		break;
	case 0x600:
		*valp |= (uint64_t) cpssp->NAME.mch_c1drb0 << 0; /* FIXME */
		*valp |= (uint64_t) cpssp->NAME.mch_c1drb1 << 16;
		*valp |= (uint64_t) cpssp->NAME.mch_c1drb2 << 32;
		*valp |= (uint64_t) cpssp->NAME.mch_c1drb3 << 48;
		break;
	default:
		fprintf(stderr, "WARNING: %s: reading from unimplemented register 0x%" PRIx64 ".\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG) fprintf(stderr, "%s: addr=0x%" PRIx64 ", bs=0x%x, val=0x%" PRIx64 "\n",
			__FUNCTION__, addr, bs, *valp);

	return 0;
}

static int
NAME_(mch_mw)(struct cpssp *cpssp, uint64_t addr, uint8_t bs, uint64_t val)
{
	if (! cpssp->NAME.mchbaren
	 || cpssp->NAME.mchbar != (addr & ~0x3fff)) {
		return 1;
	}

	addr &= 0x3fff;
	switch (addr) {
	case 0x200:
		/* Channel 0 DRAM Rank Boundary Address 0 Register */
		/* 5.2.2 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.mch_c0drb0 &= ~(0xff << 0);
			cpssp->NAME.mch_c0drb0 |= (val >> 0) & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.mch_c0drb0 &= ~(0x03 << 8);
			cpssp->NAME.mch_c0drb0 |= (val >> 0) & (0x03 << 8);
			/* Bit 10-15: Reserved */
		}
		/* Channel 0 DRAM Rank Boundary Address 1 Register */
		/* 5.2.3 */
		if ((bs >> 2) & 1) {
			cpssp->NAME.mch_c0drb1 &= ~(0xff << 0);
			cpssp->NAME.mch_c0drb1 |= (val >> 16) & (0xff << 0);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.mch_c0drb1 &= ~(0x03 << 8);
			cpssp->NAME.mch_c0drb1 |= (val >> 16) & (0x03 << 8);
			/* Bit 10-15: Reserved */
		}
		/* Channel 0 DRAM Rank Boundary Address 2 Register */
		/* 5.2.4 */
		if ((bs >> 4) & 1) {
			cpssp->NAME.mch_c0drb2 &= ~(0xff << 0);
			cpssp->NAME.mch_c0drb2 |= (val >> 32) & (0xff << 0);
		}
		if ((bs >> 5) & 1) {
			cpssp->NAME.mch_c0drb2 &= ~(0x03 << 8);
			cpssp->NAME.mch_c0drb2 |= (val >> 32) & (0x03 << 8);
			/* Bit 10-15: Reserved */
		}
		/* Channel 0 DRAM Rank Boundary Address 3 Register */
		/* 5.2.5 */
		if ((bs >> 6) & 1) {
			cpssp->NAME.mch_c0drb3 &= ~(0xff << 0);
			cpssp->NAME.mch_c0drb3 |= (val >> 48) & (0xff << 0);
		}
		if ((bs >> 7) & 1) {
			cpssp->NAME.mch_c0drb3 &= ~(0x03 << 8);
			cpssp->NAME.mch_c0drb3 |= (val >> 48) & (0x03 << 8);
			/* Bit 10-15: Reserved */
		}
		break;
	case 0x600:
		/* Channel 1 DRAM Rank Boundary Address 0 Register */
		/* FIXME */
		if ((bs >> 0) & 1) {
			cpssp->NAME.mch_c1drb0 &= ~(0xff << 0);
			cpssp->NAME.mch_c1drb0 |= (val >> 0) & (0xff << 0);
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.mch_c1drb0 &= ~(0x03 << 8);
			cpssp->NAME.mch_c1drb0 |= (val >> 0) & (0x03 << 8);
			/* Bit 10-15: Reserved */
		}
		/* Channel 1 DRAM Rank Boundary Address 1 Register */
		/* 5.2.16 */
		if ((bs >> 2) & 1) {
			cpssp->NAME.mch_c1drb1 &= ~(0xff << 0);
			cpssp->NAME.mch_c1drb1 |= (val >> 16) & (0xff << 0);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.mch_c1drb1 &= ~(0x03 << 8);
			cpssp->NAME.mch_c1drb1 |= (val >> 16) & (0x03 << 8);
			/* Bit 10-15: Reserved */
		}
		/* Channel 1 DRAM Rank Boundary Address 2 Register */
		/* 5.2.17 */
		if ((bs >> 4) & 1) {
			cpssp->NAME.mch_c1drb2 &= ~(0xff << 0);
			cpssp->NAME.mch_c1drb2 |= (val >> 32) & (0xff << 0);
		}
		if ((bs >> 5) & 1) {
			cpssp->NAME.mch_c1drb2 &= ~(0x03 << 8);
			cpssp->NAME.mch_c1drb2 |= (val >> 32) & (0x03 << 8);
			/* Bit 10-15: Reserved */
		}
		/* Channel 1 DRAM Rank Boundary Address 3 Register */
		/* 5.2.18 */
		if ((bs >> 6) & 1) {
			cpssp->NAME.mch_c1drb3 &= ~(0xff << 0);
			cpssp->NAME.mch_c1drb3 |= (val >> 48) & (0xff << 0);
		}
		if ((bs >> 7) & 1) {
			cpssp->NAME.mch_c1drb3 &= ~(0x03 << 8);
			cpssp->NAME.mch_c1drb3 |= (val >> 48) & (0x03 << 8);
			/* Bit 10-15: Reserved */
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: writing to unimplemented register 0x%" PRIx64 ".\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG) fprintf(stderr, "%s: addr=0x%" PRIx64 ", bs=0x%x, val=0x%" PRIx64 "\n",
			__FUNCTION__, addr, bs, val);

	return 0;
}

static int
NAME_(pciex_mr)(struct cpssp *cpssp, uint64_t addr, uint8_t bs, uint64_t *valp)
{
	if (cpssp->NAME.pciexbaren
	 && cpssp->NAME.pciexbar == (addr & cpssp->NAME.pciexmsk)) {
		uint8_t bus;
		uint8_t dev;
		uint8_t func;
		uint16_t reg;
		int ret;

		bus = (addr >> 20) & 0xff;
		dev = (addr >> 15) & 0x1f;
		func = (addr >> 12) & 0x7;
		reg = (addr >> 0) & 0xfff;

		if (bus == 0x00) {
			addr = (uint64_t) 1 << (11 + dev);
			addr |= (uint64_t) func << 8;
			addr |= (uint64_t) reg << 0;
			ret = NAME_(dmi_out_main_c0r)(cpssp, addr, bs, valp);
		} else {
			addr = (uint64_t) bus << 16;
			addr |= (uint64_t) dev << 11;
			addr |= (uint64_t) func << 8;
			addr |= (uint64_t) reg << 0;
			ret = NAME_(dmi_out_c1r)(cpssp, addr, bs, valp);
		}

		if (DEBUG) fprintf(stderr, "%s: addr=0x%" PRIx64 ", bs=0x%x, val=0x%" PRIx64 "\n",
				__FUNCTION__, addr, bs, *valp);

		return ret;

	} else {
		return 1;
	}
}

static int
NAME_(pciex_mw)(struct cpssp *cpssp, uint64_t addr, uint8_t bs, uint64_t val)
{
	if (cpssp->NAME.pciexbaren
	 && cpssp->NAME.pciexbar == (addr & cpssp->NAME.pciexmsk)) {
		uint8_t bus;
		uint8_t dev;
		uint8_t func;
		uint16_t reg;
		int ret;

		bus = (addr >> 20) & 0xff;
		dev = (addr >> 15) & 0x1f;
		func = (addr >> 12) & 0x7;
		reg = (addr >> 0) & 0xfff;

		if (bus == 0x00) {
			addr = (uint64_t) 1 << (11 + dev);
			addr |= (uint64_t) func << 8;
			addr |= (uint64_t) reg << 0;
			ret = NAME_(dmi_out_main_c0w)(cpssp, addr, bs, val);
		} else {
			addr = (uint64_t) bus << 16;
			addr |= (uint64_t) dev << 11;
			addr |= (uint64_t) func << 8;
			addr |= (uint64_t) reg << 0;
			ret = NAME_(dmi_out_c1w)(cpssp, addr, bs, val);
		}

		if (DEBUG) fprintf(stderr, "%s: addr=0x%" PRIx64 ", bs=0x%x, val=0x%" PRIx64 "\n",
				__FUNCTION__, addr, bs, val);

		return ret;
	} else {
		return 1;
	}
}

static int
NAME_(dmi_mr)(struct cpssp *cpssp, uint64_t addr, uint8_t bs, uint64_t *valp)
{
	if (cpssp->NAME.dmibaren
	 && cpssp->NAME.dmibar == (addr & ~0xfff)) {
		fprintf(stderr, "WARNING: %s reading from unimplemented register 0x%" PRIx64 "/0x%x.\n",
				__FUNCTION__, addr, bs);
		return 0;
	} else {
		return 1;
	}
}

static int
NAME_(dmi_mw)(struct cpssp *cpssp, uint64_t addr, uint8_t bs, uint64_t val)
{
	if (cpssp->NAME.dmibaren
	 && cpssp->NAME.dmibar == (addr & ~0xfff)) {
		fprintf(stderr, "WARNING: %s writing to unimplemented register 0x%" PRIx64 "/0x%x.\n",
				__FUNCTION__, addr, bs);
		return 0;
	} else {
		return 1;
	}
}

static void
NAME_(host_pci_bridge_c0r)(
	struct cpssp *cpssp,
	uint64_t addr,
	unsigned int bs,
	uint64_t *valp
)
{
	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%02lx, bs=0x%x\n",
				__FUNCTION__, addr, bs);
	}

	*valp = 0;
	switch (addr) {
	case 0x00:
		/* Vendor ID Register */
		/* 5.1.1 */
		*valp |= 0x8086 << 0;

		/* Device ID Register */
		/* 5.1.2 */
		*valp |= 0x2e20 << 16;

		/* PCI Command Register */
		/* 5.1.3 */
		*valp |= (uint64_t) 0b0 << (0 + 32); /* Read-only */
		*valp |= (uint64_t) 0b1 << (1 + 32); /* Read-only */
		*valp |= (uint64_t) 0b1 << (2 + 32); /* Read-only */
		*valp |= (uint64_t) 0b0 << (3 + 32); /* Read-only */
		*valp |= (uint64_t) 0b0 << (4 + 32); /* Read-only */
		*valp |= (uint64_t) 0b0 << (5 + 32); /* Read-only */
		*valp |= (uint64_t) cpssp->NAME.perre << (6 + 32);
		*valp |= (uint64_t) 0b0 << (7 + 32); /* Read-only */
		*valp |= (uint64_t) cpssp->NAME.serre << (8 + 32);
		*valp |= (uint64_t) 0b0 << (9 + 32); /* Read-only */
		*valp |= (uint64_t) 0b000000 << (10 + 32);

		/* Status Register */
		/* 5.1.4 */
		*valp |= (uint64_t) 0b0000 << (0 + 48); /* Reserved */
		*valp |= (uint64_t) 0b1 << (4 + 48); /* Read-only */
		*valp |= (uint64_t) 0b0 << (5 + 48); /* Read-only */
		*valp |= (uint64_t) 0b0 << (6 + 48); /* Reserved */
		*valp |= (uint64_t) 0b0 << (7 + 48); /* Read-only */
		*valp |= (uint64_t) 0b0 << (8 + 48); /* FIXME */
		*valp |= (uint64_t) 0b00 << (9 + 48); /* Read-only */
		*valp |= (uint64_t) 0b0 << (11 + 48); /* Read-only */
		*valp |= (uint64_t) 0b0 << (12 + 48); /* FIXME */
		*valp |= (uint64_t) 0b0 << (13 + 48); /* FIXME */
		*valp |= (uint64_t) 0b0 << (14 + 48); /* FIXME */
		*valp |= (uint64_t) 0b0 << (15 + 48); /* FIXME */
		break;

	case 0x08:
		/* Revision Identification Register */
		/* 5.1.5 */
		*valp |= 0x03 << 0;

		/* Class Code Register */
		/* 5.1.6 */
		*valp |= 0x00 << (0 + 8); /* Programming Interface */
		*valp |= 0x00 << (8 + 8); /* Host Bridge */
		*valp |= 0x06 << (16 + 8); /* Bridge Device */

		/* Cache Line Size Register */
		/* DRAM Controller will never become bus master. */
		*valp |= (uint64_t) 0x00 << 32;

		/* Master Latency Timer Register */
		/* 5.1.7 */
		*valp |= (uint64_t) 0x00 << (8 + 32);

		/* Header Type Register */
		/* 5.1.8 */
		*valp |= (uint64_t) 0x00 << (16 + 32);

		/* BIST Register */
		*valp |= (uint64_t) 0x00 << (24 + 32);
		break;

	case 0x10:
	case 0x18:
	case 0x20:
		*valp = 0x0000000000000000; /* Reserved */
		break;

	case 0x28:
		*valp |= 0x00000000 << 0; /* Reserved */

		/* Subsystem Vendor Identification Register */
		/* 5.1.9 */
		*valp |= (uint64_t) cpssp->NAME.svid << 32;

		/* Subsystem Identification Register */
		/* 5.1.10 */
		*valp |= (uint64_t) cpssp->NAME.sid << 40;
		break;

	case 0x30:
		*valp |= 0x00000000 << 0; /* Reserved */

		/* Capabilities Pointer */
		/* 5.1.11 */
		*valp |= (uint64_t) 0xe0 << 32;

		*valp |= (uint64_t) 0x000000 << (8 + 32); /* Reserved */
		break;
		
	case 0x38:
		*valp |= 0x0000000000000000 << 0; /* Reserved */
		break;

	case 0x40:
		/* PCI Express Egress Port Base Address Register */
		/* 5.1.12 */
		*valp |= cpssp->NAME.pxpepbaren << 0;
		*valp |= cpssp->NAME.pxpepbar << (1-1);
		break;

	case 0x48:
		/* Memory Mapped Register Range Base Address Register */
		/* 5.1.13 */
		*valp |= cpssp->NAME.mchbaren << 0;
		*valp |= cpssp->NAME.mchbar << (1-1);
		break;

	case 0x50:
	case 0x58:
		*valp |= 0x0000000000000000; /* Reserved */
		break;

	case 0x60:
		/* PCI Express Register Range Base Address Register */
		/* 5.1.16 */
		*valp |= cpssp->NAME.pciexbaren << 0;
		*valp |= cpssp->NAME.length << 1;
		*valp |= 0 << 3; /* Reserved */
		*valp |= cpssp->NAME.pciexbar << (26-26);
		break;

	case 0x68:
		/* Root Complex Register Range Base Address Register */
		/* 5.1.17 */
		*valp |= cpssp->NAME.dmibaren << 0;
		*valp |= cpssp->NAME.dmibar << (1-1);
		break;

	case 0x70:
	case 0x78:
	case 0x80:
	case 0x88:
		*valp |= 0x0000000000000000 << 0; /* Reserved */
		break;

	case 0x90:
		/* Programable Attribute Map Register 0 */
		/* 5.1.18 */
		*valp |= 0b0000 << 0; /* Reserved */
		*valp |= cpssp->NAME.pam0 << 4;
		*valp |= 0b00 << 6; /* Reserved */

		/* Programable Attribute Map Register 1 */
		/* 5.1.19 */
		*valp |= cpssp->NAME.pam1[0] << 8;
		*valp |= 0b00 << 10; /* Reserved */
		*valp |= cpssp->NAME.pam1[1] << 12;
		*valp |= 0b00 << 14; /* Reserved */

		/* Programable Attribute Map Register 2 */
		/* 5.1.20 */
		*valp |= cpssp->NAME.pam1[2] << 16;
		*valp |= 0b00 << 18; /* Reserved */
		*valp |= cpssp->NAME.pam1[3] << 20;
		*valp |= 0b00 << 22; /* Reserved */

		/* Programable Attribute Map Register 3 */
		/* 5.1.21 */
		*valp |= cpssp->NAME.pam1[4] << 24;
		*valp |= 0b00 << 26; /* Reserved */
		*valp |= cpssp->NAME.pam1[5] << 28;
		*valp |= 0b00 << 30; /* Reserved */

		/* Programable Attribute Map Register 4 */
		/* 5.1.22 */
		*valp |= (uint64_t) cpssp->NAME.pam1[6] << 32;
		*valp |= (uint64_t) 0b00 << 34; /* Reserved */
		*valp |= (uint64_t) cpssp->NAME.pam1[7] << 36;
		*valp |= (uint64_t) 0b00 << 38; /* Reserved */

		/* Programable Attribute Map Register 5 */
		/* 5.1.23 */
		*valp |= (uint64_t) cpssp->NAME.pam1[8] << 40;
		*valp |= (uint64_t) 0b00 << 42; /* Reserved */
		*valp |= (uint64_t) cpssp->NAME.pam1[9] << 44;
		*valp |= (uint64_t) 0b00 << 46; /* Reserved */

		/* Programable Attribute Map Register 6 */
		/* 5.1.24 */
		*valp |= (uint64_t) cpssp->NAME.pam1[10] << 48;
		*valp |= (uint64_t) 0b00 << 50; /* Reserved */
		*valp |= (uint64_t) cpssp->NAME.pam1[11] << 52;
		*valp |= (uint64_t) 0b00 << 54; /* Reserved */

		/* Legacy Access Control Register */
		/* 5.1.25 */
		/* FIXME */
		break;

	case 0x98:
		/* Remap Base Address Register */
		/* 5.1.26 */
		*valp |= (cpssp->NAME.remapbase >> 26) << 0;
		*valp |= 0b000000 << 10;

		/* Remap Limit Address Register */
		/* 5.1.27 */
		*valp |= (cpssp->NAME.remaplmt >> 26) << (0 + 16);
		*valp |= 0b000000 << (10 + 16);

		/* Reserved */
		*valp |= (uint64_t) 0x00 << 32;

		/* System Management RAM Control Register */
		/* 5.1.28 */
		/* FIXME */
		*valp |= (uint64_t) 0b010 << (0 + 40);
		*valp |= (uint64_t) cpssp->NAME.g_smrame << (3 + 40);
		*valp |= (uint64_t) cpssp->NAME.d_lck << (4 + 40);
		*valp |= (uint64_t) cpssp->NAME.d_cls << (5 + 40);
		*valp |= (uint64_t) cpssp->NAME.d_open << (6 + 40);
		*valp |= (uint64_t) 0b0 << (7 + 40); /* Reserved */

		/* Extended System Management RAM Control Register */
		/* 5.1.29 */
		/* FIXME */
		*valp |= (uint64_t) 0x00 << 48;

		*valp |= (uint64_t) 0x00 << 56; /* Reserved */
		break;

	case 0xb0:
		/* Top of Low Usable DRAM */
		/* 5.1.35 */
		*valp |= 0b0000 << 0; /* Reserved */
		*valp |= (cpssp->NAME.tolud >> 20) << 4;
		*valp |= 0x000000000000 << 16; /* Reserved */
		break;

	/* FIXME */

	case 0xd8:
		/* Reserved */
		*valp |= 0x00000000 << 0;

		/* Scratchpad Data */
		/* 5.1.39 */
		*valp |= (uint64_t) cpssp->NAME.skpd << 32;
		break;

	case 0xe0:
		/* Capability Identifier */
		/* 5.1.40 */
		*valp |= 0x09 << 0; /* Vendor Dependent Capability Pointer */
		*valp |= 0x00 << 8; /* Next Capability Pointer (NULL) */
		*valp |= 12 << 16; /* Indicate Structure Length */
		*valp |= 0x01 << 24; /* Revision */

		*valp |= (uint64_t) 0x00000000 << 32;
		break;

	case 0xe8:
		/* cont. */
		*valp |= 0x00000000 << 0;

		*valp |= (uint64_t) 0x00000000 << 32; /* Reserved */
		break;

	case 0xf0:
	case 0xf8:
		*valp |= 0x0000000000000000 << 0; /* Reserved */
		break;

	default:
		fprintf(stderr, "WARNING: %s: addr=0x%" PRIx64 "\n",
				__FUNCTION__, addr);
		*valp |= 0x0000000000000000 << 0; /* FIXME */
		break;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: -> val=0x%016" PRIx64 "\n",
				__FUNCTION__, *valp);
	}
}

static void
NAME_(host_pci_bridge_c0w)(
	struct cpssp *cpssp,
	uint64_t addr,
	unsigned int bs,
	uint64_t val
)
{
	if (DEBUG) {
		fprintf(stderr, "%s: addr=0x%02lx, bs=0x%x, val=0x%016lx\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x00:
		/* Vendor ID Register */
		/* 5.1.1 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Read-only */
		}

		/* Device ID Register */
		/* 5.1.2 */
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}

		/* PCI Command Register */
		/* 5.1.3 */
		if ((bs >> 4) & 1) {
			/* Bits 0-5: Read-only */
			cpssp->NAME.perre = (val >> (6 + 32)) & 0b1;
			/* Bit 7: Read-only */
		}
		if ((bs >> 5) & 1) {
			cpssp->NAME.serre = (val >> (8 + 32)) & 0b1;
			/* Bit 9: Read-only */
			/* Bit 10-15: Reserved */
		}

		/* Status Register */
		/* 5.1.4 */
		if ((bs >> 6) & 1) {
			/* FIXME */
		}
		if ((bs >> 7) & 1) {
			/* FIXME */
		}
		break;

	case 0x08:
		/* Revision ID Register */
		/* 5.1.5 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}

		/* Class Code Register */
		/* 5.1.6 */
		if ((bs >> 1) & 1) {
			/* Read-only */
		}
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}

		/* Cache Line Size Register */
		/* DRAM Controller will never become bus master. */
		if ((bs >> 4) & 1) {
			/* Reserved */
		}

		/* Latency Timer Register */
		/* 5.1.7 */
		/* DRAM Controller will never become bus master. */
		if ((bs >> 5) & 1) {
			/* Reserved */
		}

		/* Header Type Register */
		/* 5.1.8 */
		if ((bs >> 6) & 1) {
			/* Read-only */
		}

		/* BIST Register */
		if ((bs >> 7) & 1) {
			/* Read-only */
		}
		break;

	case 0x10:
	case 0x18:
	case 0x20:
		/* Reserved */
		break;

	case 0x28:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}

		/* Subsystem Vendor Identification Register */
		/* 5.1.9 */
		if (cpssp->NAME.svid == 0x0000) {
			if ((bs >> 4) & 1) {
				cpssp->NAME.svid &= ~(0xff << 0);
				cpssp->NAME.svid |= (val >> 32) & (0xff << 0);
			}
			if ((bs >> 5) & 1) {
				cpssp->NAME.svid &= ~(0xff << 8);
				cpssp->NAME.svid |= (val >> 32) & (0xff << 8);
			}
		}

		/* Subsystem Identification Register */
		/* 5.1.10 */
		if (cpssp->NAME.sid == 0x0000) {
			if ((bs >> 4) & 1) {
				cpssp->NAME.sid &= ~(0xff << 0);
				cpssp->NAME.sid |= (val >> 48) & (0xff << 0);
			}
			if ((bs >> 5) & 1) {
				cpssp->NAME.sid &= ~(0xff << 8);
				cpssp->NAME.sid |= (val >> 48) & (0xff << 8);
			}
		}
		break;

	case 0x30:
		if ((bs >> 0) & 1) {
			/* Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}

		/* Capabilities Pointer Register */
		/* 5.1.11 */
		if ((bs >> 4) & 1) {
			/* Read-only */
		}

		if ((bs >> 5) & 1) {
			/* Reserved */
		}
		if ((bs >> 6) & 1) {
			/* Reserved */
		}
		if ((bs >> 7) & 1) {
			/* Reserved */
		}
		break;

	case 0x38:
		/* Reserved */
		break;

	case 0x40:
		/* PCI Express Egress Port Base Address Register */
		/* 5.1.12 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.pxpepbaren = (val >> 0) & 0b1;
			/* Bits 1-7: Hardwired to '0' */
		}
		if ((bs >> 1) & 1) {
			/* Bits 8-11: Hardwired to '0' */
			cpssp->NAME.pxpepbar &= ~(0xf0 << 8);
			cpssp->NAME.pxpepbar |= val & (0xf0 << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.pxpepbar &= ~(0xff << 16);
			cpssp->NAME.pxpepbar |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.pxpepbar &= ~(0xff << 24);
			cpssp->NAME.pxpepbar |= val & (0xff << 24);
		}
		if ((bs >> 4) & 1) {
			cpssp->NAME.pxpepbar &= ~((uint64_t) 0x0f << 32);
			cpssp->NAME.pxpepbar |= val & ((uint64_t) 0x0f << 32);
			/* Bits 36-39: Hardwired to '0' */
		}
		if ((bs >> 5) & 1) {
			/* Hardwired to '0' */
		}
		if ((bs >> 6) & 1) {
			/* Hardwired to '0' */
		}
		if ((bs >> 7) & 1) {
			/* Hardwired to '0' */
		}
		break;

	case 0x48:
		/* Memory Mapped Register Range Base Address Register */
		/* 5.1.13 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.mchbaren = (val >> 0) & 0b1;
			/* Bit 1-7: Hardwired to '0' */
		}
		if ((bs >> 1) & 1) {
			/* Bit 8-13: Hardwired to '0' */
			cpssp->NAME.mchbar &= ~(0xc0 << 8);
			cpssp->NAME.mchbar |= val & (0xc0 << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.mchbar &= ~(0xff << 16);
			cpssp->NAME.mchbar |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.mchbar &= ~(0xff << 24);
			cpssp->NAME.mchbar |= val & (0xff << 24);
		}
		if ((bs >> 4) & 1) {
			cpssp->NAME.mchbar &= ~((uint64_t) 0x0f << 32);
			cpssp->NAME.mchbar |= val & ((uint64_t) 0x0f << 32);
			/* Bit 36-39: Hardwired to '0' */
		}
		if ((bs >> 5) & 1) {
			/* Hardwired to '0' */
		}
		if ((bs >> 6) & 1) {
			/* Hardwired to '0' */
		}
		if ((bs >> 7) & 1) {
			/* Hardwired to '0' */
		}
		break;

	case 0x50:
	case 0x58:
		/* Reserved */
		break;

	case 0x60:
		/* PCI Express Register Range Base Address Register */
		/* 5.1.16 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.pciexbaren = (val >> 0) & 0b1;
			cpssp->NAME.length = (val >> 1) & 0b11;
			switch (cpssp->NAME.length) {
			case 0b00:
				/* 256 Buses */
				cpssp->NAME.pciexmsk = ~(uint64_t) ((1 << 28) - 1);
				break;
			case 0b01:
				/* 128 Buses */
				cpssp->NAME.pciexmsk = ~(uint64_t) ((1 << 27) - 1);
				break;
			case 0b10:
				/* 64 Buses */
				cpssp->NAME.pciexmsk = ~(uint64_t) ((1 << 26) - 1);
				break;
			case 0b11:
				/* Reserved */
				fprintf(stderr, "WARNING: %s length=%d.\n",
						__FUNCTION__, cpssp->NAME.length);
				break;
			}
			/* Bit 3-7: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.pciexbar &= ~(0xfc << 24);
			cpssp->NAME.pciexbar |= val & (0xfc << 24);
			cpssp->NAME.pciexbar &= cpssp->NAME.pciexmsk;
		}
		if ((bs >> 4) & 1) {
			cpssp->NAME.pciexbar &= ~((uint64_t) 0x0f << 32);
			cpssp->NAME.pciexbar |= val & ((uint64_t) 0x0f << 32);
			/* Bit 36-39: Hardwired to '0' */
		}
		if ((bs >> 5) & 1) {
			/* Hardwired to '0'. */
		}
		if ((bs >> 6) & 1) {
			/* Hardwired to '0'. */
		}
		if ((bs >> 7) & 1) {
			/* Hardwired to '0'. */
		}
		break;

	case 0x68:
		/* Root Complex Register Range Base Address Register */
		/* 5.1.17 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.dmibaren = (val >> 0) & 0b1;
			/* Bits 1-7: Hardwired to '0' */
		}
		if ((bs >> 1) & 1) {
			/* Bits 8-11: Hardwired to '0' */
			cpssp->NAME.dmibar &= ~(0xf0 << 8);
			cpssp->NAME.dmibar |= val & (0xf0 << 8);
		}
		if ((bs >> 2) & 1) {
			cpssp->NAME.dmibar &= ~(0xff << 16);
			cpssp->NAME.dmibar |= val & (0xff << 16);
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.dmibar &= ~(0xff << 24);
			cpssp->NAME.dmibar |= val & (0xff << 24);
		}
		if ((bs >> 4) & 1) {
			cpssp->NAME.dmibar &= ~((uint64_t) 0x0f << 32);
			cpssp->NAME.dmibar |= val & ((uint64_t) 0x0f << 32);
			/* Bit 36-39: Hardwired to '0' */
		}
		if ((bs >> 5) & 1) {
			/* Hardwired to '0'. */
		}
		if ((bs >> 6) & 1) {
			/* Hardwired to '0'. */
		}
		if ((bs >> 7) & 1) {
			/* Hardwired to '0'. */
		}
		break;

	case 0x70:
	case 0x78:
	case 0x80:
	case 0x88:
		/* Reserved */
		break;

	case 0x90:
		/* Programable Attribute Map Register 0 */
		/* 5.1.18 */
		if ((bs >> 0) & 1) {
			/* Bit 0-3: Reserved */
			cpssp->NAME.pam0 = (val >> 4) & 0b11;
			/* Bit 6-7: Reserved */

			NAME_(host_out_unmap)(cpssp, 0x000f0000, 0x10000);
			NAME_(dmi_out_unmap)(cpssp, 0x000f0000, 0x10000);
		}
		/* Programable Attribute Map Register 1 */
		/* 5.1.19 */
		if ((bs >> 1) & 1) {
			cpssp->NAME.pam1[0] = (val >> 8) & 0b11;
			/* Bit 2-3: Reserved */
			cpssp->NAME.pam1[1] = (val >> 12) & 0b11;
			/* Bit 6-7: Reserved */

			NAME_(host_out_unmap)(cpssp, 0x000c0000, 0x8000);
			NAME_(dmi_out_unmap)(cpssp, 0x000c0000, 0x8000);
		}
		/* Programable Attribute Map Register 2 */
		/* 5.1.20 */
		if ((bs >> 2) & 1) {
			cpssp->NAME.pam1[2] = (val >> 16) & 0b11;
			/* Bit 2-3: Reserved */
			cpssp->NAME.pam1[3] = (val >> 20) & 0b11;
			/* Bit 6-7: Reserved */

			NAME_(host_out_unmap)(cpssp, 0x000c8000, 0x8000);
			NAME_(dmi_out_unmap)(cpssp, 0x000c8000, 0x8000);
		}
		/* Programable Attribute Map Register 3 */
		if ((bs >> 3) & 1) {
			cpssp->NAME.pam1[4] = (val >> 24) & 0b11;
			/* Bit 2-3: Reserved */
			cpssp->NAME.pam1[5] = (val >> 28) & 0b11;
			/* Bit 6-7: Reserved */

			NAME_(host_out_unmap)(cpssp, 0x000d0000, 0x8000);
			NAME_(dmi_out_unmap)(cpssp, 0x000d0000, 0x8000);
		}
		/* Programable Attribute Map Register 4 */
		if ((bs >> 4) & 1) {
			cpssp->NAME.pam1[6] = (val >> 32) & 0b11;
			/* Bit 2-3: Reserved */
			cpssp->NAME.pam1[7] = (val >> 36) & 0b11;
			/* Bit 6-7: Reserved */

			NAME_(host_out_unmap)(cpssp, 0x000d8000, 0x8000);
			NAME_(dmi_out_unmap)(cpssp, 0x000d8000, 0x8000);
		}
		/* Programable Attribute Map Register 5 */
		if ((bs >> 5) & 1) {
			cpssp->NAME.pam1[8] = (val >> 40) & 0b11;
			/* Bit 2-3: Reserved */
			cpssp->NAME.pam1[9] = (val >> 44) & 0b11;
			/* Bit 6-7: Reserved */

			NAME_(host_out_unmap)(cpssp, 0x000e0000, 0x8000);
			NAME_(dmi_out_unmap)(cpssp, 0x000e0000, 0x8000);
		}
		/* Programable Attribute Map Register 6 */
		if ((bs >> 6) & 1) {
			cpssp->NAME.pam1[10] = (val >> 48) & 0b11;
			/* Bit 2-3: Reserved */
			cpssp->NAME.pam1[11] = (val >> 52) & 0b11;
			/* Bit 6-7: Reserved */

			NAME_(host_out_unmap)(cpssp, 0x000e8000, 0x8000);
			NAME_(dmi_out_unmap)(cpssp, 0x000e8000, 0x8000);
		}

		/* Legacy Access Control Register */
		/* 5.1.25 */
		if ((bs >> 7) & 1) {
			/* FIXME */
		}
		break;

	case 0x98:
		/* Remap Base Address Register */
		/* 5.1.26 */
		if ((bs >> 0) & 1) {
			cpssp->NAME.remapbase &= ~((uint64_t) 0xff << 26);
			cpssp->NAME.remapbase |= ((val >> 0) & 0xff) << 26;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.remapbase &= ~((uint64_t) 0b11 << (8 + 26));
			cpssp->NAME.remapbase |= ((val >> 8) & 0b11) << (8 + 26);
			/* Bit 10-15: Reserved */
		}

		/* Remap Limit Address Register */
		/* 5.1.27 */
		if ((bs >> 2) & 1) {
			cpssp->NAME.remaplmt &= ~((uint64_t) 0xff << 26);
			cpssp->NAME.remaplmt |= ((val >> 16) & 0xff) << 26;
		}
		if ((bs >> 3) & 1) {
			cpssp->NAME.remaplmt &= ~((uint64_t) 0b11 << (8 + 26));
			cpssp->NAME.remaplmt |= ((val >> 24) & 0b11) << (8 + 26);
			/* Bit 10-15: Reserved */
		}

		/* Reserved */
		if ((bs >> 4) & 1) {
			/* Reserved */
		}

		/* System Management RAM Control Register */
		/* 5.1.28 */
		/* FIXME */
		if ((bs >> 5) & 1) {
			/* Bits 16-18: Hardwired to 0b010 */
			cpssp->NAME.g_smrame = (val >> (3 + 40)) & 0b1;
			cpssp->NAME.d_lck |= (val >> (4 + 40)) & 0b1;
			cpssp->NAME.d_cls = (val >> (5 + 40)) & 0b1;
			cpssp->NAME.d_open = (! cpssp->NAME.d_lck)
					& (val >> (6 + 40)) & 0b1;
			/* Bit 7: Reserved */

			/* SMRAM mapping may have changed */
			NAME_(host_out_unmap)(cpssp, 0x000a0000, 0x00020000);
			NAME_(dmi_out_unmap)(cpssp, 0x000a0000, 0x00020000);
		}

		if ((bs >> 6) & 1) {
			/* Extended System Management RAM Control Register */
			/* 5.1.29 */
			/* FIXME */
		}
		if ((bs >> 7) & 1) {
			/* Reserved */
		}
		break;

	case 0xb0:
		/* Top of Low Usable DRAM */
		/* 5.1.35 */
		if ((bs >> 0) & 1) {
			/* Bits 0-3: Reserved */
			cpssp->NAME.tolud &= ~(0xf0 << 16);
			cpssp->NAME.tolud |= ((val >> 0) & 0xf0) << 16;
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.tolud &= ~(0xff << 24);
			cpssp->NAME.tolud |= ((val >> 8) & 0xff) << 24;
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}
		if ((bs >> 4) & 1) {
			/* Reserved */
		}
		if ((bs >> 5) & 1) {
			/* Reserved */
		}
		if ((bs >> 6) & 1) {
			/* Reserved */
		}
		if ((bs >> 7) & 1) {
			/* Reserved */
		}
		break;

	/* FIXME */

	case 0xd8:
		/* Reserved */
		if ((bs >> 0) & 1) {
			/* Reserved */
		}
		if ((bs >> 1) & 1) {
			/* Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Reserved */
		}
		if ((bs >> 3) & 1) {
			/* Reserved */
		}

		/* Scratchpad Data */
		/* 5.1.39 */
		if ((bs >> 4) & 1) {
			cpssp->NAME.skpd &= ~(0xff << 0);
			cpssp->NAME.skpd |= (val >> 32) & (0xff << 0);
		}
		if ((bs >> 5) & 1) {
			cpssp->NAME.skpd &= ~(0xff << 8);
			cpssp->NAME.skpd |= (val >> 32) & (0xff << 8);
		}
		if ((bs >> 6) & 1) {
			cpssp->NAME.skpd &= ~(0xff << 16);
			cpssp->NAME.skpd |= (val >> 32) & (0xff << 16);
		}
		if ((bs >> 7) & 1) {
			cpssp->NAME.skpd &= ~(0xff << 24);
			cpssp->NAME.skpd |= (val >> 32) & (0xff << 24);
		}
		break;

	case 0xe0:
		/* Capability Identifier */
		/* 5.1.40 */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Read-only */
		}
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}
		if ((bs >> 4) & 1) {
			/* Read-only */
		}
		if ((bs >> 5) & 1) {
			/* Read-only */
		}
		if ((bs >> 6) & 1) {
			/* Read-only */
		}
		if ((bs >> 7) & 1) {
			/* Read-only */
		}
		break;

	case 0xe8:
		/* cont */
		if ((bs >> 0) & 1) {
			/* Read-only */
		}
		if ((bs >> 1) & 1) {
			/* Read-only */
		}
		if ((bs >> 2) & 1) {
			/* Read-only */
		}
		if ((bs >> 3) & 1) {
			/* Read-only */
		}

		if ((bs >> 4) & 1) {
			/* Reserved */
		}
		if ((bs >> 5) & 1) {
			/* Reserved */
		}
		if ((bs >> 6) & 1) {
			/* Reserved */
		}
		if ((bs >> 7) & 1) {
			/* Reserved */
		}
		break;

	default:
		fprintf(stderr, "WARNING: %s addr=0x%" PRIx64 "\n",
				__FUNCTION__, addr);
		break;
	}
}

static int
NAME_(c0r)(
	struct cpssp *cpssp,
	uint64_t addr,
	unsigned int *bsp,
	uint64_t *valp
)
{
	if (((addr >> 11) & 1)		/* Device 0 */
	 && ((addr >> 8) & 7) == 0) {	/* Function 0 */
		/* Host/PCI Bridge */
		addr &= 0xff;
		NAME_(host_pci_bridge_c0r)(cpssp, addr, *bsp, valp);
		*bsp &= ~0xff;
		return 0;

	} else if (((addr >> 12) & 1)		/* Device 1 */
		&& ((addr >> 8) & 7) == 0) {	/* Function 0 */
		/* AGP - FIXME VOSSI */
		return -1;

	} else {
		return -1;
	}
}

static int
NAME_(c0w)(
	struct cpssp *cpssp,
	uint64_t addr,
	unsigned int *bsp,
	uint64_t val
)
{
	if (((addr >> 11) & 1)		/* Device 0 */
	 && ((addr >> 8) & 7) == 0) {	/* Function 0 */
		/* Host/PCI Bridge */
		addr &= 0xff;
		NAME_(host_pci_bridge_c0w)(cpssp, addr, *bsp, val);
		*bsp &= ~0xff;
		return 0;

	} else if (((addr >> 12) & 1)		/* Device 1 */
		&& ((addr >> 8) & 7) == 0) {	/* Function 0 */
		/* AGP - FIXME VOSSI */
		return -1;

	} else {
		return -1;
	}
}

/* --------------------------------------------------------------------- */
/* Memory Interface                                                      */
/* --------------------------------------------------------------------- */

static int
NAME_(region)(
	struct cpssp *cpssp,
	int wflag,
	unsigned int smm,
	uint64_t addr
)
{
	int region;
	unsigned int rw;

	if (cpssp->NAME.tolud <= addr) {
		return -1;
	} else if ((addr >> 26) < cpssp->NAME.mch_c0drb0) {
		region = 0;
	} else if ((addr >> 26) < cpssp->NAME.mch_c0drb1) {
		region = 1;
	} else if ((addr >> 26) < cpssp->NAME.mch_c0drb2) {
		region = 2;
	} else if ((addr >> 26) < cpssp->NAME.mch_c0drb3) {
		region = 3;
	} else if ((addr >> 26) < cpssp->NAME.mch_c1drb0) {
		region = 4;
	} else if ((addr >> 26) < cpssp->NAME.mch_c1drb1) {
		region = 5;
	} else if ((addr >> 26) < cpssp->NAME.mch_c1drb2) {
		region = 6;
	} else if ((addr >> 26) < cpssp->NAME.mch_c1drb3) {
		region = 7;
	} else {
		/* Shortcut */
		return -1;
	}

	if (/* 0x00000000 <= addr && */ addr < 0x000a0000) {
		/*
		 * Low memory.
		 */
		rw = 3;

	} else if (0x000a0000 <= addr && addr < 0x000c0000) {
		/*
		 * VGA hole / SMRAM
		 */
		if (cpssp->NAME.g_smrame
		 && (smm || cpssp->NAME.d_open)) {
			/* SMRAM */
			rw = 3;
		} else {
			/* VGA Hole */
			/* Shortcut */
			return -1;
		}

	} else if (0x000c0000 <= addr && addr < 0x000c4000) {
		rw = cpssp->NAME.pam1[0];

	} else if (0x000c4000 <= addr && addr < 0x000c8000) {
		rw = cpssp->NAME.pam1[1];

	} else if (0x000c8000 <= addr && addr < 0x000cc000) {
		rw = cpssp->NAME.pam1[2];

	} else if (0x000cc000 <= addr && addr < 0x000d0000) {
		rw = cpssp->NAME.pam1[3];

	} else if (0x000d0000 <= addr && addr < 0x000d4000) {
		rw = cpssp->NAME.pam1[4];

	} else if (0x000d4000 <= addr && addr < 0x000d8000) {
		rw = cpssp->NAME.pam1[5];

	} else if (0x000d8000 <= addr && addr < 0x000dc000) {
		rw = cpssp->NAME.pam1[6];

	} else if (0x000dc000 <= addr && addr < 0x000e0000) {
		rw = cpssp->NAME.pam1[7];

	} else if (0x000e0000 <= addr && addr < 0x000e4000) {
		rw = cpssp->NAME.pam1[8];

	} else if (0x000e4000 <= addr && addr < 0x000e8000) {
		rw = cpssp->NAME.pam1[9];

	} else if (0x000e8000 <= addr && addr < 0x000ec000) {
		rw = cpssp->NAME.pam1[10];

	} else if (0x000ec000 <= addr && addr < 0x000f0000) {
		rw = cpssp->NAME.pam1[11];

	} else if (0x000f0000 <= addr && addr < 0x00100000) {
		/*
		 * BIOS Area configured by PAM register 0x59.
		 */
		rw = cpssp->NAME.pam0;

	} else {
		/*
		 * High memory.
		 */
		rw = 3;
	}

	if (! (rw & (1 << wflag))) {
		return -1;
	}

	return region;
}

static int
NAME_(mem_mr)(
	struct cpssp *cpssp,
	int smm,
	uint64_t addr,
	unsigned int bs,
	uint64_t *valp
)
{
	int region;

	region = NAME_(region)(cpssp, 0, smm, addr);
	if (0 <= region
	 && region <= 7) {
		return NAME_(mem_out_read)(cpssp, region, addr, bs, valp);
	}
	return 1;
}

static int
NAME_(mem_mw)(
	struct cpssp *cpssp,
	int smm,
	uint64_t addr,
	unsigned int bs,
	uint64_t val
)
{
	int region;

	region = NAME_(region)(cpssp, 1, smm, addr);
	if (0 <= region
	 && region <= 7) {
		return NAME_(mem_out_write)(cpssp, region, addr, bs, val);
	}
	return 1;
}

static int
NAME_(mem_map_r)(
	struct cpssp *cpssp,
	enum NAME_(orig) orig,
	unsigned int smm,
	uint64_t addr,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t *),
	void **csp,
	char **haddr_p
)
{
	int region;

	region = NAME_(region)(cpssp, 0, smm, addr);
	if (0 <= region
	 && region <= 7) {
		return NAME_(mem_out_map_r)(cpssp, region, addr,
				cfp, csp, haddr_p);
	}
	return 1;
}

static int
NAME_(mem_map_w)(
	struct cpssp *cpssp,
	enum NAME_(orig) orig,
	unsigned int smm,
	uint64_t addr,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t),
	void **csp,
	char **haddr_p
)
{
	int region;

	region = NAME_(region)(cpssp, 1, smm, addr);
	if (0 <= region
	 && region <= 7) {
		return NAME_(mem_out_map_w)(cpssp, region, addr,
				cfp, csp, haddr_p);
	}
	return 1;
}

static void
NAME_(memX_unmap)(void *_cpssp, uint64_t pa, uint64_t len)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	/* Should unmap interval [pa, pa + len) only - FIXME VOSSI */
	NAME_(host_out_unmap)(cpssp, 0, 1024*1024*1024);
	NAME_(dmi_out_unmap)(cpssp, 0, 1024*1024*1024);
}

/* --------------------------------------------------------------------- */
/* DMI Interface                                                         */
/* --------------------------------------------------------------------- */

static int
NAME_(dmi_read)(
	struct cpssp *cpssp,
	unsigned int type,
	uint64_t addr,
	unsigned int bs,
	uint64_t *valp
)
{
	int ret;

	switch (type) {
	case SIG_DMI_BUS_IOR:
		ret = NAME_(dmi_out_ior)(cpssp, addr, bs, valp);
		break;
	case SIG_DMI_BUS_C0R:
		ret = NAME_(dmi_out_main_c0r)(cpssp, addr, bs, valp);
		break;
	case SIG_DMI_BUS_C1R:
		ret = NAME_(dmi_out_c1r)(cpssp, addr, bs, valp);
		break;
	case SIG_DMI_BUS_MR:
		ret = NAME_(dmi_out_mr)(cpssp, addr, bs, valp);
		break;
	default:
		assert(0);
	}
	if (ret) {
		NAME_(dmi_out_main_addr_type)(cpssp, addr, type);
		ret = NAME_(dmi_out_main_read_data)(cpssp, bs, valp);
	}
	return ret;
}

static int
NAME_(dmi_write)(
	struct cpssp *cpssp,
	unsigned int type,
	uint64_t addr,
	unsigned int bs,
	uint64_t val
)
{
	int ret;

	switch (type) {
	case SIG_DMI_BUS_IOW:
		ret = NAME_(dmi_out_iow)(cpssp, addr, bs, val);
		break;
	case SIG_DMI_BUS_C0W:
		ret = NAME_(dmi_out_main_c0w)(cpssp, addr, bs, val);
		break;
	case SIG_DMI_BUS_C1W:
		ret = NAME_(dmi_out_c1w)(cpssp, addr, bs, val);
		break;
	case SIG_DMI_BUS_MW:
		ret = NAME_(dmi_out_mw)(cpssp, addr, bs, val);
		break;
	default:
		assert(0);
	}
	if (ret) {
		NAME_(dmi_out_main_addr_type)(cpssp, addr, type);
		ret = NAME_(dmi_out_main_write_data)(cpssp, bs, val);
	}
	return ret;
}

static int
NAME_(dmi_map_r)(
	struct cpssp *cpssp,
	uint64_t addr,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t *),
	void **csp,
	char **haddr_p
)
{
	return NAME_(dmi_out_map_r)(cpssp, addr, cfp, csp, haddr_p);
}

static int
NAME_(dmi_map_w)(
	struct cpssp *cpssp,
	uint64_t addr,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t),
	void **csp,
	char **haddr_p
)
{
	return NAME_(dmi_out_map_w)(cpssp, addr, cfp, csp, haddr_p);
}

/* --------------------------------------------------------------------- */
/* Internal Bus                                                          */
/* --------------------------------------------------------------------- */

static int
NAME_(ior)(
	void *_cpssp,
	uint64_t port,
	unsigned int bs,
	uint64_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	unsigned int type;
	int ret;

	type = SIG_HOST775_BUS_IOR;
	ret = 1;
	ret &= NAME_(confadd_read)(cpssp, port, &bs, valp);
	if (bs) {
		NAME_(confdata1)(cpssp, &type, &port, &bs, NULL);
	}
	if (type == SIG_DMI_BUS_C0R) {
		ret &= NAME_(c0r)(cpssp, port, &bs, valp);
	}
	if (bs) {
		ret &= NAME_(dmi_read)(cpssp, type, port, bs, valp);
	}
	NAME_(confdata2)(cpssp, type, bs, valp);

	return ret;
}

static int
NAME_(iow)(
	void *_cpssp,
	uint64_t port,
	unsigned int bs,
	uint64_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	unsigned int type;
	int ret;

	type = SIG_HOST775_BUS_IOW;
	ret = 1;
	ret &= NAME_(confadd_write)(cpssp, port, &bs, val);
	if (bs) {
		NAME_(confdata1)(cpssp, &type, &port, &bs, &val);
	}
	if (type == SIG_DMI_BUS_C0W) {
		ret &= NAME_(c0w)(cpssp, port, &bs, val);
	}
	if (bs) {
		ret &= NAME_(dmi_write)(cpssp, type, port, bs, val);
	}
	NAME_(confdata2)(cpssp, type, bs, NULL);

	return ret;
}

static int
NAME_(ior_info)(
	struct cpssp *cpssp,
	enum NAME_(orig) orig,
	uint64_t port,
	unsigned int bs,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t *),
	void **csp
)
{
	if (port == 0x0cf8
	 || port == 0x0cfc) {
		*cfp = NAME_(ior);
		*csp = cpssp;
		return 0;
	} else {
		return sig_dmi_bus_ior_info(cpssp->port_dmi_bus, cpssp,
				port, bs, cfp, csp);
	}
}

static int
NAME_(iow_info)(
	struct cpssp *cpssp,
	enum NAME_(orig) orig,
	uint64_t port,
	unsigned int bs,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t),
	void **csp
)
{
	if (port == 0x0cf8
	 || port == 0x0cfc) {
		*cfp = NAME_(iow);
		*csp = cpssp;
		return 0;
	} else {
		return sig_dmi_bus_iow_info(cpssp->port_dmi_bus, cpssp,
				port, bs, cfp, csp);
	}
}

static int
NAME_(mem_read)(
	struct cpssp *cpssp,
	enum NAME_(orig) orig,
	unsigned int smm,
	uint64_t addr,
	unsigned int bs,
	uint64_t *valp
)
{
	int ret;

	if (DEBUG) {
		fprintf(stderr, "%s: smm=%d, addr=0x%lx, bs=0x%x\n",
				__FUNCTION__, smm, addr, bs);
	}

	if (NAME_(pxpep_mr)(cpssp, addr, bs, valp) == 0
	 || NAME_(mch_mr)(cpssp, addr, bs, valp) == 0
	 || NAME_(pciex_mr)(cpssp, addr, bs, valp) == 0
	 || NAME_(dmi_mr)(cpssp, addr, bs, valp) == 0
	 || NAME_(mem_mr)(cpssp, smm, addr, bs, valp) == 0
	 || (orig != ORIG_PCI_BUS
	  && NAME_(dmi_read)(cpssp, SIG_DMI_BUS_MR, addr, bs, valp) == 0)) {
		ret = 0;
	} else {
		ret = 1;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: -> ret=%d (*valp=0x%lx)\n",
				__FUNCTION__, ret, *valp);
	}

	return ret;
}

static int
NAME_(mem_write)(
	struct cpssp *cpssp,
	enum NAME_(orig) orig,
	unsigned int smm,
	uint64_t addr,
	unsigned int bs,
	uint64_t val
)
{
	int ret;

	if (DEBUG) {
		fprintf(stderr, "%s: smm=%d, addr=0x%lx, bs=0x%x val=0x%lx\n",
				__FUNCTION__, smm, addr, bs, val);
	}

	if (NAME_(pxpep_mw)(cpssp, addr, bs, val) == 0
	 || NAME_(mch_mw)(cpssp, addr, bs, val) == 0
	 || NAME_(pciex_mw)(cpssp, addr, bs, val) == 0
	 || NAME_(dmi_mw)(cpssp, addr, bs, val) == 0
	 || NAME_(mem_mw)(cpssp, smm, addr, bs, val) == 0
	 || (orig != ORIG_PCI_BUS
	  && NAME_(dmi_write)(cpssp, SIG_DMI_BUS_MW, addr, bs, val) == 0)) {
		ret = 0;
	} else {
		ret = 1;
	}

	if (DEBUG) {
		fprintf(stderr, "%s: -> ret=%d\n", __FUNCTION__, ret);
	}

	return ret;
}

static int
NAME_(map_r)(
	void *_cpssp,
	enum NAME_(orig) orig,
	unsigned int smm,
	uint64_t addr,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t *),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (NAME_(mem_map_r)(cpssp, ORIG_HOST_BUS, smm, addr, cfp, csp, haddr_p) == 0
	 || NAME_(dmi_map_r)(cpssp, addr, cfp, csp, haddr_p) == 0) {
		return 0;
	}
	return 1;
}

static int
NAME_(map_w)(
	void *_cpssp,
	enum NAME_(orig) orig,
	unsigned int smm,
	uint64_t addr,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	if (NAME_(mem_map_w)(cpssp, ORIG_HOST_BUS, smm, addr, cfp, csp, haddr_p) == 0
	 || NAME_(dmi_map_w)(cpssp, addr, cfp, csp, haddr_p) == 0) {
		return 0;
	}
	return 1;
}

/* ----------------------------------------------------------------- */
/* Host Bus Interface                                                */
/* ----------------------------------------------------------------- */

static int
NAME_(host_bus_ior)(
	void *_cpssp,
	uint64_t port,
	unsigned int bs,
	uint64_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(ior)(cpssp, port, bs, valp);
}

static int
NAME_(host_bus_iow)(
	void *_cpssp,
	uint64_t port,
	unsigned int bs,
	uint64_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(iow)(cpssp, port, bs, val);
}

static int
NAME_(host_bus_ior_info)(
	void *_cpssp,
	uint64_t port,
	unsigned int bs,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t *),
	void **csp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(ior_info)(cpssp, ORIG_HOST_BUS, port, bs, cfp, csp);
}

static int
NAME_(host_bus_iow_info)(
	void *_cpssp,
	uint64_t port,
	unsigned int bs,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t),
	void **csp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(iow_info)(cpssp, ORIG_HOST_BUS, port, bs, cfp, csp);
}

static int
NAME_(host_bus_inta_addr)(
	void *_cpssp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(dmi_out_inta_addr)(cpssp);
}

static int
NAME_(host_bus_inta_data)(
	void *_cpssp,
	uint8_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(dmi_out_inta_data)(cpssp, valp);
}

static int
NAME_(host_bus_mr)(
	void *_cpssp,
	unsigned int smm,
	uint64_t addr,
	unsigned int bs,
	uint64_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(mem_read)(cpssp, ORIG_HOST_BUS,
			smm, addr, bs, valp);
}

static int
NAME_(host_bus_mw)(
	void *_cpssp,
	unsigned int smm,
	uint64_t addr,
	unsigned int bs,
	uint64_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(mem_write)(cpssp, ORIG_HOST_BUS,
			smm, addr, bs, val);
}

/* Re-used as NAME_(host_bus_map_x_check). */
static int
NAME_(host_bus_map_r_check)(
	void *_cpssp,
	unsigned int smm,
	uint64_t pa
)
{
	struct cpssp *cpssp = _cpssp;

	/* SMM? FIXME */
	return NAME_(dmi_out_map_r_check)(cpssp, pa);
}

/* Re-used as NAME_(host_bus_map_x). */
static int
NAME_(host_bus_map_r)(
	void *_cpssp,
	unsigned int smm,
	uint64_t pa,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t *),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(map_r)(cpssp, ORIG_HOST_BUS, smm, pa, cfp, csp, haddr_p);
}

static int
NAME_(host_bus_map_w_check)(
	void *_cpssp,
	unsigned int smm,
	uint64_t pa
)
{
	struct cpssp *cpssp = _cpssp;

	/* SMM? FIXME */
	return NAME_(dmi_out_map_w_check)(cpssp, pa);
}

static int
NAME_(host_bus_map_w)(
	void *_cpssp,
	unsigned int smm,
	uint64_t pa,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = _cpssp;

	return NAME_(map_w)(cpssp, ORIG_HOST_BUS, smm, pa, cfp, csp, haddr_p);
}

static int
NAME_(host_bus_addr_type)(
	void *_cpssp,
	unsigned int smm,
	uint64_t addr,
	unsigned int type
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	cpssp->NAME.selected_smm = smm;
	cpssp->NAME.selected_addr = addr;
	cpssp->NAME.selected_type = type;

	return 0;
}

static int
NAME_(host_bus_read_data)(
	void *_cpssp,
	unsigned int bs,
	uint64_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	int smm;
	unsigned int type;
	uint64_t addr;

	smm = cpssp->NAME.selected_smm;
	addr = cpssp->NAME.selected_addr;
	type = cpssp->NAME.selected_type;

	cpssp->NAME.selected_addr += 8;

	switch (type) {
	case SIG_HOST775_BUS_IOR:
		return NAME_(host_bus_ior)(cpssp, addr, bs, valp);
	case SIG_HOST775_BUS_MR:
		return NAME_(host_bus_mr)(cpssp, smm, addr, bs, valp);
	default:
		assert(0);
	}
}

static int
NAME_(host_bus_write_data)(
	void *_cpssp,
	unsigned int bs,
	uint64_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	int smm;
	unsigned int type;
	uint64_t addr;

	smm = cpssp->NAME.selected_smm;
	addr = cpssp->NAME.selected_addr;
	type = cpssp->NAME.selected_type;

	cpssp->NAME.selected_addr += 8;

	switch (type) {
	case SIG_HOST775_BUS_IOW:
		return NAME_(host_bus_iow)(cpssp, addr, bs, val);
	case SIG_HOST775_BUS_MW:
		return NAME_(host_bus_mw)(cpssp, smm, addr, bs, val);
	default:
		assert(0);
	}
}

/* ----------------------------------------------------------------- */
/* PCI Bus Interface                                                 */
/* ----------------------------------------------------------------- */

static int
NAME_(dmi_bus_mr)(
	void *_cpssp,
	uint64_t addr,
	unsigned int bs,
	uint64_t *valp
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(mem_read)(cpssp, ORIG_PCI_BUS,
			0, addr, bs, valp);
}

static int
NAME_(dmi_bus_mw)(
	void *_cpssp,
	uint64_t addr,
	unsigned int bs,
	uint64_t val
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(mem_write)(cpssp, ORIG_PCI_BUS,
			0, addr, bs, val);
}

static int
NAME_(dmi_bus_map_r_check)(
	void *_cpssp,
	uint64_t pa
)
{
	struct cpssp *cpssp = _cpssp;

	/* SMM? FIXME */
	return NAME_(host_out_map_r_check)(cpssp, 0, pa);
}

static int
NAME_(dmi_bus_map_r)(
	void *_cpssp,
	uint64_t pa,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t *),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(map_r)(cpssp, ORIG_PCI_BUS, 0, pa, cfp, csp, haddr_p);
}

static int
NAME_(dmi_bus_map_w_check)(
	void *_cpssp,
	uint64_t pa
)
{
	struct cpssp *cpssp = _cpssp;

	/* SMM? FIXME */
	return NAME_(host_out_map_w_check)(cpssp, 0, pa);
}

static int
NAME_(dmi_bus_map_w)(
	void *_cpssp,
	uint64_t pa,
	int (**cfp)(void *, uint64_t, unsigned int, uint64_t),
	void **csp,
	char **haddr_p
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	return NAME_(map_w)(cpssp, ORIG_PCI_BUS, 0, pa, cfp, csp, haddr_p);
}

static void
NAME_(dmi_bus_unmap)(
	void *_cpssp,
	uint64_t pa,
	uint64_t len
)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;

	NAME_(host_out_unmap)(cpssp, pa, len);
}

static void
NAME_(power_set)(void *_cpssp, unsigned int val)
{
	/* FIXME */
}

static void
NAME_(n_reset_set)(void *_cpssp, unsigned int n_val)
{
	struct cpssp *cpssp = (struct cpssp *) _cpssp;
	int i;

	/* PCI Command */
	/* 5.1.3 */
	cpssp->NAME.perre = 0b0;
	cpssp->NAME.serre = 0b0;

	/* Subsystem Vendor Identification Register */
	/* 5.1.9 */
	cpssp->NAME.svid = 0x0000;

	/* Subsystem Identification Register */
	/* 5.1.10 */
	cpssp->NAME.sid = 0x0000;

	/* PCI Express Egress Port Base Address Register */
	/* 5.1.12 */
	cpssp->NAME.pxpepbaren = 0b0;
	cpssp->NAME.pxpepbar = 0x0000000000000000;

	/* Memory Mapped Register Range Base Address Register */
	/* 5.1.13 */
	cpssp->NAME.mchbaren = 0b0;
	cpssp->NAME.mchbar = 0x0000000000000000;

	/* PCI Express Register Range Base Address Register */
	/* 5.1.16 */
	cpssp->NAME.pciexbaren = 0;
	cpssp->NAME.length = 0b00;
	cpssp->NAME.pciexmsk = ~(uint64_t) ((1 << 28) - 1);
	cpssp->NAME.pciexbar = 0x00000000e0000000;

	/* Root Complex Register Range Base Address Register */
	/* 5.1.17 */
	cpssp->NAME.dmibaren = 0;
	cpssp->NAME.dmibar = 0x0000000000000000;

	/* Programable Attribute Map Registers */
	/* 5.1.18 - 5.1.24 */
	cpssp->NAME.pam0 = 0b00;
	for (i = 0; i < 12; i++) {
		cpssp->NAME.pam1[i] = 0b00;
	}

	/* Remap Base Address Register */
	/* 5.1.26 */
	cpssp->NAME.remapbase = (uint64_t) 0b1111111111 << 26;

	/* Remap Limit Address Register */
	/* 5.1.27 */
	cpssp->NAME.remaplmt = (uint64_t) 0b0000000000 << 26;

	/* System Management RAM Control Register */
	/* 5.1.28 */
	/* FIXME */
	cpssp->NAME.g_smrame = 0b0;
	cpssp->NAME.d_lck = 0b0;
	cpssp->NAME.d_cls = 0b0;
	cpssp->NAME.d_open = 0b0;

	/* Top of Low Usable DRAM */
	/* 5.1.35 */
	cpssp->NAME.tolud = 0x001 << 20;

	/* Scratchpad Data */
	/* 5.1.39 */
	cpssp->NAME.skpd = 0x00000000;

	/* DRAM Boundary Registers */
	cpssp->NAME.mch_c0drb0 = 0x01;
	cpssp->NAME.mch_c0drb1 = 0x01;
	cpssp->NAME.mch_c0drb2 = 0x01;
	cpssp->NAME.mch_c0drb3 = 0x01;
	cpssp->NAME.mch_c1drb0 = 0x01;
	cpssp->NAME.mch_c1drb1 = 0x01;
	cpssp->NAME.mch_c1drb2 = 0x01;
	cpssp->NAME.mch_c1drb3 = 0x01;
}

#endif /* BEHAVIOR */
