/*
 * Copyright (C) 2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#define DEBUG_CONTROL_FLOW	0

/*
 * 16-bit Timer/Counter
 *
 * See: ATMEL ATmega32 doc page 86 ff.
 */

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	unsigned int count;

	uint8_t temp;

	/* ??? */
	uint16_t tcnt;

	/* TCCRxA 107 */
	uint8_t comB; /* 7-6: Compare Output Mode for Compare Unit A */
	uint8_t comA; /* 5-4: Compare Output Mode for Compare Unit B */
	uint8_t focA; /* 3: ... */
	uint8_t focB; /* 2: ... */
	uint8_t wgm; /* 1-0: ... */

	/* TCCRxB 110 */
	uint8_t cs; /* 2-0: Clock Select */
	/* uint8_t wgm; See above. */

	/* OCRxAH 111 */
	/* OCRxAL 111 */
	uint16_t ocrA;

	/* OCRxBH 111 */
	/* OCRxBL 111 */
	uint16_t ocrB;

	/* TIMSK 112 */
	uint8_t ticie;
	uint8_t ocieA;
	uint8_t ocieB;
	uint8_t toie;

	/* TIFR 112 */
	uint8_t icf;
	uint8_t ocfA;
	uint8_t ocfB;
	uint8_t tov;
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

#if 0
	NAME_(irq_set)(cpssp, n, val);
#endif

static void
NAME_(update)(struct cpssp *cpssp)
{
	NAME_(irq_set)(cpssp, 0, cpssp->NAME.icf & cpssp->NAME.ticie);
	NAME_(irq_set)(cpssp, 1, cpssp->NAME.ocfA & cpssp->NAME.ocieA);
	NAME_(irq_set)(cpssp, 2, cpssp->NAME.ocfB & cpssp->NAME.ocieB);
	NAME_(irq_set)(cpssp, 3, cpssp->NAME.tov & cpssp->NAME.toie);
}

static void
NAME_(reg_set)(struct cpssp *cpssp, uint8_t addr, uint8_t val)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%02x, val=0x%02x\n", __FUNCTION__,
				addr, val);
	}

	switch (addr) {
	case 2: /* OCRxBL */
		cpssp->NAME.ocrB = (cpssp->NAME.temp << 8) | (val << 0);
		break;
	case 3: /* OCRxBH */
		cpssp->NAME.temp = val;
		break;
	case 4: /* OCRxAL */
		cpssp->NAME.ocrA = (cpssp->NAME.temp << 8) | (val << 0);
		break;
	case 5: /* OCRxAH */
		cpssp->NAME.temp = val;
		break;
	case 6: /* TCNTxL */
		cpssp->NAME.tcnt = (cpssp->NAME.temp << 8) | (val << 0);
		break;
	case 7: /* TCNTxH */
		cpssp->NAME.temp = val;
		break;
	case 8: /* TCCRxB 110 */
		/* FIXME */
		cpssp->NAME.wgm &= ~0xc;
		cpssp->NAME.wgm |= ((val >> 3) & 0x3) << 2;
		cpssp->NAME.cs = (val >> 0) & 0x7;
		break;
	case 9: /* TCCRxA 107 */
		cpssp->NAME.comB = (val >> 7) & 0x3;
		cpssp->NAME.comA = (val >> 5) & 0x3;
		cpssp->NAME.focA = (val >> 3) & 0x1;
		cpssp->NAME.focB = (val >> 2) & 0x1;
		cpssp->NAME.wgm &= ~3;
		cpssp->NAME.wgm |= (val >> 0) & 0x3;
		break;
	case 10: /* (Part of) TIFR */
		cpssp->NAME.icf &= ~((val >> 3) & 0x1); /* Write-Clear */
		cpssp->NAME.ocfA &= ~((val >> 2) & 0x1); /* Write-Clear */
		cpssp->NAME.ocfB &= ~((val >> 1) & 0x1); /* Write-Clear */
		cpssp->NAME.tov &= ~((val >> 0) & 0x1); /* Write-Clear */
		NAME_(update)(cpssp);
		break;
	case 11: /* (Part of) TIMSK */
		cpssp->NAME.ticie = (val >> 3) & 0x1;
		cpssp->NAME.ocieA = (val >> 2) & 0x1;
		cpssp->NAME.ocieB = (val >> 1) & 0x1;
		cpssp->NAME.toie = (val >> 0) & 0x1;
		NAME_(update)(cpssp);
		break;
	default:
		fprintf(stderr, "%s: addr=0x%02x\n", __FUNCTION__, addr);
		assert(0);
	}
}

static void
NAME_(reg_get)(struct cpssp *cpssp, uint8_t addr, uint8_t *valp)
{
	switch (addr) {
	case 2: /* OCRxBL */
		cpssp->NAME.temp = (cpssp->NAME.ocrB >> 8) & 0xff;
		*valp = (cpssp->NAME.ocrB >> 0) & 0xff;
		break;
	case 3: /* OCRxBH */
		*valp = cpssp->NAME.temp;
		break;
	case 4: /* OCRxAL */
		cpssp->NAME.temp = (cpssp->NAME.ocrA >> 8) & 0xff;
		*valp = (cpssp->NAME.ocrA >> 0) & 0xff;
		break;
	case 5: /* OCRxAH */
		*valp = cpssp->NAME.temp;
		break;
	case 6: /* TCNTxL */
		cpssp->NAME.temp = (cpssp->NAME.tcnt >> 8) & 0xff;
		*valp = (cpssp->NAME.tcnt >> 0) & 0xff;
		break;
	case 7: /* TCNTxH */
		*valp = cpssp->NAME.temp;
		break;
	case 8: /* TCCRxB 110 */
		*valp = 0;
		/* FIXME */
		*valp |= 0x0 << 5; /* Reserved */
		*valp |= ((cpssp->NAME.wgm >> 2) & 0x3) << 3;
		*valp |= cpssp->NAME.cs << 0;
		break;
	case 9: /* TCCRxA 107 */
		*valp = 0;
		*valp |= cpssp->NAME.comA << 6;
		*valp |= cpssp->NAME.comB << 4;
		*valp |= cpssp->NAME.focA << 3;
		*valp |= cpssp->NAME.focB << 2;
		*valp |= ((cpssp->NAME.wgm >> 0) & 0x3) << 0;
		break;
	case 10: /* (Part of) TIFR */
		*valp = 0;
		*valp |= cpssp->NAME.icf << 3;
		*valp |= cpssp->NAME.ocfA << 2;
		*valp |= cpssp->NAME.ocfB << 1;
		*valp |= cpssp->NAME.tov << 0;
		break;
	case 11: /* (Part of) TIMSK */
		*valp = 0;
		*valp |= cpssp->NAME.ticie << 3;
		*valp |= cpssp->NAME.ocieA << 2;
		*valp |= cpssp->NAME.ocieB << 1;
		*valp |= cpssp->NAME.toie << 0;
		break;
	default:
		fprintf(stderr, "%s: addr=0x%02x\n", __FUNCTION__, addr);
		assert(0);
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%02x, val=0x%02x\n", __FUNCTION__,
				addr, *valp);
	}
}

static void
NAME_(ack)(struct cpssp *cpssp, unsigned int n)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: n=%u\n", __FUNCTION__, n);
	}

	switch (n) {
	case 0:
		cpssp->NAME.icf = 0;
		break;
	case 1:
		cpssp->NAME.ocfA = 0;
		break;
	case 2:
		cpssp->NAME.ocfB = 0;
		break;
	case 3:
		cpssp->NAME.tov = 0;
		break;
	default:
		assert(0); /* Mustn't happen. */
	}
	NAME_(update)(cpssp);
}

static void
NAME_(tick)(struct cpssp *cpssp)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s\n", __FUNCTION__);
	}

	cpssp->NAME.tcnt++;

	switch (cpssp->NAME.wgm) {
	case 0x0:
	case 0x2:
	case 0x3:
		goto todo;

	case 0x4: /* CTC using OCRA. */
		if (cpssp->NAME.tcnt == cpssp->NAME.ocrA) {
			cpssp->NAME.ocfA = 1;
			NAME_(update)(cpssp);
			cpssp->NAME.tcnt = 0;
		}
		break;

	case 0x5:
	case 0x6:
	case 0x7:
	case 0x8:
	case 0x9:
	case 0xa:
	case 0xb:
	case 0xc:
	case 0xd:
	case 0xe:
	case 0xf:
	todo:	;
		fprintf(stderr, "WARNING: %s: wgm=0x%x\n", __FUNCTION__,
				cpssp->NAME.wgm);
		break;
	default:
		assert(0); /* Cannot happen. */
	}
}

static void
NAME_(step)(struct cpssp *cpssp)
{
	unsigned int end;

	switch (cpssp->NAME.cs) {
	case 1: /* clk / 1 */
		end = 1 << 0;
		break;
	case 2: /* clk / 8 */
		end = 1 << 3;
		break;
	case 3: /* clk / 64 */
		end = 1 << 6;
		break;
	case 4: /* clk / 256 */
		end = 1 << 8;
		break;
	case 5:
		/* clk / 1024 */
		end = 1 << 10;
		break;
	case 0: /* No clock source. */
	case 6: /* External clock source (Falling edge). */
	case 7: /* External clock source (Raising edge). */
		end = 0;
		break;
	default:
		assert(0); /* Cannot happen. */
	}

	if (end) {
		cpssp->NAME.count++;
		if ((cpssp->NAME.count & (end - 1)) == 0) {
			NAME_(tick)(cpssp);
			cpssp->NAME.count = 0;
		}
	}
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	cpssp->NAME.count = 0;

	cpssp->NAME.tcnt = 0x0000;
	cpssp->NAME.cs = 0;
	cpssp->NAME.temp = 0x00;

	/* TIMSK 112 */
	cpssp->NAME.ticie = 0;
	cpssp->NAME.ocieA = 0;
	cpssp->NAME.ocieB = 0;
	cpssp->NAME.toie = 0;

	/* TIFR 112 */
	cpssp->NAME.icf = 0;
	cpssp->NAME.ocfA = 0;
	cpssp->NAME.ocfB = 0;
	cpssp->NAME.tov = 0;
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	NAME_(reset)(cpssp);
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
