/*
 * Copyright (C) 2014-2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * For most of the comments, infos, ... see:
 * ARM Architecture Reference Manual (ARM DDI 0100I)
 */

#define DEBUG_CONTROL_FLOW	0
#define DEBUG_CONTROL_FLOW_REGS	0

#ifdef INCLUDE

#include <assert.h>
#include <inttypes.h>
#include <stdio.h>

#include "glue.h"
#include "gdb.h"

#endif /* INCLUDE */
#ifdef STATE

struct {
	struct gdb *gdb;
	uint32_t brkstart[4];
	uint32_t brkend[4];

	int wfi;
	int exception_pending;

	uint32_t reg[13];
	uint32_t sp_main;
	uint32_t sp_process;
	uint32_t lr;
	enum { MODE_THREAD, MODE_HANDLER } current_mode;
	int control_spsel;
	int control_npriv;
	int primask_pm;
	int flag_shift_c;
	/*
	 * Program status register
	 * A2.5, A2-11
	 */
	int flag_c;
	int flag_n;
	int flag_v;
	int flag_z;
	int flag_t;
	int flag_align;
	int ipsr;

	uint32_t pc;
	uint32_t pc_next;
	uint32_t pc_next_next;
	uint16_t insn_prev;
	uint32_t insn;
	uint32_t insn_next;
	int flush;
	int stall;
	uint8_t itstate;

	uint32_t vtor;
	uint8_t prigroup;

	uint8_t sevonpend;
	uint8_t sleepdeep;
	uint8_t sleeponexit;
} NAME;

#endif /* STATE */
#ifdef BEHAVIOR

#if DEBUG_CONTROL_FLOW_REGS
static void
NAME_(dump)(struct cpssp *cpssp)
{
	fprintf(stderr, "r00=%08lx r01=%08lx r02=%08lx r03=%08lx\n",
		cpssp->NAME.reg[0], cpssp->NAME.reg[1],
		cpssp->NAME.reg[2], cpssp->NAME.reg[3]);
	fprintf(stderr, "r04=%08lx r05=%08lx r06=%08lx r07=%08lx\n",
		cpssp->NAME.reg[4], cpssp->NAME.reg[5],
		cpssp->NAME.reg[6], cpssp->NAME.reg[7]);
	fprintf(stderr, "r08=%08lx r09=%08lx r10=%08lx r11=%08lx\n",
		cpssp->NAME.reg[8], cpssp->NAME.reg[9],
		cpssp->NAME.reg[10], cpssp->NAME.reg[11]);
	fprintf(stderr, "r12=%08lx lr=%08lx\n",
		cpssp->NAME.reg[12], cpssp->NAME.lr);
	fprintf(stderr, "sp_main=%08lx sp_process=%08lx\n",
		cpssp->NAME.sp_main, cpssp->NAME.sp_process);
	fprintf(stderr, "c=%d n=%d v=%d z=%d t=%d align=%d ipsr=0x%02x\n",
		cpssp->NAME.flag_c, cpssp->NAME.flag_n,
		cpssp->NAME.flag_v, cpssp->NAME.flag_z,
		cpssp->NAME.flag_t, cpssp->NAME.flag_align,
		cpssp->NAME.ipsr);
#if 0
	fprintf(stderr, "r00=%e ", *(float *) &cpssp->NAME.reg[0]);
	fprintf(stderr, "r01=%e ", *(float *) &cpssp->NAME.reg[1]);
	fprintf(stderr, "r02=%e ", *(float *) &cpssp->NAME.reg[2]);
	fprintf(stderr, "r03=%e ", *(float *) &cpssp->NAME.reg[3]);
	fprintf(stderr, "\n");
	fprintf(stderr, "r04=%e ", *(float *) &cpssp->NAME.reg[4]);
	fprintf(stderr, "r05=%e ", *(float *) &cpssp->NAME.reg[5]);
	fprintf(stderr, "r06=%e ", *(float *) &cpssp->NAME.reg[6]);
	fprintf(stderr, "r07=%e ", *(float *) &cpssp->NAME.reg[7]);
	fprintf(stderr, "\n");
	fprintf(stderr, "r08=%e ", *(float *) &cpssp->NAME.reg[8]);
	fprintf(stderr, "r09=%e ", *(float *) &cpssp->NAME.reg[9]);
	fprintf(stderr, "r10=%e ", *(float *) &cpssp->NAME.reg[10]);
	fprintf(stderr, "r11=%e ", *(float *) &cpssp->NAME.reg[11]);
	fprintf(stderr, "\n");
#endif
}
#endif

static void
NAME_(store_flags)(struct cpssp *cpssp, int c, int n, int v, int z)
{
	cpssp->NAME.flag_c = c;
	cpssp->NAME.flag_n = n;
	cpssp->NAME.flag_v = v;
	cpssp->NAME.flag_z = z;
}

static uint32_t
NAME_(load_sp)(struct cpssp *cpssp)
{
	switch (cpssp->NAME.control_spsel) {
	case 0:
		return cpssp->NAME.sp_main;
	case 1:
		return cpssp->NAME.sp_process;
	default:
		assert(0);
	}
}

static void
NAME_(store_sp)(struct cpssp *cpssp, uint32_t val)
{
	switch (cpssp->NAME.control_spsel) {
	case 0:
		cpssp->NAME.sp_main = val;
		break;
	case 1:
		cpssp->NAME.sp_process = val;
		break;
	default:
		assert(0);
	}
}

static uint32_t
NAME_(load_lr)(struct cpssp *cpssp)
{
	return cpssp->NAME.lr;
}

static void
NAME_(store_lr)(struct cpssp *cpssp, uint32_t val)
{
	cpssp->NAME.lr = val;
}

static uint32_t
NAME_(load_pc)(struct cpssp *cpssp)
{
	return cpssp->NAME.pc_next_next;
}

static void
NAME_(store_pc)(struct cpssp *cpssp, uint32_t val)
{
	cpssp->NAME.pc_next_next = val;
	cpssp->NAME.flush = 1;
}

static uint32_t
NAME_(load_reg)(struct cpssp *cpssp, int nr)
{
	switch (nr) {
	case 13:
		return NAME_(load_sp)(cpssp);
	case 14:
		return NAME_(load_lr)(cpssp);
	case 15:
		return NAME_(load_pc)(cpssp);
	default:
		return cpssp->NAME.reg[nr];
	}
}

static void
NAME_(store_reg)(struct cpssp *cpssp, int nr, uint32_t val)
{
	switch (nr) {
	case 13:
		NAME_(store_sp)(cpssp, val);
		break;
	case 14:
		NAME_(store_lr)(cpssp, val);
		break;
	case 15:
		NAME_(store_pc)(cpssp, val);
		break;
	default:
		cpssp->NAME.reg[nr] = val;
		break;
	}
}

#if 5 <= CONFIG_VERSION && CONFIG_E
static uint64_t
NAME_(load_reg64)(struct cpssp *cpssp, int nr)
{
	uint32_t val0;
	uint32_t val1;

	assert(! (nr & 1));

	val0 = NAME_(load_reg)(cpssp, nr + 0);
	val1 = NAME_(load_reg)(cpssp, nr + 1);

	return ((uint64_t) val1 << 32) | ((uint64_t) val0 << 0);
}

static void
NAME_(store_reg64)(struct cpssp *cpssp, int nr, uint64_t val)
{
	assert(! (nr & 1));

	NAME_(store_reg)(cpssp, nr + 0, ((val >>  0) & 0xffffffff));
	NAME_(store_reg)(cpssp, nr + 1, ((val >> 32) & 0xffffffff));
}
#endif /* 5 <= CONFIG_VERSION && CONFIG_E */

/*
 * Program status register
 * A2.5, A2-11
 */
static uint32_t
NAME_(load_apsr)(struct cpssp *cpssp)
{
	return (cpssp->NAME.flag_n << 31)
		| (cpssp->NAME.flag_z << 30)
		| (cpssp->NAME.flag_c << 29)
		| (cpssp->NAME.flag_v << 28);
}

static void
NAME_(store_apsr)(struct cpssp *cpssp, uint32_t val)
{
	cpssp->NAME.flag_n = (val >> 31) & 0b1;
	cpssp->NAME.flag_z = (val >> 30) & 0b1;
	cpssp->NAME.flag_c = (val >> 29) & 0b1;
	cpssp->NAME.flag_v = (val >> 28) & 0b1;
}

static uint32_t
NAME_(load_epsr)(struct cpssp *cpssp)
{
	return (cpssp->NAME.flag_t << 24)
		| (cpssp->NAME.flag_align << 9);
}

static void
NAME_(store_epsr)(struct cpssp *cpssp, uint32_t val)
{
	cpssp->NAME.flag_t = (val >> 24) & 0b1;
	cpssp->NAME.flag_align = (val >> 9) & 0b1;
}

static uint32_t
NAME_(load_ipsr)(struct cpssp *cpssp)
{
	return cpssp->NAME.ipsr << 0;
}

static void
NAME_(store_ipsr)(struct cpssp *cpssp, uint32_t val)
{
	cpssp->NAME.ipsr = val & 0x3f;
}

static uint32_t
NAME_(load_xpsr)(struct cpssp *cpssp)
{
	return NAME_(load_apsr)(cpssp)
		| NAME_(load_epsr)(cpssp)
		| NAME_(load_ipsr)(cpssp);
}

static void
NAME_(store_xpsr)(struct cpssp *cpssp, uint32_t val)
{
	NAME_(store_apsr)(cpssp, val);
	NAME_(store_epsr)(cpssp, val);
	NAME_(store_ipsr)(cpssp, val);
}

static uint32_t
NAME_(load_cpsr)(struct cpssp *cpssp)
{
	return NAME_(load_xpsr)(cpssp);
}

static void
NAME_(store_cpsr)(struct cpssp *cpssp, uint32_t val)
{
	NAME_(store_xpsr)(cpssp, val);
}

static uint32_t
NAME_(load_spsr)(struct cpssp *cpssp)
{
	assert(0);
	return 0; /* FIXME */
}

static void
NAME_(store_spsr)(struct cpssp *cpssp, uint32_t val)
{
	assert(0); /* FIXME */
}

#if CONFIG_M
static void
NAME_(store_spec)(struct cpssp *cpssp, uint32_t sysm, uint32_t val)
{
	switch ((sysm >> 3) & 0b11111) {
	case 0b00000:
		if (((sysm >> 2) & 1) == 0) {
			val &= 0xf8000000;
			NAME_(store_apsr)(cpssp, val);
		}
		break;
	case 0b00001:
		if (cpssp->NAME.current_mode == MODE_HANDLER
		 || ! cpssp->NAME.control_npriv) {
			val &= ~0b11;
			switch ((sysm >> 0) & 0b111) {
			case 0b000:
				cpssp->NAME.sp_main = val;
				break;
			case 0b001:
				cpssp->NAME.sp_process = val;
				break;
			default:
				assert(0); /* FIXME */
			}
		}
		break;
	case 0b00010:
		if (cpssp->NAME.current_mode == MODE_HANDLER
		 || ! cpssp->NAME.control_npriv) {
			switch ((sysm >> 0) & 0b111) {
			case 0b000:
				NAME_(primask_set)(cpssp, (val >> 0) & 1);
				break;
			case 0b001:
				/* BASEPRI */
				/* FIXME */
				break;
			case 0b010:
				/* BASEPRI_MAX */
				assert(0); /* FIXME */
				break;
			case 0b011:
				assert(0); /* FIXME */
				break;
			case 0b100:
				if (cpssp->NAME.current_mode == MODE_THREAD) {
					cpssp->NAME.control_spsel = (val >> 1) & 1;
					cpssp->NAME.control_npriv = (val >> 0) & 1;
				}
				break;
			default:
				assert(0); /* FIXME */
			}
		}
		break;
	default:
		assert(0); /* FIXME */
	}
}

static uint32_t
NAME_(load_spec)(struct cpssp *cpssp, uint32_t sysm)
{
	uint32_t val;

	switch ((sysm >> 3) & 0b11111) {
	case 0b00000:
		val = 0;
		if ((sysm >> 0) & 1) {
			val |= NAME_(load_ipsr(cpssp));
		}
		if ((sysm >> 1) & 1) {
			val |= 0 << 24; /* T-bit reads as zero */
		}
		if (((sysm >> 2) & 1) == 0) {
			val |= NAME_(load_apsr)(cpssp);
		}
		break;
	case 0b00001:
		switch ((sysm >> 0) & 0b111) {
		case 0b000:
			val = cpssp->NAME.sp_main;
			break;
		case 0b001:
			val = cpssp->NAME.sp_process;
			break;
		default:
			assert(0);
		}
		break;
	case 0b00010:
		switch ((sysm >> 0) & 0b111) {
		case 0b000:
			val = NAME_(primask_get)(cpssp);
			break;
		case 0b001:
			/* BASEPRI */
		case 0b010:
			/* BASEPRI_MAX */
			val = 0; /* FIXME */
			break;
		case 0b011:
			assert(0); /* FIXME */
			break;
		case 0b100:
			val = (cpssp->NAME.control_spsel << 1)
				| (cpssp->NAME.control_npriv << 0);
			break;
		default:
			assert(0); /* FIXME */
		}
		break;
	default:
		assert(0); /* FIXME */
	}

	return val;
}
#endif /* CONFIG_M */

#if ! CONFIG_ARM
static void
NAME_(SCB_ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	/*
	 * System Control and ID Registers
	 * B3.2.2
	 * ARMv7-M: B3-652
	 */
	*valp = 0;
	switch (addr & 0xff) {
	case 0x00:
		/*
		 * CPUID Base Register
		 * B3.2.3
		 * ARMv7-M: B3-655
		 */
		*valp = ('F' << 24) /* Implementer 'F'AU */
			| (0x0 << 20) /* Implementation Defined */
			| (0xc0 << 16) /* ARMv6-M */
			| (0x000 << 4) /* Implementation Defined */
			| (0x0 << 0); /* Implementation Defined */
		break;
	case 0x04:
		/*
		 * Interrupt Control State Register (ICSR)
		 * B3-265
		 * ARMv7-M: B3-655
		 */
#if 0 /* FIXME */
		*valp |= cpssp->NAME.NMI_pending << 31;
		/* 30-29: Reserved */
		*valp |= cpssp->NAME.PendSV_pending << 28;
		/* 27: Write-only */
		*valp |= cpssp->NAME.SysTick_pending << 26;
#endif
		/* 25: Write-only */
		/* 24: Reserved */
#if CONFIG_DEBUG
		assert(0); /* FIXME */
#else
		/* 23-22: Reserved */
#endif
		/* 21: Reserved */
		// assert(0); /* 20-12: FIXME */
		/* 11-9: Reserved */
		*valp |= cpssp->NAME.ipsr << 0;
		fprintf(stderr, "WARNING: %s: addr=0x%x\n", __FUNCTION__, addr);
		break;
	case 0x08:
		/*
		 * Vector Table Offset Register (VTOR)
		 * B3-267
		 * ARMv7-M: B3-657
		 */
		*valp = cpssp->NAME.vtor;
		break;
	case 0x0c:
		/*
		 * Application Interrupt and Reset Control Register (AIRCR)
		 * B3-268
		 * ARMv7-M: B3-658
		 */
		*valp = 0;
		*valp |= 0xfa05 << 16;
		*valp |= 0 << 15; /* Little Endian */
		/* Bit 14-11: Reserved */
		*valp |= cpssp->NAME.prigroup << 8;
		/* Bit 7-3: Reserved */
		*valp |= 0 << 2; /* Write-only */
		*valp |= 0 << 1; /* Write-only */
		*valp |= 0 << 0; /* Write-only */
		break;
	case 0x10:
		/* System Control Register (SCR) */
		/* B3-269 */
		/* 31-5: Reserved */
		*valp |= cpssp->NAME.sevonpend << 4;
		/* 3: Reserved */
		*valp |= cpssp->NAME.sleepdeep << 2;
		*valp |= cpssp->NAME.sleeponexit << 1;
		/* 0: Reserved */
		break;
	case 0x14:
		/* Configuration and Control Register (CCR) */
		/* B3-271 */
		/* 31-10: Reserved */
		*valp |= 1 << 9; /* Stack Align */
		*valp |= 0b11111 << 4; /* 8-4: Reserved */
		*valp |= 1 << 3; /* Unaligned Trap */
		/* 2-0: Reserved */
		break;
	case 0x1c:
		/* System Handler Priority Register 2 (SHPR2) */
		/* B3-272 */
		*valp |= NAME_(pri_get)(cpssp, 11) << 24;
		/* 29-0: Reserved */
		break;
	case 0x20:
		/* System Handler Priority Register 3 (SHPR3) */
		/* B3-273 */
		*valp |= NAME_(pri_get)(cpssp, 15) << 24;
		/* 29-24: Reserved */
		*valp |= NAME_(pri_get)(cpssp, 14) << 16;
		/* 21-0: Reserved */
		break;
#if CONFIG_DEBUG
	case 0x24:
		/* System Handler Control and State Register (SHCSR) */
		/* C1-329 */
		assert(0); /* FIXME */
	case 0x30:
		/* Debug Fault Status Register (DFSR) */
		/* C1-330 */
		assert(0); /* FIXME */
#endif /* CONFIG_DEBUG */
	case 0x88:
		/* Coprocessor Access Control Register */
		*valp |= 0; /* FIXME */
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx\n",
				__FUNCTION__, addr);
		assert(0); /* FIXME */
	}
}

static void
NAME_(SCB_st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	/*
	 * System Control Block
	 * B3.2.2
	 * ARMv7-M: B3-652
	 */
	switch (addr & 0xff) {
	case 0x00:
		/*
		 * CPUID Base Register
		 * B3.2.3
		 * ARMv7-M: B3-655
		 */
		/* Read-only */
		break;
	case 0x04:
		/*
		 * Interrupt Control State Register (ICSR)
		 * B3-265
		 * ARMv7-M: B3-655
		 */
		if ((val >> 31) & 1) {
			/* Set pending NMI. */
			NAME_(NMI_pending_set)(cpssp, 1);
		}
		/* 30-29: Reserved */
		switch ((val >> 27) & 0b11) {
		case 0b00:
			/* Nothing to do... */
			break;
		case 0b01:
			/* Clear pending PendSV. */
			NAME_(PendSV_pending_set)(cpssp, 0);
			break;
		case 0b10:
			/* Set pending PendSV. */
			NAME_(PendSV_pending_set)(cpssp, 1);
			break;
		case 0b11:
		default:
			assert(0); /* FIXME */
		}
		switch ((val >> 25) & 0b11) {
		case 0b00:
			/* Nothing to do... */
			break;
		case 0b01:
			/* Clear pending SysTick. */
			NAME_(SysTick_pending_set)(cpssp, 0);
			break;
		case 0b10:
			/* Set pending SysTick. */
			NAME_(SysTick_pending_set)(cpssp, 1);
			break;
		case 0b11:
		default:
			assert(0); /* FIXME */
		}
		/* 24: Reserved */
#if CONFIG_DEBUG
		assert(0); /* FIXME */
#else
		/* 23-22: Reserved */
#endif
		/* 21: Reserved */
		/* 20-12: Read-only */
		/* 11-9: Reserved */
#if CONFIG_DEBUG
		assert(0); /* FIXME */
#else
		/* 8-0: Reserved */
#endif
		break;
	case 0x08:
		/*
		 * Vector Table Offset Register (VTOR)
		 * B3-267
		 * ARMv7-M: B3-657
		 */
		assert(bs == 0b1111);

		cpssp->NAME.vtor = val & 0xffffff80;
		break;
	case 0x0c:
		/*
		 * Application Interrupt and Reset Control Register (AIRCR)
		 * B3-268
		 * ARMv7-M: B3-658
		 */
		assert(bs == 0b1111);

		assert(((val >> 16) & 0xffff) == 0x05fa); /* FIXME */

		/* Bit 15: Read-only */
		/* Bit 14-11: Reserved */
		cpssp->NAME.prigroup = (val >> 8) & 0x7;
		/* Bit 7-3: Reserved */
		if ((val >> 2) & 1) {
			/* sysresetreq */
			assert(0); /* FIXME */
		}
		if ((val >> 1) & 1) {
			/* vectclractive */
			assert(0); /* FIXME */
		}
		if ((val >> 0) & 1) {
			/* vectreset */
			assert(0); /* FIXME */
		}
		break;
	case 0x10:
		/* System Control Register (SCR) */
		/* B3-269 */
		assert(bs == 0b1111);

		/* 31-5: Reserved */
		cpssp->NAME.sevonpend = (val >> 4) & 1;
		/* 3: Reserved */
		cpssp->NAME.sleepdeep = (val >> 2) & 1;
		cpssp->NAME.sleeponexit = (val >> 1) & 1;
		/* 0: Reserved */
		break;
	case 0x14:
		/* Configuration and Control Register (CCR) */
		/* B3-271 */
		assert(bs == 0b1111);

		/* Read-only */
		break;
	case 0x1c:
		/* System Handler Priority Register 2 (SHPR2) */
		/* B3-272 */
		if ((bs >> 3) & 1) {
			NAME_(pri_set)(cpssp, 11, (val >> 24) & 0xff);
		}
		if ((bs >> 2) & 1) {
			/* 23-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			/* 15-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-0: Reserved */
		}
		break;
	case 0x20:
		/* System Handler Priority Register 3 (SHPR3) */
		/* B3-273 */
		if ((bs >> 3) & 1) {
			NAME_(pri_set)(cpssp, 15, (val >> 24) & 0xff);
		}
		if ((bs >> 2) & 1) {
			NAME_(pri_set)(cpssp, 14, (val >> 16) & 0xff);
		}
		if ((bs >> 1) & 1) {
			/* 15-8: Reserved */
		}
		if ((bs >> 0) & 1) {
			/* 7-0: Reserved */
		}
		break;
#if CONFIG_DEBUG
	case 0x24:
		/* System Handler Control and State Register (SHCSR) */
		/* C1-329 */
		assert(bs == 0b1111);

		assert(0); /* FIXME */
	case 0x30:
		/* Debug Fault Status Register (DFSR) */
		/* C1-330 */
		assert(bs == 0b1111);

		assert(0); /* FIXME */
#endif /* CONFIG_DEBUG */
	case 0x88:
		/* Coprocessor Access Control Register */
		/* FIXME */
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08lx val=0x%08lx\n",
				__FUNCTION__, addr, val);
		assert(0); /* FIXME */
	}
}
#endif /* ! CONFIG_ARM */

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, int bs, uint32_t *valp)
{
#if ! CONFIG_ARM
	if (0xf8000000 <= addr /* && addr < 0x100000000 */) {
		/* Direct GPIO access. */
		NAME_(gpio_ld)(cpssp, addr, bs, valp);
	} else
#endif
	{
		NAME_(mr)(cpssp, addr, bs, valp);
	}
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, int bs, uint32_t val)
{
#if ! CONFIG_ARM	/* FIXME */
	if (0xf8000000 <= addr /* && addr < 0x100000000 */) {
		/* Direct GPIO access. */
		NAME_(gpio_st)(cpssp, addr, bs, val);
	} else
#endif
	{
		NAME_(mw)(cpssp, addr, bs, val);
	}
}

#if CONFIG_THUMB
static uint16_t
NAME_(ld16_code)(struct cpssp *cpssp, uint32_t addr)
{
	uint32_t val32;
	uint16_t val;

	switch (addr & 3) {
	case 0:
		NAME_(mx)(cpssp, addr & ~3, 0b0011, &val32);
		val = val32 >> 0;
		break;
	case 1:
		assert(0); /* Unaligned - FIXME */
	case 2:
		NAME_(mx)(cpssp, addr & ~3, 0b1100, &val32);
		val = val32 >> 16;
		break;
	case 3:
		assert(0); /* Unaligned - FIXME */
	default:
		assert(0); /* Cannot happen. */
	}

	return val;
}
#endif /* CONFIG_THUMB */

#if CONFIG_ARM
static uint32_t
NAME_(ld32_code)(struct cpssp *cpssp, uint32_t addr)
{
	uint32_t val;

	NAME_(mx)(cpssp, addr, 0b1111, &val);

	return val;
}
#endif /* CONFIG_ARM */

static uint8_t
NAME_(ld8)(struct cpssp *cpssp, uint32_t addr)
{
	uint32_t val32;
	uint8_t val;

	cpssp->NAME.stall = 1;

	switch (addr & 3) {
	case 0:
		NAME_(ld)(cpssp, addr & ~3, 0b0001, &val32);
		val = val32 >> 0;
		break;
	case 1:
		NAME_(ld)(cpssp, addr & ~3, 0b0010, &val32);
		val = val32 >> 8;
		break;
	case 2:
		NAME_(ld)(cpssp, addr & ~3, 0b0100, &val32);
		val = val32 >> 16;
		break;
	case 3:
		NAME_(ld)(cpssp, addr & ~3, 0b1000, &val32);
		val = val32 >> 24;
		break;
	default:
		assert(0); /* Cannot happen. */
	}

	return val;
}

static void
NAME_(st8)(struct cpssp *cpssp, uint32_t addr, uint8_t val)
{
	uint32_t val32;

	cpssp->NAME.stall = 1;

	switch (addr & 3) {
	case 0:
		val32 = val << 0;
		NAME_(st)(cpssp, addr & ~3, 0b0001, val32);
		break;
	case 1:
		val32 = val << 8;
		NAME_(st)(cpssp, addr & ~3, 0b0010, val32);
		break;
	case 2:
		val32 = val << 16;
		NAME_(st)(cpssp, addr & ~3, 0b0100, val32);
		break;
	case 3:
		val32 = val << 24;
		NAME_(st)(cpssp, addr & ~3, 0b1000, val32);
		break;
	default:
		assert(0); /* Cannot happen. */
	}
}

static uint16_t
NAME_(ld16)(struct cpssp *cpssp, uint32_t addr)
{
	uint32_t val32;
	uint16_t val;

	cpssp->NAME.stall = 1;

	switch (addr & 3) {
	case 0:
		NAME_(ld)(cpssp, addr & ~3, 0b0011, &val32);
		val = val32 >> 0;
		break;
	case 1:
		assert(0); /* Unaligned - FIXME */
	case 2:
		NAME_(ld)(cpssp, addr & ~3, 0b1100, &val32);
		val = val32 >> 16;
		break;
	case 3:
		assert(0); /* Unaligned - FIXME */
	default:
		assert(0); /* Cannot happen. */
	}

	return val;
}

static void
NAME_(st16)(struct cpssp *cpssp, uint32_t addr, uint16_t val)
{
	uint32_t val32;

	cpssp->NAME.stall = 1;

	switch (addr & 3) {
	case 0:
		val32 = val << 0;
		NAME_(st)(cpssp, addr & ~3, 0b0011, val32);
		break;
	case 1:
		assert(0); /* Unaligned - FIXME */
	case 2:
		val32 = val << 16;
		NAME_(st)(cpssp, addr & ~3, 0b1100, val32);
		break;
	case 3:
		assert(0); /* Unaligned - FIXME */
	default:
		assert(0); /* Cannot happen. */
	}
}

static uint32_t
NAME_(ld32)(struct cpssp *cpssp, uint32_t addr)
{
	uint32_t val;

	assert(! (addr & 3));

	cpssp->NAME.stall = 1;

	NAME_(ld)(cpssp, addr, 0b1111, &val);

	return val;
}

static void
NAME_(st32)(struct cpssp *cpssp, uint32_t addr, uint32_t val)
{
	assert(! (addr & 3));

	cpssp->NAME.stall = 1;

	NAME_(st)(cpssp, addr, 0b1111, val);
}

#if 5 <= CONFIG_VERSION && CONFIG_E
static uint64_t
NAME_(ld64)(struct cpssp *cpssp, uint32_t addr)
{
	uint32_t val0;
	uint32_t val1;

	assert(! (addr & 7));

	cpssp->NAME.stall = 1;

	NAME_(ld)(cpssp, addr + 0, 0b1111, &val0);
	NAME_(ld)(cpssp, addr + 4, 0b1111, &val1);

	return ((uint64_t) val1 << 32) | ((uint64_t) val0 << 0);
}

static void
NAME_(st64)(struct cpssp *cpssp, uint32_t addr, uint64_t val)
{
	assert(! (addr & 7));

	cpssp->NAME.stall = 1;

	NAME_(st)(cpssp, addr + 0, 0b1111, (val >>  0) & 0xffffffff);
	NAME_(st)(cpssp, addr + 4, 0b1111, (val >> 32) & 0xffffffff);
}
#endif /* 5 <= CONFIG_VERSION && CONFIG_E */

/*
 * Addressing Mode 2 - Load and Store Word or Unsigned Byte
 * A5.2.1, page A5-19.
 */
static uint32_t
NAME_(addr_mode_2_pre)(
	struct cpssp *cpssp,
	int p,
	int u,
	int w,
	uint8_t rn,
	uint32_t offset
)
{
	uint32_t addr;

	addr = NAME_(load_reg)(cpssp, rn);
	if (rn == 15
	 && cpssp->NAME.flag_t) {
		addr &= ~3;
	}
	if (p) {
		if (u) {
			addr += offset;
		} else {
			addr -= offset;
		}
	}

	return addr;
}

static void
NAME_(addr_mode_2_post)(
	struct cpssp *cpssp,
	int p,
	int u,
	int w,
	uint8_t rn,
	uint32_t addr,
	uint32_t offset
)
{
	if (! p) {
		if (u) {
			addr += offset;
		} else {
			addr -= offset;
		}
	}
	if (! p || w) {
		NAME_(store_reg)(cpssp, rn, addr);
	}
}

static void
NAME_(ldst)(
	struct cpssp *cpssp,
	int p,
	int b,
	int w,
	int l,
	uint8_t rd,
	uint32_t addr
)
{
	uint32_t tmp32;

	assert(! (! p && w)); /* User-mode-access - FIXME */

	if (b) {
		/* Unsigned Byte Access */
		if (l) {
			/* Load */
			tmp32 = (uint32_t) (uint8_t) NAME_(ld8)(cpssp, addr);
			NAME_(store_reg)(cpssp, rd, tmp32);
		} else {
			/* Store */
			tmp32 = NAME_(load_reg)(cpssp, rd);
			NAME_(st8)(cpssp, addr, (uint8_t) tmp32);
		}
	} else {
		/* Word Access */
		if (l) {
			/* Load */
			tmp32 = NAME_(ld32)(cpssp, addr);
			NAME_(store_reg)(cpssp, rd, tmp32);
		} else {
			/* Store */
			tmp32 = NAME_(load_reg)(cpssp, rd);
			NAME_(st32)(cpssp, addr, tmp32);
		}
	}
}

/*
 * Addressing Mode 4 - Load and Store Multiple, A5.4, A5-41
 * Encoding, A5.4.1, A5-42
 */
static uint32_t
NAME_(addr_mode_4_pre)(struct cpssp *cpssp, uint8_t rn)
{
	return NAME_(load_reg)(cpssp, rn);
}

static void
NAME_(addr_mode_4_post)(struct cpssp *cpssp, int w, uint8_t rn, uint32_t addr)
{
	if (w) {
		NAME_(store_reg)(cpssp, rn, addr);
	}
}

static uint32_t
NAME_(ldstm)(
	struct cpssp *cpssp,
	int p,
	int u,
	int s,
	int l,
	uint32_t addr,
	uint16_t reglist
)
{
	uint32_t tmp32;
	int i;

	assert(! s); /* FIXME */

	if (u) {
		for (i = 0; i <= 15; i++) {
			if ((reglist >> i) & 1) {
				if (p) {
					addr += 4;
				}
				if (l) {
					tmp32 = NAME_(ld32)(cpssp, addr);
					if (i == 15) {
						cpssp->NAME.flag_t = tmp32 & 1;
						NAME_(store_pc)(cpssp, tmp32 & ~1);
					} else {
						NAME_(store_reg)(cpssp, i, tmp32);
					}
				} else {
					if (i == 15) {
						tmp32 = NAME_(load_pc)(cpssp);
						tmp32 |= cpssp->NAME.flag_t;
					} else {
						tmp32 = NAME_(load_reg)(cpssp, i);
					}
					NAME_(st32)(cpssp, addr, tmp32);
				}
				if (! p) {
					addr += 4;
				}
			}
		}
	} else {
		for (i = 15; 0 <= i; i--) {
			if ((reglist >> i) & 1) {
				if (p) {
					addr -= 4;
				}
				if (l) {
					tmp32 = NAME_(ld32)(cpssp, addr);
					if (i == 15) {
						cpssp->NAME.flag_t = tmp32 & 1;
						NAME_(store_pc)(cpssp, tmp32 & ~1);
					} else {
						NAME_(store_reg)(cpssp, i, tmp32);
					}
				} else {
					if (i == 15) {
						tmp32 = NAME_(load_pc)(cpssp);
						tmp32 |= cpssp->NAME.flag_t;
					} else {
						tmp32 = NAME_(load_reg)(cpssp, i);
					}
					NAME_(st32)(cpssp, addr, tmp32);
				}
				if (! p) {
					addr -= 4;
				}
			}
		}
	}

	return addr;
}

/* AND, A4.1.4, A4-8 */
static uint32_t
NAME_(ands)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	int *cp,
	int *vp
)
{
	uint32_t res;

	res = op0 & op1;
	*cp = cpssp->NAME.flag_shift_c;
	*vp = cpssp->NAME.flag_v;

	return res;
}

/* EOR, A4.1.18, A4-32 */
static uint32_t
NAME_(eors)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	int *cp,
	int *vp
)
{
	uint32_t res;

	res = op0 ^ op1;
	*cp = cpssp->NAME.flag_shift_c;
	*vp = cpssp->NAME.flag_v;

	return res;
}

/* ORR, A4.1.42, A4-84 */
static uint32_t
NAME_(orrs)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	int *cp,
	int *vp
)
{
	uint32_t res;

	res = op0 | op1;
	*cp = cpssp->NAME.flag_shift_c;
	*vp = cpssp->NAME.flag_v;

	return res;
}

/* BIC, A4.1.6, A4-12 */
static uint32_t
NAME_(bics)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	int *cp,
	int *vp
)
{
	uint32_t res;

	res = op0 & ~op1;
	*cp = cpssp->NAME.flag_shift_c;
	*vp = cpssp->NAME.flag_v;

	return res;
}

/* ADC, A4.1.2, A4-4 */
static uint32_t
NAME_(adcs)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	int c,
	int *cp,
	int *vp
)
{
	uint64_t unsigned_sum;
	int64_t signed_sum;
	uint32_t result;

	unsigned_sum = (uint64_t) op0 + (uint64_t) op1 + (uint64_t) c;
	signed_sum = (int64_t) (int32_t) op0
			+ (int64_t) (int32_t) op1 + (int64_t) (int32_t) c;
	result = (uint32_t) unsigned_sum;
	*cp = (uint64_t) result != unsigned_sum;
	*vp = (int64_t) (int32_t) result != signed_sum;

	return result;
}

/* SBC, A4.1.65, A4-125 */
static uint32_t
NAME_(sbcs)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	int c,
	int *cp,
	int *vp
)
{
	uint64_t unsigned_sum;
	int64_t signed_sum;
	uint32_t result;

	unsigned_sum = (uint64_t) op0 + ~(uint64_t) op1 + (uint64_t) c;
	signed_sum = (int64_t) (int32_t) op0 + ~(int64_t) (int32_t) op1 + (int64_t) (int32_t) c;
	result = (uint32_t) unsigned_sum;
	*cp = (uint64_t) result == unsigned_sum;
	*vp = (int64_t) (int32_t) result != signed_sum;

	return result;
}

/* MOV, A4.1.35, A4-68 */
static uint32_t
NAME_(movs)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	int *cp,
	int *vp
)
{
	uint32_t res;

	(void) op0;
	res = op1;
	*cp = cpssp->NAME.flag_shift_c;
	*vp = cpssp->NAME.flag_v;

	return res;
}

/* MVN, A4.1.41, A4-82 */
static uint32_t
NAME_(mvns)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	int *cp,
	int *vp
)
{
	uint32_t res;

	(void) op0;
	res = ~op1;
	*cp = cpssp->NAME.flag_shift_c;
	*vp = cpssp->NAME.flag_v;

	return res;
}

/*
 * Data-processing instructions
 * A3.4, page A3-7.
 */
static uint32_t
NAME_(alu)(
	struct cpssp *cpssp,
	uint8_t opcode,
	uint32_t op0,
	uint32_t op1,
	int *cp,
	int *np,
	int *vp,
	int *zp
)
{
	uint32_t res;

	switch (opcode) {
	case 0x0:
		/* AND, A4.1.4, A4-8 */
		res = NAME_(ands)(cpssp, op0, op1, cp, vp);
		break;
	case 0x1:
		/* EOR, A4.1.18, A4-32 */
		res = NAME_(eors)(cpssp, op0, op1, cp, vp);
		break;
	case 0x2:
		/* SUB, A4.1.106, A4-208 */
		res = NAME_(sbcs)(cpssp, op0, op1, 1, cp, vp);
		break;
	case 0x3:
		/* RSB, A4.1.60, A4-115 */
		res = NAME_(sbcs)(cpssp, op1, op0, 1, cp, vp);
		break;
	case 0x4:
		/* ADD, A4.1.3, A4-6 */
		res = NAME_(adcs)(cpssp, op0, op1, 0, cp, vp);
		break;
	case 0x5:
		/* ADC, A4.1.2, A4-4 */
		res = NAME_(adcs)(cpssp, op0, op1, cpssp->NAME.flag_c, cp, vp);
		break;
	case 0x6:
		/* SBC, A4.1.65, A4-125 */
		res = NAME_(sbcs)(cpssp, op0, op1, cpssp->NAME.flag_c, cp, vp);
		break;
	case 0x7:
		/* RSC, A4.1.61, A4-117 */
		res = NAME_(sbcs)(cpssp, op1, op0, cpssp->NAME.flag_c, cp, vp);
		break;
	case 0x8:
		/* TST, A4.1.117, A4-230 */
		res = NAME_(ands)(cpssp, op0, op1, cp, vp);
		/* res not used. */
		break;
	case 0x9:
		/* TEQ, A4.1.116, A4-228 */
		res = NAME_(eors)(cpssp, op0, op1, cp, vp);
		/* res not used. */
		break;
	case 0xa:
		/* CMP, A4.1.15, A4-28 */
		res = NAME_(sbcs)(cpssp, op0, op1, 1, cp, vp);
		/* res not used. */
		break;
	case 0xb:
		/* CMN, A4.1.14, A4-26 */
		res = NAME_(adcs)(cpssp, op0, op1, 0, cp, vp);
		/* res not used. */
		break;
	case 0xc:
		/* ORR, A4.1.42, A4-84 */
		res = NAME_(orrs)(cpssp, op0, op1, cp, vp);
		break;
	case 0xd:
		/* MOV, A4.1.35, A4-68 */
		res = NAME_(movs)(cpssp, op0, op1, cp, vp);
		break;
	case 0xe:
		/* BIC, A4.1.6, A4-12 */
		res = NAME_(bics)(cpssp, op0, op1, cp, vp);
		break;
	case 0xf:
		/* MVN, A4.1.41, A4-82 */
		res = NAME_(mvns)(cpssp, op0, op1, cp, vp);
		break;
	default: assert(0); /* Cannot happen. */
	}

	*np = (res >> 31) & 1;
	*zp = res == 0;

	return res;
}

/* CLZ, A4.1.13, page A4-25 */
static uint32_t
NAME_(clz)(struct cpssp *cpssp, uint32_t op)
{
	uint32_t res;

	for (res = 0; res < 32; res++) {
		if ((op >> 31) & 1) {
			break;
		}
		op <<= 1;
	}

	return res;
}

#if CONFIG_VERSION <= 7
/*
 * BFC, ARMv7-M: A7-209
 * BFI, ARMv7-M: A7-210
 */
static uint32_t
NAME_(bfi)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	uint8_t msb,
	uint8_t lsb
)
{
	uint32_t mask;
	uint32_t res;

	mask = ((1 << (msb - lsb + 1)) - 1) << lsb;

	res = (op0 & ~mask) | ((op1 << lsb) & mask);

	return res;
}
#endif /* CONFIG_VERSION <= 7 */

/* UMULL, A4.1.129, A4-251 */
static uint64_t
NAME_(umulls)(
	struct cpssp *cpssp,
	uint32_t op0,
	uint32_t op1,
	int *cp,
	int *np,
	int *vp,
	int *zp
)
{
	uint64_t res;

	res = (uint64_t) op0 * (uint64_t) op1;
	*cp = cpssp->NAME.flag_c;
	*vp = cpssp->NAME.flag_v;
	*np = (res >> 63) & 1;
	*zp = res == 0;

	return res;
}

/* MLA, A4.1.34, A4-66 */
static uint32_t
NAME_(mlas)(
	struct cpssp *cpssp,
	uint32_t a,
	uint32_t b,
	uint32_t c,
	int *cp,
	int *np,
	int *vp,
	int *zp
)
{
	uint64_t unsigned_sum;
	// int64_t signed_sum;
	uint32_t res;

	unsigned_sum = (uint64_t) a * (uint64_t) b
			+ (uint64_t) c;
	// signed_sum = (int64_t) (int32_t) a * (int64_t) (int32_t) b
	// 		+ (int64_t) (int32_t) c;
	res = (uint32_t) unsigned_sum;
	*cp = cpssp->NAME.flag_c;
	*np = (res >> 31) & 1;
	*vp = cpssp->NAME.flag_v;
	*zp = res == 0;

	return res;
}

static int
NAME_(condition)(struct cpssp *cpssp, uint8_t cond)
{
	int res;

	assert(cond != 0b1111);

	switch ((cond >> 1) & 0x7) {
	case 0x0:
		/* beq/bne */
		res = cpssp->NAME.flag_z;
		break;
	case 0x1:
		/* bcs/bcc */
		res = cpssp->NAME.flag_c;
		break;
	case 0x2:
		/* bmi/bpl */
		res = cpssp->NAME.flag_n;
		break;
	case 0x3:
		/* bvs/bvc */
		res = cpssp->NAME.flag_v;
		break;
	case 0x4:
		/* bhi/bls */
		res = cpssp->NAME.flag_c & ! cpssp->NAME.flag_z;
		break;
	case 0x5:
		/* bge/blt */
		res = ! (cpssp->NAME.flag_n ^ cpssp->NAME.flag_v);
		break;
	case 0x6:
		/* bgt/ble */
		res = ! cpssp->NAME.flag_z
			& ! (cpssp->NAME.flag_n ^ cpssp->NAME.flag_v);
		break;
	case 0x7:
		res = 1;
		break;
	default: assert(0); /* Cannot happen. */
	}
	res ^= cond & 1;

	return res;
}

static int
NAME_(it_true)(struct cpssp *cpssp)
{
	if (cpssp->NAME.itstate == 0x00) {
		return 1;

	} else {
		int res;

		res = NAME_(condition)(cpssp, (cpssp->NAME.itstate >> 4) & 0xf);

		if ((cpssp->NAME.itstate & 0xf) == 0x8) {
			cpssp->NAME.itstate = 0x00;
		} else {
			cpssp->NAME.itstate = (cpssp->NAME.itstate & 0xe0)
					| ((cpssp->NAME.itstate & 0x0f) << 1);
		}

		return res;
	}
}

static uint32_t
NAME_(shift)(
	struct cpssp *cpssp,
	unsigned int rm,
	unsigned int op,
	unsigned int shift
)
{
	uint32_t val;

	val = NAME_(load_reg)(cpssp, rm);

	switch (op) {
	case 0x0: /* LSL */
		if (shift == 0) {
			cpssp->NAME.flag_shift_c = cpssp->NAME.flag_c;
		} else {
			cpssp->NAME.flag_shift_c = (val >> (32 - shift)) & 1;
			val <<= shift;
		}
		break;
	case 0x1: /* LSR */
		if (shift == 0) {
			cpssp->NAME.flag_shift_c = 0;
			val = 0x00000000;
		} else {
			cpssp->NAME.flag_shift_c = (val >> (shift - 1)) & 1;
			val >>= shift;
		}
		break;
	case 0x2: /* ASR */
		if (shift == 0) {
			if ((val >> 31) & 1) {
				val = 0xffffffff;
				cpssp->NAME.flag_shift_c = 1;
			} else {
				val = 0x00000000;
				cpssp->NAME.flag_shift_c = 0;
			}
		} else {
			cpssp->NAME.flag_shift_c = (val >> (shift - 1)) & 1;
			val = (uint32_t) ((int32_t) val >> shift);
		}
		break;
	case 0x3:
		if (shift == 0) {
			/* RRX */
			cpssp->NAME.flag_shift_c = (val >> 0) & 1;
			val = (cpssp->NAME.flag_c << 31) | (val >> 1);
		} else {
			/* ROR */
			cpssp->NAME.flag_shift_c = (val >> (shift - 1)) & 1;
			val = (val << (32 - shift)) | (val >> shift);
		}
		break;
	default:
		assert(0);
	}

	return val;
}

/*
 * Immediate shifts
 * A5.1.1, page A5-3.
 */
static uint32_t
NAME_(imm_shift)(struct cpssp *cpssp, uint32_t insn)
{
	uint8_t shift;
	uint8_t op;
	unsigned int rm;

	shift = (insn >> 7) & 0x1f;
	op = (insn >> 5) & 0x3;
	rm = (insn >> 0) & 0xf;

	return NAME_(shift)(cpssp, rm, op, shift);
}

/*
 * Register shifts
 * A5.1.1, page A5-3.
 */
static uint32_t
NAME_(reg_shift)(struct cpssp *cpssp, uint32_t insn)
{
	unsigned int rs;
	uint8_t shift;
	uint8_t op;
	unsigned int rm;
	uint32_t val;

	rs = (insn >> 8) & 0xf;
	op = (insn >> 5) & 0x3;
	rm = (insn >> 0) & 0xf;

	shift = NAME_(load_reg)(cpssp, rs) & 0xff;
	val = NAME_(load_reg)(cpssp, rm);

	switch (op) {
	case 0x0: /* LSL */
		if (shift == 0) {
			cpssp->NAME.flag_shift_c = cpssp->NAME.flag_c;
		} else if (shift < 32) {
			cpssp->NAME.flag_shift_c = (val >> (32 - shift)) & 1;
			val <<= shift;
		} else if (shift == 32) {
			cpssp->NAME.flag_shift_c = (val >> 0) & 1;
			val = 0x00000000;
		} else {
			cpssp->NAME.flag_shift_c = 0;
			val = 0x00000000;
		}
		break;
	case 0x1: /* LSR */
		if (shift == 0) {
			cpssp->NAME.flag_shift_c = cpssp->NAME.flag_c;
		} else if (shift < 32) {
			cpssp->NAME.flag_shift_c = (val >> (shift - 1)) & 1;
			val >>= shift;
		} else if (shift == 32) {
			cpssp->NAME.flag_shift_c = (val >> 31) & 1;
			val = 0x00000000;
		} else {
			cpssp->NAME.flag_shift_c = 0;
			val = 0x00000000;
		}
		break;
	case 0x2: /* ASR */
		if (shift == 0) {
			cpssp->NAME.flag_shift_c = cpssp->NAME.flag_c;
		} else if (shift < 32) {
			cpssp->NAME.flag_shift_c = (val >> (shift - 1)) & 1;
			val = (uint32_t) ((int32_t) val >> shift);
		} else if (32 <= shift) {
			if ((val >> 31) & 1) {
				val = 0xffffffff;
				cpssp->NAME.flag_shift_c = 1;
			} else {
				val = 0x00000000;
				cpssp->NAME.flag_shift_c = 0;
			}
		}
		break;
	case 0x3: /* ROR */
		if (shift == 0) {
			cpssp->NAME.flag_shift_c = cpssp->NAME.flag_c;
		} else if ((shift & 0x1f) == 0) {
			cpssp->NAME.flag_shift_c = (val >> 31) & 1;
		} else {
			shift &= 0x1f;
			cpssp->NAME.flag_shift_c = (val >> (shift - 1)) & 1;
			val = (val << (32 - shift)) | (val >> shift);
		}
		break;
	default:
		assert(0); /* Cannot happen. */
	}

	return val;
}

/*
 * Immediate
 * A5.1.3, page A5-6.
 */
static uint32_t
NAME_(imm)(struct cpssp *cpssp, uint32_t insn)
{
	uint32_t val;
	uint8_t shift;

	shift = ((insn >> 8) & 0xf) << 1;
	val = insn & 0xff;

	if (shift == 0) {
		val = val;
		cpssp->NAME.flag_shift_c = cpssp->NAME.flag_c;
	} else {
		val = (val << (32 - shift)) | (val >> shift);
		cpssp->NAME.flag_shift_c = (val >> 31) & 1;
	}

	return val;
}

/*
 * Immediate Offset
 * A5.2.2, page A5-20.
 */
static uint32_t
NAME_(imm_offset)(struct cpssp *cpssp, uint32_t insn)
{
	return insn & 0xfff;
}

/*
 * Scaled Register Offset
 * A5.2.4, page A5-22.
 */
static uint32_t
NAME_(reg_offset)(struct cpssp *cpssp, uint32_t insn)
{
	uint8_t shift;
	uint8_t op;
	unsigned int rm;
	uint32_t val;

	shift = (insn >> 7) & 0x1f;
	op = (insn >> 5) & 0x3;
	rm = (insn >> 0) & 0xf;

	val = NAME_(load_reg)(cpssp, rm);

	switch (op) {
	case 0x0: /* LSL */
		if (shift == 0) {
			/* Nothing to do... */
		} else {
			val <<= shift;
		}
		break;
	case 0x1: /* LSR */
		if (shift == 0) {
			val = 0x00000000;
		} else {
			val >>= shift;
		}
		break;
	case 0x2: /* ASR */
		if (shift == 0) {
			if ((val >> 31) & 1) {
				val = 0xffffffff;
			} else {
				val = 0x00000000;
			}
		} else {
			val = (uint32_t) ((int32_t) val >> shift);
		}
		break;
	case 0x3:
		if (shift == 0) {
			/* RRX */
			val = (cpssp->NAME.flag_c << 31) | (val >> 1);
		} else {
			/* ROR */
			val = (val << (32 - shift)) | (val >> shift);
		}
		break;
	default:
		assert(0);
	}

	return val;
}

/*
 * Instruction set encoding
 *
 * A3.1, A3-2
 * Figure A3-1, A3-2
 */
static void
NAME_(insn_exec)(struct cpssp *cpssp, uint32_t insn)
{
	uint8_t opcode;
	int p, u, b, i, w, l, s, h;
	uint8_t rn, rd, rm, rs;
	uint16_t reglist;
	uint32_t addr, offset;
	uint32_t op0, op1, op2, res;
	uint64_t res64;
	int c, n, v, z;

	if ((insn & 0xf0000000) == 0xf0000000) {
		/*
		 * Unconditional instructions.
		 *
		 * Figure A3-6, A3-42
		 */
#if 6 <= CONFIG_VERSION
		if ((insn & 0xfff10020) == 0xf1000000) {
			/*
			 * Change Processor State
			 *
			 * CPS, A4.1.16, A4-29
			 */
			uint8_t imod;
			uint8_t mmod;
			uint8_t a;
			uint8_t i;
			uint8_t f;
			uint8_t mode;

			assert((insn & 0x0000fe00) == 0x00000000);

			imod = (insn >> 18) & 3;
			mmod = (insn >> 17) & 1;
			a = (insn >> 8) & 1;
			i = (insn >> 7) & 1;
			f = (insn >> 6) & 1;
			mode = (insn >> 0) & 0b11111;

			assert(imod != 0b01); /* UNPREDICTABLE */
			assert(imod != 0b00 || mmod); /* UNPREDICTABLE */

			if ((imod >> 1) & 1) {
				/* Set A, I, F flags. */
				if (a) {
					assert(0); /* FIXME */
				}
				if (i) {
					NAME_(primask_set)(cpssp, imod & 1);
				}
				if (f) {
					assert(0); /* FIXME */
				}
			}
			if (mmod) {
				/* Set mode. */
				assert(0); /* FIXME */
			}
		} else
#endif /* 6 <= CONFIG_VERSION */

#if 6 <= CONFIG_VERSION
		if ((insn & 0xffff00f0) == 0xf1010000) {
			/*
			 * Set Endianness
			 *
			 * SETEND
			 */
			assert(0); /* FIXME */
		} else
#endif /* 6 <= CONFIG_VERSION */

#if 6 <= CONFIG_VERSION
		if ((insn & 0xfd70f000) == 0xf550f000) {
			/*
			 * Cache Preload
			 *
			 * PLD
			 */
			assert(0); /* FIXME */
		} else
#endif /* 6 <= CONFIG_VERSION */

#if 6 <= CONFIG_VERSION
		if ((insn & 0xfe5f0f00) == 0xf84d0500) {
			/*
			 * Save Return State
			 *
			 * SRS
			 */
			assert(0); /* FIXME */
		} else
#endif /* 6 <= CONFIG_VERSION */

#if 6 <= CONFIG_VERSION
		if ((insn & 0xfe500f00) == 0xf8100a00) {
			/*
			 * Return From Exception
			 *
			 * SRS
			 */
			assert(0); /* FIXME */
		} else
#endif /* 6 <= CONFIG_VERSION */

#if 5 <= CONFIG_VERSION
		if ((insn & 0xfe000000) == 0xfa000000) {
			/*
			 * Branch with Link and change to Thumb
			 *
			 * BLX
			 */
			assert(0); /* FIXME */
		} else
#endif /* 5 <= CONFIG_VERSION */

#if 6 <= CONFIG_VERSION
		/*
		 * MCRR2
		 * MRRC2
		 */
		/* FIXME */
#endif /* 6 <= CONFIG_VERSION */

#if 5 <= CONFIG_VERSION
		/*
		 * STC2
		 * LDC2
		 * CDP2
		 * MCR2
		 * MRC2
		 */
		/* FIXME */
#endif /* 5 <= CONFIG_VERSION */

		{
			assert(0); /* FIXME */
		}

	} else if (NAME_(condition)(cpssp, (insn >> 28) & 0xf)) {
		/*
		 * NOTE:
		 * Be careful! Sequence order of "if" matters!!!
		 */
		if ((insn & 0x0f900010) == 0x01000000) {
			/*
			 * Miscellaneous instructions.
			 *
			 * Figure A3-4, A3-37
			 *
			 * MRS, A4.1.38, A4-74
			 * MSR, A4.1.39, A4-76
			 */
			if ((insn & 0x0fb000f0) == 0x01000000) {
				/* MRS, A4.1.38, A4-74 */
				assert((insn & 0x000f0f00) == 0x000f0000);

				rd = (insn >> 12) & 0xf;

				if ((insn >> 22) & 1) {
					/* MRS Rd, SPSR */
					res = NAME_(load_spsr)(cpssp);
				} else {
					/* MRS Rd, CPSR */
					res = NAME_(load_cpsr)(cpssp);
				}
				NAME_(store_reg)(cpssp, rd, res);

			} else if ((insn & 0x0fb000f0) == 0x01200000) {
				/* MSR, A4.1.39, A4-76 */
				uint32_t bytemask;

				assert((insn & 0x0000ff00) == 0x0000f000);
				assert((insn & 0x000f0000) != 0x00000000);

				rn = (insn >> 16) & 0xf;
				bytemask = 0x00000000;
				if ((rn >> 0) & 1) bytemask |= 0x000000ff;
				if ((rn >> 1) & 1) bytemask |= 0x0000ff00;
				if ((rn >> 2) & 1) bytemask |= 0x00ff0000;
				if ((rn >> 3) & 1) bytemask |= 0xff000000;

				if ((insn >> 22) & 1) {
					/* MSR SPSR, Rd */
					op0 = NAME_(load_spsr)(cpssp);
				} else {
					/* MSR CPSR, Rd */
					op0 = NAME_(load_cpsr)(cpssp);
				}
				if ((insn >> 25) & 1) {
					op1 = NAME_(imm)(cpssp, insn);
				} else {
					rm = (insn >> 0) & 0xf;
					op1 = NAME_(load_reg)(cpssp, rm);
				}
				res = (op0 & ~bytemask) | (op1 & bytemask);
				if ((insn >> 22) & 1) {
					NAME_(store_spsr)(cpssp, res);
				} else {
					NAME_(store_cpsr)(cpssp, res);
				}

			} else {
				assert(0); /* FIXME */
			}

		} else if ((insn & 0x0e000010) == 0x00000000) {
			/*
			 * Data processing immediate shift.
			 *
			 * Data-processing instructions, A3.4, A3-7
			 * Instruction encoding, A3.4.1, A3-8
			 * List of data-processing Instructions, A3.4.2, A3-9
			 * Addressing Mode 1 - Data-processing operands,
			 * A5.1, A5-2
			 * Encoding, A5.1.1, A5-3
			 */
			opcode = (insn >> 21) & 0xf;
			s = (insn >> 20) & 1;
			rn = (insn >> 16) & 0xf;
			rd = (insn >> 12) & 0xf;

			assert(! s || rd != 15); /* FIXME */

			op0 = NAME_(load_reg)(cpssp, rn);
			op1 = NAME_(imm_shift)(cpssp, insn);
			res = NAME_(alu)(cpssp, opcode,
					op0, op1, &c, &n, &v, &z);
			if (opcode < 0x8
			 || 0xc <= opcode) {
				NAME_(store_reg)(cpssp, rd, res);
			} else {
				assert(s);
			}
			if (s) {
				NAME_(store_flags)(cpssp, c, n, v, z);
			}

		} else if ((insn & 0x0f900090) == 0x01000010) {
			/*
			 * Miscellaneous instructions.
			 *
			 * Figure A3-4, A3-37
			 *
			 * BKPT, A4.1.7, A4-14
			 * BLX(2), A4.1.9, A4-16
			 * BX, A4.1.10, A4-20
			 * CLZ, A4.1.13, A4-25
			 */
#if (4 == CONFIG_VERSION && CONFIG_THUMB) || 5 <= CONFIG_VERSION
			if ((insn & 0x0ff000f0) == 0x01200010) {
				/*
				 * Branch/exchange instruction set Thumb
				 *
				 * BX, A4.1.10, A4-20
				 */
				assert((insn & 0x000fff00) == 0x000fff00);

				rm = (insn >> 0) & 0xf;
				op0 = NAME_(load_reg)(cpssp, rm);
				cpssp->NAME.flag_t = op0 & 1;
				NAME_(store_pc)(cpssp, op0 & ~1);

			} else
#endif /* (4 == CONFIG_VERSION && CONFIG_THUMB) || 5 <= CONFIG_VERSION */
#if 5 <= CONFIG_VERSION
			if ((insn & 0x0ff000f0) == 0x01600010) {
				/* CLZ, A4.1.13, A4-25 */
				assert((insn & 0x000f0f00) == 0x000f0f00);

				rd = (insn >> 12) & 0xf;
				rm = (insn >> 0) & 0xf;
				op0 = NAME_(load_reg)(cpssp, rm);
				res = NAME_(clz)(cpssp, op0);
				NAME_(store_reg)(cpssp, rd, res);

			} else
#endif /* 5 <= CONFIG_VERSION */
#if 5 <= CONFIG_VERSION
			if ((insn & 0x0ff000f0) == 0x01200030) {
				/*
				 * Branch and link/exchange instruction set Thumb
				 *
				 * BLX(2), A4.1.9, A4-16
				 */
				assert((insn & 0x000fff00) == 0x000fff00);

				rm = (insn >> 0) & 0xf;
				op0 = NAME_(load_reg)(cpssp, rm);
				NAME_(store_lr)(cpssp, cpssp->NAME.pc_next | cpssp->NAME.flag_t);
				cpssp->NAME.flag_t = op0 & 1;
				NAME_(store_pc)(cpssp, op0 & ~1);

			} else
#endif /* 5 <= CONFIG_VERSION */
#if 5 <= CONFIG_VERSION && CONFIG_E
			if ((insn & 0x0f9000f0) == 0x01000050) {
				/*
				 * Saturating add/subtract
				 */
				assert(0); /* FIXME */

			} else
#endif /* 5 <= CONFIG_VERSION && CONFIG_E */
#if 5 <= CONFIG_VERSION
			if ((insn & 0x0ff000f0) == 0x01200070) {
				/* BKPT, A4.1.7, A4-14 */
				assert(0);

			} else
#endif /* 5 <= CONFIG_VERSION */
#if 5 <= CONFIG_VERSION && CONFIG_E
			if ((insn & 0x0f9000f0) == 0x01000090) {
				/*
				 * Signed multiplies (type 2)
				 */
				assert(0); /* FIXME */

			} else
#endif /* 5 <= CONFIG_VERSION && CONFIG_E */
			{
				assert(0); /* FIXME */
			}

		} else if ((insn & 0x0e000090) == 0x00000010) {
			/*
			 * Data-processing register shift
			 *
			 * Data-processing instructions, A3.4, A3-7
			 * Instruction encoding, A3.4.1, A3-8
			 * List of data-processing Instructions, A3.4.2, A3-9
			 * Addressing Mode 1 - Data-processing operands,
			 * A5.1, A5-2
			 * Encoding, A5.1.1, A5-3
			 */
			opcode = (insn >> 21) & 0xf;
			s = (insn >> 20) & 1;
			rn = (insn >> 16) & 0xf;
			rd = (insn >> 12) & 0xf;

			assert(! s || rd != 15); /* FIXME */

			op0 = NAME_(load_reg)(cpssp, rn);
			op1 = NAME_(reg_shift)(cpssp, insn);
			res = NAME_(alu)(cpssp, opcode,
					op0, op1, &c, &n, &v, &z);
			if (opcode < 0x8
			 || 0xc <= opcode) {
				NAME_(store_reg)(cpssp, rd, res);
			} else {
				assert(s);
			}
			if (s) {
				NAME_(store_flags)(cpssp, c, n, v, z);
			}

		} else if ((insn & 0x0e000090) == 0x00000090) {
			/*
			 * Multplies/Extra load/stores
			 *
			 * Figure A3-3, A3-35
			 * Figure A3-5, A3-39
			 */
			if ((insn & 0x0fc000f0) == 0x00000090) {
				/*
				 * Figure A3-3, A3-35
				 *
				 * MUL, A4.1.40, A4-80
				 * MLA, A4.1.34, A4-66
				 */
				s = (insn >> 20) & 1;
				rd = (insn >> 16) & 0xf;
				rn = (insn >> 12) & 0xf;
				rs = (insn >> 8) & 0xf;
				rm = (insn >> 0) & 0xf;
				op0 = NAME_(load_reg)(cpssp, rm);
				op1 = NAME_(load_reg)(cpssp, rs);
				if ((insn >> 21) & 1) {
					op2 = NAME_(load_reg)(cpssp, rn);
				} else {
					assert(rn == 0x0);
					op2 = 0x00000000;
				}
				res = NAME_(mlas)(cpssp, op0, op1, op2,
						&c, &n, &v, &z);
				NAME_(store_reg)(cpssp, rd, res);
				if (s) {
					NAME_(store_flags)(cpssp, c, n, v, z);
				}

#if 6 <= CONFIG_VERSION
			} else if ((insn & 0x0ff000f0) == 0x00400090) {
				/*
				 * Figure A3-3, A3-35
				 *
				 * UMAAL
				 */
				assert(0); /* FIXME */
#endif /* 6 <= CONFIG_VERSION */
				
			} else if ((insn & 0x0f8000f0) == 0x00800090) {
				/*
				 * Figure A3-3, A3-35
				 *
				 * SMLAL, A4.1.76, A4-146
				 * SMULL, A4.1.87, A4-168
				 * UMLAL, A4.1.128, A4-249
				 * UMULL, A4.1.129, A4-251
				 */
				s = (insn >> 20) & 1;
				rd = (insn >> 16) & 0xf;
				rn = (insn >> 12) & 0xf;
				rs = (insn >> 8) & 0xf;
				rm = (insn >> 0) & 0xf;

				op0 = NAME_(load_reg)(cpssp, rm);
				op1 = NAME_(load_reg)(cpssp, rs);

				switch ((insn >> 21) & 0x3) {
				case 0:
					/* UMULL, A4.1.129, A4-251 */
					res64 = NAME_(umulls)(cpssp, op0, op1,
							&c, &n, &v, &z);
					break;
				case 1:
					/* UMLAL, A4.1.128, A4-249 */
					assert(0); /* FIXME */
				case 2:
					/* SMULL, A4.1.87, A4-168 */
					assert(0); /* FIXME */
				case 3:
					/* SMLAL, A4.1.76, A4-146 */
					assert(0); /* FIXME */
				default: assert(0); /* Cannot happen. */
				}

				NAME_(store_reg)(cpssp, rd,
						(uint32_t) (res64 >> 32));
				NAME_(store_reg)(cpssp, rn,
						(uint32_t) (res64 >> 0));
				if (s) {
					NAME_(store_flags)(cpssp, c, n, v, z);
				}

			} else if ((insn & 0x0fb000f0) == 0x01000090) {
				/*
				 * Extra Load/store instructions
				 *
				 * Figure A3-5, A3-39
				 * SWP
				 * SWPB
				 */
				assert(0); /* FIXME */

			} else
#if 6 <= CONFIG_VERSION
			if ((insn & 0x0fe000f0) == 0x01800090) {
				/*
				 * Extra Load/store instructions
				 *
				 * Figure A3-5, A3-39
				 * LDREX
				 * STREX
				 */
				assert(0); /* FIXME */

			} else
#endif /* 6 <= CONFIG_VERSION */

			if ((insn & 0x0e000090) == 0x00000090) {
				/*
				 * Addressing Mode 3, A5.3, A5-33
				 *
				 * LDRD, A4.1.26, A4-50
				 * LDRH, A4.1.28, A4-54
				 * LDRSB, A4.1.29, A4-56
				 * LDRSH, A4.1.30, A4-58
				 * STRD, A4.1.102, A4-199
				 * STRH, A4.1.104, A4-204
				 */
				p = (insn >> 24) & 1;
				u = (insn >> 23) & 1;
				i = (insn >> 22) & 1;
				w = (insn >> 21) & 1;
				l = (insn >> 20) & 1;
				s = (insn >> 6) & 1;
				h = (insn >> 5) & 1;
				rn = (insn >> 16) & 0xf;
				rd = (insn >> 12) & 0xf;

				assert(p || ! w);
				assert(rd != 0xf);

				if (i) {
					offset = (((insn >> 8) & 0xf) << 4)
							| ((insn >> 0) & 0xf);
				} else {
					rm = (insn >> 0) & 0xf;
					offset = NAME_(load_reg)(cpssp, rm);
				}
				addr = NAME_(load_reg)(cpssp, rn);
				if (p) {
					if (u) {
						addr += offset;
					} else {
						addr -= offset;
					}
				}

				switch ((l << 2) | (s << 1) | (h << 0)) {
				case 0b000:
					assert(0); /* See above. */
				case 0b001:
					/* STRH, A4.1.104, A4-204 */
					op0 = NAME_(load_reg)(cpssp, rd);
					NAME_(st16)(cpssp, addr, op0);
					break;
#if 5 <= CONFIG_VERSION && CONFIG_E
				case 0b010:
					/* LDRD, A4.1.26, A4-50 */
					res64 = NAME_(ld64)(cpssp, addr);
					NAME_(store_reg64)(cpssp, rd, res64);
					break;
				case 0b011:
					/* STRD, A4.1.102, A4-199 */
					res64 = NAME_(load_reg64)(cpssp, rd);
					NAME_(st64)(cpssp, addr, res64);
					break;
#endif /* 5 <= CONFIG_VERSION && CONFIG_E */
				case 0b100:
					assert(0); /* See above. */
				case 0b101:
					/* LDRH, A4.1.28, A4-54 */
					res = (uint32_t) (uint16_t) NAME_(ld16)(cpssp, addr);
					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 0b110:
					/* LDRSB, A4.1.29, A4-56 */
					res = (uint32_t) (int32_t) (int8_t) NAME_(ld8)(cpssp, addr);
					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 0b111:
					/* LDRSH, A4.1.30, A4-58 */
					res = (uint32_t) (int32_t) (int16_t) NAME_(ld16)(cpssp, addr);
					NAME_(store_reg)(cpssp, rd, res);
					break;
				default:
					assert(0); /* Cannot happen. */
				}

				if (! p) {
					if (u) {
						addr += offset;
					} else {
						addr -= offset;
					}
				}
				if (! p || w) {
					NAME_(store_reg)(cpssp, rn, addr);
				}

			} else {
				assert(0); /* FIXME */
			}

		} else if ((insn & 0x0fb00000) == 0x03000000) {
			/*
			 * Undefined instruction.
			 */
			assert(0); /* FIXME */

		} else if ((insn & 0x0fb00000) == 0x03200000) {
			/*
			 * Move immediate to status register.
			 */
			assert(0); /* FIXME */

		} else if ((insn & 0x0e000000) == 0x02000000) {
			/*
			 * Data processing immediate
			 *
			 * Data-processing instructions, A3.4, A3-7
			 * Instruction encoding, A3.4.1, A3-8
			 * List of data-processing Instructions, A3.4.2, A3-9
			 * Addressing Mode 1 - Data-processing operands,
			 * A5.1, A5-2
			 * Encoding, A5.1.1, A5-3
			 */
			opcode = (insn >> 21) & 0xf;
			s = (insn >> 20) & 1;
			rn = (insn >> 16) & 0xf;
			rd = (insn >> 12) & 0xf;

			assert(! s || rd != 15); /* FIXME */

			op0 = NAME_(load_reg)(cpssp, rn);
			if (rn == 15
			 && cpssp->NAME.flag_t) {
				op0 &= ~3;
			}
			op1 = NAME_(imm)(cpssp, insn);

			res = NAME_(alu)(cpssp, opcode,
					op0, op1, &c, &n, &v, &z);
			if (opcode < 0x8
			 || 0xc <= opcode) {
				NAME_(store_reg)(cpssp, rd, res);
			} else {
				assert(s);
			}
			if (s) {
				NAME_(store_flags)(cpssp, c, n, v, z);
			}

		} else if ((insn & 0x0e000000) == 0x04000000) {
			/*
			 * Load/store immediate offset.
			 *
			 * A5.2.2, page A5-20.
			 */
			p = (insn >> 24) & 1;
			u = (insn >> 23) & 1;
			b = (insn >> 22) & 1;
			w = (insn >> 21) & 1;
			l = (insn >> 20) & 1;
			rn = (insn >> 16) & 0xf;
			rd = (insn >> 12) & 0xf;
			if (! p || w) {
				assert(rn != rd); /* UNPREDICTABLE */
			}
			offset = NAME_(imm_offset)(cpssp, insn);
			addr = NAME_(addr_mode_2_pre)(cpssp, p, u, w, rn, offset);
			NAME_(ldst)(cpssp, p, b, w, l, rd, addr);
			NAME_(addr_mode_2_post)(cpssp, p, u, w, rn, addr, offset);

		} else if ((insn & 0x0e000010) == 0x06000000) {
			/*
			 * Load/store register offset.
			 *
			 * A5.2.4, page A5-22.
			 */
			p = (insn >> 24) & 1;
			u = (insn >> 23) & 1;
			b = (insn >> 22) & 1;
			w = (insn >> 21) & 1;
			l = (insn >> 20) & 1;
			rn = (insn >> 16) & 0xf;
			rd = (insn >> 12) & 0xf;
			if (! p || w) {
				assert(rn != rd); /* UNPREDICTABLE */
			}
			offset = NAME_(reg_offset)(cpssp, insn);
			addr = NAME_(addr_mode_2_pre)(cpssp, p, u, w, rn, offset);
			NAME_(ldst)(cpssp, p, b, w, l, rd, addr);
			NAME_(addr_mode_2_post)(cpssp, p, u, w, rn, addr, offset);

		} else if ((insn & 0x0ff000f0) == 0x07f000f0) {
			/*
			 * Architecturally undefined.
			 *
			 * A3.16.5
			 */
			assert(0); /* FIXME */

#if 6 <= CONFIG_VERSION
		} else if ((insn & 0x0e000010) == 0x06000010) {
			/*
			 * Media instructions.
			 *
			 * Figure A3-2, A3-34
			 */
			if ((insn & 0x0f800010) == 0x06000010) {
				/*
				 * Parallel add/subtract
				 */
				assert((insn & 0x00000f00) == 0x00000f00);

				assert(0); /* FIXME */

			} else if ((insn & 0x0ff00030) == 0x06800010) {
				/*
				 * Halfword pack
				 */
				assert(0); /* FIXME */

			} else if ((insn & 0x0fa00030) == 0x06a00010) {
				/*
				 * Word saturate
				 */
				assert(0); /* FIXME */

			} else if ((insn & 0x0fb000f0) == 0x06a00030) {
				/*
				 * Parallel halfword saturate
				 */
				assert(0); /* FIXME */

			} else if ((insn & 0x0fb00070) == 0x06b00030) {
				/*
				 * Reverse
				 *
				 * REV, A4.1.56, A4-109
				 * REV16, A4.1.57, A4-110
				 * REVSH, A4.1.58, A4-111
				 */
				assert((insn & 0x000f0f00) == 0x000f0f00);

				rd = (insn >> 12) & 0xf;
				rm = (insn >> 0) & 0xf;

				assert(rd != 15); /* UNPREDICTABLE */
				assert(rm != 15); /* UNPREDICTABLE */

				op0 = NAME_(load_reg)(cpssp, rm);

				if ((insn & 0x0ff000f0) == 0x06b00030) {
					/* REV, A4.1.56, A4-109 */
					res = ((op0 >> 24) & 0xff)
						| ((op0 >> 8) & 0xff00)
						| ((op0 << 8) & 0xff0000)
						| ((op0 << 24) & 0xff000000);

				} else if ((insn & 0x0ff000f0) == 0x06b000b0) {
					/* REV16, A4.1.57, A4-110 */
					res = ((op0 >> 8) & 0xff0000)
						| ((op0 << 8) & 0xff000000)
						| ((op0 >> 8) & 0xff)
						| ((op0 << 8) & 0xff00);

				} else if ((insn & 0x0ff000f0) == 0x06f000b0) {
					/* REVSH, A4.1.58, A4-111 */
					op0 = ((op0 >> 8) & 0xff)
						| ((op0 << 8) & 0xff00);
					res = (int32_t) (int16_t) op0;

				} else {
					assert(0); /* FIXME */
				}

				NAME_(store_reg)(cpssp, rd, res);

			} else if ((insn & 0x0ff000f0) == 0x068000b0) {
				/*
				 * Select bytes
				 */
				assert((insn & 0x00000f00) == 0x00000f00);

				assert(0); /* FIXME */

			} else if ((insn & 0x0f8000f0) == 0x06800070) {
				/*
				 * Sign/zero extend (add)
				 *
				 * SXTAB, A4.1.110, A4-216
				 * SXTB, A4.1.113, A4-222
				 * SXTAB16, A4.1.111, A4-218
				 * SXTB16, A4.1.114, A4-224
				 * SXTAH, A4.1.112, A4-220
				 * SXTH, A4.1.115, A4-226
				 * UXTAB, A4.1.143, A4-274
				 * UXTAH, A4.1.145, A4-278
				 * UXTB, A4.1.146, A4-280
				 * UXTAB16, A4.1.144, A4-276
				 * UXTB16, A4.1.147, A4-282
				 * UXTH, A4.1.148, A4-284
				 */
				unsigned int rotate;

				assert((insn & 0x00000300) == 0x00000000);

				rn = (insn >> 16) & 0xf;
				rd = (insn >> 12) & 0xf;
				rotate = ((insn >> 10) & 3) << 3;
				rm = (insn >> 0) & 0xf;

				assert(rd != 15); /* UNPREDICTABLE */
				assert(rm != 15); /* UNPREDICTABLE */

				op0 = NAME_(load_reg)(cpssp, rm);
				op0 = (op0 >> rotate) | (op0 << (32 - rotate));
				if (rn == 15) {
					op1 = 0;
				} else {
					op1 = NAME_(load_reg)(cpssp, rn);
				}
				switch ((insn >> 20) & 0x7) {
				case 0:
					/* SXTAB16, A4.1.111, A4-218 */
					/* SXTB16, A4.1.114, A4-224 */
					assert(0); /* FIXME */

				case 1:
					assert(0); /* FIXME */

				case 2:
					/* SXTAB, A4.1.110, A4-216 */
					/* SXTB, A4.1.113, A4-222 */
					res = op1 + (int32_t) (int8_t) op0;
					break;

				case 3:
					/* SXTAH, A4.1.112, A4-220 */
					/* SXTH, A4.1.115, A4-226 */
					res = op1 + (int32_t) (int16_t) op0;
					break;

				case 4:
					/* UXTAB16, A4.1.144, A4-276 */
					/* UXTB16, A4.1.147, A4-282 */
					assert(0); /* FIXME */

				case 5:
					assert(0); /* FIXME */

				case 6:
					/* UXTAB, A4.1.143, A4-274 */
					/* UXTB, A4.1.146, A4-280 */
					res = op1 + (uint32_t) (uint8_t) op0;
					break;

				case 7:
					/* UXTAH, A4.1.145, A4-278 */
					/* UXTH, A4.1.148, A4-284 */
					res = op1 + (uint32_t) (uint16_t) op0;
					break;

				default: assert(0); /* Cannot happen. */
				}
				NAME_(store_reg)(cpssp, rd, res);

			} else if ((insn & 0x0f800010) == 0x07000010) {
				/*
				 * Multiplies (type 3)
				 */
				assert(0); /* FIXME */

			} else if ((insn & 0x0ff000f0) == 0x07800010) {
				/*
				 * Unsigned sum of abs differences
				 * Unsigned sum of abs differences, acc
				 */
				assert(0); /* FIXME */

			} else {
				assert(0); /* FIXME */
			}
#endif /* 6 <= CONFIG_VERSION */

		} else if ((insn & 0x0e000000) == 0x08000000) {
			/*
			 * Load/store multiple
			 *
			 * Addressing Mode 4 - Load and Store Multiple,
			 * A5.4, A5-41
			 * Encoding, A5.4.1, A5-42
			 *
			 * LDM(1), A4.1.20, A4-36
			 * LDM(2), A4.1.21, A4-38
			 * LDM(3), A4.1.22, A4-40
			 * STM(1), A4.1.97, A4-189
			 * STM(2), A4.1.98, A4-191
			 */
			p = (insn >> 24) & 1;
			u = (insn >> 23) & 1;
			s = (insn >> 22) & 1;
			w = (insn >> 21) & 1;
			l = (insn >> 20) & 1;
			rn = (insn >> 16) & 0xf;
			reglist = (insn >> 0) & 0xffff;
			assert(! l || ! w || ! ((reglist >> rn) & 1)); /* UNPREDICTABLE */
			addr = NAME_(addr_mode_4_pre)(cpssp, rn);
			addr = NAME_(ldstm)(cpssp, p, u, s, l, addr, reglist);
			NAME_(addr_mode_4_post)(cpssp, w, rn, addr);

		} else if ((insn & 0x0e000000) == 0x0a000000) {
			/*
			 * Branch and branch with link.
			 * B, BL, A4.1.5, A4-10
			 */
			int32_t offset;

			if ((insn >> 24) & 1) {
				NAME_(store_lr)(cpssp, cpssp->NAME.pc_next | cpssp->NAME.flag_t);
			}

			offset = (int32_t) ((insn >> 0) & 0xffffff);
			offset <<= 8;
			offset >>= 8;
			if (cpssp->NAME.flag_t) {
				offset <<= 1;
			} else {
				offset <<= 2;
			}
			NAME_(store_pc)(cpssp, NAME_(load_pc)(cpssp) + offset);

#if CONFIG_ARM
		} else if ((insn & 0x0e000000) == 0x0c000000) {
			/*
			 * Coprocessor load/store and double register transfers.
			 */
			assert(0); /* FIXME */

		} else if ((insn & 0x0f000010) == 0x0e000000) {
			/*
			 * Coprocessor data processing.
			 */
			assert(0); /* FIXME */

		} else if ((insn & 0x0f000010) == 0x0e000010) {
			/*
			 * Coprocessor register transfers.
			 */
			if ((insn & 0x0f100010) == 0x0e000010) {
				/* MCR, A4.1.32, page A4-62. */
				rd = (insn >> 12) & 0xf;

				op0 = NAME_(load_reg)(cpssp, rd);
				NAME_(mcr)(cpssp, insn, op0);

			} else if ((insn & 0x0f100010) == 0x0e100010) {
				/* MRC, A4.1.36, page A4-70. */
				rd = (insn >> 12) & 0xf;

				NAME_(mrc)(cpssp, insn, &res);
				if (rd == 15) {
					assert(0); /* Store to flags */
				} else {
					NAME_(store_reg)(cpssp, rd, res);
				}
			} else {
				assert(0); /* FIXME */
			}
#endif /* CONFIG_ARM */

		} else if ((insn & 0x0f000000) == 0x0f000000) {
			/*
			 * Software interrupt.
			 *
			 * SWI, A4.1.107, A4-210
			 */
			NAME_(SVCall_pending_set)(cpssp, 1);
			assert(cpssp->NAME.exception_pending);

		} else {
			/* FIXME */
			fprintf(stderr, "insn=0x%08x\n", insn);
			assert(0);
		}
	}
}

#if CONFIG_ARM
static void
NAME_(insn_exec_arm)(struct cpssp *cpssp)
{
	cpssp->NAME.pc = cpssp->NAME.pc_next;
	cpssp->NAME.pc_next = cpssp->NAME.pc_next_next;
	cpssp->NAME.pc_next_next += 4;

	cpssp->NAME.insn = cpssp->NAME.insn_next;
	cpssp->NAME.insn_next = NAME_(ld32_code)(cpssp, cpssp->NAME.pc_next);

	if (cpssp->NAME.flush) {
		/*
		 * Flush pipeline after branch taken.
		 */
		/* Just wait... */
		cpssp->NAME.flush = 0;
		return;
	}

#if 0
	fprintf(stderr, "DIS %08x %08x\n", cpssp->NAME.pc, cpssp->NAME.insn);
#endif

	NAME_(insn_exec)(cpssp, cpssp->NAME.insn);
}
#endif /* CONFIG_ARM */

#if CONFIG_THUMB
static void
NAME_(insn_exec_thumb2)(struct cpssp *cpssp, uint32_t insn)
{
	uint32_t offset;
	uint32_t addr;
#if CONFIG_M
	uint32_t op0, op1, res;
	unsigned int op;
	uint64_t res64;
	uint32_t tmp32, shift;
	uint16_t reglist;
	uint8_t rd, rdlo, rdhi, rn, rm, rt;
	int p, u, l, w;
	int c, n, v, z;
#endif
	int s;

	/*
	 * 32-bit Thumb instruction encoding
	 * ARMv7-M Architecture Reference Manual, A5.3, A5-135
	 */
	switch ((insn >> 27) & 0b11) {
	case 0b00:
		/* Cannot happen. 16-bit instruction. */
		assert(0);

	case 0b01:
#if CONFIG_M
		switch ((insn >> 25) & 3) {
		case 0:
			if (! ((insn >> 22) & 1)) {
				/*
				 * Load/store multiple
				 * A5-142
				 *
				 * STM/STMIA/STMEA (T2), A7-422
				 * LDM/LDMIA/LDMFD (T2), A7-248
				 * POP (T2), A7-348
				 * STMDB/STMFD (T1), A7-424
				 * PUSH (T2), A7-350
				 * LDMDB/LDMEA (T1), A7-250
				 */
				p = (insn >> 24) & 1;
				u = (insn >> 23) & 1;
				s = 0;
				w = (insn >> 21) & 1;
				l = (insn >> 20) & 1;
				rn = (insn >> 16) & 0xf;
				reglist = (insn >> 0) & 0xffff;

				addr = NAME_(load_reg)(cpssp, rn);
				addr = NAME_(ldstm)(cpssp, p, u, s, l, addr, reglist);
				if (w) {
					NAME_(store_reg)(cpssp, rn, addr);
				}

			} else {
				/*
				 * Load/store dual or exclusive, table branch
				 * A5-143
				 */
				assert(0); /* FIXME */
			}
			break;
		case 1:
			/*
			 * Data processing (shifted register)
			 * ARMv7-M: A5-148
			 */
			s = (insn >> 20) & 1;
			rn = (insn >> 16) & 0xf;
			rd = (insn >> 8) & 0xf;
			rm = (insn >> 0) & 0xf;
			shift = ((insn >> 12) & 0x7) << 2;
			shift |= ((insn >> 6) & 0x3) << 0;
			op = (insn >> 4) & 0x3;

			op0 = NAME_(shift)(cpssp, rm, op, shift);
			op1 = NAME_(load_reg)(cpssp, rn);

			switch ((insn >> 21) & 0b1111) {
			case 0b0000:
				/*
				 * AND (register), ARMv7-M: A7-201
				 * TST (register), ARMv7-M: A7-466
				 */
				res = NAME_(ands)(cpssp, op0, op1, &c, &v);
				n = (res >> 31) & 1;
				z = res == 0;
				if (s) {
					NAME_(store_flags)(cpssp, c, n, v, z);
				}
				if (rd != 0xf) {
					NAME_(store_reg)(cpssp, rd, res);
				} else {
					assert(s);
				}
				break;
			case 0b0001:
				assert(0); /* FIXME */
				break;
			case 0b0010:
				/*
				 * ORR (register), ARMv7-M: A7-333
				 * MOV (register), ARMv7-M: A7-314
				 * LSL (immediate), ARMv7-M: A7-298
				 * LSR (immediate), ARMv7-M: A7-302
				 * ASR (immediate), ARMv7-M: A7-203
				 * ROR (immediate), ARMv7-M: A7-366
				 * RRX, ARMv7-M: A7-370
				 */
				if (rn == 0xf) {
					op1 = 0;
				}

				res = NAME_(orrs)(cpssp, op0, op1, &c, &v);
				n = (res >> 31) & 1;
				z = res == 0;
				if (s) {
					NAME_(store_flags)(cpssp, c, n, v, z);
				}
				NAME_(store_reg)(cpssp, rd, res);
				break;
			case 0b0011:
				assert(0); /* FIXME */
				break;
			case 0b0100:
				assert(0); /* FIXME */
				break;
			case 0b0110:
				/*
				 * PKHBT, ARMv7-M: A7-338
				 * PKHTB, ARMv7-M: A7-338
				 */
				if (! ((insn >> 5) & 1)) {
					res = (op0 & 0xffff0000) | (op1 & 0x0000ffff);
				} else {
					res = (op1 & 0xffff0000) | (op0 & 0x0000ffff);
				}
				if (s) {
					assert(0); /* UNDEFINED */
				}
				NAME_(store_reg)(cpssp, rd, res);
				break;
#if 7 <= CONFIG_VERSION
			case 0b1000:
				if (rd != 0xf) {
					/*
					 * ADD (register)
					 * ARMv7-M: A7-191
					 */
					res = NAME_(adcs)(cpssp, op0, op1, 0, &c, &v);
					n = (res >> 31) & 1;
					z = res == 0;
					if (s) {
						NAME_(store_flags)(cpssp, c, n, v, z);
					}
					NAME_(store_reg)(cpssp, rd, res);

				} else {
					assert(0); /* FIXME */
				}
				break;
#endif
			case 0b1010:
				assert(0); /* FIXME */
				break;
			case 0b1011:
				assert(0); /* FIXME */
				break;
			case 0b1101:
				assert(0); /* FIXME */
				break;
			case 0b1110:
				assert(0); /* FIXME */
				break;
			default:
				assert(0); /* UNDEFINED */
			}
			break;
		case 2:
		case 3:
#if 7 <= CONFIG_VERSION
			/*
			 * Coprocessor instructions
			 * A5-156
			 */
			fprintf(stderr, "insn=0x%08x\n", insn);
			if (((insn >> 20) & 0b100001) == 0b000000
			 && ((insn >> 20) & 0b111010) != 0b000000) {
				/*
				 * STC, STC2
				 * ARMv7: A7-420
				 */
				assert(0); /* FIXME */

			} else if (((insn >> 20) & 0b100001) == 0b000001
				&& ((insn >> 20) & 0b111010) != 0b000000) {
				/*
				 * LDC, LDC2 (immediate)
				 * ARMv7: A7-244
				 * LDC, LDC2 (literal)
				 * ARMv7: A7-246
				 */
				assert(0); /* FIXME */

			} else if (((insn >> 20) & 0b111111) == 0b000100) {
				/*
				 * MCRR, MCRR2
				 * ARMv7: A7-308
				 */
				assert(0); /* FIXME */

			} else if (((insn >> 20) & 0b111111) == 0b000101) {
				/*
				 * MRR, MRRC2
				 * ARMv7: A7-320
				 */
				assert(0); /* FIXME */

			} else if (((insn >> 20) & 0b110000) == 0b100000
				&& ((insn >> 4) & 1) == 0) {
				/*
				 * CDP, CDP2
				 * ARMv7: A7-221
				 */
				assert(0); /* FIXME */

			} else if (((insn >> 20) & 0b110001) == 0b100000
				&& ((insn >> 4) & 1) == 1) {
				/*
				 * MCR, MCR2
				 * ARMv7: A7-306
				 */
				rt = (insn >> 12) & 0xf;

				assert(rt != 0xd); /* UNPREDICTABLE */
				assert(rt != 0xf); /* UNPREDICTABLE */

				op0 = NAME_(load_reg)(cpssp, rt);

				NAME_(mcr)(cpssp, insn, op0);

			} else if (((insn >> 20) & 0b110001) == 0b100001
				&& ((insn >> 4) & 1) == 1) {
				/*
				 * MRC, MRC2
				 * ARMv7: A7-318
				 */
				rt = (insn >> 12) & 0xf;

				assert(rt != 0xd); /* UNPREDICTABLE */
				assert(rt != 0xf); /* UNPREDICTABLE */

				NAME_(mrc)(cpssp, insn, &res);

				NAME_(store_reg)(cpssp, rt, res);

			} else {
				/*
				 * UNDEFINED
				 */
				assert(0); /* FIXME */
			}
#else /* CONFIG_VERSION < 7 */
			assert(0); /* UNDEFINED */
#endif /* CONFIG_VERSION < 7 */
			break;
		default:
			assert(0); /* Cannot happen. */
		}
#else /* ! CONFIG_M */
		assert(0); /* UNDEFINED */
#endif /* ! CONFIG_M */
		break;

	case 0b10:
		if (! ((insn >> 15) & 1)) {
#if CONFIG_M
			if (! ((insn >> 25) & 1)) {
				/*
				 * Data processing (modified immediate)
				 * ARMv7-M: A5-136
				 */
				int i;
				uint8_t imm3;
				uint8_t abcdefgh;

				i = (insn >> 26) & 1;
				s = (insn >> 20) & 1;
				rn = (insn >> 16) & 0xf;
				imm3 = (insn >> 12) & 0x7;
				rd = (insn >> 8) & 0xf;
				abcdefgh = (insn >> 0) & 0xff;

				op0 = NAME_(load_reg)(cpssp, rn);
				if (((i << 1) | (imm3 >> 2)) == 0) {
					switch (imm3 & 3) {
					case 0:
						op1 = abcdefgh;
						break;
					case 1:
						assert(abcdefgh); /* UNPREDICTABLE */
						op1 = (abcdefgh << 16) | (abcdefgh << 0);
						break;
					case 2:
						assert(abcdefgh); /* UNPREDICTABLE */
						op1 = (abcdefgh << 24) | (abcdefgh << 8);
						break;
					case 3:
						assert(abcdefgh); /* UNPREDICTABLE */
						op1 = (abcdefgh << 24) | (abcdefgh << 16) | (abcdefgh << 8) | (abcdefgh << 0);
						break;
					default:
						assert(0); /* Cannot happen. */
					}
				} else {
					unsigned int count;

					count = (i << 4) | (imm3 << 1) | ((abcdefgh >> 7) & 1);
					assert(8 <= count && count < 32);
					abcdefgh |= 1 << 7;
					op1 = abcdefgh << (32 - count);
				}

				switch ((insn >> 21) & 0xf) {
				case 0:
					if (rd != 0xf) {
						/* AND, ARMv7-M: A7-199 */
						res = NAME_(ands)(cpssp, op0, op1, &c, &v);
						NAME_(store_reg)(cpssp, rd, res);
					} else {
						/* TST, ARMv7-M: A7-465 */
						res = NAME_(ands)(cpssp, op0, op1, &c, &v);
					}
					break;
				case 1:
					/* BIC, ARMv7-M: A7-211 */
					res = NAME_(bics)(cpssp, op0, op1, &c, &v);
					n = (res >> 31) & 1;
					z = res == 0;
					if (s) {
						NAME_(store_flags)(cpssp, c, n, v, z);
					}
					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 2:
					if (rn != 0xf) {
						/* ORR, ARMv7-M: A7-334 */
						res = NAME_(orrs)(cpssp, op0, op1, &c, &v);
					} else {
						/* MOV, ARMv7-M: A7-312 */
						res = NAME_(movs)(cpssp, op0, op1, &c, &v);
					}
					n = (res >> 31) & 1;
					z = res == 0;
					if (s) {
						NAME_(store_flags)(cpssp, c, n, v, z);
					}
					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 3:
					if (rn != 0xf) {
						/* ORN, ARMv7-M: A7-332 */
						res = NAME_(orrs)(cpssp, op0, ~op1, &c, &v);
					} else {
						/* MVN, ARMv7-M: A7-326 */
						res = NAME_(movs)(cpssp, op0, ~op1, &c, &v);
					}
					n = (res >> 31) & 1;
					z = res == 0;
					if (s) {
						NAME_(store_flags)(cpssp, c, n, v, z);
					}
					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 4:
					if (rd != 0xf) {
						/* EOR, ARMv7-M: A7-238 */
						res = NAME_(eors)(cpssp, op0, op1, &c, &v);
						NAME_(store_reg)(cpssp, rd, res);
					} else {
						/* TEQ, ARMv7-M: A7-463 */
						res = NAME_(eors)(cpssp, op0, op1, &c, &v);
					}
					break;
				case 5:
					assert(0); /* UNDEFINED */
					break;
				case 6:
					assert(0); /* UNDEFINED */
					break;
				case 7:
					assert(0); /* UNDEFINED */
					break;
				case 8:
					if (rd != 0xf) {
						/* ADD, ARMv7-M: A7-189 */
						res = NAME_(adcs)(cpssp, op0, op1, 0, &c, &v);
						NAME_(store_reg)(cpssp, rd, res);
					} else {
						/* CMN, ARMv7-M: A7-225 */
						res = NAME_(adcs)(cpssp, op0, op1, 0, &c, &v);
					}
					break;
				case 9:
					assert(0); /* UNDEFINED */
					break;
				case 10:
					/* ADC, ARMv7-M: A7-185 */
					res = NAME_(adcs)(cpssp, op0, op1, cpssp->NAME.flag_c, &c, &v);
					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 11:
					/* SBC, ARMv7-M: A7-379 */
					res = NAME_(sbcs)(cpssp, op0, op1, cpssp->NAME.flag_c, &c, &v);
					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 12:
					assert(0); /* UNDEFINED */
					break;
				case 13:
					if (rd != 0xf) {
						/* SUB, ARMv7-M: A7-448 */
						res = NAME_(sbcs)(cpssp, op0, op1, 1, &c, &v);
						NAME_(store_reg)(cpssp, rd, res);
					} else {
						/* CMP, ARMv7-M: A7-229 */
						res = NAME_(sbcs)(cpssp, op0, op1, 1, &c, &v);
					}
					break;
				case 14:
					/* RSB, ARMv7-M: A7-372 */
					res = NAME_(sbcs)(cpssp, op1, op0, 1, &c, &v);
					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 15:
					assert(0); /* UNDEFINED */
					break;
				default:
					assert(0); /* Cannot happen. */
				}
				n = (res >> 31) & 1;
				z = res == 0;
				if (s) {
					NAME_(store_flags)(cpssp, c, n, v, z);
				}

			} else {
				/*
				 * Data processing (plain binary immediate)
				 * ARMv7-M: A5-139
				 *
				 * ADD (immediate), ARMv7-M: A7-189
				 * ADR, ARMv7-M: A7-197
				 * BFC, ARMv7-M: A7-209
				 * BFI, ARMv7-M: A7-210
				 * MOV (immediate), ARMv7-M: A7-312
				 * MOVT, ARMv7-M: A7-317
				 * SUB (immediate), ARMv7-M: A7-448
				 * ...
				 * UBFX, ARMv7-M: A7-470
				 */
#if CONFIG_VERSION <= 7
				uint32_t op2;
				uint8_t lsb;
				uint8_t msb;
#endif

				rn = (insn >> 16) & 0xf;
				rd = (insn >> 8) & 0xf;
				offset = ((insn >> 26) & 1) << 11;
				offset |= ((insn >> 12) & 0x7) << 8;
				offset |= (insn >> 0) & 0xff;

				switch ((insn >> 20) & 0b11111) {
				case 0b00000:
					if (rn != 0xf) {
						/* ADD (immediate), A6-22 */
						assert(0); /* FIXME */
					} else {
						/* ADR (T3), A6-30 */
						assert(rd != 0xd); /* UNPREDICTABLE */
						assert(rd != 0xf); /* UNPREDICTABLE */

						assert(0); /* FIXME */
					}
					break;
				case 0b00100:
					/* MOV (immediate), A6-148 */
					assert(rd != 0xd); /* UNPREDICTABLE */
					assert(rd != 0xf); /* UNPREDICTABLE */

					offset |= rn << 12;

					NAME_(store_reg)(cpssp, rd, offset);
					break;
				case 0b01010:
					if (rn != 0xf) {
						/* SUB (immediate), A6-244 */
						assert(0); /* FIXME */
					} else {
						/* ADR, A6-30 */
						assert(0); /* FIXME */
					}
					break;
				case 0b01100:
					/* MOVT, A6-153 */
					assert(rd != 0xd); /* UNPREDICTABLE */
					assert(rd != 0xf); /* UNPREDICTABLE */

					offset |= rn << 12;

					res = NAME_(load_reg)(cpssp, rd);
					res &= 0x0000ffff;
					res |= offset << 16;
					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 0b10000:
					assert(0); /* FIXME */
				case 0b10010:
					assert(0); /* FIXME */
				case 0b10100:
					assert(0); /* FIXME */
#if CONFIG_VERSION <= 7
				case 0b10110:
					/*
					 * BFC, ARMv7-M: A7-209
					 * BFI, ARMv7-M: A7-210
					 */
					lsb = (((insn >> 12) & 0x7) << 2)
						| (((insn >> 6) & 0x3) << 0);
					msb = (insn >> 0) & 0x1f;

					assert(rd != 0xd); /* UNPREDICTABLE */
					assert(rd != 0xf); /* UNPREDICTABLE */
					assert(rn != 0xd); /* UNPREDICTABLE */
					assert(lsb <= msb); /* UNPREDICTABLE */

					op0 = NAME_(load_reg)(cpssp, rd);
					if (rn != 0xf) {
						op1 = NAME_(load_reg)(cpssp, rn);
					} else {
						op1 = 0x00000000;
					}

					res = NAME_(bfi)(cpssp, op0, op1, msb, lsb);

					NAME_(store_reg)(cpssp, rd, res);
					break;
#endif /* CONFIG_VERSION <= 7 */
				case 0b11000:
					assert(0); /* FIXME */
				case 0b11010:
					assert(0); /* FIXME */
#if CONFIG_VERSION <= 7
				case 0b11100:
					/* UBFX, ARMv7-M: A7-470 */
					assert(rn != 0xd); /* UNPREDICTABLE */
					assert(rn != 0xf); /* UNPREDICTABLE */
					assert(rd != 0xd); /* UNPREDICTABLE */
					assert(rd != 0xf); /* UNPREDICTABLE */

					op0 = NAME_(load_reg)(cpssp, rn);
					op1 = ((insn >> 12) & 0x7) << 2;
					op1 |= (insn >> 6) & 0x3;
					op2 = ((insn >> 0) & 0x1f) + 1;
					assert(op2 + op1 < 32); /* UNPREDICTABLE */

					res = op0 >> op1;
					res &= (1 << op2) - 1;

					NAME_(store_reg)(cpssp, rd, res);
					break;
#endif /* CONFIG_VERSION <= 7 */
				default:
					assert(0); /* UNDEFINED */
				}
			}
#else /* ! CONFIG_M */
			assert(0); /* UNDEFINED */
#endif /* ! CONFIG_M */

		} else {
			/*
			 * Branch and miscellaneous control
			 * ARMv7: A5-140
			 */
			uint8_t op1, op2;

			op1 = (insn >> 20) & 0x7f;
			op2 = (insn >> 12) & 0x7;

			if ((op2 & 0b101) == 0b000
			 && (op1 & 0b0111000) != 0b0111000) {
				/*
				 * Conditional branch (B) (T3)
				 * ARMv7-M: A7-207
				 */
				int j1, j2;
				uint8_t cond;

				s = (insn >> 26) & 1;
				j1 = (insn >> 13) & 1;
				j2 = (insn >> 11) & 1;
				cond = (insn >> 22) & 0xf;

				assert(cond != 0xe);

				offset = (s << 20)
					| (j2 << 19)
					| (j1 << 18)
					| (((insn >> 16) & 0x3f) << 12)
					| (((insn >> 0) & 0x7ff) << 1)
					| (0 << 0);
				offset = ((int32_t) offset << 11) >> 11;

				if (NAME_(condition)(cpssp, cond)) {
					addr = cpssp->NAME.pc_next + offset;
					NAME_(store_pc)(cpssp, addr);
				}

			} else if ((op2 & 0b101) == 0b000
				&& (op1 & 0b1111110) == 0b0111000) {
#if CONFIG_M
				/*
				 * MSR (register)
				 * ARMv7-M, A6-159
				 */
				rn = (insn >> 16) & 0xf;
				tmp32 = NAME_(load_reg)(cpssp, rn);

				NAME_(store_spec)(cpssp, (insn >> 0) & 0xff, tmp32);
#else /* ! CONFIG_M */
				assert(0); /* UNDEFINED */
#endif /* ! CONFIG_M */

			} else if ((op2 & 0b101) == 0b000
				&& op1 == 0b0111010) {
				/*
				 * Hint instructions
				 * ARMv7: A5-19
				 */
#if CONFIG_M
				assert(0); /* FIXME */
#else /* ! CONFIG_M */
				assert(0); /* UNDEFINED */
#endif /* ! CONFIG_M */
				break;

			} else if ((op2 & 0b101) == 0b000
				&& op1 == 0b0111011) {
#if CONFIG_M
				/*
				 * Miscellaneous control instructions
				 * ARMv7: A5-19
				 */
				switch ((insn >> 4) & 0b1111) {
				case 0b0100:
					/* DSB */
					break;
				case 0b0101:
					/* DMB */
					break;
				case 0b0110:
					/* ISB */
					break;
				default:
					assert(0); /* FIXME */
				}
#else /* ! CONFIG_M */
				assert(0); /* UNDEFINED */
#endif /* ! CONFIG_M */

			} else if ((op2 & 0b101) == 0b000
				&& (op1 & 0b1111110) == 0b0111110) {
#if CONFIG_M
				/*
				 * Move from Special Register (MRS)
				 * ARMv7: A6-158, B4-4
				 */
				rd = (insn >> 8) & 0xf;

				tmp32 = NAME_(load_spec)(cpssp, (insn >> 0) & 0xff);

				NAME_(store_reg)(cpssp, rd, tmp32);
#else /* ! CONFIG_M */
				assert(0); /* UNDEFINED */
#endif /* ! CONFIG_M */

			} else if (op2 == 0b010
				&& op1 == 0b1111111) {
				/*
				 * Permanently UNDEFINED
				 */
				assert(0); /* UNDEFINED */

			} else if ((op2 & 0b001) == 0b001) {
				/*
				 * B, ARMv7: A6-40
				 * BL, ARMv7: A6-49
				 */
				int s, j1, j2, i1, i2;

				s = (insn >> 26) & 1;
				j1 = (insn >> 13) & 1;
				j2 = (insn >> 11) & 1;
				i1 = ! (j1 ^ s);
				i2 = ! (j2 ^ s);

				offset = (s << 24)
					| (i1 << 23)
					| (i2 << 22)
					| (((insn >> 16) & 0x3ff) << 12)
					| (((insn >> 0) & 0x7ff) << 1)
					| (0 << 0);
				offset = ((int32_t) offset << 7) >> 7;

				addr = cpssp->NAME.pc_next + offset;
				if (op2 & 0b100) {
					NAME_(store_lr)(cpssp, cpssp->NAME.pc_next | cpssp->NAME.flag_t);
				}
				NAME_(store_pc)(cpssp, addr);
			
			} else {
				assert(0); /* UNDEFINED */
			}
		}
		break;

	case 0b11:
#if CONFIG_M
		if (((insn >> 20) & 0b1110001) == 0b0000000) {
			/*
			 * Store single data item
			 * ARMv7-M: A5-147
			 *
			 * STR (immediate), A6-222
			 * STR (register), A6-224
			 * STRB (immediate), A6-226
			 * STRB (register), A6-228
			 * STRH (immediate), A6-238
			 * STRH (register), A6-240
			 */
			if ((insn >> 23) & 1) {
				/*
				 * Long immediate
				 *
				 * STRB (immediate, T2), A6-226
				 * STRH (immediate, T2), A6-238
				 * STR (immediate, T3), A6-222
				 */
				rn = (insn >> 16) & 0xf;
				rt = (insn >> 12) & 0xf;
				offset = (insn >> 0) & 0xfff;

				tmp32 = NAME_(load_reg)(cpssp, rt);
				addr = NAME_(load_reg)(cpssp, rn);
				addr += offset;

				switch ((insn >> 21) & 3) {
				case 0:
					/* STRB (immediate, T2), A6-226 */
					NAME_(st8)(cpssp, addr, tmp32);
					break;
				case 1:
					/* STRH (immediate, T2), A6-238 */
					NAME_(st16)(cpssp, addr, tmp32);
					break;
				case 2:
					/* STR (immediate, T3), A6-222 */
					NAME_(st32)(cpssp, addr, tmp32);
					break;
				case 3:
					assert(0); /* FIXME */
					break;
				default: assert(0); /* Cannot happen. */
				}

			} else if ((insn >> 11) & 1) {
				/*
				 * Immediate, pre/post
				 *
				 * STRB (immediate, T3), A6-226
				 * STRH (immediate, T3), A6-238
				 * STR (immediate, T4), A6-222
				 */
				rn = (insn >> 16) & 0xf;
				rt = (insn >> 12) & 0xf;
				p = (insn >> 10) & 1;
				u = (insn >> 9) & 1;
				w = (insn >> 8) & 1;
				offset = (insn >> 0) & 0xff;

				tmp32 = NAME_(load_reg)(cpssp, rt);

				addr = NAME_(addr_mode_2_pre)(cpssp, p, u, w, rn, offset);
				switch ((insn >> 21) & 3) {
				case 0:
					/* STRB (immediate, T3), A6-226 */
					NAME_(st8)(cpssp, addr, tmp32);
					break;
				case 1:
					/* STRH (immediate, T3), A6-238 */
					NAME_(st16)(cpssp, addr, tmp32);
					break;
				case 2:
					/* STR (immediate, T4), A6-222 */
					NAME_(st32)(cpssp, addr, tmp32);
					break;
				case 3:
					assert(0); /* FIXME */
					break;
				}
				NAME_(addr_mode_2_post)(cpssp, p, u, w, rn, addr, offset);

			} else {
				/*
				 * Register
				 *
				 * STRB (register, T2), A6-228
				 * STRH (register, T2), A6-240
				 * STR (register, T2), A6-224
				 */
				rn = (insn >> 16) & 0xf;
				rt = (insn >> 12) & 0xf;
				shift = (insn >> 4) & 0x3;
				rm = (insn >> 0) & 0xf;

				tmp32 = NAME_(load_reg)(cpssp, rt);
				addr = NAME_(load_reg)(cpssp, rn);
				offset = NAME_(load_reg)(cpssp, rm);
				offset <<= shift;
				addr += offset;

				switch ((insn >> 21) & 3) {
				case 0:
					/* STRB (register, T2), A6-228 */
					NAME_(st8)(cpssp, addr, tmp32);
					break;
				case 1:
					/* STRH (register, T2), A6-240 */
					NAME_(st16)(cpssp, addr, tmp32);
					break;
				case 2:
					/* STR (register, T2), A6-224 */
					NAME_(st32)(cpssp, addr, tmp32);
					break;
				case 3:
					assert(0); /* FIXME */
					break;
				default: assert(0); /* Cannot happen. */
				}
			}

		} else if (((insn >> 20) & 0b1100101) == 0b0000001) {
			/*
			 * Load byte, memory hints
			 * ARMv7-M: A5-146
			 * Load halfword, memory hints
			 * ARMv7-M: A5-145
			 */
			rn = (insn >> 16) & 0xf;
			rt = (insn >> 12) & 0xf;

			if (rt == 0xf) {
				if (! ((insn >> 21) & 1)) {
					/*
					 * Memory hint
					 */
					/* Nothing to do... */

				} else {
					/*
					 * UNPREDICTABLE
					 * Unallocated memory hint
					 */
					assert(0); /* FIXME */
				}

			} else if (rn == 0xf) {
				assert(rt != 0xf);

				if (! ((insn >> 24) & 1)) {
					if (! ((insn >> 21) & 1)) {
						/*
						 * LDRB (literal)
						 * ARMv7-M: A7-260
						 */
						assert(0); /* FIXME */
					} else {
						/*
						 * LDRH (literal)
						 * ARMv7-M: A7-276
						 */
						assert(0); /* FIXME */
					}

				} else {
					if (! ((insn >> 21) & 1)) {
						/*
						 * LDRSB (literal)
						 * ARMv7-M: A7-286
						 */
						assert(0); /* FIXME */
					} else {
						/*
						 * LDRSH (literal)
						 * ARMv7-M: A7-292
						 */
						assert(0); /* FIXME */
					}
				}
			} else {
				assert(rn != 0xf);
				assert(rt != 0xf);

				if (! ((insn >> 23) & 1)
				 && ((((insn >> 8) & 0b1001) == 0b1001)
				  || (((insn >> 8) & 0b1111) == 0b1100))) {
					p = (insn >> 10) & 1;
					u = (insn >> 9) & 1;
					w = (insn >> 8) & 1;
					offset = (insn >> 0) & 0xff;

					addr = NAME_(load_reg)(cpssp, rn);
					if (p) {
						if (u) {
							addr += offset;
						} else {
							addr -= offset;
						}
					}

					if (! ((insn >> 24) & 1)) {
						if (! ((insn >> 21) & 1)) {
							/*
							 * LDRB (immediate) T3
							 * ARMv7-M: A7-258
							 */
							res = (uint32_t) (uint8_t) NAME_(ld8)(cpssp, addr);
						} else {
							/*
							 * LDRH (immediate)
							 * ARMv7-M: A7-274
							 */
							res = (uint32_t) (uint16_t) NAME_(ld16)(cpssp, addr);
						}

					} else {
						if (! ((insn >> 21) & 1)) {
							/*
							 * LDRSB (immediate)
							 * ARMv7-M: A7-282
							 */
							res = (int32_t) (int8_t) NAME_(ld8)(cpssp, addr);
						} else {
							/*
							 * LDRSH (immediate)
							 * ARMv7-M: A7-290
							 */
							res = (int32_t) (int16_t) NAME_(ld16)(cpssp, addr);
						}
					}

					if (! p) {
						if (u) {
							addr += offset;
						} else {
							addr -= offset;
						}
					}
					if (w) {
						NAME_(store_reg)(cpssp, rn, addr);
					}

					NAME_(store_reg)(cpssp, rt, res);

				} else if ((insn >> 23) & 1) {
					offset = (insn >> 0) & 0xfff;
					addr = NAME_(load_reg)(cpssp, rn);
					addr += offset;

					if (! ((insn >> 24) & 1)) {
						if (! ((insn >> 21) & 1)) {
							/*
							 * LDRB (immediate) T2
							 * ARMv7-M: A7-258
							 */
							res = (uint32_t) (uint8_t) NAME_(ld8)(cpssp, addr);
						} else {
							/*
							 * LDRH (immediate)
							 * ARMv7-M: A7-274
							 */
							res = (uint32_t) (uint16_t) NAME_(ld16)(cpssp, addr);
						}

					} else {
						if (! ((insn >> 21) & 1)) {
							/*
							 * LDRSB (immediate)
							 * ARMv7-M: A7-282
							 */
							res = (int32_t) (int8_t) NAME_(ld8)(cpssp, addr);
						} else {
							/*
							 * LDRSH (immediate)
							 * ARMv7-M: A7-290
							 */
							res = (int32_t) (int16_t) NAME_(ld16)(cpssp, addr);
						}
					}

					NAME_(store_reg)(cpssp, rt, res);

				} else if (! ((insn >> 23) & 1)
					&& ((insn >> 6) & 0b111111) == 0b000000) {
					/*
					 * LDRB (register)
					 * ARMv7-M: A7-262
					 * LDRH (register)
					 * ARMv7-M: A7-278
					 * LDRSB (register)
					 * ARMv7-M: A7-286
					 * LDRSH (register)
					 * ARMv7-M: A7-294
					 */
					offset = (insn >> 4) & 0x3;
					rm = (insn >> 0) & 0xf;

					addr = NAME_(load_reg)(cpssp, rn);
					addr += NAME_(load_reg)(cpssp, rm) << offset;
					if (! ((insn >> 24) & 1)) {
						if (! ((insn >> 21) & 1)) {
							/*
							 * LDRB (register)
							 */
							res = (uint32_t) (uint8_t) NAME_(ld8)(cpssp, addr);
						} else {
							/*
							 * LDRH (register)
							 */
							res = (uint32_t) (uint16_t) NAME_(ld16)(cpssp, addr);
						}
					} else {
						if (! ((insn >> 21) & 1)) {
							/*
							 * LDRSB (register)
							 */
							res = (int32_t) (int8_t) NAME_(ld8)(cpssp, addr);
						} else {
							/*
							 * LDRSH (register)
							 */
							res = (int32_t) (int16_t) NAME_(ld16)(cpssp, addr);
						}
					}

					NAME_(store_reg)(cpssp, rt, res);

				} else if (! ((insn >> 23) & 1)
					&& ((insn >> 6) & 0b111100) == 0b111000) {
					if (! ((insn >> 24) & 1)) {
						if (! ((insn >> 21) & 1)) {
							/*
							 * LDRBT
							 * ARMv7-M: A7-264
							 */
							assert(0); /* FIXME */
						} else {
							/*
							 * LDRHT
							 * ARMv7-M: A7-280
							 */
							assert(0); /* FIXME */
						}
					} else {
						if (! ((insn >> 21) & 1)) {
							/*
							 * LDRSBT
							 * ARMv7-M: A7-288
							 */
							assert(0); /* FIXME */
						} else {
							/*
							 * LDRSHT
							 * ARMv7-M: A7-296
							 */
							assert(0); /* FIXME */
						}
					}

				} else {
					assert(0); /* UNDEFINED */
				}
			}

		} else if (((insn >> 20) & 0b1100111) == 0b0000101) {
			/*
			 * Load word
			 * ARMv7-M: A5-144
			 */
			rn = (insn >> 16) & 0xf;
			rt = (insn >> 12) & 0xf;

			if (! ((insn >> 24) & 1)) {
				if (rn != 0xf) {
					if ((insn >> 23) & 1) {
						/* LDR (immediate, T3), A6-88 */
						offset = (insn >> 0) & 0xfff;
						addr = NAME_(load_reg)(cpssp, rn);
						addr += offset;
						res = NAME_(ld32)(cpssp, addr);
						NAME_(store_reg)(cpssp, rt, res);

					} else if (! ((insn >> 11) & 1)) {
						/* LDR (register, T2), A6-92 */
						rm = (insn >> 0) & 0xf;

						offset = NAME_(load_reg)(cpssp, rm);
						offset <<= (insn >> 4) & 3;
						addr = NAME_(load_reg)(cpssp, rn);
						addr += offset;
						res = NAME_(ld32)(cpssp, addr);
						NAME_(store_reg)(cpssp, rt, res);

					} else {
						/* LDR (immediate, T4), A6-88 */
						/* LDRT (T1), A6-133 */
						p = (insn >> 10) & 1;
						u = (insn >> 9) & 1;
						w = (insn >> 8) & 1;
						offset = (insn >> 0) & 0xff;

						addr = NAME_(addr_mode_2_pre)(cpssp, p, u, w, rn, offset);
						switch ((insn >> 8) & 7) {
						case 0:
						case 1:
						case 2:
						case 3:
						case 4:
						case 5:
						case 7:
							res = NAME_(ld32)(cpssp, addr);
							NAME_(store_reg)(cpssp, rt, res);
							break;
						case 6:
							/* LDRT (T1), A6-133 */
							assert(0); /* FIXME */
						default:
							assert(0); /* Cannot happen. */
						}
						NAME_(addr_mode_2_post)(cpssp, p, u, w, rn, addr, offset);
					}
				} else {
					/*
					 * LDR (literal)
					 * ARMv7-M: A7-254
					 */
					u = (insn >> 23) & 1;
					offset = (insn >> 0) & 0xfff;

					addr = NAME_(load_pc)(cpssp);
					addr -= 2; /* FIXME */
					addr &= ~3;
					if (u) {
						addr += offset;
					} else {
						addr -= offset;
					}
					
					res = NAME_(ld32)(cpssp, addr);

					NAME_(store_reg)(cpssp, rt, res);
				}
			} else {
				assert(0); /* UNDEFINED */
			}

		} else if (((insn >> 20) & 0b1100111) == 0b0000111) {
			/*
			 * UNDEFINED
			 */
			assert(0); /* UNDEFINED */

		} else if (((insn >> 20) & 0b1110000) == 0b0100000) {
			/*
			 * Data processing (register)
			 * ARMv7-M: A5-150
			 */
			assert(((insn >> 12) & 0xf) == 0xf); /* UNDEFINED */

			if (! ((insn >> 7) & 1)) {
				/*
				 * ASR (register, T2), A6-38
				 * LSL (register, T2), A6-136
				 * LSR (register, T2), A6-140
				 * ROR (register, T2), A6-196
				 */
				s = (insn >> 20) & 1;
				rn = (insn >> 16) & 0xf;
				rd = (insn >> 8) & 0xf;
				rm = (insn >> 0) & 0xf;

				op0 = NAME_(load_reg)(cpssp, rn);
				op1 = NAME_(load_reg)(cpssp, rm);

				switch ((insn >> 21) & 3) {
				case 0:
					/* LSL (register, T2), A6-136 */
					op1 &= 0xff;
					if (32 < op1) {
						c = 0;
						res = 0;
					} else if (32 == op1) {
						c = op0 & 1;
						res = 0;
					} else {
						c = (op0 >> (32 - op1)) & 1;
						res = op0 << op1;
					}
					break;
				case 1:
					/* LSR (register, T2), A6-140 */
					op1 &= 0xff;
					if (32 < op1) {
						c = 0;
						res = 0;
					} else if (32 == op1) {
						c = (op0 >> 31) & 1;
						res = 0;
					} else {
						c = (op0 >> (op1 - 1)) & 1;
						res = (uint32_t) op0 >> op1;
					}
					break;
				case 2:
					/* ASR (register, T2), A6-38 */
					assert(0); /* FIXME */
					break;
				case 3:
					/* ROR (register, T2), A6-196 */
					assert(0); /* FIXME */
					break;
				default: assert(0); /* Cannot happen. */
				}
				n = (res >> 31) & 1;
				v = cpssp->NAME.flag_v;
				z = res == 0;

				if (s) {
					NAME_(store_flags)(cpssp, c, n, v, z);
				}
				NAME_(store_reg)(cpssp, rd, res);

			} else if (! ((insn >> 23) & 1)) {
				/*
				 * SXTAB, ARMv7-M: A7-456
				 * SXTAH, ARMv7-M: A7-458
				 * SXTB, ARMv7-M: A7-459
				 * SXTH, ARMv7-M: A7-461
				 * UXTAH, ARMv7-M: A7-497
				 * UXTAB, ARMv7-M: A7-495
				 * UXTB, ARMv7-M: A7-498
				 * UXTH, ARMv7-M: A7-500
				 */
				unsigned int rot;

				rn = (insn >> 16) & 0xf;
				rd = (insn >> 8) & 0xf;
				rm = (insn >> 0) & 0xf;

				switch ((insn >> 20) & 0b1111) {
				case 0b0000:
					/*
					 * SXTAH, ARMv7-M: A7-458
					 * SXTH, ARMv7-M: A7-461
					 */
					rot = (insn >> 4) & 0x3;
					rot *= 8;

					op0 = NAME_(load_reg)(cpssp, rm);
					if (rot) {
						op0 = (op0 >> rot)
							| (op0 << (32 - rot));
					}

					res = (int32_t) (int16_t) op0;
					if (rn != 0xf) {
						res += NAME_(load_reg)(cpssp, rn);
					}

					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 0b0001:
					/*
					 * UXTAH, ARMv7-M: A7-497
					 * UXTH, ARMv7-M: A7-500
					 */
					rot = (insn >> 4) & 0x3;
					rot *= 8;

					op0 = NAME_(load_reg)(cpssp, rm);
					if (rot) {
						op0 = (op0 >> rot)
							| (op0 << (32 - rot));
					}

					res = (uint32_t) (uint16_t) op0;
					if (rn != 0xf) {
						res += NAME_(load_reg)(cpssp, rn);
					}

					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 0b0010:
					assert(0); /* FIXME */
				case 0b0011:
					assert(0); /* FIXME */
				case 0b0100:
					/*
					 * SXTAB, ARMv7-M: A7-456
					 * SXTB, ARMv7-M: A7-459
					 */
					rot = (insn >> 4) & 0x3;
					rot *= 8;

					op0 = NAME_(load_reg)(cpssp, rm);
					if (rot) {
						op0 = (op0 >> rot)
							| (op0 << (32 - rot));
					}

					res = (int32_t) (int8_t) op0;
					if (rn != 0xf) {
						res += NAME_(load_reg)(cpssp, rn);
					}

					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 0b0101:
					/*
					 * UXTAB, ARMv7-M: A7-495
					 * UXTB, ARMv7-M: A7-498
					 */
					rot = (insn >> 4) & 0x3;
					rot *= 8;

					op0 = NAME_(load_reg)(cpssp, rm);
					if (rot) {
						op0 = (op0 >> rot)
							| (op0 << (32 - rot));
					}

					res = (uint32_t) (uint8_t) op0;
					if (rn != 0xf) {
						res += NAME_(load_reg)(cpssp, rn);
					}

					NAME_(store_reg)(cpssp, rd, res);
					break;
				default:
					assert(0); /* UNDEFINED */
				}

			} else {
				/*
				 * Miscellaneous operations, A5-29
				 */
				assert(0); /* FIXME */
			}

		} else if (((insn >> 20) & 0b1111000) == 0b0110000) {
			/*
			 * Multiply, and multiply accumulate
			 * ARMv7-M: A5-154
			 *
			 * ...
			 * MUL, ARMv7-M: A7-324
			 * ...
			 */
			uint8_t ra;
#if CONFIG_VERSION <= 7
			uint32_t op2;
#endif

			assert(((insn >> 6) & 0b11) == 0b00); /* UNDEFINED */

			ra = (insn >> 12) & 0xf;

			switch ((insn >> 20) & 0b111) {
#if CONFIG_VERSION <= 7
			case 0b000:
				switch ((insn >> 4) & 0b11) {
				case 0b00:
					/*
					 * MLA, ARMv7-M: A7-310
					 * MUL, ARMv7-M: A7-324
					 */
					rn = (insn >> 16) & 0xf;
					rd = (insn >> 8) & 0xf;
					rm = (insn >> 0) & 0xf;

					assert(rn != 0xd); /* UNPREDICTABLE */
					assert(rn != 0xf); /* UNPREDICTABLE */
					assert(ra != 0xd); /* UNPREDICTABLE */
					assert(rd != 0xd); /* UNPREDICTABLE */
					assert(rd != 0xf); /* UNPREDICTABLE */
					assert(rm != 0xd); /* UNPREDICTABLE */
					assert(rm != 0xf); /* UNPREDICTABLE */

					op0 = NAME_(load_reg)(cpssp, rn);
					op1 = NAME_(load_reg)(cpssp, rm);
					if (ra == 0xf) {
						op2 = 0x00000000;
					} else {
						op2 = NAME_(load_reg)(cpssp, ra);
					}

					res = NAME_(mlas)(cpssp, op0, op1, op2,
							&c, &n, &v, &z);

					NAME_(store_reg)(cpssp, rd, res);
					break;
				case 0b01:
					assert(0); /* FIXME */
					break;
				default:
					assert(0); /* UNDEFINED */
					break;
				}
				break;
#endif /* CONFIG_VERSION <= 7 */
			case 0b001:
				if (ra != 0b1111) {
					assert(0); /* FIXME */
				} else {
					assert(0); /* FIXME */
				}
				break;
			case 0b010:
				switch ((insn >> 4) & 0b11) {
				case 00:
				case 01:
					if (ra != 0b1111) {
						assert(0); /* FIXME */
					} else {
						assert(0); /* FIXME */
					}
					break;
				default:
					assert(0); /* UNDEFINED */
					break;
				}
				break;
			case 0b011:
				switch ((insn >> 4) & 0b11) {
				case 00:
				case 01:
					if (ra != 0b1111) {
						assert(0); /* FIXME */
					} else {
						assert(0); /* FIXME */
					}
					break;
				default:
					assert(0); /* UNDEFINED */
					break;
				}
				break;
			case 0b100:
				switch ((insn >> 4) & 0b11) {
				case 00:
				case 01:
					if (ra != 0b1111) {
						assert(0); /* FIXME */
					} else {
						assert(0); /* FIXME */
					}
					break;
				default:
					assert(0); /* UNDEFINED */
					break;
				}
				break;
			case 0b101:
				switch ((insn >> 4) & 0b11) {
				case 00:
				case 01:
					if (ra != 0b1111) {
						assert(0); /* FIXME */
					} else {
						assert(0); /* FIXME */
					}
					break;
				default:
					assert(0); /* UNDEFINED */
					break;
				}
				break;
			case 0b110:
				switch ((insn >> 4) & 0b11) {
				case 00:
				case 01:
					assert(0); /* FIXME */
					break;
				default:
					assert(0); /* UNDEFINED */
					break;
				}
				break;
			case 0b111:
				switch ((insn >> 4) & 0b11) {
				case 00:
					if (ra != 0b1111) {
						assert(0); /* FIXME */
					} else {
						assert(0); /* FIXME */
					}
					break;
				default:
					assert(0); /* UNDEFINED */
					break;
				}
				break;
			default:
				assert(0); /* UNDEFINED */
			}

		} else if (((insn >> 20) & 0b1111000) == 0b0111000) {
			/*
			 * Long multiply, long multiply accumulate, and divide
			 * ARMv7-M: A5-154
			 */
			switch ((insn >> 20) & 7) {
			case 0:
				/* SMULL, ARMv7-M: A6-214 */
				assert(0); /* FIXME */
				break;
			case 1:
				/* SDIV, ARMv7-M: A6-210 */
				assert(0); /* FIXME */
				break;
			case 2:
				/* UMULL, ARMv7-M: A6-269 */
				rn = (insn >> 16) & 0xf;
				rdlo = (insn >> 12) & 0xf;
				rdhi = (insn >> 8) & 0xf;
				rm = (insn >> 0) & 0xf;

				op0 = NAME_(load_reg)(cpssp, rn);
				op1 = NAME_(load_reg)(cpssp, rm);

				res64 = NAME_(umulls)(cpssp, op0, op1,
						&c, &n, &v, &z);

				NAME_(store_reg)(cpssp, rdhi, (res64 >> 32) & 0xffffffff);
				NAME_(store_reg)(cpssp, rdlo, (res64 >> 0) & 0xffffffff);
				break;

			case 3:
				/* UDIV, ARMv7-M: A6-267 */
				rn = (insn >> 16) & 0xf;
				rd = (insn >> 8) & 0xf;
				rm = (insn >> 0) & 0xf;

				op0 = NAME_(load_reg)(cpssp, rn);
				op1 = NAME_(load_reg)(cpssp, rm);

				assert(op1 != 0); /* FIXME */
				res = op0 / op1;

				NAME_(store_reg)(cpssp, rd, res);
				break;

			case 4:
				/* SMLAL, ARMv7-M: A6-213 */
				assert(0); /* FIXME */
				break;
			case 5:
				/* UMLAL, ARMv7-M: A6-268 */
				assert(0); /* FIXME */
				break;
			case 6:
				assert(0); /* UNDEFINED */
				break;
			case 7:
				assert(0); /* UNDEFINED */
				break;
			default: assert(0); /* Cannot happen. */
			}

		} else if (((insn >> 20) & 0b1000000) == 0b1000000) {
			/*
			 * Coprocessor instructions
			 * ARMv7-M: A5-156
			 */
			assert(0); /* FIXME */

		} else {
			assert(0); /* Cannot happen. */
		}
#else /* ! CONFIG_M */
		assert(0); /* UNDEFINED */
#endif /* ! CONFIG_M */
	}
}

static void
NAME_(insn_exec_thumb)(struct cpssp *cpssp)
{
	uint16_t insn16;
	uint16_t reglist;
	uint16_t op;
	uint16_t rd;
	uint16_t rm;
	uint16_t rn;
	uint32_t offset;
	uint32_t insn;

	if (cpssp->NAME.stall) {
		/*
		 * Pipeline stalled because of ld/st.
		 */
		/* Just wait... */
		cpssp->NAME.stall = 0;
		return;
	}

	cpssp->NAME.pc = cpssp->NAME.pc_next;
	cpssp->NAME.pc_next = cpssp->NAME.pc_next_next;
	cpssp->NAME.pc_next_next += 2;

	cpssp->NAME.insn = cpssp->NAME.insn_next;
	cpssp->NAME.insn_next = NAME_(ld16_code)(cpssp, cpssp->NAME.pc_next);

	if (cpssp->NAME.flush) {
		/*
		 * Flush pipeline after branch taken.
		 */
		/* Just wait... */
		cpssp->NAME.flush = 0;
		return;
	}

#if 0
	fprintf(stderr, "DIS %08x %04x\n", cpssp->NAME.pc, cpssp->NAME.insn);
#endif

	insn16 = cpssp->NAME.insn;

	if (! cpssp->NAME.insn_prev) {
		/*
		 * 16-bit instruction
		 * or
		 * first part of 32-bit instruction
		 */
		switch ((insn16 >> 12) & 0xf) {
		case 0x0: case 0x1:
			if (! NAME_(it_true)(cpssp)) goto skip;

			if (((insn16 >> 11) & 0x3) != 0x3) {
				/*
				 * Move shifted register.
				 *
				 * LSL(1), A7.1.38, A7-64
				 * LSR(1), A7.1.40, A7-68
				 * ASR(1), A7.1.11, A7-15
				 */
				COUNT_INST(shift);

				offset = (insn16 >> 6) & 0x1f;
				rm = (insn16 >> 3) & 0x7;
				rd = (insn16 >> 0) & 0x7;

				switch ((insn16 >> 11) & 0b11) {
				case 0b00:
					/* LSL(1), A7.1.38, A7-64 */
					insn = 0xe1b00000;
					break;

				case 0b01:
					/* LSR(1), A7.1.40, A7-68 */
					insn = 0xe1b00020;
					break;

				case 0b10:
					/* ASR(1), A7.1.11, A7-15 */
					insn = 0xe1b00040;
					break;

				case 0b11:
				default: assert(0); /* Cannot happen. */
				}
				insn |= (rd << 12) | (offset << 7) | (rm << 0);

				NAME_(insn_exec)(cpssp, insn);

			} else if ((insn16 >> 10) & 1) {
				/*
				 * Add/subtract immediate.
				 *
				 * ADD(1), A7.1.3, A7-5
				 * SUB(1), A7.1.65, A7-113
				 */
				COUNT_INST(addsub);

				offset = (insn16 >> 6) & 0x7;
				rn = (insn16 >> 3) & 0x7;
				rd = (insn16 >> 0) & 0x7;

				switch ((insn16 >> 9) & 1) {
				case 0:
					/* ADD(1), A7.1.3, A7-5 */
					insn = 0xe2900000;
					break;
				case 1:
					/* SUB(1), A7.1.65, A7-113 */
					insn = 0xe2500000;
					break;
				default: assert(0); /* Cannot happen. */
				}
				insn |= (rn << 16) | (rd << 12) | (offset << 0);

				NAME_(insn_exec)(cpssp, insn);

			} else {
				/*
				 * Add/subtract register.
				 *
				 * ADD(3), A7.1.5, A7-7
				 * SUB(3), A7.1.67, A7-115
				 */
				COUNT_INST(addsub);

				rm = (insn16 >> 6) & 0x7;
				rn = (insn16 >> 3) & 0x7;
				rd = (insn16 >> 0) & 0x7;

				switch ((insn16 >> 9) & 1) {
				case 0:
					/* ADD(3), A7.1.5, A7-7 */
					insn = 0xe0900000;
					break;
				case 1:
					/* SUB(3), A7.1.67, A7-115 */
					insn = 0xe0500000;
					break;
				default: assert(0); /* Cannot happen. */
				}
				insn |= (rn << 16) | (rd << 12) | (rm << 0);
					
				NAME_(insn_exec)(cpssp, insn);
			}
			break;

		case 0x2: case 0x3:
			if (! NAME_(it_true)(cpssp)) goto skip;

			/*
			 * Move/compare/add/subtract immediate.
			 *
			 * ADD(2), A7.1.4, A7-6
			 * CMP(1), A7.1.21, A7-35
			 * MOV(1), A7.1.42, A7-72
			 * SUB(2), A7.1.66, A7-114
			 */
			COUNT_INST(immediate);

			rd = (insn16 >> 8) & 0x7;
			offset = (insn16 >> 0) & 0xff;

			switch ((insn16 >> 11) & 0x3) {
			case 0:
				/* MOV(1), A7.1.42, A7-72 */
				insn = 0xe3b00000;
				insn |= (0 << 16) | (rd << 12) | (offset << 0);
				break;
			case 1:
				/* CMP(1), A7.1.21, A7-35 */
				insn = 0xe3500000;
				insn |= (rd << 16) | (0 << 12) | (offset << 0);
				break;
			case 2:
				/* ADD(2), A7.1.4, A7-6 */
				insn = 0xe2900000;
				insn |= (rd << 16) | (rd << 12) | (offset << 0);
				break;
			case 3:
				/* SUB(2), A7.1.66, A7-114 */
				insn = 0xe2500000;
				insn |= (rd << 16) | (rd << 12) | (offset << 0);
				break;
			default: assert(0); /* Cannot happen. */
			}

			NAME_(insn_exec)(cpssp, insn);
			break;

		case 0x4:
			if (! NAME_(it_true)(cpssp)) goto skip;

			switch ((insn16 >> 10) & 0x3) {
			case 0x0:
				/*
				 * ALU operations
				 *
				 * ADC, A7.1.2, A7-4
				 * AND, A7.1.10, A7-14
				 * ASR(2), A7.1.12, A7-17
				 * BIC, A7.1.15, A7-23
				 * CMN, A7.1.20, A7-34
				 * CMP(2), A7.1.22, A7-36
				 * EOR, A7.1.26, A7-43
				 * LSL(2), A7.1.39, A7-66
				 * LSR(2), A7.1.41, A7-70
				 * MVN, A7.1.46, A7-79
				 * MUL, A7.1.45, A7-77
				 * NEG, A7.1.47, A7-80
				 * ORR, A7.1.48, A7-81
				 * ROR, A7.1.54, A7-92
				 * SBC, A7.1.55, A7-94
				 * TST, A7.1.72, A7-122
				 */
				COUNT_INST(alu);

				rm = (insn16 >> 3) & 0x7;
				rd = (insn16 >> 0) & 0x7;

				switch ((insn16 >> 6) & 0xf) {
				case 0x0:
					/* AND, A7.1.10, A7-14 */
					insn = 0xe0100000;
					insn |= (rd << 16) | (rd << 12) | (rm << 0);
					break;
				case 0x1:
					/* EOR, A7.1.26, A7-43 */
					insn = 0xe0300000;
					insn |= (rd << 16) | (rd << 12) | (rm << 0);
					break;
				case 0x2:
					/* LSL(2), A7.1.39, A7-66 */
					insn = 0xe1b00010;
					insn |= (0 << 16) | (rd << 12) | (rm << 8) | (rd << 0);
					break;
				case 0x3:
					/* LSR(2), A7.1.41, A7-70 */
					insn = 0xe1b00030;
					insn |= (0 << 16) | (rd << 12) | (rm << 8) | (rd << 0);
					break;
				case 0x4:
					/* ASR(2), A7.1.12, A7-17 */
					insn = 0xe1b00050;
					insn |= (0 << 16) | (rd << 12) | (rm << 8) | (rd << 0);
					break;
				case 0x5:
					/* ADC, A7.1.2, A7-4 */
					insn = 0xe0b00000;
					insn |= (rd << 16) | (rd << 12) | (rm << 0);
					break;
				case 0x6:
					/* SBC, A7.1.55, A7-94 */
					insn = 0xe0d00000;
					insn |= (rd << 16) | (rd << 12) | (rm << 0);
					break;
				case 0x7:
					/* ROR, A7.1.54, A7-92 */
					insn = 0xe1b00070;
					insn |= (0 << 16) | (rd << 12) | (rm << 8) | (rd << 0);
					break;
				case 0x8:
					/* TST, A7.1.72, A7-122 */
					insn = 0xe1100000;
					insn |= (rd << 16) | (0 << 12) | (rm << 0);
					break;
				case 0x9:
					/* NEG, A7.1.47, A7-80 */
					insn = 0xe2700000;
					insn |= (rm << 16) | (rd << 12);
					break;
				case 0xa:
					/* CMP(2), A7.1.22, A7-36 */
					insn = 0xe1500000;
					insn |= (rd << 16) | (0 << 12) | (rm << 0);
					break;
				case 0xb:
					/* CMN, A7.1.20, A7-34 */
					insn = 0xe1700000;
					insn |= (rd << 16) | (0 << 12) | (rm << 0);
					break;
				case 0xc:
					/* ORR, A7.1.48, A7-81 */
					insn = 0xe1900000;
					insn |= (rd << 16) | (rd << 12) | (rm << 0);
					break;
				case 0xd:
					/* MUL, A7.1.45, A7-77 */
					insn = 0xe0100090;
					insn |= (rd << 16) | (0 << 12) | (rd << 8) | (rm << 0);
					break;
				case 0xe:
					/* BIC, A7.1.15, A7-23 */
					insn = 0xe1d00000;
					insn |= (rd << 16) | (rd << 12) | (rm << 0);
					break;
				case 0xf:
					/* MVN, A7.1.46, A7-79 */
					insn = 0xe1f00000;
					insn |= (0 << 16) | (rd << 12) | (rm << 0);
					break;
				default: assert(0); /* Cannot happen. */
				}

				NAME_(insn_exec)(cpssp, insn);
				break;

			case 0x1:
				/*
				 * Hi register operations/branch exchange
				 *
				 * ADD(4), A7.1.6, A7-8
				 * BLX(2), A7.1.18, A7-30
				 * BX, A7.1.19, A7-32
				 * CMP(3), A7.1.23, A7-37
				 * MOV(3), A7.1.44, A7-75
				 */
				COUNT_INST(hireg);

				rm = (insn16 >> 3) & 0x7;
				rm |= ((insn16 >> 6) & 0x1) << 3;
				rd = (insn16 >> 0) & 0x7;
				rd |= ((insn16 >> 7) & 0x1) << 3;

				switch ((insn16 >> 8) & 0x3) {
				case 0x0:
					/* ADD(4), A7.1.6, A7-8 */
					insn = 0xe0800000;
					insn |= (rd << 16) | (rd << 12) | (rm << 0);
					break;
				case 0x1:
					/* CMP(3), A7.1.23, A7-37 */
					insn = 0xe1500000;
					insn |= (rd << 16) | (0 << 12) | (rm << 0);
					break;
				case 0x2:
					/* MOV(3), A7.1.44, A7-75 */
					insn = 0xe1a00000;
					insn |= (0 << 16) | (rd << 12) | (rm << 0);
					break;
				case 0x3:
					/* BLX(2), A7.1.18, A7-30 */
					/* BX, A7.1.19, A7-32 */
					insn = 0xe1200010;
					insn |= ((insn16 >> 7) & 1) << 5;
					insn |= (0xf << 16) | (0xf << 12) | (0xf << 8) | (rm << 0);
					break;
				default: assert(0); /* Cannot happen. */
				}

				NAME_(insn_exec)(cpssp, insn);
				break;

			case 0x2: case 0x3:
				/*
				 * PC-relative load
				 *
				 * LDR(3), A7.1.30
				 */
				COUNT_INST(mempcrel);

				rd = (insn16 >> 8) & 0x7;
				offset = (insn16 >> 0) & 0xff;

				insn = 0xe59f0000;
				insn |= (rd << 12) | (offset << 2);

				NAME_(insn_exec)(cpssp, insn);
				break;

			default: assert(0); /* Cannot happen. */
			}
			break;
		case 0x5:
			if (! NAME_(it_true)(cpssp)) goto skip;

			/*
			 * Load/Store
			 *
			 * LDR(2), A7.1.29, A7-49
			 * LDRB(2), A7.1.33, A7-56
			 * LDRH(2), A7.1.35, A7-59
			 * LDRSB, A7.1.36, A7-61
			 * LDRSH, A7.1.37, A7-62
			 * STR(2), A7.1.59, A7-101
			 * STRB(2), A7.1.62, A7-107
			 * STRH(2), A7.1.64, A7-111
			 */
			COUNT_INST(memreg);

			rm = (insn16 >> 6) & 0x7;
			rn = (insn16 >> 3) & 0x7;
			rd = (insn16 >> 0) & 0x7;

			switch ((insn16 >> 9) & 0b111) {
			case 0b000:
				/* STR(2), A7.1.59, A7-101 */
				insn = 0xe7800000;
				break;
			case 0b001:
				/* STRH(2), A7.1.64, A7-111 */
				insn = 0xe18000b0;
				break;
			case 0b010:
				/* STRB(2), A7.1.62, A7-107 */
				insn = 0xe7c00000;
				break;
			case 0b011:
				/* LDRSB, A7.1.36, A7-61 */
				insn = 0xe19000d0;
				break;
			case 0b100:
				/* LDR(2), A7.1.29, A7-49 */
				insn = 0xe7900000;
				break;
			case 0b101:
				/* LDRH(2), A7.1.35, A7-59 */
				insn = 0xe19000b0;
				break;
			case 0b110:
				/* LDRB(2), A7.1.33, A7-56 */
				insn = 0xe7d00000;
				break;
			case 0b111:
				/* LDRSH, A7.1.37, A7-62 */
				insn = 0xe19000f0;
				break;
			default: assert(0); /* Cannot happen. */
			}
			insn |= (rn << 16) | (rd << 12) | (rm << 0);

			NAME_(insn_exec)(cpssp, insn);
			break;

		case 0x6:
		case 0x7:
			if (! NAME_(it_true)(cpssp)) goto skip;

			/*
			 * load/store with immediate offset
			 *
			 * STR(1), A7.1.58, A7-99
			 */
			COUNT_INST(memimmediate);

			offset = (insn16 >> 6) & 0x1f;
			rn = (insn16 >> 3) & 0x7;
			rd = (insn16 >> 0) & 0x7;

			switch ((insn16 >> 11) & 0x3) {
			case 0:
				/* STR(1), A7.1.58, A7-99 */
				insn = 0xe5800000;
				insn |= (rn << 16) | (rd << 12) | (offset << 2);
				break;
			case 1:
				/* LDR(1), A7.1.58, A7-99 */
				insn = 0xe5900000;
				insn |= (rn << 16) | (rd << 12) | (offset << 2);
				break;
			case 2:
				/* STRB(1), A7.1.61, A7-105 */
				insn = 0xe5c00000;
				insn |= (rn << 16) | (rd << 12) | (offset << 0);
				break;
			case 3:
				/* LDRB(1), A7.1.32, A7-55 */
				insn = 0xe5d00000;
				insn |= (rn << 16) | (rd << 12) | (offset << 0);
				break;
			default: assert(0); /* Cannot happen. */
			}

			NAME_(insn_exec)(cpssp, insn);
			break;

		case 0x8:
			if (! NAME_(it_true)(cpssp)) goto skip;

			/*
			 * load/store halfword
			 *
			 * LDRH(1), A7.1.34, A7-57
			 * STRH(1), A7.1.63, A7-109
			 */
			COUNT_INST(memimmediate);

			offset = (insn16 >> 6) & 0x1f;
			rn = (insn16 >> 3) & 0x7;
			rd = (insn16 >> 0) & 0x7;

			switch ((insn16 >> 11) & 1) {
			case 0:
				/* STRH(1), A7.1.63, A7-109 */
				insn = 0xe1c000b0;
				break;
			case 1:
				/* LDRH(1), A7.1.34, A7-57 */
				insn = 0xe1d000b0;
				break;
			default: assert(0); /* Cannot happen. */
			}
			insn |= (rn << 16) | (rd << 12);
			insn |= ((offset >> 3) & 0x3) << 8;
			insn |= ((offset >> 0) & 0x7) << 1;

			NAME_(insn_exec)(cpssp, insn);
			break;

		case 0x9:
			if (! NAME_(it_true)(cpssp)) goto skip;

			/*
			 * SP-relative load/store
			 *
			 * LDR(4), A7.1.31, A7-53
			 * STR(3), A7.1.60, A7-103
			 */
			COUNT_INST(memsprel);

			rd = (insn16 >> 8) & 0x7;
			offset = (insn16 >> 0) & 0xff;
			
			switch ((insn16 >> 11) & 1) {
			case 0:
				/* STR(3), A7.1.60, A7-103 */
				insn = 0xe58d0000;
				break;
			case 1:
				/* LDR(4), A7.1.31, A7-53 */
				insn = 0xe59d0000;
				break;
			default: assert(0); /* Cannot happen. */
			}
			insn |= (rd << 12) | (offset << 2);

			NAME_(insn_exec)(cpssp, insn);
			break;

		case 0xa:
			if (! NAME_(it_true)(cpssp)) goto skip;

			/*
			 * Load address
			 *
			 * ADD(5), A7.1.7, A7-10
			 * ADD(6), A7.1.8, A7-11
			 */
			COUNT_INST(lea);

			rd = (insn16 >> 8) & 7;
			offset = (insn16 & 0xff);

			insn = 0xe2800f00;
			switch ((insn16 >> 11) & 1) {
			case 0:
				/* ADD(5), A7.1.7, A7-10 */
				insn |= 15 << 16; /* PC */
				break;
			case 1:
				/* ADD(6), A7.1.8, A7-11 */
				insn |= 13 << 16; /* SP */
				break;
			default: assert(0); /* Cannot happen. */
			}
			insn |= (rd << 12) | (offset << 0);
				
			NAME_(insn_exec)(cpssp, insn);
			break;

		case 0xb:
			if (! NAME_(it_true)(cpssp)) goto skip;

			/* Misc */
			op = (insn16 >> 8) & 0xf;
			switch (op) {
			case 0x0:
				/*
				 * Add offset to stack pointer
				 *
				 * ADD (SP plus immediate), ARMv7-M: A6-26
				 * SUB (SP minus immediate), ARMv7-M: A6-248
				 */
				COUNT_INST(addsubsp);

				offset = (insn16 >> 0) & 0x7f;

				switch ((insn16 >> 7) & 1) {
				case 0:
					/* ADD(7), A7.1.9, A7-12 */
					insn = 0xe28ddf00;
					break;
				case 1:
					/* SUB(4), A7.1.68, A7-116 */
					insn = 0xe24ddf00;
					break;
				default: assert(0); /* Cannot happen. */
				}
				insn |= (offset << 0);

				NAME_(insn_exec)(cpssp, insn);
				break;

			case 0x1: case 0x3:
			case 0x9: case 0xb: {
#if 7 <= CONFIG_VERSION
				/*
				 * Compare and Branch on Zero
				 *
				 * CBNZ, CBZ, ARMv7-M: A6-52
				 */
				uint32_t tmp32;
				uint32_t addr;

				offset = ((insn16 >> 9) & 1) << 5;
				offset |= ((insn16 >> 3) & 0x1f) << 0;
				rn = (insn16 >> 0) & 0x7;

				tmp32 = NAME_(load_reg)(cpssp, rn);

				if (((insn16 >> 11) & 1) ^ (tmp32 == 0)) {
					addr = NAME_(load_pc)(cpssp);
					addr += offset << 1;
					NAME_(store_pc)(cpssp, addr);
				}
#else /* CONFIG_VERSION < 7 */
				assert(0); /* UNDEFINED */
#endif /* CONFIG_VERSION < 7 */
				break;
			    }
			case 0x2:
				/*
				 * sign/zero extend.
				 *
				 * SXTB, ARMv7-M: A6-254
				 * SXTH, ARMv7-M: A6-256
				 * UXTB, ARMv7-M: A6-272
				 * UXTH, ARMv7-M: A6-274
				 */
				COUNT_INST(extend);

				rd = (insn16 >> 0) & 0x7;
				rm = (insn16 >> 3) & 0x7;

				switch ((insn16 >> 6) & 0x3) {
				case 0x0:
					/* SXTH, ARMv7-M: A6-256 */
					insn = 0xe6bf0070;
					break;
				case 0x1:
					/* SXTB, ARMv7-M: A6-254 */
					insn = 0xe6af0070;
					break;
				case 0x2:
					/* UXTH, ARMv7-M: A6-274 */
					insn = 0xe6ff0070;
					break;
				case 0x3:
					/* UXTB, ARMv7-M: A6-272 */
					insn = 0xe6ef0070;
					break;
				default: assert(0); /* Cannot happen. */
				}
				insn |= (rd << 12) | (rm << 0);

				NAME_(insn_exec)(cpssp, insn);
				break;

			case 0x4: case 0x5:
			case 0xc: case 0xd:
				/*
				 * push/pop registers
				 *
				 * POP, ARMv7-M: A6-186
				 * PUSH, ARMv7-M: A6-188
				 */
				COUNT_INST(pushpop);

				reglist = (insn16 >> 0) & 0xff;

				switch ((insn16 >> 11) & 1) {
				case 0:
					/* PUSH, ARMv7-M: A6-188 */
					insn = 0xe92d0000;
					reglist |= (((insn16 >> 8) & 1) << 14);
					break;
				case 1:
					/* POP, ARMv7-M: A6-186 */
					insn = 0xe8bd0000;
					reglist |= (((insn16 >> 8) & 1) << 15);
					break;
				default: assert(0); /* Cannot happen. */
				}
				insn |= reglist;

				NAME_(insn_exec)(cpssp, insn);
				break;

			case 0x6:
				/*
				 * Change Processor State
				 *
				 * CPS, A7.1.24, A7-39
				 */
				COUNT_INST(cps);

				switch ((insn16 >> 5) & 0x7) {
				case 0x3:
					/* CPS, A7.1.24, A7-39 */
					insn = 0xf1000000;
					insn |= (0b10 | ((insn >> 4) & 1)) << 18;
					insn |= ((insn >> 0) & 7) << 6;
					
					break;
				default:
					assert(0); /* Illegal Instruction - FIXME */
				}

				NAME_(insn_exec)(cpssp, insn);
				break;
			case 0x7:
				assert(0); /* Illegal Instruction - FIXME */
				break;
			case 0x8:
				assert(0); /* Illegal Instruction - FIXME */
				break;
			case 0xa:
				/*
				 * Reverse
				 *
				 * REV, ARMv7-M: A6-191
				 * REV16, ARMv7-M: A6-192
				 * REVSH, ARMv7-M: A6-193
				 */
				COUNT_INST(reverse);

				rd = (insn16 >> 0) & 0x7;
				rm = (insn16 >> 3) & 0x7;

				switch ((insn16 >> 6) & 0b11) {
				case 0b00:
					/* REV, ARMv7-M: A6-191 */
					insn = 0xe6b00030;
					break;
				case 0b01:
					/* REV16, ARMv7-M: A6-192 */
					insn = 0xe6b000b0;
					break;
				case 0b11:
					/* REVSH, ARMv7-M: A6-193 */
					insn = 0xe6f000b0;
					break;
				default:
					assert(0); /* Illegal Instruction - FIXME */
				}
				insn |= (rd << 12) | (rm << 0);

				NAME_(insn_exec)(cpssp, insn);
				break;

			case 0xe:
				/*
				 * Software breakpoint
				 *
				 * BKPT, A7.1.16, A7-24
				 */
				offset = (insn16 >> 0) & 0xff;

				insn = 0xe1200000;
				insn |= ((offset >> 4) & 0xf) << 8;
				insn |= ((offset >> 0) & 0xf) << 0;

				NAME_(insn_exec)(cpssp, insn);
				break;

			case 0xf:
#if CONFIG_M
				/*
				 * If-Then and Hints
				 *
				 * ARMv7-M: A5-133
				 */
				COUNT_INST(hint);

				if (((insn16 >> 0) & 0xf) != 0x0) {
					/*
					 * IT
					 * ARMv7-M: A7-242
					 */
					cpssp->NAME.itstate = insn16 & 0xff;

				} else switch ((insn16 >> 4) & 0xf) {
				case 0x0:
					/*
					 * No Operation Hint (NOP)
					 * ARMv7-M: A7-331
					 */
					break;
				case 0x1:
					/*
					 * Yield Hint (YIELD)
					 * ARMv7-M: A7-562
					 */
					break;
				case 0x2:
					/*
					 * Wait for Event Hint (WFE)
					 * ARMv7-M: A7-560
					 */
					/* FIXME */
					break;
				case 0x3:
					/*
					 * Wait for Interrupt Hint (WFI)
					 * ARMv7-M: A7-561
					 */
					cpssp->NAME.wfi = 1;
					NAME_(stop)(cpssp, cpssp->NAME.sleepdeep);
					break;
				case 0x4:
					/*
					 * Send Event Hint (SEV)
					 * ARMv7-M: A7-385
					 */
					/* FIXME */
					break;
				default:
					assert(0); /* FIXME */
				}
#else /* ! CONFIG_M */
				assert(0); /* UNDEFINED */
#endif /* ! CONFIG_M */
				break;

			default: assert(0); /* Cannot happen. */
			}
			break;

		case 0xc:
			if (! NAME_(it_true)(cpssp)) goto skip;

			/*
			 * Multiple load/store
			 *
			 * LDMIA, A7.1.27
			 * STMIA, A7.1.57
			 */
			COUNT_INST(memmultiple);

			rn = (insn16 >> 8) & 0x7;
			reglist = (insn16 >> 0) & 0xff;

			switch ((insn16 >> 11) & 1) {
			case 0:
				/* STMIA, A7.1.57 */
				insn = 0xe8a00000;
				break;
			case 1:
				/* LDMIA, A7.1.27 */
				insn = 0xe8900000;
				insn |= ! ((insn16 >> rn) & 1);
				break;
			default: assert(0); /* Cannot happen. */
			}
			insn |= (rn << 16) | (reglist << 0);

			NAME_(insn_exec)(cpssp, insn);
			break;

		case 0xd:
			if (! NAME_(it_true)(cpssp)) goto skip;

			if (((insn16 >> 8) & 0xf) != 0xf) {
				/*
				 * conditional branch
				 *
				 * B(1), A7.1.13, A7-19
				 */
				uint8_t cond;

				COUNT_INST(branchconditional);

				cond = (insn16 >> 8) & 0xf;
				offset = (int32_t) (int8_t) ((insn16 >> 0) & 0xff);

				insn = 0x0a000000;
				insn |= (cond << 28) | ((offset & 0xffffff) << 0);

				NAME_(insn_exec)(cpssp, insn);

			} else {
				/*
				 * Software interrupt
				 *
				 * SWI, A7.1.69, A7-118
				 */
				offset = (insn16 >> 0) & 0xff;

				insn = 0xef000000;
				insn |= (offset << 0);

				NAME_(insn_exec)(cpssp, insn);
			}
			break;

		case 0xe:
			if (((insn16 >> 11) & 1) == 0) {
				if (! NAME_(it_true)(cpssp)) goto skip;

				/*
				 * unconditional branch
				 *
				 * B(2), A7.1.14, A7-21
				 */
				COUNT_INST(branchunconditional);

				offset = ((int32_t) insn16 << 21) >> 21;

				insn = 0xea000000;
				insn |= (offset & 0xffffff) << 0;

				NAME_(insn_exec)(cpssp, insn);

			} else {
				/* first part of thumb2 instruction */
				/* A5.1 */
				cpssp->NAME.insn_prev = insn16;
			}
			break;

		case 0xf:
			/*
			 * first part of thumb2 instruction
			 */
			cpssp->NAME.insn_prev = insn16;
			break;

		default:
			assert(0); /* Cannot happen. */
		}
	skip:	;

	} else {
		if (! NAME_(it_true)(cpssp)) goto skip2;

		/*
		 * second part of 32-bit instruction
		 */
		COUNT_INST(inst32);

		insn = ((uint32_t) cpssp->NAME.insn_prev << 16) | insn16;

		NAME_(insn_exec_thumb2)(cpssp, insn);

	skip2:	;
		cpssp->NAME.insn_prev = 0; /* Back to 16-bit instruction. */
	}
}
#endif /* CONFIG_THUMB */

static void
NAME_(exc)(struct cpssp *cpssp)
{
	uint8_t vec;
	uint8_t frameptralign;
	uint32_t frameptr;
	uint32_t xpsr;
	uint32_t start;

	vec = NAME_(exc_ack)(cpssp);

	/*
	 * Exception entry behavior
	 * B1.5.6
	 */
	/* PushStack(); */
	if (cpssp->NAME.control_spsel
	 && cpssp->NAME.current_mode == MODE_THREAD) {
		frameptralign = (cpssp->NAME.sp_process >> 2) & 0b1;
		cpssp->NAME.sp_process = (cpssp->NAME.sp_process - 0x20)
				& ~0b100;
		frameptr = cpssp->NAME.sp_process;
	} else {
		frameptralign = (cpssp->NAME.sp_main >> 2) & 0b1;
		cpssp->NAME.sp_main = (cpssp->NAME.sp_main - 0x20)
				& ~0b100;
		frameptr = cpssp->NAME.sp_main;
	}
	NAME_(st32)(cpssp, frameptr + 0x00, NAME_(load_reg)(cpssp, 0));
	NAME_(st32)(cpssp, frameptr + 0x04, NAME_(load_reg)(cpssp, 1));
	NAME_(st32)(cpssp, frameptr + 0x08, NAME_(load_reg)(cpssp, 2));
	NAME_(st32)(cpssp, frameptr + 0x0c, NAME_(load_reg)(cpssp, 3));
	NAME_(st32)(cpssp, frameptr + 0x10, NAME_(load_reg)(cpssp, 12));
	NAME_(st32)(cpssp, frameptr + 0x14, NAME_(load_lr)(cpssp));
	switch (vec) {
	case 0x00:
		/* Reset Stack Pointer */
		assert(0);
	case 0x01:
		/* Reset */
		assert(0);
	case 0x02:
		/* NMI */
		assert(0);
	case 0x03:
		/* HardFault */
		assert(0);
	case 0x04 ... 0x0a:
		/* Reserved */
		assert(0);
	case 0x0b:
		/* SVC */
		NAME_(st32)(cpssp, frameptr + 0x18, cpssp->NAME.pc_next);
		break;
	case 0x0c ... 0x0d:
		/* Reserved */
		assert(0);
	case 0x0e:
		/* PendSV */
		/*FALLTHROUGH*/
	case 0x0f:
		/* SysTick */
		/*FALLTHROUGH*/
	case 0x10 ... 0x30:
		/* Interrupt */
		NAME_(st32)(cpssp, frameptr + 0x18, cpssp->NAME.pc_next);
		break;
	default:
		assert(0); /* FIXME */
	}
	xpsr = NAME_(load_xpsr)(cpssp);
	xpsr &= ~(1 << 9);
	xpsr |= frameptralign << 9;
	NAME_(st32)(cpssp, frameptr + 0x1c, xpsr);
	if (cpssp->NAME.current_mode == MODE_HANDLER) {
		NAME_(store_lr)(cpssp, 0xfffffff1);
	} else if (cpssp->NAME.control_spsel == 0) {
		NAME_(store_lr)(cpssp, 0xfffffff9);
	} else {
		NAME_(store_lr)(cpssp, 0xfffffffd);
	}

	/* ExceptionTaken(); */
	/* FIXME */
	cpssp->NAME.current_mode = MODE_HANDLER;
	NAME_(store_ipsr)(cpssp, vec);
	cpssp->NAME.control_spsel = 0;
	/* cpssp->NAME.control_npriv unchanged. */
	/* FIXME */
	start = NAME_(ld32)(cpssp, cpssp->NAME.vtor | (vec << 2));
	cpssp->NAME.flag_t = (start >> 0) & 1;
	NAME_(store_pc)(cpssp, start & ~1);
}

static void
NAME_(exc_return)(struct cpssp *cpssp)
{
	uint8_t exc_return;
	int vec;
	uint32_t frameptr;
	uint32_t psr;
	uint32_t pc;

	/*
	 * Exception return behavior
	 * B1.5.8
	 */
	exc_return = (cpssp->NAME.pc_next >> 0) & 0b1111;
	vec = NAME_(load_ipsr)(cpssp);

	switch (exc_return) {
	case 0b0000:
	case 0b0001:
		/* Return to Handler. */
		frameptr = cpssp->NAME.sp_main;
		cpssp->NAME.current_mode = MODE_HANDLER;
		cpssp->NAME.control_spsel = 0;
		break;
	case 0b1000:
	case 0b1001:
		/* Return to Thread using Main stack. */
		frameptr = cpssp->NAME.sp_main;
		cpssp->NAME.current_mode = MODE_THREAD;
		cpssp->NAME.control_spsel = 0;
		break;
	case 0b1100:
	case 0b1101:
		/* Return to Thread using Process stack. */
		frameptr = cpssp->NAME.sp_process;
		cpssp->NAME.current_mode = MODE_THREAD;
		cpssp->NAME.control_spsel = 1;
		break;
	default:
		assert(0); /* FIXME */
	}

	/* Deactivate */
	NAME_(exc_eoi)(cpssp, vec);

	/* PopStack */
	NAME_(store_reg)(cpssp, 0, NAME_(ld32)(cpssp, frameptr + 0x00));
	NAME_(store_reg)(cpssp, 1, NAME_(ld32)(cpssp, frameptr + 0x04));
	NAME_(store_reg)(cpssp, 2, NAME_(ld32)(cpssp, frameptr + 0x08));
	NAME_(store_reg)(cpssp, 3, NAME_(ld32)(cpssp, frameptr + 0x0c));
	NAME_(store_reg)(cpssp, 12, NAME_(ld32)(cpssp, frameptr + 0x10));
	NAME_(store_lr)(cpssp, NAME_(ld32)(cpssp, frameptr + 0x14));
	pc = NAME_(ld32)(cpssp, frameptr + 0x18);
	NAME_(store_pc)(cpssp, pc & ~1);
	cpssp->NAME.flag_t = pc & 1;
	psr = NAME_(ld32)(cpssp, frameptr + 0x1c);
	switch (exc_return) {
	case 0b0000:
	case 0b0001: /* Returning to Handler mode. */
	case 0b1000:
	case 0b1001: /* Returning to Thread mode using Main stack. */
		cpssp->NAME.sp_main = (cpssp->NAME.sp_main + 0x20)
				| (((psr >> 9) & 1) << 2);
		break;
	case 0b1100:
	case 0b1101: /* Returning to Thread mode using Process stack. */
		cpssp->NAME.sp_process = (cpssp->NAME.sp_process + 0x20)
				| (((psr >> 9) & 1) << 2);
		break;
	default:
		assert(0); /* FIXME */
	}
	NAME_(store_apsr)(cpssp, psr & 0xf0000000);
	if (cpssp->NAME.current_mode == MODE_THREAD
	 && cpssp->NAME.control_npriv) {
		NAME_(store_ipsr)(cpssp, 0x00000000);
	} else {
		NAME_(store_ipsr)(cpssp, psr & 0x0000003f);
	}
	NAME_(store_epsr)(cpssp, psr & (1 << 24));
}

static void
NAME_(insn_step)(struct cpssp *cpssp)
{
	uint32_t pc_next;

	pc_next = cpssp->NAME.pc_next;

	loglevel = 1;
#if 0
	if (pc_next == 0xc1080000) {
		loglevel = 1;
	}
#endif
	if (unlikely(
	    (cpssp->NAME.brkstart[0] <= pc_next && pc_next < cpssp->NAME.brkend[0])
	 || (cpssp->NAME.brkstart[1] <= pc_next && pc_next < cpssp->NAME.brkend[1])
	 || (cpssp->NAME.brkstart[2] <= pc_next && pc_next < cpssp->NAME.brkend[2])
	 || (cpssp->NAME.brkstart[3] <= pc_next && pc_next < cpssp->NAME.brkend[3]))) {
		gdb_stop(cpssp->NAME.gdb, 5); /* SIGTRAP */

	} else if (unlikely(cpssp->NAME.wfi)) {
		/*
		 * Waiting for Interrupt
		 */
		/* Nothing to do... */
		COUNT_INST(wfi);

#if ! CONFIG_ARM /* FIXME */
	} else if (! cpssp->NAME.flush
		&& ((pc_next >> 28) & 0xf) == 0xf) {
		/*
		 * Exception Return
		 */
		if (DEBUG_CONTROL_FLOW
		 && loglevel) {
			fprintf(stderr, "Exception return\n");
		}
		NAME_(exc_return)(cpssp);
#endif /* CONFIG_ARM */

	} else if (! cpssp->NAME.flush
		&& cpssp->NAME.exception_pending
		&& ! cpssp->NAME.insn_prev) {
		/*
		 * Exception/Interrupt Pending
		 */
		if (DEBUG_CONTROL_FLOW
		 && loglevel) {
			fprintf(stderr, "Exception at %08lx\n",
					cpssp->NAME.pc_next);
		}
		NAME_(exc)(cpssp);
	} else {
		/*
		 * Execute Instruction
		 */
		if (DEBUG_CONTROL_FLOW
		 && loglevel) {
			fprintf(stderr, "Executing %08lx at %08lx\n", cpssp->NAME.insn_next, pc_next);
#if DEBUG_CONTROL_FLOW_REGS
			NAME_(dump)(cpssp);
#endif
		}

#if CONFIG_ARM && CONFIG_THUMB
		if (! cpssp->NAME.flag_t) {
			NAME_(insn_exec_arm)(cpssp);
		} else {
			NAME_(insn_exec_thumb)(cpssp);
		}
#elif CONFIG_ARM
		NAME_(insn_exec_arm)(cpssp);
#elif CONFIG_THUMB
		NAME_(insn_exec_thumb)(cpssp);
#else
#error "Must define CONFIG_ARM and/or CONFIG_THUMB!"
#endif
	}
}

static void
NAME_(irq_set)(struct cpssp *cpssp, unsigned int val)
{
	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s val=%u\n", __FUNCTION__, val);
	}

	cpssp->NAME.exception_pending = val;
	cpssp->NAME.wfi &= ! cpssp->NAME.exception_pending;
}

static void
NAME_(clk)(struct cpssp *cpssp)
{
	NAME_(insn_step)(cpssp);
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	int i;

	if (DEBUG_CONTROL_FLOW
	 && loglevel) {
		fprintf(stderr, "Reset\n");
	}

	cpssp->NAME.wfi = 0;
	cpssp->NAME.exception_pending = 0;

	for (i = 0; i < 13; i++) {
		NAME_(store_reg)(cpssp, i, 0);
	}
	NAME_(store_lr)(cpssp, 0);
	cpssp->NAME.sp_main = 0;
	cpssp->NAME.sp_process = 0;

	cpssp->NAME.current_mode = MODE_THREAD;
	cpssp->NAME.control_spsel = 0;
	cpssp->NAME.control_npriv = 0;

	cpssp->NAME.flag_shift_c = 0;
	cpssp->NAME.flag_c = 0;
	cpssp->NAME.flag_n = 0;
	cpssp->NAME.flag_v = 0;
	cpssp->NAME.flag_z = 0;

	cpssp->NAME.itstate = 0x00;

	cpssp->NAME.vtor = 0x00000000;
	cpssp->NAME.prigroup = 0b000;

	cpssp->NAME.sevonpend = 0;
	cpssp->NAME.sleepdeep = 0;
	cpssp->NAME.sleeponexit = 0;

#if CONFIG_ARM
	/* HACK! */
	/* Read sp. */
	NAME_(store_sp)(cpssp, 0);
	/* Read pc. */
	cpssp->NAME.flag_t = 0;
	NAME_(store_pc)(cpssp, 0xfffd0000); /* Sets flush. */
#else
	/* Read sp. */
	NAME_(store_sp)(cpssp, NAME_(ld32)(cpssp, 0));
	/* Read pc. */
	cpssp->NAME.flag_t = 1;
	NAME_(store_pc)(cpssp, NAME_(ld32)(cpssp, 4) & ~1); /* Sets flush. */
#endif
	cpssp->NAME.stall = 0;
	cpssp->NAME.insn_prev = 0;
}

static int
NAME_(gdb_reg_read)(void *_cpssp, unsigned long n, void *_buf, int len)
{
	struct cpssp *cpssp = _cpssp;
	uint32_t *buf = _buf;

	switch (n) {
	case 0 ... 12:
		assert(sizeof(*buf) <= len);
		*buf = NAME_(load_reg)(cpssp, n);
		return sizeof(*buf);
	case 13:
		assert(sizeof(*buf) <= len);
		*buf = NAME_(load_sp)(cpssp);
		return sizeof(*buf);
	case 14:
		assert(sizeof(*buf) <= len);
		*buf = NAME_(load_lr)(cpssp);
		return sizeof(*buf);
	case 15:
		assert(sizeof(*buf) <= len);
		*buf = cpssp->NAME.pc_next;
		return sizeof(*buf);
	case 16 ... 23:
		assert(3 * sizeof(*buf) <= len);
		*buf++ = 0;
		*buf++ = 0;
		*buf++ = 0;
		return 3 * sizeof(*buf);
	case 24:
		assert(sizeof(*buf) <= len);
		*buf = 0;
		return sizeof(*buf);
	case 25:
		assert(sizeof(*buf) <= len);
		*buf = NAME_(load_xpsr)(cpssp);
		return sizeof(*buf);
	default:
		return -1;
	}
}

static int
NAME_(gdb_mem_read)(void *_cpssp, unsigned long addr, void *_buf, int len)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t *buf = _buf;
	int i;

	for (i = 0; i < len; i++) {
		*buf++ = NAME_(ld8)(cpssp, (uint32_t) addr++);
	}

	return len;
}

static int
NAME_(gdb_mem_write)(void *_cpssp, unsigned long addr, void *_buf, int len)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t *buf = _buf;
	int i;

	for (i = 0; i < len; i++) {
		NAME_(st8)(cpssp, (uint32_t) addr++, *buf++);
	}

	return len;
}

static int
NAME_(gdb_brk_set)(void *_cpssp, int type, unsigned long addr, int len)
{
	struct cpssp *cpssp = _cpssp;
	unsigned int i;

	assert(type == 0); /* FIXME */

	for (i = 0; ; i++) {
		if (i == 4) {
			return -1;
		}
		if (cpssp->NAME.brkstart[i] == 0
		 && cpssp->NAME.brkend[i] == 0) {
			cpssp->NAME.brkstart[i] = addr;
			cpssp->NAME.brkend[i] = addr + len;
			return 0;
		}
	}
}

static int
NAME_(gdb_brk_clr)(void *_cpssp, int type, unsigned long addr, int len)
{
	struct cpssp *cpssp = _cpssp;
	unsigned int i;

	assert(type == 0); /* FIXME */

	for (i = 0; ; i++) {
		if (i == 4) {
			return -1;
		}
		if (cpssp->NAME.brkstart[i] == addr
		 && cpssp->NAME.brkend[i] == addr + len) {
			cpssp->NAME.brkstart[i] = 0;
			cpssp->NAME.brkend[i] = 0;
			return 0;
		}
	}
}

static void
NAME_(create)(const char *name, struct cpssp *cpssp)
{
	static struct gdb_funcs NAME_(funcs) = {
		.reg_read = NAME_(gdb_reg_read),
		.mem_read = NAME_(gdb_mem_read),
		.mem_write = NAME_(gdb_mem_write),
		.brk_set = NAME_(gdb_brk_set),
		.brk_clr = NAME_(gdb_brk_clr),
	};

	cpssp->NAME.gdb = gdb_create(name, cpssp, &NAME_(funcs));

	cpssp->NAME.brkstart[0] = 0; cpssp->NAME.brkend[0] = 0;
	cpssp->NAME.brkstart[1] = 0; cpssp->NAME.brkend[1] = 0;
	cpssp->NAME.brkstart[2] = 0; cpssp->NAME.brkend[2] = 0;
	cpssp->NAME.brkstart[3] = 0; cpssp->NAME.brkend[3] = 0;
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
	gdb_destroy(cpssp->NAME.gdb);
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
#undef DEBUG_CONTROL_FLOW_REGS
