/*===========================================================================
  Copyright (C) 1991-2009 European Southern Observatory (ESO)
 
  This program is free software; you can redistribute it and/or 
  modify it under the terms of the GNU General Public License as 
  published by the Free Software Foundation; either version 2 of 
  the License, or (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public 
  License along with this program; if not, write to the Free 
  Software Foundation, Inc., 675 Massachusetts Ave, Cambridge, 
  MA 02139, USA.
 
  Correspondence concerning ESO-MIDAS should be addressed as follows:
	Internet e-mail: midas@eso.org
	Postal address: European Southern Observatory
			Data Management Division 
			Karl-Schwarzschild-Strasse 2
			D 85748 Garching bei Muenchen 
			GERMANY
===========================================================================*/

/*+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
.IDENT        splint.c
.MODULE       subroutines -- sprebin.exe
.ENVIRONMENT  UNIX
.LANGUAGE     C
.AUTHOR       Cristian Levin - ESO La Silla
.PURPOSE      This file contains two routines for making spline
              interpolation:
                  - using Hermite polynomials.
                  - using natural-cubic spline.
              The first one is currently used for rebinning.
.KEYWORDS     rebin, spline interpolation.
.VERSION 1.0  1-Apr-1991   Implementation

 090723		last modif
------------------------------------------------------------*/

#include <stdio.h>
#include <math.h>

extern void nrerror();



/***************************************************************
 *
 * splint(): spline interpolation based on Hermite polynomials.
 *
 ***************************************************************/

double splint( xp, x, y, n, istart )
double xp;	/* x-value to interpolate */
double *x;	/* x-array [1..n] */
float  *y;	/* y-array [1..n] */
int n;	
int *istart;	/* initial index */
{
    double yp1, yp2, yp;
    double xpi, xpi1, l1, l2, lp1, lp2;
    int i;

    if ( x[1] <= x[n] && (xp < x[1] || xp > x[n]) )
	return(0.0);
    if ( x[1] > x[n] && (xp > x[1] || xp < x[n]) )
	return(0.0);

    if ( x[1] <= x[n] )
    	for ( i = *istart; i <= n && xp >= x[i]; i++ )
	    ;
    else
    	for ( i = *istart; i <= n && xp <= x[i]; i++ )
	    ;

    *istart = i;
    i--;
    
    lp1 = 1.0 / (x[i] - x[i+1]);
    lp2 = -lp1;

    if ( i == 1 )
	yp1 = (y[2] - y[1]) / (x[2] - x[1]);
    else
	yp1 = (y[i+1] - y[i-1]) / (x[i+1] - x[i-1]);

    if ( i >= n - 1 )
	yp2 = (y[n] - y[n-1]) / (x[n] - x[n-1]);
    else
	yp2 = (y[i+2] - y[i]) / (x[i+2] - x[i]);

    xpi1 = xp - x[i+1];
    xpi  = xp - x[i];
    l1   = xpi1*lp1;
    l2   = xpi*lp2;

    yp = y[i]*(1 - 2.0*lp1*xpi)*l1*l1 + 
         y[i+1]*(1 - 2.0*lp2*xpi1)*l2*l2 + 
         yp1*xpi*l1*l1 + yp2*xpi1*l2*l2;

    return(yp);
}

/***************************************************************
 *
 * splint2(): natural cubic-spline interpolation.
 *
 ***************************************************************/

double splint2( xp, x, y, y2, n, kstart )
double xp;	/* x-value to interpolate */
double *x;	/* x-array [1..n] */
float  *y;	/* y-array [1..n] */
float  *y2;	/* y2-array [1..n] (2-nd derivatives) */
int n;	
int *kstart;	/* initial index */
{
    int klo, khi, k;
    double a, b, h, yp;

    klo = *kstart;
    khi = n;

    if ( xp < x[1] || xp > x[n] )
	return(0.0);
    else if ( xp == x[1] )
	return(y[1]);

    for ( k = klo; k < n && xp > x[k]; k++ )
	;

    klo = *kstart = k-1;
    khi = k;

    h = x[khi] - x[klo];
    if ( h == 0.0 )
	nrerror( "Error in spline interpolation" );

    a = (x[khi] - xp) / h;
    b = (xp - x[klo]) / h;

    yp = a*y[klo] + b*y[khi] + ((a*a*a - a)*y2[klo] + (b*b*b - b)*y2[khi])*
	 (h*h) / 6.0;

    return(yp);
}
