"""This module defines a Form class that wraps FFC forms and UFC forms
into a cpp.Form (dolfin::Form)."""

# Copyright (C) 2008 Johan Hake
#
# This file is part of DOLFIN.
#
# DOLFIN is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# DOLFIN is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with DOLFIN. If not, see <http://www.gnu.org/licenses/>.
#
# Modified by Anders Logg, 2008-2011
# Modified by Marie E. Rognes, 2011
#
# First added:  2008-12-04
# Last changed: 2009-12-11

__all__ = ["Form"]

# Import SWIG-generated extension module (DOLFIN C++)
import dolfin.cpp as cpp

# Import ufl and ufc
import ufl, ufc

# Import JIT compiler
from dolfin.compilemodules.jit import jit

from ufl import replace_integral_domains
from dolfin.cpp import deprecation

# Note that we need to store _compiled_form and _compiled_coefficients
# to prevent Python from garbage-collecting these while still in use.
# FIXME: Figure out how to solve this with shared_ptr

def get_common_domain(mesh, common_cell): # TODO: Remove common_cell argument after 1.4 release
    if mesh is not None:
        if common_cell is not None:
            raise RuntimeError("Please provide only mesh, common_cell is deprecated.")
        return mesh.ufl_domain()
    else:
        if common_cell is None:
            return None
        else:
            return ufl.as_domain(common_cell)

class Form(cpp.Form):
    def __init__(self, form,
                 form_compiler_parameters=None,
                 # TODO: Remove these after 1.4 release:
                 function_spaces=None,
                 coefficients=None,
                 subdomains=None,
                 common_cell=None,
                 mesh=None):
        "Create JIT-compiled form from any given form (compiled or not)."

        # NOTE [martinal]: I have a branch where I'm severely rewriting this, please
        #                  don't modify this code now without keeping me in the loop!

        if common_cell is not None:
            deprecation("Form", "1.4.0", "1.5.0",
                        "The common_cell argument is not to be used, pass a mesh or make the form unambiguous instead.")

        # Check form argument
        if isinstance(form, ufl.Form):
            if mesh is not None:
                deprecation("Form", "1.4.0", "1.5.0",
                            "The mesh argument is not to be used with a ufl form, include this in the form instead.")
            if function_spaces is not None:
                deprecation("Form", "1.4.0", "1.5.0",
                            "The function_spaces argument is not to be used with a ufl form, include this in the form instead.")
            if coefficients is not None:
                deprecation("Form", "1.4.0", "1.5.0",
                            "The coefficients argument is not to be used with a ufl form, include this in the form instead.")
            if subdomains is not None:
                deprecation("Form", "1.4.0", "1.5.0",
                            "The subdomains argument is not to be used with a ufl form, include this in the form instead.")

            # Complete form with domain before sending to jit
            common_domain = get_common_domain(mesh, common_cell)
            if common_domain is not None:
                form = replace_integral_domains(form, common_domain)

            if not form.empty():
                assert form.domains(), "Expecting a completed form with domains at this point."

            self._compiled_form, module, self.form_data, prefix \
                                 = jit(form, form_compiler_parameters)
            # If we got no mesh, try to get mesh from domain data,
            # to handle e.g. assemble(1*dx(mesh))
            if mesh is None:
                fd = form.form_data()
                domains = fd.preprocessed_form.domains()
                if len(domains) == 1:
                    # Note: mesh here could still be None, depends on what the user did
                    mesh = domains[0].data()

        elif isinstance(form, ufc.form):
            cpp.deprecation("Passing a ufc.form to Form constructor",
                            "1.3.0", "1.4.0",
                            "Passing a ufc.form to Form constructor"
                            " will be removed.")
            self._compiled_form = form
            self.form_data = None

        elif isinstance(form, cpp.Form):
            cpp.deprecation("Passing a cpp.Form to dolfin.Form constructor",
                            "1.3.0", "1.4.0",
                            "Passing a cpp.Form to dolfin.Form constructor"
                            " will be removed.")
            self._compiled_form = form._compiled_form
            self.form_data = form.form_data

        else:
            cpp.dolfin_error("form.py",
                             "creating dolfin.Form",
                             "Expected a ufl.Form, ufc.form or a dolfin.Form")

        # Extract function spaces
        self.function_spaces = _extract_function_spaces(self.form_data,
                                                        self._compiled_form,
                                                        function_spaces)

        # Extract coefficients
        (self.coefficients, self._compiled_coefficients) = \
            _extract_coefficients(self.form_data, coefficients)

        # Initialize base class
        cpp.Form.__init__(self, self._compiled_form,
                          self.function_spaces, self.coefficients)

        # Extract subdomains from form_data, override if given explicitly
        self.subdomains = _extract_subdomains(self.form_data, subdomains)

        # Attach subdomains if we have them
        subdomains = self.subdomains.get("cell")
        if subdomains is not None:
            self.set_cell_domains(subdomains)
        subdomains = self.subdomains.get("exterior_facet")
        if subdomains is not None:
            self.set_exterior_facet_domains(subdomains)
        subdomains = self.subdomains.get("interior_facet")
        if subdomains is not None:
            self.set_interior_facet_domains(subdomains)

        # Set mesh if specified (important for functionals without a function spaces)
        if mesh is not None:
            self.set_mesh(mesh)

def _extract_function_spaces(form_data, compiled_form, given_function_spaces):
    "Extract list of test spaces."

    # Note: Most of this function is error checking,
    # look for 'function_spaces' to see where stuff happens

    function_space_error = "Error while extracting test and/or trial spaces. "

    function_spaces = []

    r = compiled_form.rank()

    if given_function_spaces:
        # Given function spaces are necessary for assembling pure UFC forms,
        # and possibly also convenient for other rare scenarios.
        if isinstance(given_function_spaces, cpp.FunctionSpace):
            if r > 0:
                # Repeat a single given space
                function_spaces = [given_function_spaces]*r
            else:
                raise ValueError(function_space_error +
                                 "Superfluous test spaces given for a functional.")

        elif isinstance(given_function_spaces, (list,tuple)):
            if all(isinstance(fs, cpp.FunctionSpace) for fs in given_function_spaces):
                # Take given list of spaces
                function_spaces = list(given_function_spaces)
            else:
                raise ValueError(function_space_error +
                                 "Invalid given test spaces.")

        else:
            raise ValueError(function_space_error +
                             "Expecting given_function_spaces to be one or more function spaces.")
    else:
        # No given function spaces is the norm in pydolfin, which means
        # they are to be extracted from the ufl form
        if hasattr(form_data, 'original_arguments'):
            # Get spaces from Arguments found in ufl form
            function_spaces = [func.function_space() for func in form_data.original_arguments]

        elif r == 0:
            # No given spaces, no form_data, but this is a functional,
            # so no spaces needed
            function_spaces = []

        else:
            raise ValueError(function_space_error +
                             "Missing data about basis functions in form data.")

    # Consistency checks
    if len(function_spaces) != r:
        raise ValueError(function_space_error +
              " Wrong number of test spaces (should be %d)." % r)
    if not all(isinstance(fs, cpp.FunctionSpace) for fs in function_spaces):
        raise ValueError(function_space_error +
              " Invalid type of test spaces.")

    return function_spaces

def _extract_coefficients(form_data, given_coefficients):
    "Extract list of coefficients."

    # Note: Most of this function is error checking,
    # look for 'coefficients' and '_compiled_coefficients' to see where stuff happens

    coefficient_error = "Error while extracting coefficients. "

    coefficients = []
    _compiled_coefficients = []

    if given_coefficients:
        # Type checking given coefficients
        error_info = "Provide a 'list' with cpp.GenericFunctions"
        if not isinstance(given_coefficients, list):
            raise TypeError(coefficient_error + error_info)
        if not all(isinstance(c, cpp.GenericFunction) for c in given_coefficients):
            raise TypeError(coefficient_error)

        # Just use the given list
        coefficients = list(given_coefficients)
        _compiled_coefficients = list(given_coefficients)

    elif form_data is None:
        # Nothing to extract
        coefficients = []
        _compiled_coefficients = []

    else:
        # Type checking form data
        if not hasattr(form_data, "original_coefficients"):
            raise TypeError(coefficient_error +
                  "Missing data about coefficients in form data.")
        if not all(isinstance(c, cpp.GenericFunction) for c in form_data.original_coefficients):
            raise TypeError(coefficient_error +
                            "Either provide a dict of cpp.GenericFunctions, " +
                            "or use Function to define your form.")

        # Just use the list from form data
        coefficients = list(form_data.original_coefficients)
        _compiled_coefficients = [] # Is this correct? No need to store double references.

    return (coefficients, _compiled_coefficients)

def _extract_subdomains(form_data, override_subdomains):
    "Extract dict of subdomains per domain type."

    override_subdomains = override_subdomains or {}
    if form_data is None:
        return override_subdomains

    integral_types = ("cell", "exterior_facet", "interior_facet")

    additional_keys = set(override_subdomains.keys()) - set(integral_types)
    if additional_keys:
        raise TypeError, "Invalid keys in integral_types: %s" % additional_keys

    # Temporary hack for getting subdomain_data from new ufl structure,
    # assuming a single domain like it has been up to now.
    # To support multiple domains this needs to change so we
    # get the right subdomains for the right mesh.
    domains = form_data.preprocessed_form.domains()
    assert len(domains) == 1, "This code assumes a single domain per form."
    domain, = domains
    subdomain_data = form_data.subdomain_data[domain.label()]

    all_subdomains = {}
    for integral_type in integral_types:
        subdomains = override_subdomains.get(integral_type)
        if subdomains is None:
            subdomains = subdomain_data.get(integral_type)

        if subdomains is not None:
            # FIXME: Add test for MeshFunctions here. Like this?
            #if not isinstance(subdomains, MeshFunction):
            #    raise TypeError, "Invalid subdomains type %s" % type(subdomains)
            all_subdomains[integral_type] = subdomains

    return all_subdomains
