# Copyright (C) 2008-2011 Dejan Muhamedagic <dmuhamedagic@suse.de>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2 of the License, or (at your option) any later version.
#
# This software is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
#

import shlex
import re
from lxml import etree
import vars
from ra import disambiguate_ra_type, ra_type_validate
from schema import Schema
from utils import keyword_cmp, verify_boolean, lines2cli
from utils import get_boolean, olist, is_boolean_true
from msg import common_err, syntax_err
from cibobjects import Primitive, RscTemplate, Group, Clone, Master
from cibobjects import Location, Colocation, Order
from cibobjects import Monitor, Node, Property, RscTicket
from cibobjects import FencingTopology, ACLRight, Role, User, RawXML


class ParseError(Exception):
    '''
    Raised by parsers when parsing fails.
    No error message, parsers should write
    error messages before raising the exception.
    '''


class BaseParser(object):
    _NVPAIR_RE = re.compile(r'([^=]+)=(.+)$')
    _IDENT_RE = re.compile(r'([a-z0-9_#$-].*)$', re.IGNORECASE)
    _DISPATCH_RE = re.compile(r'[a-z0-9_]+$', re.IGNORECASE)
    _DESC_RE = re.compile(r'description=(.+)$', re.IGNORECASE)
    _RESOURCE_RE = re.compile(r'([a-z_#$][^=]*)$', re.IGNORECASE)
    _IDSPEC_RE = re.compile(r'(\$id-ref|\$id)=(.*)$', re.IGNORECASE)

    def can_parse(self):
        "Returns a list of commands this parser understands"
        raise NotImplementedError

    def parse(self, cmd):
        "Called by do_parse(). Raises ParseError if parsing fails."
        raise NotImplementedError

    def init(self, validation):
        self.validation = validation

    def err(self, errmsg):
        "Report a parse error and abort."
        token = None
        if self.has_tokens():
            token = self._cmd[self._currtok]
        syntax_err(self._cmd, context=self._cmd[0], token=token, msg=errmsg)
        raise ParseError

    def begin(self, cmd, min_args=-1):
        self._cmd = cmd
        self._currtok = 0
        self._lastmatch = None
        if min_args > -1 and len(cmd) < min_args + 1:
            self.err("Expected at least %d arguments" % (min_args))

    def begin_dispatch(self, cmd, min_args=-1):
        """
        Begin parsing cmd.
        Dispatches to parse_<resource> based on the first token.
        """
        self.begin(cmd, min_args=min_args)
        return self.match_dispatch(errmsg="Unknown command")

    def do_parse(self, cmd):
        """
        Called by CliParser. Calls parse()
        Parsers should pass their return value through this method.
        """
        out = self.parse(cmd)
        if self.has_tokens():
            self.err("Unknown arguments: " + ' '.join(self._cmd[self._currtok:]))
        return out

    def try_match(self, rx):
        """
        Try to match the given regex with the curren token.
        rx: compiled regex or string
        returns: the match object, if the match is successful
        """
        tok = self.current_token()
        if not tok:
            return None
        if isinstance(rx, basestring):
            if not rx.endswith('$'):
                rx = rx + '$'
            self._lastmatch = re.match(rx, tok, re.IGNORECASE)
        else:
            self._lastmatch = rx.match(tok)
        if self._lastmatch is not None:
            if not self.has_tokens():
                self.err("Unexpected end of line")
            self._currtok += 1
        return self._lastmatch

    def match(self, rx, errmsg=None):
        """
        Match the given regex with the current token.
        If match fails, parse is aborted and an error reported.
        rx: compiled regex or string.
        errmsg: optional error message if match fails.
        Returns: The matched token.
        """
        if not self.try_match(rx):
            if errmsg:
                self.err(errmsg)
            elif isinstance(rx, basestring):
                self.err("Expected " + rx)
            else:
                self.err("Expected " + rx.pattern.rstrip('$'))
        return self.matched(0)

    def matched(self, idx=0):
        """
        After a successful match, returns
        the groups generated by the match.
        """
        return self._lastmatch.group(idx)

    def lastmatch(self):
        return self._lastmatch

    def rewind(self):
        "useful for when validation fails, to undo the match"
        if self._currtok > 0:
            self._currtok -= 1

    def current_token(self):
        if self.has_tokens():
            return self._cmd[self._currtok]
        return None

    def has_tokens(self):
        return self._currtok < len(self._cmd)

    def match_rest(self):
        '''
        matches and returns the rest
        of the tokens in a list
        '''
        ret = self._cmd[self._currtok:]
        self._currtok = len(self._cmd)
        return ret

    def match_any(self):
        if not self.has_tokens():
            self.err("Unexpected end of line")
        tok = self.current_token()
        self._currtok += 1
        self._lastmatch = tok
        return tok

    def match_nvpairs(self, minpairs=1):
        """
        Matches string of p=v tokens
        """
        ret = []
        while self.try_match(self._NVPAIR_RE):
            ret.append((self.matched(1), self.matched(2)))
        if len(ret) < minpairs:
            if minpairs == 1:
                self.err("Expected at least one name-value pair")
            else:
                self.err("Expected at least %d name-value pairs" % (minpairs))
        return ret

    def try_match_nvpairs(self, name):
        """
        Matches sequence of <name> [<key>=<value> [<key>=<value> ...] ...]
        """
        if self.try_match(name):
            self._lastmatch = self.match_nvpairs(minpairs=1)
        else:
            self._lastmatch = []
        return self._lastmatch

    def match_identifier(self):
        return self.match(self._IDENT_RE, errmsg="Expected identifier")

    def match_resource(self):
        return self.match(self._RESOURCE_RE, errmsg="Expected resource")

    def match_idspec(self):
        """
        matches $id=<id> | $id-ref=<id>
        matched(1) = $id|$id-ref
        matched(2) = <id>
        """
        return self.match(self._IDSPEC_RE, errmsg="Expected $id-ref=<id> or $id=<id>")

    def try_match_idspec(self):
        """
        matches $id=<value> | $id-ref=<value>
        matched(1) = $id|$id-ref
        matched(2) = <value>
        """
        return self.try_match(self._IDSPEC_RE)

    def match_split(self, sep=':', order=None):
        """
        matches *:*:*...
        order is an optional list of indices (0-based),
        order (0, 1, 2) looks for a max-three split, and returns
        (0, 1, 2) if string splits into (0, 1, 2)
        (0, 1, None) if string splits into (0, 1)
        (0, None, None) if string splits into (0)
        order (2, 1, 0) returns
        (foo, bar, baz) if string splits into (foo, bar, baz)
        (None, foo, bar) if string splits into (foo, bar)
        (None, None, foo) if string splits into (foo)
        """
        if not order:
            if not self.current_token():
                self.err("Expected val[%sval ...]" % (sep))
            return self.match_any().split(sep)
        if not self.current_token():
            fmt = "Expected val[%sval ...] (max %d parts)"
            self.err(fmt % (sep, len(order)))
        sp = self.current_token().split(sep)
        if len(sp) > len(order):
            fmt = "Expected val[%sval ...] (max %d parts)"
            self.err(fmt % (sep, len(order)))
        try:
            while len(sp) < len(order):
                sp.insert(order.index(len(sp)), None)
        except ValueError:
            self.err("Internal error, please file a bug report (order = %s)" %
                     (repr(order)))
        self.match_any()
        return sp

    def match_dispatch(self, errmsg=None):
        """
        Match on the next token. Looks
        for a method named parse_<token>.
        If found, the named function is called.
        Else, an error is reported.
        """
        t = self.match(self._DISPATCH_RE, errmsg=errmsg)
        t = 'parse_' + t.lower()
        if hasattr(self, t) and callable(getattr(self, t)):
            return getattr(self, t)()
        self.rewind()  # rewind for more accurate error message
        self.err(errmsg)

    def try_match_description(self):
        """
        reads a description=? token if one is next
        """
        if self.try_match(self._DESC_RE):
            return self.matched(1)
        return None


class NodeParser(BaseParser):
    _ID_RE = re.compile(r'\$id=(.+)$', re.IGNORECASE)
    _UNAME_RE = re.compile(r'([^:]+)(:(normal|member|ping))?$', re.IGNORECASE)

    def can_parse(self):
        return ('node',)

    def match_arguments(self, out, names):
        """matches [<name> <n>=<v> <n>=<v> ...]
        adds the nvpairs to dicts named out.<name>
        """
        names = olist(names)
        while self.has_tokens() and (self.current_token() in names):
            for name in names:
                for n, v in self.try_match_nvpairs(name):
                    getattr(out, name)[n] = v

    def parse(self, cmd):
        """
        node [$id=<id>] <uname>[:<type>]
          [description=<description>]
          [attributes <param>=<value> [<param>=<value>...]]
          [utilization <param>=<value> [<param>=<value>...]]

        type :: normal | member | ping
        """
        self.begin(cmd, min_args=1)
        self.match('node')
        out = Node()
        if self.try_match(self._ID_RE):
            out.id = self.matched(1)
        self.match(self._UNAME_RE, errmsg="Expected uname[:type]")
        out.uname, out.type = self.matched(1), self.matched(3)
        out.description = self.try_match_description()
        self.match_arguments(out, ('attributes', 'utilization'))
        return out


class ResourceParser(BaseParser):
    _TEMPLATE_RE = re.compile(r'@(.+)$')
    _RA_TYPE_RE = re.compile(r'[a-z0-9_:-]+$', re.IGNORECASE)
    _OPTYPE_RE = re.compile(r'(%s)$' % ('|'.join(vars.op_cli_names)), re.IGNORECASE)

    def can_parse(self):
        return ('primitive', 'group', 'clone', 'ms', 'master', 'rsc_template')

    def match_ra_type(self, out):
        "[<class>:[<provider>:]]<type>"
        if not self.current_token():
            self.err("Expected resource type")
        cpt = self.validation.class_provider_type(self.current_token())
        if not cpt:
            self.err("Unknown resource type")
        self.match_any()
        out.ra_class, out.ra_type = cpt[0], cpt[2]
        if out.ra_class == 'ocf':
            out.ra_provider = cpt[1]

    def match_op(self, out):
        "op <optype> [<n>=<v> ...]"
        self.match('op')
        op_type = self.match(self._OPTYPE_RE, errmsg="Expected operation type")
        op_attrs = self.match_nvpairs(minpairs=0)
        has_interval = False
        for kv in op_attrs:
            if kv[0].lower() == 'interval':
                has_interval = True
                break
        if not has_interval:
            op_attrs.append(('interval', '0'))
        out.operations.append((op_type, op_attrs))

    def match_operations(self, out):
        self.match('operations')
        self.match_idspec()
        out.operations.append((self.matched(1).lower(), self.matched(2)))
        if self.matched(1).lower() == '$id':
            self.match_op(out)  # require at least one op
            while self.has_tokens() and self.current_token().lower() == 'op':
                self.match_op(out)

    def match_attr_list(self, name, out):
        attr_list = self.try_match_nvpairs(name)
        if not attr_list:
            return False
        # handle $id and $id-ref
        attr0 = attr_list[0][0].lower()
        if attr0 == '$id-ref':
            if len(attr_list) > 1:
                self.err("Only a single $id-ref is allowed")
        elif attr0 == '$id':
            if len(attr_list) < 2:
                self.err("At least one param is required")
        for n, v in attr_list:
            getattr(out, name)[n] = v
        return True

    def match_arguments(self, out, names):
        """
        [<name> attr_list]
        [operations id_spec]
        [op op_type [<attribute>=<value> ...] ...]

        attr_list :: [$id=<id>] <attr>=<val> [<attr>=<val>...] | $id-ref=<id>
        id_spec :: $id=<id> | $id-ref=<id>
        op_type :: start | stop | monitor
        """
        names = olist(names)
        while self.has_tokens() and self.current_token() in names:
            t = self.current_token()
            if t == 'operations':
                self.match_operations(out)
            elif t.lower() == 'op':
                self.match_op(out)
            else:
                for name in names:
                    if self.match_attr_list(name, out):
                        break

    def parse(self, cmd):
        return self.begin_dispatch(cmd, min_args=2)

    def _primitive_or_template(self):
        """
        primitive <rsc> {[<class>:[<provider>:]]<type>|@<template>]
          [params attr_list]
          [meta attr_list]
          [utilization attr_list]
          [operations id_spec]
          [op op_type [<attribute>=<value> ...] ...]

        attr_list :: [$id=<id>] <attr>=<val> [<attr>=<val> ...] | $id-ref=<id>
        id_spec :: $id=<id> | $id-ref=<id>
        op_type :: start | stop | monitor
        """
        t = self.matched(0).lower()
        if t == 'primitive':
            out = Primitive()
        else:
            out = RscTemplate()
        out.id = self.match_identifier()
        if t == 'primitive' and self.try_match(self._TEMPLATE_RE):
            out.template = self.matched(1)
        else:
            self.match_ra_type(out)
        out.description = self.try_match_description()
        self.match_arguments(out, ('params', 'meta', 'utilization', 'operations', 'op'))
        return out

    parse_primitive = _primitive_or_template
    parse_rsc_template = _primitive_or_template

    def _master_or_clone(self):
        if self.matched(0).lower() == 'clone':
            out = Clone()
        else:
            out = Master()
        out.id = self.match_identifier()
        out.children = [self.match_resource()]
        out.description = self.try_match_description()
        self.match_arguments(out, ('params', 'meta'))
        return out

    parse_master = _master_or_clone
    parse_ms = _master_or_clone
    parse_clone = _master_or_clone

    def _try_group_resource(self):
        t = self.current_token()
        if (not t) or ('=' in t) or (t.lower() in ('params', 'meta')):
            return None
        return self.match_any()

    def parse_group(self):
        out = Group()
        out.id = self.match_identifier()
        while self._try_group_resource():
            child = self.lastmatch()
            if child in out.children:
                self.err("child %s listed more than once in group %s" %
                         (child, out.id))
            out.children.append(child)
        out.description = self.try_match_description()
        self.match_arguments(out, ('params', 'meta'))
        return out


class ConstraintParser(BaseParser):
    _SCORE_RE = re.compile(r"([^:]+):$")
    _ROLE_RE = re.compile(r"\$role=(.+)$", re.IGNORECASE)
    _ROLE2_RE = re.compile(r"role=(.+)$", re.IGNORECASE)
    _BOOLOP_RE = re.compile(r'(%s)$' % ('|'.join(vars.boolean_ops)), re.IGNORECASE)
    _UNARYOP_RE = re.compile(r'(%s)$' % ('|'.join(vars.unary_ops)), re.IGNORECASE)
    _BINOP_RE = None

    def can_parse(self):
        return ('location', 'colocation', 'collocation', 'order', 'rsc_ticket')

    def parse(self, cmd):
        return self.begin_dispatch(cmd, min_args=2)

    def match_rules(self, out):
        '''parse rule definitions'''
        while self.try_match('rule'):
            head = []
            rule = ['rule', head]
            idref = False
            if self.try_match_idspec():
                head.append([self.matched(1), self.matched(2)])
                if self.matched(1) == '$id-ref':
                    idref = True
            if self.try_match(self._ROLE_RE):
                head.append(['$role', self.matched(1)])
            if idref:
                out.rules.append(rule)
                continue
            self.match(self._SCORE_RE)
            score = self.validate_score(self.matched(1))
            head.append(score)
            boolop, expr = self.match_rule_expression()
            if boolop and not keyword_cmp(boolop, 'and'):
                head.append(['boolean-op', boolop])
            out.rules.append(rule)
            out.rules.extend(expr)

    def match_rule_expression(self):
        """
        expression :: <simple_exp> [bool_op <simple_exp> ...]
        bool_op :: or | and
        simple_exp :: <attribute> [type:]<binary_op> <value>
                      | <unary_op> <attribute>
                      | date <date_expr>
        type :: string | version | number
        binary_op :: lt | gt | lte | gte | eq | ne
        unary_op :: defined | not_defined

        date_expr :: lt <end>
                     | gt <start>
                     | in_range start=<start> end=<end>
                     | in_range start=<start> <duration>
                     | date_spec <date_spec>
        duration|date_spec ::
                     hours=<value>
                     | monthdays=<value>
                     | weekdays=<value>
                     | yearsdays=<value>
                     | months=<value>
                     | weeks=<value>
                     | years=<value>
                     | weekyears=<value>
                     | moon=<value>
        """
        boolop = None
        exprs = [self._match_simple_exp()]
        while self.try_match(self._BOOLOP_RE):
            if boolop and self.matched(1) != boolop:
                self.err("Mixing bool ops not allowed: %s != %s" % (boolop, self.matched(1)))
            else:
                boolop = self.matched(1)
            exprs.append(self._match_simple_exp())
        return boolop, exprs

    def _match_simple_exp(self):
        if self.try_match('date'):
            return ['date_expression', self.match_date()]
        elif self.try_match(self._UNARYOP_RE):
            unary_op = self.matched(1)
            attr = self.match_identifier()
            return ['expression', [['operation', unary_op], ['attribute', attr]]]
        else:
            attr = self.match_identifier()
            if not self._BINOP_RE:
                self._BINOP_RE = re.compile(r'((%s):)?(%s)$' % (
                    '|'.join(self.validation.expression_types()),
                    '|'.join(vars.binary_ops)), re.IGNORECASE)
            self.match(self._BINOP_RE)
            optype = self.matched(2)
            binop = self.matched(3)
            val = self.match_any()
            ret = ['expression', [['attribute', attr]]]
            if optype:
                ret[1].extend([['type', optype], ['operation', binop], ['value', val]])
            else:
                ret[1].extend([['operation', binop], ['value', val]])
            return ret

    def match_date(self):
        """
        returns: [['operation', <op>], (n, v), (n, v)]
        """
        self.match('(%s)$' % ('|'.join(self.validation.date_ops())))
        op = self.matched(1)
        if op in olist(vars.simple_date_ops):
            val = self.match_any()
            if keyword_cmp(op, 'lt'):
                return [['operation', op], ('end', val)]
            return [['operation', op], ('start', val)]
        elif op == 'in_range':
            nv = self.match_nvpairs(minpairs=2)
            if not keyword_cmp(nv[0][0], 'start'):
                self.err("Expected start=<start>")
            return [['operation', op]] + nv
        else:  # lt, gt, date_spec, ?
            return [['operation', op]] + self.match_nvpairs(minpairs=1)

    def validate_score(self, score, noattr=False):
        if not noattr and score in olist(vars.score_types):
            return vars.score_types[score.lower()]
        elif re.match("^[+-]?(inf(inity)?|INF(INITY)?|[0-9]+)$", score):
            score = re.sub("inf(inity)?|INF(INITY)?", "INFINITY", score)
            return ["score", score]
        if noattr:
            # orders have the special kind attribute
            kind = self.validation.canonize(score, self.validation.rsc_order_kinds())
            if not kind:
                self.err("Invalid kind: " + score)
            return ['kind', kind]
        else:
            return ['score-attribute', score]

    def match_until(self, end_token):
        tokens = []
        while self.current_token() is not None and self.current_token() != end_token:
            tokens.append(self.match_any())
        return tokens

    def parse_location(self):
        """
        location <id> rsc <score>: <node> [role=<role>]
        location <id> rsc [rule ...]
        rsc :: /<rsc-pattern>/
            | { <rsc-set> }
            | <rsc>
        """
        out = Location()
        out.id = self.match_identifier()
        if self.try_match('^/(.+)/$'):
            out.rsc_pattern = self.matched(1)
        elif self.try_match('{'):
            tokens = self.match_until('}')
            self.match('}')
            if not tokens:
                self.err("Empty resource set")
            parser = ResourceSet('role', tokens, self)
            out.rsc_set = parser.parse()
        else:
            out.resource = self.match_resource()
        if self.try_match(self._SCORE_RE):
            out.score = self.validate_score(self.matched(1))
            out.node = self.match_identifier()
            if self.try_match(self._ROLE_RE):
                out.role = self.matched(1)
            elif self.try_match(self._ROLE2_RE):
                out.role = self.matched(1)
        else:
            self.match_rules(out)
        return out

    def parse_colocation(self):
        """
        colocation <id> <score>: <rsc>[:<role>] <rsc>[:<role>] ...
          [node-attribute=<node_attr>]
        """
        out = Colocation()
        out.id = self.match_identifier()
        self.match(self._SCORE_RE)
        out.score = self.validate_score(self.matched(1))
        if self.try_match_tail('node-attribute=(.+)$'):
            out.node_attribute = self.matched(1).lower()
        out.simple, out.resources = self.match_resource_set('role')
        return out

    parse_collocation = parse_colocation

    def parse_order(self):
        '''
        order <id> {kind|<score>}: <rsc>[:<action>] <rsc>[:<action>] ...
          [symmetrical=<bool>]

        kind :: Mandatory | Optional | Serialize
        '''
        out = Order()
        out.id = self.match_identifier()
        if self.try_match('(%s):$' % ('|'.join(self.validation.rsc_order_kinds()))):
            out.kind = self.validation.canonize(self.matched(1), self.validation.rsc_order_kinds())
        else:
            self.match(self._SCORE_RE)
            out.score = self.validate_score(self.matched(1), noattr=True)
        if self.try_match_tail('symmetrical=(true|false|yes|no|on|off)$'):
            out.symmetrical = is_boolean_true(self.matched(1))
        out.simple, out.resources = self.match_resource_set('action')
        return out

    def parse_rsc_ticket(self):
        '''
        rsc_ticket <id> <ticket_id>: <rsc>[:<role>] [<rsc>[:<role>] ...]
        [loss-policy=<loss_policy_action>]

        loss_policy_action :: stop | demote | fence | freeze
        '''
        out = RscTicket()
        out.id = self.match_identifier()
        self.match(self._SCORE_RE)
        out.ticket = self.matched(1)
        if self.try_match_tail('loss-policy=(stop|demote|fence|freeze)$'):
            out.loss_policy = self.matched(1)
        out.simple, out.resources = self.match_resource_set('role', simple_count=1)
        return out

    def try_match_tail(self, rx):
        "ugly hack to prematurely extract a tail attribute"
        pos = self._currtok
        self._currtok = len(self._cmd) - 1
        ret = self.try_match(rx)
        if ret:
            self._cmd = self._cmd[:-1]
        self._currtok = pos
        return ret

    def remaining_tokens(self):
        return len(self._cmd) - self._currtok

    def match_resource_set(self, suffix_type, simple_count=2):
        simple = False
        if self.remaining_tokens() == simple_count:
            simple = True
            if suffix_type == 'role':
                return True, self.match_simple_role_set(simple_count)
            else:
                return True, self.match_simple_action_set()
        tokens = self.match_rest()
        parser = ResourceSet(suffix_type, tokens, self)
        return simple, parser.parse()

    def match_simple_role_set(self, count):
        def rsc_role():
            rsc, role = self.match_split(order=(0, 1))
            role, t = self.validation.classify_role(role)
            if role and not t:
                self.err("Invalid role '%s' for '%s'" % (role, rsc))
            return rsc, role, t

        def fmt(info, name):
            if info[1]:
                return [[name, info[0]], [name + '-' + info[2], info[1]]]
            return [[name, info[0]]]
        ret = fmt(rsc_role(), 'rsc')
        if count == 2:
            ret += fmt(rsc_role(), 'with-rsc')
        return ret

    def match_simple_action_set(self):
        def rsc_action():
            rsc, action = self.match_split(order=(0, 1))
            action, t = self.validation.classify_action(action)
            if action and not t:
                self.err('invalid action: ' + action)
            return rsc, action, t

        def fmt(info, name):
            if info[1]:
                return [[name, info[0]], [name + '-' + info[2], info[1]]]
            return [[name, info[0]]]
        ret = fmt(rsc_action(), 'first')
        return ret + fmt(rsc_action(), 'then')


class OpParser(BaseParser):
    def can_parse(self):
        return ('monitor',)

    def parse(self, cmd):
        return self.begin_dispatch(cmd, min_args=2)

    def parse_monitor(self):
        out = Monitor()
        out.resource, out.role = self.match_split(order=(0, 1))
        if out.role:
            out.role, out.role_class = self.validation.classify_role(out.role)
            if not out.role_class:
                self.err("Invalid role '%s' for resource '%s'" % (out.role, out.resource))
        out.interval, out.timeout = self.match_split(order=(0, 1))
        return out


class PropertyParser(BaseParser):
    def can_parse(self):
        return ('property', 'rsc_defaults', 'op_defaults')

    def parse(self, cmd):
        self.begin(cmd, min_args=1)
        self.match('(%s)$' % '|'.join(self.can_parse()))
        out = Property()
        out.type = self.matched(1)
        if self.try_match_idspec():
            out.values.append(('$id', self.matched(2)))
        out.values.extend(self.match_nvpairs())
        return out


class FencingOrderParser(BaseParser):
    _TARGET_RE = re.compile(r'([^:]+):$')

    def can_parse(self):
        return ('fencing_topology',)

    def parse(self, cmd):
        self.begin(cmd, min_args=1)
        self.match("fencing_topology")
        target = "@@"
        out = FencingTopology()
        while self.has_tokens():
            if self.try_match(self._TARGET_RE):
                target = self.matched(1)
            else:
                out.add_level(target, self.match_any())
        return out


class AclParser(BaseParser):
    _ACL_RIGHT_RE = re.compile(r'(%s)$' % ('|'.join(vars.acl_rule_names)), re.IGNORECASE)
    _ROLE_REF_RE = re.compile(r'role:(.+)$', re.IGNORECASE)

    def can_parse(self):
        return ('user', 'role')

    def parse(self, cmd):
        return self.begin_dispatch(cmd, min_args=2)

    def parse_role(self):
        out = Role()
        out.role_id = self.match_identifier()
        while self.has_tokens():
            out.rules.append(self._add_rule())
        return out

    def parse_user(self):
        out = User()
        out.uid = self.match_identifier()
        while self.has_tokens():
            # role identifier
            if self.try_match(self._ROLE_REF_RE):
                out.roles.append(self.matched(1))
            # acl right rule
            else:
                out.rules.append(self._add_rule())
        return out

    def _add_rule(self):
        rule = ACLRight()
        rule.right = self.match(self._ACL_RIGHT_RE).lower()
        eligible_specs = vars.acl_spec_map.values()
        while self.has_tokens():
            a = self._expand_shortcuts(self.current_token().split(':', 1))
            if len(a) != 2 or a[0] not in eligible_specs:
                break
            self.match_any()
            rule.specs.append((a[0], a[1]))
            if self._remove_spec(eligible_specs, a[0]):
                break
        return rule

    def _remove_spec(self, speclist, spec):
        """
        Remove spec from list of eligible specs.
        Returns true if spec parse is complete.
        """
        try:
            speclist.remove(spec)
            if spec == 'xpath':
                speclist.remove('ref')
                speclist.remove('tag')
            elif spec in ('ref', 'tag'):
                speclist.remove('xpath')
            else:
                return True
        except ValueError:
            pass
        return False

    def _expand_shortcuts(self, l):
        '''
        Expand xpath shortcuts. The input list l contains the user
        input. If no shortcut was found, just return l.
        In case of syntax error, return empty list. Otherwise, l[0]
        contains 'xpath' and l[1] the expansion as found in
        vars.acl_shortcuts. The id placeholders '@@' are replaced
        with the given attribute names or resource references.
        '''
        try:
            expansion = vars.acl_shortcuts[l[0]]
        except KeyError:
            return l
        l[0] = "xpath"
        if len(l) == 1:
            if '@@' in expansion[0]:
                return []
            l.append(expansion[0])
            return l
        a = l[1].split(':')
        xpath = ""
        exp_i = 0
        for tok in a:
            try:
                # some expansions may contain no id placeholders
                # of course, they don't consume input tokens
                if '@@' not in expansion[exp_i]:
                    xpath += expansion[exp_i]
                    exp_i += 1
                xpath += expansion[exp_i].replace('@@', tok)
                exp_i += 1
            except:
                return []
        # need to remove backslash chars which were there to escape
        # special characters in expansions when used as regular
        # expressions (mainly '[]')
        l[1] = xpath.replace("\\", "")
        return l


class RawXMLParser(BaseParser):
    def can_parse(self):
        return ('xml',)

    def parse(self, cmd):
        self.begin(cmd, min_args=1)
        self.match('xml')
        if not self.has_tokens():
            self.err("Expected XML data")
        xml_data = ' '.join(self.match_rest())
        # strip spaces between elements
        # they produce text elements
        try:
            e = etree.fromstring(xml_data)
        except Exception, e:
            common_err("Cannot parse XML data: %s" % xml_data)
            self.err(e)
        try:
            el_type = vars.cib_cli_map[e.tag]
        except:
            self.err("Element %s not recognized" % (e.tag))
        out = RawXML()
        out.id = e.get("id")
        out.type = el_type
        out.raw = xml_data
        return out


class ResourceSet(object):
    '''
    Constraint resource set parser. Parses sth like:
    a ( b c:start ) d:Master e ...
    Appends one or more lists to cli_list.
    Lists are in form:
        list :: ["resource_set", set_pl]
        set_pl :: [["sequential","false"], ["action"|"role", action|role],
            ["resource_ref", ["id", rsc]], ...]
        (the first two elements of set_pl are optional)
    Action/role change makes a new resource set.
    '''
    open_set = ('(', '[')
    close_set = (')', ']')
    matching = {
        '[': ']',
        '(': ')',
    }

    def __init__(self, type, s, parent):
        self.parent = parent
        self.type = type
        self.q_attr = (type == "order") and "action" or "role"
        self.tokens = s
        self.cli_list = []
        self.reset_set()
        self.opened = ''
        self.sequential = True
        self.require_all = True
        self.fix_parentheses()

    def fix_parentheses(self):
        newtoks = []
        for p in self.tokens:
            if p[0] in self.open_set and len(p) > 1:
                newtoks.append(p[0])
                newtoks.append(p[1:])
            elif p[len(p)-1] in self.close_set and len(p) > 1:
                newtoks.append(p[0:len(p)-1])
                newtoks.append(p[len(p)-1])
            else:
                newtoks.append(p)
        self.tokens = newtoks

    def reset_set(self):
        self.set_pl = []
        self.prev_q = ''  # previous qualifier (action or role)
        self.curr_attr = ''  # attribute (action or role)

    def save_set(self):
        if not self.set_pl:
            return
        if self.curr_attr:
            self.set_pl.insert(0, [self.curr_attr, self.prev_q])
        if not self.sequential:
            self.set_pl.insert(0, ["sequential", "false"])
        if not self.require_all:
            self.set_pl.insert(0, ["require-all", "false"])
        self.cli_list.append(["resource_set", self.set_pl])
        self.reset_set()

    def parseattr(self, p):
        attrs = {"sequential": "sequential",
                 "require-all": "require_all"}
        l = p.split('=')
        if len(l) != 2 or l[0] not in attrs:
            return False
        k, v = l
        if not verify_boolean(v):
            return False
        setattr(self, attrs[k], get_boolean(v))
        return True

    def splitrsc(self, p):
        l = p.split(':')
        if len(l) == 2:
            if self.q_attr == 'action':
                l[1] = self.parent.validation.canonize(
                    l[1],
                    self.parent.validation.resource_actions())
            else:
                l[1] = self.parent.validation.canonize(
                    l[1],
                    self.parent.validation.resource_roles())
            if not l[1]:
                self.err('Invalid %s for %s' % (self.q_attr, p))
        elif len(l) == 1:
            l = [p, '']
        return l

    def err(self, token, errmsg):
        syntax_err(self.parent._cmd,
                   context=self.type,
                   token=token,
                   msg=errmsg)
        raise ParseError

    def update_attrs(self, bracket, tokpos):
        if bracket in ('(', '['):
            if self.opened:
                self.err(token=self.tokens[tokpos],
                         errmsg='Cannot nest resource sets')
            self.sequential = False
            if bracket == '[':
                self.require_all = False
            self.opened = bracket
        elif bracket in (')', ']'):
            if not self.opened:
                self.err(token=self.tokens[tokpos],
                         errmsg='Unmatched closing bracket')
            if bracket != self.matching[self.opened]:
                self.err(token=self.tokens[tokpos],
                         errmsg='Mismatched closing bracket')
            self.sequential = True
            self.require_all = True
            self.opened = ''

    def parse(self):
        tokpos = -1
        for p in self.tokens:
            tokpos += 1
            if p == "_rsc_set_":
                continue  # a degenerate resource set
            if p in self.open_set:
                self.save_set()
                self.update_attrs(p, tokpos)
                continue
            if p in self.close_set:
                # empty sets not allowed
                if not self.set_pl:
                    self.err(token=self.tokens[tokpos],
                             errmsg='Empty resource set')
                self.save_set()
                self.update_attrs(p, tokpos)
                continue
            if '=' in p:
                if not self.parseattr(p):
                    self.err(token=self.tokens[tokpos],
                             errmsg='Unknown attribute')
                continue
            rsc, q = self.splitrsc(p)
            if q != self.prev_q:  # one set can't have different roles/actions
                self.save_set()
                self.prev_q = q
            if q:
                if not self.curr_attr:
                    self.curr_attr = self.q_attr
            else:
                self.curr_attr = ''
            self.set_pl.append(["resource_ref", ["id", rsc]])
        if self.opened:  # no close
            self.err(token=self.tokens[tokpos],
                     errmsg='Unmatched opening bracket')
        if self.set_pl:  # save the final set
            self.save_set()
        ret = self.cli_list
        self.cli_list = []
        return ret


class Validation(object):
    def __init__(self, schema):
        self.schema = schema

    def resource_roles(self):
        'returns list of valid resource roles'
        return self.schema.rng_attr_values('resource_set', 'role')

    def resource_actions(self):
        'returns list of valid resource actions'
        return self.schema.rng_attr_values('resource_set', 'action')

    def date_ops(self):
        'returns list of valid date operations'
        return self.schema.rng_attr_values_l('date_expression', 'operation')

    def expression_types(self):
        'returns list of valid expression types'
        return self.schema.rng_attr_values_l('expression', 'type')

    def rsc_order_kinds(self):
        return self.schema.rng_attr_values('rsc_order', 'kind')

    def class_provider_type(self, value):
        """
        Unravel [class:[provider:]]type
        returns: (class, provider, type)
        """
        c_p_t = disambiguate_ra_type(value)
        if not ra_type_validate(value, *c_p_t):
            return None
        return c_p_t

    def canonize(self, value, lst):
        'case-normalizes value to what is in lst'
        value = value.lower()
        for x in lst:
            if value == x.lower():
                return x
        return None

    def classify_role(self, role):
        if not role:
            return role, None
        elif role in olist(self.resource_roles()):
            return self.canonize(role, self.resource_roles()), 'role'
        elif role.isdigit():
            return role, 'instance'
        return role, None

    def classify_action(self, action):
        if not action:
            return action, None
        elif action in olist(self.resource_actions()):
            return self.canonize(action, self.resource_actions()), 'action'
        elif action.isdigit():
            return action, 'instance'
        return action, None


class CliParser(object):
    parsers = {}

    def __init__(self):
        self.comments = []
        validation = Validation(Schema.getInstance())
        if not self.parsers:
            def add(*parsers):
                for pcls in parsers:
                    p = pcls()
                    p.init(validation)
                    for n in p.can_parse():
                        self.parsers[n] = p
            add(ResourceParser,
                ConstraintParser,
                OpParser,
                NodeParser,
                PropertyParser,
                FencingOrderParser,
                AclParser,
                RawXMLParser)

    def _xml_lex(self, s):
        try:
            l = lines2cli(s)
            a = []
            for p in l:
                a += p.split()
            return a
        except ValueError, e:
            common_err(e)
            return False

    def _normalize(self, s):
        '''
        Handles basic normalization of the input string.
        Converts unicode to ascii, XML data to CLI format,
        lexing etc.
        '''
        if isinstance(s, unicode):
            try:
                s = s.encode('ascii')
            except Exception, e:
                common_err(e)
                return False
        if isinstance(s, str):
            if s and s.startswith('#'):
                self.comments.append(s)
                return None
            if s.startswith('xml'):
                s = self._xml_lex(s)
            else:
                s = shlex.split(s)
        # but there shouldn't be any newlines (?)
        while '\n' in s:
            s.remove('\n')
        if s:
            s[0] = s[0].lower()
        return s

    def parse(self, s):
        '''
        Input: a list of tokens (or a CLI format string).
        Return: a cibobject
            On failure, returns either False or None.
        '''
        s = self._normalize(s)
        if not s:
            return s
        kw = s[0]
        if kw in self.parsers:
            parser = self.parsers[kw]
            try:
                ret = parser.do_parse(s)
                if self.comments:
                    ret.comments = self.comments
                    self.comments = []
                return ret
            except ParseError:
                return False
        syntax_err(s, token=s[0], msg="Unknown command")
        return False

    def parse2(self, s):
        p = self.parse(s)
        if not p:
            return p
        return p.to_list()

# vim:ts=4:sw=4:et:
