# +==========================================================================+
# || CipUX::RPC::Test::Client                                               ||
# ||                                                                        ||
# || Copyright (C) 2008 - 2010 by Christian Kuelker. All rights reserved!   ||
# ||                                                                        ||
# || License: GNU GPL - GNU General Public License - version 2              ||
# ||          or (at your opinion) any later version.                       ||
# +==========================================================================+
# ID:       $Id$
# Revision: $Revision$
# Head URL: $HeadURL$
# Date:     $Date$
# Source:   $Source$

package CipUX::RPC::Test::Client;

use 5.008001;
use strict;
use warnings;
use Carp;
use Class::Std;
use base qw(CipUX::RPC);
use CipUX::Task;
use Data::Dumper;
use Date::Manip;
use English qw( -no_match_vars);
use Frontier::Client;
use Getopt::Long;
use Log::Log4perl qw(get_logger :levels);
use Pod::Usage;
use Readonly;

{

    use version; our $VERSION = qv('3.4.0.9');
    use re 'taint';    # Keep data captured by parens tainted
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safe

    # CONSTANTS
    Readonly::Scalar my $SCRIPT       => 'CipUX::RPC::Test::Client';
    Readonly::Scalar my $EMPTY_STRING => q{};
    Readonly::Scalar my $LINEWIDTH    => 78;

    Readonly::Scalar my $W => 98;    # line width of test. Dep. on task names.
    Readonly::Scalar my $BORDER            => 4;
    Readonly::Scalar my $BORDER_DIFF_LEFT  => 28;
    Readonly::Scalar my $BORDER_DIFF_RIGHT => 38;
    Readonly::Scalar my $A_LITTLE_WHILE    => 3;    # sec of sleep
    Readonly::Scalar my $ENTITY_OBJ_HR     => {
        admin_account     => 'rpctestadmin',
        assistant_account => 'rpctestassistant',
        lecturer_account  => 'rpctestlecturer',
        professor_account => 'rpctestprofessor',
        pupil_account     => 'rpctestpupil',
        student_account   => 'rpcteststudent',
        teacher_account   => 'rpctestteacher',
        tutor_account     => 'rpctesttutor',
        class_share       => 'rpctestclass',
        course_share      => 'rpctestcourse',
        lecture_share     => 'rpctestlecture',
        reading_share     => 'rpctestreading',
        seminar_share     => 'rpctestseminar',
        studygroup_share  => 'rpcteststudygroup',
        team_share        => 'rpctestteam',
        tutorial_share    => 'rpctesttutorial',
        workshop_share    => 'rpctestworkshop',
    };
    Readonly::Scalar my $PLAIN_ENTITY_HR => {
        netgroup => {
            entity     => 'netgroup',
            entity_obj => 'mytestnetgroup',
            member     => 'client',
            member_obj => '(testhost1,,)',
        },
    };
    Readonly::Scalar my $TEST_COUNT_START => '0000';
    Readonly::Scalar my $TEST_COUNT_INC   => '0001';
    Readonly::Scalar my $TEST_MAX         => '9999';
    Readonly::Scalar my $TEST_SUM_LEFT    => 3;
    Readonly::Scalar my $TEST_SUM_RIGHT   => 4;
    Readonly::Scalar my $TEST_SUM_RESULT  => 7;

    Readonly::Array my @ACTION => qw(cipux_rpc_test_client);

    Readonly::Scalar my $L4P_CONF => ( -e '/etc/cipux/log4perl.conf' )
        ? '/etc/cipux/log4perl.conf'
        : $EMPTY_STRING;

    # OBJECT
    # register client with name: cipux_rpc_test_client
    my %name_of : ATTR( init_arg => 'name');

    # GLOBAL
    my $L = q{=} x $LINEWIDTH;
    $L .= "\n";
    my $url             = $EMPTY_STRING;
    my $host            = 'localhost';
    my $port            = 8001;
    my $ssl             = 0;
    my $ttl             = 0;
    my $config_hr       = {};
    my $task_hr         = {};
    my %opt             = ();
    my $destroy         = 1;
    my $login           = $EMPTY_STRING;
    my $ticket          = $EMPTY_STRING;
    my $password        = $EMPTY_STRING;
    my %test            = ();
    my $test            = $TEST_COUNT_START;
    my %title           = ();
    my $testcount       = $TEST_COUNT_INC;
    my $maxtest         = $TEST_MAX;
    my $success_summary = '0';
    my $failure_summary = '0';
    my %option          = (
        'cipux_rpc_test_client' => {
            'must' => [],
            'may' =>
                [qw(D debug h ? help host maxtest no-ssl port ssl version)],
            'not' => [],
        },
    );

    sub run {    ## no critic Subroutines::ProhibitExcessComplexity

        # +------------------------------------------------------------------+
        # | API
        my ( $self, $arg_r ) = @_;

        # constructor pram from CipUX::Object::Client
        my $run_action = $name_of{ ident $self};

        # test right away if we have a valid action
        # is $run_action inside @action?
        if ( scalar( grep { $_ eq $run_action } @ACTION ) < 1 ) {
            $self->exc( { msg => 'unknown action', value => $run_action } );
        }

        # ENVIRONMENT
        Getopt::Long::Configure('bundling');

        # first thing will be defined: debug,help,version
        GetOptions(
            \%opt,       'debug|D', 'help|h|?', 'host=s',
            'maxtest=i', 'port=i',  'no-ssl',   'ssl',
            'version|V',
            )
            or pod2usage(
            -exitstatus => 2,
            -msg        => "$L problems parsing command line!\n$L"
            );

        if ( $L4P_CONF eq $EMPTY_STRING ) {
            $self->exc( { msg => 'log4perl.conf not found' } );
        }

        if ( exists $opt{debug} and defined $opt{debug} ) {
            Log::Log4perl::init_once($L4P_CONF);
            print "Writing debug output to cipux-test.log\n";
        }

        my $logger = get_logger(__PACKAGE__);

        my $date = UnixDate( 'today', '%O' );
        $logger->debug("    CipUX : $VERSION   ");
        $logger->debug("    date  : $date");
        $logger->debug("    action: $run_action");

        # display help page
        if ( exists $opt{help} ) {
            pod2usage( -exitstatus => 0, -verbose => 0 );
        }

        if ( exists $opt{version} ) {
            $self->out( "$run_action $VERSION\n", );
            exit 0;
        }

        my $cipux_task = CipUX::Task->new();
        $task_hr = $cipux_task->get_task_api_cfg();

        my $cipux_rpc = CipUX::RPC->new();
        $config_hr = $cipux_rpc->get_rpc_cfg();

        $host
            = ( exists $opt{host} and defined $opt{host} ) ? $opt{host}
            : ( exists $config_hr->{xml_rpc_address}
                and defined $config_hr->{xml_rpc_address} )
            ? $config_hr->{xml_rpc_address}
            : 'localhost';

        $port
            = ( exists $opt{port} and defined $opt{port} ) ? $opt{port}
            : ( exists $config_hr->{xml_rpc_port}
                and defined $config_hr->{xml_rpc_port} )
            ? $config_hr->{xml_rpc_port}
            : 8001;

        $ssl
            = ( exists $opt{ssl} and defined $opt{ssl} ) ? 1
            : ( exists $opt{'no-ssl'} and defined $opt{'no-ssl'} ) ? 0
            : ( exists $config_hr->{xml_rpc_ssl}
                and defined $config_hr->{xml_rpc_ssl} )
            ? $config_hr->{xml_rpc_ssl}
            : 0;

        $ttl
            = ( exists $config_hr->{xml_rpc_ttl}
                and defined $config_hr->{xml_rpc_ttl} )
            ? $config_hr->{xml_rpc_ttl}
            : 600;

        $logger->debug("    host  : $host");
        $logger->debug("    port  : $port");
        $logger->debug("    ssl   : $ssl");
        $logger->debug("    ttl   : $ttl");

        # url  = 'https://localhost:8000/RPC2';
        # url  = 'http://localhost:8001/RPC2';
        my $proto = ($ssl) ? 'https' : 'http';
        $url = $proto . q{://} . $host . q{:} . $port . q{/RPC2};
        $logger->debug("    url   : $url");

        if ( exists $opt{maxtest} ) {
            $maxtest = $opt{maxtest};
            $logger->debug( 'maxtest: ', $maxtest, "\n" );
        }

        my $ret = $self->test_cli_option(
            {
                script   => $run_action,
                logic_hr => \%option,
                opt_hr   => \%opt,
                debug    => 0,
            }
        );

        # MAIN
        $self->test_ping;
        $self->test_version();
        $self->test_sum1();
        $self->test_sum2();
        $self->test_login();
        $self->test_list_admin();
        $self->test_ttl();
        $self->test_session();
        $self->test_entity1();
        $self->test_logout();
        $self->test_list_users();
        $self->test_login_again();
        $self->test_entity2();

        # this test is disabled. We do not need to add orga nodes for the
        # moment under ou=CipUX and we do not have write access on Debian Edu
        # on the hole subtree dc=skole,dc=skolelinux,dc=no for now.
        # $self->test_create_list_destroy1();
        $self->test_create_list_destroy2();
        $self->test_plain_entity();
        $self->test_summary();

        return;
    }

    sub test_ping {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::RPC ping exec',
                url      => $url,
                login    => 'dummy',
                ticket   => 'dummy',
                cmd      => 'ping',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        return;
    }

    sub test_version {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::RPC version exec',
                url      => $url,
                login    => 'dummy',
                ticket   => 'dummy',
                cmd      => 'version',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        my $version_hr = $answer_hr->{cmdres_r};

        $self->record_test_result(
            {
                start => 1,
                title => 'CipUX::RPC version result',
                mode  => 'true',
                true  => (
                            $version_hr->{cipux_version} ne $EMPTY_STRING
                        and $version_hr->{rpc_version}   ne $EMPTY_STRING
                        and $version_hr->{cipux_version} ne $EMPTY_STRING
                ),
                print => ' CipUX version: '
                    . $version_hr->{cipux_version} . ', '
                    . ' RPC version: '
                    . $version_hr->{rpc_version} . ', '
                    . ' server version: '
                    . $version_hr->{cipux_version},
                success => 'Got version',
                failure => 'Did not got version'
            }
        );

        return;
    }

    sub test_sum1 {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::RPC sum 1 exec',
                url      => $url,
                login    => 'dummy',
                ticket   => 'dummy',
                cmd      => 'sum',
                param_hr => {
                    summand1 => $TEST_SUM_LEFT,
                    summand2 => $TEST_SUM_RIGHT,
                },
                expect => 'TRUE',
            }
        );

        $logger->debug( 'answer_hr: ',
            { filter => \&Dumper, value => $answer_hr } );

        my $sum = $answer_hr->{cmdres_r}->[0];

        $self->record_test_result(
            {
                start => 1,
                title => 'CipUX::RPC sum 1 result check',
                mode  => 'true',
                true  => ( $sum eq "$TEST_SUM_RESULT" ),
                success =>
                    "the sum function is working. $TEST_SUM_LEFT plus $TEST_SUM_RIGHT equal $TEST_SUM_RESULT.",
                failure =>
                    "the sum function is NOT working. $TEST_SUM_LEFT plus $TEST_SUM_RIGHT not equal $TEST_SUM_RESULT."
            }
        );

        return;
    }

    sub test_sum2 {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task sum 2 test',
                url      => $url,
                login    => 'dummy',
                ticket   => 'dummy',
                cmd      => 'cipux_task_sum',
                param_hr => {
                    summand1 => $TEST_SUM_LEFT,
                    summand2 => $TEST_SUM_RIGHT,
                },
                expect => 'TRUE',
            }
        );

        my $sum2 = $answer_hr->{cmdres_r}->[0];
        $self->record_test_result(
            {
                start => 1,
                title => 'CipUX::Task sum 2 result check',
                mode  => 'true',
                true  => ( $sum2 eq "$TEST_SUM_RESULT" ),
                success =>
                    "GOOD: the sum2 function is working. $TEST_SUM_LEFT plus $TEST_SUM_RIGHT equal $TEST_SUM_RESULT",
                failure =>
                    "BAD: the sum2 function is NOT working.  $TEST_SUM_LEFT plus $TEST_SUM_RIGHT NOT equal $TEST_SUM_RESULT"
            }
        );

        return;
    }

    sub test_login {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        #1
        $self->manual_test( { title => 'CipUX login prompt exec' } );
        $self->out("     Give a valid login (cipadmin) and a password,\n");
        $self->out("     to log in to the CipUX XML-RPC server.\n");
        $self->out(
            "     ATTENTION: This will test XML-RPC server! It will add\n");
        $self->out(
            "     add and DELETE accounts on the LDAP server! You should\n");
        $self->out(
            "     NOT do this on a production server, you might lose\n");
        $self->out("     data! If you want to chancel press CTRl-C now.\n");

        $login = $self->login_prompt( { prompt => '     Login: ' } );
        $password = $self->password_prompt( { prompt => '     Password: ' } );

        $self->login( { login => $login, password => $password } );

        return;
    }

    sub test_list_admin {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);
        $logger->debug('list admin accounts exec');

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task list admin accounts exec',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'cipux_task_list_admin_accounts',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        $logger->debug('search member list for rpctestadmin');

#my $result = $self->member_list({answer_hr => $answer_hr, member => 'rpctestadmin'});
        my @admin = $self->member_list( { answer_hr => $answer_hr } );
        $logger->debug( 'result: ', scalar @admin );

        $self->record_test_result(
            {
                start   => 1,
                title   => 'CipUX::Task list admin accounts result',
                mode    => 'compare',
                minor   => 0,
                major   => scalar @admin,
                success => 'OK CipUX::Task list admin test',
                failure => 'BAD CipUX::Task list admin test'
            }
        );

        return;
    }

    sub test_ttl {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::RPC ttl exec',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'ttl',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        my $got_ttl = $answer_hr->{cmdres_r}->{ttl} || 0;

        $self->record_test_result(
            {
                start => 1,
                title => 'CipUX::RPC ttl result',

                mode    => 'compare',
                minor   => 0,
                major   => $got_ttl,
                print   => 'TTL is ' . $got_ttl . ' seconds.',
                success => 'We got a TTL',
                failure => 'We got no TTL'
            }
        );
        $self->record_test_result(
            {
                start   => 1,
                title   => 'CipUX::RPC ttl value',
                mode    => 'true',
                true    => ( $ttl == $got_ttl ),
                major   => $got_ttl,
                success => 'Configuration TTL and Server TTL is equal',
                failure => 'Configuration TTL and Server TTL is NOT equal'
            }
        );

        return;
    }

    sub test_session {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        sleep $A_LITTLE_WHILE;

        my $ticket_save = $ticket;

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::RPC session exec',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'session',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        $self->record_test_result(
            {
                start   => 1,
                title   => 'CipUX::RPC session result',
                mode    => 'true',
                true    => ( $ticket ne $ticket_save ),
                success => 'GOOD: we got a different ticket.',
                failure => 'BAD: tickets are the same.'
            }
        );

        return;
    }

    #  cipux_task_{create|list|destroy}_student_account
    sub test_entity1 {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        foreach my $entity ( sort keys %{$ENTITY_OBJ_HR} ) {
            $self->big_entity_test(
                { entity => $entity, obj => $ENTITY_OBJ_HR->{$entity} } );
        }

        return;
    }

    sub test_logout {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task logout exec',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'logout',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        return;
    }

    sub test_list_users {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task list user accounts exec',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'cipux_task_list_user_accounts',
                param_hr => {},
                expect   => 'FALSE',
            }
        );

        if ( defined $answer_hr->{msg} ) {
            $self->out("     Server answer: $answer_hr->{msg}\n");
        }
        my $result = $self->member_list(
            { answer_hr => $answer_hr, member => 'cipadmin' } );

        $self->record_test_result(
            {
                start => 1,
                title => 'CipUX::Task list user account result',
                mode  => 'compare',
                minor => $result,
                major => 1,
                print => 'cipux_task_list_user_accounts should fail,'
                    . ' because we are logged out.',
                success => 'ok, got no objects',
                failure => 'bad, got objects'
            }
        );

        return;
    }

    sub test_login_again {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        $self->login( { login => $login, password => $password } );

        return;
    }

    # cipux_task_{create|list|destroy}_course_share

    sub test_entity2 {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        $self->big_entity_test(
            { entity => 'course_share', obj => 'testcourse' } );

        return;
    }

    # cipux_task_{create|list|destroy}_ldap_orga_node
    sub test_create_list_destroy1 {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        $self->create_list_destroy_test(
            {
                title   => 'CipUX::Task',
                create  => 'cipux_task_create_ldap_orga_node',
                destroy => 'cipux_task_destroy_ldap_orga_node',
                list    => 'cipux_task_list_ldap_orga_nodes',
                testobj => 'testorganode',
                add_hr  => {}
            }
        );

        return;
    }

    # cipux_task_{create|list|destroy}_cat_module

    sub test_create_list_destroy2 {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        $self->create_list_destroy_test(
            {
                title   => 'CipUX::Task',
                create  => 'cipux_task_register_cat_module',
                destroy => 'cipux_task_deregister_cat_module',
                list    => 'cipux_task_list_cat_modules',
                testobj => 'testcatmodule.cgi',
                add_hr  => {
                    cipuxTemplate         => 'simpeladmin',
                    cipuxName             => 'Test CAT module',
                    cipuxEntity           => 'student_account',
                    cipuxYear             => '2008',
                    cipuxModality         => 'student',
                    cipuxIcon             => 'student.png',
                    cipuxIsModuleArray    => 'FALSE',
                    cipuxShortDescription => 'Short Description',
                    cipuxIsEnabled        => 'FALSE',
                }
            }
        );

        return;
    }

    sub test_plain_entity {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        foreach my $entity ( keys %{$PLAIN_ENTITY_HR} ) {
            $self->plain_entity_test(
                { entity_hr => $PLAIN_ENTITY_HR->{$entity} } );
        }

        return;
    }

    sub test_summary {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        $self->out($L);
        my $width   = $W - $BORDER;
        my $tlength = $W - $BORDER_DIFF_LEFT;
        my $slength = $W - $BORDER_DIFF_RIGHT;
        $self->out("    Summary:\n");
        $self->out( '    +' . q{-} x ( $width + 1 ) . "+\n" );
        foreach my $test ( sort { $a <=> $b } keys %test ) {
            printf '    |  Test %04s: %-' . $tlength . "s %-10s |\n", $test,
                $title{$test}, $test{$test};
        }
        $self->out( '    +' . q{-} x ( $width + 1 ) . "+\n" );

        if ( $success_summary > 0 ) { $success_summary--; }
        printf '    | '
            . q{ } x $slength
            . "SUCCESS: %-6s FAILURE: %-6s   |\n",
            $success_summary, $failure_summary;
        $self->out( '    +' . q{-} x ( $width + 1 ) . "+\n" );

        return;
    }

   #
   # out("="x76,"\n");
   # out("($test) connect to $url over SSL ...\n");
   # out("    try to get the userlist with cipux_task_change_own_password\n");
   # @args=();
   # push(@args,$login);
   # push(@args,"password");
   # $server->call('cipux_task_change_own_password',@args);
   # exit 0;

   # out("="x76,"\n");
   # out("($test) connect to $url over SSL ...\n");
   # out("    try to get the userlist with cipux_task_change_own_password\n");
   # @args=();
   # push(@args,$login);
   # push(@args,"password");
   # $server->call('cipux_task_change_own_password',@args);
   #
   # exit 0;

    sub xmlrpc {    ## no critic Subroutines::ProhibitExcessComplexity

        # API
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        $ticket
            = exists $arg_r->{ticket}
            ? $self->l( $arg_r->{ticket} )
            : $self->perr('ticket');

        my $cmd
            = exists $arg_r->{cmd}
            ? $self->l( $arg_r->{cmd} )
            : $self->perr('cmd');

        my $param_hr
            = exists $arg_r->{param_hr}
            ? $self->h( $arg_r->{param_hr} )
            : $self->perr('param_hr');

        my $url
            = exists $arg_r->{url}
            ? $self->l( $arg_r->{url} )
            : $self->perr('url');

        my $title
            = exists $arg_r->{title}
            ? $self->l( $arg_r->{title} )
            : $self->perr('title');

        my $expect
            = exists $arg_r->{expect}
            ? $self->l( $arg_r->{expect} )
            : $self->perr('expect');

        my $print = $self->l( $arg_r->{print} ) || $EMPTY_STRING;

        #if ( defined $title{$test} ) {
        #    $logger->debug( 'title  : ', $title{$test} );
        #}
        $logger->debug( 'test   : ', $test );
        $logger->debug( 'test nr: ', $testcount );
        $logger->debug( 'maxtest: ', $maxtest );
        $logger->debug( 'command: ', $cmd );
        $logger->debug( 'url    : ', $url );

        exit 0 if $test >= $maxtest;

        $test++;
        $title{$test} = $title;
        $testcount++;
        $self->out($L);
        $self->out("($test) ==> $title{$test} <== (RPC) \n");
        $self->out("     command $cmd\n");
        $self->out("     connect to $url ...\n");
        if ( $print ne $EMPTY_STRING ) { $self->out("     $print\n"); }

        my $HEADER_HREF = {
            'cipux_version'  => '3.4.0.0',
            'client_name'    => 'cipux_rpc_test_client',
            'client_version' => '3.4.0.0',
            'rpc_version'    => '2.0',
            'client_key'     => $EMPTY_STRING,
            'client_cred'    => $EMPTY_STRING,
            'gmt_time'       => time(),
        };

        my $pay_hr = {
            header_hr => $HEADER_HREF,
            login     => $login,
            ticket    => $ticket,
            cmd       => $cmd,
            param_hr  => $param_hr
        };

        my $cmd_type = $cmd;
        if ( $cmd =~ m/cipux_task/xms ) {
            $cmd_type = 'task';
        }

        my $socket = Frontier::Client->new( url => $url );

        my $answer_hr = $socket->call( $cmd_type, $pay_hr );

        my $status = $answer_hr->{status} || 'UNKNOWN';
        $self->out("     the expected status should be: [$expect] \n");
        $self->out("     the status of the request was: [$status] \n");
        if ( $status eq $expect ) {
            $self->out("     Test $test ... SUCCESS\n");
            $test{$test} = 'SUCCESS';
            $success_summary++;
            $logger->debug( 'status : ', $test{$test} );
        }
        else {
            $self->out("     Test $test ... FAILURE\n");
            $test{$test} = 'FAILURE';
            $failure_summary++;
            $logger->debug( 'status : ', $test{$test} );
            if ( defined $answer_hr->{msg} ) {
                $self->out(
                    '    Message from Server: ' . $answer_hr->{msg} . "\n" );
            }
        }

        # remember, if we got new ticket on the default channel
        if ( defined $answer_hr->{ticket}
            and $answer_hr->{ticket} ne $EMPTY_STRING )
        {
            $ticket = $answer_hr->{ticket};
        }

        # but if we got a ticket explcit: eq. login we should use this
        if (    ref( $answer_hr->{cmdres_r} ) eq 'HASH'
            and defined $answer_hr->{cmdres_r}->{ticket}
            and $answer_hr->{cmdres_r}->{ticket} ne $EMPTY_STRING )
        {
            $ticket = $answer_hr->{cmdres_r}->{ticket};
        }

        return $answer_hr;

    }

    sub manual_test {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $title
            = exists $arg_r->{title}
            ? $self->l( $arg_r->{title} )
            : $self->perr('title');

        exit 0 if $test >= $maxtest;

        $test++;
        $title{$test} = $title;
        $testcount++;

        $self->out($L);
        $self->out("($test) ==> $title{$test} <== (M) \n");

        return;

    }

    sub record_test_result {

        #API
        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $mode
            = exists $arg_r->{mode}
            ? $self->l( $arg_r->{mode} )
            : $self->perr('mode');

        my $title
            = exists $arg_r->{title}
            ? $self->l( $arg_r->{title} )
            : $self->perr('title');

        my $failure
            = exists $arg_r->{failure}
            ? $self->l( $arg_r->{failure} )
            : $self->perr('failure');

        my $success
            = exists $arg_r->{success}
            ? $self->l( $arg_r->{success} )
            : $self->perr('success');

        my $start = exists $arg_r->{start} ? $self->l( $arg_r->{start} ) : 0;

        my $print
            = exists $arg_r->{print}
            ? $self->l( $arg_r->{print} )
            : $EMPTY_STRING;

        if ($start) {
            exit 0 if $test >= $maxtest;
            $test++;
            $title{$test} = $title;
            $testcount++;

            $self->out($L);
            $self->out("($test) ==> $title{$test} <== (M) \n");
        }
        else {
            $title{$test} = $title;
        }

        #    if ($print) {
        #        $self->out( q{     } . $print . "\n" );
        #    }

        my $result = 0;
        if ( $mode eq 'true' ) {

            if ( $self->l( $arg_r->{true} ) ) {    # API2
                $self->out("     Test $test SUCCESS\n");
                $test{$test} = 'SUCCESS';
                $success_summary++;
                $result = 1;
                $logger->debug( 'status : ', $test{$test} );
            }
            else {
                $self->out("     Test $test FAILURE\n");
                $test{$test} = 'FAILURE';
                $failure_summary++;
                $logger->debug( 'status : ', $test{$test} );
            }

        }
        elsif ( $mode eq 'compare' ) {

            # API2
            my $minor
                = exists $arg_r->{minor} ? $self->l( $arg_r->{minor} ) : 0;
            my $major
                = exists $arg_r->{major} ? $self->l( $arg_r->{major} ) : 0;

            if ( $minor < $major ) {
                $self->out("     Test $test SUCCESS\n");
                $test{$test} = 'SUCCESS';
                $success_summary++;
                $result = 1;
                $logger->debug( 'status : ', $test{$test} );
            }
            else {
                $self->out("     Test $test FAILURE\n");
                $test{$test} = 'FAILURE';
                $failure_summary++;
                $logger->debug( 'status : ', $test{$test} );
            }

        }

        if ($result) {
            $self->out( q{    (} . $success . ")\n" );
        }
        else {
            $self->out( q{    (} . $failure . ")\n" );
        }

        return $result;
    }

    sub login {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $login
            = exists $arg_r->{login}
            ? $self->l( $arg_r->{login} )
            : $self->perr('login');

        my $password
            = exists $arg_r->{password}
            ? $self->l( $arg_r->{password} )
            : $self->perr('password');

        my $result = $self->record_test_result(
            {
                start => 0,
                title => 'CipUX login prompt exec',
                mode  => 'true',
                true  => (
                            $login eq 'cipadmin'
                        and defined $password
                        and $password ne $EMPTY_STRING
                ),
                success => 'login prompt works',
                failure => 'We will not continue. We stop the test now.'
                    . 'Please provide login - cipadmin - and password next time.'
            }
        );

        exit 1 if not $result;

        #2
        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::RPC    login test exec',
                url      => $url,
                login    => $login,
                ticket   => 'dummy',
                cmd      => 'login',
                param_hr => { password => $password, },
                expect   => 'TRUE',
            }
        );

        #3
        $result = $self->record_test_result(
            {
                start   => 1,
                title   => 'CipUX::RPC    login result',
                mode    => 'true',
                true    => ( $answer_hr->{status} eq 'TRUE' ),
                success => 'GOOD, we are in.',
                failure => 'BAD, we are not in. We will not continue.'
                    . ' We stop the test now.'
                    . ' Please provide valid login and password next time.'
            }
        );

        exit 1 if not $result;

        #$ticket = $answer_hr->{cmdres_r}->{ticket};

        #4
        $result = $self->record_test_result(
            {
                start => 1,
                title => 'CipUX::RPC    login ticket result',
                mode  => 'true',
                true  => ( defined $ticket and $ticket ne $EMPTY_STRING ),
                success =>
                    'OK, got secret ticket. You should not print that.',
                failure => 'BAD: wrong login or password'
            }
        );

        exit 1 if not $result;

        return;

    }

    #  cipux_task_{create|list|destroy}_{$entity}
    # cipux_task_create_{entity}
    # cipux_task_change_{entity}_password
    # cipux_task_add_member_to_{entity}
    # cipux_task_list_members_of_{entity}
    # cipux_task_remove_member_from_{entity}
    # cipux_task_list_{entity}s
    # cipux_task_destroy_{entity}
    #
    sub big_entity_test {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $entity
            = exists $arg_r->{entity}
            ? $self->l( $arg_r->{entity} )
            : $self->perr('entity');

        my $obj
            = ( exists $arg_r->{obj} and defined $arg_r->{obj} )
            ? $self->l( $arg_r->{obj} )
            : $self->perr('obj');

        $logger->debug("probably delete $obj of $entity");

        $self->probably_delete(
            {
                title   => 'CipUX::Task 01 maybe destroy ' . $entity,
                list    => 'cipux_task_list_' . $entity . 's',
                destroy => 'cipux_task_destroy_' . $entity,
                testobj => $obj,
            }
        );

        # cipux_task_create_{ $entity }
        $logger->debug( 'RPC CREATE: cipux_task_create_' . $entity );
        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task 03 create ' . $entity . ' exec',
                print    => 'operating on: ' . $obj,
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'cipux_task_create_' . $entity . $EMPTY_STRING,
                param_hr => {
                    object         => $obj,
                    cipuxFirstname => 'Test',
                    cipuxLastname  => 'Object',
                },
                expect => 'TRUE',
            }
        );

        $logger->debug( 'RPC CREATE: cipux_task_list_' . $entity . 's' );
        $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task 04 create ' . $entity . ' list',
                print    => 'operating on: ' . $obj,
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'cipux_task_list_' . $entity . 's',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        $logger->debug('test if account has been created');
        my $result = $self->member_list(
            { answer_hr => $answer_hr, member => $obj } );
        $logger->debug("result: $result");

        my $test_account_created = $result;

        $logger->debug('Record CREATE: recording the result');
        $self->record_test_result(
            {
                start   => 1,
                title   => 'CipUX::Task 05 create ' . $entity . ' result',
                print   => 'operating on: ' . $obj,
                mode    => 'true',
                true    => ($test_account_created),
                print   => 'test if it was created ...',
                success => 'test account created.',
                failure => 'Test account not created.'
            }
        );

        # cipux_task_change_{. $entity . }_password
        $logger->debug( 'RPC CHANGE PWD: change_' . $entity . '_password' );
        my $rnd_password = $self->random_password();

        $answer_hr = $self->xmlrpc(
            {
                title => 'CipUX::Task 07 change password of ' 
                    . $entity . ' exec',
                print  => 'operating on: ' . $obj,
                url    => $url,
                login  => $login,
                ticket => $ticket,
                cmd    => 'cipux_task_change_' . $entity . '_password',

               #param_hr => { object => $obj, userPassword => $rnd_password },
                param_hr => { object => $obj, value => $rnd_password },
                expect   => 'TRUE',
            }
        );

        # cipux_task_add_member_{ $entity }
        # cipux_task_add_member_to_course_share
        # cipux_task_list_members_of_course_share
        # cipux_task_remove_member_from_course_share

        my $member = 'testmember';

        $answer_hr = $self->xmlrpc(
            {
                title  => "CipUX::Task 08 add $member to $entity exec",
                print  => 'operating on: ' . $obj . ' and ' . $member,
                url    => $url,
                login  => $login,
                ticket => $ticket,
                cmd => 'cipux_task_add_member_to_' . $entity . $EMPTY_STRING,

                #param_hr => { object => $obj, memberUid => $member },
                param_hr => { object => $obj, value => $member },
                expect   => 'TRUE',
            }
        );

        $answer_hr = $self->xmlrpc(
            {
                title  => "CipUX::Task 09 add $member to $entity list",
                url    => $url,
                login  => $login,
                ticket => $ticket,
                cmd    => 'cipux_task_list_members_of_' 
                    . $entity
                    . $EMPTY_STRING,
                param_hr => { object => $obj },
                expect   => 'TRUE',
            }
        );

        $logger->debug( 'answer_hr: ',
            { filter => \&Dumper, value => $answer_hr } );
        $logger->debug( 'obj      : ', $obj, "\n" );

        $result = $self->member_list(
            { answer_hr => $answer_hr, member => $member } );

        $self->record_test_result(
            {
                start => 1,
                title => 'CipUX::Task 10 add ' 
                    . $member . ' to ' 
                    . $entity
                    . ' result',
                mode    => 'true',
                true    => $result,
                print   => 'test if it was created ... ' . $member,
                success => 'GOOD: test member ' . $member . ' is there',
                failure => 'BAD: ' . $member . ' not there.'
            }
        );

        $answer_hr = $self->xmlrpc(
            {
                title  => "CipUX::Task 11 remove $member from $entity remove",
                url    => $url,
                login  => $login,
                ticket => $ticket,
                cmd    => 'cipux_task_remove_member_from_' 
                    . $entity
                    . $EMPTY_STRING,
                param_hr => { object => $obj, value => $member },
                expect   => 'TRUE',
            }
        );

        $answer_hr = $self->xmlrpc(
            {
                title  => "CipUX::Task 12 remove $member from $entity list",
                url    => $url,
                login  => $login,
                ticket => $ticket,
                cmd    => 'cipux_task_list_members_of_' 
                    . $entity
                    . $EMPTY_STRING,
                param_hr => { object => $obj },
                expect   => 'TRUE',
            }
        );

        $result = $self->member_list(
            { answer_hr => $answer_hr, member => $member } );

        $self->record_test_result(
            {
                start => 1,
                title => 'CipUX::Task 13 add member to ' 
                    . $entity
                    . ' result',
                mode    => 'true',
                true    => ( not $result ),
                print   => 'test if it was created ... ' . $member,
                success => 'GOOD: test member ' . $member . ' is not there',
                failure => 'BAD: ' . $member . ' there.'
            }
        );

        #  cipux_task_destroy_{ $entity }
        if ($destroy) {
            my $answer_hr = $self->xmlrpc(
                {
                    title  => 'CipUX::Task 14 destroy ' . $entity . ' exec',
                    url    => $url,
                    login  => $login,
                    ticket => $ticket,
                    cmd    => 'cipux_task_destroy_' . $entity . $EMPTY_STRING,
                    param_hr => { object => $obj, },
                    expect   => 'TRUE',
                }
            );
        }

        $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task 15 destroy ' . $entity . ' list',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'cipux_task_list_' . $entity . 's',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        $result = $self->member_list(
            { answer_hr => $answer_hr, member => $obj } );

        $self->record_test_result(
            {
                start   => 1,
                title   => 'CipUX::Task 16 destroy ' . $entity . ' result',
                mode    => 'true',
                true    => ( not $result ),
                print   => 'test if it was created ...',
                success => 'test account deleted.',
                failure => 'Test account not deleted.'
            }
        );

        return;

    }

    # plain_entity_test
    # cipux_task_create_{entity}
    # cipux_task_list_{entity}s
    # cipux_task_addmodify_all_clients_of_netgroup  (add)
    # cipux_task_list_{target}s_of_{entity}
    # cipux_task_erase_clients_from_netgroup
    # cipux_task_list_{target}s_of_{entity}
    # cipux_task_destroy_{entity}
    #
    sub plain_entity_test {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $e_hr
            = exists $arg_r->{entity_hr}
            ? $self->h( $arg_r->{entity_hr} )
            : $self->perr('entity_hr');

        my $entity
            = exists $e_hr->{entity}
            ? $self->l( $e_hr->{entity} )
            : $self->perr('e_hr->{entity}');

        my $entity_obj
            = exists $e_hr->{entity_obj}
            ? $self->l( $e_hr->{entity_obj} )
            : $self->perr('e_hr->{entity_obj}');

        my $member
            = exists $e_hr->{member}
            ? $self->l( $e_hr->{member} )
            : $self->perr('e_hr->{member}');

        my $member_obj
            = exists $e_hr->{member_obj}
            ? $self->l( $e_hr->{member_obj} )
            : $self->perr('e_hr->{member_obj}');

        $logger->debug( "probably plain delete $entity_obj of $entity",
            "\n" );

        $self->probably_delete(
            {
                title   => 'CipUX::Task 01 maybe plain destroy ' . $entity,
                list    => 'cipux_task_list_' . $entity . 's',
                destroy => 'cipux_task_destroy_' . $entity,
                testobj => $entity_obj,
            }
        );

        # cipux_task_create_{ $entity }

        my $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task 03 create ' . $entity . ' exec',
                print    => 'operating on: ' . $entity_obj,
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'cipux_task_create_' . $entity . $EMPTY_STRING,
                param_hr => { object => $entity_obj, },
                expect   => 'TRUE',
            }
        );

        $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task 04 create ' . $entity . ' list',
                print    => 'operating on: ' . $entity_obj,
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'cipux_task_list_' . $entity . 's',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        my $result = $self->member_list(
            { answer_hr => $answer_hr, member => $entity_obj } );

        $self->record_test_result(
            {
                start   => 1,
                title   => 'CipUX::Task 05 create ' . $entity . ' result',
                print   => 'operating on: ' . $entity_obj,
                mode    => 'true',
                true    => $result,
                print   => 'test if it was created ...',
                success => 'test account created.',
                failure => 'Test account not created.'
            }
        );

        #  cipux_task_add_member_{ $entity }
        # cipux_task_addmodify_all_clients_of_netgroup
        # cipux_task_list_clients_of_netgroup
        # cipux_task_erase_clients_from_netgroup

       #cmd => 'cipux_task_add_'.$member_obj.'_to_' . $entity . $EMPTY_STRING,
        my $cmd
            = ( $entity eq 'netgroup' )
            ? 'cipux_task_addmodify_all_clients_of_netgroup'
            : 'cipux_task_change_' . $entity . '_' . $member;

        $answer_hr = $self->xmlrpc(
            {
                title => "CipUX::Task 06 add $member to $entity exec",
                print => 'operating on: '
                    . $entity_obj . ' and '
                    . $member_obj,
                url    => $url,
                login  => $login,
                ticket => $ticket,
                cmd    => $cmd,

                #param_hr => { object => $obj, memberUid => $member },
                param_hr => { object => $entity_obj, value => $member_obj },
                expect   => 'TRUE',
            }
        );

        my $tsk07cmd = "cipux_task_list_$member";
        $tsk07cmd .= "s_of_$entity" . $EMPTY_STRING;

        $answer_hr = $self->xmlrpc(
            {
                title    => "CipUX::Task 07 add $member to $entity list",
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => $tsk07cmd,
                param_hr => { object => $entity_obj },
                expect   => 'TRUE',
            }
        );

        my @member = sort
            @{ $answer_hr->{cmdres_r}->{$entity_obj}->{nisNetgroupTriple} };
        my $memberlist = join q{, }, @member;
        my %member = ();

        my $z = 0;
        foreach my $m (@member) {
            $member{$m} = 1;
            $self->out("got member $m\n");
            $z++;
        }

        $self->record_test_result(
            {
                start => 1,
                title => 'CipUX::Task 08 add member to ' 
                    . $entity
                    . ' result',
                mode    => 'true',
                true    => ( defined $member{$member_obj} ),
                print   => 'test if it was created ... ' . $member_obj,
                success => 'GOOD: test member ' . $member_obj . ' is there',
                failure => 'BAD: ' . $member_obj . ' not there.'
            }
        );

        $answer_hr = $self->xmlrpc(
            {
                title  => "CipUX::Task 09 erase $member 's of $entity list",
                url    => $url,
                login  => $login,
                ticket => $ticket,

                cmd => 'cipux_task_erase_' . $member . 's_of_' . $entity
                    . $EMPTY_STRING,

                #param_hr => { object => $entity_obj, value => $member_obj },
                param_hr => { object => $entity_obj, value => $EMPTY_STRING },
                expect   => 'TRUE',
            }
        );

        # cipux_task_destroy_{ $entity }

        if ($destroy) {
            my $answer_hr = $self->xmlrpc(
                {
                    title  => 'CipUX::Task 10 destroy ' . $entity . ' exec',
                    url    => $url,
                    login  => $login,
                    ticket => $ticket,
                    cmd    => 'cipux_task_destroy_' . $entity . $EMPTY_STRING,
                    param_hr => { object => $entity_obj, },
                    expect   => 'TRUE',
                }
            );
        }

        $answer_hr = $self->xmlrpc(
            {
                title    => 'CipUX::Task 11 destroy ' . $entity . ' list',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => 'cipux_task_list_' . $entity . 's',
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        $result = $self->member_list(
            { answer_hr => $answer_hr, member => $entity_obj } );

        $self->record_test_result(
            {
                start   => 1,
                title   => 'CipUX::Task 12 destroy ' . $entity . ' result',
                mode    => 'true',
                true    => ( not $result ),
                print   => 'test if it was created ...',
                success => 'test account deleted.',
                failure => 'Test account not deleted.'
            }
        );

        return;

    }

    sub probably_delete {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $testobj
            = exists $arg_r->{testobj}
            ? $self->l( $arg_r->{testobj} )
            : $self->perr('testobj');

        my $list
            = exists $arg_r->{list}
            ? $self->l( $arg_r->{list} )
            : $self->perr('list');

        my $destroy
            = exists $arg_r->{destroy}
            ? $self->l( $arg_r->{destroy} )
            : $self->perr('destroy');

        my $title
            = exists $arg_r->{title}
            ? $self->l( $arg_r->{title} )
            : $self->perr('title');

        $logger->debug("testobj: $testobj");
        $logger->debug("list   : $list");
        $logger->debug("destroy: $destroy");
        $logger->debug("title  : $title");

        # cipux_task_destroy_'.$entity.' (maybe)
        # (1) first list, to see if it is there
        my $answer_hr = $self->xmlrpc(
            {
                title    => $title . q{ } . $testobj . ' list',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => $list,
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        $logger->debug( 'answer_hr: ',
            { filter => \&Dumper, value => $answer_hr } );

        # remember status, if we have to destroy the testaccount first
        my $result = $self->member_list(
            { answer_hr => $answer_hr, member => $testobj } );

        #(2) the, if it is there destroy it
        if ($result) {

            if ($destroy) {
                my $answer_hr = $self->xmlrpc(
                    {
                        title    => $title . q{ } . $testobj . ' exec',
                        print    => 'operating on: ' . $testobj,
                        url      => $url,
                        login    => $login,
                        ticket   => $ticket,
                        cmd      => $destroy,
                        param_hr => { object => $testobj, },
                        expect   => 'TRUE',
                    }
                );
            }
        }
        else {

            $title =~ s/01/02/xms;

            # this is a present
            $self->record_test_result(
                {
                    start   => 1,
                    title   => $title . q{ } . $testobj . ' exec',
                    print   => 'operating on: ' . $testobj,
                    mode    => 'true',
                    true    => ( 1 == 1 ),
                    success => 'GOOD: This is a present.',
                    failure => 'BAD: Not possible.'
                }
            );
        }

        return;

    }

    sub create_list_destroy_test {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $title
            = exists $arg_r->{title}
            ? $self->l( $arg_r->{title} )
            : $self->perr('title');

        my $create
            = exists $arg_r->{create}
            ? $self->l( $arg_r->{create} )
            : $self->perr('create');

        my $destroy
            = exists $arg_r->{destroy}
            ? $self->l( $arg_r->{destroy} )
            : $self->perr('destroy');

        my $list
            = exists $arg_r->{list}
            ? $self->l( $arg_r->{list} )
            : $self->perr('list');

        my $testobj
            = exists $arg_r->{testobj}
            ? $self->l( $arg_r->{testobj} )
            : $self->perr('testobj');

        my $add_hr
            = exists $arg_r->{add_hr}
            ? $self->h( $arg_r->{add_hr} )
            : $self->perr('add_hr');

        $self->probably_delete(
            {
                title   => $title . ' probably delete',
                list    => $list,
                destroy => $destroy,
                testobj => $testobj,
            }
        );

        $add_hr->{object} = $testobj;

        my $answer_hr = $self->xmlrpc(
            {
                title    => $title . q{ } . $create . ' exec',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => $create,
                param_hr => $add_hr,
                expect   => 'TRUE',
            }
        );

        $answer_hr = $self->xmlrpc(
            {
                title    => $title . q{ } . $list . ' list',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => $list,
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        if ( defined $answer_hr->{msg} ) {
            $self->out("     Server answer: $answer_hr->{msg}\n");
        }

        my $result = $self->member_list(
            { answer_hr => $answer_hr, member => $testobj } );

        $self->record_test_result(
            {
                start   => 1,
                title   => $title . q{ } . $create . ' result',
                mode    => 'true',
                true    => $result,
                print   => $create . 'should create something',
                success => 'GOOD: ' . $testobj . ' was created',
                failure => 'BAD: ' . $testobj . ' was NOT created'
            }
        );

        $answer_hr = $self->xmlrpc(
            {
                title    => $title . q{ } . $destroy . ' exec',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => $destroy,
                param_hr => { object => $testobj },
                expect   => 'TRUE',
            }
        );

        $answer_hr = $self->xmlrpc(
            {
                title    => $title . q{ } . $list . ' list',
                url      => $url,
                login    => $login,
                ticket   => $ticket,
                cmd      => $list,
                param_hr => {},
                expect   => 'TRUE',
            }
        );

        $result = $self->member_list(
            { answer_hr => $answer_hr, member => $testobj } );

        $self->record_test_result(
            {
                start   => 1,
                title   => $title . q{ } . $destroy . ' result 1',
                mode    => 'true',
                true    => ( not $result ),
                print   => $destroy . 'should destroyed ' . $testobj,
                success => 'GOOD: ' . $testobj . ' was destroyed',
                failure => 'BAD: ' . $testobj . ' was NOT destroyed'
            }
        );

        return;
    }

   # @member = $self->member_list({answer_hr => $answer_hr});
   # will give back memberUid of answer
   #
   # $member_hr = $self->member_list({answer_hr => $answer_hr});
   # will give back $member{member} = 1 hash reference
   #
   # $true = $self->member_list({answer_hr => $answer_hr, member => $member});
   # will give bach TRUE if member is in answer, else will give back FALSE
   #
   # optional target_attr may be specified, default = memberUid
   #
    sub member_list {

        my ( $self, $arg_r ) = @_;

        my $logger = get_logger(__PACKAGE__);

        my $answer_hr
            = ( exists $arg_r->{answer_hr} )
            ? $self->h( $arg_r->{answer_hr} )
            : $self->perr('answer_hr');

        my $member
            = ( exists $arg_r->{member} )
            ? $self->l( $arg_r->{member} )
            : $EMPTY_STRING;

        my $task = $answer_hr->{cmd};
        $logger->debug("task: $task");

        # This is only used for groups
        # TODO: remove this and use CipUX::RPC::Client
        my %target_attr = (
            cipux_task_list_admin_accounts               => 'memberUid',
            cipux_task_list_student_accounts             => 'memberUid',
            cipux_task_list_teacher_accounts             => 'memberUid',
            cipux_task_list_pupil_accounts               => 'memberUid',
            cipux_task_list_lecturer_accounts            => 'memberUid',
            cipux_task_list_assistant_accounts           => 'memberUid',
            cipux_task_list_professor_accounts           => 'memberUid',
            cipux_task_list_tutor_accounts               => 'memberUid',
            cipux_task_list_members_of_role_account      => 'memberUid',
            cipux_task_list_members_of_admin_account     => 'memberUid',
            cipux_task_list_members_of_student_account   => 'memberUid',
            cipux_task_list_members_of_teacher_account   => 'memberUid',
            cipux_task_list_members_of_pupil_account     => 'memberUid',
            cipux_task_list_members_of_lecturer_account  => 'memberUid',
            cipux_task_list_members_of_assistant_account => 'memberUid',
            cipux_task_list_members_of_professor_account => 'memberUid',
            cipux_task_list_members_of_tutor_account     => 'memberUid',
            cipux_task_list_course_shares                => 'uid',
            cipux_task_list_class_shares                 => 'uid',
            cipux_task_list_lecture_shares               => 'uid',
            cipux_task_list_seminar_shares               => 'uid',
            cipux_task_list_workshop_shares              => 'uid',
            cipux_task_list_reading_shares               => 'uid',
            cipux_task_list_studygroup_shares            => 'uid',
            cipux_task_list_team_shares                  => 'uid',
            cipux_task_list_tutorial_shares              => 'uid',
            cipux_task_list_members_of_course_share      => 'memberUid',
            cipux_task_list_members_of_class_share       => 'memberUid',
            cipux_task_list_members_of_lecture_share     => 'memberUid',
            cipux_task_list_members_of_seminar_share     => 'memberUid',
            cipux_task_list_members_of_workshop_share    => 'memberUid',
            cipux_task_list_members_of_reading_share     => 'memberUid',
            cipux_task_list_members_of_studygroup_share  => 'memberUid',
            cipux_task_list_members_of_team_share        => 'memberUid',
            cipux_task_list_members_of_tutorial_share    => 'memberUid',
        );
        my $target_attr
            = ( defined $target_attr{$task} ) ? $target_attr{$task}
            : ( exists $arg_r->{taget_attr} )
            ? $self->l( $arg_r->{target_attr} )
            : 'cn';

        #$target_attr = $target_attr{$task}; # memberUid

        $logger->debug( 'answer_hr: ',
            { filter => \&Dumper, value => $answer_hr } );
        $logger->debug("member: $member");
        $logger->debug("target_attr: $target_attr");

        my $z       = 0;
        my $user_hr = $answer_hr->{cmdres_r};
        my @member  = ();
        my %member  = ();
        my $result  = 0;

        # admins, pupil, ...
        foreach my $key ( sort keys %{$user_hr} ) {
            $logger->debug( 'OK, got object key : ', $key );

            # cn, memberUid
            foreach my $value ( @{ $user_hr->{$key}->{$target_attr} } ) {
                push @member, $value;
                $member{$value} = 1;
                if ( $value eq $member ) { $result = 1; }
                {
                    $self->out("     OK, got obj: $value\n");
                    $logger->debug("OK, got object $z: $value");
                    $z++;
                }
            }
        }

        if ( $member ne $EMPTY_STRING ) {
            return $result;
        }

        return @member if wantarray;
        return \%member;
    }

}    # END INSIDE-OUT CLASS

1;

__END__


=pod

=head1 NAME

CipUX::RPC::Test::Client - libray for test clients

=head1 VERSION

version 3.4.0.9

=head1 USAGE

    use CipUX::RPC::Test::Client;

=head1 DESCRIPTION

This is the library for CipUX XML-RPC test clients. See man page of those
clients (for example cipux_rpc_test_client) for details on the clients.

=head1 METHODS

=head2 create_list_destroy_test
=head2 big_entity_test
=head2 login
=head2 manual_test
=head2 member_list
=head2 plain_entity_test
=head2 probably_delete
=head2 record_test_result
=head2 run
=head2 test_create_list_destroy1
=head2 test_create_list_destroy2
=head2 test_entity1
=head2 test_entity2
=head2 test_list_admin
=head2 test_list_users
=head2 test_login
=head2 test_login_again
=head2 test_logout
=head2 test_ping
=head2 test_plain_entity
=head2 test_session
=head2 test_sum1
=head2 test_sum2
=head2 test_summary
=head2 test_ttl
=head2 test_version
=head2 xmlrpc

=head1 DIAGNOSTICS

TODO

=head1 EXIT STATUS

 1 on failure

 0 on success

 other from XML-RPC server

=head1 CONFIGURATION

Not needed.

=head1 DEPENDENCIES

 Carp
 Class::Std
 CipUX::RPC
 CipUX::Task
 Data::Dumper
 Date::Manip
 English
 Frontier::Client
 Getopt::Long
 Log::Log4perl
 Pod::Usage
 Readonly

=head1 INCOMPATIBILITIES

Not known.

=head1 BUGS AND LIMITATIONS

Not known.

=head1 AUTHOR

Christian Kuelker  E<lt>christian.kuelker@cipworx.orgE<gt>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2008 - 2009 by Christian Kuelker

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA


=cut

