// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/google/google_brand_code_map_chromeos.h"

#include "base/containers/flat_map.h"
#include "base/no_destructor.h"
#include "base/stl_util.h"

namespace google_brand {
namespace chromeos {

std::string GetRlzBrandCode(
    const std::string& static_brand_code,
    base::Optional<policy::MarketSegment> market_segment) {
  struct BrandCodeValueEntry {
    const char* unenrolled_brand_code;
    const char* education_enrolled_brand_code;
    const char* enterprise_enrolled_brand_code;
  };
  static const base::NoDestructor<
      base::flat_map<std::string, BrandCodeValueEntry>>
      kBrandCodeMap({{"ACAC", {"CFZM", "BEUH", "GUTN"}},
                     {"ACAG", {"KSOU", "MUHR", "YYJR"}},
                     {"ACAH", {"KEFG", "RYNH", "HHAZ"}},
                     {"ACAI", {"BKWQ", "CMVE", "VNFQ"}},
                     {"ACAJ", {"KVPC", "UHAI", "CPNG"}},
                     {"ACAK", {"PQNT", "MDWH", "AJKH"}},
                     {"ACAM", {"HBCZ", "ZGSZ", "MFUO"}},
                     {"ACAO", {"MWDF", "BNNY", "SYIY"}},
                     {"ACAP", {"LKNW", "SVFL", "FGKR"}},
                     {"ACAQ", {"JXWA", "PZLO", "AKLZ"}},
                     {"ACAR", {"EAQE", "UHHJ", "ZYFW"}},
                     {"ACAS", {"JEHD", "XUID", "FNGA"}},
                     {"ACAT", {"RJNJ", "CKCB", "VHGI"}},
                     {"ACAU", {"DXIN", "LPNB", "FIXM"}},
                     {"ACAV", {"TTSD", "XTQQ", "TIQC"}},
                     {"ACAX", {"CFKW", "QKXN", "VXIS"}},
                     {"ACAY", {"HKDC", "RYKK", "KSIY"}},
                     {"ACAZ", {"VHDQ", "AOTU", "WAMY"}},
                     {"ACBA", {"TVZD", "HLQR", "DOWV"}},
                     {"ACBB", {"ABIB", "LFJO", "ZQSG"}},
                     {"ACBC", {"UFPX", "WVQM", "MLYC"}},
                     {"ACBD", {"BRFU", "HBXU", "FAQM"}},
                     {"ACBE", {"JQFF", "GOJK", "ALHM"}},
                     {"ACBF", {"SSNP", "VHIH", "QMFD"}},
                     {"ADGK", {"PKUQ", "AEMI", "CUUL"}},
                     {"ALRH", {"XDKE", "TDIH", "VLER"}},
                     {"AOKF", {"ZKSY", "KRPA", "QAZL"}},
                     {"AOPA", {"TTBL", "HHHA", "SADO"}},
                     {"AOPB", {"WFJJ", "ZQCN", "OMBM"}},
                     {"AOPC", {"CAKV", "AASR", "BXLX"}},
                     {"AOPD", {"PFAH", "NIUP", "MQIF"}},
                     {"AOPE", {"ZMYO", "BBQM", "UOUV"}},
                     {"ARBI", {"GFHA", "FBQD", "WRQR"}},
                     {"ASCT", {"CTRF", "LBBD", "YBND"}},
                     {"ASUA", {"IEIT", "JAIV", "MURN"}},
                     {"ASUB", {"QBBW", "RUGL", "UVVX"}},
                     {"ASUD", {"QLMM", "CRUA", "JSID"}},
                     {"ASUE", {"XLEN", "KECH", "HBGX"}},
                     {"ASUF", {"IVGE", "VNTM", "XELD"}},
                     {"ASUG", {"TSGW", "DZUL", "HFLO"}},
                     {"ASUH", {"DDNS", "OMSX", "VVWZ"}},
                     {"ASUJ", {"HJUL", "XWWL", "WSCY"}},
                     {"ASUK", {"RGUX", "OXBQ", "LDTL"}},
                     {"ASUL", {"ZJXP", "HGDC", "OFPC"}},
                     {"ASUN", {"ERAF", "HZQI", "JBDP"}},
                     {"ASUO", {"RCMY", "NLPS", "JOKT"}},
                     {"AYMH", {"BBMB", "VBWP", "BVTP"}},
                     {"BAQN", {"YJJJ", "LDCA", "QSJF"}},
                     {"BAUA", {"UWIF", "EOEW", "RPDR"}},
                     {"BCOL", {"YJDV", "GSIC", "BAUL"}},
                     {"BDIW", {"UDUG", "TRYQ", "PWFV"}},
                     {"CBUY", {"POUW", "GHJY", "USXU"}},
                     {"CLSF", {"OWOB", "RLJX", "OZWK"}},
                     {"CPPT", {"CQFF", "PCCZ", "HZEW"}},
                     {"CQPQ", {"GATZ", "QAVU", "WRXC"}},
                     {"CSLV", {"BZSH", "ZDXA", "HGGZ"}},
                     {"CTIE", {"EURD", "HZJV", "WBJJ"}},
                     {"CYQR", {"XGJJ", "DRMC", "RUQD"}},
                     {"CYSQ", {"NHHD", "TAVM", "FHSA"}},
                     {"DEAA", {"HXUG", "BJUN", "IYTV"}},
                     {"DEAB", {"ARPQ", "MFRJ", "JWTH"}},
                     {"DEAC", {"DSMM", "IXET", "KQDV"}},
                     {"DEAD", {"QJXG", "AGGP", "GYQT"}},
                     {"DEAE", {"NZAS", "IHEL", "JSYE"}},
                     {"DEAF", {"TATK", "RWXF", "DQDT"}},
                     {"DEAG", {"JFEX", "CVLN", "UFWN"}},
                     {"DEAH", {"HRBU", "DJKF", "CMPZ"}},
                     {"DRYI", {"LWTQ", "OLEY", "NWUA"}},
                     {"DVUG", {"HJHV", "KPAH", "DCQS"}},
                     {"DXZT", {"WNSK", "WNDA", "DZWQ"}},
                     {"ELQA", {"GTJZ", "DTIH", "IXVN"}},
                     {"FCVS", {"HOBX", "YMDN", "GKTP"}},
                     {"FQPJ", {"ZTQG", "ZNEO", "LYMZ"}},
                     {"FQZI", {"WPBA", "YZDA", "FXCI"}},
                     {"FSFR", {"ZDAR", "BERM", "COKX"}},
                     {"FSGY", {"PJQC", "RHZW", "POVI"}},
                     {"FWVK", {"MUTD", "GWKK", "SQSC"}},
                     {"GBWE", {"DKLE", "OUDI", "VWJC"}},
                     {"GJZV", {"BUSA", "GIOS", "UYOM"}},
                     {"GOKU", {"PRAG", "PQVF", "PIDI"}},
                     {"HDPY", {"JCUZ", "TMKK", "XMVQ"}},
                     {"HFRG", {"YGYA", "IWET", "PSFN"}},
                     {"HHRN", {"IGZW", "ICRP", "QQKJ"}},
                     {"HOMH", {"BXHI", "WXYD", "VRZY"}},
                     {"HOWM", {"MJNG", "XPYN", "IRWY"}},
                     {"HPZO", {"SICM", "XEGH", "TDJJ"}},
                     {"HPZP", {"NQDY", "QIMT", "QKAK"}},
                     {"HPZQ", {"XGER", "OLTF", "DVQA"}},
                     {"HPZR", {"ZAQH", "WPSK", "TCHA"}},
                     {"HPZS", {"QRFK", "SQGI", "VESI"}},
                     {"HPZT", {"IUCU", "WDAV", "LOLH"}},
                     {"HPZV", {"WAFN", "PQVW", "MJVM"}},
                     {"HPZW", {"TLLY", "WNPD", "XIFO"}},
                     {"HPZX", {"DNXN", "VHRG", "XTRZ"}},
                     {"HPZY", {"RAWP", "CNRC", "TPIA"}},
                     {"HPZZ", {"FJGP", "GMLT", "SZQX"}},
                     {"HUIJ", {"EVJI", "RNMR", "JQZR"}},
                     {"IHZG", {"MLLN", "EZTK", "GJEJ"}},
                     {"INUT", {"BRSN", "OJOO", "DWSP"}},
                     {"JBPA", {"VUZL", "XYPI", "XOWE"}},
                     {"JFZB", {"PFDC", "XJDX", "CPXX"}},
                     {"JLRH", {"SAMJ", "GLJZ", "SKTN"}},
                     {"JPZQ", {"CCBQ", "ABTW", "KFNE"}},
                     {"JXIS", {"ZYZD", "TEIT", "ILLN"}},
                     {"JYXK", {"USZT", "XXPU", "LJHH"}},
                     {"KABJ", {"ISGW", "KOHG", "BPGB"}},
                     {"KLKW", {"PIDD", "JIKU", "QTVN"}},
                     {"KXUH", {"RIFT", "DZUO", "ZSEI"}},
                     {"LASN", {"ILWC", "BQYG", "RROZ"}},
                     {"LEAA", {"DHUB", "OBDS", "YMSJ"}},
                     {"LEAB", {"LRHX", "EFFC", "SZFH"}},
                     {"LEAC", {"DMEA", "EXWD", "PBTU"}},
                     {"LEAD", {"QXLJ", "GDOH", "RJNB"}},
                     {"LEAE", {"QFVM", "GACH", "BMXB"}},
                     {"LEAF", {"KGXB", "OUVB", "GTLI"}},
                     {"LEAG", {"XTLW", "WLQO", "QVKP"}},
                     {"LEAH", {"QIDR", "XBTQ", "QYUO"}},
                     {"LEAI", {"KCSV", "PRBF", "FVDO"}},
                     {"LEAJ", {"OBPJ", "NJJS", "WOFS"}},
                     {"LEAK", {"CGWM", "ZLOS", "JGTD"}},
                     {"LEAL", {"EYPX", "SOCH", "PFPW"}},
                     {"LEAM", {"ZGEL", "KZQA", "PSAL"}},
                     {"LEAN", {"MEYH", "PXTT", "LFVK"}},
                     {"LEAO", {"MKOE", "YJSI", "QQMN"}},
                     {"LEAP", {"AEZG", "JOYE", "JHWK"}},
                     {"LGAA", {"YOGJ", "UGWO", "DAMU"}},
                     {"LOEB", {"HPPW", "LGZO", "NZIZ"}},
                     {"LOEM", {"XQSP", "HYHH", "GQQF"}},
                     {"LOFP", {"DGNA", "ZXHN", "ARBG"}},
                     {"LOFR", {"ZOWO", "ZSTS", "JXBM"}},
                     {"LOFY", {"SCIO", "HJKR", "TBOP"}},
                     {"LYLN", {"XXWY", "JEUV", "RSOC"}},
                     {"MAII", {"EOHR", "XZOT", "VJJS"}},
                     {"MCDN", {"BAOV", "GLVV", "XHGO"}},
                     {"MCOO", {"IPNW", "CRSK", "QTAX"}},
                     {"MNZG", {"PPTP", "OFXE", "ROJJ"}},
                     {"MXUY", {"IRZH", "ADQR", "PCST"}},
                     {"NBQS", {"KMJF", "MFWA", "UWRX"}},
                     {"NOMD", {"GZLV", "UNZR", "FVOP"}},
                     {"NPEC", {"BMGD", "YETH", "XAWJ"}},
                     {"OFPE", {"YFOO", "UIGY", "PFGZ"}},
                     {"OKWC", {"RGFB", "UPFP", "HUVK"}},
                     {"PAZD", {"VARX", "KZSU", "WPLH"}},
                     {"PGQF", {"USPJ", "SFKO", "KNBH"}},
                     {"PHYB", {"EGXD", "KHYC", "QUPU"}},
                     {"PXDO", {"ZXCF", "TQWC", "HOAL"}},
                     {"QKTA", {"USGV", "UPMS", "ZVTZ"}},
                     {"QYGU", {"FYBR", "QLFJ", "OLRV"}},
                     {"QZUX", {"HNBM", "BUJY", "FFDE"}},
                     {"RKRB", {"OPOY", "QMZZ", "FAGR"}},
                     {"RVRM", {"MZJU", "IGXP", "DSJP"}},
                     {"SHAN", {"OERN", "XNHK", "GVYX"}},
                     {"SMAC", {"FDEX", "ZFXY", "DJMW"}},
                     {"SMAD", {"AADC", "URZK", "UBVE"}},
                     {"SMAE", {"SUUV", "QXWL", "LYKX"}},
                     {"SMAF", {"HKPA", "NFCE", "UBOP"}},
                     {"SMAG", {"DPGH", "PQFA", "ROEP"}},
                     {"SMAH", {"EXLB", "YYYY", "LLLA"}},
                     {"SMAI", {"PPDO", "ISMM", "BKNT"}},
                     {"SMAJ", {"PVCB", "UCIK", "XVBK"}},
                     {"SMAK", {"WOMZ", "OHAX", "JSTF"}},
                     {"SMAL", {"OWLX", "YXSA", "TXJR"}},
                     {"SOCA", {"AJGR", "IYZW", "NPDX"}},
                     {"TAAB", {"ZBMY", "NYDT", "CXYZ"}},
                     {"TAAC", {"YBVP", "RXXN", "HMDY"}},
                     {"TKER", {"KOSM", "IUCL", "LIIM"}},
                     {"TKZT", {"KWCM", "APLN", "STGO"}},
                     {"TMTX", {"CNAW", "BEDK", "HGOT"}},
                     {"UGAY", {"YDHM", "HVCY", "ILHO"}},
                     {"UMAU", {"FKAK", "JCTZ", "GDUU"}},
                     {"VEUT", {"JDFA", "ALIR", "DDJM"}},
                     {"VHUH", {"JYDF", "SFJY", "JMBU"}},
                     {"VVUC", {"WQCU", "YUMW", "YHYC"}},
                     {"WBZQ", {"LAYK", "LQDM", "QBFV"}},
                     {"WJOZ", {"BASQ", "BRTL", "CQAV"}},
                     {"WMVU", {"GMMR", "AVVS", "IMDF"}},
                     {"XFUX", {"UHAM", "NEHU", "SHMG"}},
                     {"XVTK", {"TMUU", "BTWW", "THQH"}},
                     {"XVYQ", {"UAVB", "OEMI", "VQVK"}},
                     {"XWJE", {"KDZI", "IYPJ", "ERIM"}},
                     {"YHYU", {"CDLM", "QDXQ", "HPTE"}},
                     {"YMJL", {"LBTX", "YPBE", "LHMF"}},
                     {"YMMU", {"ZVIA", "CFKN", "ERLO"}},
                     {"YPCE", {"CCCC", "VHQK", "PYBL"}},
                     {"YVRQ", {"LBMS", "AKKB", "UFNF"}},
                     {"YXBK", {"VKAU", "HUNQ", "AFRP"}},
                     {"YXED", {"KDUD", "MTUI", "WLHI"}},
                     {"ZDKS", {"UBRP", "AWQF", "GOVG"}},
                     {"ZFCZ", {"JQUA", "SEEH", "RJVV"}},
                     {"ZIWS", {"GSAE", "JJUF", "ZPRA"}},
                     {"ZKJH", {"OBDQ", "OUAQ", "SPYY"}},
                     {"ZSKM", {"JPEZ", "FTUS", "ZFUF"}},
                     {"ZZAB", {"WVIK", "IUXK", "ZCIK"}},
                     {"ZZAC", {"MBDD", "SMUW", "JEIY"}},
                     {"ZZAD", {"KSTH", "CBJY", "TSID"}},
                     {"ZZAF", {"OTWH", "RRNB", "VNXA"}}});

  const auto it = kBrandCodeMap->find(static_brand_code);
  if (it == kBrandCodeMap->end())
    return static_brand_code;
  const auto& entry = it->second;
  // An empty value indicates the device is not enrolled.
  if (!market_segment.has_value())
    return entry.unenrolled_brand_code;

  switch (market_segment.value()) {
    case policy::MarketSegment::EDUCATION:
      return entry.education_enrolled_brand_code;
    case policy::MarketSegment::ENTERPRISE:
    case policy::MarketSegment::UNKNOWN:
      // If the device is enrolled but market segment is unknown, it's fine to
      // treat it as enterprise enrolled.
      return entry.enterprise_enrolled_brand_code;
  }
  NOTREACHED();
  return static_brand_code;
}

}  // namespace chromeos
}  // namespace google_brand
