/*
 * log.c - logging functions
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#include <config.h>

#include "log.h"

static FILE* util_log_file = NULL;
static log_function_type *current_log_function = log_file;
static uint8_t verbosity = 0;

/** Open logfile */
void
open_logfile(const char *filename, uint8_t level)
{
	if (filename)
	{
		FILE *file = fopen(filename, "a");
		if (!file)
			error("cannot open %s for appending, logging to stderr",
								filename);
		else
			util_log_file = file;
	}
	verbosity = level;
}

/** Close logfile */
void
close_logfile(void)
{
	if (util_log_file)
		fclose(util_log_file);
}

/** Set log function */
void
set_logfunction(log_function_type* log_function)
{
	current_log_function = log_function;
}

void
log_file(int level, char* message)
{
	if (util_log_file)
	{
		if (level != LOG_RAW)  	    fprintf(util_log_file, "[%s] ",
								AUTOTRUST_NAME);
		if (level == LOG_ERROR)	    fprintf(util_log_file, "error: ");
		else if (level==LOG_FATAL)  fprintf(util_log_file, "fatal: ");
		else if (level==LOG_WARN)   fprintf(util_log_file, "warning: ");
		else if (level==LOG_VERBOSE)fprintf(util_log_file, "verbose: ");
		else if (level==LOG_DBG)  fprintf(util_log_file, "debug: ");
		else if (level == LOG_MSG)  fprintf(util_log_file, "notice: ");
		fprintf(util_log_file, "%s", message);
		if (level != LOG_RAW)       fprintf(util_log_file, "\n");
		return;
	}

	if (level != LOG_RAW)  	    fprintf(stderr, "[%s] ", AUTOTRUST_NAME);
	if (level == LOG_ERROR)		fprintf(stderr, "error: ");
	else if (level == LOG_FATAL)	fprintf(stderr, "fatal: ");
	else if (level == LOG_WARN)	fprintf(stderr, "warning: ");
	else if (level == LOG_VERBOSE)	fprintf(stderr, "verbose: ");
	else if (level == LOG_DBG)	fprintf(stderr, "debug: ");
	else if (level == LOG_MSG)	fprintf(stderr, "notice: ");
	fprintf(stderr, "%s", message);
	if (level != LOG_RAW)		fprintf(stderr, "\n");
}

void
log_syslog(int level, char* message)
{
#ifdef HAVE_SYSLOG_H
	syslog(level, "%s", message);
	return;
#endif /* HAVE_SYSLOG_H */
	log_file(level, message);
}

/** Log message wrapper */
static void
log_vmsg(log_level level, const char *format, va_list args)
{
	char message[MAXLEN];
	vsnprintf(message, sizeof(message), format, args);
	current_log_function((int) level, message);
}

/** Log notify message */
void
notice(const char *format, ...)
{
	va_list args;
	va_start(args, format);
	log_vmsg(LOG_MSG, format, args);
	va_end(args);
}

/** Log warnings */
void
warning(const char *format, ...)
{
	va_list args;
	va_start(args, format);
	log_vmsg(LOG_WARN, format, args);
	va_end(args);
}

/** Log error message */
void
error(const char *format, ...)
{
	va_list args;
	va_start(args, format);
	log_vmsg(LOG_ERROR, format, args);
	va_end(args);
}


/** Log fatal error message */
void
fatal(const char *format, ...)
{
	va_list args;
	va_start(args, format);
	log_vmsg(LOG_FATAL, format, args);
	va_end(args);
}

/** A raw message, with no schmuck around it */
void
raw(const char *format, ...)
{
	va_list args;
	va_start(args, format);
	log_vmsg(LOG_RAW, format, args);
	va_end(args);
}

/** Verbose messages */
void
verbos(uint8_t level, const char *format, ...)
{
	if (verbosity >= level)
	{
		va_list args;
		va_start(args, format);
		log_vmsg(LOG_VERBOSE, format, args);
		va_end(args);
	}
}

/** Flush the logs */
void
flush(void)
{
	fflush(util_log_file);
}

/** Debug printing */
void
print_debug(const char *format, ...)
{
	va_list args;
	va_start(args, format);
	log_vmsg(LOG_DBG, format, args);
	va_end(args);
}

void
print_verbose(const char *format, ...)
{
	if (verbosity >= 100) {
		va_list args;
		va_start(args, format);
		log_vmsg(LOG_DBG, format, args);
		va_end(args);
	}
}

/** Get message by status id */
const char*
status_by_id(int status)
{
	switch (status)
	{
		case STATUS_OK: return "ok";
		case STATUS_USAGE: return "show usage";
		case STATUS_ERR_MALLOC: return "malloc failed";
		case STATUS_ERR_GETOPT: return "parse error on "
					       "command line";
		case STATUS_ERR_CHDIR: return "cannot chdir";
		case STATUS_ERR_CONFIG: return "error loading configfile";
		case STATUS_ERR_PARSE_ERROR: return "parse error";
		case STATUS_ERR_BUFFERSIZE: return "buffer too small";
		case STATUS_ERR_CFG_OPTION: return "unknown configuration "
						   "setting";
		case STATUS_ERR_QUERY: return "query error";
		case STATUS_ERR_INCLUDE: return "error include config";
		case STATUS_ERR_AUTO_UPDATE: return "cannot auto update";
		case STATUS_UNKNOWN:
		default: return "unknown status";
	}

        return "unknown status";
}
