/* InvariantListSaver.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io.list;

import java.io.IOException;
import java.io.OutputStream;
import java.util.logging.Logger;

import org.grinvin.invariants.Invariant;
import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.io.SectionSaver;
import org.grinvin.list.invariants.InvariantList;

import org.jdom.DocType;
import org.jdom.Element;

/**
 *
 */
public class InvariantListSaver {    

    //
    private static final String PUBLIC_ID = "-//GrInvIn IO//InvariantList 1.0//EN";
    
    //
    private static final Logger LOGGER = Logger.getLogger("org.grinvin.io");
    
    private InvariantListSaver() { }
    
    /**
     * Save a list of invariants as an XML file. Creates a file with contents
     * of the form
     * <pre>
     *    &lt;invariants&gt;
     *        &lt;invariant invariantId="..."/&gt;
     *        ...
     *    &lt;/invariants&gt;
     * </pre>
     * @param list List(model) of invariants to be stored
     * @param file File into which these invariants should be stored
     */
    private static void save(InvariantList list, OutputStream out) throws IOException {
        Element element = new Element("invariantlist");
        for (Invariant inv: list)
            element.addContent(new Element("invariant").setAttribute("id", inv.getId()));
        LoaderSaverHelper.outputXML(element,
                new DocType("invariantlist", PUBLIC_ID, "http://downloads.grinvin.org/dtds/io/invariantlist-1.0.dtd"),
                out);
    }
    
    
    /**
     * Save a (session) invariant list into a (workspace) directory. Uses the uri
     * of the invariant list to determine the destination file name. If no URI was
     * assigned, a new URI is created.<p>
     * @throws IllegalArgumentException if the URI of the list is non null and
     * not a session URI
     */
    public static void saveIntoWorkspace(InvariantList list, SectionSaver ssaver) throws IOException {
        String name = LoaderSaverHelper.prepareSessionURI(list, "invariantlists", "gil");
        save(list, ssaver.createSection(name));
    }
    
}
