/* GraphSaver.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io.graphs;

import java.io.IOException;
import java.io.OutputStream;

import org.grinvin.graphs.Edge;
import org.grinvin.graphs.GraphView;
import org.grinvin.graphs.Vertex;
import org.grinvin.io.LoaderSaverHelper;

import org.jdom.DocType;
import org.jdom.Element;

/**
 * Saves a {@link GraphView} as a <tt>graph.xml</tt>-section onto an output
 * stream.
 */
public class GraphSaver {
    
    //
    private GraphSaver() { }
    
    /**
     * Create a JDOM-element that represents the abstract graph
     * for this initializer.
     */
    public static Element toElement(GraphView graph) {
        
        int numberOfVertices = 0;
        for (Vertex v : graph.vertices())
            numberOfVertices++;
        
        Element element = new Element("graph");
        element.setAttribute("nrofvertices", Integer.toString(numberOfVertices));
        
        // vertices are not explicitly saved
        
        // edges (all edges)
        for (Edge e: graph.edges()) {
            Element edge = new Element("edge");
            edge.setAttribute("from", Integer.toString(e.getFirstEndpoint().getIndex()));
            edge.setAttribute("to", Integer.toString(e.getSecondEndpoint().getIndex()));
            element.addContent(edge);
        }
        return element;
    }
    
    /**
     * Save the graph onto the output stream.
     * @param out Output stream to which the result shall be written.
     */
    public static void save(GraphView graph, OutputStream out) throws IOException{
        LoaderSaverHelper.outputXML(toElement(graph), new DocType("graph", "-//GrInvIn IO//Graph 1.1//EN", "http://downloads.grinvin.org/dtds/io/graph-1.1.dtd"), out);
    }
    
}
