/* SmallGraphIcon.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.icons;

import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;

import org.grinvin.graphs.AnnotationView;
import org.grinvin.graphs.Edge;
import org.grinvin.graphs.EmbeddingView;
import org.grinvin.graphs.GraphBundleView;
import org.grinvin.graphs.GraphView;
import org.grinvin.graphs.Vertex;
import org.grinvin.graphs.render.Renderer;

/**
 * Icon that represents a Graph. It uses the first embedding of the
 * GraphBundleView to draw the icon.
 */
public class SmallGraphIcon extends AbstractGraphIcon {
    
    //
    private GraphView graph;
    
    //
    private EmbeddingView embedding;
    
    //
    private AnnotationView annotation;
    
    //
    private Renderer renderer;
    
    /**
     * Size of the bounding square of this icon.
     */
    private int size;
    
    // Scale factor to be used for coordinates
    private int scale;
   
    /**
     * Create a graph icon for the given embedded graph.
     * The (two-dimensional) embedding is assumed to be restricted to
     * coordinates in the range (-1.0,-1.0) - (1.0,1.0) which will be scaled
     * to a square of the indicated size.
     * @param embedding Two-dimensional embedding of a graph
     * @param renderer Renderer which is used to draw the graph
     * @param size Size in pixels for the width and height of the enclosing square
     */
    public SmallGraphIcon(EmbeddingView embedding, AnnotationView annotation, Renderer renderer, int size) {
        this.embedding = embedding;
        this.graph = embedding.getPeer();
        this.annotation = annotation;
        this.renderer = renderer;
        this.size = size;
        this.scale = size/2 - 6;
    }
    
    /**
     * Create a graph icon based on the first embedding stored in the given
     * graph bundle.
     */
    public SmallGraphIcon(GraphBundleView bundle, Renderer renderer, int size) {
        this(bundle.getEmbedding(), bundle.getAnnotation(), renderer, size);
    }

    /**
     * Set or change the size of the bounding square of this icon.
     */
    public void setSize(int size) {
        this.size = size;
        this.scale = size/2 - 6;
    }
    
    // implements Icon
    public int getIconHeight() {
        return size;
    }
    
    // implements Icon
    public int getIconWidth() {
        return size;
    }
    
    /**
     * {@inheritDoc} First paints the edges and then the vertices.
     */
    public void paintIcon(Component c, Graphics g, int x, int y) {
        Graphics2D g2 = (Graphics2D)g.create(); // TODO: regression test for forgetting
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
            RenderingHints.VALUE_ANTIALIAS_ON);
        g2.translate(size/2+x,size/2+y);
        paintEdges(g2);
        paintVertices(g2);
    }
    
    /**
     * Paint all edges of the graph using the current renderer.<p>
     * See {@link #paintIcon} for the order in which the various
     * paint methods are executed.
     */
    protected void paintEdges(Graphics2D g2) {
        for(Edge e : graph.edges()) {
            double[] firstCoordinates = embedding.getCoordinates(e.getFirstEndpoint());
            double[] secondCoordinates = embedding.getCoordinates(e.getSecondEndpoint());
            /* Zoom */
            firstCoordinates[0] *= scale;
            firstCoordinates[1] *= -scale;
            secondCoordinates[0] *= scale;
            secondCoordinates[1] *= -scale;
            if (annotation != null)
                renderer.paint(e, firstCoordinates, secondCoordinates, false, selected, annotation.getAnnotation(e),  g2);
            else
                renderer.paint(e, firstCoordinates, secondCoordinates, false, selected, null,  g2);
        }
    }
    
    /**
     * Paint all vertices of the graph using the current vertex renderer.<p>
     * See {@link #paintIcon} for the order in which the various
     * paint methods are executed.
     */
    protected void paintVertices(Graphics2D g2) {
        for (Vertex v : graph.vertices()) {
            double[] coordinates = embedding.getCoordinates(v);
            /* Zoom */
            coordinates[0] *= scale;
            coordinates[1] *= -scale;
            if (annotation != null)
                renderer.paint(v, coordinates, false, selected, annotation.getAnnotation(v), g2);
            else
                renderer.paint(v, coordinates, false, selected, null, g2);
        }
    }
    
}
