/* EdgesOnlyIcon.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.icons;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.RenderingHints;
import java.awt.Stroke;
import java.awt.geom.Line2D;

import org.grinvin.graphs.Edge;
import org.grinvin.graphs.EmbeddingView;
import org.grinvin.graphs.GraphBundleView;
import org.grinvin.graphs.GraphView;

/**
 * Creates an icon for the first embedding of a graph, but draws
 * only the edges. Typically used for chemical graphs.
 */
public class EdgesOnlyIcon extends AbstractGraphIcon {
    
    // TODO: factor out code in common with SmallGraphIcon
    
    //
    private GraphView graph;
    
    //
    private EmbeddingView embedding;
    
    /**
     * Size of the bounding square of this icon.
     */
    private int size;
    
    // Scale factor to be used for coordinates
    private int scale;
    
    /**
     * Create a graph icon for the given embedded graph.
     * The (two-dimensional) embedding is assumed to be restricted to
     * coordinates in the range (-1.0,-1.0) - (1.0,1.0) which will be scaled
     * to a square of the indicated size.
     * @param embedding Two-dimensional embedding of a graph
     * @param size Size in pixels for the width and height of the enclosing square
     */
    public EdgesOnlyIcon(EmbeddingView embedding, int size) {
        this.embedding = embedding;
        this.graph = embedding.getPeer();
        this.size = size;
        this.scale = size/2 - 6;
    }
    
    /**
     * Create a graph icon based on the first embedding stored in the given
     * graph bundle.
     */
    public EdgesOnlyIcon(GraphBundleView bundle, int size) {
        this(bundle.getEmbedding(), size);
    }
    
    /**
     * Set or change the size of the bounding square of this icon.
     */
    public void setSize(int size) {
        this.size = size;
        this.scale = size/2 - 6;
    }
    
    // implements Icon
    public int getIconHeight() {
        return size;
    }
    
    // implements Icon
    public int getIconWidth() {
        return size;
    }
    

    // Stroke used to draw an edge
    private static final Stroke STROKE = new BasicStroke(1.0f);
    
    // Stroke used to draw an edge
    private static final Stroke SHADOW_STROKE = new BasicStroke(3.0f);
    
    //  Paint used for normal edges
    private static final Paint PAINT = Color.BLACK;
    
    //  Paint used for selected edges
    private static final Paint SELECTED_PAINT  = new Color(0, 127, 255);
    
    // Paint used for shadow edges
    private static final Paint SHADOW_PAINT = new Color (142, 142, 127);

    /**
     * {@inheritDoc} Only paints the edges, first as a shadow and then as
     * true edges.
     */
    public void paintIcon(Component c, Graphics g, int x, int y) {
        Graphics2D g2 = (Graphics2D)g.create(); // TODO: regression test for forgetting
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                RenderingHints.VALUE_ANTIALIAS_ON);
        g2.translate(size/2+x+1,size/2+y);
        g2.setPaint(SHADOW_PAINT);
        g2.setStroke (SHADOW_STROKE);
        paintEdges(g2);
        g2.translate(-2,-1);
        g2.setPaint(selected ? SELECTED_PAINT : PAINT);
        g2.setStroke (STROKE);
        paintEdges(g2);
    }
    
    /**
     * Paint all edges of the graph using the current renderer.<p>
     * See {@link #paintIcon} for the order in which the various
     * paint methods are executed.
     */
    protected void paintEdges(Graphics2D g2) {
        for(Edge e : graph.edges()) {
            double[] firstCoordinates = embedding.getCoordinates(e.getFirstEndpoint());
            double[] secondCoordinates = embedding.getCoordinates(e.getSecondEndpoint());
            /* Zoom */
            firstCoordinates[0] *= scale;
            firstCoordinates[1] *= -scale;
            secondCoordinates[0] *= scale;
            secondCoordinates[1] *= -scale;
            Line2D line = new Line2D.Double(firstCoordinates[0], firstCoordinates[1],
                    secondCoordinates[0], secondCoordinates[1]);
            g2.draw(line);
        }
    }
    
}
